@tool
class_name StoryFileSource extends Object

static var _singleton_instance_mutex := Mutex.new()
static var _singleton_instance: StoryFileSource = null
var _files_mutex := Mutex.new()
var _files_by_path: Dictionary[String, StoryFile]
var _files: Array[StoryFile]


static func singleton() -> StoryFileSource:
	_singleton_instance_mutex.lock()
	_singleton_instance_mutex.unlock.call_deferred()
	if _singleton_instance == null:
		_singleton_instance = new()
	return _singleton_instance


static func reload_file_list() -> void:
	singleton()._reload_file_list()


static func save_all_files() -> void:
	singleton()._save_all_files()


func get_item_count() -> int:
	return _files.size()


func get_item_icon(_index: int) -> Texture2D:
	return StoryFile.FileIcon


func get_item_metadata(index: int) -> Variant:
	return _files[index]


func get_item_text(index: int) -> String:
	return _files[index].resource_path.get_file()


func get_item_tooltip(index: int) -> String:
	return _files[index].resource_path


func _reload_file_list() -> void:
	_files_mutex.lock()
	_files_mutex.unlock.call_deferred()

	# remove deleted files
	var is_deleted_file = func(file: StoryFile) -> bool:
		if file == null:
			return false
		return file.resource_path != ""
	_files = _files.filter(is_deleted_file)
	_files_by_path.clear()
	for file in _files:
		_files_by_path.set(file.resource_path, file)

	# append missing file paths
	var file_paths := PackedStringArray()
	_append_files(file_paths, EditorInterface.get_resource_filesystem().get_filesystem())
	for file_path in file_paths:
		# tricky part: resource filesystem contains also deleted file paths
		if !FileAccess.file_exists(file_path):
			continue

		if _files_by_path.has(file_path):
			continue

		var file = ResourceLoader.load(file_path)
		if file != null:
			_files_by_path.set(file_path, file)
			_files.append(file)

	# sort by file path
	_files.sort_custom(func(a, b): return a.resource_path < b.resource_path)


func _save_all_files() -> void:
	_files_mutex.lock()
	_files_mutex.unlock.call_deferred()

	for file in _files:
		file._save()


static func _append_files(to: PackedStringArray, dir: EditorFileSystemDirectory) -> void:
	for i in dir.get_subdir_count():
		_append_files(to, dir.get_subdir(i))
	for i in dir.get_file_count():
		if dir.get_file(i).ends_with(StoryFile.DOT_EXTENSION):
			to.append(dir.get_file_path(i))
