@tool
class_name StoryPartSelectorDialog extends Window

@onready var _panel: Panel = $Panel
@onready var _files: FilterableItemList = $Margin/Content/Columns/Files
@onready var _parts: FilterableItemList = $Margin/Content/Columns/Parts

var _part: StoryFile.Part
var _on_story_part_selector_dialog_close: Callable


func _ready() -> void:
	var style_box := StyleBoxFlat.new()
	style_box.bg_color = EditorInterface.get_editor_theme().get_color("base_color", "Editor")
	_panel.add_theme_stylebox_override("panel", style_box)
	_files.set_item_source(StoryFileSource.singleton())
	min_size = get_contents_minimum_size()
	reset_size()


func popup_assign(part: StoryFile.Part, preselected: Resource, on_story_part_selector_dialog_close: Callable) -> void:
	_set_part(part)
	_on_story_part_selector_dialog_close = on_story_part_selector_dialog_close
	StoryFileSource.singleton().reload_file_list()
	_files.refresh_items()
	if preselected != null:
		var file: StoryFile = load(preselected.file_path)
		if file != null:
			var part_index = file.find_part_index(_part, StoryPartRef.part_name(_part, preselected))
			if part_index >= 0:
				_files.select_item_by_metadata(file)
				_parts.set_item_source(file.get_part_source(_part))
				_parts.select_item_by_metadata(part_index)
	popup_centered()


func _on_close_requested() -> void:
	_close_story_part_selector_dialog(false)


func _on_window_input(event: InputEvent) -> void:
	if event.is_action("ui_cancel"):
		_close_story_part_selector_dialog(false)


func _on_files_item_selected(_text: String, metadata: Variant) -> void:
	var file: StoryFile = metadata
	_parts.set_item_source(file.get_part_source(_part))


func _on_parts_item_activated(_text: String, _metadata: Variant) -> void:
	_close_story_part_selector_dialog(true)


func _on_assign_pressed() -> void:
	_close_story_part_selector_dialog(true)


func _on_cancel_pressed() -> void:
	_close_story_part_selector_dialog(false)


func _close_story_part_selector_dialog(ok: bool) -> void:
	if _on_story_part_selector_dialog_close:
		var selected = null
		var file: StoryFile = _files.get_selected_item_metadata()
		if file != null && _parts.is_anything_selected():
			var part_index: int = _parts.get_selected_item_metadata()
			selected = StoryPartRef.by(_part, file.resource_path, file.get_part_name(_part, part_index))
		_on_story_part_selector_dialog_close.call(selected, ok)
	hide()


func _on_focus_entered() -> void:
	if _files:
		_files.grab_focus.call_deferred()


func _set_part(part: StoryFile.Part) -> void:
	_part = part
	match _part:
		StoryFile.Part.NOTE:
			title = tr("Select a Story Note")
			_parts.filter_text = tr("Filter Notes")

		StoryFile.Part.TALK:
			title = tr("Select a Story Talk")
			_parts.filter_text = tr("Filter Talks")
