@tool
class_name TalkEdit extends VSplitContainer

signal on_line_changed()

@onready var name_dialog := $NameDialog
@onready var tools_line_move_up := $Lines/Tools/LineMoveUp
@onready var tools_line_move_down := $Lines/Tools/LineMoveDown
@onready var tools_line_move_delta := $Lines/Tools/LineMoveDelta
@onready var tools_line_delete := $Lines/Tools/LineDelete
@onready var lines := $Lines/Lines
@onready var current_label := $Line/CurrentLabel
@onready var current_text := $Line/CurrentText
@onready var translate_label := $Line/TranslateLabel
@onready var translate_text := $Line/TranslateText
@onready var key_text := $Line/Details/KeyText
@onready var key_randomize := $Line/Details/KeyRandomize
@onready var time_enabled := $Line/Details/TimeEnabled
@onready var time_value := $Line/Details/TimeValue
@onready var time_auto := $Line/Details/TimeAuto

var _icon_translation_missing := _get_editor_icon("StatusError")
var _icon_translation_partial := _get_editor_icon("StatusWarning")
var _icon_translation_done := _get_editor_icon("StatusSuccess")
var _locale_current: String = ""
var _locale_translate: String = ""
var _locales: PackedStringArray
var _story_file: StoryFile = null
var _talk_index: int = -1
var _lines_selected_index: int = -1


func _ready() -> void:
	tools_line_move_up.icon = _get_editor_icon("ArrowUp")
	tools_line_move_down.icon = _get_editor_icon("ArrowDown")
	tools_line_delete.icon = _get_editor_icon("Remove")
	key_randomize.icon = _get_editor_icon("NewKey")
	time_auto.icon = _get_editor_icon("Time")


func action_line_insert_new(before: bool) -> void:
	if _is_empty():
		return
	var index := 0 if before else _story_file.get_talk_line_count(_talk_index)
	if _lines_selected_index >= 0:
		index = _lines_selected_index if before else _lines_selected_index+1
	_story_file.create_talk_line(_talk_index, index)
	_refresh_lines(index)


func action_line_move(delta: int) -> void:
	if _is_empty() || _lines_selected_index < 0:
		return
	var index := _story_file.move_talk_line(_talk_index, _lines_selected_index, delta)
	_refresh_lines(index)


func action_line_delete() -> void:
	if _is_empty() || _lines_selected_index < 0:
		return
	var index_after_deletion := _index_after_deletion(_lines_selected_index)
	_story_file.delete_talk_line(_talk_index, _lines_selected_index)
	_refresh_lines(index_after_deletion)


func get_current_locale() -> String:
	return _locale_current


func set_current_locale(locale: String) -> void:
	_locale_current = locale
	current_label.text = TranslationServer.get_locale_name(locale)+" ("+locale+")"
	_refresh_locales()
	_refresh_lines(_lines_selected_index)


func get_translate_locale() -> String:
	return _locale_translate


func set_translate_locale(locale: String) -> void:
	_locale_translate = locale
	translate_label.text = TranslationServer.get_locale_name(locale)+" ("+locale+")"
	_refresh_locales()
	_refresh_lines(_lines_selected_index)


func get_story_file() -> StoryFile:
	return _story_file


func set_story_file(file: StoryFile) -> void:
	_story_file = file
	_talk_index = -1
	_refresh_lines(-1)


func get_talk_index() -> int:
	return _talk_index


func set_talk_index(talk_index: int) -> void:
	_talk_index = talk_index
	_refresh_lines(-1)


func get_selected_talk_line_text() -> String:
	if _is_empty() || _lines_selected_index < 0:
		return ""
	return lines.get_item_text(_lines_selected_index)


func _on_tools_line_insert_before_pressed() -> void:
	action_line_insert_new(true)


func _on_tools_line_insert_after_pressed() -> void:
	action_line_insert_new(false)


func _on_tools_line_move_up_pressed() -> void:
	action_line_move(-tools_line_move_delta.value)


func _on_tools_line_move_down_pressed() -> void:
	action_line_move(+tools_line_move_delta.value)


func _on_tools_line_delete_pressed() -> void:
	var callback := func(_value: String, ok: bool):
		if ok:
			action_line_delete()
	name_dialog.popup_delete(tr("Line"), get_selected_talk_line_text(), callback)


func _on_key_randomize_pressed() -> void:
	while true:
		var key := "%08x" % randi()
		if _story_file.is_talk_line_key_unique(_talk_index, key):
			key_text.set_text(key)
			_story_file.set_talk_line_key(_talk_index, _lines_selected_index, key)
			return


func _on_time_auto_pressed() -> void:
	var time := maxf(current_text.text.length() * 0.08, 2.0)
	_set_time_value(time)
	_story_file.set_talk_line_time(_talk_index, _lines_selected_index, time)


func _set_locales(locales: PackedStringArray) -> void:
	_locales = locales
	for i in lines.get_item_count():
		_refresh_line_translation_status(i)


func _refresh_locales() -> void:
	var has_translate := _locale_current != _locale_translate
	translate_label.visible = has_translate
	translate_text.visible = has_translate


func _refresh_lines(select_after_refresh: int) -> void:
	if _is_empty():
		lines.clear()
		_set_lines_selected_index(-1)
		return
	var lines_count_old: int = lines.get_item_count()
	var lines_count_new := _story_file.get_talk_line_count(_talk_index)
	for i in lines_count_new:
		if i >= lines_count_old:
			lines.add_item("")
			lines.set_item_tooltip_enabled(i, false)
		_lines_set_item_text(i, _story_file.get_talk_line_translation(_talk_index, i, _locale_current))
		_refresh_line_translation_status(i)
	while lines_count_new < lines.get_item_count():
		lines.remove_item(lines.get_item_count()-1)
	if select_after_refresh < 0 || select_after_refresh >= lines.get_item_count():
		lines.deselect_all()
		_set_lines_selected_index(-1)
	else:
		lines.select(select_after_refresh)
		_set_lines_selected_index(select_after_refresh)


func _refresh_line() -> void:
	if _lines_selected_index < 0:
		tools_line_move_up.disabled = true
		tools_line_move_down.disabled = true
		tools_line_delete.disabled = true
		current_text.editable = false
		translate_text.editable = false
		current_text.set_text("")
		translate_text.set_text("")
		key_text.set_text("")
		_set_time_value(NAN)
		key_text.editable = false
		key_randomize.disabled = true
		time_enabled.disabled = true
		return
	current_text.set_text(_story_file.get_talk_line_translation(_talk_index, _lines_selected_index, _locale_current))
	translate_text.set_text(_story_file.get_talk_line_translation(_talk_index, _lines_selected_index, _locale_translate))
	key_text.set_text(_story_file.get_talk_line_key(_talk_index, _lines_selected_index))
	_set_time_value(_story_file.get_talk_line_time(_talk_index, _lines_selected_index))
	tools_line_move_up.disabled = false
	tools_line_move_down.disabled = false
	tools_line_delete.disabled = false
	current_text.editable = true
	translate_text.editable = true
	key_text.editable = true
	key_randomize.disabled = false
	time_enabled.disabled = false


func _refresh_line_translation_status(index: int) -> void:
	if _locale_current == _locale_translate:
		lines.set_item_icon(index, null)
	else:
		var icon = _icon_translation_missing
		if _story_file.has_talk_line_translation(_talk_index, index, _locale_translate):
			icon = _icon_translation_partial if _story_file.missing_talk_line_translation(_talk_index, index, _locales) else _icon_translation_done
		lines.set_item_icon(index, icon)


func _on_lines_selected(index: int) -> void:
	_set_lines_selected_index(index)


func _on_current_text_focus_exited() -> void:
	if _lines_selected_index < 0:
		return
	var text: String = current_text.get_text()
	_story_file.set_talk_line_translation(_talk_index, _lines_selected_index, _locale_current, text)
	_lines_set_item_text(_lines_selected_index, text)
	on_line_changed.emit()


func _on_translate_text_focus_exited() -> void:
	if _lines_selected_index < 0:
		return
	var text: String = translate_text.get_text()
	_story_file.set_talk_line_translation(_talk_index, _lines_selected_index, _locale_translate, text)
	_refresh_line_translation_status(_lines_selected_index)
	on_line_changed.emit()


func _on_key_text_focus_exited() -> void:
	if _lines_selected_index < 0:
		return
	var text: String = key_text.get_text()
	_story_file.set_talk_line_key(_talk_index, _lines_selected_index, text)


func _on_time_enabled_toggled(toggled_on: bool) -> void:
	if _lines_selected_index < 0:
		return
	time_value.editable = toggled_on
	_story_file.set_talk_line_time(_talk_index, _lines_selected_index, time_value.value if toggled_on else NAN)


func _on_time_value_value_changed(value: float) -> void:
	if _lines_selected_index < 0 || !time_value.editable:
		return
	_story_file.set_talk_line_time(_talk_index, _lines_selected_index, value)


func _index_after_deletion(index: int) -> int:
	var count_after_deletion: int = lines.get_item_count()-1
	return index if index < count_after_deletion else count_after_deletion-1


func _set_lines_selected_index(index: int) -> void:
	_lines_selected_index = index
	_refresh_line()


func _set_time_value(value: float) -> void:
	if is_nan(value):
		time_value.set_value(1)
		time_enabled.button_pressed = false
		time_value.editable = false
	else:
		time_value.set_value(value)
		time_enabled.button_pressed = true
		time_value.editable = true


func _is_empty() -> bool:
	return _story_file == null || _talk_index < 0


func _lines_set_item_text(idx: int, text: String) -> void:
	var prefix := "%04d — " % (idx+1)
	lines.set_item_text(idx, prefix+("∅" if text == "" else text))


static func _get_editor_icon(name: StringName) -> Texture2D:
	return EditorInterface.get_editor_theme().get_icon(name, "EditorIcons")
