//go:build ignore

package main

import (
	"fmt"
	"log"
	"os"
	"os/exec"
	"strings"

	"gopkg.in/yaml.v3"
)

type FieldInfo struct {
	Path string `yaml:"path"`
	Type string `yaml:"type"`
}

type FieldsConfig map[string]FieldInfo

func toCamelCase(snakeCase string) string {
	parts := strings.Split(snakeCase, "_")
	var camelCase strings.Builder
	for _, part := range parts {
		if len(part) > 0 {
			camelCase.WriteString(strings.ToUpper(part[:1]))
			if len(part) > 1 {
				camelCase.WriteString(part[1:])
			}
		}
	}
	return camelCase.String()
}

func main() {
	// 读取 YAML 文件
	data, err := os.ReadFile("fields.yaml")
	if err != nil {
		log.Fatal(err)
	}

	var config FieldsConfig
	if err := yaml.Unmarshal(data, &config); err != nil {
		log.Fatal(err)
	}

	// 保持 YAML 文件中的顺序
	yamlContent := string(data)
	lines := strings.Split(yamlContent, "\n")

	var orderedKeys []string
	for _, line := range lines {
		if strings.Contains(line, ":") && !strings.HasPrefix(strings.TrimSpace(line), "path:") &&
			!strings.HasPrefix(strings.TrimSpace(line), "type:") && !strings.HasPrefix(strings.TrimSpace(line), "desc:") &&
			!strings.HasPrefix(line, " ") && !strings.HasPrefix(line, "\t") {
			key := strings.TrimSpace(strings.Split(line, ":")[0])
			if key != "" {
				orderedKeys = append(orderedKeys, key)
			}
		}
	}

	// 生成代码
	var builder strings.Builder

	// 文件头
	builder.WriteString(`/**
 * Copyright (c) 2025 Yunshan Networks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Code generated by go run field_gen.go. DO NOT EDIT.

package pcap_verify

`)

	// 生成 map
	builder.WriteString("var ResponseFieldToAppProtoLogsDataFieldPath = map[string]string{\n")
	for _, key := range orderedKeys {
		if field, exists := config[key]; exists {
			builder.WriteString(fmt.Sprintf("\t\"%s\": \"%s\",\n", key, field.Path))
		}
	}
	builder.WriteString("}\n\n")

	// 生成 struct
	builder.WriteString("type PartialConvertedAppProtoLogsData struct {\n")
	for _, key := range orderedKeys {
		if field, exists := config[key]; exists {
			camelKey := toCamelCase(key)
			builder.WriteString(fmt.Sprintf("\t%s %s `json:\"%s,omitempty\"`\n", camelKey, field.Type, key))
		}
	}
	builder.WriteString("}\n")

	// 写入文件
	if err := os.WriteFile("fields.gen.go", []byte(builder.String()), 0644); err != nil {
		log.Fatal(err)
	}

	// 格式化生成的文件
	cmd := exec.Command("go", "fmt", "fields.gen.go")
	if err := cmd.Run(); err != nil {
		log.Printf("Warning: failed to format fields.gen.go: %v\n", err)
	}

	fmt.Printf("Successfully generated fields.gen.go with %d fields\n", len(orderedKeys))
}
