package com.felhr.serialportexample;

import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.ServiceConnection;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.StrictMode;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v7.app.AppCompatActivity;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.TextView;
import android.widget.Toast;
import android.widget.ProgressBar;

import android.net.Uri;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import android.os.Environment;
import java.io.InputStream;

import android.Manifest;
import android.os.Build;
import android.content.pm.PackageManager;

import java.lang.ref.WeakReference;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Set;
import java.util.zip.CRC32;
import java.util.zip.CheckedInputStream;

import android.provider.DocumentsContract;
import android.content.ContentUris;
import android.database.Cursor;
import android.content.ContentResolver;
import android.provider.MediaStore;
import android.provider.OpenableColumns;
import java.io.ByteArrayOutputStream;

import java.io.IOException;

public class MainActivity extends AppCompatActivity {

    public static String path = null;
    public static byte permission_flag = 0;

    private static String[] PERMISSIONS_STORAGE = {
            Manifest.permission.READ_EXTERNAL_STORAGE,
            Manifest.permission.WRITE_EXTERNAL_STORAGE};
    private static int REQUEST_PERMISSION_CODE = 1;

    private UsbService usbService = null;
    private TextView display = null;
    private MyHandler mHandler = null;
    private OtaThread mThread = null;

    private static DatagramSocket mServerSocket = null;
    private DatagramSocket mClientSocket = null;
    private static final int SOCKET_SERVER_PORT = 3000;
    private static final int SOCKET_CLIENT_PORT = 3001;
    private static final int SOCKET_TIMEOUT_RECV = 3000;
    private byte[] mBuf = new byte[1024];
    private DatagramPacket mDatagramPacket = new DatagramPacket(mBuf, mBuf.length);

    /*
     * Notifications from UsbService will be received here.
     */
    private final BroadcastReceiver mUsbReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            switch (intent.getAction()) {
                case UsbService.ACTION_USB_PERMISSION_GRANTED: // USB PERMISSION GRANTED
                    Toast.makeText(context, "USB Ready", Toast.LENGTH_SHORT).show();
                    break;
                case UsbService.ACTION_USB_PERMISSION_NOT_GRANTED: // USB PERMISSION NOT GRANTED
                    Toast.makeText(context, "USB Permission not granted", Toast.LENGTH_SHORT).show();
                    break;
                case UsbService.ACTION_NO_USB: // NO USB CONNECTED
                    Toast.makeText(context, "No USB connected", Toast.LENGTH_SHORT).show();
                    break;
                case UsbService.ACTION_USB_DISCONNECTED: // USB DISCONNECTED
                    Toast.makeText(context, "USB disconnected", Toast.LENGTH_SHORT).show();
                    break;
                case UsbService.ACTION_USB_NOT_SUPPORTED: // USB NOT SUPPORTED
                    Toast.makeText(context, "USB device not supported", Toast.LENGTH_SHORT).show();
                    break;
            }
        }
    };

    private final ServiceConnection usbConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName arg0, IBinder arg1) {
            usbService = ((UsbService.UsbBinder) arg1).getService();
            usbService.setHandler(mHandler);
        }

        @Override
        public void onServiceDisconnected(ComponentName arg0) {
            usbService = null;
        }
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        if (android.os.Build.VERSION.SDK_INT > 9) {
            StrictMode.ThreadPolicy policy = new StrictMode.ThreadPolicy.Builder().permitAll().build();
            StrictMode.setThreadPolicy(policy);
        }

        try {
            mServerSocket = new DatagramSocket(SOCKET_SERVER_PORT);
            mClientSocket = new DatagramSocket(SOCKET_CLIENT_PORT);
            mClientSocket.setSoTimeout(SOCKET_TIMEOUT_RECV);
        } catch (SocketException e) {
            e.printStackTrace();
        }

        mHandler = new MyHandler(this);
        mThread = new OtaThread();

        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP) {
            if (ActivityCompat.checkSelfPermission(this, Manifest.permission.WRITE_EXTERNAL_STORAGE) != PackageManager.PERMISSION_GRANTED) {
                ActivityCompat.requestPermissions(this, PERMISSIONS_STORAGE, REQUEST_PERMISSION_CODE);
            }
        }

        display = (TextView) findViewById(R.id.textView1);
        Button sendButton = (Button) findViewById(R.id.buttonSend);
        Button selectButton = (Button) findViewById(R.id.buttonSelect);

        sendButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mThread != null && mThread.isAlive()) {
                    display.append("OTA: updating now" + "\r\n");
                }else{
                    mThread = new OtaThread();
                    mThread.start();
                    display.setText("");
                    display.append("OTA: start updating...\r\n");
                }
            }
        });

        selectButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                Intent intent = new Intent(Intent.ACTION_GET_CONTENT);
                intent.addCategory(Intent.CATEGORY_OPENABLE);
                intent.setType("*/*");
                startActivityForResult(intent, 1);
            }
        });
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == REQUEST_PERMISSION_CODE) {
            permission_flag = 1;
        }
    }

    @Override
    protected void onActivityResult(final int requestCode, final int resultCode, final Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (data == null) {
            return;
        }
        path = null;
        Uri uri = data.getData(); // 获取用户选择文件的URI

        path = getFileAbsolutePath(getBaseContext(), uri);



        display.append("Selected: " + path + "\r\n");
        return;
    }

    @Override
    public void onResume() {
        super.onResume();
        setFilters();  // Start listening notifications from UsbService
        startService(UsbService.class, usbConnection, null); // Start UsbService(if it was not started before) and Bind it
    }

    @Override
    public void onPause() {
        super.onPause();
        unregisterReceiver(mUsbReceiver);
        unbindService(usbConnection);
    }

    private void startService(Class<?> service, ServiceConnection serviceConnection, Bundle extras) {
        if (!UsbService.SERVICE_CONNECTED) {
            Intent startService = new Intent(this, service);
            if (extras != null && !extras.isEmpty()) {
                Set<String> keys = extras.keySet();
                for (String key : keys) {
                    String extra = extras.getString(key);
                    startService.putExtra(key, extra);
                }
            }
            startService(startService);
        }
        Intent bindingIntent = new Intent(this, service);
        bindService(bindingIntent, serviceConnection, Context.BIND_AUTO_CREATE);
    }

    private void setFilters() {
        IntentFilter filter = new IntentFilter();
        filter.addAction(UsbService.ACTION_USB_PERMISSION_GRANTED);
        filter.addAction(UsbService.ACTION_NO_USB);
        filter.addAction(UsbService.ACTION_USB_DISCONNECTED);
        filter.addAction(UsbService.ACTION_USB_NOT_SUPPORTED);
        filter.addAction(UsbService.ACTION_USB_PERMISSION_NOT_GRANTED);
        registerReceiver(mUsbReceiver, filter);
    }

    /*
     * This handler will be passed to UsbService. Data received from serial port is displayed through this handler
     */
    private static class MyHandler extends Handler {
        private final WeakReference<MainActivity> mActivity;

        public MyHandler(MainActivity activity) {
            mActivity = new WeakReference<>(activity);
        }

        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
                case UsbService.MESSAGE_FROM_SERIAL_PORT:
                    byte[] data = (byte[])(msg.obj);

                    try {
                        DatagramPacket dp_send = new DatagramPacket(data, data.length, InetAddress.getLocalHost(), SOCKET_CLIENT_PORT);
                        mServerSocket.send(dp_send);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }

                    break;
                case UsbService.CTS_CHANGE:
                    Toast.makeText(mActivity.get(), "CTS_CHANGE",Toast.LENGTH_LONG).show();
                    break;
                case UsbService.DSR_CHANGE:
                    Toast.makeText(mActivity.get(), "DSR_CHANGE",Toast.LENGTH_LONG).show();
                    break;
            }
        }
    }


    private static String getFileAbsolutePath(Context context, Uri imageUri) {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.KITKAT && android.os.Build.VERSION.SDK_INT < Build.VERSION_CODES.Q && DocumentsContract.isDocumentUri(context, imageUri)) {
            if (isExternalStorageDocument(imageUri)) {
                String docId = DocumentsContract.getDocumentId(imageUri);
                String[] split = docId.split(":");
                String type = split[0];
                if ("primary".equalsIgnoreCase(type)) {
                    return Environment.getExternalStorageDirectory() + "/" + split[1];
                }
            } else if (isDownloadsDocument(imageUri)) {
                String id = DocumentsContract.getDocumentId(imageUri);
                Uri contentUri = ContentUris.withAppendedId(Uri.parse("content://downloads/public_downloads"), Long.valueOf(id));
                return getDataColumn(context, contentUri, null, null);
            } else if (isMediaDocument(imageUri)) {
                String docId = DocumentsContract.getDocumentId(imageUri);
                String[] split = docId.split(":");
                String type = split[0];
                Uri contentUri = null;
                if ("image".equals(type)) {
                    contentUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
                } else if ("video".equals(type)) {
                    contentUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI;
                } else if ("audio".equals(type)) {
                    contentUri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
                }
                String selection = MediaStore.Images.Media._ID + "=?";
                String[] selectionArgs = new String[]{split[1]};
                return getDataColumn(context, contentUri, selection, selectionArgs);
            }
        } // MediaStore (and general)
        if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q){
            return uriToFileApiQ(context,imageUri);
        }
        else if ("content".equalsIgnoreCase(imageUri.getScheme())) {
            // Return the remote address
            if (isGooglePhotosUri(imageUri)) {
                return imageUri.getLastPathSegment();
            }
            return getDataColumn(context, imageUri, null, null);
        }
        // File
        else if ("file".equalsIgnoreCase(imageUri.getScheme())) {
            return imageUri.getPath();
        }
        return null;
    }

    private static boolean isExternalStorageDocument(Uri uri) {
        return "com.android.externalstorage.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is DownloadsProvider.
     */
    private static boolean isDownloadsDocument(Uri uri) {
        return "com.android.providers.downloads.documents".equals(uri.getAuthority());
    }

    private static String getDataColumn(Context context, Uri uri, String selection, String[] selectionArgs) {
        Cursor cursor = null;
        String column = MediaStore.Images.Media.DATA;
        String[] projection = {column};
        try {
            cursor = context.getContentResolver().query(uri, projection, selection, selectionArgs, null);
            if (cursor != null && cursor.moveToFirst()) {
                int index = cursor.getColumnIndexOrThrow(column);
                return cursor.getString(index);
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return null;
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is MediaProvider.
     */
    private static boolean isMediaDocument(Uri uri) {
        return "com.android.providers.media.documents".equals(uri.getAuthority());
    }

    /**
     * @param uri The Uri to check.
     * @return Whether the Uri authority is Google Photos.
     */
    private static boolean isGooglePhotosUri(Uri uri) {
        return "com.google.android.apps.photos.content".equals(uri.getAuthority());
    }

    private static String uriToFileApiQ(Context context, Uri uri) {
        File file = null;

        if (uri.getScheme().equals(ContentResolver.SCHEME_FILE)) {
            file = new File(uri.getPath());
        } else if (uri.getScheme().equals(ContentResolver.SCHEME_CONTENT)) {
            ContentResolver contentResolver = context.getContentResolver();
            Cursor cursor = contentResolver.query(uri, null, null, null, null);
            if (cursor.moveToFirst()) {
                String displayName = cursor.getString(cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME));
                try {
                    InputStream is = contentResolver.openInputStream(uri);
                    File cache = new File(context.getExternalCacheDir().getAbsolutePath(), Math.round((Math.random() + 1) * 1000) + displayName);
                    FileOutputStream fos = new FileOutputStream(cache);
                    ByteArrayOutputStream arrayOutputStream = new ByteArrayOutputStream();
                    byte[] buffer = new byte[1024 * 10];
                    while (true) {
                        int len = is.read(buffer);
                        if (len == -1) {
                            break;
                        }
                        arrayOutputStream.write(buffer, 0, len);
                    }
                    arrayOutputStream.close();

                    byte[] data = arrayOutputStream.toByteArray();
                    fos.write(data);

                    file = cache;
                    fos.close();
                    is.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return file.getAbsolutePath();
    }

    private static byte[] toByteArray(String hexString) {
        if (hexString.isEmpty())
            return null;

        hexString = hexString.toLowerCase();
        final byte[] byteArray = new byte[hexString.length() >> 1];
        int index = 0;
        for (int i = 0; i < hexString.length(); i++) {
            if (index  > hexString.length() - 1)
                return byteArray;
            byte highDit = (byte) (Character.digit(hexString.charAt(index), 16) & 0xFF);
            byte lowDit = (byte) (Character.digit(hexString.charAt(index + 1), 16) & 0xFF);
            byteArray[i] = (byte) (highDit << 4 | lowDit);
            index += 2;
        }
        return byteArray;
    }

    private static String bytesToHexString(byte[] src, int offset, int length) {
        StringBuilder builder = new StringBuilder();
        if (src == null || src.length <= 0 || length <= 0) {
            return null;
        }
        String hv;
        for (int i = offset; i < length; i++) {
            hv = Integer.toHexString(src[i] & 0xFF).toUpperCase();
            if (hv.length() < 2) {
                builder.append(0);
            }
            builder.append(hv);
        }

        return builder.toString();
    }

    private class OtaThread extends Thread{
        private final int CMD_SYNC   = 1;
        private final int CMD_JUMP   = 2;
        private final int CMD_FLASH  = 3;
        private final int CMD_ERASE  = 4;
        private final int CMD_VERIFY = 5;
        private final int CMD_REBOOT = 12;

        private final int OTA_BASE_ADDR = 0x0800D000;
        private final int REQ_RETRY_TIMES = 3;

        private void wait_at_response() throws Exception {
            mClientSocket.receive(mDatagramPacket);
            String recvStr = new String(mDatagramPacket.getData(), 0, mDatagramPacket.getLength());

            if(recvStr.compareTo("\r\nOK+SEND\r\n") != 0){
                throw new Exception("Wrong AT response");
            }
        }

        private byte[] wait_response() throws Exception {
            mClientSocket.receive(mDatagramPacket);
            String recvStr = new String(mDatagramPacket.getData(), 0, mDatagramPacket.getLength());

            if(!recvStr.contains("\r\nAT+DATA=0"))
                throw new Exception("Wrong response");

            recvStr = recvStr.replaceAll("\r|\n", "");

            String[] recvSplits = recvStr.split(",");

            byte[] resp_pkt = toByteArray(recvSplits[4]);

            if (resp_pkt[0] != (byte)0xFE || resp_pkt[resp_pkt.length-1] != (byte)0xEF)
                throw new Exception("Wrong response pkt");

            ByteBuffer resp_pkt_buffer = ByteBuffer.wrap(resp_pkt);
            resp_pkt_buffer.order(ByteOrder.LITTLE_ENDIAN);

            byte status = resp_pkt_buffer.get(1);
            short len = resp_pkt_buffer.getShort(2);
            int checksum = resp_pkt_buffer.getInt(4+len);

            CRC32 crc = new CRC32();
            crc.update(resp_pkt_buffer.array(), 0, 4+len);
            if(crc.getValue() != checksum)
                throw new Exception("Response pkt crc error");

            byte[] resp = new byte[1+len];
            resp[0] = status;
            for(int i =0; i<len; i++)
                resp[1+i] = resp_pkt[4+i];

            return resp;
        }

        private void send_request(int cmd, byte[] data) throws Exception {
            ByteBuffer pkt = ByteBuffer.allocate(1024);
            pkt.order(ByteOrder.LITTLE_ENDIAN);
            pkt.put((byte) 0xFE);
            pkt.put((byte) cmd);
            if(data == null || data.length<=0) {
                pkt.putShort((short) 0);
            }else{
                pkt.putShort((short) data.length);
                pkt.put(data);
            }

            CRC32 crc = new CRC32();
            crc.update(pkt.array(), 0, pkt.position());
            pkt.putInt((int) crc.getValue());
            pkt.put((byte)0xEF);

            String pkt_str = bytesToHexString(pkt.array(), 0, pkt.position());
            String at_str = String.format("AT+TX=%d,%s\r\n", pkt.position(), pkt_str);
            Log.i("OTA", at_str);
            byte[] at_bytes = at_str.getBytes();

            usbService.write(at_bytes);
        }

        private void sync() throws Exception {
            int i=0;
            for (i=0; i<REQ_RETRY_TIMES; i++) {
                try {
                    send_request(CMD_SYNC, null);
                    wait_at_response();
                    byte[] resp = wait_response();
                    if (resp[0] != 0)
                        throw new Exception("Sync failed");
                    else
                        break;
                } catch (Exception e){
                    e.printStackTrace();
                }
            }
            if (i==REQ_RETRY_TIMES)
                throw new Exception("Sync timeout");
        }

        private void erase(int addr, int size) throws Exception {
            ByteBuffer data = ByteBuffer.allocate(8);
            data.order(ByteOrder.LITTLE_ENDIAN);
            data.putInt(addr);
            data.putInt(size);

            int i=0;
            for (i=0; i<REQ_RETRY_TIMES; i++) {
                try {
                    send_request(CMD_ERASE, data.array());
                    wait_at_response();
                    byte[] resp = wait_response();
                    if (resp[0] != 0)
                        throw new Exception("Erase failed");
                    else
                        break;
                } catch (Exception e){
                    e.printStackTrace();
                }
            }
            if (i==REQ_RETRY_TIMES)
                throw new Exception("Erase timeout");
        }

        private void flash(int addr, byte[] image_data, int len) throws Exception {
            ByteBuffer data = ByteBuffer.allocate(8 + len);
            data.order(ByteOrder.LITTLE_ENDIAN);
            data.putInt(addr);
            data.putInt(image_data.length);
            data.put(image_data, 0, len);

            int i=0;
            for (i=0; i<REQ_RETRY_TIMES; i++) {
                try {
                    send_request(CMD_FLASH, data.array());
                    wait_at_response();
                    byte[] resp = wait_response();
                    if (resp[0] != 0)
                        throw new Exception("Flash failed");
                    else
                        break;
                } catch (Exception e){
                    e.printStackTrace();
                }
            }
            if (i==REQ_RETRY_TIMES)
                throw new Exception("Flash timeout");
        }

        private void verify(int addr, int size, int checksum) throws Exception {
            ByteBuffer data = ByteBuffer.allocate(12);
            data.order(ByteOrder.LITTLE_ENDIAN);
            data.putInt(addr);
            data.putInt(size);
            data.putInt(checksum);

            int i=0;
            for (i=0; i<REQ_RETRY_TIMES; i++) {
                try {
                    send_request(CMD_VERIFY, data.array());
                    wait_at_response();
                    byte[] resp = wait_response();
                    if (resp[0] != 0)
                        throw new Exception("Verify failed");
                    else
                        break;
                } catch (Exception e){
                    e.printStackTrace();
                }
            }
            if (i==REQ_RETRY_TIMES)
                throw new Exception("Verify timeout");
        }

        private void jump(int addr) throws Exception {
            ByteBuffer data = ByteBuffer.allocate(4);
            data.order(ByteOrder.LITTLE_ENDIAN);
            data.putInt(addr);

            int i=0;
            for (i=0; i<REQ_RETRY_TIMES; i++) {
                try {
                    send_request(CMD_JUMP, data.array());
                    wait_at_response();
                    byte[] resp = wait_response();
                    if (resp[0] != 0)
                        throw new Exception("Jump failed");
                    else
                        break;
                } catch (Exception e){
                    e.printStackTrace();
                }
            }
            if (i==REQ_RETRY_TIMES)
                throw new Exception("Jump timeout");
        }

        private void reboot(int mode) throws Exception {
            ByteBuffer data = ByteBuffer.allocate(4);
            data.order(ByteOrder.LITTLE_ENDIAN);
            data.putInt(mode);

            int i=0;
            for (i=0; i<REQ_RETRY_TIMES; i++) {
                try {
                    send_request(CMD_REBOOT, data.array());
                    wait_at_response();
                    byte[] resp = wait_response();
                    if (resp[0] != 0)
                        throw new Exception("Reboot failed");
                    else
                        break;
                } catch (Exception e){
                    e.printStackTrace();
                }
            }
            if (i==REQ_RETRY_TIMES)
                throw new Exception("Reboot timeout");
        }

        @Override
        public void run() {
            super.run();
            ProgressBar progressBar = (ProgressBar) findViewById(R.id.progressBar4);

            try {
                sync();

                if(path != null) {
                    FileInputStream fis = new FileInputStream(new File(path));
                    int file_size = fis.available();
                    byte[] buf = new byte[200];
                    int len = 0;
                    int total_len = 0;

                    //erase
                    erase(OTA_BASE_ADDR, file_size);

                    //flash
                    while ((len = fis.read(buf)) > 0) {
                        flash((OTA_BASE_ADDR+total_len), buf, len);
                        total_len += len;

                        int progress_bar_value = (total_len * 100) / file_size;
                        progressBar.setProgress(progress_bar_value);
                    }
                    fis.close();

                    //verify
                    CRC32 crc32 = new CRC32();
                    FileInputStream fileinputstream = new FileInputStream(new File(path));
                    CheckedInputStream checkedinputstream = new CheckedInputStream(fileinputstream, crc32);
                    while (checkedinputstream.read() != -1) {
                    }
                    checkedinputstream.close();
                    fileinputstream.close();
                    int checksum = (int)crc32.getValue();
                    verify(OTA_BASE_ADDR, file_size, checksum);

                    //reboot
                    reboot(0);
                    display.append("OTA: done\r\n");
                }else{
                    display.append("ERROR: No update file is selected\r\n");
                }
            } catch (Exception e) {
                e.printStackTrace();
                display.append("ERROR: " + e.toString() + "\r\n");
            }
        }
    }
}