## SDRangel — Copilot instructions

Goal: help an AI coding agent become productive quickly in the SDRangel repo by summarizing the architecture, developer workflows, project conventions and integration points with concrete file references and examples.

- Big picture
  - SDRangel is a Qt-based (Qt5/Qt6 optional) GUI and server frontend for software-defined radio devices. It processes I/Q samples from various SDR hardware, applies signal processing and demodulation/modulation plugins, and exposes a WebAPI for remote control. The interface with hardware is done via "device" plugins. The modulation/demodulation is done via "channel" plugins. Another kind of plugins provide additional features not necessarily depending on the I/Q stream(s) (e.g., rotator control, map display). SDRangel can run as a desktop GUI application or as a headless server exposing a WebAPI.
  Key top-level directories:
    - `app/`, `appbench/`, `appsrv/` — application entry points
    - `sdrbase/` — core application framework shared by app, appsrv, appbench and plugins
    - `sdrgui/` — shared GUI components used by the main app and plugins
    - `sdrsrv/` — shared server components used by appsrv and plugins
    - `settings/` — application settings management
    - `plugins/` — channel, device and feature plugins (primary extension mechanism)
    - `plugins/channelmimo/` — MIMO channel plugins for channels processing multiple synchronized streams in both transmit and receive directions
    - `plugins/channelrx/` — receive channel plugins for channels processing single streams in receive direction
    - `plugins/channeltx/` — transmit channel plugins for channels processing single streams in transmit direction
    - `plugins/feature/` — feature plugins providing additional functionalities not directly related to I/Q stream processing
    - `plugins/samplemimo/` — device plugins for SDR hardware supporting multiple synchronized streams in both transmit and receive directions
    - `plugins/samplesource/` — device plugins for SDR hardware receivers (sources) with single I/Q stream
    - `plugins/samplesink/` — device plugins for SDR hardware transmitters (sinks) with single I/Q stream
    - `httpserver/` — HTTP server implementation for the WebAPI
    - `logging/` — logging framework used by the application and plugins
    - `ft8/` — FT8 modem implementation as a separate module
    - `modemm17/` — M17 modem implementation as a separate module
    - `qrtplib/` — QRTPLib library for RTP protocol used by some modem plugins
    - `wdsp/` — signal processing library used by demod/modem plugins
    - `swagger/` — OpenAPI specs and generated client/server code (used to expose and consume the WebAPI)
    - `scriptsapi/` — Scripts written in Python to interact with the WebAPI
    - `doc/img/` — images used in the documentation
    - `doc/model/` — architecture diagrams and models
- Architecture notes (why things are organized this way)
  - Plugin-first design: features, device drivers and channel implementations are implemented as plugins under `plugins/`. Inspect `plugins/*/*plugin.h` and `*worker` classes for lifecycle and messaging patterns.
  - Core real-time DSP lives in `wdsp/`. Plugins call into WDSP classes (e.g. `wdsp/wcpAGC.*`, `wdsp/fmd.*`) for audio/DSP routines.
  - Swagger/OpenAPI under `swagger/` drives a generated client and server API. The code generator output for Qt lives under `swagger/sdrangel/code/qt5/` and is not meant to be hand-edited.

- Build & developer flows (concrete commands)
  - Preferred build: CMake. Preconfigured presets in `CMakePresets.json`. Typical Linux flow:
    - Configure: `cmake --preset default` (uses `build-default` and local cache variables)
    - Build: `cmake --build --preset default` or `cmake --build build-default -j$(nproc)`
    - Install (optional): `cmake --install build-default --prefix /opt/install/sdrangel`
  - For Qt6 builds use preset `default-qt6` or `default-qt6-windows` for Windows.
  - Notes: presets contain many cache variables pointing to external dependencies (`UHD_DIR`, `SoapySDR`, `LimeSuite`, `MBE_DIR`, etc.). When reproducing CI or a developer environment, set those to installed dependencies or use Docker (there is a separate `sdrangel-docker` project).

- Tests & verification
  - There are no central unit-test runners in the top-level; run small tests by building the relevant plugin or running example programs. Use `build/Makefile` targets produced by CMake.

- Common conventions and patterns
  - Message passing: many subsystems use `util/message.h` derived `Message` classes and `Msg*` naming (see `plugins/*/*plugin.h`). When adding a command or event, follow that pattern.
  - Worker threads: long-running hardware or DSP tasks are implemented as `*worker` classes and often use Qt threads (see `plugins/*/*worker.h`).
  - Generated code: files under `swagger/sdrangel/code/*` are generated by swagger-codegen; do not edit them manually. To regenerate see `swagger/sdrangel/README.md` and start the include HTTP server for includes.

- Integration points & external deps
  - Hardware/backends: plugins interface to external SDKs. Look in `CMakePresets.json` for common dependency variables (e.g., `AIRSPY_DIR`, `LIMESUITE_DIR`, `UHD_DIR`).
  - Web API: OpenAPI spec lives at `swagger/sdrangel/api/swagger/swagger.yaml`. Many plugins expose a WebAPI adapter class `*webapiadapter.cpp` which uses generated SWG classes under `swagger/sdrangel/code/qt5/client/`.
  - Scripts: automation and API helpers live in `scriptsapi/` (examples: `dump.py`, `config.py`). They show how to call the WebAPI.

- Files to inspect for common tasks (examples)
  - Add a new plugin: inspect existing `plugins/channelrx/wdsprx/` and `plugins/samplesource/*` for the plugin lifecycle, settings classes and `readme.md` documentation.
  - Debug audio/DSP: `wdsp/` (e.g., `wcpAGC.cpp`, `fmd.*`) and plugin readmes `plugins/channelrx/wdsprx/readme.md` explain AGC behavior and parameters.
  - Work with WebAPI: `swagger/sdrangel/api/swagger/swagger.yaml` + `scriptsapi/Readme.md` + `swagger/sdrangel/README.md` for generation and usage.

- Quick examples (copyable snippets agents may use)
  - Configure + build (Linux): `cmake --preset default && cmake --build --preset default -j$(nproc)`
  - Regenerate swagger Qt client (developer machine with swagger-codegen): follow `swagger/sdrangel/README.md` and run `/opt/install/swagger/swagger-codegen generate -i api/swagger/swagger.yaml -l qt5cpp -c qt5cpp-config.json -o code/qt5`

- Keep in mind
  - Do not edit generated code under `swagger/.../code/` — change the OpenAPI spec or generator configuration instead.
  - Respect plugin message naming and settings classes (`*settings.h/.cpp`) for backwards compatibility with saved session files.
  - Many device-specific behaviors are documented in `plugins/*/*/readme.md` — use them as primary references for device behavior and configuration.

If anything above is unclear or you need a different focus (for example: test harness, CI, or contribution guidelines), tell me which area to expand and I'll iterate.

## Short additions: CI, Docker, Swagger tips

- CI / reproducible builds
  - There is no single top-level CI config in the repo root — check `cmake/ci/` for CI helper scripts and the `build/` or `build-default/` directories produced by CMake to inspect test/run artifacts.
  - Use `cmake --preset default` on Linux; ensure the various *_DIR cache vars in `CMakePresets.json` are set to installed dependency locations or point to Docker images that provide them.

- Docker
  - The repo references `sdrangel-docker` in the README. For reproducible environments prefer using that project which pre-installs SDKs listed in presets.

- Swagger codegen
  - The OpenAPI spec is at `swagger/sdrangel/api/swagger/swagger.yaml`. Regenerate Qt client with the command (developer machine with generator installed):
    - `/opt/install/swagger/swagger-codegen generate -i api/swagger/swagger.yaml -l qt5cpp -c qt5cpp-config.json -o code/qt5`
  - Do not edit generated files under `swagger/sdrangel/code/*` — change the spec or generator config instead (`swagger/sdrangel/config/`).

## Where to look next (fast open targets)
- `plugins/channelrx/wdsprx/` — canonical receive channel plugin and `readme.md` explaining many signal chain choices
- `wdsp/` — DSP implementations (AGC, filters, demodulators) referenced by many plugins
- `scriptsapi/` — example scripts showing how to call the WebAPI for automation

Please review these additions and tell me if you want me to: (A) expand CI/Docker commands with example Dockerfile and Compose, (B) add a short checklist for contributing patches, or (C) generate quick search shortcuts for common code patterns.
