/******************************************************************************/
// Bullfrog Engine Emulation Library - for use to remake classic games like
// Syndicate Wars, Magic Carpet or Dungeon Keeper.
/******************************************************************************/
/** @file bflib_math.c
 *     Math routines.
 * @par Purpose:
 *     Fast math routines, mostly fixed point.
 * @par Comment:
 *     None.
 * @author   Tomasz Lis
 * @date     24 Jan 2009 - 08 Mar 2009
 * @par  Copying and copyrights:
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 */
/******************************************************************************/
#include "pre_inc.h"
#include "bflib_math.h"

#include "bflib_basics.h"

#ifdef _MSC_VER
#include <intrin.h>
#endif // _MSC_VER

#ifdef FUNCTESTING
  #include "ftests/ftest.h"
#endif
#include <math.h>
#include "post_inc.h"

/******************************************************************************/
static const int lbSinTable[2048];
static const int lbCosTable[2048];

unsigned short const lbSqrTable[] = {
   0x0001, 0x0002, 0x0002, 0x0004, 0x0005, 0x0008, 0x000B, 0x0010,
   0x0016, 0x0020, 0x002D, 0x0040, 0x005A, 0x0080, 0x00B5, 0x0100,
   0x016A, 0x0200, 0x02D4, 0x0400, 0x05A8, 0x0800, 0x0B50, 0x1000,
   0x16A0, 0x2000, 0x2D41, 0x4000, 0x5A82, 0x8000, 0xB504, 0xFFFF,};

unsigned short const lbArcTanFactors[] = {
     0,  1,  2,  3,  5,  6,  7,  8, 10, 11, 12, 13, 15, 16, 17, 19,
    20, 21, 22, 24, 25, 26, 27, 29, 30, 31, 32, 34, 35, 36, 38, 39,
    40, 41, 43, 44, 45, 46, 48, 49, 50, 51, 53, 54, 55, 56, 57, 59,
    60, 61, 62, 64, 65, 66, 67, 68, 70, 71, 72, 73, 75, 76, 77, 78,
    79, 81, 82, 83, 84, 85, 87, 88, 89, 90, 91, 92, 94, 95, 96, 97,
    98, 99,101,102,103,104,105,106,107,109,110,111,112,113,114,115,
   116,118,119,120,121,122,123,124,125,126,127,129,130,131,132,133,
   134,135,136,137,138,139,140,141,142,143,144,145,147,148,149,150,
   151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,
   167,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,
   182,182,183,184,185,186,187,188,189,190,191,191,192,193,194,195,
   196,197,198,198,199,200,201,202,203,203,204,205,206,207,208,208,
   209,210,211,212,212,213,214,215,216,216,217,218,219,220,220,221,
   222,223,223,224,225,226,226,227,228,229,229,230,231,232,232,233,
   234,235,235,236,237,237,238,239,239,240,241,242,242,243,244,244,
   245,246,246,247,248,248,249,250,250,251,252,252,253,254,254,255,
   256 };

const struct Proportion proportions[] = {
    {-256, 11585},
    {-255, 11563},
    {-255, 11540},
    {-254, 11518},
    {-253, 11495},
    {-253, 11473},
    {-252, 11450},
    {-251, 11428},
    {-251, 11406},
    {-250, 11383},
    {-250, 11361},
    {-249, 11339},
    {-248, 11317},
    {-248, 11295},
    {-247, 11273},
    {-246, 11251},
    {-245, 11229},
    {-245, 11207},
    {-244, 11185},
    {-243, 11164},
    {-243, 11142},
    {-242, 11120},
    {-241, 11099},
    {-241, 11077},
    {-240, 11056},
    {-239, 11034},
    {-239, 11013},
    {-238, 10991},
    {-237, 10970},
    {-236, 10949},
    {-236, 10928},
    {-235, 10906},
    {-234, 10885},
    {-234, 10864},
    {-233, 10843},
    {-232, 10822},
    {-231, 10801},
    {-231, 10781},
    {-230, 10760},
    {-229, 10739},
    {-228, 10718},
    {-228, 10698},
    {-227, 10677},
    {-226, 10657},
    {-225, 10636},
    {-225, 10616},
    {-224, 10596},
    {-223, 10575},
    {-222, 10555},
    {-222, 10535},
    {-221, 10515},
    {-220, 10495},
    {-219, 10475},
    {-219, 10455},
    {-218, 10435},
    {-217, 10415},
    {-216, 10396},
    {-215, 10376},
    {-215, 10356},
    {-214, 10337},
    {-213, 10317},
    {-212, 10298},
    {-211, 10279},
    {-211, 10259},
    {-210, 10240},
    {-209, 10221},
    {-208, 10202},
    {-207, 10183},
    {-206, 10164},
    {-206, 10145},
    {-205, 10126},
    {-204, 10107},
    {-203, 10088},
    {-202, 10070},
    {-201, 10051},
    {-201, 10033},
    {-200, 10014},
    {-199, 9996},
    {-198, 9978},
    {-197, 9959},
    {-196, 9941},
    {-195, 9923},
    {-195, 9905},
    {-194, 9887},
    {-193, 9869},
    {-192, 9851},
    {-191, 9834},
    {-190, 9816},
    {-189, 9798},
    {-188, 9781},
    {-188, 9764},
    {-187, 9746},
    {-186, 9729},
    {-185, 9712},
    {-184, 9694},
    {-183, 9677},
    {-182, 9660},
    {-181, 9643},
    {-180, 9627},
    {-179, 9610},
    {-178, 9593},
    {-177, 9577},
    {-177, 9560},
    {-176, 9544},
    {-175, 9527},
    {-174, 9511},
    {-173, 9495},
    {-172, 9479},
    {-171, 9462},
    {-170, 9447},
    {-169, 9431},
    {-168, 9415},
    {-167, 9399},
    {-166, 9383},
    {-165, 9368},
    {-164, 9352},
    {-163, 9337},
    {-162, 9322},
    {-161, 9306},
    {-160, 9291},
    {-159, 9276},
    {-158, 9261},
    {-157, 9246},
    {-156, 9232},
    {-155, 9217},
    {-154, 9202},
    {-153, 9188},
    {-152, 9173},
    {-151, 9159},
    {-150, 9145},
    {-149, 9130},
    {-148, 9116},
    {-147, 9102},
    {-146, 9089},
    {-145, 9075},
    {-144, 9061},
    {-143, 9047},
    {-142, 9034},
    {-141, 9020},
    {-140, 9007},
    {-139, 8994},
    {-138, 8981},
    {-137, 8968},
    {-135, 8955},
    {-134, 8942},
    {-133, 8929},
    {-132, 8916},
    {-131, 8904},
    {-130, 8891},
    {-129, 8879},
    {-128, 8866},
    {-127, 8854},
    {-126, 8842},
    {-125, 8830},
    {-124, 8818},
    {-122, 8807},
    {-121, 8795},
    {-120, 8783},
    {-119, 8772},
    {-118, 8760},
    {-117, 8749},
    {-116, 8738},
    {-115, 8727},
    {-114, 8716},
    {-112, 8705},
    {-111, 8694},
    {-110, 8684},
    {-109, 8673},
    {-108, 8662},
    {-107, 8652},
    {-106, 8642},
    {-104, 8632},
    {-103, 8622},
    {-102, 8612},
    {-101, 8602},
    {-100, 8592},
    {-99, 8583},
    {-98, 8573},
    {-96, 8564},
    {-95, 8555},
    {-94, 8545},
    {-93, 8536},
    {-92, 8527},
    {-91, 8519},
    {-89, 8510},
    {-88, 8501},
    {-87, 8493},
    {-86, 8484},
    {-85, 8476},
    {-83, 8468},
    {-82, 8460},
    {-81, 8452},
    {-80, 8444},
    {-79, 8436},
    {-77, 8429},
    {-76, 8421},
    {-75, 8414},
    {-74, 8407},
    {-73, 8400},
    {-71, 8393},
    {-70, 8386},
    {-69, 8379},
    {-68, 8372},
    {-67, 8366},
    {-65, 8359},
    {-64, 8353},
    {-63, 8347},
    {-62, 8341},
    {-60, 8335},
    {-59, 8329},
    {-58, 8323},
    {-57, 8318},
    {-55, 8312},
    {-54, 8307},
    {-53, 8302},
    {-52, 8296},
    {-51, 8291},
    {-49, 8287},
    {-48, 8282},
    {-47, 8277},
    {-46, 8273},
    {-44, 8268},
    {-43, 8264},
    {-42, 8260},
    {-41, 8256},
    {-39, 8252},
    {-38, 8248},
    {-37, 8244},
    {-36, 8241},
    {-34, 8237},
    {-33, 8234},
    {-32, 8231},
    {-30, 8228},
    {-29, 8225},
    {-28, 8222},
    {-27, 8220},
    {-25, 8217},
    {-24, 8215},
    {-23, 8212},
    {-22, 8210},
    {-20, 8208},
    {-19, 8206},
    {-18, 8204},
    {-17, 8203},
    {-15, 8201},
    {-14, 8200},
    {-13, 8198},
    {-11, 8197},
    {-10, 8196},
    { -9, 8195},
    { -8, 8194},
    { -6, 8194},
    { -5, 8193},
    { -4, 8193},
    { -3, 8192},
    { -1, 8192},
    {  0, 8192},
    {  1, 8192},
    {  3, 8192},
    {  4, 8193},
    {  5, 8193},
    {  6, 8194},
    {  8, 8194},
    {  9, 8195},
    { 10, 8196},
    { 11, 8197},
    { 13, 8198},
    { 14, 8200},
    { 15, 8201},
    { 17, 8203},
    { 18, 8204},
    { 19, 8206},
    { 20, 8208},
    { 22, 8210},
    { 23, 8212},
    { 24, 8215},
    { 25, 8217},
    { 27, 8220},
    { 28, 8222},
    { 29, 8225},
    { 30, 8228},
    { 32, 8231},
    { 33, 8234},
    { 34, 8237},
    { 36, 8241},
    { 37, 8244},
    { 38, 8248},
    { 39, 8252},
    { 41, 8256},
    { 42, 8260},
    { 43, 8264},
    { 44, 8268},
    { 46, 8273},
    { 47, 8277},
    { 48, 8282},
    { 49, 8287},
    { 51, 8291},
    { 52, 8296},
    { 53, 8302},
    { 54, 8307},
    { 55, 8312},
    { 57, 8318},
    { 58, 8323},
    { 59, 8329},
    { 60, 8335},
    { 62, 8341},
    { 63, 8347},
    { 64, 8353},
    { 65, 8359},
    { 67, 8366},
    { 68, 8372},
    { 69, 8379},
    { 70, 8386},
    { 71, 8393},
    { 73, 8400},
    { 74, 8407},
    { 75, 8414},
    { 76, 8421},
    { 77, 8429},
    { 79, 8436},
    { 80, 8444},
    { 81, 8452},
    { 82, 8460},
    { 83, 8468},
    { 85, 8476},
    { 86, 8484},
    { 87, 8493},
    { 88, 8501},
    { 89, 8510},
    { 91, 8519},
    { 92, 8527},
    { 93, 8536},
    { 94, 8545},
    { 95, 8555},
    { 96, 8564},
    { 98, 8573},
    { 99, 8583},
    {100, 8592},
    {101, 8602},
    {102, 8612},
    {103, 8622},
    {104, 8632},
    {106, 8642},
    {107, 8652},
    {108, 8662},
    {109, 8673},
    {110, 8684},
    {111, 8694},
    {112, 8705},
    {114, 8716},
    {115, 8727},
    {116, 8738},
    {117, 8749},
    {118, 8760},
    {119, 8772},
    {120, 8783},
    {121, 8795},
    {122, 8807},
    {124, 8818},
    {125, 8830},
    {126, 8842},
    {127, 8854},
    {128, 8866},
    {129, 8879},
    {130, 8891},
    {131, 8904},
    {132, 8916},
    {133, 8929},
    {134, 8942},
    {135, 8955},
    {137, 8968},
    {138, 8981},
    {139, 8994},
    {140, 9007},
    {141, 9020},
    {142, 9034},
    {143, 9047},
    {144, 9061},
    {145, 9075},
    {146, 9089},
    {147, 9102},
    {148, 9116},
    {149, 9130},
    {150, 9145},
    {151, 9159},
    {152, 9173},
    {153, 9188},
    {154, 9202},
    {155, 9217},
    {156, 9232},
    {157, 9246},
    {158, 9261},
    {159, 9276},
    {160, 9291},
    {161, 9306},
    {162, 9322},
    {163, 9337},
    {164, 9352},
    {165, 9368},
    {166, 9383},
    {167, 9399},
    {168, 9415},
    {169, 9431},
    {170, 9447},
    {171, 9462},
    {172, 9479},
    {173, 9495},
    {174, 9511},
    {175, 9527},
    {176, 9544},
    {177, 9560},
    {177, 9577},
    {178, 9593},
    {179, 9610},
    {180, 9627},
    {181, 9643},
    {182, 9660},
    {183, 9677},
    {184, 9694},
    {185, 9712},
    {186, 9729},
    {187, 9746},
    {188, 9764},
    {188, 9781},
    {189, 9798},
    {190, 9816},
    {191, 9834},
    {192, 9851},
    {193, 9869},
    {194, 9887},
    {195, 9905},
    {195, 9923},
    {196, 9941},
    {197, 9959},
    {198, 9978},
    {199, 9996},
    {200, 10014},
    {201, 10033},
    {201, 10051},
    {202, 10070},
    {203, 10088},
    {204, 10107},
    {205, 10126},
    {206, 10145},
    {206, 10164},
    {207, 10183},
    {208, 10202},
    {209, 10221},
    {210, 10240},
    {211, 10259},
    {211, 10279},
    {212, 10298},
    {213, 10317},
    {214, 10337},
    {215, 10356},
    {215, 10376},
    {216, 10396},
    {217, 10415},
    {218, 10435},
    {219, 10455},
    {219, 10475},
    {220, 10495},
    {221, 10515},
    {222, 10535},
    {222, 10555},
    {223, 10575},
    {224, 10596},
    {225, 10616},
    {225, 10636},
    {226, 10657},
    {227, 10677},
    {228, 10698},
    {228, 10718},
    {229, 10739},
    {230, 10760},
    {231, 10781},
    {231, 10801},
    {232, 10822},
    {233, 10843},
    {234, 10864},
    {234, 10885},
    {235, 10906},
    {236, 10928},
    {236, 10949},
    {237, 10970},
    {238, 10991},
    {239, 11013},
    {239, 11034},
    {240, 11056},
    {241, 11077},
    {241, 11099},
    {242, 11120},
    {243, 11142},
    {243, 11164},
    {244, 11185},
    {245, 11207},
    {245, 11229},
    {246, 11251},
    {247, 11273},
    {248, 11295},
    {248, 11317},
    {249, 11339},
    {250, 11361},
    {250, 11383},
    {251, 11406},
    {251, 11428},
    {252, 11450},
    {253, 11473},
    {253, 11495},
    {254, 11518},
    {255, 11540},
    {255, 11563},
    {256, 11585},
};
/******************************************************************************/
/**
 * Gives sinus of given angle.
 * @param x Angle as integer with reference to DEGREES_180.
 * @return Value ranged -65536 to 65536.
 */
long LbSinL(long x)
{
    return lbSinTable[(unsigned long)x & ANGLE_MASK];
}

/**
 * Gives cosinus of given angle.
 * @param x Angle as integer with reference to DEGREES_180.
 * @return Value ranged -65536 to 65536.
 */
long LbCosL(long x)
{
    return lbCosTable[(unsigned long)x & ANGLE_MASK];
}

/** Computes angle between negative Y axis and the line that crosses (0,0) and given (x,y).
 *  Uses arctan(x/y) with proper shift to get the angle.
 *  Returning 0 means direction towards negative y; 512 is towards positive x;
 *  1024 towards positive y, and 1536 towards negative x. Output range is between 0 (=0 rad)
 *  and 2048 (=2*pi rad), zero included.
 *  Value of the angle is properly rounded, up or down.
 *
 * @param x
 * @param y
 * @return
 */
int32_t LbArcTanAngle(int32_t x,int32_t y)
{
    int32_t ux;
    int32_t uy;
    uint32_t index;
    if ((x == 0) && (y == 0))
        return 0;
    if (x < 0)
    {
        ux = -x;
        if (y < 0)
        {
            uy = -y;
            // Make sure we'll have smaller value * 256 / greater value.
            // This way we won't exceed factors array bounds (which is 256 elements).
            if (ux < uy) {
                index = (ux << 8)/uy;
                return DEGREES_360   - (int32_t)lbArcTanFactors[index];
            } else {
                index = (uy << 8)/ux;
                return ANGLE_WEST + (int32_t)lbArcTanFactors[index];
            }
        } else
        {
            uy = y;
            // Make sure we'll have smaller value * 256 / greater value.
            if (ux < uy) {
                index = (ux << 8)/uy;
                return   DEGREES_180   + (int32_t)lbArcTanFactors[index];
            } else {
                index = (uy << 8)/ux;
                return ANGLE_WEST - (int32_t)lbArcTanFactors[index];
            }
        }
    } else
    {
        ux = x;
        if (y < 0)
        {
            uy = -y;
            // Make sure we'll have smaller value * 256 / greater value.
            if (ux < uy) {
                index = (ux << 8)/uy;
                return                 (int32_t)lbArcTanFactors[index];
            } else {
                index = (uy << 8)/ux;
                return ANGLE_EAST - (int32_t)lbArcTanFactors[index];
            }
        } else
        {
            uy = y;
            // Make sure we'll have smaller value * 256 / greater value.
            if (ux < uy) {
                index = (ux << 8)/uy;
                return DEGREES_180   - (int32_t)lbArcTanFactors[index];
            } else {
                index = (uy << 8)/ux;
                return ANGLE_EAST + (int32_t)lbArcTanFactors[index];
            }
        }
    }
}

static long bitScanReverse(long s)
{
  unsigned long source = (unsigned long)s;
#if defined(_MSC_VER)
    DWORD i;
    uint8_t success = _BitScanReverse(&i, source);
    return success != 0 ? i : -1;
#elif defined(__GNUC__)
    int result = source == 0 ? -1 : __builtin_clz(source) ^ 31;
    return result;
#else
#pragma message "Falling back to iterative bitscan reverse, consider using intrinsics"
    if (source != 0)
    {
        for (int32_t i = 31; i > -1; i--)
        {
            if (source & (1u << i))
            {
                return i;
            }
        }
    }
    return -1;
#endif
}

long LbSqrL(long x)
{
  if (x <= 0)
    return 0;
  //
  long y = bitScanReverse(x);
  y = lbSqrTable[y];
  while ((x/y) < y)
    y = ((x/y) + y) >> 1;
  return y;
}

long LbMathOperation(unsigned char opkind, long first_operand, long second_operand)
{
  switch (opkind)
  {
    case MOp_EQUAL:
      return first_operand == second_operand;
    case MOp_NOT_EQUAL:
      return first_operand != second_operand;
    case MOp_SMALLER:
      return first_operand < second_operand;
    case MOp_GREATER:
      return first_operand > second_operand;
    case MOp_SMALLER_EQ:
      return first_operand <= second_operand;
    case MOp_GREATER_EQ:
      return first_operand >= second_operand;
    case MOp_LOGIC_AND:
      return first_operand && second_operand;
    case MOp_LOGIC_OR:
      return first_operand || second_operand;
    case MOp_LOGIC_XOR:
      return (first_operand!=0) ^ (second_operand!=0);
    case MOp_BITWS_AND:
      return first_operand & second_operand;
    case MOp_BITWS_OR:
      return first_operand | second_operand;
    case MOp_BITWS_XOR:
      return first_operand ^ second_operand;
    case MOp_SUM:
      return first_operand + second_operand;
    case MOp_SUBTRACT:
      return first_operand - second_operand;
    case MOp_MULTIPLY:
      return first_operand * second_operand;
    case MOp_DIVIDE:
      return first_operand / second_operand;
    case MOp_MODULO:
      return first_operand % second_operand;
    default:
      return first_operand;
  }
}

unsigned long LbRandomSeries(unsigned long range, uint32_t *seed, const char *func_name, unsigned long place)
{
  if (range == 0)
    return 0;
  unsigned long i = 9377 * (*seed) + 9439;
#ifndef FUNCTESTING // don't modify seeds when functional testing is enabled
  *seed = (i >> 13) | (i << ((sizeof(int32_t) * 8) - 13));
#endif // FUNCTESTING
  i = (*seed) % range;
  return i;
}

TbBool LbNumberSignsSame(long num_a, long num_b)
{
    int sign_a;
    int sign_b;
    if (num_a >= 0)
        sign_a = (num_a != 0);
    else
        sign_a = -1;
    if (num_b >= 0)
        sign_b = (num_b != 0);
    else
        sign_b = -1;
    return (sign_a == sign_b);
}

char LbCompareMultiplications(long mul1a, long mul1b, long mul2a, long mul2b)
{
    long long mul1 = (long long)mul1a * (long long)mul1b;
    long long mul2 = (long long)mul2a * (long long)mul2b;
    if (mul1 > mul2)
        return 1;
    if (mul1 < mul2)
        return -1;
    return 0;
}

/**
 *  Optimized function for computing vector length.
 *  Returns diagonal of a rectangle with sides of given length.
 * @param a Length of one of the sides.
 * @param b Length of second of the sides.
 * @return Root of sum of squares of given lengths, sqrt(a*a + b*b).
 */
long LbDiagonalLength(long a, long b)
{
    int propidx;
    long long tmpval;
    if (a > b) {
        if (a == 0) {
            return 0; // don't divide by zero
        }
        propidx = (b << 8)/a;
        tmpval = a;
    } else {
        if (b == 0) {
            return 0; // Just to avoid dividing by 0
        }
        propidx = (a << 8)/b;
        tmpval = b;
    }
    tmpval = tmpval * (long long)proportions[propidx + 256].distance_ratio;
    return (tmpval >> 13);
}

float LbLerp(float low, float high, float interval)
{
    return (low * (1.0 - interval)) + (high * interval);
}

float LbFmodf(float x, float y)
{
    float result = x - ((int)(x / y)) * y;
    if (result < 0) {
        result += y;
    }
    return result;
}

float lerp_angle(float from, float to, float weight) {
    float angle_difference = to - from;
    float wrapped_difference = LbFmodf(angle_difference + DEGREES_180, DEGREES_360);
    float shortest_distance = wrapped_difference - DEGREES_180;
    float step = shortest_distance * weight;
    float result = LbFmodf(from + step, DEGREES_360);
    return result;
}

double fastPow(double a, double b)
{
  union {
    double d;
    int x[2];
  } u = { a };
  u.x[1] = (int)(b * (u.x[1] - 1072632447) + 1072632447);
  u.x[0] = 0;
  return u.d;
}
/******************************************************************************/

static const int lbSinTable[2048] = {
    0,     201,   402,   603,   804,   1005,  1206,  1407,  1608,  1809,  2010,
    2211,  2412,  2613,  2814,  3014,  3215,  3416,  3617,  3818,  4018,  4219,
    4420,  4620,  4821,  5021,  5222,  5422,  5622,  5823,  6023,  6223,  6423,
    6623,  6823,  7023,  7223,  7423,  7623,  7822,  8022,  8221,  8421,  8620,
    8819,  9019,  9218,  9417,  9616,  9814,  10013, 10212, 10410, 10609, 10807,
    11006, 11204, 11402, 11600, 11797, 11995, 12193, 12390, 12588, 12785, 12982,
    13179, 13376, 13573, 13769, 13966, 14162, 14359, 14555, 14751, 14946, 15142,
    15338, 15533, 15728, 15923, 16118, 16313, 16508, 16702, 16897, 17091, 17285,
    17479, 17672, 17866, 18059, 18253, 18446, 18638, 18831, 19024, 19216, 19408,
    19600, 19792, 19983, 20175, 20366, 20557, 20748, 20938, 21129, 21319, 21509,
    21699, 21889, 22078, 22267, 22456, 22645, 22833, 23022, 23210, 23398, 23586,
    23773, 23960, 24147, 24334, 24521, 24707, 24893, 25079, 25265, 25450, 25635,
    25820, 26005, 26189, 26373, 26557, 26741, 26925, 27108, 27291, 27473, 27656,
    27838, 28020, 28201, 28383, 28564, 28745, 28925, 29105, 29285, 29465, 29645,
    29824, 30003, 30181, 30360, 30538, 30715, 30893, 31070, 31247, 31424, 31600,
    31776, 31952, 32127, 32302, 32477, 32651, 32826, 32999, 33173, 33346, 33519,
    33692, 33864, 34036, 34208, 34379, 34550, 34721, 34891, 35061, 35231, 35400,
    35569, 35738, 35906, 36074, 36242, 36409, 36576, 36743, 36909, 37075, 37241,
    37406, 37571, 37736, 37900, 38064, 38227, 38390, 38553, 38716, 38878, 39039,
    39201, 39362, 39522, 39682, 39842, 40002, 40161, 40319, 40478, 40636, 40793,
    40950, 41107, 41263, 41419, 41575, 41730, 41885, 42040, 42194, 42347, 42501,
    42653, 42806, 42958, 43110, 43261, 43412, 43562, 43712, 43862, 44011, 44160,
    44308, 44456, 44603, 44750, 44897, 45043, 45189, 45335, 45480, 45624, 45768,
    45912, 46055, 46198, 46340, 46482, 46624, 46765, 46906, 47046, 47186, 47325,
    47464, 47602, 47740, 47878, 48015, 48151, 48288, 48423, 48558, 48693, 48828,
    48961, 49095, 49228, 49360, 49492, 49624, 49755, 49886, 50016, 50146, 50275,
    50403, 50532, 50660, 50787, 50914, 51040, 51166, 51291, 51416, 51541, 51665,
    51788, 51911, 52033, 52155, 52277, 52398, 52518, 52639, 52758, 52877, 52996,
    53114, 53231, 53348, 53465, 53581, 53696, 53811, 53926, 54040, 54153, 54266,
    54379, 54491, 54602, 54713, 54823, 54933, 55043, 55152, 55260, 55368, 55475,
    55582, 55688, 55794, 55899, 56004, 56108, 56212, 56315, 56417, 56519, 56621,
    56722, 56822, 56922, 57022, 57120, 57219, 57316, 57414, 57510, 57606, 57702,
    57797, 57892, 57986, 58079, 58172, 58264, 58356, 58447, 58538, 58628, 58718,
    58807, 58895, 58983, 59070, 59157, 59243, 59329, 59414, 59499, 59583, 59666,
    59749, 59831, 59913, 59994, 60075, 60155, 60235, 60313, 60392, 60470, 60547,
    60624, 60700, 60775, 60850, 60924, 60998, 61071, 61144, 61216, 61288, 61359,
    61429, 61499, 61568, 61637, 61705, 61772, 61839, 61905, 61971, 62036, 62100,
    62164, 62228, 62291, 62353, 62414, 62475, 62536, 62596, 62655, 62714, 62772,
    62829, 62886, 62942, 62998, 63053, 63108, 63162, 63215, 63268, 63320, 63371,
    63422, 63473, 63522, 63571, 63620, 63668, 63715, 63762, 63808, 63854, 63899,
    63943, 63987, 64030, 64073, 64115, 64156, 64197, 64237, 64276, 64315, 64353,
    64391, 64428, 64465, 64501, 64536, 64571, 64605, 64638, 64671, 64703, 64735,
    64766, 64796, 64826, 64855, 64884, 64912, 64939, 64966, 64992, 65018, 65043,
    65067, 65091, 65114, 65136, 65158, 65179, 65200, 65220, 65239, 65258, 65276,
    65294, 65311, 65327, 65343, 65358, 65372, 65386, 65400, 65412, 65424, 65436,
    65446, 65457, 65466, 65475, 65483, 65491, 65498, 65505, 65511, 65516, 65520,
    65524, 65528, 65531, 65533, 65534, 65535,
    65536,  65535,  65534,  65533,  65531,  65528,  65524,  65520,  65516,
    65511,  65505,  65498,  65491,  65483,  65475,  65466,  65457,  65446,
    65436,  65424,  65412,  65400,  65386,  65372,  65358,  65343,  65327,
    65311,  65294,  65276,  65258,  65239,  65220,  65200,  65179,  65158,
    65136,  65114,  65091,  65067,  65043,  65018,  64992,  64966,  64939,
    64912,  64884,  64855,  64826,  64796,  64766,  64735,  64703,  64671,
    64638,  64605,  64571,  64536,  64501,  64465,  64428,  64391,  64353,
    64315,  64276,  64237,  64197,  64156,  64115,  64073,  64030,  63987,
    63943,  63899,  63854,  63808,  63762,  63715,  63668,  63620,  63571,
    63522,  63473,  63422,  63371,  63320,  63268,  63215,  63162,  63108,
    63053,  62998,  62942,  62886,  62829,  62772,  62714,  62655,  62596,
    62536,  62475,  62414,  62353,  62291,  62228,  62164,  62100,  62036,
    61971,  61905,  61839,  61772,  61705,  61637,  61568,  61499,  61429,
    61359,  61288,  61216,  61144,  61071,  60998,  60924,  60850,  60775,
    60700,  60624,  60547,  60470,  60392,  60313,  60235,  60155,  60075,
    59994,  59913,  59831,  59749,  59666,  59583,  59499,  59414,  59329,
    59243,  59157,  59070,  58983,  58895,  58807,  58718,  58628,  58538,
    58447,  58356,  58264,  58172,  58079,  57986,  57892,  57797,  57702,
    57606,  57510,  57414,  57316,  57219,  57120,  57022,  56922,  56822,
    56722,  56621,  56519,  56417,  56315,  56212,  56108,  56004,  55899,
    55794,  55688,  55582,  55475,  55368,  55260,  55152,  55043,  54933,
    54823,  54713,  54602,  54491,  54379,  54266,  54153,  54040,  53926,
    53811,  53696,  53581,  53465,  53348,  53231,  53114,  52996,  52877,
    52758,  52639,  52518,  52398,  52277,  52155,  52033,  51911,  51788,
    51665,  51541,  51416,  51291,  51166,  51040,  50914,  50787,  50660,
    50532,  50403,  50275,  50146,  50016,  49886,  49755,  49624,  49492,
    49360,  49228,  49095,  48961,  48828,  48693,  48558,  48423,  48288,
    48151,  48015,  47878,  47740,  47602,  47464,  47325,  47186,  47046,
    46906,  46765,  46624,  46482,  46340,  46198,  46055,  45912,  45768,
    45624,  45480,  45335,  45189,  45043,  44897,  44750,  44603,  44456,
    44308,  44160,  44011,  43862,  43712,  43562,  43412,  43261,  43110,
    42958,  42806,  42653,  42501,  42347,  42194,  42040,  41885,  41730,
    41575,  41419,  41263,  41107,  40950,  40793,  40636,  40478,  40319,
    40161,  40002,  39842,  39682,  39522,  39362,  39201,  39039,  38878,
    38716,  38553,  38390,  38227,  38064,  37900,  37736,  37571,  37406,
    37241,  37075,  36909,  36743,  36576,  36409,  36242,  36074,  35906,
    35738,  35569,  35400,  35231,  35061,  34891,  34721,  34550,  34379,
    34208,  34036,  33864,  33692,  33519,  33346,  33173,  32999,  32826,
    32651,  32477,  32302,  32127,  31952,  31776,  31600,  31424,  31247,
    31070,  30893,  30715,  30538,  30360,  30181,  30003,  29824,  29645,
    29465,  29285,  29105,  28925,  28745,  28564,  28383,  28201,  28020,
    27838,  27656,  27473,  27291,  27108,  26925,  26741,  26557,  26373,
    26189,  26005,  25820,  25635,  25450,  25265,  25079,  24893,  24707,
    24521,  24334,  24147,  23960,  23773,  23586,  23398,  23210,  23022,
    22833,  22645,  22456,  22267,  22078,  21889,  21699,  21509,  21319,
    21129,  20938,  20748,  20557,  20366,  20175,  19983,  19792,  19600,
    19408,  19216,  19024,  18831,  18638,  18446,  18253,  18059,  17866,
    17672,  17479,  17285,  17091,  16897,  16702,  16508,  16313,  16118,
    15923,  15728,  15533,  15338,  15142,  14946,  14751,  14555,  14359,
    14162,  13966,  13769,  13573,  13376,  13179,  12982,  12785,  12588,
    12390,  12193,  11995,  11797,  11600,  11402,  11204,  11006,  10807,
    10609,  10410,  10212,  10013,  9814,   9616,   9417,   9218,   9019,
    8819,   8620,   8421,   8221,   8022,   7822,   7623,   7423,   7223,
    7023,   6823,   6623,   6423,   6223,   6023,   5823,   5622,   5422,
    5222,   5021,   4821,   4620,   4420,   4219,   4018,   3818,   3617,
    3416,   3215,   3014,   2814,   2613,   2412,   2211,   2010,   1809,
    1608,   1407,   1206,   1005,   804,    603,    402,    201,    0,
    -201,   -402,   -603,   -804,   -1005,  -1206,  -1407,  -1608,  -1809,
    -2010,  -2211,  -2412,  -2613,  -2814,  -3014,  -3215,  -3416,  -3617,
    -3818,  -4018,  -4219,  -4420,  -4620,  -4821,  -5021,  -5222,  -5422,
    -5622,  -5823,  -6023,  -6223,  -6423,  -6623,  -6823,  -7023,  -7223,
    -7423,  -7623,  -7822,  -8022,  -8221,  -8421,  -8620,  -8819,  -9019,
    -9218,  -9417,  -9616,  -9814,  -10013, -10212, -10410, -10609, -10807,
    -11006, -11204, -11402, -11600, -11797, -11995, -12193, -12390, -12588,
    -12785, -12982, -13179, -13376, -13573, -13769, -13966, -14162, -14359,
    -14555, -14751, -14946, -15142, -15338, -15533, -15728, -15923, -16118,
    -16313, -16508, -16702, -16897, -17091, -17285, -17479, -17672, -17866,
    -18059, -18253, -18446, -18638, -18831, -19024, -19216, -19408, -19600,
    -19792, -19983, -20175, -20366, -20557, -20748, -20938, -21129, -21319,
    -21509, -21699, -21889, -22078, -22267, -22456, -22645, -22833, -23022,
    -23210, -23398, -23586, -23773, -23960, -24147, -24334, -24521, -24707,
    -24893, -25079, -25265, -25450, -25635, -25820, -26005, -26189, -26373,
    -26557, -26741, -26925, -27108, -27291, -27473, -27656, -27838, -28020,
    -28201, -28383, -28564, -28745, -28925, -29105, -29285, -29465, -29645,
    -29824, -30003, -30181, -30360, -30538, -30715, -30893, -31070, -31247,
    -31424, -31600, -31776, -31952, -32127, -32302, -32477, -32651, -32826,
    -32999, -33173, -33346, -33519, -33692, -33864, -34036, -34208, -34379,
    -34550, -34721, -34891, -35061, -35231, -35400, -35569, -35738, -35906,
    -36074, -36242, -36409, -36576, -36743, -36909, -37075, -37241, -37406,
    -37571, -37736, -37900, -38064, -38227, -38390, -38553, -38716, -38878,
    -39039, -39201, -39362, -39522, -39682, -39842, -40002, -40161, -40319,
    -40478, -40636, -40793, -40950, -41107, -41263, -41419, -41575, -41730,
    -41885, -42040, -42194, -42347, -42501, -42653, -42806, -42958, -43110,
    -43261, -43412, -43562, -43712, -43862, -44011, -44160, -44308, -44456,
    -44603, -44750, -44897, -45043, -45189, -45335, -45480, -45624, -45768,
    -45912, -46055, -46198, -46340, -46482, -46624, -46765, -46906, -47046,
    -47186, -47325, -47464, -47602, -47740, -47878, -48015, -48151, -48288,
    -48423, -48558, -48693, -48828, -48961, -49095, -49228, -49360, -49492,
    -49624, -49755, -49886, -50016, -50146, -50275, -50403, -50532, -50660,
    -50787, -50914, -51040, -51166, -51291, -51416, -51541, -51665, -51788,
    -51911, -52033, -52155, -52277, -52398, -52518, -52639, -52758, -52877,
    -52996, -53114, -53231, -53348, -53465, -53581, -53696, -53811, -53926,
    -54040, -54153, -54266, -54379, -54491, -54602, -54713, -54823, -54933,
    -55043, -55152, -55260, -55368, -55475, -55582, -55688, -55794, -55899,
    -56004, -56108, -56212, -56315, -56417, -56519, -56621, -56722, -56822,
    -56922, -57022, -57120, -57219, -57316, -57414, -57510, -57606, -57702,
    -57797, -57892, -57986, -58079, -58172, -58264, -58356, -58447, -58538,
    -58628, -58718, -58807, -58895, -58983, -59070, -59157, -59243, -59329,
    -59414, -59499, -59583, -59666, -59749, -59831, -59913, -59994, -60075,
    -60155, -60235, -60313, -60392, -60470, -60547, -60624, -60700, -60775,
    -60850, -60924, -60998, -61071, -61144, -61216, -61288, -61359, -61429,
    -61499, -61568, -61637, -61705, -61772, -61839, -61905, -61971, -62036,
    -62100, -62164, -62228, -62291, -62353, -62414, -62475, -62536, -62596,
    -62655, -62714, -62772, -62829, -62886, -62942, -62998, -63053, -63108,
    -63162, -63215, -63268, -63320, -63371, -63422, -63473, -63522, -63571,
    -63620, -63668, -63715, -63762, -63808, -63854, -63899, -63943, -63987,
    -64030, -64073, -64115, -64156, -64197, -64237, -64276, -64315, -64353,
    -64391, -64428, -64465, -64501, -64536, -64571, -64605, -64638, -64671,
    -64703, -64735, -64766, -64796, -64826, -64855, -64884, -64912, -64939,
    -64966, -64992, -65018, -65043, -65067, -65091, -65114, -65136, -65158,
    -65179, -65200, -65220, -65239, -65258, -65276, -65294, -65311, -65327,
    -65343, -65358, -65372, -65386, -65400, -65412, -65424, -65436, -65446,
    -65457, -65466, -65475, -65483, -65491, -65498, -65505, -65511, -65516,
    -65520, -65524, -65528, -65531, -65533, -65534, -65535, -65536, -65535,
    -65534, -65533, -65531, -65528, -65524, -65520, -65516, -65511, -65505,
    -65498, -65491, -65483, -65475, -65466, -65457, -65446, -65436, -65424,
    -65412, -65400, -65386, -65372, -65358, -65343, -65327, -65311, -65294,
    -65276, -65258, -65239, -65220, -65200, -65179, -65158, -65136, -65114,
    -65091, -65067, -65043, -65018, -64992, -64966, -64939, -64912, -64884,
    -64855, -64826, -64796, -64766, -64735, -64703, -64671, -64638, -64605,
    -64571, -64536, -64501, -64465, -64428, -64391, -64353, -64315, -64276,
    -64237, -64197, -64156, -64115, -64073, -64030, -63987, -63943, -63899,
    -63854, -63808, -63762, -63715, -63668, -63620, -63571, -63522, -63473,
    -63422, -63371, -63320, -63268, -63215, -63162, -63108, -63053, -62998,
    -62942, -62886, -62829, -62772, -62714, -62655, -62596, -62536, -62475,
    -62414, -62353, -62291, -62228, -62164, -62100, -62036, -61971, -61905,
    -61839, -61772, -61705, -61637, -61568, -61499, -61429, -61359, -61288,
    -61216, -61144, -61071, -60998, -60924, -60850, -60775, -60700, -60624,
    -60547, -60470, -60392, -60313, -60235, -60155, -60075, -59994, -59913,
    -59831, -59749, -59666, -59583, -59499, -59414, -59329, -59243, -59157,
    -59070, -58983, -58895, -58807, -58718, -58628, -58538, -58447, -58356,
    -58264, -58172, -58079, -57986, -57892, -57797, -57702, -57606, -57510,
    -57414, -57316, -57219, -57120, -57022, -56922, -56822, -56722, -56621,
    -56519, -56417, -56315, -56212, -56108, -56004, -55899, -55794, -55688,
    -55582, -55475, -55368, -55260, -55152, -55043, -54933, -54823, -54713,
    -54602, -54491, -54379, -54266, -54153, -54040, -53926, -53811, -53696,
    -53581, -53465, -53348, -53231, -53114, -52996, -52877, -52758, -52639,
    -52518, -52398, -52277, -52155, -52033, -51911, -51788, -51665, -51541,
    -51416, -51291, -51166, -51040, -50914, -50787, -50660, -50532, -50403,
    -50275, -50146, -50016, -49886, -49755, -49624, -49492, -49360, -49228,
    -49095, -48961, -48828, -48693, -48558, -48423, -48288, -48151, -48015,
    -47878, -47740, -47602, -47464, -47325, -47186, -47046, -46906, -46765,
    -46624, -46482, -46340, -46198, -46055, -45912, -45768, -45624, -45480,
    -45335, -45189, -45043, -44897, -44750, -44603, -44456, -44308, -44160,
    -44011, -43862, -43712, -43562, -43412, -43261, -43110, -42958, -42806,
    -42653, -42501, -42347, -42194, -42040, -41885, -41730, -41575, -41419,
    -41263, -41107, -40950, -40793, -40636, -40478, -40319, -40161, -40002,
    -39842, -39682, -39522, -39362, -39201, -39039, -38878, -38716, -38553,
    -38390, -38227, -38064, -37900, -37736, -37571, -37406, -37241, -37075,
    -36909, -36743, -36576, -36409, -36242, -36074, -35906, -35738, -35569,
    -35400, -35231, -35061, -34891, -34721, -34550, -34379, -34208, -34036,
    -33864, -33692, -33519, -33346, -33173, -32999, -32826, -32651, -32477,
    -32302, -32127, -31952, -31776, -31600, -31424, -31247, -31070, -30893,
    -30715, -30538, -30360, -30181, -30003, -29824, -29645, -29465, -29285,
    -29105, -28925, -28745, -28564, -28383, -28201, -28020, -27838, -27656,
    -27473, -27291, -27108, -26925, -26741, -26557, -26373, -26189, -26005,
    -25820, -25635, -25450, -25265, -25079, -24893, -24707, -24521, -24334,
    -24147, -23960, -23773, -23586, -23398, -23210, -23022, -22833, -22645,
    -22456, -22267, -22078, -21889, -21699, -21509, -21319, -21129, -20938,
    -20748, -20557, -20366, -20175, -19983, -19792, -19600, -19408, -19216,
    -19024, -18831, -18638, -18446, -18253, -18059, -17866, -17672, -17479,
    -17285, -17091, -16897, -16702, -16508, -16313, -16118, -15923, -15728,
    -15533, -15338, -15142, -14946, -14751, -14555, -14359, -14162, -13966,
    -13769, -13573, -13376, -13179, -12982, -12785, -12588, -12390, -12193,
    -11995, -11797, -11600, -11402, -11204, -11006, -10807, -10609, -10410,
    -10212, -10013, -9814,  -9616,  -9417,  -9218,  -9019,  -8819,  -8620,
    -8421,  -8221,  -8022,  -7822,  -7623,  -7423,  -7223,  -7023,  -6823,
    -6623,  -6423,  -6223,  -6023,  -5823,  -5622,  -5422,  -5222,  -5021,
    -4821,  -4620,  -4420,  -4219,  -4018,  -3818,  -3617,  -3416,  -3215,
    -3014,  -2814,  -2613,  -2412,  -2211,  -2010,  -1809,  -1608,  -1407,
    -1206,  -1005,  -804,   -603,   -402
};

static const int lbCosTable[2048] = {
    65536,  65535,  65534,  65533,  65531,  65528,  65524,  65520,  65516,
    65511,  65505,  65498,  65491,  65483,  65475,  65466,  65457,  65446,
    65436,  65424,  65412,  65400,  65386,  65372,  65358,  65343,  65327,
    65311,  65294,  65276,  65258,  65239,  65220,  65200,  65179,  65158,
    65136,  65114,  65091,  65067,  65043,  65018,  64992,  64966,  64939,
    64912,  64884,  64855,  64826,  64796,  64766,  64735,  64703,  64671,
    64638,  64605,  64571,  64536,  64501,  64465,  64428,  64391,  64353,
    64315,  64276,  64237,  64197,  64156,  64115,  64073,  64030,  63987,
    63943,  63899,  63854,  63808,  63762,  63715,  63668,  63620,  63571,
    63522,  63473,  63422,  63371,  63320,  63268,  63215,  63162,  63108,
    63053,  62998,  62942,  62886,  62829,  62772,  62714,  62655,  62596,
    62536,  62475,  62414,  62353,  62291,  62228,  62164,  62100,  62036,
    61971,  61905,  61839,  61772,  61705,  61637,  61568,  61499,  61429,
    61359,  61288,  61216,  61144,  61071,  60998,  60924,  60850,  60775,
    60700,  60624,  60547,  60470,  60392,  60313,  60235,  60155,  60075,
    59994,  59913,  59831,  59749,  59666,  59583,  59499,  59414,  59329,
    59243,  59157,  59070,  58983,  58895,  58807,  58718,  58628,  58538,
    58447,  58356,  58264,  58172,  58079,  57986,  57892,  57797,  57702,
    57606,  57510,  57414,  57316,  57219,  57120,  57022,  56922,  56822,
    56722,  56621,  56519,  56417,  56315,  56212,  56108,  56004,  55899,
    55794,  55688,  55582,  55475,  55368,  55260,  55152,  55043,  54933,
    54823,  54713,  54602,  54491,  54379,  54266,  54153,  54040,  53926,
    53811,  53696,  53581,  53465,  53348,  53231,  53114,  52996,  52877,
    52758,  52639,  52518,  52398,  52277,  52155,  52033,  51911,  51788,
    51665,  51541,  51416,  51291,  51166,  51040,  50914,  50787,  50660,
    50532,  50403,  50275,  50146,  50016,  49886,  49755,  49624,  49492,
    49360,  49228,  49095,  48961,  48828,  48693,  48558,  48423,  48288,
    48151,  48015,  47878,  47740,  47602,  47464,  47325,  47186,  47046,
    46906,  46765,  46624,  46482,  46340,  46198,  46055,  45912,  45768,
    45624,  45480,  45335,  45189,  45043,  44897,  44750,  44603,  44456,
    44308,  44160,  44011,  43862,  43712,  43562,  43412,  43261,  43110,
    42958,  42806,  42653,  42501,  42347,  42194,  42040,  41885,  41730,
    41575,  41419,  41263,  41107,  40950,  40793,  40636,  40478,  40319,
    40161,  40002,  39842,  39682,  39522,  39362,  39201,  39039,  38878,
    38716,  38553,  38390,  38227,  38064,  37900,  37736,  37571,  37406,
    37241,  37075,  36909,  36743,  36576,  36409,  36242,  36074,  35906,
    35738,  35569,  35400,  35231,  35061,  34891,  34721,  34550,  34379,
    34208,  34036,  33864,  33692,  33519,  33346,  33173,  32999,  32826,
    32651,  32477,  32302,  32127,  31952,  31776,  31600,  31424,  31247,
    31070,  30893,  30715,  30538,  30360,  30181,  30003,  29824,  29645,
    29465,  29285,  29105,  28925,  28745,  28564,  28383,  28201,  28020,
    27838,  27656,  27473,  27291,  27108,  26925,  26741,  26557,  26373,
    26189,  26005,  25820,  25635,  25450,  25265,  25079,  24893,  24707,
    24521,  24334,  24147,  23960,  23773,  23586,  23398,  23210,  23022,
    22833,  22645,  22456,  22267,  22078,  21889,  21699,  21509,  21319,
    21129,  20938,  20748,  20557,  20366,  20175,  19983,  19792,  19600,
    19408,  19216,  19024,  18831,  18638,  18446,  18253,  18059,  17866,
    17672,  17479,  17285,  17091,  16897,  16702,  16508,  16313,  16118,
    15923,  15728,  15533,  15338,  15142,  14946,  14751,  14555,  14359,
    14162,  13966,  13769,  13573,  13376,  13179,  12982,  12785,  12588,
    12390,  12193,  11995,  11797,  11600,  11402,  11204,  11006,  10807,
    10609,  10410,  10212,  10013,  9814,   9616,   9417,   9218,   9019,
    8819,   8620,   8421,   8221,   8022,   7822,   7623,   7423,   7223,
    7023,   6823,   6623,   6423,   6223,   6023,   5823,   5622,   5422,
    5222,   5021,   4821,   4620,   4420,   4219,   4018,   3818,   3617,
    3416,   3215,   3014,   2814,   2613,   2412,   2211,   2010,   1809,
    1608,   1407,   1206,   1005,   804,    603,    402,    201,    0,
    -201,   -402,   -603,   -804,   -1005,  -1206,  -1407,  -1608,  -1809,
    -2010,  -2211,  -2412,  -2613,  -2814,  -3014,  -3215,  -3416,  -3617,
    -3818,  -4018,  -4219,  -4420,  -4620,  -4821,  -5021,  -5222,  -5422,
    -5622,  -5823,  -6023,  -6223,  -6423,  -6623,  -6823,  -7023,  -7223,
    -7423,  -7623,  -7822,  -8022,  -8221,  -8421,  -8620,  -8819,  -9019,
    -9218,  -9417,  -9616,  -9814,  -10013, -10212, -10410, -10609, -10807,
    -11006, -11204, -11402, -11600, -11797, -11995, -12193, -12390, -12588,
    -12785, -12982, -13179, -13376, -13573, -13769, -13966, -14162, -14359,
    -14555, -14751, -14946, -15142, -15338, -15533, -15728, -15923, -16118,
    -16313, -16508, -16702, -16897, -17091, -17285, -17479, -17672, -17866,
    -18059, -18253, -18446, -18638, -18831, -19024, -19216, -19408, -19600,
    -19792, -19983, -20175, -20366, -20557, -20748, -20938, -21129, -21319,
    -21509, -21699, -21889, -22078, -22267, -22456, -22645, -22833, -23022,
    -23210, -23398, -23586, -23773, -23960, -24147, -24334, -24521, -24707,
    -24893, -25079, -25265, -25450, -25635, -25820, -26005, -26189, -26373,
    -26557, -26741, -26925, -27108, -27291, -27473, -27656, -27838, -28020,
    -28201, -28383, -28564, -28745, -28925, -29105, -29285, -29465, -29645,
    -29824, -30003, -30181, -30360, -30538, -30715, -30893, -31070, -31247,
    -31424, -31600, -31776, -31952, -32127, -32302, -32477, -32651, -32826,
    -32999, -33173, -33346, -33519, -33692, -33864, -34036, -34208, -34379,
    -34550, -34721, -34891, -35061, -35231, -35400, -35569, -35738, -35906,
    -36074, -36242, -36409, -36576, -36743, -36909, -37075, -37241, -37406,
    -37571, -37736, -37900, -38064, -38227, -38390, -38553, -38716, -38878,
    -39039, -39201, -39362, -39522, -39682, -39842, -40002, -40161, -40319,
    -40478, -40636, -40793, -40950, -41107, -41263, -41419, -41575, -41730,
    -41885, -42040, -42194, -42347, -42501, -42653, -42806, -42958, -43110,
    -43261, -43412, -43562, -43712, -43862, -44011, -44160, -44308, -44456,
    -44603, -44750, -44897, -45043, -45189, -45335, -45480, -45624, -45768,
    -45912, -46055, -46198, -46340, -46482, -46624, -46765, -46906, -47046,
    -47186, -47325, -47464, -47602, -47740, -47878, -48015, -48151, -48288,
    -48423, -48558, -48693, -48828, -48961, -49095, -49228, -49360, -49492,
    -49624, -49755, -49886, -50016, -50146, -50275, -50403, -50532, -50660,
    -50787, -50914, -51040, -51166, -51291, -51416, -51541, -51665, -51788,
    -51911, -52033, -52155, -52277, -52398, -52518, -52639, -52758, -52877,
    -52996, -53114, -53231, -53348, -53465, -53581, -53696, -53811, -53926,
    -54040, -54153, -54266, -54379, -54491, -54602, -54713, -54823, -54933,
    -55043, -55152, -55260, -55368, -55475, -55582, -55688, -55794, -55899,
    -56004, -56108, -56212, -56315, -56417, -56519, -56621, -56722, -56822,
    -56922, -57022, -57120, -57219, -57316, -57414, -57510, -57606, -57702,
    -57797, -57892, -57986, -58079, -58172, -58264, -58356, -58447, -58538,
    -58628, -58718, -58807, -58895, -58983, -59070, -59157, -59243, -59329,
    -59414, -59499, -59583, -59666, -59749, -59831, -59913, -59994, -60075,
    -60155, -60235, -60313, -60392, -60470, -60547, -60624, -60700, -60775,
    -60850, -60924, -60998, -61071, -61144, -61216, -61288, -61359, -61429,
    -61499, -61568, -61637, -61705, -61772, -61839, -61905, -61971, -62036,
    -62100, -62164, -62228, -62291, -62353, -62414, -62475, -62536, -62596,
    -62655, -62714, -62772, -62829, -62886, -62942, -62998, -63053, -63108,
    -63162, -63215, -63268, -63320, -63371, -63422, -63473, -63522, -63571,
    -63620, -63668, -63715, -63762, -63808, -63854, -63899, -63943, -63987,
    -64030, -64073, -64115, -64156, -64197, -64237, -64276, -64315, -64353,
    -64391, -64428, -64465, -64501, -64536, -64571, -64605, -64638, -64671,
    -64703, -64735, -64766, -64796, -64826, -64855, -64884, -64912, -64939,
    -64966, -64992, -65018, -65043, -65067, -65091, -65114, -65136, -65158,
    -65179, -65200, -65220, -65239, -65258, -65276, -65294, -65311, -65327,
    -65343, -65358, -65372, -65386, -65400, -65412, -65424, -65436, -65446,
    -65457, -65466, -65475, -65483, -65491, -65498, -65505, -65511, -65516,
    -65520, -65524, -65528, -65531, -65533, -65534, -65535, -65536, -65535,
    -65534, -65533, -65531, -65528, -65524, -65520, -65516, -65511, -65505,
    -65498, -65491, -65483, -65475, -65466, -65457, -65446, -65436, -65424,
    -65412, -65400, -65386, -65372, -65358, -65343, -65327, -65311, -65294,
    -65276, -65258, -65239, -65220, -65200, -65179, -65158, -65136, -65114,
    -65091, -65067, -65043, -65018, -64992, -64966, -64939, -64912, -64884,
    -64855, -64826, -64796, -64766, -64735, -64703, -64671, -64638, -64605,
    -64571, -64536, -64501, -64465, -64428, -64391, -64353, -64315, -64276,
    -64237, -64197, -64156, -64115, -64073, -64030, -63987, -63943, -63899,
    -63854, -63808, -63762, -63715, -63668, -63620, -63571, -63522, -63473,
    -63422, -63371, -63320, -63268, -63215, -63162, -63108, -63053, -62998,
    -62942, -62886, -62829, -62772, -62714, -62655, -62596, -62536, -62475,
    -62414, -62353, -62291, -62228, -62164, -62100, -62036, -61971, -61905,
    -61839, -61772, -61705, -61637, -61568, -61499, -61429, -61359, -61288,
    -61216, -61144, -61071, -60998, -60924, -60850, -60775, -60700, -60624,
    -60547, -60470, -60392, -60313, -60235, -60155, -60075, -59994, -59913,
    -59831, -59749, -59666, -59583, -59499, -59414, -59329, -59243, -59157,
    -59070, -58983, -58895, -58807, -58718, -58628, -58538, -58447, -58356,
    -58264, -58172, -58079, -57986, -57892, -57797, -57702, -57606, -57510,
    -57414, -57316, -57219, -57120, -57022, -56922, -56822, -56722, -56621,
    -56519, -56417, -56315, -56212, -56108, -56004, -55899, -55794, -55688,
    -55582, -55475, -55368, -55260, -55152, -55043, -54933, -54823, -54713,
    -54602, -54491, -54379, -54266, -54153, -54040, -53926, -53811, -53696,
    -53581, -53465, -53348, -53231, -53114, -52996, -52877, -52758, -52639,
    -52518, -52398, -52277, -52155, -52033, -51911, -51788, -51665, -51541,
    -51416, -51291, -51166, -51040, -50914, -50787, -50660, -50532, -50403,
    -50275, -50146, -50016, -49886, -49755, -49624, -49492, -49360, -49228,
    -49095, -48961, -48828, -48693, -48558, -48423, -48288, -48151, -48015,
    -47878, -47740, -47602, -47464, -47325, -47186, -47046, -46906, -46765,
    -46624, -46482, -46340, -46198, -46055, -45912, -45768, -45624, -45480,
    -45335, -45189, -45043, -44897, -44750, -44603, -44456, -44308, -44160,
    -44011, -43862, -43712, -43562, -43412, -43261, -43110, -42958, -42806,
    -42653, -42501, -42347, -42194, -42040, -41885, -41730, -41575, -41419,
    -41263, -41107, -40950, -40793, -40636, -40478, -40319, -40161, -40002,
    -39842, -39682, -39522, -39362, -39201, -39039, -38878, -38716, -38553,
    -38390, -38227, -38064, -37900, -37736, -37571, -37406, -37241, -37075,
    -36909, -36743, -36576, -36409, -36242, -36074, -35906, -35738, -35569,
    -35400, -35231, -35061, -34891, -34721, -34550, -34379, -34208, -34036,
    -33864, -33692, -33519, -33346, -33173, -32999, -32826, -32651, -32477,
    -32302, -32127, -31952, -31776, -31600, -31424, -31247, -31070, -30893,
    -30715, -30538, -30360, -30181, -30003, -29824, -29645, -29465, -29285,
    -29105, -28925, -28745, -28564, -28383, -28201, -28020, -27838, -27656,
    -27473, -27291, -27108, -26925, -26741, -26557, -26373, -26189, -26005,
    -25820, -25635, -25450, -25265, -25079, -24893, -24707, -24521, -24334,
    -24147, -23960, -23773, -23586, -23398, -23210, -23022, -22833, -22645,
    -22456, -22267, -22078, -21889, -21699, -21509, -21319, -21129, -20938,
    -20748, -20557, -20366, -20175, -19983, -19792, -19600, -19408, -19216,
    -19024, -18831, -18638, -18446, -18253, -18059, -17866, -17672, -17479,
    -17285, -17091, -16897, -16702, -16508, -16313, -16118, -15923, -15728,
    -15533, -15338, -15142, -14946, -14751, -14555, -14359, -14162, -13966,
    -13769, -13573, -13376, -13179, -12982, -12785, -12588, -12390, -12193,
    -11995, -11797, -11600, -11402, -11204, -11006, -10807, -10609, -10410,
    -10212, -10013, -9814,  -9616,  -9417,  -9218,  -9019,  -8819,  -8620,
    -8421,  -8221,  -8022,  -7822,  -7623,  -7423,  -7223,  -7023,  -6823,
    -6623,  -6423,  -6223,  -6023,  -5823,  -5622,  -5422,  -5222,  -5021,
    -4821,  -4620,  -4420,  -4219,  -4018,  -3818,  -3617,  -3416,  -3215,
    -3014,  -2814,  -2613,  -2412,  -2211,  -2010,  -1809,  -1608,  -1407,
    -1206,  -1005,  -804,   -603,   -402,   -201,   0,      201,    402,
    603,    804,    1005,   1206,   1407,   1608,   1809,   2010,   2211,
    2412,   2613,   2814,   3014,   3215,   3416,   3617,   3818,   4018,
    4219,   4420,   4620,   4821,   5021,   5222,   5422,   5622,   5823,
    6023,   6223,   6423,   6623,   6823,   7023,   7223,   7423,   7623,
    7822,   8022,   8221,   8421,   8620,   8819,   9019,   9218,   9417,
    9616,   9814,   10013,  10212,  10410,  10609,  10807,  11006,  11204,
    11402,  11600,  11797,  11995,  12193,  12390,  12588,  12785,  12982,
    13179,  13376,  13573,  13769,  13966,  14162,  14359,  14555,  14751,
    14946,  15142,  15338,  15533,  15728,  15923,  16118,  16313,  16508,
    16702,  16897,  17091,  17285,  17479,  17672,  17866,  18059,  18253,
    18446,  18638,  18831,  19024,  19216,  19408,  19600,  19792,  19983,
    20175,  20366,  20557,  20748,  20938,  21129,  21319,  21509,  21699,
    21889,  22078,  22267,  22456,  22645,  22833,  23022,  23210,  23398,
    23586,  23773,  23960,  24147,  24334,  24521,  24707,  24893,  25079,
    25265,  25450,  25635,  25820,  26005,  26189,  26373,  26557,  26741,
    26925,  27108,  27291,  27473,  27656,  27838,  28020,  28201,  28383,
    28564,  28745,  28925,  29105,  29285,  29465,  29645,  29824,  30003,
    30181,  30360,  30538,  30715,  30893,  31070,  31247,  31424,  31600,
    31776,  31952,  32127,  32302,  32477,  32651,  32826,  32999,  33173,
    33346,  33519,  33692,  33864,  34036,  34208,  34379,  34550,  34721,
    34891,  35061,  35231,  35400,  35569,  35738,  35906,  36074,  36242,
    36409,  36576,  36743,  36909,  37075,  37241,  37406,  37571,  37736,
    37900,  38064,  38227,  38390,  38553,  38716,  38878,  39039,  39201,
    39362,  39522,  39682,  39842,  40002,  40161,  40319,  40478,  40636,
    40793,  40950,  41107,  41263,  41419,  41575,  41730,  41885,  42040,
    42194,  42347,  42501,  42653,  42806,  42958,  43110,  43261,  43412,
    43562,  43712,  43862,  44011,  44160,  44308,  44456,  44603,  44750,
    44897,  45043,  45189,  45335,  45480,  45624,  45768,  45912,  46055,
    46198,  46340,  46482,  46624,  46765,  46906,  47046,  47186,  47325,
    47464,  47602,  47740,  47878,  48015,  48151,  48288,  48423,  48558,
    48693,  48828,  48961,  49095,  49228,  49360,  49492,  49624,  49755,
    49886,  50016,  50146,  50275,  50403,  50532,  50660,  50787,  50914,
    51040,  51166,  51291,  51416,  51541,  51665,  51788,  51911,  52033,
    52155,  52277,  52398,  52518,  52639,  52758,  52877,  52996,  53114,
    53231,  53348,  53465,  53581,  53696,  53811,  53926,  54040,  54153,
    54266,  54379,  54491,  54602,  54713,  54823,  54933,  55043,  55152,
    55260,  55368,  55475,  55582,  55688,  55794,  55899,  56004,  56108,
    56212,  56315,  56417,  56519,  56621,  56722,  56822,  56922,  57022,
    57120,  57219,  57316,  57414,  57510,  57606,  57702,  57797,  57892,
    57986,  58079,  58172,  58264,  58356,  58447,  58538,  58628,  58718,
    58807,  58895,  58983,  59070,  59157,  59243,  59329,  59414,  59499,
    59583,  59666,  59749,  59831,  59913,  59994,  60075,  60155,  60235,
    60313,  60392,  60470,  60547,  60624,  60700,  60775,  60850,  60924,
    60998,  61071,  61144,  61216,  61288,  61359,  61429,  61499,  61568,
    61637,  61705,  61772,  61839,  61905,  61971,  62036,  62100,  62164,
    62228,  62291,  62353,  62414,  62475,  62536,  62596,  62655,  62714,
    62772,  62829,  62886,  62942,  62998,  63053,  63108,  63162,  63215,
    63268,  63320,  63371,  63422,  63473,  63522,  63571,  63620,  63668,
    63715,  63762,  63808,  63854,  63899,  63943,  63987,  64030,  64073,
    64115,  64156,  64197,  64237,  64276,  64315,  64353,  64391,  64428,
    64465,  64501,  64536,  64571,  64605,  64638,  64671,  64703,  64735,
    64766,  64796,  64826,  64855,  64884,  64912,  64939,  64966,  64992,
    65018,  65043,  65067,  65091,  65114,  65136,  65158,  65179,  65200,
    65220,  65239,  65258,  65276,  65294,  65311,  65327,  65343,  65358,
    65372,  65386,  65400,  65412,  65424,  65436,  65446,  65457,  65466,
    65475,  65483,  65491,  65498,  65505,  65511,  65516,  65520,  65524,
    65528,  65531,  65533,  65534,  65535 };
