/******************************************************************************/
// Bullfrog Engine Emulation Library - for use to remake classic games like
// Syndicate Wars, Magic Carpet or Dungeon Keeper.
/******************************************************************************/
/** @file bflib_netsp.cpp
 *     Network ServiceProvider class declaration.
 * @par Purpose:
 *     Defines ServiceProvider for network library.
 * @par Comment:
 *     None.
 * @author   Tomasz Lis
 * @date     08 Mar 2009 - 09 Aug 2009
 * @par  Copying and copyrights:
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 */
/******************************************************************************/
#include "pre_inc.h"
#include "bflib_netsp.hpp"

#include "bflib_basics.h"
#include "bflib_netsession.h"
#include "post_inc.h"

/******************************************************************************/
// Nil callbacks declaration
void NilAddMsgCallback(unsigned long , char *, void *);
void NilDeleteMsgCallback(unsigned long, void *);
void NilHostMsgCallback(unsigned long, void *);
void NilUserSysMsgCallback(void *);
void *NilUserDataMsgCallback(unsigned long, unsigned long, unsigned long, void *);
void NilRequestExchangeDataMsgCallback(unsigned long, unsigned long, void *);
void NilRequestCompositeExchangeDataMsgCallback(unsigned long, unsigned long, void *);
void *NilUnidirectionalMsgCallback(unsigned long, unsigned long, void *);
void NilSystemUserMsgCallback(unsigned long, void *, unsigned long, void *);

struct ReceiveCallbacks nilReceiveAspect = {
  NilAddMsgCallback,
  NilDeleteMsgCallback,
  NilHostMsgCallback,
  NilUserSysMsgCallback,
  NilUserDataMsgCallback,
  NilRequestExchangeDataMsgCallback,
  NilRequestCompositeExchangeDataMsgCallback,
  NilUnidirectionalMsgCallback,
  NilSystemUserMsgCallback,
  NULL,
};
/******************************************************************************/
class ServiceProvider *spPtr;
/******************************************************************************/
// Nil callbacks content
void NilAddMsgCallback(unsigned long player_id, char *message, void *data)
{
  WARNLOG("hit(%lu, \"%s\", *)",player_id,message);
}

void NilDeleteMsgCallback(unsigned long player_id, void *data)
{
  WARNLOG("hit(%lu, *)",player_id);
}

void NilHostMsgCallback(unsigned long player_id, void *data)
{
  WARNLOG("hit(%lu, *)",player_id);
}

void NilUserSysMsgCallback(void *data)
{
  WARNLOG("hit(*)");
}

void *NilUserDataMsgCallback(unsigned long player_id, unsigned long message_type, unsigned long data_size, void *data)
{
  WARNLOG("hit(%lu, %lu, %lu, *)",player_id,message_type,data_size);
  return NULL;
}

void NilRequestExchangeDataMsgCallback(unsigned long player_id, unsigned long data_size, void *data)
{
  WARNLOG("hit(%lu, %lu, *)",player_id,data_size);
}

void NilRequestCompositeExchangeDataMsgCallback(unsigned long player_id, unsigned long data_size, void *data)
{
  WARNLOG("hit(%lu, %lu, *)",player_id,data_size);
}

void *NilUnidirectionalMsgCallback(unsigned long player_id, unsigned long message_type, void *data)
{
  WARNLOG("hit(%lu, %lu, *)",player_id,message_type);
  return NULL;
}

void NilSystemUserMsgCallback(unsigned long player_id, void *system_data, unsigned long data_size, void *user_data)
{
  WARNLOG("hit(%lu, *, %lu, *)",player_id,data_size);
}
/******************************************************************************/
// methods of virtual class ServiceProvider

void ServiceProvider::DecodeMessageStub(const void *msgHeader, uint32_t *dataLen,
      unsigned char *messageType, uint32_t *seqNbr)
{
  unsigned long k;
  if (msgHeader == NULL)
  {
      WARNLOG("NULL ptr");
      return;
  }
  if (dataLen != NULL)
  {
      k = *(uint32_t *)msgHeader;
      *dataLen = k & 0xFFFFF; //does not include length of header (which is 4 bytes)
  }
  if (seqNbr != NULL)
  {
      k = *(uint32_t *)msgHeader;
      *seqNbr = ((k >> 20) & 0xF);
  }
  if (messageType != NULL)
  {
      k = *(uint32_t *)msgHeader;
      *messageType = (k >> 24)  & 0xFF;
  }
}

ServiceProvider::ServiceProvider() :
      nextSessionId(1),
      nextPlayerId(2)
{
  long i;
  this->localPlayerId = 0;
  for (i=0; i < SESSION_ENTRIES_COUNT; i++)
  {
    nsnames[i].id = 0;
    nsnames[i].in_use = false;
    memset(nsnames[i].text, 0, SESSION_NAME_MAX_LEN);
  }
  this->reference_count = 0;
  this->status_flags = 0;
  this->players_count = 0;
  strcpy(this->session_identifier,"");
  this->started = 0;
  this->callback_context = 0;
  this->recvCallbacks = &nilReceiveAspect;
}

ServiceProvider::~ServiceProvider()
{
  NETMSG("Destructing Service Provider");
}

TbError ServiceProvider::Initialise(struct ReceiveCallbacks *nCallbacks, void *a2)
{
  NETMSG("Initializing Service Provider");
  if (this->reference_count)
    WARNLOG("Service Provider already set up!");
  this->players_count = 0;
  ClearSessions();
  this->callback_context = a2;
  if (nCallbacks != NULL)
    this->recvCallbacks = nCallbacks;
  this->status_flags = 0;
  this->started = false;
  this->reference_count++;
  return Lb_OK;
}

TbError ServiceProvider::Send(unsigned long plr_id, void *buf)
{
  unsigned char messageType;
  uint32_t dataLen;
  uint32_t seqNbr;
  unsigned long player_id_from_buffer;
  void *imsg;
  char str[32];
  long i;
  if (!this->started)
  {
    WARNLOG("not initialized");
    return Lb_FAIL;
  }
  messageType = 0;
  seqNbr = 0;
  dataLen = 0;
  DecodeMessageStub(buf,&dataLen,&messageType,&seqNbr);
  if (plr_id != this->localPlayerId)
  {
    i = (messageType != 0) && (messageType != 5) && (messageType != 6) && (messageType != 8);
    if (this->SendMessage(plr_id, buf, i) != Lb_OK)
    {
      WARNLOG("failure on SendMessage()");
      return Lb_FAIL;
    }
    return Lb_OK;
  }
  switch (messageType)
  {
  case NETMSGTYPE_MULTIPLAYER:
      if (recvCallbacks->multiPlayer == NULL)
      {
        WARNLOG("NIL target for userDataMsgCallbackProc");
        break;
      }
      imsg = recvCallbacks->multiPlayer(this->localPlayerId, dataLen+4, seqNbr, this->callback_context);
      if (imsg == NULL)
        break;
      memcpy(imsg, buf, dataLen+4);
      break;
  case NETMSGTYPE_ADD:
      memcpy(&player_id_from_buffer, (uchar *)buf+4, sizeof(uint32_t));
      snprintf(str, sizeof(str), "%s", (char*)buf + 8);
      this->AddPlayer(player_id_from_buffer, str, 0, 0);
      if (recvCallbacks->addMsg == NULL)
      {
        break;
      }
      recvCallbacks->addMsg(player_id_from_buffer, str, this->callback_context);
      break;
  case NETMSGTYPE_DELETE:
      this->CheckForDeletedHost(buf);
      memcpy(&dataLen, (uchar *)buf+4, 4);
      this->DeletePlayer(dataLen);
      if (recvCallbacks->deleteMsg == NULL)
      {
        break;
      }
      recvCallbacks->deleteMsg(dataLen, this->callback_context);
      break;
  case NETMSGTYPE_PROBABLYHOST:
      break;
  case NETMSGTYPE_SYSUSER:
      if (recvCallbacks->systemUserMsg == NULL)
      {
        WARNLOG("NIL target for systemUserMsgCallbackProc");
        break;
      }
      recvCallbacks->systemUserMsg(this->localPlayerId, (char *)buf+4, dataLen, this->callback_context);
      break;
  case NETMSGTYPE_MPREQEXDATA:
      memcpy(&player_id_from_buffer, (uchar *)buf+4, sizeof(uint32_t));
      if (recvCallbacks->mpReqExDataMsg == NULL)
      {
        break;
      }
      recvCallbacks->mpReqExDataMsg(player_id_from_buffer, seqNbr, this->callback_context);
      break;
  case NETMSGTYPE_MPREQCOMPEXDATA:
      memcpy(&player_id_from_buffer, (uchar *)buf+4, sizeof(uint32_t));
      if (recvCallbacks->mpReqCompsExDataMsg == NULL)
      {
        break;
      }
      recvCallbacks->mpReqCompsExDataMsg(player_id_from_buffer, seqNbr, this->callback_context);
      break;
  case NETMSGTYPE_UNKNOWN:
      // This callback seems to never be used
      player_id_from_buffer = 0;
      if (recvCallbacks->unhandledMessageTypeCallback == NULL)
      {
        break;
      }
      recvCallbacks->unhandledMessageTypeCallback(player_id_from_buffer, buf);
      break;
  default:
      WARNLOG("messageType is out of range");
      break;
  }
  return Lb_OK;
}

TbError ServiceProvider::Receive(unsigned long flags)
{
    uint32_t playerId;
    uint32_t dataLen;
    uint32_t seqNbr;
    unsigned char messageType;
    unsigned long id;
    TbBool keepExchanging;
    char msgBuffer[1028];
    uint32_t msgLen;
    char array2[32];
    char array3[32];
    char * msgBufferPtr;
    char * msgBufferPtr2;
    char * array3Ptr;
    void * somePtr;
    long tmpInt1;
    long tmpInt2;
    //TbError result; // unused

    //result = 0;

    if (!this->started)
    {
      WARNLOG("not initialized");
      return Lb_FAIL;
    }
    keepExchanging = true;
    while (keepExchanging)
    {
      msgLen = 1028;
      if (!PeekMessage(&playerId, msgBuffer, &msgLen))
      {
        keepExchanging = false;
        break;
      }

      somePtr = NULL;

      DecodeMessageStub(msgBuffer, &dataLen, &messageType, &seqNbr);
      if (flags & 0x08)
        flags = 0xFF;

      switch (messageType)
      {
      case NETMSGTYPE_MULTIPLAYER:
          if (flags & 1) {
              somePtr = 0;
              if (recvCallbacks->multiPlayer) {
                  somePtr = recvCallbacks->multiPlayer(playerId, dataLen + 4, seqNbr, callback_context);
              }
              else {
                  NETMSG("NIL target for userDataMsgCallbackProc"); //rename
              }
              if (somePtr == NULL) {
                  somePtr = msgBuffer;
                  msgLen = 1028;
              }
              if (!(ReadMessage(&playerId, msgBuffer, &msgLen))) {
                  NETMSG("Inconsistency between reads");
                  //result = 0xff;
              }
          }

          keepExchanging = false;
          break;
      case NETMSGTYPE_ADD:
          if (!(flags & 2)) {
              keepExchanging = false;
              break;
          }

          msgLen = 1028;
          if (!ReadMessage(&playerId, msgBuffer, &msgLen)) {
              NETMSG("Inconsistency on receive");
              break;
          }

          memcpy(&tmpInt1, msgBuffer, sizeof(tmpInt1));
          {
              msgBufferPtr = msgBuffer + 4;
              msgBufferPtr2 = array2;
              bool cond;
              do {
                  msgBufferPtr2[0] = msgBufferPtr[0];
                  if (!msgBufferPtr[0]) {
                      break;
                  }

                  cond = msgBufferPtr[1];
                  msgBufferPtr += 2;
                  msgBufferPtr2[1] = cond;
                  msgBufferPtr2 += 2;
              } while (cond);
          }
          AddPlayer(tmpInt1, array2, 0, 0);

          memcpy(&tmpInt2, msgBuffer, sizeof(tmpInt2));
          memcpy(&tmpInt1, msgBuffer + 4, sizeof(tmpInt1));
          {
              msgBufferPtr = msgBuffer + 4;
              array3Ptr = array3;
              bool cond;
              do {
                  array3Ptr[0] = msgBufferPtr[0];
                  if (!msgBufferPtr[0]) {
                      break;
                  }

                  cond = msgBufferPtr[1];
                  msgBufferPtr += 2;
                  array3Ptr[1] = cond;
                  array3Ptr += 2;
              } while (cond);
          }

          if (recvCallbacks->addMsg) {
              recvCallbacks->addMsg(tmpInt2, array3, callback_context);
          }

          break;
      case NETMSGTYPE_DELETE:
          if (!(flags & 4)) {
              keepExchanging = false;
              break;
          }

          msgLen = 1028;
          if (!ReadMessage(&playerId, msgBuffer, &msgLen)) {
              NETMSG("Inconsistency on receive");
              break;
          }

          CheckForDeletedHost(msgBuffer);
          memcpy(&id, msgBuffer, sizeof(id));
          DeletePlayer(id);
          memcpy(&tmpInt1, msgBuffer, sizeof(tmpInt1));
          if (recvCallbacks->deleteMsg) {
              recvCallbacks->deleteMsg(tmpInt1, callback_context);
          }

          break;
      case NETMSGTYPE_PROBABLYHOST:
          continue;
      case NETMSGTYPE_SYSUSER:
          if (!(flags & 0x80)) {
              keepExchanging = false;
              break;
          }

          msgLen = 1028;
          if (!ReadMessage(&playerId, msgBuffer, &msgLen)) {
              NETMSG("Inconsistency on receive");
              break;
          }

          if (recvCallbacks->systemUserMsg) {
              recvCallbacks->systemUserMsg(playerId, msgBuffer, (*(uint32_t *) msgBuffer) & 0xFFFFF, callback_context);
          }

          break;
      case NETMSGTYPE_MPREQEXDATA:
          if (!(flags & 0x10)) {
              keepExchanging = false;
              break;
          }

          msgLen = 1028;
          if (!ReadMessage(&playerId, msgBuffer, &msgLen)) {
              NETMSG("Inconsistency on receive");
              break;
          }

          memcpy(&tmpInt1, msgBuffer + 4, sizeof(tmpInt1));
          if (recvCallbacks->mpReqExDataMsg) {
              recvCallbacks->mpReqExDataMsg(tmpInt1, seqNbr, callback_context);
          }

          break;
      case NETMSGTYPE_MPREQCOMPEXDATA:
          if (!(flags & 0x20)) {
              keepExchanging = false;
              break;
          }

          if (!ReadMessage(&playerId, msgBuffer, &msgLen)) {
              NETMSG("Inconsistency on receive");
              break;
          }

          memcpy(&tmpInt1, msgBuffer + 4, sizeof(tmpInt1));
          if (recvCallbacks->mpReqCompsExDataMsg) {
              recvCallbacks->mpReqCompsExDataMsg(tmpInt1, seqNbr, callback_context);
          }

          break;
      case NETMSGTYPE_UNIDIRECTIONAL:
          if (flags & 0x40) {
              if (recvCallbacks->unidirectionalMsg) {
                  somePtr = recvCallbacks->unidirectionalMsg(playerId, dataLen + 4, callback_context);
              }
              else {
                  NETMSG("NIL target for unidirectionalMsgCallbackProc");
              }

              if (somePtr == NULL) {
                  somePtr = msgBuffer;
                  msgLen = 1028;
              }

              if (!ReadMessage(&playerId, somePtr, &msgLen)) {
                  NETMSG("Inconsistency on receive");
                  break;
              }
          }

          keepExchanging = false;
          break;
      case NETMSGTYPE_UNKNOWN:
          if (!(flags & 0x01)) {
              break;
          }

          if (!recvCallbacks->unhandledMessageTypeCallback) {
              break;
          }

          msgLen = dataLen + 4;
          somePtr = calloc(msgLen, 1); //TODO NET check that this is freed somewhere...
          ReadMessage(&playerId, somePtr, &msgLen);
          recvCallbacks->unhandledMessageTypeCallback(playerId, somePtr);

          break;
      default:
          WARNLOG("messageType is out of range");
          break;
      }
    }
    return Lb_OK;
}

TbError ServiceProvider::Release(void)
{
  NETMSG("Releasing Service Provider");
  this->status_flags = 0;
  if (this->reference_count != 1)
    WARNLOG("Service Provider not set up!");
  this->reference_count--;
  return Lb_OK;
}

long ServiceProvider::PlayerIndex(unsigned long plyr_id)
{
  struct TbNetworkPlayerEntry *netplyr;
  long i;
  for (i=0; i < this->players_count; i++)
  {
    netplyr = &this->players[i];
    if (plyr_id == netplyr->id)
      return i;
  }
  return -1;
}

TbError ServiceProvider::AddPlayer(unsigned long plyr_id, const char *namestr, unsigned long a3, unsigned long a4)
{
  struct TbNetworkPlayerEntry *netplyr;
  long i;
  SYNCDBG(7, "Starting");
  // Check if we already have the player on list
  if (PlayerIndex(plyr_id) >= 0)
    return Lb_OK;
  // Get a structure for new player
  i = this->players_count;
  if (i >= NETSP_PLAYERS_COUNT)
  {
    WARNLOG("player table is full");
    return Lb_FAIL;
  }
  netplyr = &this->players[i];
  netplyr->id = plyr_id;
  net_copy_name_string(netplyr->name,namestr,NETSP_PLAYER_NAME_MAX_LEN);
  netplyr->reserved_data = a3;
  netplyr->is_active = a4;
  this->players_count++;
  return Lb_OK;
}

TbError ServiceProvider::DeletePlayer(unsigned long plyr_id)
{
  long i;
  SYNCDBG(7, "Starting");
  // Check if we have the player on list
  i = PlayerIndex(plyr_id);
  if (i < 0)
    return Lb_OK;
  while (i < this->players_count-1)
  {
    memcpy(&this->players[i], &this->players[i+1], sizeof(struct TbNetworkPlayerEntry));
    i++;
  }
  this->players_count--;
  return Lb_OK;
}

long ServiceProvider::SessionIndex(unsigned long sess_id)
{
  struct TbNetworkSessionNameEntry *nsname;
  int i;
  for (i=0; i < SESSION_ENTRIES_COUNT; i++)
  {
    nsname = &this->nsnames[i];
    if ((nsname->in_use) && (nsname->id == sess_id))
        return i;
  }
  return -1;
}

/**
 * Adds a session with given sess_id and name string.
 * @param sess_id ID of the session, or -1 if new session should be added.
 * @param namestr Text name of the new session, or NULL if session is unnamed.
 * @return Returns session name structure, or NULL if couldn't add.
 */
struct TbNetworkSessionNameEntry *ServiceProvider::AddSession(unsigned long sess_id, const char *namestr)
{
  struct TbNetworkSessionNameEntry *nsname;
  TbBool got;
  long i;
  // Check if the session i already in list
  i = SessionIndex(sess_id);
  if (i >= 0)
    return &this->nsnames[i];

  if (sess_id != -1) {
      ERRORLOG("Expected existing session ID which wasn't found");
      return NULL;
  }

  // Search for unused slot
  got = false;
  for (i=0; i < SESSION_ENTRIES_COUNT; i++)
  {
    nsname = &this->nsnames[i];
    if (!nsname->in_use && !(i == SESSION_ENTRIES_COUNT - 1 && localPlayerId == 0)) //reserves last slot if no game is being hosted
    {
      got = true;
      break;
    }
  }
  if (!got)
    return NULL;
  // Fill the new entry
  nsname->id = nextSessionId++;
  net_copy_name_string(nsname->text,namestr,SESSION_NAME_MAX_LEN);
  nsname->in_use = true;
  nsname->joinable = true; //for now
  return nsname;
}

/**
 * Sets all sessions as unused.
 */
void ServiceProvider::ClearSessions(void)
{
  long i;
  for (i=0; i < SESSION_ENTRIES_COUNT; i++)
  {
    nsnames[i].in_use = false;
  }
}

TbError ServiceProvider::EnumeratePlayers(TbNetworkCallbackFunc callback, void *ptr)
{
  struct TbNetworkPlayerEntry *netplyr;
  TbError result;
  long i;
  result = Lb_OK;
  for (i=0; i < players_count; i++)
  {
    netplyr = &this->players[i];
    callback((struct TbNetworkCallbackData *)netplyr, ptr);
  }
  return result;
}

TbBool ServiceProvider::DecodeAddPlayerMsg(const unsigned char *enc_buf, unsigned long &id, char *msg_str)
{
  const unsigned char *inp;
  inp = enc_buf;
  inp += sizeof(uint32_t);
  memcpy(&id, inp, sizeof(uint32_t));
  if (msg_str == NULL)
    return true;
  inp += sizeof(uint32_t);
  strcpy(msg_str, (const char *)inp);
  return true;
}

TbError ServiceProvider::SystemAddPlayerHandler(const void *enc_buf)
{
  char name[NETSP_PLAYER_NAME_MAX_LEN];
  const unsigned char *inp;
  unsigned long plyr_id;
  inp = (const unsigned char *)enc_buf;
  inp += sizeof(uint32_t);
  memcpy(&plyr_id, inp, sizeof(uint32_t));
  inp += sizeof(uint32_t);
  net_copy_name_string(name,(const char *)inp,NETSP_PLAYER_NAME_MAX_LEN);
  if (AddPlayer(plyr_id, name, 0, 0) == Lb_OK)
    return Lb_OK;
  return Lb_FAIL;
}

TbError ServiceProvider::SystemDeletePlayerHandler(const void *enc_buf)
{
  const unsigned char *inp;
  unsigned long plyr_id;
  CheckForDeletedHost(enc_buf);
  inp = (const unsigned char *)enc_buf;
  inp += sizeof(uint32_t);
  memcpy(&plyr_id, inp, sizeof(uint32_t));
  if (DeletePlayer(plyr_id) == Lb_OK)
    return Lb_OK;
  return Lb_FAIL;
}

TbError ServiceProvider::CheckForDeletedHost(const void *enc_buf)
{
  struct TbNetworkPlayerEntry *netplyr;
  const unsigned char *inp;
  unsigned long plyr_id;
  //unsigned long idx1;
  TbBool got;
  long i;
  inp = (const unsigned char *)enc_buf;
  inp += sizeof(uint32_t);
  memcpy(&plyr_id, inp, sizeof(uint32_t));


  //TODO NET CheckForDeletedHost
  got = 0;
  for (i=0; i < players_count; i++)
  {
    netplyr = &this->players[i];
    if ((plyr_id == netplyr->id) && (netplyr->is_active))
    {
      //idx1 = i;
      got = 1;
      break;
    }
  }
  if (!got)
    return Lb_OK;
/*
  got = 0;
  for (i=0; i < players_count; i++)
  {
    netplyr = &this->players[i];
    if (!netplyr->is_active)
    {
      if ( got )
      {
        if ( ebp0 > *(_DWORD *)&v5->session_identifier[1] )
        {
          ebp0 = *(_DWORD *)&v5->session_identifier[1];
          idx2 = i;
        }
      } else
      {
        got = 1;
        idx2 = i;
        ebp0 = *(_DWORD *)&v5->session_identifier[1];
      }
    }
  }

  if ( got )
  {
    netplyr = &this->players[idx1];
    netplyr->is_active = 1;
    netplyr = &this->players[idx1];
    netplyr->is_active = 0;
    if (recvCallbacks->deleteMsg != NULL)
      (recvCallbacks->deleteMsg(ebp0, *(_DWORD *)&this1[1].nsnames[23].field_9[16]);
  }
*/
  return Lb_OK;
}

TbError ServiceProvider::LookForSystemMessages(void)
{
  return this->Receive(0xB6);
}

TbError ServiceProvider::BroadcastSystemMessage(void *enc_msg)
{
  struct TbNetworkPlayerEntry *netplyr;
  unsigned char messageType;
  unsigned long id;
  TbError result;
  unsigned char *inp;
  long i;
  inp = (unsigned char *)enc_msg;
  messageType = 0;
  this->DecodeMessageStub(inp, NULL, &messageType, NULL);
  inp += sizeof(uint32_t);
  if ( (messageType < 1) || ((messageType > 1) && (messageType != 2)) )
  {
    WARNLOG("invalid message type: %02X", (int)messageType);
    return Lb_FAIL;
  } else
  {
    memcpy(&id, inp, sizeof(uint32_t));
  }
  inp += sizeof(uint32_t);
  result = Lb_OK;
  for (i=0; i < this->players_count; i++)
  {
    netplyr = &this->players[i];
    if ((netplyr->id != this->localPlayerId) && (netplyr->id != id))
    {
      if (this->SendMessage(netplyr->id,inp,1) != Lb_OK)
      {
        WARNLOG("failure while sending message");
        result = Lb_FAIL;
      }
    }
  }
  return result;
}

TbError ServiceProvider::EnumeratePlayersForSessionRunning(TbNetworkCallbackFunc callback, void *a2)
{
  if (this->LookForSystemMessages() != Lb_OK)
  {
    WARNLOG("failure while looking for system messages");
    return Lb_FAIL;
  }
  if (this->EnumeratePlayers(callback, a2) != Lb_OK)
  {
    WARNLOG("failure while enumerating players");
    return Lb_FAIL;
  }
  return Lb_OK;
}

TbError ServiceProvider::EnableNewPlayers(TbBool allow)
{
  return Lb_OK;
}

/******************************************************************************/
