/******************************************************************************/
// Bullfrog Engine Emulation Library - for use to remake classic games like
// Syndicate Wars, Magic Carpet or Dungeon Keeper.
/******************************************************************************/
/** @file bflib_netsp.hpp
 *     Header file for bflib_netsp.cpp.
 * @par Purpose:
 *     Network ServiceProvider class declaration.
 * @par Comment:
 *     Just a header file - #defines, typedefs, function prototypes etc.
 * @author   Tomasz Lis
 * @date     08 Mar 2009 - 09 Aug 2009
 * @par  Copying and copyrights:
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 */
/******************************************************************************/
#ifndef BFLIB_NETSP_H
#define BFLIB_NETSP_H

#include "bflib_basics.h"
#include "bflib_netsession.h"

#include "globals.h"
/******************************************************************************/
class ServiceProvider {
private:
    unsigned long nextSessionId;
protected:
    //see if these can be moved to private later
    bool started;
    unsigned long players_count;
    struct TbNetworkPlayerEntry players[NETSP_PLAYERS_COUNT];
    unsigned long nextPlayerId;
    unsigned long localPlayerId; //local player ID

    TbError Initialise(struct ReceiveCallbacks *nCallbacks, void *context);

    //session management
    struct TbNetworkSessionNameEntry *AddSession(unsigned long sess_id, const char *namestr);
    void ClearSessions(void);
    long SessionIndex(unsigned long sess_id);

    TbError EnumeratePlayers(TbNetworkCallbackFunc callback, void *context);
    long PlayerIndex(unsigned long plyr_id);
    TbError AddPlayer(unsigned long plyr_id, const char *namestr, unsigned long player_flags, unsigned long unused_param);
    TbError DeletePlayer(unsigned long plyr_id);

    /**
     * Reads a message from some player.
     * @param playerId Holds the player that the message was sent from on return.
     * @param msgBuffer The buffer of the message. This must be at least the initial value of len bytes.
     * @param len The initial value specifies the maximum size of the message that may be received. It is
     * modified to contain the actual length of the message (including header).
     * @return True if a message was read (which implies there may be more to read).
     */
    virtual bool ReadMessage(uint32_t * playerId, void * msgBuffer, uint32_t * len) = 0;

    /**
     * Same as ReadMessage but does not remove the received message (if any), which means it can still be Read.
     */
    virtual bool PeekMessage(uint32_t * playerId, void * msgBuffer, uint32_t * len) = 0;

    /**
     * Sends a message to a specific player.
     * @param playerId The player that the message should be sent to.
     * @param msgBuffer The buffer of the message.
     * @param Not sure... Anyway, it can be deduced from message type.
     * @return Whether operation was a success or a failure.
     */
    virtual TbError SendMessage(unsigned long playerId, void * msgBuffer, unsigned char) = 0;
public:
    ServiceProvider();
    virtual ~ServiceProvider();
    static void DecodeMessageStub(const void *enc_msg, uint32_t *a2, unsigned char *a3, uint32_t *a4);
    TbError Send(unsigned long a1, void *a2);
    TbError Receive(unsigned long a1);
    TbBool DecodeAddPlayerMsg(const unsigned char *enc_buf, unsigned long &id, char *msg_str);
    TbError SystemAddPlayerHandler(const void *enc_buf);
    TbError SystemDeletePlayerHandler(const void *enc_buf);
    TbError CheckForDeletedHost(const void *enc_buf);
    TbError LookForSystemMessages(void);
    TbError BroadcastSystemMessage(void *enc_msg);
    TbError EnumeratePlayersForSessionRunning(TbNetworkCallbackFunc callback, void *);
    virtual TbError EnableNewPlayers(TbBool allow);
    virtual TbError Start(struct TbNetworkSessionNameEntry *, char *, void *) = 0;
    virtual TbError Start(char *, char *, unsigned long, void *) = 0;
    virtual TbError Stop(void) = 0;
    virtual TbError Enumerate(TbNetworkCallbackFunc sessionCb, void * ptr) = 0;
    virtual TbError Enumerate(struct TbNetworkSessionNameEntry * sessionEntry, TbNetworkCallbackFunc playerCb, void * ptr) = 0;
    virtual TbError Init(struct ReceiveCallbacks *, void *) = 0;
    virtual TbError Release(void);
    virtual TbError ChangeSettings(unsigned long, void *) = 0;
    virtual void update() = 0; //in case SP needs execution time once per frame

    struct TbNetworkSessionNameEntry nsnames[SESSION_ENTRIES_COUNT];
    unsigned long reference_count;
    unsigned long status_flags;
    char session_identifier[32];
    struct ReceiveCallbacks *recvCallbacks;
    void *callback_context;
};

/******************************************************************************/
extern class ServiceProvider *spPtr;
/******************************************************************************/
#endif
