# Contributing to nvim-lspconfig

## Requirements

- [Lint requirements](#lint)
- Documentation is generated by `scripts/docgen.lua`.
  - Only works on linux and macOS

## Scope of nvim-lspconfig

The purpose of nvim-lspconfig is to provide configuration so that users can activate LSP with a single `vim.lsp.enable('foo')` call.
It must not provide its own "framework". Any "framework" or "util" code must be upstreamed to Nvim core.

## Pull requests (PRs)

- Mark your pull request as "draft" until it's ready for review.
- Avoid cosmetic changes to unrelated files in the same commit.
- Use a **rebase workflow** for small PRs.
  - After addressing review comments, it's fine to rebase and force-push.

## New config

### Criteria

New configs must meet these criteria (to avoid spam/quasi-marketing/vanity projects):

- GitHub Stars: The server repository should have at least 100 stars, or some other evidence (such as vscode marketplace downloads) that the LSP server is reasonably popular and is not spam/quasi-marketing/vanity projects.
- Provide some reference or evidence that the language targeted by the LSP server has an active user base.

This helps ensure that we only include actively maintained and widely used servers to provide a better experience for
the community.

### Walkthrough

To add a new config, copy an existing config from `lsp/`. Start with `lsp/lua_ls.lua` for a simple a config, or `lsp/jdtls.lua` or `lsp/pyright.lua` for more complex examples.

When choosing a config name, convert dashes (`-`) to underscores (`_`). If the name of the server is a unique name (`pyright`, `clangd`) or a commonly used abbreviation (`zls`), prefer this as the server name. If the server instead follows the pattern x-language-server, prefer the convention `x_ls` (`jsonnet_ls`). 

The minimal config properties are:

* `cmd`: command defined as a string list, where the first item is an executable and following items are its arguments (`--stdio` is common).
  ```lua
  cmd = { 'typescript-language-server', '--stdio' }
  ```
* `filetypes`: list of filetypes that should activate this config.
* `root_markers`: a list of files that mark the root of the project/workspace.
    * See `:help lsp-config`.
    * See `vim.fs.root()`

### Commands

LSP servers may provide custom `workspace/executeCommand` commands. Because LSP does not provide any way for clients to programmatically discover/list these commands, configs may define user commands in Nvim which invoke the `workspace/executeCommand` commands. To keep things maintainable and discoverable, configs must follow these guidelines:

- The server-specific user commands must be buffer-local and must be created in the `on_attach` handler.
  - Be sure to use the `bufnr` passed as the 2nd argument to `on_attach` to refer to the buffer that the server has been attached to, as it may be different from the current buffer!
- The names of these commands must be prefixed with `:Lsp`. This is a crude way to improve "discoverability".
- Do NOT create commands that merely alias existing *code-actions* or *code-lenses*, which are *already* auto-discoverable via the ["gra" keymap](https://neovim.io/doc/user/lsp.html#gra) (or `vim.lsp.buf.code_action()`)
- Use `client:exec_cmd()` (instead of `request(..., 'workspace/executeCommand')`)

### Example

Following is an example new config for `lsp/pyright.lua`:

```lua
---@brief
---
--- https://github.com/microsoft/pyright
---
--- `pyright`, a static type checker and language server for python
return {
  cmd = { 'pyright-langserver', '--stdio' },
  filetypes = { 'python' },
  root_markers = {
    'pyproject.toml',
    'setup.py',
    'setup.cfg',
    'requirements.txt',
    'Pipfile',
    'pyrightconfig.json',
  },
  settings = {
    python = {
      analysis = {
        autoSearchPaths = true,
        useLibraryCodeForTypes = true,
        diagnosticMode = 'workspace',
      },
    },
  },
  on_attach = function(client, bufnr)
    vim.api.nvim_buf_create_user_command(bufnr, 'LspPyrightOrganizeImports', function()
      client:exec_cmd({
        command = 'pyright.organizeimports',
        arguments = { vim.uri_from_bufnr(bufnr) },
      })
    end, {
      desc = 'Organize Imports',
    })
  end,
}
```

### Root marker priority

By default, the [`root_markers` field](https://neovim.io/doc/user/lsp.html#lsp-root_markers) is ordered by priority.
However, configs can specify "equal priority" (since Nvim 0.11.3) by placing names in a nested list.

For example in this config, `'package.json'` and `'tsconfig.json'` have *equal* priority, whereas `'.git'` has lower priority.

```lua
return {
  …
  root_markers = { { 'package.json', 'tsconfig.json' }, { '.git' } }
}
```

Since nvim-lspconfig still supports Nvim older than 0.11.3, avoid the "nested list" form on older versions of Nvim:

```lua
return {
  …
  root_markers = vim.fn.has('nvim-0.11.3') == 1
    and { { 'package.json', 'tsconfig.json' }, { '.git' } }
    or { 'package.json', 'tsconfig.json', '.git' }
}
```

## Rename or deprecate a config

If a config needs to be renamed or deprecated, changes its contents like this:

```lua
---@brief
---
--- Renamed to [systemd_lsp](#systemd_lsp)

---@type vim.lsp.Config
return vim.tbl_extend('force', vim.lsp.config.systemd_lsp, {
  on_init = function(...)
    vim.deprecate('systemd_ls', 'systemd_lsp', '3.0.0', 'nvim-lspconfig', false)
    if vim.lsp.config.systemd_lsp.on_init then
      vim.lsp.config.systemd_lsp.on_init(...)
    end
  end,
})
```

## Commit style

Follow the Neovim core [commit message guidelines](https://github.com/neovim/neovim/blob/master/CONTRIBUTING.md#commit-messages). Examples:

* Adding a new config for "lua_ls":
  ```
  feat: lua_ls
  ```
* Fixing a bug for "lua_ls":
  ```
  fix(lua_ls): update root directory pattern

  Problem:
  Root directory incorrectly prefers "foo".

  Solution:
  Rearrange the root dir definition.
  ```

## Lint

PRs are checked with the following analyzers:
- [luals](https://github.com/LuaLS/lua-language-server)
- [stylua](https://github.com/JohnnyMorganz/StyLua).

To run the linter locally:

    make lint

If using nix, you can use `nix develop` to install these to a local nix shell.

## Generating docs

GitHub Actions automatically generates `configs.md`. Only modify
`scripts/docs_template.md` or the docstrings in the source of the config file.
Do not modify `configs.md` directly.

To preview the generated `configs.md` locally, run `scripts/docgen.lua` from
`nvim` (from the project root):

    HOME=./ nvim --clean -R -Es -V1 +'set rtp+=$PWD' +'luafile scripts/docgen.lua'
