"""HarmonyOS application package name mappings.

Maps user-friendly app names to HarmonyOS bundle names.
These bundle names are used with the 'hdc shell aa start -b <bundle>' command.
"""

# Custom ability names for apps that don't use the default "EntryAbility"
# Maps bundle_name -> ability_name
# Generated by: python test/find_abilities.py
APP_ABILITIES: dict[str, str] = {
    # Third-party apps
    "cn.wps.mobileoffice.hap": "DocumentAbility",
    "com.ccb.mobilebank.hm": "CcbMainAbility",
    "com.dewu.hos": "HomeAbility",
    "com.larus.nova.hm": "MainAbility",
    "com.luna.hm.music": "MainAbility",
    "com.meitu.meitupic": "MainAbility",
    "com.ss.hm.article.news": "MainAbility",
    "com.ss.hm.ugc.aweme": "MainAbility",
    "com.taobao.taobao4hmos": "Taobao_mainAbility",
    "com.tencent.videohm": "AppAbility",
    "com.ximalaya.ting.xmharmony": "MainBundleAbility",
    "com.zhihu.hmos": "PhoneAbility",

    # Huawei system apps
    "com.huawei.hmos.browser": "MainAbility",
    "com.huawei.hmos.calculator": "com.huawei.hmos.calculator.CalculatorAbility",
    "com.huawei.hmos.calendar": "MainAbility",
    "com.huawei.hmos.camera": "com.huawei.hmos.camera.MainAbility",
    "com.huawei.hmos.clock": "com.huawei.hmos.clock.phone",
    "com.huawei.hmos.clouddrive": "MainAbility",
    "com.huawei.hmos.email": "ApplicationAbility",
    "com.huawei.hmos.filemanager": "MainAbility",
    "com.huawei.hmos.health": "Activity_card_entryAbility",
    "com.huawei.hmos.notepad": "MainAbility",
    "com.huawei.hmos.photos": "MainAbility",
    "com.huawei.hmos.screenrecorder": "com.huawei.hmos.screenrecorder.ServiceExtAbility",
    "com.huawei.hmos.screenshot": "com.huawei.hmos.screenshot.ServiceExtAbility",
    "com.huawei.hmos.settings": "com.huawei.hmos.settings.MainAbility",
    "com.huawei.hmos.soundrecorder": "MainAbility",
    "com.huawei.hmos.vassistant": "AiCaptionServiceExtAbility",
    "com.huawei.hmos.wallet": "MainAbility",

    # Huawei services
    "com.huawei.hmsapp.appgallery": "MainAbility",
    "com.huawei.hmsapp.books": "MainAbility",
    "com.huawei.hmsapp.himovie": "MainAbility",
    "com.huawei.hmsapp.hisearch": "MainAbility",
    "com.huawei.hmsapp.music": "MainAbility",
    "com.huawei.hmsapp.thememanager": "MainAbility",
    "com.huawei.hmsapp.totemweather": "com.huawei.hmsapp.totemweather.MainAbility",

    # OHOS system apps
    "com.ohos.callui": "com.ohos.callui.ServiceAbility",
    "com.ohos.contacts": "com.ohos.contacts.MainAbility",
    "com.ohos.mms": "com.ohos.mms.MainAbility",
}

APP_PACKAGES: dict[str, str] = {
    # Social & Messaging
    "微信": "com.tencent.wechat",
    "QQ": "com.tencent.mqq",
    "微博": "com.sina.weibo.stage",
    # E-commerce
    "淘宝": "com.taobao.taobao4hmos",
    "京东": "com.jd.hm.mall",
    "拼多多": "com.xunmeng.pinduoduo.hos",
    "淘宝闪购": "com.taobao.taobao4hmos",
    "京东秒送": "com.jd.hm.mall",
    # Lifestyle & Social
    "小红书": "com.xingin.xhs_hos",
    "知乎": "com.zhihu.hmos",
    # "豆瓣": "com.douban.frodo",  # 未在 hdc 列表中找到
    # Maps & Navigation
    "高德地图": "com.amap.hmapp",
    "百度地图": "com.baidu.hmmap",
    # Food & Services
    "美团": "com.sankuai.hmeituan",
    "美团外卖": "com.meituan.takeaway",
    "大众点评": "com.sankuai.dianping",
    # "肯德基": "com.yek.android.kfc.activitys",  # 未在 hdc 列表中找到
    # Travel
    # "携程": "ctrip.android.view",  # 未在 hdc 列表中找到
    "铁路12306": "com.chinarailway.ticketingHM",
    "12306": "com.chinarailway.ticketingHM",
    # "去哪儿": "com.Qunar",  # 未在 hdc 列表中找到
    # "去哪儿旅行": "com.Qunar",  # 未在 hdc 列表中找到
    "滴滴出行": "com.sdu.didi.hmos.psnger",
    # Video & Entertainment
    "bilibili": "yylx.danmaku.bili",
    "抖音": "com.ss.hm.ugc.aweme",
    "快手": "com.kuaishou.hmapp",
    "腾讯视频": "com.tencent.videohm",
    "爱奇艺": "com.qiyi.video.hmy",
    "芒果TV": "com.mgtv.phone",
    # "优酷视频": "com.youku.phone",  # 未在 hdc 列表中找到
    # "红果短剧": "com.phoenix.read",  # 未在 hdc 列表中找到
    # Music & Audio
    # "网易云音乐": "com.netease.cloudmusic",  # 未在 hdc 列表中找到
    "QQ音乐": "com.tencent.hm.qqmusic",
    "汽水音乐": "com.luna.hm.music",
    "喜马拉雅": "com.ximalaya.ting.xmharmony",
    # Reading
    # "番茄小说": "com.dragon.read",  # 未在 hdc 列表中找到
    # "番茄免费小说": "com.dragon.read",  # 未在 hdc 列表中找到
    # "七猫免费小说": "com.kmxs.reader",  # 未在 hdc 列表中找到
    # Productivity
    "飞书": "com.ss.feishu",
    # "QQ邮箱": "com.tencent.androidqqmail",  # 未在 hdc 列表中找到
    # AI & Tools
    "豆包": "com.larus.nova.hm",
    # Health & Fitness
    # "keep": "com.gotokeep.keep",  # 未在 hdc 列表中找到
    # "美柚": "com.lingan.seeyou",  # 未在 hdc 列表中找到
    # News & Information
    # "腾讯新闻": "com.tencent.news",  # 未在 hdc 列表中找到
    "今日头条": "com.ss.hm.article.news",
    # Real Estate
    # "贝壳找房": "com.lianjia.beike",  # 未在 hdc 列表中找到
    # "安居客": "com.anjuke.android.app",  # 未在 hdc 列表中找到
    # Finance
    # "同花顺": "com.hexin.plat.android",  # 未在 hdc 列表中找到
    # Games
    # "星穹铁道": "com.miHoYo.hkrpg",  # 未在 hdc 列表中找到
    # "崩坏：星穹铁道": "com.miHoYo.hkrpg",  # 未在 hdc 列表中找到
    # "恋与深空": "com.papegames.lysk.cn",  # 未在 hdc 列表中找到
    
    # HarmonyOS 第三方应用
    "百度": "com.baidu.baiduapp",
    "阿里巴巴": "com.alibaba.wireless_hmos",
    "WPS": "cn.wps.mobileoffice.hap",
    "企业微信": "com.tencent.wework.hmos",
    "同程": "com.tongcheng.hmos",
    "同程旅行": "com.tongcheng.hmos",
    "唯品会": "com.vip.hosapp",
    "支付宝": "com.alipay.mobile.client",
    "UC浏览器": "com.uc.mobile",
    "闲鱼": "com.taobao.idlefish4ohos",
    "转转": "com.zhuanzhuan.hmoszz",
    "迅雷": "com.xunlei.thunder",
    "搜狗输入法": "com.sogou.input",
    "扫描全能王": "com.intsig.camscanner.hap",
    "美图秀秀": "com.meitu.meitupic",
    "58同城": "com.wuba.life",
    "得物": "com.dewu.hos",
    "海底捞": "com.haidilao.haros",
    "中国移动": "com.droi.tong",
    "中国联通": "com.sinovatech.unicom.ha",
    "国家税务总局": "cn.gov.chinatax.gt4.hm",
    "建设银行": "com.ccb.mobilebank.hm",
    "快手极速版": "com.kuaishou.hmnebula",
    
    # HarmonyOS 系统应用 - 工具类
    "浏览器": "com.huawei.hmos.browser",
    "计算器": "com.huawei.hmos.calculator",
    "日历": "com.huawei.hmos.calendar",
    "相机": "com.huawei.hmos.camera",
    "时钟": "com.huawei.hmos.clock",
    "云盘": "com.huawei.hmos.clouddrive",
    "云空间": "com.huawei.hmos.clouddrive",
    "邮件": "com.huawei.hmos.email",
    "文件管理器": "com.huawei.hmos.filemanager",
    "文件": "com.huawei.hmos.files",
    "查找设备": "com.huawei.hmos.finddevice",
    "查找手机": "com.huawei.hmos.finddevice",
    "录音机": "com.huawei.hmos.soundrecorder",
    "录音": "com.huawei.hmos.soundrecorder",
    "录屏": "com.huawei.hmos.screenrecorder",
    "截屏": "com.huawei.hmos.screenshot",
    "笔记": "com.huawei.hmos.notepad",
    "备忘录": "com.huawei.hmos.notepad",
    
    # HarmonyOS 系统应用 - 媒体类
    "相册": "com.huawei.hmos.photos",
    "图库": "com.huawei.hmos.photos",
    # "视频": "com.huawei.hmos.mediaplayer",  # 未在 hdc 列表中找到，但有 com.huawei.hmsapp.himovie
    
    # HarmonyOS 系统应用 - 通讯类
    "联系人": "com.ohos.contacts",
    "通讯录": "com.ohos.contacts",
    "短信": "com.ohos.mms",
    "信息": "com.ohos.mms",
    "电话": "com.ohos.callui",
    "拨号": "com.ohos.callui",
    
    # HarmonyOS 系统应用 - 设置类
    "设置": "com.huawei.hmos.settings",
    "系统设置": "com.huawei.hmos.settings",
    "AndroidSystemSettings": "com.huawei.hmos.settings",
    "Android System Settings": "com.huawei.hmos.settings",
    "Android  System Settings": "com.huawei.hmos.settings",
    "Android-System-Settings": "com.huawei.hmos.settings",
    "Settings": "com.huawei.hmos.settings",
    
    # HarmonyOS 系统应用 - 生活服务
    "健康": "com.huawei.hmos.health",
    "运动健康": "com.huawei.hmos.health",
    "地图": "com.huawei.hmos.maps.app",
    "华为地图": "com.huawei.hmos.maps.app",
    "钱包": "com.huawei.hmos.wallet",
    "华为钱包": "com.huawei.hmos.wallet",
    "智慧生活": "com.huawei.hmos.ailife",
    "智能助手": "com.huawei.hmos.vassistant",
    "小艺": "com.huawei.hmos.vassistant",
    
    # HarmonyOS 服务
    "应用市场": "com.huawei.hmsapp.appgallery",
    "华为应用市场": "com.huawei.hmsapp.appgallery",
    "音乐": "com.huawei.hmsapp.music",
    "华为音乐": "com.huawei.hmsapp.music",
    "主题": "com.huawei.hmsapp.thememanager",
    "主题管理": "com.huawei.hmsapp.thememanager",
    "天气": "com.huawei.hmsapp.totemweather",
    "华为天气": "com.huawei.hmsapp.totemweather",
    "视频": "com.huawei.hmsapp.himovie",
    "华为视频": "com.huawei.hmsapp.himovie",
    "阅读": "com.huawei.hmsapp.books",
    "华为阅读": "com.huawei.hmsapp.books",
    "游戏中心": "com.huawei.hmsapp.gamecenter",
    "华为游戏中心": "com.huawei.hmsapp.gamecenter",
    "搜索": "com.huawei.hmsapp.hisearch",
    "华为搜索": "com.huawei.hmsapp.hisearch",
    "指南针": "com.huawei.hmsapp.compass",
    "会员中心": "com.huawei.hmos.myhuawei",
    "我的华为": "com.huawei.hmos.myhuawei",
    "华为会员": "com.huawei.hmos.myhuawei",
}


def get_package_name(app_name: str) -> str | None:
    """
    Get the package name for an app.

    Args:
        app_name: The display name of the app.

    Returns:
        The HarmonyOS bundle name, or None if not found.
    """
    return APP_PACKAGES.get(app_name)


def get_app_name(package_name: str) -> str | None:
    """
    Get the app name from a package name.

    Args:
        package_name: The HarmonyOS bundle name.

    Returns:
        The display name of the app, or None if not found.
    """
    for name, package in APP_PACKAGES.items():
        if package == package_name:
            return name
    return None


def list_supported_apps() -> list[str]:
    """
    Get a list of all supported app names.

    Returns:
        List of app names.
    """
    return list(APP_PACKAGES.keys())
