WEB_DIRECTORY = "./web"

import os
import sys
import hashlib
import importlib.util


def _load_module(py_path: str):
    p = os.path.abspath(py_path)
    if not os.path.isfile(p):
        raise RuntimeError(f"missing file: {p}")

    base = os.path.basename(p).lower()
    is_pkg = base == "__init__.py"
    pkg_dir = os.path.dirname(p)

    name = "mktools_" + hashlib.sha1(p.encode("utf-8", errors="ignore")).hexdigest()[:12]

    if is_pkg:
        spec = importlib.util.spec_from_file_location(
            name, p, submodule_search_locations=[pkg_dir]
        )
    else:
        spec = importlib.util.spec_from_file_location(name, p)

    if spec is None or spec.loader is None:
        raise RuntimeError(f"cannot load spec: {p}")

    m = importlib.util.module_from_spec(spec)
    sys.modules[name] = m
    spec.loader.exec_module(m)
    return m


def _get_maps(m):
    cls_map = getattr(m, "NODE_CLASS_MAPPINGS", None)
    disp_map = getattr(m, "NODE_DISPLAY_NAME_MAPPINGS", None)
    if not isinstance(cls_map, dict):
        raise RuntimeError("NODE_CLASS_MAPPINGS not found or not dict")
    if disp_map is None:
        disp_map = {}
    if not isinstance(disp_map, dict):
        raise RuntimeError("NODE_DISPLAY_NAME_MAPPINGS not dict")
    return cls_map, disp_map


_here = os.path.dirname(os.path.abspath(__file__))

mods = []
mods.append(_load_module(os.path.join(_here, "nodes.py")))

_cam_candidates = ["CameraMove-Pro", "camera_move_pro", "CameraMovePro"]
for d in _cam_candidates:
    p = os.path.join(_here, d, "__init__.py")
    if os.path.isfile(p):
        mods.append(_load_module(p))
        break

_story_init = os.path.join(_here, "mk_storyboard", "__init__.py")
if os.path.isfile(_story_init):
    mods.append(_load_module(_story_init))

NODE_CLASS_MAPPINGS = {}
NODE_DISPLAY_NAME_MAPPINGS = {}

for m in mods:
    cm, dm = _get_maps(m)
    for k, v in cm.items():
        if k in NODE_CLASS_MAPPINGS:
            raise RuntimeError(f"duplicate NODE_CLASS_MAPPINGS key: {k}")
        NODE_CLASS_MAPPINGS[k] = v
    for k, v in dm.items():
        if k not in NODE_DISPLAY_NAME_MAPPINGS:
            NODE_DISPLAY_NAME_MAPPINGS[k] = v

