import { app } from "/scripts/app.js";

/**
 * Canvas Toolkit (fixed9)
 * - 线条按类型自定义：支持任意类型；面板显示类型文本
 *   通过“从选中节点添加规则”来捕获类型关键字（内部保存）
 */

const LS_KEY = "ct_settings_v7";
const LS_KEYS_OLD = ["ct_settings_v6","ct_settings_v5","ct_settings_v4"];
const LS_POS = "ct_fab_pos_v3";
const LS_CANVAS_LABELS = "ct_canvas_labels_v1";

const ID_FAB = "ct-fab";
const ID_PANEL = "ct-panel";
const ID_LABEL_LAYER = "ct-canvas-label-layer";
const ID_ALIGN_PANEL = "ct-align-panel";

const LINK_ICON_PRESETS = ["🦆","🐟","🐳","🐬","🧊","💠","🔷","🧿","✨","⚡","🌊","❄️"];
const LINK_ICON_FLIP = new Set(["🦆","🐟","🐳","🐬"]);

const STICKY_FEATURE_ENABLED = false;
const CANVAS_LABEL_ENABLED = false;

const TITLE_BAR_PRESETS = {
  proMuted: {
    label: "专业灰（低饱和）",
    bar: "#3b3432,#2b2726",
    text: "#d7cbc6",
    gloss: 0.18,
    divider: 0.22,
    radius: 12,
    bleed: 2.2,
  },
  graphite: {
    label: "石墨蓝",
    bar: "#2b3a4a,#1a2430",
    text: "#dbe6f3",
    gloss: 0.22,
    divider: 0.28,
    radius: 12,
    bleed: 2.0,
  },
  teal: {
    label: "冷青",
    bar: "#1f6f78,#1a3f4a",
    text: "#d7f2f4",
    gloss: 0.20,
    divider: 0.25,
    radius: 12,
    bleed: 2.0,
  },
  wine: {
    label: "雾紫红",
    bar: "#6a3d4b,#3a1f28",
    text: "#f0dbe2",
    gloss: 0.20,
    divider: 0.26,
    radius: 12,
    bleed: 2.0,
  },
  clean: {
    label: "简洁黑",
    bar: "#141619,#0f1113",
    text: "#e9eef5",
    gloss: 0.10,
    divider: 0.18,
    radius: 10,
    bleed: 1.8,
  },
};

const NODE_BG_PRESETS = [
  { label: "极光暮蓝", colors: ["#0f2027","#203a43","#2c5364"], angle: 135 },
  { label: "暮霞霓光", colors: ["#f953c6","#b91d73","#4b1248"], angle: 120 },
  { label: "冷海电流", colors: ["#1a2980","#26d0ce"], angle: 135 },
  { label: "熔岩夜色", colors: ["#0f0c29","#302b63","#24243e"], angle: 45 },
  { label: "金属墨影", colors: ["#232526","#414345"], angle: 0 },
  { label: "薄荷极光", colors: ["#43cea2","#185a9d"], angle: 135 },
  { label: "琥珀流金", colors: ["#f7971e","#ffd200"], angle: 30 },
  { label: "星云紫电", colors: ["#5f2c82","#49a09d"], angle: 160 },
];

const DEFAULTS = {
  hoverOnly: true,
  enabled: true,
  showSelected: true,
  neighborDepth: 1,
  showAllKey: "alt",

  effectsEnabled: true,
  fps: 30,
  style: "dual",
  lineStyleGlobal: "auto",
  quality: 12,

  colorMode: "theme",   // theme | followNode | gradient
  themeColor: "#d4af37",
  gradStart: "#d4af37",
  gradEnd: "#fff1b8",

  speed: 1.0,
  beamWidth: 2.0,
  glowWidth: 6.0,
  scanPeriod: 1.35,

  particles: false,
  particleCount: 4,

  nodeBgColor: "#141414",
  nodeBgPresets: [],
  nodeBgAngle: 135,
  nodeBgGradientSpan: 1,
  nodeBgGlobal: false,
  nodeBgPresetDisabled: false,
  nodeStylePreview: true,
  alignHotkeys: true,
  nodeTitleBarColor: "#0f0f0f",
  nodeTitleTextColor: "#e6e6e6",
  nodeTitleGlobal: false,
  nodeTitleBarEnabled: true,
  nodeTitleRadius: 10,
  nodeTitleGloss: 0.28,
  nodeTitleDivider: 0.35,
  nodeTitleBleed: 2,
  nodeTitlePresets: [],
  stickyEnabled: true,
  stickyBg: "#2b2f36,#1e2228",
  stickyBgPresets: [],
  stickyText: "#dbe2ea",
  stickyTitleBar: false,
  stickyMatch: "站贴,sticky note,stickynote,sticky,note,comment",
  canvasLabelFontSize: 16,
  canvasLabelTextColor: "#d9e1ea",
  canvasLabelBg: "#1f2328,#101317",

  compatMode: "auto", // auto | on | off

  chainColorEnabled: true,
  chainPickColor: "#d4af37",
  chainRules: [], // [{id,startNodeId,startSlot,color,enabled}]

  linkIconEnabled: false,
  linkIconText: "🦆",
  linkIconSize: 14,
  linkIconSpeed: 40,
  linkIconSpacing: 28,
  linkIconMaxCount: 8,
  linkIconAutoCycle: false,
  linkIconCyclePeriod: 6,
  linkTextIconAnimate: true,
  linkIconColorEnabled: false,
  linkIconColor: "#ffffff",
  linkIconLoopFade: true,
  perfMode: false,

  unlinkedHighlightEnabled: false,
  unlinkedHighlightColor: "#ffd166",
  unlinkedHighlightWidth: 2,
  unlinkedHighlightPulse: 1.2,
  unlinkedHighlightText: "未连线",
  unlinkedHighlightTextSize: 12,
  unlinkedHighlightTextEnabled: true,
};

function loadSettings() {
  try {
    let raw = localStorage.getItem(LS_KEY);
    if (!raw) {
      for (const key of LS_KEYS_OLD) {
        raw = localStorage.getItem(key);
        if (raw) break;
      }
    }
    if (!raw) return { ...DEFAULTS };
    const v = JSON.parse(raw);
    delete v.titleBarColor;
    delete v.titleTextColor;
    delete v.forceTitleBar;
    delete v.titleBarAlpha;
    delete v.nodeMaterialEnabled;
    delete v.nodeMaterialGlobal;
    delete v.materialPreset;
    delete v.materialIntensity;
    delete v.materialTint;
    delete v.arrow;
    delete v.arrowReverse;
    delete v.bgGradientEnabled;
    delete v.bgGradientStart;
    delete v.bgGradientEnd;
    delete v.bgGradientAngle;
    delete v.nodeGradientEnabled;
    delete v.nodeGradientGlobal;
    delete v.nodeGradientStart;
    delete v.nodeGradientEnd;
    delete v.nodeGradientAngle;
    delete v.nodeGradientPreset;
    delete v.nodeCollapsedWidthScale;
    delete v.linkLabelEnabled;
    delete v.linkLabelSpeed;
    delete v.linkLabelSpacing;
    const chainRules = Array.isArray(v.chainRules)
      ? v.chainRules.filter(r => r && (typeof r.startNodeId === "number" || typeof r.startNodeId === "string"))
      : null;
    delete v.typeRules;
    const next = { ...DEFAULTS, ...v, chainRules: chainRules ?? DEFAULTS.chainRules };
    if(!["theme","followNode","gradient"].includes(next.colorMode)){
      next.colorMode = "theme";
    }
    if(!["alt","shift","ctrl","meta","none"].includes(next.showAllKey)){
      next.showAllKey = DEFAULTS.showAllKey;
    }
    if(typeof next.nodeTitleGlobal !== "boolean") next.nodeTitleGlobal = DEFAULTS.nodeTitleGlobal;
    if(typeof next.nodeTitleBarEnabled !== "boolean") next.nodeTitleBarEnabled = DEFAULTS.nodeTitleBarEnabled;
    if(!Number.isFinite(next.nodeTitleRadius)) next.nodeTitleRadius = DEFAULTS.nodeTitleRadius;
    if(!Number.isFinite(next.nodeTitleGloss)) next.nodeTitleGloss = DEFAULTS.nodeTitleGloss;
    if(!Number.isFinite(next.nodeTitleDivider)) next.nodeTitleDivider = DEFAULTS.nodeTitleDivider;
    if(!Number.isFinite(next.nodeTitleBleed)) next.nodeTitleBleed = DEFAULTS.nodeTitleBleed;
    if(!Array.isArray(next.nodeTitlePresets)) next.nodeTitlePresets = DEFAULTS.nodeTitlePresets.slice();
    if(typeof next.stickyEnabled !== "boolean") next.stickyEnabled = DEFAULTS.stickyEnabled;
    if(typeof next.stickyTitleBar !== "boolean") next.stickyTitleBar = DEFAULTS.stickyTitleBar;
    if(typeof next.stickyBg !== "string") next.stickyBg = DEFAULTS.stickyBg;
    if(!Array.isArray(next.nodeBgPresets)) next.nodeBgPresets = DEFAULTS.nodeBgPresets.slice();
    if(!Array.isArray(next.stickyBgPresets)) next.stickyBgPresets = DEFAULTS.stickyBgPresets.slice();
    if(typeof next.stickyText !== "string") next.stickyText = DEFAULTS.stickyText;
    if(typeof next.stickyMatch !== "string") next.stickyMatch = DEFAULTS.stickyMatch;
    if(!Number.isFinite(next.nodeBgAngle)) next.nodeBgAngle = DEFAULTS.nodeBgAngle;
    if(!Number.isFinite(next.nodeBgGradientSpan)) next.nodeBgGradientSpan = DEFAULTS.nodeBgGradientSpan;
    if(typeof next.nodeBgGlobal !== "boolean") next.nodeBgGlobal = DEFAULTS.nodeBgGlobal;
    if(typeof next.nodeBgPresetDisabled !== "boolean") next.nodeBgPresetDisabled = DEFAULTS.nodeBgPresetDisabled;
    if(typeof next.nodeStylePreview !== "boolean") next.nodeStylePreview = DEFAULTS.nodeStylePreview;
    if(typeof next.alignHotkeys !== "boolean") next.alignHotkeys = DEFAULTS.alignHotkeys;
    if(!Number.isFinite(next.canvasLabelFontSize)) next.canvasLabelFontSize = DEFAULTS.canvasLabelFontSize;
    if(typeof next.canvasLabelTextColor !== "string") next.canvasLabelTextColor = DEFAULTS.canvasLabelTextColor;
    if(typeof next.canvasLabelBg !== "string") next.canvasLabelBg = DEFAULTS.canvasLabelBg;
    if(typeof next.linkIconColorEnabled !== "boolean") next.linkIconColorEnabled = DEFAULTS.linkIconColorEnabled;
    if(typeof next.linkIconColor !== "string") next.linkIconColor = DEFAULTS.linkIconColor;
    if(typeof next.linkIconLoopFade !== "boolean") next.linkIconLoopFade = DEFAULTS.linkIconLoopFade;
    if(typeof next.unlinkedHighlightEnabled !== "boolean") next.unlinkedHighlightEnabled = DEFAULTS.unlinkedHighlightEnabled;
    if(typeof next.unlinkedHighlightColor !== "string") next.unlinkedHighlightColor = DEFAULTS.unlinkedHighlightColor;
    if(!Number.isFinite(next.unlinkedHighlightWidth)) next.unlinkedHighlightWidth = DEFAULTS.unlinkedHighlightWidth;
    if(!Number.isFinite(next.unlinkedHighlightPulse)) next.unlinkedHighlightPulse = DEFAULTS.unlinkedHighlightPulse;
    if(typeof next.unlinkedHighlightText !== "string") next.unlinkedHighlightText = DEFAULTS.unlinkedHighlightText;
    if(!Number.isFinite(next.unlinkedHighlightTextSize)) next.unlinkedHighlightTextSize = DEFAULTS.unlinkedHighlightTextSize;
    if(typeof next.unlinkedHighlightTextEnabled !== "boolean") next.unlinkedHighlightTextEnabled = DEFAULTS.unlinkedHighlightTextEnabled;
    const sm = (next.stickyMatch || "").toLowerCase();
    if(sm && !sm.includes("stickynote") && !sm.includes("sticky note")){
      next.stickyMatch = `${next.stickyMatch},sticky note,stickynote`;
    }
    if(sm && !sm.includes("label")){
      next.stickyMatch = `${next.stickyMatch},label`;
    }
    try { localStorage.setItem(LS_KEY, JSON.stringify(next)); } catch {}
    return next;
  } catch {
    return { ...DEFAULTS };
  }
}
function saveSettings(s) { try { localStorage.setItem(LS_KEY, JSON.stringify(s)); } catch {} }

function loadPos() {
  try {
    const raw = localStorage.getItem(LS_POS);
    if (!raw) return null;
    const p = JSON.parse(raw);
    if (Number.isFinite(p?.x) && Number.isFinite(p?.y)) return p;
  } catch {}
  return null;
}
function savePos(p) { try { localStorage.setItem(LS_POS, JSON.stringify(p)); } catch {} }

function loadCanvasLabels(){
  try{
    const raw = localStorage.getItem(LS_CANVAS_LABELS);
    if(!raw) return [];
    const data = JSON.parse(raw);
    if(!Array.isArray(data)) return [];
    return data.filter(it => it && typeof it.id === "string" && Number.isFinite(it.x) && Number.isFinite(it.y));
  }catch{
    return [];
  }
}

function saveCanvasLabels(list){
  try{
    const safe = Array.isArray(list) ? list : [];
    localStorage.setItem(LS_CANVAS_LABELS, JSON.stringify(safe));
  }catch{}
}

function el(tag, props={}, children=[]) {
  const e = document.createElement(tag);
  Object.assign(e, props);
  for (const c of children) e.appendChild(typeof c === "string" ? document.createTextNode(c) : c);
  return e;
}
function clamp(v,a,b){ return Math.max(a, Math.min(b, v)); }
function hexToRgba(hex,a=1){
  const h=hex.replace("#","").trim();
  const n=parseInt(h.length===3?h.split("").map(x=>x+x).join(""):h,16);
  const r=(n>>16)&255,g=(n>>8)&255,b=n&255;
  return `rgba(${r},${g},${b},${a})`;
}
function hexToRgb(hex){
  if(typeof hex !== "string") return null;
  const h = hex.replace("#","").trim();
  if(!/^[0-9a-fA-F]{3}([0-9a-fA-F]{3})?$/.test(h)) return null;
  const full = (h.length===3) ? h.split("").map(x=>x+x).join("") : h;
  const n = parseInt(full,16);
  return { r:(n>>16)&255, g:(n>>8)&255, b:n&255 };
}
function mixHex(a,b,t=0.5){
  const ra = hexToRgb(a); const rb = hexToRgb(b);
  if(!ra || !rb) return a || b || "#ffffff";
  const k = clamp(t,0,1);
  const r = Math.round(ra.r + (rb.r - ra.r) * k);
  const g = Math.round(ra.g + (rb.g - ra.g) * k);
  const b2 = Math.round(ra.b + (rb.b - ra.b) * k);
  const toHex = (v)=>v.toString(16).padStart(2,"0");
  return `#${toHex(r)}${toHex(g)}${toHex(b2)}`;
}

function ensureFab(openPanelFn) {
  let btn = document.getElementById(ID_FAB);
  if (btn) return btn;

  btn = el("div", { id: ID_FAB, innerText:"CT" });
  Object.assign(btn.style, {
    position:"fixed",
    right:"18px",
    bottom:"150px",
    width:"44px",
    height:"44px",
    display:"flex",
    alignItems:"center",
    justifyContent:"center",
    borderRadius:"14px",
    border:"1px solid rgba(212,175,55,0.55)",
    background:"rgba(15,15,15,0.75)",
    color:"#d4af37",
    fontWeight:"900",
    letterSpacing:"0.5px",
    cursor:"grab",
    zIndex:999999,
    boxShadow:"0 10px 30px rgba(0,0,0,0.55)",
    backdropFilter:"blur(8px)",
    userSelect:"none",
    opacity:"1",
    visibility:"visible",
  });

  const FORCE_FLOAT = true;
  const resolveDockTarget = ()=>{
    const selectors = [
      ".actionbar-container",
      "#topbar",
      ".comfy-menu",
      ".comfyui-toolbar",
      ".comfyui-topbar",
      ".topbar",
      "#topbar-container",
    ];
    for(const sel of selectors){
      const list = Array.from(document.querySelectorAll(sel));
      if(!list.length) continue;
      let best = null;
      let bestTop = Infinity;
      for(const el of list){
        try{
          const rect = el.getBoundingClientRect();
          const style = window.getComputedStyle(el);
          const hidden = rect.width < 20 || rect.height < 20 || style.display === "none" || style.visibility === "hidden" || parseFloat(style.opacity || "1") === 0;
          if(hidden) continue;
          const top = rect.top;
          if(top < bestTop && top < 140){
            bestTop = top;
            best = el;
          }
        }catch{}
      }
      if(best) return best;
    }
    return null;
  };

  const dockToTopbar = ()=>{
    const target = resolveDockTarget();
    if(!target) return false;
    btn.__ct_docked = true;
    Object.assign(btn.style,{
      position:"relative",
      left:"auto",
      top:"auto",
      right:"auto",
      bottom:"auto",
      margin:"0 -8px 0 0",
      width:"32px",
      height:"32px",
      borderRadius:"10px",
      fontSize:"12px",
      lineHeight:"32px",
      cursor:"pointer",
      display:"flex",
      visibility:"visible",
      opacity:"1",
    });
    btn.style.boxShadow = "0 6px 16px rgba(0,0,0,0.35)";
    btn.onclick = ()=>openPanelFn?.();
    try{
      target.insertBefore(btn, target.firstChild);
    }catch{
      try{ target.appendChild(btn); }catch{}
    }
    return true;
  };

  const fallbackToFloating = ()=>{
    btn.__ct_docked = false;
    btn.onclick = null;
    Object.assign(btn.style,{
      position:"fixed",
      right:"18px",
      bottom:"150px",
      left:"auto",
      top:"auto",
      margin:"0",
      cursor:"grab",
      width:"44px",
      height:"44px",
      borderRadius:"14px",
      fontSize:"14px",
      lineHeight:"44px",
      display:"flex",
      visibility:"visible",
      opacity:"1",
    });
    if(!btn.__ct_float_handlers){
      btn.__ct_float_handlers = true;
      let dragging=false, moved=false;
      let startX=0,startY=0, originLeft=0, originTop=0;

      function setPos(x,y){
        btn.style.left=`${x}px`;
        btn.style.top=`${y}px`;
        btn.style.right="auto";
        btn.style.bottom="auto";
        savePos({x,y});
      }

      const saved = loadPos();
      if (saved) {
        const x = clamp(saved.x, 6, window.innerWidth - 50);
        const y = clamp(saved.y, 6, window.innerHeight - 50);
        setPos(x, y);
      }

      btn.addEventListener("dblclick", (e)=>{
        e.preventDefault();
        btn.style.left="auto"; btn.style.top="auto";
        btn.style.right="18px"; btn.style.bottom="150px";
        localStorage.removeItem(LS_POS);
      });

      btn.addEventListener("pointerdown", (e)=>{
        if(btn.__ct_docked) return;
        dragging=true; moved=false;
        btn.setPointerCapture(e.pointerId);
        btn.style.cursor="grabbing";
        const r = btn.getBoundingClientRect();
        startX=e.clientX; startY=e.clientY;
        originLeft=r.left; originTop=r.top;
        e.preventDefault();
      });
      btn.addEventListener("pointermove", (e)=>{
        if(!dragging) return;
        const dx=e.clientX-startX, dy=e.clientY-startY;
        if(Math.abs(dx)+Math.abs(dy)>3) moved=true;
        setPos(clamp(originLeft+dx,6,window.innerWidth-50), clamp(originTop+dy,6,window.innerHeight-50));
      });
      btn.addEventListener("pointerup", (e)=>{
        if(!dragging) return;
        dragging=false;
        btn.style.cursor="grab";
        btn.releasePointerCapture(e.pointerId);
        if(!moved) openPanelFn?.();
      });
    }
    const host = document.body || document.documentElement;
    if(host && btn.parentElement !== host){
      host.appendChild(btn);
    }
  };

  if(FORCE_FLOAT){
    fallbackToFloating();
    return btn;
  }
  if(dockToTopbar()){
    setTimeout(()=>{
      try{
        const rect = btn.getBoundingClientRect();
        const style = window.getComputedStyle(btn);
        const hidden = rect.width < 10 || rect.height < 10 || style.display === "none" || style.visibility === "hidden" || parseFloat(style.opacity || "1") === 0;
        if(hidden){
          fallbackToFloating();
        }
      }catch{
        fallbackToFloating();
      }
    }, 0);
    return btn;
  }
  fallbackToFloating();
  if(!btn.__ct_dock_watch){
    btn.__ct_dock_watch = true;
    let tries = 0;
    const tryDock = ()=>{
      if(btn.__ct_docked) return;
      if(dockToTopbar()){
        btn.__ct_docked = true;
        try{ localStorage.removeItem(LS_POS); }catch{}
      }
      if(++tries > 20){
        clearInterval(timer);
      }
    };
    const timer = setInterval(tryDock, 800);
    const obs = new MutationObserver(()=>tryDock());
    try{ obs.observe(document.body || document.documentElement, {childList:true, subtree:true}); }catch{}
    btn.__ct_dock_stop = ()=>{ try{ obs.disconnect(); }catch{}; clearInterval(timer); };
  }
  if(!btn.__ct_presence_watch){
    btn.__ct_presence_watch = true;
    const ensurePresence = ()=>{
      const host = document.body || document.documentElement;
      if(!btn.isConnected){
        if(!dockToTopbar() && host) host.appendChild(btn);
        return;
      }
      let hidden = false;
      try{
        const rect = btn.getBoundingClientRect();
        const style = window.getComputedStyle(btn);
        hidden = rect.width < 10 || rect.height < 10 ||
          style.display === "none" || style.visibility === "hidden" ||
          parseFloat(style.opacity || "1") < 0.05;
        let p = btn.parentElement;
        let depth = 0;
        while(!hidden && p && depth < 4){
          const ps = window.getComputedStyle(p);
          if(ps.display === "none" || ps.visibility === "hidden" || parseFloat(ps.opacity || "1") < 0.05){
            hidden = true;
            break;
          }
          p = p.parentElement;
          depth++;
        }
      }catch{}
      if(hidden){
        fallbackToFloating();
        if(!btn.isConnected && host) host.appendChild(btn);
      }
    };
    setInterval(ensurePresence, 1200);
  }
  setTimeout(()=>{
    if(!btn.__ct_docked && dockToTopbar()){
      try{ localStorage.removeItem(LS_POS); }catch{}
    }
  }, 800);
  setTimeout(()=>{
    if(!btn.__ct_docked && dockToTopbar()){
      try{ localStorage.removeItem(LS_POS); }catch{}
    }
  }, 2200);
  // If still off-screen (bad saved coords), reset to default
  try{
    const r = btn.getBoundingClientRect();
    if(r.right < 0 || r.left > window.innerWidth || r.bottom < 0 || r.top > window.innerHeight){
      btn.style.left="auto"; btn.style.top="auto";
      btn.style.right="18px"; btn.style.bottom="150px";
      localStorage.removeItem(LS_POS);
    }
  }catch{}
  return btn;
}

function toggle(v, cb){ const t=el("input",{type:"checkbox"}); t.checked=!!v; t.onchange=()=>cb(t.checked); return t; }
function slider(v,min,max,step,cb){
  const wrap=el("div");
  Object.assign(wrap.style,{display:"flex",alignItems:"center",gap:"10px",minWidth:"230px",justifyContent:"flex-end"});
  const s=el("input",{type:"range",min,max,step,value:v}); s.style.width="170px";
  const val=el("div",{innerText:String(v)}); Object.assign(val.style,{width:"44px",textAlign:"right",fontSize:"12px",opacity:"0.85"});
  s.oninput=()=>{ val.innerText=String(s.value); cb(parseFloat(s.value)); };
  wrap.appendChild(s); wrap.appendChild(val);
  return wrap;
}
function color(v,cb){
  const c=el("input",{type:"color",value:v});
  const fire=()=>cb(c.value);
  c.oninput=fire;
  c.onchange=fire;
  c.addEventListener("pointerdown",(e)=>e.stopPropagation());
  c.addEventListener("mousedown",(e)=>e.stopPropagation());
  c.addEventListener("click",(e)=>e.stopPropagation());
  return c;
}
function select(v,opts,cb){
  const s=el("select");
  Object.assign(s.style,{background:"rgba(0,0,0,.35)",color:"rgba(255,255,255,.92)",border:"1px solid rgba(212,175,55,.25)",borderRadius:"8px",padding:"4px 8px"});
  for (const [value,name] of opts) s.appendChild(el("option",{value,innerText:name}));
  s.value=v; s.onchange=()=>cb(s.value);
  return s;
}
function btn(label, fn, tone="normal"){
  const b=el("button",{innerText:label});
  Object.assign(b.style,{
    border:"1px solid rgba(212,175,55,.25)",
    background:tone==="gold"?"rgba(212,175,55,.12)":"rgba(0,0,0,.35)",
    color:tone==="gold"?"#fff1b8":"rgba(255,255,255,.92)",
    borderRadius:"10px",
    padding:"8px 10px",
    cursor:"pointer",
    fontWeight:"900",
  });
  b.onclick=()=>fn();
  return b;
}
function row(panel,label,control){
  const r=el("div");
  Object.assign(r.style,{display:"flex",alignItems:"center",justifyContent:"space-between",gap:"10px",padding:"6px 0"});
  const l=el("div",{innerText:label});
  Object.assign(l.style,{fontSize:"13px",opacity:"0.92"});
  r.appendChild(l); r.appendChild(control);
  panel.appendChild(r);
}
function section(panel,name){
  const h = el("div",{innerText:name});
  Object.assign(h.style,{marginTop:"10px",marginBottom:"6px",fontWeight:"900",color:"rgba(255,255,255,0.95)"});
  panel.appendChild(h);
}
function sep(panel){
  const hr=el("div");
  Object.assign(hr.style,{height:"1px",background:"rgba(255,255,255,0.08)",margin:"10px 0"});
  panel.appendChild(hr);
}

function ensureAlignPanel(actions){
  let panel = document.getElementById(ID_ALIGN_PANEL);
  if(panel) return panel;
  panel = el("div");
  panel.id = ID_ALIGN_PANEL;
  Object.assign(panel.style,{
    position:"fixed",
    right:"18px",
    top:"140px",
    width:"260px",
    zIndex: 10000,
    background:"rgba(10,10,10,0.92)",
    border:"1px solid rgba(212,175,55,0.25)",
    borderRadius:"12px",
    boxShadow:"0 12px 28px rgba(0,0,0,0.45)",
    padding:"12px",
    color:"rgba(255,255,255,0.92)",
    display:"none",
  });
  const header = el("div");
  Object.assign(header.style,{display:"flex",alignItems:"center",justifyContent:"space-between",marginBottom:"8px"});
  const title = el("div",{innerText:"对齐 / 分布"});
  Object.assign(title.style,{fontWeight:"900",fontSize:"13px"});
  const closeBtn = el("button",{innerText:"×"});
  Object.assign(closeBtn.style,{
    border:"1px solid rgba(255,255,255,0.2)",
    background:"rgba(0,0,0,0.35)",
    color:"#fff",
    borderRadius:"8px",
    width:"24px",
    height:"24px",
    cursor:"pointer",
  });
  closeBtn.onclick = ()=>{ panel.style.display="none"; };
  header.appendChild(title);
  header.appendChild(closeBtn);
  panel.appendChild(header);

  const grid = el("div");
  Object.assign(grid.style,{display:"grid",gridTemplateColumns:"1fr 1fr",gap:"8px"});
  grid.appendChild(btn("左对齐", actions.alignLeft));
  grid.appendChild(btn("右对齐", actions.alignRight));
  grid.appendChild(btn("上对齐", actions.alignTop));
  grid.appendChild(btn("下对齐", actions.alignBottom));
  grid.appendChild(btn("水平居中", actions.alignHCenter));
  grid.appendChild(btn("垂直居中", actions.alignVCenter));
  grid.appendChild(btn("水平等间距", actions.distH));
  grid.appendChild(btn("垂直等间距", actions.distV));
  grid.appendChild(btn("对齐+水平分布", actions.alignDistH, "gold"));
  grid.appendChild(btn("对齐+垂直分布", actions.alignDistV, "gold"));
  panel.appendChild(grid);

  const sz = el("div");
  Object.assign(sz.style,{display:"flex",gap:"8px",marginTop:"10px"});
  sz.appendChild(btn("同步大小", actions.syncSizeBoth, "gold"));
  sz.appendChild(btn("同步宽度", actions.syncSizeW));
  sz.appendChild(btn("同步高度", actions.syncSizeH));
  panel.appendChild(sz);

  const arrange = el("div");
  Object.assign(arrange.style,{display:"flex",gap:"8px",marginTop:"8px"});
  arrange.appendChild(btn("自动横排", actions.autoArrangeH));
  arrange.appendChild(btn("自动竖排", actions.autoArrangeV));
  panel.appendChild(arrange);

  document.body.appendChild(panel);
  return panel;
}


function textInput(value, placeholder, cb){
  const i = el("input", { type:"text", value: value ?? "", placeholder: placeholder || "" });
  Object.assign(i.style,{
    background:"rgba(0,0,0,.35)",
    color:"rgba(255,255,255,.92)",
    border:"1px solid rgba(212,175,55,.25)",
    borderRadius:"8px",
    padding:"4px 8px",
    width:"180px",
  });
  i.oninput = () => cb(i.value);
  i.onchange = () => cb(i.value);
  i.addEventListener("pointerdown",(e)=>e.stopPropagation());
  i.addEventListener("mousedown",(e)=>e.stopPropagation());
  i.addEventListener("click",(e)=>e.stopPropagation());
  return i;
}

function gradientPicker(value, cb){
  const wrap = el("div");
  Object.assign(wrap.style,{display:"flex",alignItems:"center",gap:"6px",flexWrap:"wrap",justifyContent:"flex-end"});

  const parse = (val)=>{
    const parts = (val || "").split(",").map(s=>s.trim()).filter(Boolean);
    const s = parts[0] || "#fafafa";
    const m = parts[1] || s;
    const e = parts[2] || m;
    const mode = parts.length >= 3 ? "3" : (parts.length === 2 ? "2" : "1");
    return {s,m,e,mode};
  };

  const modeSel = select("1", [
    ["1","单色"],
    ["2","双段"],
    ["3","三段"],
  ], () => { syncFromInputs(); });

  const start = color("#fafafa", () => syncFromInputs());
  const mid = color("#fafafa", () => syncFromInputs());
  const end = color("#fafafa", () => syncFromInputs());

  const input = el("input", { type:"text", value: value || "#fafafa", placeholder:"#fafafa 或 #FF0000,#0000FF" });
  Object.assign(input.style,{
    background:"rgba(0,0,0,.35)",
    color:"rgba(255,255,255,.92)",
    border:"1px solid rgba(212,175,55,.25)",
    borderRadius:"8px",
    padding:"4px 8px",
    width:"200px",
  });
  input.addEventListener("pointerdown",(e)=>e.stopPropagation());
  input.addEventListener("mousedown",(e)=>e.stopPropagation());
  input.addEventListener("click",(e)=>e.stopPropagation());

  let syncing = false;
  const syncFromValue = (val)=>{
    if(syncing) return;
    syncing = true;
    const {s,m,e,mode} = parse(val);
    modeSel.value = mode;
    start.value = s;
    mid.value = m;
    end.value = e;
    mid.style.display = (mode === "3") ? "" : "none";
    end.style.display = (mode === "1") ? "none" : "";
    input.value = val || s;
    syncing = false;
  };
  const syncFromInputs = ()=>{
    if(syncing) return;
    syncing = true;
    const mode = modeSel.value;
    const s = start.value;
    const m = mid.value || s;
    const e = end.value || m;
    let val = s;
    if(mode === "2") val = `${s},${e}`;
    if(mode === "3") val = `${s},${m},${e}`;
    input.value = val;
    mid.style.display = (mode === "3") ? "" : "none";
    end.style.display = (mode === "1") ? "none" : "";
    cb(val);
    syncing = false;
  };

  input.oninput = () => { cb(input.value); syncFromValue(input.value); };
  input.onchange = () => { cb(input.value); syncFromValue(input.value); };

  wrap.appendChild(modeSel);
  wrap.appendChild(start);
  wrap.appendChild(mid);
  wrap.appendChild(end);
  wrap.appendChild(input);
  syncFromValue(value || "#fafafa");
  return wrap;
}

function parseGradientValue(val){
  const parts = (val || "").split(",").map(s=>s.trim()).filter(Boolean);
  const norm = (p)=>{
    if(/^#([0-9a-fA-F]{6})$/.test(p)) return p;
    const m = /^#([0-9a-fA-F]{3})$/.exec(p);
    if(m){
      return `#${m[1].split("").map(ch=>ch+ch).join("")}`;
    }
    return null;
  };
  const ok = parts.map(norm).filter(Boolean);
  return ok.length ? ok : [];
}

function nodeHasLinks(node, graph){
  if(!node) return false;
  if(Array.isArray(node.inputs)){
    for(const inp of node.inputs){
      if(inp && inp.link != null) return true;
    }
  }
  if(Array.isArray(node.outputs)){
    for(const out of node.outputs){
      if(out && Array.isArray(out.links) && out.links.length) return true;
    }
  }
  const g = graph || node?.graph || app?.graph || null;
  const links = g?.links;
  if(!links) return false;
  const now = (typeof performance !== "undefined" && performance.now) ? performance.now() : Date.now();
  const cache = g.__ct_link_cache;
  let set;
  if(cache && cache.t && (now - cache.t) < 200 && cache.set instanceof Set){
    set = cache.set;
  }else{
    set = new Set();
    if(links instanceof Map){
      for(const link of links.values()){
        if(!link) continue;
        if(link.origin_id != null) set.add(link.origin_id);
        if(link.target_id != null) set.add(link.target_id);
      }
    }else if(Array.isArray(links)){
      for(const link of links){
        if(!link) continue;
        if(link.origin_id != null) set.add(link.origin_id);
        if(link.target_id != null) set.add(link.target_id);
      }
    }else if(typeof links === "object"){
      for(const k in links){
        const link = links[k];
        if(!link) continue;
        if(link.origin_id != null) set.add(link.origin_id);
        if(link.target_id != null) set.add(link.target_id);
      }
    }
    g.__ct_link_cache = { t: now, set };
  }
  const id = node.id;
  return set.has(id) || set.has(String(id)) || set.has(Number(id));
}

function addColorPreset(list, name, value, angle){
  const colors = parseGradientValue(value);
  const next = Array.isArray(list) ? list.slice() : [];
  if(!colors.length) return next;
  const cleanName = (name || "").trim() || `自定义预设${(list?.length || 0) + 1}`;
  const key = colors.join(",");
  const existing = next.find(p => (p?.colors || []).join(",") === key);
  if(existing){
    existing.name = cleanName;
    if(Number.isFinite(angle)) existing.angle = angle;
  }else{
    const entry = { name: cleanName, colors };
    if(Number.isFinite(angle)) entry.angle = angle;
    next.push(entry);
  }
  return next;
}

function addTitlePreset(list, name, payload){
  const next = Array.isArray(list) ? list.slice() : [];
  if(!payload || typeof payload !== "object") return next;
  const bar = typeof payload.bar === "string" ? payload.bar.trim() : "";
  const text = typeof payload.text === "string" ? payload.text.trim() : "";
  if(!bar && !text) return next;
  const cleanName = (name || "").trim() || `自定义预设${(list?.length || 0) + 1}`;
  const key = `${bar}__${text}__${payload.gloss ?? ""}__${payload.divider ?? ""}__${payload.radius ?? ""}__${payload.bleed ?? ""}`;
  const existing = next.find(p => {
    const k = `${p?.bar || ""}__${p?.text || ""}__${p?.gloss ?? ""}__${p?.divider ?? ""}__${p?.radius ?? ""}__${p?.bleed ?? ""}`;
    return k === key;
  });
  if(existing){
    existing.name = cleanName;
  }else{
    next.push({ name: cleanName, ...payload });
  }
  return next;
}

function isStickyNode(node, state){
  if(!STICKY_FEATURE_ENABLED) return false;
  if(!node) return false;
  const type = (node?.type || "").toString().toLowerCase();
  const comfyClass = (node?.comfyClass || "").toString().toLowerCase();
  const ctor = (node?.constructor?.name || "").toString().toLowerCase();
  if(type.includes("label") || comfyClass.includes("label") || ctor === "label"){
    return true;
  }
  const hay = `${node?.title || ""} ${node?.type || ""} ${node?.comfyClass || ""} ${node?.name || ""}`.toLowerCase();
  const match = (state?.stickyMatch || "").split(",").map(s=>s.trim().toLowerCase()).filter(Boolean);
  if(!match.length) return false;
  return match.some(k => hay.includes(k));
}

function isRgthreeAnySwitch(node){
  if(!node) return false;
  const type = (node?.type || "").toString().toLowerCase();
  const comfyClass = (node?.comfyClass || "").toString().toLowerCase();
  const ctor = (node?.constructor?.name || "").toString().toLowerCase();
  return type.includes("any switch") || comfyClass.includes("any switch") || ctor.includes("rgthreeanyswitch");
}

function isCollapsedNode(node){
  return !!(node?.flags?.collapsed || node?.collapsed || node?._collapsed || node?.flags?.minimized || node?.minimized);
}

function isLabelLikeNode(node){
  if(!node) return false;
  const type = (node?.type || "").toString().toLowerCase();
  const comfyClass = (node?.comfyClass || "").toString().toLowerCase();
  const ctor = (node?.constructor?.name || "").toString().toLowerCase();
  if(type.includes("label") || type.includes("sticky") || type.includes("note")) return true;
  if(comfyClass.includes("label") || comfyClass.includes("sticky") || comfyClass.includes("note")) return true;
  if(ctor.includes("label")) return true;
  const props = node?.properties || {};
  const hasTextProps = ("fontSize" in props) && ("fontFamily" in props) && ("fontColor" in props);
  const hasNoIO = (!node?.inputs || node.inputs.length === 0) && (!node?.outputs || node.outputs.length === 0);
  return hasTextProps && hasNoIO;
}

function disableLabelEditing(node){
  if(!node || node.__ct_noedit) return;
  node.__ct_noedit = true;
  node.__ct_edit_backup = {
    onDblClick: node.onDblClick,
    onShowCustomPanelInfo: node.onShowCustomPanelInfo,
  };
  node.onDblClick = function(){ return false; };
  node.onShowCustomPanelInfo = function(){};
}

function getCanvasElement(canvas){
  return canvas?.canvas || canvas?.canvasEl || canvas?.ctx?.canvas || null;
}

function ensureLabelLayer(canvas){
  const el = getCanvasElement(canvas);
  if(!el) return null;
  const host = document.body;
  if(!host) return null;
  let layer = host.querySelector(`#${ID_LABEL_LAYER}`);
  if(!layer){
    layer = document.createElement("div");
    layer.id = ID_LABEL_LAYER;
    Object.assign(layer.style,{
      position:"fixed",
      left:"0px",
      top:"0px",
      width:"0px",
      height:"0px",
      pointerEvents:"auto",
      zIndex: 9999,
      background:"transparent",
    });
    host.appendChild(layer);
  }
  return layer;
}

function graphToScreen(canvas, x, y){
  const scale = canvas?.ds?.scale || 1;
  const off = canvas?.ds?.offset || [0,0];
  return {
    x: (x + (off?.[0] || 0)) * scale,
    y: (y + (off?.[1] || 0)) * scale,
    scale,
  };
}

function screenToGraph(canvas, clientX, clientY){
  const el = getCanvasElement(canvas);
  const rect = el?.getBoundingClientRect?.();
  const scale = canvas?.ds?.scale || 1;
  const off = canvas?.ds?.offset || [0,0];
  const px = (clientX - (rect?.left || 0));
  const py = (clientY - (rect?.top || 0));
  return {
    x: (px / scale) - (off?.[0] || 0),
    y: (py / scale) - (off?.[1] || 0),
  };
}

function setupCanvasLabelOverlay(getCanvas, getState){
  if(!CANVAS_LABEL_ENABLED){
    return {
      addLabel(){},
      refresh(){},
      updatePositions(){},
      setLabels(){},
      getLabels(){ return []; },
      applyToAll(){},
      graphKey: "__ct_canvas_labels",
    };
  }
  let labels = loadCanvasLabels();
  let layer = null;
  let labelMap = new Map();
  let lastKey = "";
  let drawPatched = false;
  const GRAPH_KEY = "__ct_canvas_labels";
  let selectedId = null;

  const getDefaults = ()=>{
    const s = getState?.() || {};
    return {
      fontSize: Number.isFinite(s.canvasLabelFontSize) ? s.canvasLabelFontSize : DEFAULTS.canvasLabelFontSize,
      textColor: (typeof s.canvasLabelTextColor === "string" ? s.canvasLabelTextColor : DEFAULTS.canvasLabelTextColor),
      bg: (typeof s.canvasLabelBg === "string" ? s.canvasLabelBg : DEFAULTS.canvasLabelBg),
    };
  };

  const applyLabelStyle = (el, style)=>{
    const fs = Number.isFinite(style?.fontSize) ? style.fontSize : DEFAULTS.canvasLabelFontSize;
    const tc = (typeof style?.textColor === "string" && style.textColor.startsWith("#")) ? style.textColor : DEFAULTS.canvasLabelTextColor;
    const bg = (typeof style?.bg === "string") ? style.bg : DEFAULTS.canvasLabelBg;
    el.style.fontSize = `${fs}px`;
    el.style.color = tc;
    const colors = parseGradientValue(bg);
    if(colors.length <= 1){
      el.style.background = colors[0] || "transparent";
    }else{
      el.style.background = `linear-gradient(135deg, ${colors.join(",")})`;
    }
  };

  const syncLabels = ()=>{
    const canvas = getCanvas?.();
    layer = ensureLabelLayer(canvas);
    if(!layer) return;
    if(!layer.__ct_forward_events){
      layer.__ct_forward_events = true;
      const forwardPointer = (e)=>{
        if(e.target !== layer) return;
        const canvasEl = getCanvasElement(getCanvas?.());
        if(!canvasEl) return;
        try{
          const evt = new PointerEvent(e.type, e);
          canvasEl.dispatchEvent(evt);
        }catch{}
      };
      const forwardWheel = (e)=>{
        if(e.target !== layer) return;
        const canvasEl = getCanvasElement(getCanvas?.());
        if(!canvasEl) return;
        try{
          const evt = new WheelEvent(e.type, e);
          canvasEl.dispatchEvent(evt);
        }catch{}
      };
      layer.addEventListener("pointerdown", forwardPointer);
      layer.addEventListener("pointermove", forwardPointer);
      layer.addEventListener("pointerup", forwardPointer);
      layer.addEventListener("pointercancel", forwardPointer);
      layer.addEventListener("contextmenu", forwardPointer);
      layer.addEventListener("wheel", forwardWheel, { passive: true });
    }
    const alive = new Set();
    for(const label of labels){
      alive.add(label.id);
      let el = labelMap.get(label.id);
      if(!el){
        el = document.createElement("div");
        el.className = "ct-canvas-label";
        el.contentEditable = "false";
        el.spellcheck = false;
        Object.assign(el.style,{
          position:"absolute",
          minWidth:"40px",
          padding:"6px 10px",
          borderRadius:"8px",
          background:"rgba(20,20,20,0.75)",
          color:"#d9e1ea",
          fontSize:"16px",
          fontWeight:"600",
          letterSpacing:"0.2px",
          whiteSpace:"pre-wrap",
          pointerEvents:"auto",
          userSelect:"text",
          outline:"none",
          boxShadow:"0 6px 18px rgba(0,0,0,0.35)",
          transformOrigin:"top left",
          backdropFilter:"blur(4px)",
          cursor:"grab",
          overflow:"visible",
        });
        const actions = document.createElement("div");
        actions.className = "ct-label-actions";
        Object.assign(actions.style,{
          position:"absolute",
          top:"4px",
          right:"4px",
          display:"flex",
          gap:"6px",
          opacity:"1",
          transition:"opacity 120ms ease",
          pointerEvents:"auto",
          zIndex:"2",
          padding:"2px",
          borderRadius:"8px",
          background:"rgba(0,0,0,0.35)",
        });
        const btnBase = {
          width:"22px",
          height:"22px",
          borderRadius:"8px",
          border:"1px solid rgba(255,255,255,0.18)",
          background:"rgba(0,0,0,0.55)",
          color:"#fff",
          cursor:"pointer",
          fontWeight:"700",
          fontSize:"12px",
          lineHeight:"20px",
          padding:"0",
        };
        const delBtn = document.createElement("button");
        delBtn.innerText = "×";
        Object.assign(delBtn.style, btnBase);
        const colorBtn = document.createElement("button");
        colorBtn.innerText = "🎨";
        Object.assign(colorBtn.style, btnBase);
        actions.appendChild(colorBtn);
        actions.appendChild(delBtn);
        el.appendChild(actions);

        const resizeHandle = document.createElement("div");
        resizeHandle.className = "ct-label-resize";
        Object.assign(resizeHandle.style,{
          position:"absolute",
          right:"-6px",
          bottom:"-6px",
          width:"16px",
          height:"16px",
          borderRadius:"4px",
          border:"1px solid rgba(255,255,255,0.6)",
          background:"rgba(0,0,0,0.65)",
          cursor:"nwse-resize",
          pointerEvents:"auto",
          opacity:"0.85",
          zIndex:"1",
        });
        el.appendChild(resizeHandle);

        const setSelected = (on)=>{
          if(on){
            selectedId = label.id;
            actions.style.opacity = "1";
            el.style.outline = "1px solid rgba(212,175,55,0.55)";
          }else{
            actions.style.opacity = "0";
            el.style.outline = "none";
          }
        };
        el.addEventListener("mouseenter", ()=>{ actions.style.opacity = "1"; });
        el.addEventListener("mouseleave", ()=>{ if(selectedId !== label.id) actions.style.opacity = "1"; });
        el.addEventListener("pointerdown", (e)=>{
          e.stopPropagation();
          if(e.target === resizeHandle || e.target === delBtn || e.target === colorBtn) return;
          selectedId = label.id;
          for(const [_, other] of labelMap.entries()){
            other.style.outline = "none";
            const act = other.querySelector(".ct-label-actions");
            if(act) act.style.opacity = "0";
          }
          setSelected(true);
        });
        const setEditing = (on)=>{
          el.contentEditable = on ? "true" : "false";
          if(on) el.focus();
        };
        el.addEventListener("dblclick", (e)=>{
          e.stopPropagation();
          setEditing(true);
        });
        el.addEventListener("blur", ()=>{
          setEditing(false);
        });
        el.addEventListener("input", ()=>{
          label.text = el.innerText ?? "";
          saveCanvasLabels(labels);
          const graph = getCanvas?.()?.graph;
          if(graph) graph[GRAPH_KEY] = labels;
        });
        el.addEventListener("keydown", (e)=>{
          e.stopPropagation();
        });
        delBtn.addEventListener("click", (e)=>{
          e.stopPropagation();
          try{
            const ok = confirm("删除此标签？");
            if(ok){
              labels = labels.filter(it => it.id !== label.id);
              saveCanvasLabels(labels);
              const graph = getCanvas?.()?.graph;
              if(graph) graph[GRAPH_KEY] = labels;
              if(el?.parentElement) el.parentElement.removeChild(el);
              labelMap.delete(label.id);
            }
          }catch{}
        });
        colorBtn.addEventListener("click", (e)=>{
          e.stopPropagation();
          const defaults = getDefaults();
          let nextText = label.style?.textColor || defaults.textColor;
          let nextBg = label.style?.bg || defaults.bg;
          try{
            const t = prompt("标签文字色 (#RRGGBB)", nextText);
            if(t !== null && t.trim()) nextText = t.trim();
            const b = prompt("标签背景 (#RRGGBB 或 #RRGGBB,#RRGGBB,...)", nextBg);
            if(b !== null && b.trim()) nextBg = b.trim();
          }catch{}
          label.style = { ...(label.style || defaults), textColor: nextText, bg: nextBg };
          applyLabelStyle(el, label.style);
          saveCanvasLabels(labels);
          const graph = getCanvas?.()?.graph;
          if(graph) graph[GRAPH_KEY] = labels;
        });
        el.addEventListener("contextmenu", (e)=>{
          e.preventDefault();
          try{
            const ok = confirm("删除此标签？");
            if(ok){
              labels = labels.filter(it => it.id !== label.id);
              saveCanvasLabels(labels);
              const graph = getCanvas?.()?.graph;
              if(graph) graph[GRAPH_KEY] = labels;
              if(el?.parentElement) el.parentElement.removeChild(el);
              labelMap.delete(label.id);
            }
          }catch{}
        });
        let dragging = false;
        let resizing = false;
        let wasEditing = false;
        let startSize = 0;
        let startX = 0;
        let startY = 0;
        let dragOff = {x:0,y:0};
        let onWinMove = null;
        let onWinUp = null;
        el.addEventListener("pointerdown", (e)=>{
          e.stopPropagation();
          if(e.button !== 0) return;
          const canvas = getCanvas?.();
          if(!canvas) return;
          wasEditing = (el.contentEditable === "true");
          if(wasEditing) return;
          if(e.target === resizeHandle){
            resizing = true;
            const defaults = getDefaults();
            startSize = Number.isFinite(label.style?.fontSize) ? label.style.fontSize : defaults.fontSize;
            startX = e.clientX; startY = e.clientY;
            e.preventDefault();
          }else if(e.target !== delBtn && e.target !== colorBtn){
            dragging = true;
            const g = screenToGraph(canvas, e.clientX, e.clientY);
            dragOff = { x: g.x - label.x, y: g.y - label.y };
            e.preventDefault();
          }
          if(dragging || resizing){
            onWinMove = (ev)=>{
              const canvas = getCanvas?.();
              if(!canvas) return;
              if(dragging){
                const g = screenToGraph(canvas, ev.clientX, ev.clientY);
                label.x = g.x - dragOff.x;
                label.y = g.y - dragOff.y;
                saveCanvasLabels(labels);
                updatePositions(true);
              }
              if(resizing){
                const dx = ev.clientX - startX;
                const dy = ev.clientY - startY;
                const delta = (dx + dy) * 0.25;
                const next = clamp(startSize + delta, 8, 72);
                label.style = { ...(label.style || getDefaults()), fontSize: next };
                applyLabelStyle(el, label.style);
                saveCanvasLabels(labels);
                const graph = getCanvas?.()?.graph;
                if(graph) graph[GRAPH_KEY] = labels;
              }
            };
            onWinUp = ()=>{
              dragging = false;
              resizing = false;
              if(onWinMove){
                window.removeEventListener("pointermove", onWinMove, true);
                onWinMove = null;
              }
              if(onWinUp){
                window.removeEventListener("pointerup", onWinUp, true);
                window.removeEventListener("pointercancel", onWinUp, true);
                onWinUp = null;
              }
            };
            window.addEventListener("pointermove", onWinMove, true);
            window.addEventListener("pointerup", onWinUp, true);
            window.addEventListener("pointercancel", onWinUp, true);
          }
        });
        labelMap.set(label.id, el);
        layer.appendChild(el);
      }
      const defaults = getDefaults();
      const style = label.style || defaults;
      applyLabelStyle(el, style);
      el.innerText = label.text || "标签";
    }
    for(const [id, el] of labelMap.entries()){
      if(!alive.has(id)){
        if(el?.parentElement) el.parentElement.removeChild(el);
        labelMap.delete(id);
      }
    }
  };

  const updatePositions = (force=false)=>{
    const canvas = getCanvas?.();
    if(!canvas) return;
    layer = ensureLabelLayer(canvas);
    if(!layer) return;
    const el = getCanvasElement(canvas);
    const rect = el?.getBoundingClientRect?.();
    const scale = canvas?.ds?.scale || 1;
    const off = canvas?.ds?.offset || [0,0];
    const key = `${rect?.width||0}|${rect?.height||0}|${scale}|${off?.[0]||0}|${off?.[1]||0}`;
    if(!force && key === lastKey) return;
    lastKey = key;
    if(rect){
      layer.style.left = `${rect.left}px`;
      layer.style.top = `${rect.top}px`;
      layer.style.width = `${rect.width}px`;
      layer.style.height = `${rect.height}px`;
    }
    for(const label of labels){
      const el = labelMap.get(label.id);
      if(!el) continue;
      const p = graphToScreen(canvas, label.x, label.y);
      el.style.left = `${p.x}px`;
      el.style.top = `${p.y}px`;
      el.style.transform = `scale(${p.scale})`;
    }
  };

  const addLabel = (text="新建标签")=>{
    const canvas = getCanvas?.();
    if(!canvas) return;
    const el = getCanvasElement(canvas);
    const rect = el?.getBoundingClientRect?.();
    const scale = canvas?.ds?.scale || 1;
    const off = canvas?.ds?.offset || [0,0];
    const gx = ((rect?.width || 0) / 2) / scale - (off?.[0] || 0);
    const gy = ((rect?.height || 0) / 2) / scale - (off?.[1] || 0);
    const id = `lbl_${Date.now().toString(36)}_${Math.random().toString(36).slice(2,7)}`;
    const defaults = getDefaults();
    labels.push({ id, text, x: gx, y: gy, style: { ...defaults } });
    saveCanvasLabels(labels);
    const graph = canvas?.graph;
    if(graph) graph[GRAPH_KEY] = labels;
      syncLabels();
      updatePositions(true);
    const newEl = labelMap.get(id);
    if(newEl){
      selectedId = id;
      const act = newEl.querySelector(".ct-label-actions");
      if(act) act.style.opacity = "1";
      newEl.style.outline = "1px solid rgba(212,175,55,0.55)";
      setTimeout(()=>{ try{ newEl.focus(); }catch{} }, 0);
    }
  };

  const setLabels = (list, opts={})=>{
    labels = Array.isArray(list) ? list : [];
      if(opts.persist !== false){
        saveCanvasLabels(labels);
        const graph = getCanvas?.()?.graph;
        if(graph) graph[GRAPH_KEY] = labels;
      }
      syncLabels();
    updatePositions(true);
  };

  const getLabels = ()=>labels.slice();

  const applyToAll = (stylePatch)=>{
    labels = labels.map(l => ({
      ...l,
      style: { ...(l.style || getDefaults()), ...stylePatch },
    }));
    saveCanvasLabels(labels);
    const graph = getCanvas?.()?.graph;
    if(graph) graph[GRAPH_KEY] = labels;
    syncLabels();
    updatePositions(true);
  };

    const refresh = ()=>{
      const graph = getCanvas?.()?.graph;
      if(graph && Array.isArray(graph[GRAPH_KEY])){
        labels = graph[GRAPH_KEY];
      }else{
        labels = loadCanvasLabels();
      }
      syncLabels();
      updatePositions(true);
    };

  const patchDraw = ()=>{
    if(drawPatched) return;
    const LGraphCanvas = window.LGraphCanvas;
    if(!LGraphCanvas || LGraphCanvas.prototype.__ct_label_layer_patched) return;
    const _draw = LGraphCanvas.prototype.draw;
    if(typeof _draw !== "function") return;
    LGraphCanvas.prototype.__ct_label_layer_patched = true;
    LGraphCanvas.prototype.draw = function(){
      const out = _draw.apply(this, arguments);
      updatePositions(false);
      return out;
    };
    drawPatched = true;
  };

  syncLabels();
  patchDraw();
  window.addEventListener("resize", ()=>updatePositions(true));
  return { addLabel, refresh, updatePositions, setLabels, getLabels, applyToAll, graphKey: GRAPH_KEY };
}

function isStickyGroup(group, state){
  if(!STICKY_FEATURE_ENABLED) return false;
  if(!group) return false;
  const hay = `${group?.title || ""} ${group?.name || ""}`.toLowerCase();
  const match = (state?.stickyMatch || "").split(",").map(s=>s.trim().toLowerCase()).filter(Boolean);
  if(!match.length) return false;
  return match.some(k => hay.includes(k));
}

function drawGroupGradient(ctx, group, gradValue){
  const colors = parseGradientValue(gradValue);
  if(!colors.length) return;
  const pos = group?.pos || [0,0];
  const size = group?.size || [0,0];
  const x = pos[0] ?? 0;
  const y = pos[1] ?? 0;
  const w = size[0] ?? 0;
  const h = size[1] ?? 0;
  if(w <= 0 || h <= 0) return;
  const radius = clamp(group?.round || group?.roundRect || 10, 0, 20);
  ctx.save();
  ctx.beginPath();
  if(typeof ctx.roundRect === "function" && radius > 0){
    ctx.roundRect(x, y, w, h, radius);
  }else{
    ctx.rect(x, y, w, h);
  }
  ctx.clip();
  if(colors.length === 1){
    ctx.fillStyle = colors[0];
  }else{
    const g = ctx.createLinearGradient(x, y, x + w, y + h);
    const denom = colors.length - 1;
    for(let i=0;i<colors.length;i++){
      g.addColorStop(i/denom, colors[i]);
    }
    ctx.fillStyle = g;
  }
  ctx.fillRect(x, y, w, h);
  ctx.restore();
}

function patchLabelDraw(stateRef){
  if(!STICKY_FEATURE_ENABLED) return;
  if(patchLabelDraw._patched) return;
  try{
    const nodes = app?.graph?._nodes || [];
    const labelNode = nodes.find(n => {
      const t = (n?.type || "").toString().toLowerCase();
      const c = (n?.comfyClass || "").toString().toLowerCase();
      const k = (n?.constructor?.name || "").toString().toLowerCase();
      return t.includes("label") || c.includes("label") || k === "label";
    });
    const proto = labelNode?.constructor?.prototype;
    if(!proto || proto.__ct_label_patched) return;
    const _draw = proto.draw;
    if(typeof _draw !== "function") return;
    proto.__ct_label_patched = true;
    proto.draw = function(ctx){
      const s = stateRef?.();
      const isSticky = !!s?.stickyEnabled && isStickyNode(this, s);
      if(!isSticky){
        return _draw.call(this, ctx);
      }
      const stickyBg = typeof s?.stickyBg === "string" ? s.stickyBg : "";
      const stickyText = typeof s?.stickyText === "string" ? s.stickyText : "";
      const colors = parseGradientValue(stickyBg);
      const props = this.properties || {};
      const origBg = props["backgroundColor"];
      const origFont = props["fontColor"];
      if(stickyText) props["fontColor"] = stickyText;
      if(colors.length === 1){
        props["backgroundColor"] = colors[0];
      }else if(colors.length > 1){
        props["backgroundColor"] = "transparent";
      }
      const out = _draw.call(this, ctx);
      if(colors.length > 1){
        ctx.save();
        const angleDeg = parseInt(String(props["angle"] ?? 0)) || 0;
        if(angleDeg){
          const cx = (this.size?.[0] || 0) / 2;
          const cy = (this.size?.[1] || 0) / 2;
          ctx.translate(cx, cy);
          ctx.rotate((angleDeg * Math.PI) / 180);
          ctx.translate(-cx, -cy);
        }
        ctx.globalCompositeOperation = "destination-over";
        const w = this.size?.[0] || 0;
        const h = this.size?.[1] || 0;
        const radius = Number(props["borderRadius"]) || 0;
        ctx.beginPath();
        if(typeof ctx.roundRect === "function" && radius > 0){
          ctx.roundRect(0, 0, w, h, [radius]);
        }else{
          ctx.rect(0, 0, w, h);
        }
        ctx.clip();
        const g = ctx.createLinearGradient(0, 0, w, h);
        const denom = colors.length - 1;
        for(let i=0;i<colors.length;i++){
          g.addColorStop(i/denom, colors[i]);
        }
        ctx.fillStyle = g;
        ctx.fillRect(0, 0, w, h);
        ctx.restore();
      }
      props["backgroundColor"] = origBg;
      props["fontColor"] = origFont;
      return out;
    };
    patchLabelDraw._patched = true;
  }catch(err){
    console.warn("[CanvasToolkit] Label 样式补丁失败", err);
  }
}

function drawNodeTitleBar(ctx, node, barValue, textColor, styleCfg){
  if(!barValue && !textColor) return;
  const lg = window?.LiteGraph;
  const titleH = lg?.NODE_TITLE_HEIGHT || 30;
  const isCollapsed = !!(node?.flags?.collapsed || node?.collapsed);
  const nodeH = node?.size?.[1] || titleH;
  const overlayTitle = !isCollapsed && nodeH <= titleH * 1.1;
  const width = isCollapsed
    ? (node?._collapsed_width || lg?.NODE_COLLAPSED_WIDTH || node?.size?.[0] || 140)
    : (node?.size?.[0] || 140);
  const title = (node?.title ?? node?.name ?? "").toString();
  if(width <= 0 || titleH <= 0) return;
  const colors = parseGradientValue(barValue);
  if(!colors.length && !textColor) return;
  const baseRadius = node?.roundRadius ?? node?.cornerRadius ?? node?.borderRadius ?? (lg?.NODE_CORNER_RADIUS || 8);
  const radius = clamp(styleCfg?.radius ?? baseRadius, 0, 18);
  const gloss = clamp(styleCfg?.gloss ?? 0.28, 0, 0.6);
  const dividerBase = clamp(styleCfg?.divider ?? 0.35, 0, 0.9);
  const divider = overlayTitle ? dividerBase * 0.45 : dividerBase;
  const bleed = clamp(styleCfg?.bleed ?? 1.5, 0, 6);
  const overshoot = (overlayTitle ? Math.min(bleed, 1.0) : bleed) + 1;
  const barH = overlayTitle ? Math.min(titleH, nodeH) : titleH;
  ctx.save();
  const y = overlayTitle ? 0 : -titleH;
  if(colors.length){
    const xBar = -overshoot;
    const yBar = y - overshoot;
    const wBar = width + overshoot * 2;
    const hBar = barH + 2 + overshoot * 2;
    if(colors.length === 1){
      ctx.fillStyle = colors[0];
    }else{
      const g = ctx.createLinearGradient(xBar, yBar, xBar + wBar, yBar + hBar);
      const denom = colors.length - 1;
      for(let i=0;i<colors.length;i++){
        g.addColorStop(i/denom, colors[i]);
      }
      ctx.fillStyle = g;
    }
    ctx.beginPath();
    if(typeof ctx.roundRect === "function" && radius > 0){
      ctx.roundRect(xBar, yBar, wBar, hBar, radius + overshoot);
    }else{
      ctx.rect(xBar, yBar, wBar, hBar);
    }
    ctx.fill();
    // subtle gloss + divider
    if(gloss > 0){
      ctx.save();
      ctx.clip();
      const glossGrad = ctx.createLinearGradient(0, yBar, 0, yBar + Math.max(8, titleH * 0.6));
      glossGrad.addColorStop(0, `rgba(255,255,255,${gloss})`);
      glossGrad.addColorStop(1, "rgba(255,255,255,0)");
      ctx.fillStyle = glossGrad;
      ctx.fillRect(xBar, yBar, wBar, Math.max(8, titleH * 0.6));
      ctx.restore();
    }
    if(divider > 0){
      ctx.save();
      ctx.clip();
      ctx.strokeStyle = `rgba(0,0,0,${divider})`;
      ctx.lineWidth = 1;
      ctx.beginPath();
      ctx.moveTo(xBar, y + titleH + 0.5);
      ctx.lineTo(xBar + wBar, y + titleH + 0.5);
      ctx.stroke();
      ctx.restore();
    }
  }
  if(textColor && title){
    ctx.fillStyle = textColor;
    ctx.font = `bold ${Math.max(10, Math.round(barH * 0.55))}px sans-serif`;
    ctx.textAlign = "left";
    ctx.textBaseline = "middle";
    ctx.fillText(title, 10, y + barH / 2);
  }
  ctx.restore();
}

function drawNodeGradientBg(ctx, node, gradValue, angle, span){
  const colors = parseGradientValue(gradValue);
  if(!colors.length) return;
  const lg = window?.LiteGraph;
  const size = node?.size;
  let w = size?.[0] ?? 140;
  let h = size?.[1] ?? 60;
  const isCollapsed = !!(node?.flags?.collapsed || node?.collapsed || node?._collapsed || node?.flags?.minimized || node?.minimized);
  if(isCollapsed){
    const titleH = lg?.NODE_TITLE_HEIGHT || 30;
    const cw = node?._collapsed_width || lg?.NODE_COLLAPSED_WIDTH || w;
    w = cw;
    h = titleH;
  }
  if(w <= 0 || h <= 0) return;

  ctx.save();
  if(isCollapsed){
    ctx.translate(0, -(lg?.NODE_TITLE_HEIGHT || 30));
  }
  ctx.beginPath();
  if(typeof ctx.roundRect === "function"){
    ctx.roundRect(0, 0, w, h, 10);
  }else{
    ctx.rect(0, 0, w, h);
  }
  ctx.clip();

  if(colors.length === 1){
    ctx.fillStyle = colors[0];
  }else{
    const useDeg = Number.isFinite(angle) ? angle : null;
    const spanFactor = clamp(Number.isFinite(span) ? span : 1, 0.5, 2.5);
    let x0 = 0, y0 = 0, x1 = w, y1 = h;
    if(useDeg !== null){
      const rad = (useDeg % 360) * Math.PI / 180;
      const cx = w / 2, cy = h / 2;
      const len = (Math.sqrt(w*w + h*h) / 2) * spanFactor;
      const dx = Math.cos(rad) * len;
      const dy = Math.sin(rad) * len;
      x0 = cx - dx; y0 = cy - dy;
      x1 = cx + dx; y1 = cy + dy;
    }
    const g = ctx.createLinearGradient(x0, y0, x1, y1);
    const denom = colors.length - 1;
    for(let i=0;i<colors.length;i++){
      g.addColorStop(i/denom, colors[i]);
    }
    ctx.fillStyle = g;
  }
  ctx.fillRect(0, 0, w, h);

  {
    const spanFactor = clamp(Number.isFinite(span) ? span : 1, 0.5, 2.5);
    const core = clamp(0.10 * spanFactor, 0.07, 0.22);
    const glow = clamp(0.36 * spanFactor, 0.22, 0.55);
    const leftGlow = clamp(0.5 - glow, 0.03, 0.45);
    const leftCore = clamp(0.5 - core, 0.08, 0.48);
    const rightCore = clamp(0.5 + core, 0.52, 0.92);
    const rightGlow = clamp(0.5 + glow, 0.55, 0.97);

    ctx.save();
    ctx.globalCompositeOperation = "screen";
    const gHi = ctx.createLinearGradient(0, 0, w, 0);
    gHi.addColorStop(0, "rgba(255,255,255,0.00)");
    gHi.addColorStop(leftGlow, "rgba(255,255,255,0.06)");
    gHi.addColorStop(leftCore, "rgba(255,255,255,0.16)");
    gHi.addColorStop(0.5, "rgba(255,255,255,0.18)");
    gHi.addColorStop(rightCore, "rgba(255,255,255,0.16)");
    gHi.addColorStop(rightGlow, "rgba(255,255,255,0.06)");
    gHi.addColorStop(1, "rgba(255,255,255,0.00)");
    ctx.fillStyle = gHi;
    ctx.fillRect(0, 0, w, h);
    ctx.restore();

    ctx.save();
    ctx.globalCompositeOperation = "multiply";
    const gDark = ctx.createLinearGradient(0, 0, w, 0);
    gDark.addColorStop(0, "rgba(0,0,0,0.14)");
    gDark.addColorStop(0.20, "rgba(0,0,0,0.06)");
    gDark.addColorStop(0.5, "rgba(0,0,0,0.00)");
    gDark.addColorStop(0.80, "rgba(0,0,0,0.06)");
    gDark.addColorStop(1, "rgba(0,0,0,0.14)");
    ctx.fillStyle = gDark;
    ctx.fillRect(0, 0, w, h);
    ctx.restore();
  }
  ctx.restore();
}

function drawUnlinkedHint(ctx, node, color, width, now, text, textSize, showText, pulseSpeed){
  const lg = window?.LiteGraph;
  const titleH = lg?.NODE_TITLE_HEIGHT || 30;
  const isCollapsed = !!(node?.flags?.collapsed || node?.collapsed || node?._collapsed || node?.flags?.minimized || node?.minimized);
  const w = isCollapsed ? (node?._collapsed_width || lg?.NODE_COLLAPSED_WIDTH || node?.size?.[0] || 140) : (node?.size?.[0] || 140);
  const h = isCollapsed ? titleH : (node?.size?.[1] || 60);
  if(w <= 0 || h <= 0) return;
  const baseRadius = node?.roundRadius ?? node?.cornerRadius ?? node?.borderRadius ?? (lg?.NODE_CORNER_RADIUS || 8);
  const radius = clamp(baseRadius, 0, 18);
  const strokeW = clamp(width ?? 2, 1, 6);
  const t = (now ?? performance.now()) / 1000;
  const speed = clamp(Number.isFinite(pulseSpeed) ? pulseSpeed : 1.2, 0.2, 6);
  const pulse = 0.5 + 0.5 * Math.sin(t * Math.PI * 2 * speed);
  const alpha = 0.25 + pulse * 0.45;
  const glow = 6 + pulse * 10;
  ctx.save();
  ctx.restore();

  if(showText && text){
    ctx.save();
    const size = clamp(textSize ?? 12, 9, 18);
    ctx.font = `bold ${size}px sans-serif`;
    ctx.textAlign = "left";
    ctx.textBaseline = "top";
    const fill = (typeof color === "string" && color.startsWith("#"))
      ? hexToRgba(color, 0.9)
      : (color || "#ffd166");
    ctx.fillStyle = fill;
    ctx.shadowColor = (typeof color === "string" ? color : "#ffd166");
    ctx.shadowBlur = 8;
    ctx.fillText(text, w + 8, 6);
    ctx.restore();
  }
}

function setupHighlightPanel(panel){
  const wrap = el("div");
  wrap.id = "ct-highlight-panel";
  panel.appendChild(wrap);

  const render = () => {
    wrap.innerHTML = "";
    const api = window.__ct_highlight_api;
    if(!api){
      section(wrap, "运行高亮（加载中）");
      const tip = el("div",{innerText:"高亮模块未就绪，稍后自动加载。"});
      Object.assign(tip.style,{fontSize:"12px",opacity:"0.7"});
      wrap.appendChild(tip);
      return;
    }
    const cfg = api.getConfig ? api.getConfig() : {};
    const presets = api.getPresets ? api.getPresets() : [];

    section(wrap, "节点聚焦");
    row(wrap, "启用聚焦框", toggle(cfg.highlightEnabled, v=>api.setConfig({highlightEnabled:v})));
    row(wrap, "鼠标触发脉冲", toggle(cfg.breathingEnabled, v=>api.setConfig({breathingEnabled:v})));
    row(wrap, "持续脉冲", toggle(cfg.autoBreathingEnabled, v=>api.setConfig({autoBreathingEnabled:v})));
    row(wrap, "脉冲周期(s)", slider((cfg.breathingPeriodMs ?? 500)/1000, 0.1, 3.0, 0.1, v=>api.setConfig({breathingPeriodSec:v})));
    row(wrap, "脉冲强度(%)", slider(cfg.breathingStrength ?? 100, 0, 100, 1, v=>api.setConfig({breathingStrength:v})));
    row(wrap, "辉光半径", slider(cfg.breathingSizeScale ?? 2, 0.2, 6, 0.1, v=>api.setConfig({breathingSizeScale:v})));
    row(wrap, "辉光配色", gradientPicker(cfg.breathingColor || "#fafafa", v=>api.setConfig({breathingColor:v})));
    if(api.addPreset){
      const presetWrap = el("div");
      Object.assign(presetWrap.style,{display:"flex",alignItems:"center",gap:"8px",justifyContent:"flex-end"});
      const presetInput = el("input", { type:"text", placeholder:"预设名称（可选）" });
      Object.assign(presetInput.style,{
        background:"rgba(0,0,0,.35)",
        color:"rgba(255,255,255,.92)",
        border:"1px solid rgba(212,175,55,.25)",
        borderRadius:"8px",
        padding:"4px 8px",
        width:"150px",
      });
      presetInput.addEventListener("pointerdown",(e)=>e.stopPropagation());
      presetInput.addEventListener("mousedown",(e)=>e.stopPropagation());
      presetInput.addEventListener("click",(e)=>e.stopPropagation());
      const saveBtn = btn("保存为预设", ()=>{
        const liveCfg = api.getConfig ? api.getConfig() : cfg;
        const colors = parseGradientValue(liveCfg?.breathingColor || "");
        if(!colors.length){
          alert("辉光配色无效，请使用 #RRGGBB 或 #RGB");
          return;
        }
        api.addPreset(presetInput.value, colors.join(","));
        api.setConfig?.({ breathingColor: colors.join(",") });
        presetInput.value = "";
        render();
      }, "gold");
      presetWrap.appendChild(presetInput);
      presetWrap.appendChild(saveBtn);
      row(wrap, "保存配色", presetWrap);
    }
    if(presets.length){
      const current = (cfg.breathingColor || "").trim();
      const idxMatch = presets.findIndex(p => (p?.colors || []).join(",") === current);
      const selected = idxMatch >= 0 ? String(idxMatch) : "";
      const opts = [["", "选择预设"], ...presets.map((p,i)=>[String(i), p?.name || `预设${i+1}`])];
      row(wrap, "配色方案", select(selected, opts, v=>{
        const idx = parseInt(v,10);
        if(Number.isFinite(idx) && presets[idx]){
          const val = (presets[idx].colors || []).join(",");
          api.setConfig({breathingColor: val});
          render();
        }
      }));
      row(wrap, "删除预设", btn("删除当前预设", ()=>{
        if(!api.removePreset){
          alert("当前版本不支持删除预设");
          return;
        }
        const liveCfg = api.getConfig ? api.getConfig() : cfg;
        const colors = parseGradientValue(liveCfg?.breathingColor || "");
        if(!colors.length){
          alert("当前辉光配色无效");
          return;
        }
        const res = api.removePreset(colors.join(","));
        if(res === "removed"){
          render();
          return;
        }
        if(res === "readonly"){
          alert("内置预设不可删除");
          return;
        }
        alert("未找到匹配的自定义预设");
      }, "gold"));
    }
    row(wrap, "显示耗时", toggle(cfg.timeEnabled, v=>api.setConfig({timeEnabled:v})));
    sep(wrap);
  };

  render();
  window.addEventListener("ct-highlight-ready", () => render(), { once: false });
}

function ensurePanel(state, onChange, actions) {
  let panel = document.getElementById(ID_PANEL);
  if (panel) return panel;

  panel = el("div", { id: ID_PANEL });
  panel.__ct_state = state;
  panel.__ct_refresh = ()=>{
    const visible = panel.style.display;
    const parent = panel.parentElement;
    const nextState = panel.__ct_state || state;
    const scrollTop = panel.scrollTop;
    if(parent) parent.removeChild(panel);
    const next = ensurePanel(nextState, onChange, actions);
    next.style.display = visible;
    try{ next.scrollTop = scrollTop; }catch{}
    return next;
  };
  Object.assign(panel.style, {
    position:"fixed",
    right:"18px",
    bottom:"210px",
    width:"420px",
    maxHeight:"74vh",
    overflow:"auto",
    borderRadius:"14px",
    border:"1px solid rgba(212,175,55,0.35)",
    background:"rgba(10,10,10,0.90)",
    color:"rgba(255,255,255,0.92)",
    padding:"12px",
    zIndex:999999,
    boxShadow:"0 18px 60px rgba(0,0,0,0.7)",
    backdropFilter:"blur(10px)",
    display:"none",
  });

  const header = el("div");
  Object.assign(header.style,{display:"flex",alignItems:"center",justifyContent:"space-between",gap:"10px",marginBottom:"10px"});
  const title = el("div",{innerText:"画布工具箱（CT）"});
  Object.assign(title.style,{fontWeight:"900",color:"#d4af37"});
  const close = el("button",{innerText:"×"});
  Object.assign(close.style,{
    border:"1px solid rgba(212,175,55,.35)",
    background:"rgba(0,0,0,.35)",
    color:"#d4af37",
    borderRadius:"10px",
    padding:"4px 10px",
    cursor:"pointer",
  });
  close.onclick=()=>panel.style.display="none";
  header.appendChild(title); header.appendChild(close);
  panel.appendChild(header);

  section(panel,"总开关");
  row(panel,"启用插件", toggle(state.enabled, x=>onChange({enabled:x})));
  sep(panel);

  section(panel,"撤销");
  const ur = el("div"); Object.assign(ur.style,{display:"flex",gap:"8px"});
  ur.appendChild(btn("撤销（Ctrl+Z）", actions.undo, "gold"));
  panel.appendChild(ur);
  sep(panel);

  section(panel,"连线显示（降噪）");
  row(panel,"悬停显示连线", toggle(state.hoverOnly, x=>onChange({hoverOnly:x})));
  row(panel,"选中节点也显示连线", toggle(state.showSelected, x=>onChange({showSelected:x})));
  row(panel,"关联层级（0/1/2）", slider(state.neighborDepth,0,2,1, x=>onChange({neighborDepth:x})));
  row(panel,"临时显示全部键", select(state.showAllKey || "alt", [
    ["alt","Alt"],
    ["shift","Shift"],
    ["ctrl","Ctrl"],
    ["meta","Meta"],
    ["none","禁用"],
  ], x=>onChange({showAllKey:x})));
  sep(panel);

  section(panel,"连线动效（创意风格）");
  row(panel,"启用动效", toggle(state.effectsEnabled, x=>onChange({effectsEnabled:x})));
  row(panel,"兼容模式", select(state.compatMode, [
    ["auto","自动"],
    ["on","开启"],
    ["off","关闭"],
  ], x=>onChange({compatMode:x})));
  row(panel,"显示线条图标", toggle(state.linkIconEnabled, x=>onChange({linkIconEnabled:x})));
  row(panel,"文字/图标流动", toggle(state.linkTextIconAnimate, x=>onChange({linkTextIconAnimate:x})));
  const iconInput = el("input",{type:"text",value:(state.linkIconText || ""),placeholder:"例如：🦆 / ★ / 🔷"});
  Object.assign(iconInput.style,{background:"rgba(0,0,0,.35)",color:"rgba(255,255,255,.92)",border:"1px solid rgba(212,175,55,.25)",borderRadius:"8px",padding:"4px 8px",width:"140px"});
  iconInput.onchange=()=>onChange({linkIconText: iconInput.value});
  iconInput.addEventListener("pointerdown",(e)=>e.stopPropagation());
  iconInput.addEventListener("mousedown",(e)=>e.stopPropagation());
  iconInput.addEventListener("click",(e)=>e.stopPropagation());
  row(panel,"图标内容", iconInput);
  const iconPresets = el("div");
  Object.assign(iconPresets.style,{display:"flex",flexWrap:"wrap",gap:"6px",justifyContent:"flex-end"});
  for(const v of LINK_ICON_PRESETS){
    const b = el("button",{innerText:v});
    Object.assign(b.style,{minWidth:"34px",height:"28px",borderRadius:"8px",border:"1px solid rgba(212,175,55,.25)",background:"rgba(0,0,0,.35)",color:"#fff",cursor:"pointer"});
    b.onclick=()=>{
      iconInput.value = v;
      onChange({linkIconText: v});
    };
    iconPresets.appendChild(b);
  }
  row(panel,"图标预设", iconPresets);
  row(panel,"图标大小", slider(state.linkIconSize,8,26,1, x=>onChange({linkIconSize:x})));
  row(panel,"图标流速", slider(state.linkIconSpeed,10,120,2, x=>onChange({linkIconSpeed:x})));
  row(panel,"图标间距", slider(state.linkIconSpacing,12,60,1, x=>onChange({linkIconSpacing:x})));
  row(panel,"每条线最多图标", slider(state.linkIconMaxCount,1,20,1, x=>onChange({linkIconMaxCount:x})));
  row(panel,"图标颜色自定义", toggle(state.linkIconColorEnabled, x=>onChange({linkIconColorEnabled:x})));
  row(panel,"图标颜色", color(state.linkIconColor, x=>onChange({linkIconColor:x})));
  row(panel,"循环过渡", toggle(state.linkIconLoopFade, x=>onChange({linkIconLoopFade:x})));
  row(panel,"图标自动轮播", toggle(state.linkIconAutoCycle, x=>onChange({linkIconAutoCycle:x})));
  row(panel,"轮播周期（秒）", slider(state.linkIconCyclePeriod,2,20,1, x=>onChange({linkIconCyclePeriod:x})));
  row(panel,"动效帧率上限", slider(state.fps,10,60,1, x=>onChange({fps:x})));
  row(panel,"性能模式", toggle(state.perfMode, x=>onChange({perfMode:x})));
  const styleValue = (state.lineStyleGlobal && state.lineStyleGlobal !== "auto")
    ? `line:${state.lineStyleGlobal}`
    : `fx:${state.style}`;
  row(panel,"动效/线条风格", select(styleValue, [
    ["fx:neon","动效：霓虹电缆（高级）"],
    ["fx:fiber","动效：玻璃光纤（质感）"],
    ["fx:pulse","动效：脉冲能量束"],
    ["fx:metal","动效：细金属轨迹"],
    ["fx:beam","动效：能量束（扫光）"],
    ["fx:dual","动效：双纤流光（高级）"],
    ["fx:lightning","动效：闪电电弧（炫）"],
    ["fx:wave","动效：波动（声波）"],
    ["fx:spiral","动效：螺旋（绕线）"],
    ["fx:twist","动效：扭曲（抖动）"],
    ["line:curve","线条：曲线"],
    ["line:arc","线条：弧线"],
    ["line:orthogonal","线条：直角"],
    ["line:dashed","线条：断点虚线"],
  ], (v)=>{
    if(v.startsWith("line:")){
      onChange({lineStyleGlobal: v.slice(5)});
    }else if(v.startsWith("fx:")){
      onChange({style: v.slice(3), lineStyleGlobal: "auto"});
    }else{
      onChange({style: v, lineStyleGlobal: "auto"});
    }
  }));
  row(panel,"质量（采样段）", slider(state.quality,6,32,1, x=>onChange({quality:x})));

  row(panel,"颜色模式", select(state.colorMode, [
    ["theme","黑金主题"],
    ["followNode","跟随节点颜色"],
    ["gradient","自定义渐变（起始/结束）"],
  ], x=>onChange({colorMode:x})));
  row(panel,"主题金色", color(state.themeColor, x=>onChange({themeColor:x})));
  row(panel,"渐变起始", color(state.gradStart, x=>onChange({gradStart:x})));
  row(panel,"渐变结束", color(state.gradEnd, x=>onChange({gradEnd:x})));

  row(panel,"速度", slider(state.speed,0.1,3.0,0.05, x=>onChange({speed:x})));
  row(panel,"主线粗细", slider(state.beamWidth,0.5,5.0,0.1, x=>onChange({beamWidth:x})));
  row(panel,"光晕宽度", slider(state.glowWidth,0.0,16.0,0.5, x=>onChange({glowWidth:x})));
  row(panel,"扫光周期（秒）", slider(state.scanPeriod,0.4,3.0,0.05, x=>onChange({scanPeriod:x})));
  // 方向箭头已移除
  row(panel,"粒子（更吃性能）", toggle(state.particles, x=>onChange({particles:x})));
  row(panel,"粒子数量", slider(state.particleCount,0,16,1, x=>onChange({particleCount:x})));
  sep(panel);

  section(panel,"链路锁色");
  row(panel,"启用链路锁色", toggle(state.chainColorEnabled, x=>onChange({chainColorEnabled:x})));
  row(panel,"链路颜色", color(state.chainPickColor, x=>onChange({chainPickColor:x})));
  const chainOps = el("div"); Object.assign(chainOps.style,{display:"flex",gap:"8px",flexWrap:"wrap"});
  chainOps.appendChild(btn("从选中节点设为起点", actions.addChainRuleFromSelected, "gold"));
  chainOps.appendChild(btn("清空规则", actions.clearChainRules));
  panel.appendChild(chainOps);

  const chainList = el("div"); chainList.id="ct-chain-list";
  Object.assign(chainList.style,{marginTop:"8px",display:"grid",gridTemplateColumns:"minmax(0,1fr) 90px 60px",gap:"8px",alignItems:"center"});
  panel.appendChild(chainList);

  function renderChainRules(){
    chainList.innerHTML="";
    const live = panel.__ct_state || state;
    const rules = Array.isArray(live.chainRules) ? live.chainRules : [];
    for(let i=0;i<rules.length;i++){
      const r = rules[i];
      chainList.appendChild(el("div",{innerText:`规则 ${i+1}`}));
      const c = el("input",{type:"color",value:(r.color || live.chainPickColor || "#d4af37")});
      c.onchange=()=>{
        const next = rules.map((x,idx)=> idx===i ? ({...x, color: c.value}) : x);
        onChange({chainRules: next});
      };
      c.addEventListener("pointerdown",(e)=>e.stopPropagation());
      c.addEventListener("mousedown",(e)=>e.stopPropagation());
      c.addEventListener("click",(e)=>e.stopPropagation());
      Object.assign(c.style,{width:"90px",height:"28px",border:"none",background:"transparent",cursor:"pointer"});
      chainList.appendChild(c);
      chainList.appendChild(btn("删除", ()=>{
        const next = rules.filter((_,idx)=>idx!==i);
        onChange({chainRules: next});
      }));
    }
  }
  renderChainRules();

  sep(panel);

  setupHighlightPanel(panel);

  section(panel,"节点样式（选中节点）");
  row(panel,"实时预览", toggle(state.nodeStylePreview, x=>onChange({nodeStylePreview:x})));
  row(panel,"全局背景", toggle(state.nodeBgGlobal, x=>onChange({nodeBgGlobal:x})));
  row(panel,"背景配色", gradientPicker(state.nodeBgColor, x=>onChange({nodeBgColor:x})));
  row(panel,"背景角度", slider(state.nodeBgAngle,0,360,1, x=>onChange({nodeBgAngle:x})));
  row(panel,"渐变厚度", slider(state.nodeBgGradientSpan,0.5,2.5,0.05, x=>onChange({nodeBgGradientSpan:x})));
  row(panel,"禁用预设", toggle(state.nodeBgPresetDisabled, x=>onChange({nodeBgPresetDisabled:x})));
  if(!state.nodeBgPresetDisabled){
    {
      const curColors = parseGradientValue(state.nodeBgColor).join(",");
      const presetIdx = NODE_BG_PRESETS.findIndex(p => (p.colors || []).join(",") === curColors);
      const selected = presetIdx >= 0 ? String(presetIdx) : "";
      const opts = [["", "高级预设"], ...NODE_BG_PRESETS.map((p,i)=>[String(i), p.label])];
      row(panel,"高级渐变", select(selected, opts, v=>{
        const idx = parseInt(v,10);
        const preset = NODE_BG_PRESETS?.[idx];
        if(preset){
          onChange({nodeBgColor: preset.colors.join(","), nodeBgAngle: preset.angle});
        }
      }));
    }
    if(Array.isArray(state.nodeBgPresets) && state.nodeBgPresets.length){
      const curColors = parseGradientValue(state.nodeBgColor).join(",");
      const presetIdx = state.nodeBgPresets.findIndex(p => (p?.colors || []).join(",") === curColors);
      const selected = presetIdx >= 0 ? String(presetIdx) : "";
      const opts = [["", "选择预设"], ...state.nodeBgPresets.map((p,i)=>[String(i), p?.name || `预设${i+1}`])];
      row(panel,"背景预设", select(selected, opts, v=>{
        const idx = parseInt(v,10);
        const preset = state.nodeBgPresets?.[idx];
        if(preset){
          const val = (preset.colors || []).join(",");
          if(val) onChange({nodeBgColor: val, nodeBgAngle: Number.isFinite(preset.angle) ? preset.angle : state.nodeBgAngle});
        }
      }));
      row(panel,"删除预设", btn("删除当前预设", ()=>{
        const live = panel.__ct_state || state;
        const cur = parseGradientValue(live.nodeBgColor).join(",");
        const idx = live.nodeBgPresets.findIndex(p => (p?.colors || []).join(",") === cur);
        if(idx < 0){
          alert("当前配色未匹配到自定义预设");
          return;
        }
        const next = live.nodeBgPresets.filter((_,i)=>i!==idx);
        onChange({nodeBgPresets: next});
        panel.__ct_refresh?.();
      }, "gold"));
    }
    {
      const presetWrap = el("div");
      Object.assign(presetWrap.style,{display:"flex",alignItems:"center",gap:"8px",justifyContent:"flex-end"});
      const presetInput = el("input", { type:"text", placeholder:"预设名称（可选）" });
      Object.assign(presetInput.style,{
        background:"rgba(0,0,0,.35)",
        color:"rgba(255,255,255,.92)",
        border:"1px solid rgba(212,175,55,.25)",
        borderRadius:"8px",
        padding:"4px 8px",
        width:"150px",
      });
      presetInput.addEventListener("pointerdown",(e)=>e.stopPropagation());
      presetInput.addEventListener("mousedown",(e)=>e.stopPropagation());
      presetInput.addEventListener("click",(e)=>e.stopPropagation());
      const saveBtn = btn("保存背景预设", ()=>{
        const live = panel.__ct_state || state;
        const colors = parseGradientValue(live.nodeBgColor || "");
        if(!colors.length){
          alert("背景配色无效，请使用 #RRGGBB 或 #RGB");
          return;
        }
        const next = addColorPreset(live.nodeBgPresets, presetInput.value, colors.join(","), live.nodeBgAngle);
        onChange({nodeBgPresets: next});
        presetInput.value = "";
        panel.__ct_refresh?.();
      }, "gold");
      presetWrap.appendChild(presetInput);
      presetWrap.appendChild(saveBtn);
      row(panel,"保存背景", presetWrap);
    }
  }
  row(panel,"标题栏启用", toggle(state.nodeTitleBarEnabled, x=>onChange({nodeTitleBarEnabled:x})));
  row(panel,"全局默认标题色", toggle(state.nodeTitleGlobal, x=>onChange({nodeTitleGlobal:x})));
  row(panel,"标题栏风格", select("custom", [
    ["custom","自定义"],
    ...Object.entries(TITLE_BAR_PRESETS).map(([k,v])=>[k,v.label]),
  ], (v)=>{
    const p = TITLE_BAR_PRESETS[v];
    if(p){
      onChange({
        nodeTitleBarColor: p.bar,
        nodeTitleTextColor: p.text,
        nodeTitleGloss: p.gloss,
        nodeTitleDivider: p.divider,
        nodeTitleRadius: p.radius,
        nodeTitleBleed: p.bleed,
      });
    }
  }));
  row(panel,"标题栏配色", gradientPicker(state.nodeTitleBarColor, x=>onChange({nodeTitleBarColor:x})));
  row(panel,"标题文字色", color(state.nodeTitleTextColor, x=>onChange({nodeTitleTextColor:x})));
  row(panel,"标题圆角", slider(state.nodeTitleRadius,4,14,1, x=>onChange({nodeTitleRadius:x})));
  row(panel,"高光强度", slider(state.nodeTitleGloss,0,0.5,0.02, x=>onChange({nodeTitleGloss:x})));
  row(panel,"分隔线强度", slider(state.nodeTitleDivider,0,0.8,0.05, x=>onChange({nodeTitleDivider:x})));
  row(panel,"标题外扩", slider(state.nodeTitleBleed,0,3,0.1, x=>onChange({nodeTitleBleed:x})));
  if(Array.isArray(state.nodeTitlePresets) && state.nodeTitlePresets.length){
    const curKey = `${state.nodeTitleBarColor || ""}__${state.nodeTitleTextColor || ""}__${state.nodeTitleGloss ?? ""}__${state.nodeTitleDivider ?? ""}__${state.nodeTitleRadius ?? ""}__${state.nodeTitleBleed ?? ""}`;
    const presetIdx = state.nodeTitlePresets.findIndex(p => {
      const k = `${p?.bar || ""}__${p?.text || ""}__${p?.gloss ?? ""}__${p?.divider ?? ""}__${p?.radius ?? ""}__${p?.bleed ?? ""}`;
      return k === curKey;
    });
    const selected = presetIdx >= 0 ? String(presetIdx) : "";
    const opts = [["", "标题预设"], ...state.nodeTitlePresets.map((p,i)=>[String(i), p?.name || `预设${i+1}`])];
    row(panel,"标题预设", select(selected, opts, v=>{
      const idx = parseInt(v,10);
      const preset = state.nodeTitlePresets?.[idx];
      if(preset){
        onChange({
          nodeTitleBarColor: preset.bar || state.nodeTitleBarColor,
          nodeTitleTextColor: preset.text || state.nodeTitleTextColor,
          nodeTitleGloss: Number.isFinite(preset.gloss) ? preset.gloss : state.nodeTitleGloss,
          nodeTitleDivider: Number.isFinite(preset.divider) ? preset.divider : state.nodeTitleDivider,
          nodeTitleRadius: Number.isFinite(preset.radius) ? preset.radius : state.nodeTitleRadius,
          nodeTitleBleed: Number.isFinite(preset.bleed) ? preset.bleed : state.nodeTitleBleed,
        });
      }
    }));
  }
  {
    const presetWrap = el("div");
    Object.assign(presetWrap.style,{display:"flex",alignItems:"center",gap:"8px",justifyContent:"flex-end"});
    const presetInput = el("input", { type:"text", placeholder:"预设名称（可选）" });
    Object.assign(presetInput.style,{
      background:"rgba(0,0,0,.35)",
      color:"rgba(255,255,255,.92)",
      border:"1px solid rgba(212,175,55,.25)",
      borderRadius:"8px",
      padding:"4px 8px",
      width:"150px",
    });
    presetInput.addEventListener("pointerdown",(e)=>e.stopPropagation());
    presetInput.addEventListener("mousedown",(e)=>e.stopPropagation());
    presetInput.addEventListener("click",(e)=>e.stopPropagation());
    const saveBtn = btn("保存标题预设", ()=>{
      const live = panel.__ct_state || state;
      const payload = {
        bar: live.nodeTitleBarColor || "",
        text: live.nodeTitleTextColor || "",
        gloss: live.nodeTitleGloss,
        divider: live.nodeTitleDivider,
        radius: live.nodeTitleRadius,
        bleed: live.nodeTitleBleed,
      };
      const next = addTitlePreset(live.nodeTitlePresets, presetInput.value, payload);
      onChange({nodeTitlePresets: next});
      presetInput.value = "";
      panel.__ct_refresh?.();
    }, "gold");
    const delBtn = btn("删除当前标题预设", ()=>{
      const live = panel.__ct_state || state;
      const curKey = `${live.nodeTitleBarColor || ""}__${live.nodeTitleTextColor || ""}__${live.nodeTitleGloss ?? ""}__${live.nodeTitleDivider ?? ""}__${live.nodeTitleRadius ?? ""}__${live.nodeTitleBleed ?? ""}`;
      const idx = live.nodeTitlePresets.findIndex(p => {
        const k = `${p?.bar || ""}__${p?.text || ""}__${p?.gloss ?? ""}__${p?.divider ?? ""}__${p?.radius ?? ""}__${p?.bleed ?? ""}`;
        return k === curKey;
      });
      if(idx < 0){
        alert("当前标题配色未匹配到自定义预设");
        return;
      }
      const next = live.nodeTitlePresets.filter((_,i)=>i!==idx);
      onChange({nodeTitlePresets: next});
      panel.__ct_refresh?.();
    });
    presetWrap.appendChild(presetInput);
    presetWrap.appendChild(saveBtn);
    presetWrap.appendChild(delBtn);
    row(panel,"保存标题", presetWrap);
  }
  sep(panel);
  section(panel,"提醒");
  row(panel,"无连线高亮", toggle(state.unlinkedHighlightEnabled, x=>onChange({unlinkedHighlightEnabled:x})));
  row(panel,"高亮颜色", color(state.unlinkedHighlightColor, x=>onChange({unlinkedHighlightColor:x})));
  row(panel,"提示文字", textInput(state.unlinkedHighlightText, "未连线", x=>onChange({unlinkedHighlightText:x})));
  row(panel,"文字大小", slider(state.unlinkedHighlightTextSize,9,18,1, x=>onChange({unlinkedHighlightTextSize:x})));
  row(panel,"显示文字", toggle(state.unlinkedHighlightTextEnabled, x=>onChange({unlinkedHighlightTextEnabled:x})));
  const ns = el("div"); Object.assign(ns.style,{display:"flex",gap:"8px",marginTop:"6px"});
  ns.appendChild(btn("应用到选中节点", actions.applyNodeStyle, "gold"));
  ns.appendChild(btn("清除选中节点样式", actions.clearNodeStyle));
  panel.appendChild(ns);
  sep(panel);

  if(CANVAS_LABEL_ENABLED){
    section(panel,"画布标签（前端元素）");
    const newLabelRow = el("div");
    Object.assign(newLabelRow.style,{display:"flex",gap:"8px",alignItems:"center",flexWrap:"wrap"});
    const labelInput = el("input", { type:"text", placeholder:"标签文本" });
    Object.assign(labelInput.style,{
      background:"rgba(0,0,0,.35)",
      color:"rgba(255,255,255,.92)",
      border:"1px solid rgba(212,175,55,.25)",
      borderRadius:"8px",
      padding:"4px 8px",
      width:"160px",
    });
    labelInput.addEventListener("pointerdown",(e)=>e.stopPropagation());
    labelInput.addEventListener("mousedown",(e)=>e.stopPropagation());
    labelInput.addEventListener("click",(e)=>e.stopPropagation());
    const createBtn = btn("新建画布标签", ()=>{
      const text = labelInput.value?.trim() || "新建标签";
      actions.addLabel(text);
      labelInput.value = "";
    }, "gold");
    newLabelRow.appendChild(labelInput);
    newLabelRow.appendChild(createBtn);
    const tip = el("div",{innerText:"拖动：直接拖拽｜缩放：右下角拖拽｜双击编辑｜点击显示删除/配色"});
    Object.assign(tip.style,{fontSize:"11px",opacity:"0.6"});
    newLabelRow.appendChild(tip);
    panel.appendChild(newLabelRow);
    row(panel,"标签字体大小", slider(state.canvasLabelFontSize,10,40,1, x=>onChange({canvasLabelFontSize:x})));
    row(panel,"标签文字色", color(state.canvasLabelTextColor, x=>onChange({canvasLabelTextColor:x})));
    row(panel,"标签背景", gradientPicker(state.canvasLabelBg, x=>onChange({canvasLabelBg:x})));
    const applyLabelStyleRow = el("div");
    Object.assign(applyLabelStyleRow.style,{display:"flex",gap:"8px"});
    applyLabelStyleRow.appendChild(btn("应用到全部标签", ()=>{
      const stylePatch = {
        fontSize: state.canvasLabelFontSize,
        textColor: state.canvasLabelTextColor,
        bg: state.canvasLabelBg,
      };
      actions.applyLabelStyle?.(stylePatch);
    }, "gold"));
    panel.appendChild(applyLabelStyleRow);
    sep(panel);
  }

  section(panel,"对齐 / 分布 / 同步大小（选中节点）");
  const f8Tip = el("div",{innerText:"F8 快捷打开对齐面板"});
  Object.assign(f8Tip.style,{fontSize:"11px",opacity:"0.6",marginBottom:"6px"});
  panel.appendChild(f8Tip);
  const grid = el("div"); Object.assign(grid.style,{display:"grid",gridTemplateColumns:"1fr 1fr",gap:"8px"});
  grid.appendChild(btn("左对齐", actions.alignLeft));
  grid.appendChild(btn("右对齐", actions.alignRight));
  grid.appendChild(btn("上对齐", actions.alignTop));
  grid.appendChild(btn("下对齐", actions.alignBottom));
  grid.appendChild(btn("水平居中", actions.alignHCenter));
  grid.appendChild(btn("垂直居中", actions.alignVCenter));
  grid.appendChild(btn("水平等间距", actions.distH));
  grid.appendChild(btn("垂直等间距", actions.distV));
  grid.appendChild(btn("对齐+水平分布", actions.alignDistH, "gold"));
  grid.appendChild(btn("对齐+垂直分布", actions.alignDistV, "gold"));
  panel.appendChild(grid);

  const sz = el("div"); Object.assign(sz.style,{display:"flex",gap:"8px",marginTop:"8px"});
  sz.appendChild(btn("同步大小（宽+高）", actions.syncSizeBoth, "gold"));
  sz.appendChild(btn("同步宽度", actions.syncSizeW));
  sz.appendChild(btn("同步高度", actions.syncSizeH));
  panel.appendChild(sz);

  const arrange = el("div"); Object.assign(arrange.style,{display:"flex",gap:"8px",marginTop:"8px"});
  arrange.appendChild(btn("自动横排", actions.autoArrangeH));
  arrange.appendChild(btn("自动竖排", actions.autoArrangeV));
  panel.appendChild(arrange);

  panel.__ct_renderChainRules = renderChainRules;

  document.body.appendChild(panel);
  return panel;
}

// ---------- Graph helpers ----------
function toLinkId(v){
  if(typeof v === "number" && Number.isFinite(v)) return v;
  if(typeof v === "string" && v.trim() !== ""){
    const n = Number(v);
    if(Number.isFinite(n)) return n;
  }
  return null;
}
function nodeIdKey(v){
  return String(v);
}

function getSelectedNodes(canvas){
  if(!canvas) return [];
  const sel = canvas.selected_nodes;
  if(sel && typeof sel==="object") return Object.values(sel);
  const g=canvas.graph;
  if(!g||!g._nodes) return [];
  return g._nodes.filter(n=>n?.selected);
}
function gatherLinkIdsFromNode(node){
  const out=new Set();
  if(!node) return out;
  if(Array.isArray(node.inputs)){
    for(const inp of node.inputs){
      if(!inp) continue;
      const lid = toLinkId(inp.link);
      if(lid !== null) out.add(lid);
      if(Array.isArray(inp.links)) for(const id of inp.links){ const v = toLinkId(id); if(v !== null) out.add(v); }
    }
  }
  if(Array.isArray(node.outputs)){
    for(const o of node.outputs){
      if(!o) continue;
      if(Array.isArray(o.links)) for(const id of o.links){ const v = toLinkId(id); if(v !== null) out.add(v); }
      const lid = toLinkId(o.link);
      if(lid !== null) out.add(lid);
    }
  }
  return out;
}
function getNodeById(graph,id){
  if(!graph) return null;
  if(typeof graph.getNodeById==="function"){
    const direct = graph.getNodeById(id);
    if(direct) return direct;
    const nid = toLinkId(id);
    if(nid !== null){
      const alt = graph.getNodeById(nid);
      if(alt) return alt;
    }
  }
  if(Array.isArray(graph._nodes)) return graph._nodes.find(n=>nodeIdKey(n?.id)===nodeIdKey(id))||null;
  return null;
}
function neighborExpand(graph, seedNodes, depth){
  const nodes=new Set(seedNodes.filter(Boolean));
  if(!graph||depth<=0) return nodes;
  let frontier=new Set(nodes);
  for(let d=0; d<depth; d++){
    const next=new Set();
    for(const n of frontier){
      const linkIds=gatherLinkIdsFromNode(n);
      for(const lid of linkIds){
        const link=graph.links?.[lid];
        if(!link) continue;
        const a=getNodeById(graph, link.origin_id);
        const b=getNodeById(graph, link.target_id);
        if(a && !nodes.has(a)){ nodes.add(a); next.add(a); }
        if(b && !nodes.has(b)){ nodes.add(b); next.add(b); }
      }
    }
    frontier=next;
    if(frontier.size===0) break;
  }
  return nodes;
}
function computeVisibleLinks(canvas,state,showAllOverride){
  const graph=canvas?.graph;
  const links=graph?.links;
  if(!canvas||!graph||!links) return [];
  if(showAllOverride || !state.hoverOnly){
    return Object.keys(links).map(k=>toLinkId(k)).filter(v=>v!==null);
  }
  const hovered=canvas.node_over||null;
  const selected=state.showSelected?getSelectedNodes(canvas):[];
  const seed=[];
  if(hovered) seed.push(hovered);
  for(const s of selected) seed.push(s);
  if(seed.length===0) return [];
  const expanded=neighborExpand(graph, seed, state.neighborDepth);
  const out=new Set();
  for(const n of expanded){
    const lids=gatherLinkIdsFromNode(n);
    for(const id of lids) out.add(id);
  }
  return Array.from(out);
}

// ---------- Chain color helpers ----------
function computeDownstreamLinkSet(graph, startNodeId, startSlot){
  const out = new Set();
  if(!graph || startNodeId === null || typeof startNodeId === "undefined") return out;
  const start = getNodeById(graph, startNodeId);
  if(!start) return out;
  const queue = [start];
  const visitedNodes = new Set();
  const originMap = new Map();

  if(graph?.links){
    for(const k of Object.keys(graph.links)){
      const lid = toLinkId(k);
      const link = graph.links[k];
      if(lid === null || !link) continue;
      const okey = nodeIdKey(link.origin_id);
      if(!originMap.has(okey)) originMap.set(okey, []);
      originMap.get(okey).push(lid);
    }
  }

  function pushLinksFromNode(node, slot){
    if(!node || !Array.isArray(node.outputs)) return;
    for(let i=0;i<node.outputs.length;i++){
      if(slot !== -1 && i !== slot) continue;
      const o = node.outputs[i];
      if(!o) continue;
      if(Array.isArray(o.links)){
        for(const lid of o.links){
          const id = toLinkId(lid);
          if(id === null) continue;
          out.add(id);
          const link = graph.links?.[id];
          const target = link ? getNodeById(graph, link.target_id) : null;
          if(target && !visitedNodes.has(nodeIdKey(target.id))) queue.push(target);
        }
      }else if(typeof o.link !== "undefined" && o.link !== null){
        const lid = toLinkId(o.link);
        if(lid === null) continue;
        out.add(lid);
        const link = graph.links?.[lid];
        const target = link ? getNodeById(graph, link.target_id) : null;
        if(target && !visitedNodes.has(nodeIdKey(target.id))) queue.push(target);
      }
    }
    if(slot === -1){
      const list = originMap.get(nodeIdKey(node.id)) || [];
      for(const lid of list){
        out.add(lid);
        const link = graph.links?.[lid];
        const target = link ? getNodeById(graph, link.target_id) : null;
        if(target && !visitedNodes.has(nodeIdKey(target.id))) queue.push(target);
      }
    }
  }

  while(queue.length){
    const node = queue.shift();
    if(!node || visitedNodes.has(nodeIdKey(node.id))) continue;
    visitedNodes.add(nodeIdKey(node.id));
    const slot = (nodeIdKey(node.id) === nodeIdKey(startNodeId)) ? (startSlot ?? -1) : -1;
    pushLinksFromNode(node, slot);
  }
  return out;
}

function rebuildChainColorMap(graph, chainRules){
  const map = new Map();
  if(!graph || !Array.isArray(chainRules)) return map;
  for(const rule of chainRules){
    if(!rule || rule.enabled === false) continue;
    const startId = rule.startNodeId;
    const startSlot = typeof rule.startSlot === "number" ? rule.startSlot : -1;
    const color = rule.color || "#d4af37";
    const links = computeDownstreamLinkSet(graph, startId, startSlot);
    for(const lid of links){
      map.set(lid, color);
    }
  }
  return map;
}

// Bezier helpers
function bezierPoints(x0,y0,x1,y1){
  const dx=x1-x0, dy=y1-y0;
  const dist=Math.sqrt(dx*dx+dy*dy);
  const offset=Math.min(80, Math.max(30, dist*0.25));
  return {cx1:x0+offset, cy1:y0, cx2:x1-offset, cy2:y1};
}
function bezierPointsScaled(x0,y0,x1,y1, scale=1){
  const dx=x1-x0, dy=y1-y0;
  const dist=Math.sqrt(dx*dx+dy*dy);
  const offset=Math.min(140, Math.max(40, dist*0.25*scale));
  return {cx1:x0+offset, cy1:y0, cx2:x1-offset, cy2:y1};
}
function evalCubic(t,p0,p1,p2,p3){
  const it=1-t;
  return it*it*it*p0 + 3*it*it*t*p1 + 3*it*t*t*p2 + t*t*t*p3;
}
function evalCubicDeriv(t,p0,p1,p2,p3){
  const it=1-t;
  return 3*it*it*(p1-p0) + 6*it*t*(p2-p1) + 3*t*t*(p3-p2);
}
// 方向箭头功能已移除
function getConnPos(canvas,node,isInput,slot){
  try{
    if(canvas && typeof canvas.getConnectionPos==="function"){
      const p=canvas.getConnectionPos(node,isInput,slot);
      if(Array.isArray(p)&&p.length>=2) return p;
    }
  }catch{}
  try{
    if(node && typeof node.getConnectionPos==="function"){
      const p=node.getConnectionPos(isInput,slot);
      if(Array.isArray(p)&&p.length>=2) return p;
    }
  }catch{}
  const x=(node?.pos?.[0]??0)+(isInput?0:(node?.size?.[0]??140));
  const yBase=(node?.pos?.[1]??0)+24;
  const y=yBase+ (slot|0)*14;
  return [x,y];
}

// gradient cache
const gradCache=new Map();
function makeLinearGradient(ctx,x0,y0,x1,y1, stops){
  const key=`${(x0|0)}|${(y0|0)}|${(x1|0)}|${(y1|0)}|${stops.join(",")}`;
  const hit=gradCache.get(key);
  if(hit) return hit;
  const g=ctx.createLinearGradient(x0,y0,x1,y1);
  for(let i=0;i<stops.length;i+=2) g.addColorStop(stops[i], stops[i+1]);
  gradCache.set(key,g);
  if(gradCache.size>700){
    let i=0;
    for(const k of gradCache.keys()){ gradCache.delete(k); if(++i>350) break; }
  }
  return g;
}
function noise01(n){
  const x = Math.sin(n*127.1 + 311.7) * 43758.5453;
  return x - Math.floor(x);
}
const iconStampCache = new Map();
function getIconStamp(text, fontSize, color){
  const font = `${fontSize}px "Segoe UI Emoji", "Apple Color Emoji", "Segoe UI Symbol", sans-serif`;
  const key = `${text}|${font}|${color}`;
  const hit = iconStampCache.get(key);
  if(hit) return hit;
  const c = document.createElement("canvas");
  const g = c.getContext("2d");
  g.font = font;
  const m = g.measureText(text);
  const w = Math.ceil((m.width || fontSize) + 8);
  const h = Math.ceil((fontSize * 1.2) + 8);
  c.width = w; c.height = h;
  g.font = font;
  g.textAlign = "center";
  g.textBaseline = "middle";
  g.strokeStyle = "rgba(0,0,0,0.5)";
  g.lineWidth = Math.max(2, fontSize * 0.22);
  g.fillStyle = color || "#ffffff";
  g.strokeText(text, w/2, h/2);
  g.fillText(text, w/2, h/2);
  const stamp = { canvas: c, w, h };
  iconStampCache.set(key, stamp);
  if(iconStampCache.size > 120){
    let i=0;
    for(const k of iconStampCache.keys()){ iconStampCache.delete(k); if(++i>40) break; }
  }
  return stamp;
}

function resolveLinkBaseColor(canvas, link, state){
  if(state.colorMode==="theme") return state.themeColor;
  if(state.colorMode==="gradient") return null;
  if(state.colorMode==="followNode"){
    const origin=getNodeById(canvas.graph, link.origin_id);
    const c=origin?.color || origin?.bgcolor || null;
    if(typeof c==="string" && c.startsWith("#")) return c;
    return state.themeColor;
  }
  return state.themeColor;
}

function strokeBezier(ctx, x0,y0,cx1,cy1,cx2,cy2,x1,y1){
  ctx.beginPath();
  ctx.moveTo(x0,y0);
  ctx.bezierCurveTo(cx1,cy1,cx2,cy2,x1,y1);
  ctx.stroke();
}

function drawLinkEffect(canvas, ctx, link, state, tNow, overrideColor, styleOverride, curveStyle){
  const origin=getNodeById(canvas.graph, link.origin_id);
  const target=getNodeById(canvas.graph, link.target_id);
  if(!origin||!target) return;

  const start=getConnPos(canvas, origin, false, link.origin_slot);
  const end=getConnPos(canvas, target, true, link.target_slot);
  const x0=start[0], y0=start[1], x1=end[0], y1=end[1];
  const {cx1,cy1,cx2,cy2} = (curveStyle === "arc") ? bezierPointsScaled(x0,y0,x1,y1,1.6) : bezierPoints(x0,y0,x1,y1);

  const baseHex = (overrideColor || resolveLinkBaseColor(canvas, link, state) || state.themeColor);
  const perf = !!state?.perfMode;
  const speed = state.speed || 1;
  const scanPeriod = Math.max(0.2, state.scanPeriod || 1.35);
  const phase = (tNow/1000) * speed;
  const scan = ((tNow/1000) % scanPeriod) / scanPeriod;
  const bw = state.beamWidth;
  const glowW = state.effectsEnabled ? (perf ? Math.min(state.glowWidth || 0, 4) : state.glowWidth) : 0;
  const style = styleOverride || state.style || "dual";
  const seg = clamp(Math.floor((state.quality|0) * (perf ? 0.65 : 1)), 6, 32);

  ctx.save();
  ctx.lineCap="round";
  ctx.lineJoin="round";

  if(glowW>0){
    ctx.lineWidth = bw + glowW;
    if(state.colorMode==="gradient"){
      ctx.strokeStyle = makeLinearGradient(ctx,x0,y0,x1,y1,[0,hexToRgba(state.gradStart,0.16),1,hexToRgba(state.gradEnd,0.16)]);
    }else{
      ctx.strokeStyle = hexToRgba(baseHex,0.20);
    }
    strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
  }

  const band=0.10;
  function mainStrokeStyle(alphaMid=0.92, scanValue=scan){
    const s = scanValue;
    const a0=clamp(s-band,0,1), a1=clamp(s,0,1), a2=clamp(s+band,0,1);
    if(state.colorMode==="gradient"){
      return makeLinearGradient(ctx,x0,y0,x1,y1,[
        0,hexToRgba(state.gradStart,0.82),
        1,hexToRgba(state.gradEnd,0.82),
        a0,hexToRgba("#ffffff",0.0),
        a1,hexToRgba("#ffffff",0.30),
        a2,hexToRgba("#ffffff",0.0),
      ]);
    }
    return makeLinearGradient(ctx,x0,y0,x1,y1,[
      0,hexToRgba(baseHex,0.82),
      a0,hexToRgba(baseHex,0.82),
      a1,hexToRgba("#fff7d6",alphaMid),
      a2,hexToRgba(baseHex,0.82),
      1,hexToRgba(baseHex,0.82),
    ]);
  }
  function solidStyle(alpha=0.70){
    if(state.colorMode==="gradient"){
      return makeLinearGradient(ctx,x0,y0,x1,y1,[0,hexToRgba(state.gradStart,alpha),1,hexToRgba(state.gradEnd,alpha)]);
    }
    return hexToRgba(baseHex,alpha);
  }
  function wrapScans(s){
    if(!state.effectsEnabled) return [s];
    if(s < band) return [s, s+1];
    if(s > 1 - band) return [s, s-1];
    return [s];
  }

  if(style==="neon"){
    const outer = bw + Math.max(4, glowW * 1.6);
    ctx.lineWidth = outer;
    ctx.strokeStyle = state.colorMode==="gradient"
      ? makeLinearGradient(ctx,x0,y0,x1,y1,[0,hexToRgba(state.gradStart,0.20),1,hexToRgba(state.gradEnd,0.20)])
      : hexToRgba(baseHex,0.26);
    strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);

    ctx.lineWidth = Math.max(1, bw*0.9);
    if(state.effectsEnabled){
      for(const s of wrapScans(scan)){
        ctx.strokeStyle = mainStrokeStyle(0.90, s);
        strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
      }
    }else{
      ctx.strokeStyle = solidStyle(0.70);
      strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
    }

    ctx.lineWidth = Math.max(1, bw*0.45);
    ctx.strokeStyle = state.colorMode==="gradient"
      ? makeLinearGradient(ctx,x0,y0,x1,y1,[0,hexToRgba("#ffffff",0.85),1,hexToRgba("#ffffff",0.55)])
      : hexToRgba("#ffffff",0.82);
    strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
  }

  if(style==="fiber"){
    ctx.lineWidth = bw + 2;
    ctx.strokeStyle = solidStyle(0.28);
    strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);

    ctx.lineWidth = Math.max(1, bw*0.65);
    if(state.colorMode==="gradient"){
      ctx.strokeStyle = makeLinearGradient(ctx,x0,y0,x1,y1,[
        0,hexToRgba(state.gradStart,0.55),
        0.5,hexToRgba("#ffffff",0.90),
        1,hexToRgba(state.gradEnd,0.55),
      ]);
    }else{
      ctx.strokeStyle = makeLinearGradient(ctx,x0,y0,x1,y1,[
        0,hexToRgba(baseHex,0.55),
        0.5,hexToRgba("#ffffff",0.90),
        1,hexToRgba(baseHex,0.55),
      ]);
    }
    strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
  }

  if(style==="pulse"){
    ctx.lineWidth = bw;
    ctx.strokeStyle = solidStyle(0.35);
    strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
    if(state.effectsEnabled){
      ctx.lineWidth = Math.max(1, bw*0.85);
      ctx.setLineDash([14, 12]);
      ctx.lineDashOffset = -(tNow/1000) * 40 * speed;
      ctx.strokeStyle = state.colorMode==="gradient"
        ? makeLinearGradient(ctx,x0,y0,x1,y1,[0,hexToRgba(state.gradStart,0.85),1,hexToRgba(state.gradEnd,0.85)])
        : hexToRgba(baseHex,0.90);
      strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
      ctx.setLineDash([]);
    }
  }

  if(style==="metal"){
    ctx.lineWidth = Math.max(1, bw*0.65);
    ctx.strokeStyle = solidStyle(0.55);
    strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
    if(state.effectsEnabled){
      ctx.lineWidth = Math.max(1, bw*0.35);
      for(const s of wrapScans(scan)){
        ctx.strokeStyle = mainStrokeStyle(0.55, s);
        strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
      }
    }
  }

  if(style==="beam"){
    ctx.lineWidth = bw;
    if(state.effectsEnabled){
      for(const s of wrapScans(scan)){
        ctx.strokeStyle = mainStrokeStyle(0.92, s);
        strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
      }
    }else{
      ctx.strokeStyle = solidStyle(0.70);
      strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
    }
  }

  if(style==="dual"){
    const off = 2.2 + (bw*0.25);
    const t = 0.55;
    const dx = evalCubicDeriv(t, x0,cx1,cx2,x1);
    const dy = evalCubicDeriv(t, y0,cy1,cy2,y1);
    const len = Math.max(1, Math.sqrt(dx*dx+dy*dy));
    const nx = -dy/len, ny = dx/len;

    function strokeOne(sign, phaseOffset){
      const sx0=x0+nx*off*sign, sy0=y0+ny*off*sign;
      const scx1=cx1+nx*off*sign, scy1=cy1+ny*off*sign;
      const scx2=cx2+nx*off*sign, scy2=cy2+ny*off*sign;
      const sx1=x1+nx*off*sign, sy1=y1+ny*off*sign;

      ctx.lineWidth = Math.max(1, bw*0.85);
      const p = (scan + phaseOffset) % 1;
      if(state.effectsEnabled){
        const a0=clamp(p-band,0,1), a1=clamp(p,0,1), a2=clamp(p+band,0,1);
        if(state.colorMode==="gradient"){
          ctx.strokeStyle = makeLinearGradient(ctx,sx0,sy0,sx1,sy1,[
            0,hexToRgba(state.gradStart,0.74),
            1,hexToRgba(state.gradEnd,0.74),
            a0,hexToRgba("#ffffff",0.0),
            a1,hexToRgba("#ffffff",0.38),
            a2,hexToRgba("#ffffff",0.0),
          ]);
        }else{
          ctx.strokeStyle = makeLinearGradient(ctx,sx0,sy0,sx1,sy1,[
            0,hexToRgba(baseHex,0.74),
            a0,hexToRgba(baseHex,0.74),
            a1,hexToRgba("#fff7d6",0.98),
            a2,hexToRgba(baseHex,0.74),
            1,hexToRgba(baseHex,0.74),
          ]);
        }
      }else{
        ctx.strokeStyle = solidStyle(0.62);
      }
      ctx.beginPath(); ctx.moveTo(sx0,sy0); ctx.bezierCurveTo(scx1,scy1,scx2,scy2,sx1,sy1); ctx.stroke();
      if(state.effectsEnabled && (p < band || p > 1 - band)){
        const pp = p < band ? p + 1 : p - 1;
        const a0b=clamp(pp-band,0,1), a1b=clamp(pp,0,1), a2b=clamp(pp+band,0,1);
        if(state.colorMode==="gradient"){
          ctx.strokeStyle = makeLinearGradient(ctx,sx0,sy0,sx1,sy1,[
            0,hexToRgba(state.gradStart,0.74),
            1,hexToRgba(state.gradEnd,0.74),
            a0b,hexToRgba("#ffffff",0.0),
            a1b,hexToRgba("#ffffff",0.38),
            a2b,hexToRgba("#ffffff",0.0),
          ]);
        }else{
          ctx.strokeStyle = makeLinearGradient(ctx,sx0,sy0,sx1,sy1,[
            0,hexToRgba(baseHex,0.74),
            a0b,hexToRgba(baseHex,0.74),
            a1b,hexToRgba("#fff7d6",0.98),
            a2b,hexToRgba(baseHex,0.74),
            1,hexToRgba(baseHex,0.74),
          ]);
        }
        ctx.beginPath(); ctx.moveTo(sx0,sy0); ctx.bezierCurveTo(scx1,scy1,scx2,scy2,sx1,sy1); ctx.stroke();
      }
    }

    strokeOne(+1, 0.00);
    strokeOne(-1, 0.45);

    ctx.lineWidth = Math.max(1.0, bw*0.55);
    ctx.strokeStyle = solidStyle(0.35);
    strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
  }

  if(style==="wave" || style==="spiral" || style==="twist" || style==="lightning"){
    const pts = [];
    for(let i=0;i<=seg;i++){
      const t = i/seg;
      const x = evalCubic(t, x0,cx1,cx2,x1);
      const y = evalCubic(t, y0,cy1,cy2,y1);
      const dx = evalCubicDeriv(t, x0,cx1,cx2,x1);
      const dy = evalCubicDeriv(t, y0,cy1,cy2,y1);
      const l = Math.max(1, Math.sqrt(dx*dx+dy*dy));
      const nx = -dy/l, ny = dx/l;
      pts.push({t,x,y,nx,ny});
    }

    ctx.lineWidth = bw;
    ctx.strokeStyle = state.effectsEnabled ? mainStrokeStyle(0.92, scan) : solidStyle(0.70);

    ctx.beginPath();
    for(let i=0;i<pts.length;i++){
      const p = pts[i];
      const tt = p.t;
      let amp = 0;

      if(style==="wave"){
        amp = (2.0 + bw*0.5) * Math.sin((tt*10 + phase*1.2) * Math.PI*2);
      }
      if(style==="spiral"){
        const env = Math.sin(tt*Math.PI);
        amp = (3.2 + bw*0.8) * env * Math.sin((tt*14 + phase*1.8) * Math.PI*2);
      }
      if(style==="twist"){
        const n = noise01(tt*17.0 + phase*0.8 + link.id*0.1);
        amp = (3.0 + bw*0.7) * (n-0.5) * 2.0;
      }
      if(style==="lightning"){
        const env = Math.sin(tt*Math.PI);
        const n = noise01(tt*31.0 + phase*1.5 + link.id*0.33);
        amp = (8.0 + bw*1.2) * env * (n-0.5) * 2.0;
      }

      const x = p.x + p.nx*amp;
      const y = p.y + p.ny*amp;

      if(i===0) ctx.moveTo(x,y);
      else ctx.lineTo(x,y);
    }
    ctx.stroke();
    if(state.effectsEnabled){
      const scans = wrapScans(scan);
      if(scans.length > 1){
        ctx.strokeStyle = mainStrokeStyle(0.92, scans[1]);
        ctx.stroke();
      }
    }

    if(style==="lightning" && state.effectsEnabled){
      ctx.lineWidth = Math.max(1, bw*0.55);
      ctx.strokeStyle = state.colorMode==="gradient"
        ? makeLinearGradient(ctx,x0,y0,x1,y1,[0,hexToRgba("#ffffff",0.35),1,hexToRgba("#ffffff",0.10)])
        : hexToRgba("#ffffff",0.22);
      ctx.beginPath();
      for(let i=0;i<pts.length;i++){
        const p=pts[i];
        const tt=p.t;
        const env=Math.sin(tt*Math.PI);
        const n=noise01(tt*51.0 + phase*2.0 + link.id*0.77);
        const amp=(4.5 + bw*0.6)*env*(n-0.5)*2.0;
        const x=p.x + p.nx*amp;
        const y=p.y + p.ny*amp;
        if(i===0) ctx.moveTo(x,y); else ctx.lineTo(x,y);
      }
      ctx.stroke();
    }
  }

  if(state.effectsEnabled && state.particles && state.particleCount>0){
    const n = perf ? Math.min(8, Math.max(1, Math.floor((state.particleCount|0) * 0.4))) : Math.min(16, Math.max(1, state.particleCount|0));
    ctx.fillStyle = (state.colorMode==="gradient") ? hexToRgba("#ffffff",0.22) : hexToRgba(baseHex,0.50);
    for(let i=0;i<n;i++){
      const tt=(phase*0.22 + i/n) % 1.0;
      const x=evalCubic(tt, x0,cx1,cx2,x1);
      const y=evalCubic(tt, y0,cy1,cy2,y1);
      ctx.beginPath(); ctx.arc(x,y,1.4,0,Math.PI*2); ctx.fill();
    }
  }

  ctx.restore();
}

function drawSimpleLink(canvas, ctx, link, state, overrideColor, curveStyle, dashed, tNow){
  const origin=getNodeById(canvas.graph, link.origin_id);
  const target=getNodeById(canvas.graph, link.target_id);
  if(!origin||!target) return;

  const start=getConnPos(canvas, origin, false, link.origin_slot);
  const end=getConnPos(canvas, target, true, link.target_slot);
  const x0=start[0], y0=start[1], x1=end[0], y1=end[1];
  const {cx1,cy1,cx2,cy2} = (curveStyle === "arc") ? bezierPointsScaled(x0,y0,x1,y1,1.6) : bezierPoints(x0,y0,x1,y1);

  ctx.save();
  ctx.lineCap="round";
  ctx.lineJoin="round";
  ctx.lineWidth = clamp(state.beamWidth || 2, 1, 6);
  if(dashed){
    ctx.setLineDash([6,4]);
    if(state.effectsEnabled && Number.isFinite(tNow)){
      const sp = clamp(state.speed || 1, 0.1, 3);
      ctx.lineDashOffset = -(tNow / 1000) * 18 * sp;
    }
  }
  if(state.colorMode==="gradient" && !overrideColor){
    ctx.strokeStyle = makeLinearGradient(ctx,x0,y0,x1,y1,[0,hexToRgba(state.gradStart,0.9),1,hexToRgba(state.gradEnd,0.9)]);
  }else{
    const base = overrideColor || resolveLinkBaseColor(canvas, link, state) || state.themeColor;
    ctx.strokeStyle = hexToRgba(base, 0.9);
  }
  strokeBezier(ctx,x0,y0,cx1,cy1,cx2,cy2,x1,y1);
  ctx.restore();
}

function drawOrthogonalLink(canvas, ctx, link, state, overrideColor, dashed, tNow){
  const origin=getNodeById(canvas.graph, link.origin_id);
  const target=getNodeById(canvas.graph, link.target_id);
  if(!origin||!target) return;
  const start=getConnPos(canvas, origin, false, link.origin_slot);
  const end=getConnPos(canvas, target, true, link.target_slot);
  const x0=start[0], y0=start[1], x1=end[0], y1=end[1];
  const midX = x0 + (x1 - x0) * 0.5;

  ctx.save();
  ctx.lineCap="round";
  ctx.lineJoin="round";
  ctx.lineWidth = clamp(state.beamWidth || 2, 1, 6);
  if(dashed){
    ctx.setLineDash([6,4]);
    if(state.effectsEnabled && Number.isFinite(tNow)){
      const sp = clamp(state.speed || 1, 0.1, 3);
      ctx.lineDashOffset = -(tNow / 1000) * 18 * sp;
    }
  }
  const base = overrideColor || resolveLinkBaseColor(canvas, link, state) || state.themeColor;
  ctx.strokeStyle = hexToRgba(base, 0.9);
  ctx.beginPath();
  ctx.moveTo(x0,y0);
  ctx.lineTo(midX,y0);
  ctx.lineTo(midX,y1);
  ctx.lineTo(x1,y1);
  ctx.stroke();
  ctx.restore();
}

const bezierSampleCache = new Map();
const polylineSampleCache = new Map();

function buildBezierSamples(x0,y0,cx1,cy1,cx2,cy2,x1,y1, steps){
  const key = `${x0|0}|${y0|0}|${cx1|0}|${cy1|0}|${cx2|0}|${cy2|0}|${x1|0}|${y1|0}|${steps|0}`;
  const hit = bezierSampleCache.get(key);
  if(hit) return hit;
  const pts=[];
  let lastX=x0, lastY=y0;
  let dist=0;
  pts.push({x:x0,y:y0,d:0});
  for(let i=1;i<=steps;i++){
    const t=i/steps;
    const x=evalCubic(t, x0,cx1,cx2,x1);
    const y=evalCubic(t, y0,cy1,cy2,y1);
    const dx=x-lastX, dy=y-lastY;
    dist += Math.sqrt(dx*dx+dy*dy);
    pts.push({x,y,d:dist});
    lastX=x; lastY=y;
  }
  const out = {pts, length: dist};
  bezierSampleCache.set(key, out);
  if(bezierSampleCache.size > 900){
    let i = 0;
    for(const k of bezierSampleCache.keys()){
      bezierSampleCache.delete(k);
      if(++i > 450) break;
    }
  }
  return out;
}

function buildPolylineSamples(points){
  const p0 = points?.[0];
  const p1 = points?.[1];
  const p2 = points?.[2];
  const p3 = points?.[3];
  const key = `${p0?.x|0}|${p0?.y|0}|${p1?.x|0}|${p1?.y|0}|${p2?.x|0}|${p2?.y|0}|${p3?.x|0}|${p3?.y|0}`;
  const hit = polylineSampleCache.get(key);
  if(hit) return hit;
  const pts=[];
  let dist=0;
  pts.push({x:p0.x,y:p0.y,d:0});
  for(let i=1;i<points.length;i++){
    const a=points[i-1], b=points[i];
    const dx=b.x-a.x, dy=b.y-a.y;
    const seg=Math.sqrt(dx*dx+dy*dy);
    dist += seg;
    pts.push({x:b.x,y:b.y,d:dist});
  }
  const out = {pts, length: dist};
  polylineSampleCache.set(key, out);
  if(polylineSampleCache.size > 900){
    let i = 0;
    for(const k of polylineSampleCache.keys()){
      polylineSampleCache.delete(k);
      if(++i > 450) break;
    }
  }
  return out;
}

function getPointAtDistance(samples, dist){
  const pts = samples.pts;
  const total = samples.length || 0;
  if(total<=0) return {x:pts[0].x,y:pts[0].y,angle:0};
  let d = dist % total;
  if(d<0) d += total;
  for(let i=1;i<pts.length;i++){
    const p0=pts[i-1], p1=pts[i];
    if(d <= p1.d){
      const segLen = p1.d - p0.d;
      const t = segLen > 0 ? (d - p0.d) / segLen : 0;
      const x = p0.x + (p1.x - p0.x) * t;
      const y = p0.y + (p1.y - p0.y) * t;
      const ang = Math.atan2(p1.y - p0.y, p1.x - p0.x);
      return {x,y,angle:ang};
    }
  }
  const last = pts[pts.length-1];
  const prev = pts[pts.length-2] || last;
  return {x:last.x,y:last.y,angle:Math.atan2(last.y-prev.y,last.x-prev.x)};
}

function forEachPoint(samples, startDist, step, count, cb){
  const pts = samples.pts;
  const total = samples.length || 0;
  if(!pts || pts.length < 2 || total <= 0 || count <= 0) return;
  let d = startDist % total;
  if(d < 0) d += total;
  let idx = 1;
  while(idx < pts.length && d > pts[idx].d) idx++;
  for(let i=0;i<count;i++){
    if(idx >= pts.length){
      idx = 1;
    }
    const p0 = pts[idx-1];
    const p1 = pts[idx];
    const segLen = p1.d - p0.d;
    const t = segLen > 0 ? (d - p0.d) / segLen : 0;
    const x = p0.x + (p1.x - p0.x) * t;
    const y = p0.y + (p1.y - p0.y) * t;
    const ang = Math.atan2(p1.y - p0.y, p1.x - p0.x);
    cb(x, y, ang, i, d, total);
    d += step;
    if(d > total){
      d = d % total;
      idx = 1;
    }
    while(idx < pts.length && d > pts[idx].d) idx++;
  }
}

  function drawFlowIcon(ctx, iconText, samples, tNow, speed, spacing, size, color, maxCount, enabled, loopFade){
    if(!iconText || !samples || samples.length<=0) return;
    const str = iconText.toString();
    if(!str) return;
    let step = clamp(typeof spacing === "number" ? spacing : 28, 10, 120);
    const total = samples.length || 0;
    const offset = enabled ? (tNow/1000) * speed : 0;
    if(total <= 0) return;
    const maxC = clamp(typeof maxCount === "number" ? maxCount : 8, 1, 20);
  if(total / step > maxC){
    step = total / maxC;
  }
  const fontSize = clamp(typeof size === "number" ? size : 14, 8, 28);
  const start = offset % total;
  const count = Math.max(1, Math.floor(total / step) + 1);
  const stamp = getIconStamp(str, fontSize, color || "#ffffff");
  const flip = LINK_ICON_FLIP.has(str);
  ctx.save();
  const fadeOn = !!loopFade;
  const fadeSpan = Math.min(total * 0.18, Math.max(12, step * 1.5));
  forEachPoint(samples, start, step, count, (x,y,ang, _i, d, totalLen)=>{
    let alpha = 1;
    if(fadeOn && totalLen > 0 && fadeSpan > 0){
      const a0 = Math.min(1, d / fadeSpan);
      const a1 = Math.min(1, (totalLen - d) / fadeSpan);
      alpha = Math.max(0.12, Math.min(a0, a1));
    }
    ctx.save();
    ctx.translate(x, y);
    ctx.rotate(ang);
    if(flip) ctx.scale(-1, 1);
    if(alpha < 1) ctx.globalAlpha *= alpha;
    ctx.drawImage(stamp.canvas, -stamp.w/2, -stamp.h/2);
    ctx.restore();
  });
  ctx.restore();
}

function shouldUseCompat(state, baseConn, baseNode){
  const mode = state?.compatMode || "auto";
  if(mode === "on") return true;
  if(mode === "off") return false;
  const connStr = baseConn?.toString?.() || "";
  const nodeStr = baseNode?.toString?.() || "";
  const connName = baseConn?.name || "";
  const nodeName = baseNode?.name || "";
  const connLen = connStr.length;
  const nodeLen = nodeStr.length;
  const connTokens = /links|graph|getConnectionPos|drawConnections/i.test(connStr);
  const nodeTokens = /drawNode|widgets|slots|title/i.test(nodeStr);
  return (
    connLen > 4200 ||
    nodeLen > 4200 ||
    (connName && connName !== "drawConnections") ||
    (nodeName && nodeName !== "drawNode") ||
    !connTokens ||
    !nodeTokens
  );
}

// undo fallback
function dispatchKeyCombo(code, ctrl, shift){
  const key = code === "KeyZ" ? "z" : (code === "KeyY" ? "y" : "");
  const evDown = new KeyboardEvent("keydown", { code, key, ctrlKey: !!ctrl, shiftKey: !!shift, bubbles:true });
  const evUp = new KeyboardEvent("keyup", { code, key, ctrlKey: !!ctrl, shiftKey: !!shift, bubbles:true });
  document.dispatchEvent(evDown);
  document.dispatchEvent(evUp);
}

function setupCanvasOverrides(stateRef, animRef) {
  const LGraphCanvas = window.LGraphCanvas;
  if(!LGraphCanvas) return;
  if(LGraphCanvas.prototype.__ct_patched_v9) return;
  LGraphCanvas.prototype.__ct_patched_v9 = true;

  const _drawConnections = LGraphCanvas.prototype.drawConnections;
  const _drawNode = LGraphCanvas.prototype.drawNode;

  let chainCacheKey = "";
  let chainCacheMap = new Map();
  let chainRulesRef = null;
  let chainRulesVer = 0;
  let visibleCacheKey = "";
  let visibleCache = null;
  let compatCacheMode = "";
  let compatCacheValue = false;

  let showAllHold=false;
  const matchesShowAllKey = (e, key)=>{
    if(!key || key === "none") return false;
    if(key === "alt") return e.code === "AltLeft" || e.code === "AltRight";
    if(key === "shift") return e.code === "ShiftLeft" || e.code === "ShiftRight";
    if(key === "ctrl") return e.code === "ControlLeft" || e.code === "ControlRight";
    if(key === "meta") return e.code === "MetaLeft" || e.code === "MetaRight";
    return false;
  };
  const getShowAllKey = ()=> (stateRef()?.showAllKey || "alt");
  const markShowAllDirty = ()=>{
    const canvas = app?.canvas || window.app?.canvas;
    canvas?.setDirty?.(true,true);
  };
  window.addEventListener("keydown",(e)=>{
    if(!matchesShowAllKey(e, getShowAllKey())) return;
    e.preventDefault();
    e.stopPropagation();
    showAllHold = true;
    markShowAllDirty();
  });
  window.addEventListener("keyup",(e)=>{
    if(matchesShowAllKey(e, getShowAllKey())) showAllHold = false;
    if(matchesShowAllKey(e, getShowAllKey())){
      e.preventDefault();
      e.stopPropagation();
      markShowAllDirty();
    }
  });
  window.addEventListener("blur", ()=>{ showAllHold = false; });
  document.addEventListener("visibilitychange", ()=>{
    if(document.hidden) showAllHold = false;
  });

  function getAllLinkIds(links){
    const out = [];
    for(const k in links){
      const id = toLinkId(k);
      if(id !== null) out.push(id);
    }
    return out;
  }

  function getSelectedSignature(canvas, state){
    if(!state?.showSelected) return "";
    const sel = canvas?.selected_nodes;
    if(sel && typeof sel === "object"){
      const keys = Object.keys(sel);
      if(keys.length === 0) return "";
      keys.sort();
      return keys.join(",");
    }
    const nodes = getSelectedNodes(canvas);
    if(!nodes.length) return "";
    return nodes.map(n=>nodeIdKey(n?.id)).sort().join(",");
  }

  function getVisibleLinksCached(canvas, state, links, linksCount, nodesCount){
    if(showAllHold) return getAllLinkIds(links);
    const hovered = canvas.node_over || null;
    const hoveredId = hovered ? nodeIdKey(hovered.id) : "";
    const selSig = getSelectedSignature(canvas, state);
    if(!hoveredId && !selSig) return [];
    const key = `${hoveredId}|${selSig}|${state.neighborDepth}|${linksCount}|${nodesCount}`;
    if(key === visibleCacheKey && visibleCache) return visibleCache;
    visibleCacheKey = key;
    visibleCache = computeVisibleLinks(canvas, state, false);
    return visibleCache;
  }

  function getCompat(state){
    const mode = state?.compatMode || "auto";
    if(mode === "on") return true;
    if(mode === "off") return false;
    if(compatCacheMode === "auto") return compatCacheValue;
    compatCacheMode = "auto";
    compatCacheValue = shouldUseCompat(state, _drawConnections, _drawNode);
    return compatCacheValue;
  }

  LGraphCanvas.prototype.drawConnections = function(ctx){
    const canvas=this;
    const state=stateRef();

    try{
      const compat = getCompat(state); // [CT] 兼容模式降级逻辑
      const useChain = !!state?.chainColorEnabled && Array.isArray(state.chainRules) && state.chainRules.length>0;
      const useCustom = !!state && !!state.enabled && (
        state.hoverOnly ||
        useChain ||
        !!state.effectsEnabled ||
        !!state.linkIconEnabled ||
        (state.lineStyleGlobal && state.lineStyleGlobal !== "auto")
      );
      if(!useCustom){
        animRef(false);
        return _drawConnections.call(this, ctx);
      }

      const links=canvas.graph?.links;
      if(!links){
        animRef(false);
        return;
      }
      const linksCount = Object.keys(links).length;
      const nodesCount = canvas.graph?._nodes ? canvas.graph._nodes.length : 0;

      const visible = state.hoverOnly
        ? getVisibleLinksCached(canvas, state, links, linksCount, nodesCount)
        : getAllLinkIds(links);
      if(!visible || visible.length===0){
        animRef(false);
        return;
      }

      let chainMap = null; // [CT] 链路锁色 map
      if(useChain){
        if(state.chainRules !== chainRulesRef){
          chainRulesRef = state.chainRules;
          chainRulesVer++;
        }
        const key = `${linksCount}|${nodesCount}|${chainRulesVer}`;
        if(key !== chainCacheKey){
          chainCacheKey = key;
          chainCacheMap = rebuildChainColorMap(canvas.graph, state.chainRules);
        }
        chainMap = chainCacheMap;
      }

      const now = performance.now();
      const hasActiveEffects = !!state.effectsEnabled && !compat;
      animRef(hasActiveEffects && !!state.enabled);

      for(const id of visible){
        const link = links[id];
        if(!link) continue;
        const canAnimate = !!state.effectsEnabled && !compat;
        const overrideColor = chainMap?.get(id) || null;
        const origin = getNodeById(canvas.graph, link.origin_id);
        const globalLineStyle = state?.lineStyleGlobal || "auto";
        const lineStyle = (globalLineStyle && globalLineStyle !== "auto") ? globalLineStyle : "auto";
        let iconText = state.linkIconEnabled ? (state.linkIconText || "") : "";
        if(state.linkIconEnabled && state.linkIconAutoCycle && LINK_ICON_PRESETS.length){
          const period = clamp(state.linkIconCyclePeriod || 6, 2, 30);
          const idx = Math.floor((now/1000) / period) % LINK_ICON_PRESETS.length;
          iconText = LINK_ICON_PRESETS[idx];
        }

        if(lineStyle === "orthogonal"){
          drawOrthogonalLink(canvas, ctx, link, state, overrideColor, false, now);
        }else if(lineStyle === "dashed"){
          drawSimpleLink(canvas, ctx, link, state, overrideColor, "curve", true, now);
        }else if(lineStyle === "arc"){
          if(canAnimate){
            drawLinkEffect(canvas, ctx, link, state, now, overrideColor, null, "arc");
          }else{
            drawSimpleLink(canvas, ctx, link, state, overrideColor, "arc", false, now);
          }
        }else if(lineStyle === "beam"){
          if(canAnimate){
            drawLinkEffect(canvas, ctx, link, state, now, overrideColor, "beam", "curve");
          }else{
            drawSimpleLink(canvas, ctx, link, state, overrideColor, "curve", false, now);
          }
        }else if(lineStyle === "curve"){
          if(canAnimate){
            drawLinkEffect(canvas, ctx, link, state, now, overrideColor, null, "curve");
          }else{
            drawSimpleLink(canvas, ctx, link, state, overrideColor, "curve", false, now);
          }
        }else{
          if(canAnimate){
            drawLinkEffect(canvas, ctx, link, state, now, overrideColor);
          }else{
            drawSimpleLink(canvas, ctx, link, state, overrideColor, null, false, now);
          }
        }

        if(iconText){
          const start=getConnPos(canvas, origin, false, link.origin_slot);
          const target=getNodeById(canvas.graph, link.target_id);
          const end=target ? getConnPos(canvas, target, true, link.target_slot) : null;
          if(end){
            const x0=start[0], y0=start[1], x1=end[0], y1=end[1];
            const perf = !!state?.perfMode;
            const iconSpeed = clamp(state.linkIconSpeed ?? 40, 10, 200);
            const iconSpacing = clamp((state.linkIconSpacing ?? 28) * (perf ? 1.2 : 1), 10, 140);
            const iconSize = clamp(state.linkIconSize ?? 14, 8, 28);
            const iconMax = clamp(Math.floor((state.linkIconMaxCount ?? 8) * (perf ? 0.6 : 1)), 1, 20);
            let iconColor = overrideColor || (state.colorMode==="gradient" ? "#ffffff" : (resolveLinkBaseColor(canvas, link, state) || state.themeColor));
            if(state.linkIconColorEnabled && typeof state.linkIconColor === "string" && state.linkIconColor.startsWith("#")){
              iconColor = state.linkIconColor;
            }
            const animText = !!state.linkTextIconAnimate && !!state.effectsEnabled;
            if(lineStyle === "orthogonal"){
              const midX = x0 + (x1 - x0) * 0.5;
              const samples = buildPolylineSamples([
                {x:x0,y:y0},
                {x:midX,y:y0},
                {x:midX,y:y1},
                {x:x1,y:y1},
              ]);
              if(iconText) drawFlowIcon(ctx, iconText, samples, now, iconSpeed, iconSpacing, iconSize, iconColor, iconMax, animText, state.linkIconLoopFade);
            }else{
              const cps = (lineStyle === "arc") ? bezierPointsScaled(x0,y0,x1,y1,1.6) : bezierPoints(x0,y0,x1,y1);
              const baseQ = clamp((state.quality|0) || 12, 6, 32);
              const steps = clamp(Math.floor(baseQ * (perf ? 1.0 : 1.6) + (iconText ? 8 : 0)), 16, 64);
              const samples = buildBezierSamples(x0,y0,cps.cx1,cps.cy1,cps.cx2,cps.cy2,x1,y1,steps);
              if(iconText) drawFlowIcon(ctx, iconText, samples, now, iconSpeed, iconSpacing, iconSize, iconColor, iconMax, animText, state.linkIconLoopFade);
            }
          }
        }
      }
    }catch(err){
      console.warn("[CanvasToolkit] drawConnections 出错，回退原版", err);
      animRef(false);
      return _drawConnections.call(this, ctx);
    }
  };

  if(typeof _drawNode === "function"){
    LGraphCanvas.prototype.drawNode = function(node, ctx){
      if(!patchLabelDraw._patched){
        patchLabelDraw(stateRef);
      }
      const baseStyle = node?.properties?.__ct_style;
      const s = stateRef?.();
      const collapsed = isCollapsedNode(node);
      const skipCustomCollapsed = collapsed && isRgthreeAnySwitch(node);
      const previewOn = !!s?.nodeStylePreview;
      let style = baseStyle;
      if(previewOn && node && this){
        const sel = this.selected_nodes;
        const hasSel = sel && Object.keys(sel).length > 0;
        const isTarget = hasSel ? !!sel?.[node.id] : (this.node_over === node);
        if(isTarget){
          const colors = parseGradientValue(s?.nodeBgColor);
          style = {
            ...(baseStyle || {}),
            bgGradient: colors.length ? colors.join(",") : null,
            bgGradientAngle: Number.isFinite(s?.nodeBgAngle) ? s.nodeBgAngle : DEFAULTS.nodeBgAngle,
            bgGradientSpan: Number.isFinite(s?.nodeBgGradientSpan) ? s.nodeBgGradientSpan : DEFAULTS.nodeBgGradientSpan,
            titleBarColor: (typeof s?.nodeTitleBarColor === "string" && s.nodeTitleBarColor.trim()) ? s.nodeTitleBarColor.trim() : null,
            titleTextColor: (typeof s?.nodeTitleTextColor === "string" && s.nodeTitleTextColor.startsWith("#")) ? s.nodeTitleTextColor : null,
            titleBarEnabled: s?.nodeTitleBarEnabled !== false,
          };
        }
      }
      if(isLabelLikeNode(node)){
        disableLabelEditing(node);
      }
      const isSticky = !!s?.stickyEnabled && isStickyNode(node, s);
      const stickyBg = isSticky ? s?.stickyBg : null;
      const stickyText = isSticky ? s?.stickyText : null;
      const baseGrad = style?.bgGradient || stickyBg;
      const globalGrad = s?.nodeBgGlobal ? s?.nodeBgColor : null;
      const grad = skipCustomCollapsed ? null : (baseGrad || globalGrad);
      const titleBar = style?.titleBarColor;
      const titleText = style?.titleTextColor;
      const globalTitle = !!s?.nodeTitleGlobal;
      const globalBar = s?.nodeTitleBarColor;
      const globalText = s?.nodeTitleTextColor;
      const applyBar = skipCustomCollapsed ? null : ((typeof titleBar === "string" && titleBar.trim())
        ? titleBar.trim()
        : (globalTitle && typeof globalBar === "string" && globalBar.trim() ? globalBar.trim() : null));
      let applyText = skipCustomCollapsed ? null : ((typeof titleText === "string" && titleText.startsWith("#"))
        ? titleText
        : (globalTitle && typeof globalText === "string" && globalText.startsWith("#") ? globalText : null));
      if(isSticky && typeof stickyText === "string" && stickyText.startsWith("#")){
        applyText = stickyText;
      }
      const barEnabled = (style?.titleBarEnabled === false)
        ? false
        : (s?.nodeTitleBarEnabled !== false && (!isSticky || !!s?.stickyTitleBar));
      const drawUnlinked = ()=>{
        if(s?.unlinkedHighlightEnabled && !isLabelLikeNode(node) && !nodeHasLinks(node, this?.graph)){
          animRef?.(true);
          drawUnlinkedHint(
            ctx,
            node,
            s?.unlinkedHighlightColor,
            s?.unlinkedHighlightWidth,
            performance.now(),
            s?.unlinkedHighlightText,
            s?.unlinkedHighlightTextSize,
            s?.unlinkedHighlightTextEnabled,
            s?.unlinkedHighlightPulse
          );
        }
      };

      if(grad || applyBar || applyText){
        let hadBg = false;
        let origBg;
        let origOnBg;
        let origColor;
        let origTitleColor;
        let didDraw = false;
        try{
          hadBg = Object.prototype.hasOwnProperty.call(node, "bgcolor");
          origBg = node.bgcolor;
          origOnBg = node.onDrawBackground;
          origColor = node.color;
          origTitleColor = node.title_color;
          if(applyBar && barEnabled){
            const colors = parseGradientValue(applyBar);
            if(colors.length === 1) node.color = colors[0];
          }
          if(applyText) node.title_color = applyText;
          if(grad){
            node.bgcolor = "rgba(0,0,0,0)";
            node.onDrawBackground = function(bgCtx){
              didDraw = true;
              bgCtx.save();
              drawNodeGradientBg(
                bgCtx,
                node,
                grad,
                style?.bgGradientAngle ?? s?.nodeBgAngle,
                style?.bgGradientSpan ?? s?.nodeBgGradientSpan
              );
              bgCtx.restore();
              if(typeof origOnBg === "function") return origOnBg.call(node, bgCtx);
            };
          }
          const out = _drawNode.call(this, node, ctx);
          if((applyBar || applyText) && barEnabled){
            const textColor = applyText || origTitleColor || "#e6e6e6";
            drawNodeTitleBar(ctx, node, applyBar, textColor, {
              radius: s?.nodeTitleRadius,
              gloss: s?.nodeTitleGloss,
              divider: s?.nodeTitleDivider,
              bleed: s?.nodeTitleBleed,
            });
          }
          if(grad && !didDraw){
            ctx.save();
            drawNodeGradientBg(
              ctx,
              node,
              grad,
              style?.bgGradientAngle ?? s?.nodeBgAngle,
              style?.bgGradientSpan ?? s?.nodeBgGradientSpan
            );
            ctx.restore();
          }
          drawUnlinked();
          return out;
        }catch(err){
          console.warn("[CanvasToolkit] drawNode 样式覆盖出错，回退原版", err);
          return _drawNode.call(this, node, ctx);
        }finally{
          if(hadBg) node.bgcolor = origBg;
          else delete node.bgcolor;
          node.onDrawBackground = origOnBg;
          node.color = origColor;
          node.title_color = origTitleColor;
        }
      }
      const out = _drawNode.call(this, node, ctx);
      drawUnlinked();
      return out;
    };
  }

  const LGraphGroup = window.LGraphGroup;
  if(LGraphGroup && !LGraphGroup.prototype.__ct_group_patched_v9){
    LGraphGroup.prototype.__ct_group_patched_v9 = true;
    const _groupDraw = LGraphGroup.prototype.draw;
    if(typeof _groupDraw === "function"){
      LGraphGroup.prototype.draw = function(ctx, canvas){
        const s = stateRef?.();
        const useSticky = !!s?.stickyEnabled && isStickyGroup(this, s) && typeof s?.stickyBg === "string";
        if(!useSticky){
          return _groupDraw.call(this, ctx, canvas);
        }
        try{
          const out = _groupDraw.call(this, ctx, canvas);
          ctx.save();
          const prevComp = ctx.globalCompositeOperation;
          ctx.globalCompositeOperation = "source-atop";
          drawGroupGradient(ctx, this, s.stickyBg);
          ctx.globalCompositeOperation = prevComp || "source-over";
          ctx.restore();
          if(typeof s?.stickyText === "string" && s.stickyText.startsWith("#")){
            const title = (this.title || this.name || "").toString();
            if(title){
              const pos = this.pos || [0,0];
              const fontSize = Math.max(12, Math.round((this.font_size || 18)));
              ctx.save();
              ctx.fillStyle = s.stickyText;
              ctx.font = `${fontSize}px sans-serif`;
              ctx.textAlign = "left";
              ctx.textBaseline = "top";
              ctx.fillText(title, (pos[0] ?? 0) + 6, (pos[1] ?? 0) + 4);
              ctx.restore();
            }
          }
          return out;
        }catch(err){
          console.warn("[CanvasToolkit] drawGroup 样式覆盖出错，回退原版", err);
          return _groupDraw.call(this, ctx, canvas);
        }
    };
  }

  const _getNodeMenu = LGraphCanvas.prototype.getNodeMenuOptions;
  if(typeof _getNodeMenu === "function" && !LGraphCanvas.prototype.__ct_menu_patched_v9){
    LGraphCanvas.prototype.__ct_menu_patched_v9 = true;
    LGraphCanvas.prototype.getNodeMenuOptions = function(node){
      const opts = _getNodeMenu.apply(this, arguments);
      if(!node || !isLabelLikeNode(node) || !Array.isArray(opts)) return opts;
      return opts.filter(o => {
        const label = (o?.content || o?.label || "").toString().toLowerCase();
        if(label.includes("edit title")) return false;
        if(label.includes("rename")) return false;
        if(label.includes("重命名") || label.includes("编辑")) return false;
        return true;
      });
    };
  }
}
}

function makeActions(getCanvas, getState, setState, panelRef, labelApi) {
  function firstSelected(nodes){
    return nodes.slice().sort((a,b)=>(a?.id??0)-(b?.id??0))[0];
  }
  function beginChange(graph){ try{ graph?.beforeChange?.(); }catch{} }
  function endChange(graph){ try{ graph?.afterChange?.(); }catch{} }
  function markDirty(canvas){ try{ canvas?.setDirty?.(true,true); }catch{} }
  function getTargetNodes(canvas){
    const nodes = getSelectedNodes(canvas);
    if(nodes.length) return nodes;
    const hovered = canvas?.node_over || null;
    return hovered ? [hovered] : [];
  }

  function withUndo(fn){
    const canvas = getCanvas();
    const graph = canvas?.graph;
    beginChange(graph);
    try { fn(canvas, graph); }
    finally { endChange(graph); markDirty(canvas); }
  }

  function align(kind){
    withUndo((canvas)=>{
      const nodes=getSelectedNodes(canvas);
      if(!nodes.length) return;
      const xs=nodes.map(n=>n.pos[0]);
      const ys=nodes.map(n=>n.pos[1]);
      const ws=nodes.map(n=>n.size?.[0] ?? 0);
      const hs=nodes.map(n=>n.size?.[1] ?? 0);
      const minX=Math.min(...xs), minY=Math.min(...ys);
      const maxX=Math.max(...xs.map((x,i)=>x+ws[i]));
      const maxY=Math.max(...ys.map((y,i)=>y+hs[i]));
      const cx=(minX+maxX)/2, cy=(minY+maxY)/2;

      for(let i=0;i<nodes.length;i++){
        const n=nodes[i], w=ws[i], h=hs[i];
        if(kind==="left") n.pos[0]=minX;
        if(kind==="right") n.pos[0]=maxX-w;
        if(kind==="top") n.pos[1]=minY;
        if(kind==="bottom") n.pos[1]=maxY-h;
        if(kind==="hcenter") n.pos[0]=cx-w/2;
        if(kind==="vcenter") n.pos[1]=cy-h/2;
      }
    });
  }

  function dist(axis){
    withUndo((canvas)=>{
      const nodes=getSelectedNodes(canvas);
      if(nodes.length<3) return;
      const list=nodes.slice().sort((a,b)=>(axis==="x"?a.pos[0]-b.pos[0]:a.pos[1]-b.pos[1]));
      if(axis === "x"){
        const left = list[0].pos[0];
        const last = list[list.length-1];
        const lastW = last?.size?.[0] ?? 0;
        const right = last.pos[0] + lastW;
        const sizes = list.map(n => n?.size?.[0] ?? 0);
        const total = sizes.reduce((a,b)=>a+b,0);
        const gap = (right - left - total) / (list.length - 1);
        let x = left;
        for(let i=0;i<list.length;i++){
          list[i].pos[0] = x;
          x += sizes[i] + gap;
        }
      }else{
        const top = list[0].pos[1];
        const last = list[list.length-1];
        const lastH = last?.size?.[1] ?? 0;
        const bottom = last.pos[1] + lastH;
        const sizes = list.map(n => n?.size?.[1] ?? 0);
        const total = sizes.reduce((a,b)=>a+b,0);
        const gap = (bottom - top - total) / (list.length - 1);
        let y = top;
        for(let i=0;i<list.length;i++){
          list[i].pos[1] = y;
          y += sizes[i] + gap;
        }
      }
    });
  }

  function syncSize(mode){
    withUndo((canvas)=>{
      const nodes=getSelectedNodes(canvas);
      if(nodes.length<2) return;
      const ref=firstSelected(nodes);
      const w=ref?.size?.[0], h=ref?.size?.[1];
      if(typeof w!=="number" || typeof h!=="number") return;
      for(const n of nodes){
        if(n===ref) continue;
        if(!Array.isArray(n.size)) n.size=[w,h];
        if(mode==="both"){ n.size[0]=w; n.size[1]=h; }
        if(mode==="w"){ n.size[0]=w; }
        if(mode==="h"){ n.size[1]=h; }
      }
    });
  }

  function alignAndDistribute(axis){
    withUndo((canvas)=>{
      const nodes=getSelectedNodes(canvas);
      if(nodes.length<2) return;
      if(axis==="x"){
        align("top");
        dist("x");
      }else{
        align("left");
        dist("y");
      }
    });
  }

  function autoArrange(axis){
    withUndo((canvas)=>{
      const nodes=getSelectedNodes(canvas);
      if(nodes.length<2) return;
      const list = nodes.slice().sort((a,b)=>(axis==="x"?a.pos[0]-b.pos[0]:a.pos[1]-b.pos[1]));
      const gap = 80;
      if(axis==="x"){
        const y = Math.min(...list.map(n=>n.pos[1]));
        let x = Math.min(...list.map(n=>n.pos[0]));
        for(const n of list){
          n.pos[0] = x;
          n.pos[1] = y;
          const w = n?.size?.[0] ?? 0;
          x += w + gap;
        }
      }else{
        const x = Math.min(...list.map(n=>n.pos[0]));
        let y = Math.min(...list.map(n=>n.pos[1]));
        for(const n of list){
          n.pos[0] = x;
          n.pos[1] = y;
          const h = n?.size?.[1] ?? 0;
          y += h + gap;
        }
      }
    });
  }

  function applyNodeStyle(){
    withUndo((canvas)=>{
      const nodes=getTargetNodes(canvas);
      if(!nodes.length) return;
      const s=getState();
      const colors = parseGradientValue(s.nodeBgColor);
      const isGradient = colors.length > 1;
      const first = colors[0] || s.nodeBgColor;
      const titleBar = s.nodeTitleBarColor;
      const titleText = s.nodeTitleTextColor;
      const barColors = parseGradientValue(titleBar);
      const barIsSingle = barColors.length === 1;
      const barFirst = barColors[0] || (typeof titleBar === "string" ? titleBar : null);
      for(const n of nodes){
        n.bgcolor = first;
        if(barIsSingle && typeof barFirst === "string" && barFirst.startsWith("#")) n.color = barFirst;
        if(typeof titleText === "string" && titleText.startsWith("#")) n.title_color = titleText;
        if(!n.properties) n.properties = {};
          n.properties.__ct_style = {
            bgcolor: first,
            bgGradient: isGradient ? colors.join(",") : null,
            bgGradientAngle: Number.isFinite(s.nodeBgAngle) ? s.nodeBgAngle : DEFAULTS.nodeBgAngle,
            bgGradientSpan: Number.isFinite(s.nodeBgGradientSpan) ? s.nodeBgGradientSpan : DEFAULTS.nodeBgGradientSpan,
            titleBarColor: (typeof titleBar === "string" && titleBar.trim()) ? titleBar.trim() : null,
            titleTextColor: (typeof titleText === "string" && titleText.startsWith("#")) ? titleText : null,
            titleBarEnabled: s.nodeTitleBarEnabled !== false,
          };
      }
    });
  }

  function clearNodeStyle(){
    withUndo((canvas)=>{
      const nodes=getTargetNodes(canvas);
      if(!nodes.length) return;
      for(const n of nodes){
        if(n.properties?.__ct_style) delete n.properties.__ct_style;
        n.bgcolor = undefined;
        n.color = undefined;
        n.title_color = undefined;
      }
    });
  }

  function addLabel(text){
    const titleText = (text || "").trim() || "新建标签";
    labelApi?.addLabel?.(titleText);
  }

  function applyLabelStyle(stylePatch){
    labelApi?.applyToAll?.(stylePatch);
  }

  function undo(){
    const canvas=getCanvas();
    const graph=canvas?.graph || app?.graph || window.app?.graph || null;
    if(graph && typeof graph.undo === "function") {
      const res = graph.undo();
      if(res !== false) { markDirty(canvas); return; }
    }
    dispatchKeyCombo("KeyZ", true, false);
  }
  // redo removed

  function addChainRuleFromSelected(){
    const canvas = getCanvas();
    const nodes = getSelectedNodes(canvas);
    if(!nodes.length) return;
    const n = firstSelected(nodes);
    if(n?.id === null || typeof n?.id === "undefined") return;
    const s = getState();
    const rules = Array.isArray(s.chainRules) ? s.chainRules.slice() : [];
    const id = Date.now() + Math.floor(Math.random()*1000);
    rules.push({
      id,
      startNodeId: n?.id ?? -1,
      startSlot: -1,
      color: s.chainPickColor || s.themeColor,
      enabled: true,
    });
    setState({ chainRules: rules });
    const panel = panelRef();
    panel?.__ct_renderChainRules?.();
  }

  function clearChainRules(){
    setState({ chainRules: [] });
    const panel = panelRef();
    panel?.__ct_renderChainRules?.();
  }

  return {
    undo,
    alignLeft: ()=>align("left"),
    alignRight: ()=>align("right"),
    alignTop: ()=>align("top"),
    alignBottom: ()=>align("bottom"),
    alignHCenter: ()=>align("hcenter"),
    alignVCenter: ()=>align("vcenter"),
    distH: ()=>dist("x"),
    distV: ()=>dist("y"),
    alignDistH: ()=>alignAndDistribute("x"),
    alignDistV: ()=>alignAndDistribute("y"),
    autoArrangeH: ()=>autoArrange("x"),
    autoArrangeV: ()=>autoArrange("y"),
    syncSizeBoth: ()=>syncSize("both"),
    syncSizeW: ()=>syncSize("w"),
    syncSizeH: ()=>syncSize("h"),
    applyNodeStyle,
    clearNodeStyle,
    addLabel,
    applyLabelStyle,
    addChainRuleFromSelected,
    clearChainRules,
  };
}

// ---------- Extension ----------
app.registerExtension({
  name: "comfyui.canvas_toolkit",
  setup() {
    console.log("[CanvasToolkit] loaded (fixed9)");

    let state = loadSettings();
    const getState = () => state;
    const setState = (patch) => {
      state = { ...state, ...patch };
      saveSettings(state);
      const p = document.getElementById(ID_PANEL);
      if (p) {
        p.__ct_state = state;
        if (typeof p.__ct_renderChainRules === "function") p.__ct_renderChainRules();
      }
      getCanvas()?.setDirty?.(true,true);
    };
    const getCanvas = () => app?.canvas || window.app?.canvas || null;

    let rafId = 0;
    let lastTick = 0;
    let animWanted = false;

    function setAnimWanted(on){
      const next = !!on;
      if(next === animWanted) return;
      animWanted = next;
      if(animWanted){
        ensureRAF();
      }else if(rafId){
        cancelAnimationFrame(rafId);
        rafId = 0;
      }
    }
    function ensureRAF(){
      if(rafId || !animWanted) return;
      const tick = (t)=>{
        if(!animWanted){
          rafId = 0;
          return;
        }
        rafId = requestAnimationFrame(tick);
        const s = getState();
        const canvas = getCanvas();
        if(!canvas) return;

        const fps = clamp(s.fps || 30, 10, 60);
        const minDt = 1000 / fps;
        if(t - lastTick < minDt) return;
        lastTick = t;

        if (s.enabled && s.effectsEnabled && animWanted) {
          canvas.setDirty(true,true);
        }
      };
      rafId = requestAnimationFrame(tick);
    }

    const tryPatch = () => {
      const canvas = getCanvas();
      if (!canvas || !window.LGraphCanvas) return false;
      try {
        setupCanvasOverrides(getState, setAnimWanted);
        ensureRAF();
        return true;
      } catch (err) {
        console.warn("[CanvasToolkit] setupCanvasOverrides 失败，稍后重试", err);
        return false;
      }
    };

    if (!tryPatch()) {
      let tries=0;
      const timer=setInterval(()=>{
        tries++;
        if(tryPatch() || tries>80) clearInterval(timer);
      }, 200);
    }

    const panelRef = () => document.getElementById(ID_PANEL);
    const labelApi = CANVAS_LABEL_ENABLED ? setupCanvasLabelOverlay(getCanvas, getState) : null;
    const actions = makeActions(getCanvas, getState, setState, panelRef, labelApi);

    // Persist canvas labels with workflow
    if(CANVAS_LABEL_ENABLED){
      try{
        const g = app?.graph;
        const lg = window?.LGraph;
        if(g && lg && !lg.prototype.__ct_labels_persist){
          lg.prototype.__ct_labels_persist = true;
          const _serialize = lg.prototype.serialize;
          const _configure = lg.prototype.configure;
          if(typeof _serialize === "function"){
            lg.prototype.serialize = function(){
              const data = _serialize.apply(this, arguments);
              const labels = labelApi?.getLabels?.() || [];
              if(labels.length){
                data.extra = data.extra || {};
                data.extra.ct_canvas_labels = labels;
              }
              return data;
            };
          }
          if(typeof _configure === "function"){
            lg.prototype.configure = function(info){
              const res = _configure.apply(this, arguments);
              const labels = info?.extra?.ct_canvas_labels;
              if(Array.isArray(labels)){
                labelApi?.setLabels?.(labels, { persist: false });
              }
              return res;
            };
          }
        }
      }catch(err){
        console.warn("[CanvasToolkit] 标签持久化挂载失败", err);
      }
    }else{
      try{
        const layer = document.getElementById(ID_LABEL_LAYER);
        if(layer?.parentElement) layer.parentElement.removeChild(layer);
      }catch{}
    }

    const panel = ensurePanel(state, (patch)=>{ setState(patch); }, actions);
    const togglePanel = () => { panel.style.display = (panel.style.display==="none") ? "block" : "none"; };
    ensureFab(togglePanel);
    setTimeout(() => { if (!document.getElementById(ID_FAB)) ensureFab(togglePanel); }, 800);
    setTimeout(() => { if (!document.getElementById(ID_FAB)) ensureFab(togglePanel); }, 2200);

    const restore = setInterval(()=>{
      const canvas = getCanvas();
      const g = canvas?.graph;
      if(!g || !Array.isArray(g._nodes)) return;
      for(const n of g._nodes){
        const st = n?.properties?.__ct_style;
        if(st?.bgcolor) n.bgcolor = st.bgcolor;
        if(st?.titleBarColor) n.color = st.titleBarColor;
        if(st?.titleTextColor) n.title_color = st.titleTextColor;
      }
      clearInterval(restore);
      canvas?.setDirty?.(true,true);
    }, 600);

    if(!window.__ct_align_panel_hotkey){
      window.__ct_align_panel_hotkey = true;
      const isTyping = (e)=>{
        const t = e?.target;
        const tag = t?.tagName?.toLowerCase?.() || "";
        if(tag === "input" || tag === "textarea" || tag === "select") return true;
        if(t?.isContentEditable) return true;
        return false;
      };
      window.addEventListener("keydown", (e)=>{
        if(isTyping(e)) return;
        if(e.code !== "F8") return;
        if(e.repeat) return;
        const panel = ensureAlignPanel(actions);
        panel.style.display = (panel.style.display === "none") ? "block" : "none";
        e.preventDefault();
        e.stopPropagation();
      }, true);
    }
  },
});
