% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CellChat_class.R
\name{createCellChat}
\alias{createCellChat}
\title{Create a new CellChat object from a data matrix, Seurat or SingleCellExperiment object}
\usage{
createCellChat(
  object,
  meta = NULL,
  group.by = NULL,
  datatype = c("RNA", "spatial"),
  coordinates = NULL,
  spatial.factors = NULL,
  assay = NULL,
  do.sparse = T
)
}
\arguments{
\item{object}{a normalized (NOT count) data matrix (genes by cells), Seurat or SingleCellExperiment object}

\item{meta}{a data frame (rows are cells with rownames) consisting of cell information, which will be used for defining cell groups.
If input is a Seurat or SingleCellExperiment object, the meta data in the object will be used}

\item{group.by}{a char name of the variable in meta data, defining cell groups.
If input is a data matrix and group.by is NULL, the input `meta` should contain a column named 'labels',
If input is a Seurat or SingleCellExperiment object, USER must provide `group.by` to define the cell groups. e.g, group.by = "ident" for Seurat object}

\item{datatype}{By default datatype = "RNA"; when running CellChat on spatial imaging data, set datatype = "spatial" and input `spatial.factors`}

\item{coordinates}{a data matrix in which each row gives the spatial locations/coordinates of each cell/spot}

\item{spatial.factors}{a data frame containing two distance factors `ratio` and `tol`, which is dependent on spatial transcriptomics technologies (and specific datasets).

USER must input this data frame when datatype = "spatial". spatial.factors must contain an element named `ratio`, which is the conversion factor when converting spatial coordinates from Pixels or other units to Micrometers (i.e.,Microns). For example, setting `ratio = 0.18` indicates that 1 pixel equals 0.18um in the coordinates,

and another element named `tol`, which is the tolerance factor to increase the robustness when comparing the center-to-center distance against the `interaction.range`. This can be the half value of cell/spot size in the unit of um. If the cell/spot size is not known, we provide a function `computeCellDistance` to compute the cell center-to-center distance. `tol` can be the the half value of the minimum center-to-center distance. Of note, CellChat does not need an accurate tolerance factor, which is used for determining whether considering the cell-pair as spatially proximal if their distance is greater than `interaction.range` but smaller than "`interaction.range` + `tol`".}

\item{assay}{Assay to use when the input is a Seurat or SingleCellExperiment object. NB: The data in the `integrated` assay in Seurat is not suitable for CellChat analysis because it contains negative values.}

\item{do.sparse}{whether use sparse format}
}
\description{
Create a new CellChat object from a data matrix, Seurat or SingleCellExperiment object
}
\examples{
\dontrun{
Create a CellChat object from single-cell transcriptomics data
# Input is a data matrix
## create a dataframe consisting of the cell labels
meta = data.frame(labels = cell.labels, row.names = names(cell.labels))
cellChat <- createCellChat(object = data.input, meta = meta, group.by = "labels")

# input is a Seurat object
## use the default cell identities of Seurat object
cellChat <- createCellChat(object = seurat.obj, group.by = "ident", assay = "RNA")
## use other meta information as cell groups
cellChat <- createCellChat(object = seurat.obj, group.by = "seurat.clusters")

# input is a SingleCellExperiment object
cellChat <- createCellChat(object = sce.obj, group.by = "sce.clusters")

# input is a AnnData object
sce <- zellkonverter::readH5AD(file = "adata.h5ad")
assayNames(sce) # retrieve all the available assays within sce object
counts <- assay(sce, "X") # add a new assay entry "logcounts" if not available and make sure this is the original count data matrix
library.size <- Matrix::colSums(counts)
logcounts(sce) <- log1p(Matrix::t(Matrix::t(counts)/library.size) * 10000)
meta <- as.data.frame(SingleCellExperiment::colData(sce))
cellChat <- createCellChat(object = sce, group.by = "sce.clusters")


Create a CellChat object from spatial transcriptomics data
# Input is a data matrix
cellChat <- createCellChat(object = data.input, meta = meta, group.by = "labels",
                           datatype = "spatial", coordinates = coordinates, spatial.factors = spatial.factors)

# input is a Seurat object
cellChat <- createCellChat(object = seurat.obj, group.by = "ident", assay = "SCT",
                           datatype = "spatial", spatial.factors = spatial.factors)

}
}
