#!/usr/bin/env python
"""
ros_thruster_wrench_comparison_visualizer.py

Author: Scott Mayberry
Date: 2025-01-10
Description:
    This script visualizes thruster commands and wrench data for the Miniature Underwater Robot (MUR).
    It subscribes to thruster command topics and wrench data, processes the incoming information,
    and generates 3D plots to compare thruster-generated forces and torques against desired wrench
    commands. The visualization helps in assessing the performance and accuracy of thruster control
    algorithms by providing a graphical representation of force and torque vectors in real-time.

Usage:
    Ensure that the ROS master is running and the necessary topics are being published before executing this script.
    Launch the script using ROS launch or run it directly:

    Direct Execution:
        ```bash
        rosrun mur_model ros_thruster_wrench_comparison_visualizer.py
        ```

    Using Launch File:
        ```bash
        roslaunch mur_model mur_model.launch
        ```

Dependencies:
    - rospy
    - numpy
    - std_msgs.msg (Float64MultiArray, Int32MultiArray)
    - geometry_msgs.msg (WrenchStamped, TransformStamped)
    - matplotlib.pyplot
    - mpl_toolkits.mplot3d (Axes3D)
    - threading
    - queue
    - tf2_ros
    - tf.transformations

License:
    MIT License
"""

import rospy
from std_msgs.msg import Float64MultiArray, Int32MultiArray
from geometry_msgs.msg import WrenchStamped, TransformStamped
import numpy as np
import matplotlib.pyplot as plt
from mpl_toolkits.mplot3d import Axes3D
import threading
import queue
import tf2_ros
import time
from tf.transformations import quaternion_matrix

# Initialize three subplots for visualization: thrusters, linear forces, and torques
fig = plt.figure(figsize=(15, 5))
ax_thruster = fig.add_subplot(131, projection='3d', title='Thrusters')
ax_linear = fig.add_subplot(132, projection='3d', title='Linear Forces')
ax_torque = fig.add_subplot(133, projection='3d', title='Torques')

# Initialize global variables and queues for data handling
force_thruster_commands = None
update_queue = queue.Queue()  # Queue for updating thruster and COM data
wrench_queue = queue.Queue()  # Queue for wrench data
lengthOfQuiver = 0.01  # Length scaling factor for quiver plots
torque_queue = queue.Queue()  # (Unused in current implementation)

def set_axes_equal(ax):
    """
    Sets the aspect ratio of the 3D plot axes to be equal.

    This ensures that the 3D plots are proportionally accurate, preventing distortion
    where spheres may appear as ellipsoids or cubes as rectangular prisms.

    Args:
        ax (mpl_toolkits.mplot3d.axes3d.Axes3D): The 3D axes to adjust.
    """
    '''Make axes of 3D plot have equal scale so that spheres appear as spheres,
    cubes as cubes, etc..'''
    x_limits = ax.get_xlim3d()
    y_limits = ax.get_ylim3d()
    z_limits = ax.get_zlim3d()

    x_range = abs(x_limits[1] - x_limits[0])
    x_middle = np.mean(x_limits)
    y_range = abs(y_limits[1] - y_limits[0])
    y_middle = np.mean(y_limits)
    z_range = abs(z_limits[1] - z_limits[0])
    z_middle = np.mean(z_limits)

    # The plot bounding box is a sphere in the sense of the infinity
    # norm, hence we call half the max range the plot radius.
    plot_radius = 0.5 * max([x_range, y_range, z_range])

    ax.set_xlim3d([x_middle - plot_radius, x_middle + plot_radius])
    ax.set_ylim3d([y_middle - plot_radius, y_middle + plot_radius])
    ax.set_zlim3d([z_middle - plot_radius, z_middle + plot_radius])

def update_thruster_plot(thruster_data, com_pos, wrench_pos, wrench_force):
    """
    Updates the thruster subplot with the current thruster forces.

    This function plots the force vectors generated by each thruster in 3D space.
    It also sums up the total force and optionally displays it at the center of mass.

    Args:
        thruster_data (list of tuples): Each tuple contains the position and force vector of a thruster.
        com_pos (np.ndarray): The position of the robot's center of mass.
        wrench_pos (np.ndarray): The position where the wrench is applied (unused in current context).
        wrench_force (np.ndarray): The desired wrench force vector.
    """
    global lengthOfQuiver
    ax = ax_thruster  # Select the thrusters subplot
    # Initialize total force vector
    total_force = np.array([0.0, 0.0, 0.0])

    # Draw the first thruster force with a label for the legend
    pos, force_vector = thruster_data[0]
    if ax.get_legend() is None or 'Thruster Forces' not in [l.get_label() for l in ax.get_legend().get_lines()]:
        ax.quiver(pos[0], pos[1], pos[2], force_vector[0], force_vector[1], force_vector[2],
                  color='b', length=lengthOfQuiver, label='Thruster Forces')
    else:
        ax.quiver(pos[0], pos[1], pos[2], force_vector[0], force_vector[1], force_vector[2],
                  color='b', length=lengthOfQuiver)
    total_force += force_vector

    # Draw the remaining thruster forces without adding to the legend
    for pos, force_vector in thruster_data[1:]:
        ax.quiver(pos[0], pos[1], pos[2], force_vector[0], force_vector[1], force_vector[2],
                  length=lengthOfQuiver, color='b')
        total_force += force_vector

    # # Draw the wrench force (temporarily commented out)
    # if ax.get_legend() is None or 'Wrench Force' not in [l.get_label() for l in ax.get_legend().get_lines()]:
    #     ax.quiver(wrench_pos[0], wrench_pos[1], wrench_pos[2],
    #               wrench_force[0], wrench_force[1], wrench_force[2],
    #               color='g', length=lengthOfQuiver, label='Wrench Force')
    # else:
    #     ax.quiver(wrench_pos[0], wrench_pos[1], wrench_pos[2],
    #               wrench_force[0], wrench_force[1], wrench_force[2],
    #               color='g', length=lengthOfQuiver)

    # # If com_pos is not None, draw the total force vector at the COM position (temporarily commented out)
    # if com_pos is not None:
    #     if ax.get_legend() is None or 'Total Force' not in [l.get_label() for l in ax.get_legend().get_lines()]:
    #         ax.quiver(com_pos[0], com_pos[1], com_pos[2],
    #                   total_force[0], total_force[1], total_force[2],
    #                   color='r', alpha=0.6, length=lengthOfQuiver, label='Total Force')
    #     else:
    #         ax.quiver(com_pos[0], com_pos[1], com_pos[2],
    #                   total_force[0], total_force[1], total_force[2],
    #                   color='r', alpha=0.6, length=lengthOfQuiver)

    set_axes_equal(ax)  # Ensure the axes have equal scaling

    # Set the axes labels for clarity
    ax.set_xlabel('X')
    ax.set_ylabel('Y')
    ax.set_zlabel('Z')
    # Add a legend to differentiate between thruster forces and other vectors
    ax.legend()

def compute_torque(pos, force_vector, com_pos):
    """
    Computes the torque generated by a thruster based on its position and force vector.

    Args:
        pos (np.ndarray): The position vector of the thruster relative to the center of mass.
        force_vector (np.ndarray): The force vector generated by the thruster.
        com_pos (np.ndarray): The position of the center of mass.

    Returns:
        np.ndarray: The resulting torque vector.
    """
    r = pos - com_pos  # Position vector from center of mass to thruster
    return np.cross(r, force_vector)  # Torque = r x F

def update_torque_plot(thruster_data, com_pos, wrench_pos, wrench_torque):
    """
    Updates the torque subplot with the current thruster-generated torques and desired wrench torque.

    This function plots the torque vectors generated by each thruster and compares them with the desired
    wrench torque in 3D space.

    Args:
        thruster_data (list of tuples): Each tuple contains the position and force vector of a thruster.
        com_pos (np.ndarray): The position of the robot's center of mass.
        wrench_pos (np.ndarray): The position where the wrench is applied.
        wrench_torque (np.ndarray): The desired torque vector.
    """
    global lengthOfQuiver
    ax = ax_torque  # Select the torques subplot

    # Compute the total torque from all thrusters
    total_torque = np.array([0.0, 0.0, 0.0])
    for pos, force_vector in thruster_data:
        total_torque += compute_torque(pos, force_vector, com_pos)

    # Draw the total torque from thrusters
    ax.quiver(0, 0, 0, total_torque[0], total_torque[1], total_torque[2],
              color='r', alpha=0.5, label='Thruster Torque', length=lengthOfQuiver)

    # Draw the desired wrench torque
    ax.quiver(0, 0, 0, wrench_torque[0], wrench_torque[1], wrench_torque[2],
              color='g', alpha=0.5, label='Wrench Torque', length=lengthOfQuiver)

    # Set the axes labels for clarity
    ax.set_xlabel('Roll')
    ax.set_ylabel('Pitch')
    ax.set_zlabel('Yaw')

    set_axes_equal(ax)  # Ensure the axes have equal scaling

    # Add a legend to differentiate between thruster torque and wrench torque
    ax.legend()

def update_linear_plot(thruster_data, com_pos, wrench_pos, wrench_force):
    """
    Updates the linear forces subplot with the current thruster-generated forces and desired wrench force.

    This function plots the force vectors generated by each thruster and compares them with the desired
    wrench force in 3D space.

    Args:
        thruster_data (list of tuples): Each tuple contains the position and force vector of a thruster.
        com_pos (np.ndarray): The position of the robot's center of mass.
        wrench_pos (np.ndarray): The position where the wrench is applied.
        wrench_force (np.ndarray): The desired force vector.
    """
    global lengthOfQuiver
    ax = ax_linear  # Select the linear forces subplot

    # Compute the total force from all thrusters
    total_force = np.array([0.0, 0.0, 0.0])
    for pos, force_vector in thruster_data:
        total_force += force_vector

    # Draw the total force from thrusters
    ax.quiver(0, 0, 0, total_force[0], total_force[1], total_force[2],
              color='r', alpha=0.5, label='Thruster Force', length=lengthOfQuiver)

    # Draw the desired wrench force
    ax.quiver(0, 0, 0, wrench_force[0], wrench_force[1], wrench_force[2],
              color='g', alpha=0.5, label='Wrench Force', length=lengthOfQuiver)

    # Set the axes labels for clarity
    ax.set_xlabel('X')
    ax.set_ylabel('Y')
    ax.set_zlabel('Z')

    set_axes_equal(ax)  # Ensure the axes have equal scaling

    # Add a legend to differentiate between thruster forces and wrench forces
    ax.legend()

def update_plots():
    """
    Continuously updates the 3D plots with new thruster and wrench data.

    This function runs in a separate thread, retrieving data from the update and wrench queues,
    and refreshing the plots accordingly. It ensures that the visualization remains up-to-date
    with the latest thruster commands and wrench measurements.
    """
    plt.tight_layout()  # Adjust subplots to fit into the figure area
    while not rospy.is_shutdown():
        time.sleep(0.02)  # Short delay to control update rate
        if wrench_queue.empty() or update_queue.empty():
            continue  # Skip iteration if there is no new data
        thruster_data, com_pos = update_queue.get()  # Retrieve thruster and COM data
        wrench_pos, wrench_force, wrench_torque = wrench_queue.get()  # Retrieve wrench data
        ax_thruster.clear()  # Clear the thrusters subplot for fresh plotting
        ax_linear.clear()    # Clear the linear forces subplot for fresh plotting
        ax_torque.clear()    # Clear the torques subplot for fresh plotting
        update_thruster_plot(thruster_data, com_pos, wrench_pos, wrench_force)  # Update thruster forces
        update_linear_plot(thruster_data, com_pos, wrench_pos, wrench_force)    # Update linear forces
        update_torque_plot(thruster_data, com_pos, wrench_pos, wrench_torque)    # Update torques
        plt.draw()  # Redraw the current figure
        plt.pause(0.01)  # Pause briefly to allow the plot to update

def force_thruster_commands_callback(msg):
    """
    Callback function for processing incoming thruster force commands.

    This function retrieves thruster force commands from the ROS topic, transforms the force
    vectors into the robot's base frame, and queues the data for visualization.

    Args:
        msg (Float64MultiArray): The incoming thruster force commands.
    """
    if not update_queue.empty():
        return  # Skip processing if the queue is already occupied
    force_thruster_commands = np.array(msg.data)  # Convert the message data to a NumPy array
    thruster_frames = [f'thruster_{i}' for i in range(len(force_thruster_commands))]  # Generate thruster frame names

    thruster_data = []
    for idx, thruster_frame in enumerate(thruster_frames):
        try:
            # Look up the transform from 'base_link' to the thruster frame
            transform = tf_buffer.lookup_transform('base_link', thruster_frame, rospy.Time(0))
            pos = np.array([
                transform.transform.translation.x,
                transform.transform.translation.y,
                transform.transform.translation.z
            ])  # Extract thruster position
            # Get the rotation matrix from the thruster's quaternion orientation
            quat = transform.transform.rotation
            rotation_matrix = quaternion_matrix([quat.x, quat.y, quat.z, quat.w])
            # Rotate the force vector to align with the thruster's x-axis
            force_magnitude = force_thruster_commands[idx]
            force_vector = np.dot(rotation_matrix, np.array([force_magnitude, 0, 0, 1]))[:3]
            thruster_data.append((pos, force_vector))  # Append position and force vector
        except (tf2_ros.LookupException, tf2_ros.ConnectivityException, tf2_ros.ExtrapolationException) as e:
            rospy.logwarn(f"Error getting transform for {thruster_frame}: {e}")  # Log any transformation errors

    # Fetch the center of mass (COM) position
    try:
        transform_com = tf_buffer.lookup_transform('base_link', 'com', rospy.Time(0))
        com_pos = np.array([
            transform_com.transform.translation.x,
            transform_com.transform.translation.y,
            transform_com.transform.translation.z
        ])  # Extract COM position
    except (tf2_ros.LookupException, tf2_ros.ConnectivityException, tf2_ros.ExtrapolationException) as e:
        rospy.logwarn(f"Error getting transform for com: {e}")  # Log any COM transformation errors
        com_pos = None  # Set COM position to None if transform fails

    update_queue.put((thruster_data, com_pos))  # Queue the thruster and COM data for plotting

def body_frame_wrench_callback(msg):
    """
    Callback function for processing incoming wrench data.

    This function retrieves wrench data from the ROS topic, transforms the wrench
    vectors into the robot's base frame, and queues the data for visualization.

    Args:
        msg (WrenchStamped): The incoming wrench data containing force and torque.
    """
    if not wrench_queue.empty():
        return  # Skip processing if the queue is already occupied
    try:
        # Look up the transform from 'base_link' to the wrench's frame
        transform = tf_buffer.lookup_transform('base_link', msg.header.frame_id, rospy.Time(0))
        pos = np.array([
            transform.transform.translation.x,
            transform.transform.translation.y,
            transform.transform.translation.z
        ])  # Extract wrench position

        # Extract the linear force components from the wrench message
        force = np.array([
            msg.wrench.force.x,
            msg.wrench.force.y,
            msg.wrench.force.z
        ])
        # Extract the torque components from the wrench message
        torque = np.array([
            msg.wrench.torque.x,
            msg.wrench.torque.y,
            msg.wrench.torque.z
        ])

        # Add the wrench data to the wrench queue for plotting
        wrench_queue.put((pos, force, torque))
    except (tf2_ros.LookupException, tf2_ros.ConnectivityException, tf2_ros.ExtrapolationException) as e:
        rospy.logwarn(f"Error getting transform for {msg.header.frame_id}: {e}")  # Log any transformation errors

if __name__ == '__main__':
    """
    Entry point for the Thruster Wrench Comparison Visualizer script.

    This section initializes the ROS node, sets up the TF2 buffer and listener for coordinate transformations,
    subscribes to relevant ROS topics for thruster commands and wrench data, and starts the plotting thread
    for real-time visualization.
    """
    rospy.init_node('thruster_command_visualizer_node')  # Initialize the ROS node

    tf_buffer = tf2_ros.Buffer()  # Create a TF2 buffer to store transforms
    tf_listener = tf2_ros.TransformListener(tf_buffer)  # Initialize the TF2 listener
    rospy.sleep(2)  # Allow some time for the TF listener to populate the buffer

    # Subscribers for thruster commands and wrench data
    rospy.Subscriber('force_thruster_commands', Float64MultiArray, force_thruster_commands_callback)
    rospy.Subscriber('requested_thruster_wrench', WrenchStamped, body_frame_wrench_callback)

    # Start the plotting in the main thread
    update_plots()
