import type { OnFeaturesChange } from '@/app/components/base/features/types'
import { RiEqualizer2Line } from '@remixicon/react'
import { produce } from 'immer'
import * as React from 'react'
import { useCallback, useState } from 'react'
import { useTranslation } from 'react-i18next'
import Button from '@/app/components/base/button'
import { useFeatures, useFeaturesStore } from '@/app/components/base/features/hooks'
import FeatureCard from '@/app/components/base/features/new-feature-panel/feature-card'
import VoiceSettings from '@/app/components/base/features/new-feature-panel/text-to-speech/voice-settings'
import { FeatureEnum } from '@/app/components/base/features/types'
import { TextToAudio } from '@/app/components/base/icons/src/vender/features'
import { languages } from '@/i18n-config/language'
import { TtsAutoPlay } from '@/types/app'

type Props = {
  disabled: boolean
  onChange?: OnFeaturesChange
}

const TextToSpeech = ({
  disabled,
  onChange,
}: Props) => {
  const { t } = useTranslation()
  const textToSpeech = useFeatures(s => s.features.text2speech) // .language .voice .autoPlay
  const languageInfo = languages.find(i => i.value === textToSpeech?.language)
  const [modalOpen, setModalOpen] = useState(false)
  const [isHovering, setIsHovering] = useState(false)
  const features = useFeatures(s => s.features)
  const featuresStore = useFeaturesStore()

  const handleChange = useCallback((type: FeatureEnum, enabled: boolean) => {
    const {
      features,
      setFeatures,
    } = featuresStore!.getState()

    const newFeatures = produce(features, (draft) => {
      draft[type] = {
        ...draft[type],
        enabled,
      }
    })
    setFeatures(newFeatures)
    if (onChange)
      onChange()
  }, [featuresStore, onChange])

  return (
    <FeatureCard
      icon={(
        <div className="shrink-0 rounded-lg border-[0.5px] border-divider-subtle bg-util-colors-violet-violet-600 p-1 shadow-xs">
          <TextToAudio className="h-4 w-4 text-text-primary-on-surface" />
        </div>
      )}
      title={t('feature.textToSpeech.title', { ns: 'appDebug' })}
      value={!!features.text2speech?.enabled}
      onChange={state => handleChange(FeatureEnum.text2speech, state)}
      onMouseEnter={() => setIsHovering(true)}
      onMouseLeave={() => setIsHovering(false)}
      disabled={disabled}
    >
      <>
        {!features.text2speech?.enabled && (
          <div className="line-clamp-2 min-h-8 text-text-tertiary system-xs-regular">{t('feature.textToSpeech.description', { ns: 'appDebug' })}</div>
        )}
        {!!features.text2speech?.enabled && (
          <>
            {!isHovering && !modalOpen && (
              <div className="flex items-center gap-4 pt-0.5">
                <div className="">
                  <div className="mb-0.5 text-text-tertiary system-2xs-medium-uppercase">{t('voice.voiceSettings.language', { ns: 'appDebug' })}</div>
                  <div className="text-text-secondary system-xs-regular">{languageInfo?.name || '-'}</div>
                </div>
                <div className="h-[27px] w-px rotate-12 bg-divider-subtle"></div>
                <div className="">
                  <div className="mb-0.5 text-text-tertiary system-2xs-medium-uppercase">{t('voice.voiceSettings.voice', { ns: 'appDebug' })}</div>
                  <div className="text-text-secondary system-xs-regular">{features.text2speech?.voice || t('voice.defaultDisplay', { ns: 'appDebug' })}</div>
                </div>
                <div className="h-[27px] w-px rotate-12 bg-divider-subtle"></div>
                <div className="">
                  <div className="mb-0.5 text-text-tertiary system-2xs-medium-uppercase">{t('voice.voiceSettings.autoPlay', { ns: 'appDebug' })}</div>
                  <div className="text-text-secondary system-xs-regular">{features.text2speech?.autoPlay === TtsAutoPlay.enabled ? t('voice.voiceSettings.autoPlayEnabled', { ns: 'appDebug' }) : t('voice.voiceSettings.autoPlayDisabled', { ns: 'appDebug' })}</div>
                </div>
              </div>
            )}
            {(isHovering || modalOpen) && (
              <VoiceSettings open={modalOpen && !disabled} onOpen={setModalOpen} onChange={onChange}>
                <Button className="w-full" disabled={disabled}>
                  <RiEqualizer2Line className="mr-1 h-4 w-4" />
                  {t('voice.voiceSettings.title', { ns: 'appDebug' })}
                </Button>
              </VoiceSettings>
            )}
          </>
        )}
      </>
    </FeatureCard>
  )
}

export default TextToSpeech
