// AUTO-GENERATED — DO NOT EDIT
// Source: https://github.com/anthropics/skills
import type { SkillTemplateNode } from '../types'

const children: SkillTemplateNode[] = [
  {
    "name": "SKILL.md",
    "node_type": "file",
    "content": "---\nname: pdf\ndescription: Use this skill whenever the user wants to do anything with PDF files. This includes reading or extracting text/tables from PDFs, combining or merging multiple PDFs into one, splitting PDFs apart, rotating pages, adding watermarks, creating new PDFs, filling PDF forms, encrypting/decrypting PDFs, extracting images, and OCR on scanned PDFs to make them searchable. If the user mentions a .pdf file or asks to produce one, use this skill.\nlicense: Proprietary. LICENSE.txt has complete terms\n---\n\n# PDF Processing Guide\n\n## Overview\n\nThis guide covers essential PDF processing operations using Python libraries and command-line tools. For advanced features, JavaScript libraries, and detailed examples, see REFERENCE.md. If you need to fill out a PDF form, read FORMS.md and follow its instructions.\n\n## Quick Start\n\n```python\nfrom pypdf import PdfReader, PdfWriter\n\n# Read a PDF\nreader = PdfReader(\"document.pdf\")\nprint(f\"Pages: {len(reader.pages)}\")\n\n# Extract text\ntext = \"\"\nfor page in reader.pages:\n    text += page.extract_text()\n```\n\n## Python Libraries\n\n### pypdf - Basic Operations\n\n#### Merge PDFs\n```python\nfrom pypdf import PdfWriter, PdfReader\n\nwriter = PdfWriter()\nfor pdf_file in [\"doc1.pdf\", \"doc2.pdf\", \"doc3.pdf\"]:\n    reader = PdfReader(pdf_file)\n    for page in reader.pages:\n        writer.add_page(page)\n\nwith open(\"merged.pdf\", \"wb\") as output:\n    writer.write(output)\n```\n\n#### Split PDF\n```python\nreader = PdfReader(\"input.pdf\")\nfor i, page in enumerate(reader.pages):\n    writer = PdfWriter()\n    writer.add_page(page)\n    with open(f\"page_{i+1}.pdf\", \"wb\") as output:\n        writer.write(output)\n```\n\n#### Extract Metadata\n```python\nreader = PdfReader(\"document.pdf\")\nmeta = reader.metadata\nprint(f\"Title: {meta.title}\")\nprint(f\"Author: {meta.author}\")\nprint(f\"Subject: {meta.subject}\")\nprint(f\"Creator: {meta.creator}\")\n```\n\n#### Rotate Pages\n```python\nreader = PdfReader(\"input.pdf\")\nwriter = PdfWriter()\n\npage = reader.pages[0]\npage.rotate(90)  # Rotate 90 degrees clockwise\nwriter.add_page(page)\n\nwith open(\"rotated.pdf\", \"wb\") as output:\n    writer.write(output)\n```\n\n### pdfplumber - Text and Table Extraction\n\n#### Extract Text with Layout\n```python\nimport pdfplumber\n\nwith pdfplumber.open(\"document.pdf\") as pdf:\n    for page in pdf.pages:\n        text = page.extract_text()\n        print(text)\n```\n\n#### Extract Tables\n```python\nwith pdfplumber.open(\"document.pdf\") as pdf:\n    for i, page in enumerate(pdf.pages):\n        tables = page.extract_tables()\n        for j, table in enumerate(tables):\n            print(f\"Table {j+1} on page {i+1}:\")\n            for row in table:\n                print(row)\n```\n\n#### Advanced Table Extraction\n```python\nimport pandas as pd\n\nwith pdfplumber.open(\"document.pdf\") as pdf:\n    all_tables = []\n    for page in pdf.pages:\n        tables = page.extract_tables()\n        for table in tables:\n            if table:  # Check if table is not empty\n                df = pd.DataFrame(table[1:], columns=table[0])\n                all_tables.append(df)\n\n# Combine all tables\nif all_tables:\n    combined_df = pd.concat(all_tables, ignore_index=True)\n    combined_df.to_excel(\"extracted_tables.xlsx\", index=False)\n```\n\n### reportlab - Create PDFs\n\n#### Basic PDF Creation\n```python\nfrom reportlab.lib.pagesizes import letter\nfrom reportlab.pdfgen import canvas\n\nc = canvas.Canvas(\"hello.pdf\", pagesize=letter)\nwidth, height = letter\n\n# Add text\nc.drawString(100, height - 100, \"Hello World!\")\nc.drawString(100, height - 120, \"This is a PDF created with reportlab\")\n\n# Add a line\nc.line(100, height - 140, 400, height - 140)\n\n# Save\nc.save()\n```\n\n#### Create PDF with Multiple Pages\n```python\nfrom reportlab.lib.pagesizes import letter\nfrom reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, PageBreak\nfrom reportlab.lib.styles import getSampleStyleSheet\n\ndoc = SimpleDocTemplate(\"report.pdf\", pagesize=letter)\nstyles = getSampleStyleSheet()\nstory = []\n\n# Add content\ntitle = Paragraph(\"Report Title\", styles['Title'])\nstory.append(title)\nstory.append(Spacer(1, 12))\n\nbody = Paragraph(\"This is the body of the report. \" * 20, styles['Normal'])\nstory.append(body)\nstory.append(PageBreak())\n\n# Page 2\nstory.append(Paragraph(\"Page 2\", styles['Heading1']))\nstory.append(Paragraph(\"Content for page 2\", styles['Normal']))\n\n# Build PDF\ndoc.build(story)\n```\n\n#### Subscripts and Superscripts\n\n**IMPORTANT**: Never use Unicode subscript/superscript characters (₀₁₂₃₄₅₆₇₈₉, ⁰¹²³⁴⁵⁶⁷⁸⁹) in ReportLab PDFs. The built-in fonts do not include these glyphs, causing them to render as solid black boxes.\n\nInstead, use ReportLab's XML markup tags in Paragraph objects:\n```python\nfrom reportlab.platypus import Paragraph\nfrom reportlab.lib.styles import getSampleStyleSheet\n\nstyles = getSampleStyleSheet()\n\n# Subscripts: use <sub> tag\nchemical = Paragraph(\"H<sub>2</sub>O\", styles['Normal'])\n\n# Superscripts: use <super> tag\nsquared = Paragraph(\"x<super>2</super> + y<super>2</super>\", styles['Normal'])\n```\n\nFor canvas-drawn text (not Paragraph objects), manually adjust font the size and position rather than using Unicode subscripts/superscripts.\n\n## Command-Line Tools\n\n### pdftotext (poppler-utils)\n```bash\n# Extract text\npdftotext input.pdf output.txt\n\n# Extract text preserving layout\npdftotext -layout input.pdf output.txt\n\n# Extract specific pages\npdftotext -f 1 -l 5 input.pdf output.txt  # Pages 1-5\n```\n\n### qpdf\n```bash\n# Merge PDFs\nqpdf --empty --pages file1.pdf file2.pdf -- merged.pdf\n\n# Split pages\nqpdf input.pdf --pages . 1-5 -- pages1-5.pdf\nqpdf input.pdf --pages . 6-10 -- pages6-10.pdf\n\n# Rotate pages\nqpdf input.pdf output.pdf --rotate=+90:1  # Rotate page 1 by 90 degrees\n\n# Remove password\nqpdf --password=mypassword --decrypt encrypted.pdf decrypted.pdf\n```\n\n### pdftk (if available)\n```bash\n# Merge\npdftk file1.pdf file2.pdf cat output merged.pdf\n\n# Split\npdftk input.pdf burst\n\n# Rotate\npdftk input.pdf rotate 1east output rotated.pdf\n```\n\n## Common Tasks\n\n### Extract Text from Scanned PDFs\n```python\n# Requires: pip install pytesseract pdf2image\nimport pytesseract\nfrom pdf2image import convert_from_path\n\n# Convert PDF to images\nimages = convert_from_path('scanned.pdf')\n\n# OCR each page\ntext = \"\"\nfor i, image in enumerate(images):\n    text += f\"Page {i+1}:\\n\"\n    text += pytesseract.image_to_string(image)\n    text += \"\\n\\n\"\n\nprint(text)\n```\n\n### Add Watermark\n```python\nfrom pypdf import PdfReader, PdfWriter\n\n# Create watermark (or load existing)\nwatermark = PdfReader(\"watermark.pdf\").pages[0]\n\n# Apply to all pages\nreader = PdfReader(\"document.pdf\")\nwriter = PdfWriter()\n\nfor page in reader.pages:\n    page.merge_page(watermark)\n    writer.add_page(page)\n\nwith open(\"watermarked.pdf\", \"wb\") as output:\n    writer.write(output)\n```\n\n### Extract Images\n```bash\n# Using pdfimages (poppler-utils)\npdfimages -j input.pdf output_prefix\n\n# This extracts all images as output_prefix-000.jpg, output_prefix-001.jpg, etc.\n```\n\n### Password Protection\n```python\nfrom pypdf import PdfReader, PdfWriter\n\nreader = PdfReader(\"input.pdf\")\nwriter = PdfWriter()\n\nfor page in reader.pages:\n    writer.add_page(page)\n\n# Add password\nwriter.encrypt(\"userpassword\", \"ownerpassword\")\n\nwith open(\"encrypted.pdf\", \"wb\") as output:\n    writer.write(output)\n```\n\n## Quick Reference\n\n| Task | Best Tool | Command/Code |\n|------|-----------|--------------|\n| Merge PDFs | pypdf | `writer.add_page(page)` |\n| Split PDFs | pypdf | One page per file |\n| Extract text | pdfplumber | `page.extract_text()` |\n| Extract tables | pdfplumber | `page.extract_tables()` |\n| Create PDFs | reportlab | Canvas or Platypus |\n| Command line merge | qpdf | `qpdf --empty --pages ...` |\n| OCR scanned PDFs | pytesseract | Convert to image first |\n| Fill PDF forms | pdf-lib or pypdf (see FORMS.md) | See FORMS.md |\n\n## Next Steps\n\n- For advanced pypdfium2 usage, see REFERENCE.md\n- For JavaScript libraries (pdf-lib), see REFERENCE.md\n- If you need to fill out a PDF form, follow the instructions in FORMS.md\n- For troubleshooting guides, see REFERENCE.md\n"
  },
  {
    "name": "forms.md",
    "node_type": "file",
    "content": "**CRITICAL: You MUST complete these steps in order. Do not skip ahead to writing code.**\n\nIf you need to fill out a PDF form, first check to see if the PDF has fillable form fields. Run this script from this file's directory:\n `python scripts/check_fillable_fields <file.pdf>`, and depending on the result go to either the \"Fillable fields\" or \"Non-fillable fields\" and follow those instructions.\n\n# Fillable fields\nIf the PDF has fillable form fields:\n- Run this script from this file's directory: `python scripts/extract_form_field_info.py <input.pdf> <field_info.json>`. It will create a JSON file with a list of fields in this format:\n```\n[\n  {\n    \"field_id\": (unique ID for the field),\n    \"page\": (page number, 1-based),\n    \"rect\": ([left, bottom, right, top] bounding box in PDF coordinates, y=0 is the bottom of the page),\n    \"type\": (\"text\", \"checkbox\", \"radio_group\", or \"choice\"),\n  },\n  // Checkboxes have \"checked_value\" and \"unchecked_value\" properties:\n  {\n    \"field_id\": (unique ID for the field),\n    \"page\": (page number, 1-based),\n    \"type\": \"checkbox\",\n    \"checked_value\": (Set the field to this value to check the checkbox),\n    \"unchecked_value\": (Set the field to this value to uncheck the checkbox),\n  },\n  // Radio groups have a \"radio_options\" list with the possible choices.\n  {\n    \"field_id\": (unique ID for the field),\n    \"page\": (page number, 1-based),\n    \"type\": \"radio_group\",\n    \"radio_options\": [\n      {\n        \"value\": (set the field to this value to select this radio option),\n        \"rect\": (bounding box for the radio button for this option)\n      },\n      // Other radio options\n    ]\n  },\n  // Multiple choice fields have a \"choice_options\" list with the possible choices:\n  {\n    \"field_id\": (unique ID for the field),\n    \"page\": (page number, 1-based),\n    \"type\": \"choice\",\n    \"choice_options\": [\n      {\n        \"value\": (set the field to this value to select this option),\n        \"text\": (display text of the option)\n      },\n      // Other choice options\n    ],\n  }\n]\n```\n- Convert the PDF to PNGs (one image for each page) with this script (run from this file's directory):\n`python scripts/convert_pdf_to_images.py <file.pdf> <output_directory>`\nThen analyze the images to determine the purpose of each form field (make sure to convert the bounding box PDF coordinates to image coordinates).\n- Create a `field_values.json` file in this format with the values to be entered for each field:\n```\n[\n  {\n    \"field_id\": \"last_name\", // Must match the field_id from `extract_form_field_info.py`\n    \"description\": \"The user's last name\",\n    \"page\": 1, // Must match the \"page\" value in field_info.json\n    \"value\": \"Simpson\"\n  },\n  {\n    \"field_id\": \"Checkbox12\",\n    \"description\": \"Checkbox to be checked if the user is 18 or over\",\n    \"page\": 1,\n    \"value\": \"/On\" // If this is a checkbox, use its \"checked_value\" value to check it. If it's a radio button group, use one of the \"value\" values in \"radio_options\".\n  },\n  // more fields\n]\n```\n- Run the `fill_fillable_fields.py` script from this file's directory to create a filled-in PDF:\n`python scripts/fill_fillable_fields.py <input pdf> <field_values.json> <output pdf>`\nThis script will verify that the field IDs and values you provide are valid; if it prints error messages, correct the appropriate fields and try again.\n\n# Non-fillable fields\nIf the PDF doesn't have fillable form fields, you'll add text annotations. First try to extract coordinates from the PDF structure (more accurate), then fall back to visual estimation if needed.\n\n## Step 1: Try Structure Extraction First\n\nRun this script to extract text labels, lines, and checkboxes with their exact PDF coordinates:\n`python scripts/extract_form_structure.py <input.pdf> form_structure.json`\n\nThis creates a JSON file containing:\n- **labels**: Every text element with exact coordinates (x0, top, x1, bottom in PDF points)\n- **lines**: Horizontal lines that define row boundaries\n- **checkboxes**: Small square rectangles that are checkboxes (with center coordinates)\n- **row_boundaries**: Row top/bottom positions calculated from horizontal lines\n\n**Check the results**: If `form_structure.json` has meaningful labels (text elements that correspond to form fields), use **Approach A: Structure-Based Coordinates**. If the PDF is scanned/image-based and has few or no labels, use **Approach B: Visual Estimation**.\n\n---\n\n## Approach A: Structure-Based Coordinates (Preferred)\n\nUse this when `extract_form_structure.py` found text labels in the PDF.\n\n### A.1: Analyze the Structure\n\nRead form_structure.json and identify:\n\n1. **Label groups**: Adjacent text elements that form a single label (e.g., \"Last\" + \"Name\")\n2. **Row structure**: Labels with similar `top` values are in the same row\n3. **Field columns**: Entry areas start after label ends (x0 = label.x1 + gap)\n4. **Checkboxes**: Use the checkbox coordinates directly from the structure\n\n**Coordinate system**: PDF coordinates where y=0 is at TOP of page, y increases downward.\n\n### A.2: Check for Missing Elements\n\nThe structure extraction may not detect all form elements. Common cases:\n- **Circular checkboxes**: Only square rectangles are detected as checkboxes\n- **Complex graphics**: Decorative elements or non-standard form controls\n- **Faded or light-colored elements**: May not be extracted\n\nIf you see form fields in the PDF images that aren't in form_structure.json, you'll need to use **visual analysis** for those specific fields (see \"Hybrid Approach\" below).\n\n### A.3: Create fields.json with PDF Coordinates\n\nFor each field, calculate entry coordinates from the extracted structure:\n\n**Text fields:**\n- entry x0 = label x1 + 5 (small gap after label)\n- entry x1 = next label's x0, or row boundary\n- entry top = same as label top\n- entry bottom = row boundary line below, or label bottom + row_height\n\n**Checkboxes:**\n- Use the checkbox rectangle coordinates directly from form_structure.json\n- entry_bounding_box = [checkbox.x0, checkbox.top, checkbox.x1, checkbox.bottom]\n\nCreate fields.json using `pdf_width` and `pdf_height` (signals PDF coordinates):\n```json\n{\n  \"pages\": [\n    {\"page_number\": 1, \"pdf_width\": 612, \"pdf_height\": 792}\n  ],\n  \"form_fields\": [\n    {\n      \"page_number\": 1,\n      \"description\": \"Last name entry field\",\n      \"field_label\": \"Last Name\",\n      \"label_bounding_box\": [43, 63, 87, 73],\n      \"entry_bounding_box\": [92, 63, 260, 79],\n      \"entry_text\": {\"text\": \"Smith\", \"font_size\": 10}\n    },\n    {\n      \"page_number\": 1,\n      \"description\": \"US Citizen Yes checkbox\",\n      \"field_label\": \"Yes\",\n      \"label_bounding_box\": [260, 200, 280, 210],\n      \"entry_bounding_box\": [285, 197, 292, 205],\n      \"entry_text\": {\"text\": \"X\"}\n    }\n  ]\n}\n```\n\n**Important**: Use `pdf_width`/`pdf_height` and coordinates directly from form_structure.json.\n\n### A.4: Validate Bounding Boxes\n\nBefore filling, check your bounding boxes for errors:\n`python scripts/check_bounding_boxes.py fields.json`\n\nThis checks for intersecting bounding boxes and entry boxes that are too small for the font size. Fix any reported errors before filling.\n\n---\n\n## Approach B: Visual Estimation (Fallback)\n\nUse this when the PDF is scanned/image-based and structure extraction found no usable text labels (e.g., all text shows as \"(cid:X)\" patterns).\n\n### B.1: Convert PDF to Images\n\n`python scripts/convert_pdf_to_images.py <input.pdf> <images_dir/>`\n\n### B.2: Initial Field Identification\n\nExamine each page image to identify form sections and get **rough estimates** of field locations:\n- Form field labels and their approximate positions\n- Entry areas (lines, boxes, or blank spaces for text input)\n- Checkboxes and their approximate locations\n\nFor each field, note approximate pixel coordinates (they don't need to be precise yet).\n\n### B.3: Zoom Refinement (CRITICAL for accuracy)\n\nFor each field, crop a region around the estimated position to refine coordinates precisely.\n\n**Create a zoomed crop using ImageMagick:**\n```bash\nmagick <page_image> -crop <width>x<height>+<x>+<y> +repage <crop_output.png>\n```\n\nWhere:\n- `<x>, <y>` = top-left corner of crop region (use your rough estimate minus padding)\n- `<width>, <height>` = size of crop region (field area plus ~50px padding on each side)\n\n**Example:** To refine a \"Name\" field estimated around (100, 150):\n```bash\nmagick images_dir/page_1.png -crop 300x80+50+120 +repage crops/name_field.png\n```\n\n(Note: if the `magick` command isn't available, try `convert` with the same arguments).\n\n**Examine the cropped image** to determine precise coordinates:\n1. Identify the exact pixel where the entry area begins (after the label)\n2. Identify where the entry area ends (before next field or edge)\n3. Identify the top and bottom of the entry line/box\n\n**Convert crop coordinates back to full image coordinates:**\n- full_x = crop_x + crop_offset_x\n- full_y = crop_y + crop_offset_y\n\nExample: If the crop started at (50, 120) and the entry box starts at (52, 18) within the crop:\n- entry_x0 = 52 + 50 = 102\n- entry_top = 18 + 120 = 138\n\n**Repeat for each field**, grouping nearby fields into single crops when possible.\n\n### B.4: Create fields.json with Refined Coordinates\n\nCreate fields.json using `image_width` and `image_height` (signals image coordinates):\n```json\n{\n  \"pages\": [\n    {\"page_number\": 1, \"image_width\": 1700, \"image_height\": 2200}\n  ],\n  \"form_fields\": [\n    {\n      \"page_number\": 1,\n      \"description\": \"Last name entry field\",\n      \"field_label\": \"Last Name\",\n      \"label_bounding_box\": [120, 175, 242, 198],\n      \"entry_bounding_box\": [255, 175, 720, 218],\n      \"entry_text\": {\"text\": \"Smith\", \"font_size\": 10}\n    }\n  ]\n}\n```\n\n**Important**: Use `image_width`/`image_height` and the refined pixel coordinates from the zoom analysis.\n\n### B.5: Validate Bounding Boxes\n\nBefore filling, check your bounding boxes for errors:\n`python scripts/check_bounding_boxes.py fields.json`\n\nThis checks for intersecting bounding boxes and entry boxes that are too small for the font size. Fix any reported errors before filling.\n\n---\n\n## Hybrid Approach: Structure + Visual\n\nUse this when structure extraction works for most fields but misses some elements (e.g., circular checkboxes, unusual form controls).\n\n1. **Use Approach A** for fields that were detected in form_structure.json\n2. **Convert PDF to images** for visual analysis of missing fields\n3. **Use zoom refinement** (from Approach B) for the missing fields\n4. **Combine coordinates**: For fields from structure extraction, use `pdf_width`/`pdf_height`. For visually-estimated fields, you must convert image coordinates to PDF coordinates:\n   - pdf_x = image_x * (pdf_width / image_width)\n   - pdf_y = image_y * (pdf_height / image_height)\n5. **Use a single coordinate system** in fields.json - convert all to PDF coordinates with `pdf_width`/`pdf_height`\n\n---\n\n## Step 2: Validate Before Filling\n\n**Always validate bounding boxes before filling:**\n`python scripts/check_bounding_boxes.py fields.json`\n\nThis checks for:\n- Intersecting bounding boxes (which would cause overlapping text)\n- Entry boxes that are too small for the specified font size\n\nFix any reported errors in fields.json before proceeding.\n\n## Step 3: Fill the Form\n\nThe fill script auto-detects the coordinate system and handles conversion:\n`python scripts/fill_pdf_form_with_annotations.py <input.pdf> fields.json <output.pdf>`\n\n## Step 4: Verify Output\n\nConvert the filled PDF to images and verify text placement:\n`python scripts/convert_pdf_to_images.py <output.pdf> <verify_images/>`\n\nIf text is mispositioned:\n- **Approach A**: Check that you're using PDF coordinates from form_structure.json with `pdf_width`/`pdf_height`\n- **Approach B**: Check that image dimensions match and coordinates are accurate pixels\n- **Hybrid**: Ensure coordinate conversions are correct for visually-estimated fields\n"
  },
  {
    "name": "reference.md",
    "node_type": "file",
    "content": "# PDF Processing Advanced Reference\n\nThis document contains advanced PDF processing features, detailed examples, and additional libraries not covered in the main skill instructions.\n\n## pypdfium2 Library (Apache/BSD License)\n\n### Overview\npypdfium2 is a Python binding for PDFium (Chromium's PDF library). It's excellent for fast PDF rendering, image generation, and serves as a PyMuPDF replacement.\n\n### Render PDF to Images\n```python\nimport pypdfium2 as pdfium\nfrom PIL import Image\n\n# Load PDF\npdf = pdfium.PdfDocument(\"document.pdf\")\n\n# Render page to image\npage = pdf[0]  # First page\nbitmap = page.render(\n    scale=2.0,  # Higher resolution\n    rotation=0  # No rotation\n)\n\n# Convert to PIL Image\nimg = bitmap.to_pil()\nimg.save(\"page_1.png\", \"PNG\")\n\n# Process multiple pages\nfor i, page in enumerate(pdf):\n    bitmap = page.render(scale=1.5)\n    img = bitmap.to_pil()\n    img.save(f\"page_{i+1}.jpg\", \"JPEG\", quality=90)\n```\n\n### Extract Text with pypdfium2\n```python\nimport pypdfium2 as pdfium\n\npdf = pdfium.PdfDocument(\"document.pdf\")\nfor i, page in enumerate(pdf):\n    text = page.get_text()\n    print(f\"Page {i+1} text length: {len(text)} chars\")\n```\n\n## JavaScript Libraries\n\n### pdf-lib (MIT License)\n\npdf-lib is a powerful JavaScript library for creating and modifying PDF documents in any JavaScript environment.\n\n#### Load and Manipulate Existing PDF\n```javascript\nimport { PDFDocument } from 'pdf-lib';\nimport fs from 'fs';\n\nasync function manipulatePDF() {\n    // Load existing PDF\n    const existingPdfBytes = fs.readFileSync('input.pdf');\n    const pdfDoc = await PDFDocument.load(existingPdfBytes);\n\n    // Get page count\n    const pageCount = pdfDoc.getPageCount();\n    console.log(`Document has ${pageCount} pages`);\n\n    // Add new page\n    const newPage = pdfDoc.addPage([600, 400]);\n    newPage.drawText('Added by pdf-lib', {\n        x: 100,\n        y: 300,\n        size: 16\n    });\n\n    // Save modified PDF\n    const pdfBytes = await pdfDoc.save();\n    fs.writeFileSync('modified.pdf', pdfBytes);\n}\n```\n\n#### Create Complex PDFs from Scratch\n```javascript\nimport { PDFDocument, rgb, StandardFonts } from 'pdf-lib';\nimport fs from 'fs';\n\nasync function createPDF() {\n    const pdfDoc = await PDFDocument.create();\n\n    // Add fonts\n    const helveticaFont = await pdfDoc.embedFont(StandardFonts.Helvetica);\n    const helveticaBold = await pdfDoc.embedFont(StandardFonts.HelveticaBold);\n\n    // Add page\n    const page = pdfDoc.addPage([595, 842]); // A4 size\n    const { width, height } = page.getSize();\n\n    // Add text with styling\n    page.drawText('Invoice #12345', {\n        x: 50,\n        y: height - 50,\n        size: 18,\n        font: helveticaBold,\n        color: rgb(0.2, 0.2, 0.8)\n    });\n\n    // Add rectangle (header background)\n    page.drawRectangle({\n        x: 40,\n        y: height - 100,\n        width: width - 80,\n        height: 30,\n        color: rgb(0.9, 0.9, 0.9)\n    });\n\n    // Add table-like content\n    const items = [\n        ['Item', 'Qty', 'Price', 'Total'],\n        ['Widget', '2', '$50', '$100'],\n        ['Gadget', '1', '$75', '$75']\n    ];\n\n    let yPos = height - 150;\n    items.forEach(row => {\n        let xPos = 50;\n        row.forEach(cell => {\n            page.drawText(cell, {\n                x: xPos,\n                y: yPos,\n                size: 12,\n                font: helveticaFont\n            });\n            xPos += 120;\n        });\n        yPos -= 25;\n    });\n\n    const pdfBytes = await pdfDoc.save();\n    fs.writeFileSync('created.pdf', pdfBytes);\n}\n```\n\n#### Advanced Merge and Split Operations\n```javascript\nimport { PDFDocument } from 'pdf-lib';\nimport fs from 'fs';\n\nasync function mergePDFs() {\n    // Create new document\n    const mergedPdf = await PDFDocument.create();\n\n    // Load source PDFs\n    const pdf1Bytes = fs.readFileSync('doc1.pdf');\n    const pdf2Bytes = fs.readFileSync('doc2.pdf');\n\n    const pdf1 = await PDFDocument.load(pdf1Bytes);\n    const pdf2 = await PDFDocument.load(pdf2Bytes);\n\n    // Copy pages from first PDF\n    const pdf1Pages = await mergedPdf.copyPages(pdf1, pdf1.getPageIndices());\n    pdf1Pages.forEach(page => mergedPdf.addPage(page));\n\n    // Copy specific pages from second PDF (pages 0, 2, 4)\n    const pdf2Pages = await mergedPdf.copyPages(pdf2, [0, 2, 4]);\n    pdf2Pages.forEach(page => mergedPdf.addPage(page));\n\n    const mergedPdfBytes = await mergedPdf.save();\n    fs.writeFileSync('merged.pdf', mergedPdfBytes);\n}\n```\n\n### pdfjs-dist (Apache License)\n\nPDF.js is Mozilla's JavaScript library for rendering PDFs in the browser.\n\n#### Basic PDF Loading and Rendering\n```javascript\nimport * as pdfjsLib from 'pdfjs-dist';\n\n// Configure worker (important for performance)\npdfjsLib.GlobalWorkerOptions.workerSrc = './pdf.worker.js';\n\nasync function renderPDF() {\n    // Load PDF\n    const loadingTask = pdfjsLib.getDocument('document.pdf');\n    const pdf = await loadingTask.promise;\n\n    console.log(`Loaded PDF with ${pdf.numPages} pages`);\n\n    // Get first page\n    const page = await pdf.getPage(1);\n    const viewport = page.getViewport({ scale: 1.5 });\n\n    // Render to canvas\n    const canvas = document.createElement('canvas');\n    const context = canvas.getContext('2d');\n    canvas.height = viewport.height;\n    canvas.width = viewport.width;\n\n    const renderContext = {\n        canvasContext: context,\n        viewport: viewport\n    };\n\n    await page.render(renderContext).promise;\n    document.body.appendChild(canvas);\n}\n```\n\n#### Extract Text with Coordinates\n```javascript\nimport * as pdfjsLib from 'pdfjs-dist';\n\nasync function extractText() {\n    const loadingTask = pdfjsLib.getDocument('document.pdf');\n    const pdf = await loadingTask.promise;\n\n    let fullText = '';\n\n    // Extract text from all pages\n    for (let i = 1; i <= pdf.numPages; i++) {\n        const page = await pdf.getPage(i);\n        const textContent = await page.getTextContent();\n\n        const pageText = textContent.items\n            .map(item => item.str)\n            .join(' ');\n\n        fullText += `\\n--- Page ${i} ---\\n${pageText}`;\n\n        // Get text with coordinates for advanced processing\n        const textWithCoords = textContent.items.map(item => ({\n            text: item.str,\n            x: item.transform[4],\n            y: item.transform[5],\n            width: item.width,\n            height: item.height\n        }));\n    }\n\n    console.log(fullText);\n    return fullText;\n}\n```\n\n#### Extract Annotations and Forms\n```javascript\nimport * as pdfjsLib from 'pdfjs-dist';\n\nasync function extractAnnotations() {\n    const loadingTask = pdfjsLib.getDocument('annotated.pdf');\n    const pdf = await loadingTask.promise;\n\n    for (let i = 1; i <= pdf.numPages; i++) {\n        const page = await pdf.getPage(i);\n        const annotations = await page.getAnnotations();\n\n        annotations.forEach(annotation => {\n            console.log(`Annotation type: ${annotation.subtype}`);\n            console.log(`Content: ${annotation.contents}`);\n            console.log(`Coordinates: ${JSON.stringify(annotation.rect)}`);\n        });\n    }\n}\n```\n\n## Advanced Command-Line Operations\n\n### poppler-utils Advanced Features\n\n#### Extract Text with Bounding Box Coordinates\n```bash\n# Extract text with bounding box coordinates (essential for structured data)\npdftotext -bbox-layout document.pdf output.xml\n\n# The XML output contains precise coordinates for each text element\n```\n\n#### Advanced Image Conversion\n```bash\n# Convert to PNG images with specific resolution\npdftoppm -png -r 300 document.pdf output_prefix\n\n# Convert specific page range with high resolution\npdftoppm -png -r 600 -f 1 -l 3 document.pdf high_res_pages\n\n# Convert to JPEG with quality setting\npdftoppm -jpeg -jpegopt quality=85 -r 200 document.pdf jpeg_output\n```\n\n#### Extract Embedded Images\n```bash\n# Extract all embedded images with metadata\npdfimages -j -p document.pdf page_images\n\n# List image info without extracting\npdfimages -list document.pdf\n\n# Extract images in their original format\npdfimages -all document.pdf images/img\n```\n\n### qpdf Advanced Features\n\n#### Complex Page Manipulation\n```bash\n# Split PDF into groups of pages\nqpdf --split-pages=3 input.pdf output_group_%02d.pdf\n\n# Extract specific pages with complex ranges\nqpdf input.pdf --pages input.pdf 1,3-5,8,10-end -- extracted.pdf\n\n# Merge specific pages from multiple PDFs\nqpdf --empty --pages doc1.pdf 1-3 doc2.pdf 5-7 doc3.pdf 2,4 -- combined.pdf\n```\n\n#### PDF Optimization and Repair\n```bash\n# Optimize PDF for web (linearize for streaming)\nqpdf --linearize input.pdf optimized.pdf\n\n# Remove unused objects and compress\nqpdf --optimize-level=all input.pdf compressed.pdf\n\n# Attempt to repair corrupted PDF structure\nqpdf --check input.pdf\nqpdf --fix-qdf damaged.pdf repaired.pdf\n\n# Show detailed PDF structure for debugging\nqpdf --show-all-pages input.pdf > structure.txt\n```\n\n#### Advanced Encryption\n```bash\n# Add password protection with specific permissions\nqpdf --encrypt user_pass owner_pass 256 --print=none --modify=none -- input.pdf encrypted.pdf\n\n# Check encryption status\nqpdf --show-encryption encrypted.pdf\n\n# Remove password protection (requires password)\nqpdf --password=secret123 --decrypt encrypted.pdf decrypted.pdf\n```\n\n## Advanced Python Techniques\n\n### pdfplumber Advanced Features\n\n#### Extract Text with Precise Coordinates\n```python\nimport pdfplumber\n\nwith pdfplumber.open(\"document.pdf\") as pdf:\n    page = pdf.pages[0]\n    \n    # Extract all text with coordinates\n    chars = page.chars\n    for char in chars[:10]:  # First 10 characters\n        print(f\"Char: '{char['text']}' at x:{char['x0']:.1f} y:{char['y0']:.1f}\")\n    \n    # Extract text by bounding box (left, top, right, bottom)\n    bbox_text = page.within_bbox((100, 100, 400, 200)).extract_text()\n```\n\n#### Advanced Table Extraction with Custom Settings\n```python\nimport pdfplumber\nimport pandas as pd\n\nwith pdfplumber.open(\"complex_table.pdf\") as pdf:\n    page = pdf.pages[0]\n    \n    # Extract tables with custom settings for complex layouts\n    table_settings = {\n        \"vertical_strategy\": \"lines\",\n        \"horizontal_strategy\": \"lines\",\n        \"snap_tolerance\": 3,\n        \"intersection_tolerance\": 15\n    }\n    tables = page.extract_tables(table_settings)\n    \n    # Visual debugging for table extraction\n    img = page.to_image(resolution=150)\n    img.save(\"debug_layout.png\")\n```\n\n### reportlab Advanced Features\n\n#### Create Professional Reports with Tables\n```python\nfrom reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph\nfrom reportlab.lib.styles import getSampleStyleSheet\nfrom reportlab.lib import colors\n\n# Sample data\ndata = [\n    ['Product', 'Q1', 'Q2', 'Q3', 'Q4'],\n    ['Widgets', '120', '135', '142', '158'],\n    ['Gadgets', '85', '92', '98', '105']\n]\n\n# Create PDF with table\ndoc = SimpleDocTemplate(\"report.pdf\")\nelements = []\n\n# Add title\nstyles = getSampleStyleSheet()\ntitle = Paragraph(\"Quarterly Sales Report\", styles['Title'])\nelements.append(title)\n\n# Add table with advanced styling\ntable = Table(data)\ntable.setStyle(TableStyle([\n    ('BACKGROUND', (0, 0), (-1, 0), colors.grey),\n    ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),\n    ('ALIGN', (0, 0), (-1, -1), 'CENTER'),\n    ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),\n    ('FONTSIZE', (0, 0), (-1, 0), 14),\n    ('BOTTOMPADDING', (0, 0), (-1, 0), 12),\n    ('BACKGROUND', (0, 1), (-1, -1), colors.beige),\n    ('GRID', (0, 0), (-1, -1), 1, colors.black)\n]))\nelements.append(table)\n\ndoc.build(elements)\n```\n\n## Complex Workflows\n\n### Extract Figures/Images from PDF\n\n#### Method 1: Using pdfimages (fastest)\n```bash\n# Extract all images with original quality\npdfimages -all document.pdf images/img\n```\n\n#### Method 2: Using pypdfium2 + Image Processing\n```python\nimport pypdfium2 as pdfium\nfrom PIL import Image\nimport numpy as np\n\ndef extract_figures(pdf_path, output_dir):\n    pdf = pdfium.PdfDocument(pdf_path)\n    \n    for page_num, page in enumerate(pdf):\n        # Render high-resolution page\n        bitmap = page.render(scale=3.0)\n        img = bitmap.to_pil()\n        \n        # Convert to numpy for processing\n        img_array = np.array(img)\n        \n        # Simple figure detection (non-white regions)\n        mask = np.any(img_array != [255, 255, 255], axis=2)\n        \n        # Find contours and extract bounding boxes\n        # (This is simplified - real implementation would need more sophisticated detection)\n        \n        # Save detected figures\n        # ... implementation depends on specific needs\n```\n\n### Batch PDF Processing with Error Handling\n```python\nimport os\nimport glob\nfrom pypdf import PdfReader, PdfWriter\nimport logging\n\nlogging.basicConfig(level=logging.INFO)\nlogger = logging.getLogger(__name__)\n\ndef batch_process_pdfs(input_dir, operation='merge'):\n    pdf_files = glob.glob(os.path.join(input_dir, \"*.pdf\"))\n    \n    if operation == 'merge':\n        writer = PdfWriter()\n        for pdf_file in pdf_files:\n            try:\n                reader = PdfReader(pdf_file)\n                for page in reader.pages:\n                    writer.add_page(page)\n                logger.info(f\"Processed: {pdf_file}\")\n            except Exception as e:\n                logger.error(f\"Failed to process {pdf_file}: {e}\")\n                continue\n        \n        with open(\"batch_merged.pdf\", \"wb\") as output:\n            writer.write(output)\n    \n    elif operation == 'extract_text':\n        for pdf_file in pdf_files:\n            try:\n                reader = PdfReader(pdf_file)\n                text = \"\"\n                for page in reader.pages:\n                    text += page.extract_text()\n                \n                output_file = pdf_file.replace('.pdf', '.txt')\n                with open(output_file, 'w', encoding='utf-8') as f:\n                    f.write(text)\n                logger.info(f\"Extracted text from: {pdf_file}\")\n                \n            except Exception as e:\n                logger.error(f\"Failed to extract text from {pdf_file}: {e}\")\n                continue\n```\n\n### Advanced PDF Cropping\n```python\nfrom pypdf import PdfWriter, PdfReader\n\nreader = PdfReader(\"input.pdf\")\nwriter = PdfWriter()\n\n# Crop page (left, bottom, right, top in points)\npage = reader.pages[0]\npage.mediabox.left = 50\npage.mediabox.bottom = 50\npage.mediabox.right = 550\npage.mediabox.top = 750\n\nwriter.add_page(page)\nwith open(\"cropped.pdf\", \"wb\") as output:\n    writer.write(output)\n```\n\n## Performance Optimization Tips\n\n### 1. For Large PDFs\n- Use streaming approaches instead of loading entire PDF in memory\n- Use `qpdf --split-pages` for splitting large files\n- Process pages individually with pypdfium2\n\n### 2. For Text Extraction\n- `pdftotext -bbox-layout` is fastest for plain text extraction\n- Use pdfplumber for structured data and tables\n- Avoid `pypdf.extract_text()` for very large documents\n\n### 3. For Image Extraction\n- `pdfimages` is much faster than rendering pages\n- Use low resolution for previews, high resolution for final output\n\n### 4. For Form Filling\n- pdf-lib maintains form structure better than most alternatives\n- Pre-validate form fields before processing\n\n### 5. Memory Management\n```python\n# Process PDFs in chunks\ndef process_large_pdf(pdf_path, chunk_size=10):\n    reader = PdfReader(pdf_path)\n    total_pages = len(reader.pages)\n    \n    for start_idx in range(0, total_pages, chunk_size):\n        end_idx = min(start_idx + chunk_size, total_pages)\n        writer = PdfWriter()\n        \n        for i in range(start_idx, end_idx):\n            writer.add_page(reader.pages[i])\n        \n        # Process chunk\n        with open(f\"chunk_{start_idx//chunk_size}.pdf\", \"wb\") as output:\n            writer.write(output)\n```\n\n## Troubleshooting Common Issues\n\n### Encrypted PDFs\n```python\n# Handle password-protected PDFs\nfrom pypdf import PdfReader\n\ntry:\n    reader = PdfReader(\"encrypted.pdf\")\n    if reader.is_encrypted:\n        reader.decrypt(\"password\")\nexcept Exception as e:\n    print(f\"Failed to decrypt: {e}\")\n```\n\n### Corrupted PDFs\n```bash\n# Use qpdf to repair\nqpdf --check corrupted.pdf\nqpdf --replace-input corrupted.pdf\n```\n\n### Text Extraction Issues\n```python\n# Fallback to OCR for scanned PDFs\nimport pytesseract\nfrom pdf2image import convert_from_path\n\ndef extract_text_with_ocr(pdf_path):\n    images = convert_from_path(pdf_path)\n    text = \"\"\n    for i, image in enumerate(images):\n        text += pytesseract.image_to_string(image)\n    return text\n```\n\n## License Information\n\n- **pypdf**: BSD License\n- **pdfplumber**: MIT License\n- **pypdfium2**: Apache/BSD License\n- **reportlab**: BSD License\n- **poppler-utils**: GPL-2 License\n- **qpdf**: Apache License\n- **pdf-lib**: MIT License\n- **pdfjs-dist**: Apache License"
  },
  {
    "name": "scripts",
    "node_type": "folder",
    "children": [
      {
        "name": "check_bounding_boxes.py",
        "node_type": "file",
        "content": "from dataclasses import dataclass\nimport json\nimport sys\n\n\n\n\n@dataclass\nclass RectAndField:\n    rect: list[float]\n    rect_type: str\n    field: dict\n\n\ndef get_bounding_box_messages(fields_json_stream) -> list[str]:\n    messages = []\n    fields = json.load(fields_json_stream)\n    messages.append(f\"Read {len(fields['form_fields'])} fields\")\n\n    def rects_intersect(r1, r2):\n        disjoint_horizontal = r1[0] >= r2[2] or r1[2] <= r2[0]\n        disjoint_vertical = r1[1] >= r2[3] or r1[3] <= r2[1]\n        return not (disjoint_horizontal or disjoint_vertical)\n\n    rects_and_fields = []\n    for f in fields[\"form_fields\"]:\n        rects_and_fields.append(RectAndField(f[\"label_bounding_box\"], \"label\", f))\n        rects_and_fields.append(RectAndField(f[\"entry_bounding_box\"], \"entry\", f))\n\n    has_error = False\n    for i, ri in enumerate(rects_and_fields):\n        for j in range(i + 1, len(rects_and_fields)):\n            rj = rects_and_fields[j]\n            if ri.field[\"page_number\"] == rj.field[\"page_number\"] and rects_intersect(ri.rect, rj.rect):\n                has_error = True\n                if ri.field is rj.field:\n                    messages.append(f\"FAILURE: intersection between label and entry bounding boxes for `{ri.field['description']}` ({ri.rect}, {rj.rect})\")\n                else:\n                    messages.append(f\"FAILURE: intersection between {ri.rect_type} bounding box for `{ri.field['description']}` ({ri.rect}) and {rj.rect_type} bounding box for `{rj.field['description']}` ({rj.rect})\")\n                if len(messages) >= 20:\n                    messages.append(\"Aborting further checks; fix bounding boxes and try again\")\n                    return messages\n        if ri.rect_type == \"entry\":\n            if \"entry_text\" in ri.field:\n                font_size = ri.field[\"entry_text\"].get(\"font_size\", 14)\n                entry_height = ri.rect[3] - ri.rect[1]\n                if entry_height < font_size:\n                    has_error = True\n                    messages.append(f\"FAILURE: entry bounding box height ({entry_height}) for `{ri.field['description']}` is too short for the text content (font size: {font_size}). Increase the box height or decrease the font size.\")\n                    if len(messages) >= 20:\n                        messages.append(\"Aborting further checks; fix bounding boxes and try again\")\n                        return messages\n\n    if not has_error:\n        messages.append(\"SUCCESS: All bounding boxes are valid\")\n    return messages\n\nif __name__ == \"__main__\":\n    if len(sys.argv) != 2:\n        print(\"Usage: check_bounding_boxes.py [fields.json]\")\n        sys.exit(1)\n    with open(sys.argv[1]) as f:\n        messages = get_bounding_box_messages(f)\n    for msg in messages:\n        print(msg)\n"
      },
      {
        "name": "check_fillable_fields.py",
        "node_type": "file",
        "content": "import sys\nfrom pypdf import PdfReader\n\n\n\n\nreader = PdfReader(sys.argv[1])\nif (reader.get_fields()):\n    print(\"This PDF has fillable form fields\")\nelse:\n    print(\"This PDF does not have fillable form fields; you will need to visually determine where to enter data\")\n"
      },
      {
        "name": "convert_pdf_to_images.py",
        "node_type": "file",
        "content": "import os\nimport sys\n\nfrom pdf2image import convert_from_path\n\n\n\n\ndef convert(pdf_path, output_dir, max_dim=1000):\n    images = convert_from_path(pdf_path, dpi=200)\n\n    for i, image in enumerate(images):\n        width, height = image.size\n        if width > max_dim or height > max_dim:\n            scale_factor = min(max_dim / width, max_dim / height)\n            new_width = int(width * scale_factor)\n            new_height = int(height * scale_factor)\n            image = image.resize((new_width, new_height))\n        \n        image_path = os.path.join(output_dir, f\"page_{i+1}.png\")\n        image.save(image_path)\n        print(f\"Saved page {i+1} as {image_path} (size: {image.size})\")\n\n    print(f\"Converted {len(images)} pages to PNG images\")\n\n\nif __name__ == \"__main__\":\n    if len(sys.argv) != 3:\n        print(\"Usage: convert_pdf_to_images.py [input pdf] [output directory]\")\n        sys.exit(1)\n    pdf_path = sys.argv[1]\n    output_directory = sys.argv[2]\n    convert(pdf_path, output_directory)\n"
      },
      {
        "name": "create_validation_image.py",
        "node_type": "file",
        "content": "import json\nimport sys\n\nfrom PIL import Image, ImageDraw\n\n\n\n\ndef create_validation_image(page_number, fields_json_path, input_path, output_path):\n    with open(fields_json_path, 'r') as f:\n        data = json.load(f)\n\n        img = Image.open(input_path)\n        draw = ImageDraw.Draw(img)\n        num_boxes = 0\n        \n        for field in data[\"form_fields\"]:\n            if field[\"page_number\"] == page_number:\n                entry_box = field['entry_bounding_box']\n                label_box = field['label_bounding_box']\n                draw.rectangle(entry_box, outline='red', width=2)\n                draw.rectangle(label_box, outline='blue', width=2)\n                num_boxes += 2\n        \n        img.save(output_path)\n        print(f\"Created validation image at {output_path} with {num_boxes} bounding boxes\")\n\n\nif __name__ == \"__main__\":\n    if len(sys.argv) != 5:\n        print(\"Usage: create_validation_image.py [page number] [fields.json file] [input image path] [output image path]\")\n        sys.exit(1)\n    page_number = int(sys.argv[1])\n    fields_json_path = sys.argv[2]\n    input_image_path = sys.argv[3]\n    output_image_path = sys.argv[4]\n    create_validation_image(page_number, fields_json_path, input_image_path, output_image_path)\n"
      },
      {
        "name": "extract_form_field_info.py",
        "node_type": "file",
        "content": "import json\nimport sys\n\nfrom pypdf import PdfReader\n\n\n\n\ndef get_full_annotation_field_id(annotation):\n    components = []\n    while annotation:\n        field_name = annotation.get('/T')\n        if field_name:\n            components.append(field_name)\n        annotation = annotation.get('/Parent')\n    return \".\".join(reversed(components)) if components else None\n\n\ndef make_field_dict(field, field_id):\n    field_dict = {\"field_id\": field_id}\n    ft = field.get('/FT')\n    if ft == \"/Tx\":\n        field_dict[\"type\"] = \"text\"\n    elif ft == \"/Btn\":\n        field_dict[\"type\"] = \"checkbox\"  \n        states = field.get(\"/_States_\", [])\n        if len(states) == 2:\n            if \"/Off\" in states:\n                field_dict[\"checked_value\"] = states[0] if states[0] != \"/Off\" else states[1]\n                field_dict[\"unchecked_value\"] = \"/Off\"\n            else:\n                print(f\"Unexpected state values for checkbox `${field_id}`. Its checked and unchecked values may not be correct; if you're trying to check it, visually verify the results.\")\n                field_dict[\"checked_value\"] = states[0]\n                field_dict[\"unchecked_value\"] = states[1]\n    elif ft == \"/Ch\":\n        field_dict[\"type\"] = \"choice\"\n        states = field.get(\"/_States_\", [])\n        field_dict[\"choice_options\"] = [{\n            \"value\": state[0],\n            \"text\": state[1],\n        } for state in states]\n    else:\n        field_dict[\"type\"] = f\"unknown ({ft})\"\n    return field_dict\n\n\ndef get_field_info(reader: PdfReader):\n    fields = reader.get_fields()\n\n    field_info_by_id = {}\n    possible_radio_names = set()\n\n    for field_id, field in fields.items():\n        if field.get(\"/Kids\"):\n            if field.get(\"/FT\") == \"/Btn\":\n                possible_radio_names.add(field_id)\n            continue\n        field_info_by_id[field_id] = make_field_dict(field, field_id)\n\n\n    radio_fields_by_id = {}\n\n    for page_index, page in enumerate(reader.pages):\n        annotations = page.get('/Annots', [])\n        for ann in annotations:\n            field_id = get_full_annotation_field_id(ann)\n            if field_id in field_info_by_id:\n                field_info_by_id[field_id][\"page\"] = page_index + 1\n                field_info_by_id[field_id][\"rect\"] = ann.get('/Rect')\n            elif field_id in possible_radio_names:\n                try:\n                    on_values = [v for v in ann[\"/AP\"][\"/N\"] if v != \"/Off\"]\n                except KeyError:\n                    continue\n                if len(on_values) == 1:\n                    rect = ann.get(\"/Rect\")\n                    if field_id not in radio_fields_by_id:\n                        radio_fields_by_id[field_id] = {\n                            \"field_id\": field_id,\n                            \"type\": \"radio_group\",\n                            \"page\": page_index + 1,\n                            \"radio_options\": [],\n                        }\n                    radio_fields_by_id[field_id][\"radio_options\"].append({\n                        \"value\": on_values[0],\n                        \"rect\": rect,\n                    })\n\n    fields_with_location = []\n    for field_info in field_info_by_id.values():\n        if \"page\" in field_info:\n            fields_with_location.append(field_info)\n        else:\n            print(f\"Unable to determine location for field id: {field_info.get('field_id')}, ignoring\")\n\n    def sort_key(f):\n        if \"radio_options\" in f:\n            rect = f[\"radio_options\"][0][\"rect\"] or [0, 0, 0, 0]\n        else:\n            rect = f.get(\"rect\") or [0, 0, 0, 0]\n        adjusted_position = [-rect[1], rect[0]]\n        return [f.get(\"page\"), adjusted_position]\n    \n    sorted_fields = fields_with_location + list(radio_fields_by_id.values())\n    sorted_fields.sort(key=sort_key)\n\n    return sorted_fields\n\n\ndef write_field_info(pdf_path: str, json_output_path: str):\n    reader = PdfReader(pdf_path)\n    field_info = get_field_info(reader)\n    with open(json_output_path, \"w\") as f:\n        json.dump(field_info, f, indent=2)\n    print(f\"Wrote {len(field_info)} fields to {json_output_path}\")\n\n\nif __name__ == \"__main__\":\n    if len(sys.argv) != 3:\n        print(\"Usage: extract_form_field_info.py [input pdf] [output json]\")\n        sys.exit(1)\n    write_field_info(sys.argv[1], sys.argv[2])\n"
      },
      {
        "name": "extract_form_structure.py",
        "node_type": "file",
        "content": "\"\"\"\nExtract form structure from a non-fillable PDF.\n\nThis script analyzes the PDF to find:\n- Text labels with their exact coordinates\n- Horizontal lines (row boundaries)\n- Checkboxes (small rectangles)\n\nOutput: A JSON file with the form structure that can be used to generate\naccurate field coordinates for filling.\n\nUsage: python extract_form_structure.py <input.pdf> <output.json>\n\"\"\"\n\nimport json\nimport sys\nimport pdfplumber\n\n\ndef extract_form_structure(pdf_path):\n    structure = {\n        \"pages\": [],\n        \"labels\": [],\n        \"lines\": [],\n        \"checkboxes\": [],\n        \"row_boundaries\": []\n    }\n\n    with pdfplumber.open(pdf_path) as pdf:\n        for page_num, page in enumerate(pdf.pages, 1):\n            structure[\"pages\"].append({\n                \"page_number\": page_num,\n                \"width\": float(page.width),\n                \"height\": float(page.height)\n            })\n\n            words = page.extract_words()\n            for word in words:\n                structure[\"labels\"].append({\n                    \"page\": page_num,\n                    \"text\": word[\"text\"],\n                    \"x0\": round(float(word[\"x0\"]), 1),\n                    \"top\": round(float(word[\"top\"]), 1),\n                    \"x1\": round(float(word[\"x1\"]), 1),\n                    \"bottom\": round(float(word[\"bottom\"]), 1)\n                })\n\n            for line in page.lines:\n                if abs(float(line[\"x1\"]) - float(line[\"x0\"])) > page.width * 0.5:\n                    structure[\"lines\"].append({\n                        \"page\": page_num,\n                        \"y\": round(float(line[\"top\"]), 1),\n                        \"x0\": round(float(line[\"x0\"]), 1),\n                        \"x1\": round(float(line[\"x1\"]), 1)\n                    })\n\n            for rect in page.rects:\n                width = float(rect[\"x1\"]) - float(rect[\"x0\"])\n                height = float(rect[\"bottom\"]) - float(rect[\"top\"])\n                if 5 <= width <= 15 and 5 <= height <= 15 and abs(width - height) < 2:\n                    structure[\"checkboxes\"].append({\n                        \"page\": page_num,\n                        \"x0\": round(float(rect[\"x0\"]), 1),\n                        \"top\": round(float(rect[\"top\"]), 1),\n                        \"x1\": round(float(rect[\"x1\"]), 1),\n                        \"bottom\": round(float(rect[\"bottom\"]), 1),\n                        \"center_x\": round((float(rect[\"x0\"]) + float(rect[\"x1\"])) / 2, 1),\n                        \"center_y\": round((float(rect[\"top\"]) + float(rect[\"bottom\"])) / 2, 1)\n                    })\n\n    lines_by_page = {}\n    for line in structure[\"lines\"]:\n        page = line[\"page\"]\n        if page not in lines_by_page:\n            lines_by_page[page] = []\n        lines_by_page[page].append(line[\"y\"])\n\n    for page, y_coords in lines_by_page.items():\n        y_coords = sorted(set(y_coords))\n        for i in range(len(y_coords) - 1):\n            structure[\"row_boundaries\"].append({\n                \"page\": page,\n                \"row_top\": y_coords[i],\n                \"row_bottom\": y_coords[i + 1],\n                \"row_height\": round(y_coords[i + 1] - y_coords[i], 1)\n            })\n\n    return structure\n\n\ndef main():\n    if len(sys.argv) != 3:\n        print(\"Usage: extract_form_structure.py <input.pdf> <output.json>\")\n        sys.exit(1)\n\n    pdf_path = sys.argv[1]\n    output_path = sys.argv[2]\n\n    print(f\"Extracting structure from {pdf_path}...\")\n    structure = extract_form_structure(pdf_path)\n\n    with open(output_path, \"w\") as f:\n        json.dump(structure, f, indent=2)\n\n    print(f\"Found:\")\n    print(f\"  - {len(structure['pages'])} pages\")\n    print(f\"  - {len(structure['labels'])} text labels\")\n    print(f\"  - {len(structure['lines'])} horizontal lines\")\n    print(f\"  - {len(structure['checkboxes'])} checkboxes\")\n    print(f\"  - {len(structure['row_boundaries'])} row boundaries\")\n    print(f\"Saved to {output_path}\")\n\n\nif __name__ == \"__main__\":\n    main()\n"
      },
      {
        "name": "fill_fillable_fields.py",
        "node_type": "file",
        "content": "import json\nimport sys\n\nfrom pypdf import PdfReader, PdfWriter\n\nfrom extract_form_field_info import get_field_info\n\n\n\n\ndef fill_pdf_fields(input_pdf_path: str, fields_json_path: str, output_pdf_path: str):\n    with open(fields_json_path) as f:\n        fields = json.load(f)\n    fields_by_page = {}\n    for field in fields:\n        if \"value\" in field:\n            field_id = field[\"field_id\"]\n            page = field[\"page\"]\n            if page not in fields_by_page:\n                fields_by_page[page] = {}\n            fields_by_page[page][field_id] = field[\"value\"]\n    \n    reader = PdfReader(input_pdf_path)\n\n    has_error = False\n    field_info = get_field_info(reader)\n    fields_by_ids = {f[\"field_id\"]: f for f in field_info}\n    for field in fields:\n        existing_field = fields_by_ids.get(field[\"field_id\"])\n        if not existing_field:\n            has_error = True\n            print(f\"ERROR: `{field['field_id']}` is not a valid field ID\")\n        elif field[\"page\"] != existing_field[\"page\"]:\n            has_error = True\n            print(f\"ERROR: Incorrect page number for `{field['field_id']}` (got {field['page']}, expected {existing_field['page']})\")\n        else:\n            if \"value\" in field:\n                err = validation_error_for_field_value(existing_field, field[\"value\"])\n                if err:\n                    print(err)\n                    has_error = True\n    if has_error:\n        sys.exit(1)\n\n    writer = PdfWriter(clone_from=reader)\n    for page, field_values in fields_by_page.items():\n        writer.update_page_form_field_values(writer.pages[page - 1], field_values, auto_regenerate=False)\n\n    writer.set_need_appearances_writer(True)\n    \n    with open(output_pdf_path, \"wb\") as f:\n        writer.write(f)\n\n\ndef validation_error_for_field_value(field_info, field_value):\n    field_type = field_info[\"type\"]\n    field_id = field_info[\"field_id\"]\n    if field_type == \"checkbox\":\n        checked_val = field_info[\"checked_value\"]\n        unchecked_val = field_info[\"unchecked_value\"]\n        if field_value != checked_val and field_value != unchecked_val:\n            return f'ERROR: Invalid value \"{field_value}\" for checkbox field \"{field_id}\". The checked value is \"{checked_val}\" and the unchecked value is \"{unchecked_val}\"'\n    elif field_type == \"radio_group\":\n        option_values = [opt[\"value\"] for opt in field_info[\"radio_options\"]]\n        if field_value not in option_values:\n            return f'ERROR: Invalid value \"{field_value}\" for radio group field \"{field_id}\". Valid values are: {option_values}' \n    elif field_type == \"choice\":\n        choice_values = [opt[\"value\"] for opt in field_info[\"choice_options\"]]\n        if field_value not in choice_values:\n            return f'ERROR: Invalid value \"{field_value}\" for choice field \"{field_id}\". Valid values are: {choice_values}'\n    return None\n\n\ndef monkeypatch_pydpf_method():\n    from pypdf.generic import DictionaryObject\n    from pypdf.constants import FieldDictionaryAttributes\n\n    original_get_inherited = DictionaryObject.get_inherited\n\n    def patched_get_inherited(self, key: str, default = None):\n        result = original_get_inherited(self, key, default)\n        if key == FieldDictionaryAttributes.Opt:\n            if isinstance(result, list) and all(isinstance(v, list) and len(v) == 2 for v in result):\n                result = [r[0] for r in result]\n        return result\n\n    DictionaryObject.get_inherited = patched_get_inherited\n\n\nif __name__ == \"__main__\":\n    if len(sys.argv) != 4:\n        print(\"Usage: fill_fillable_fields.py [input pdf] [field_values.json] [output pdf]\")\n        sys.exit(1)\n    monkeypatch_pydpf_method()\n    input_pdf = sys.argv[1]\n    fields_json = sys.argv[2]\n    output_pdf = sys.argv[3]\n    fill_pdf_fields(input_pdf, fields_json, output_pdf)\n"
      },
      {
        "name": "fill_pdf_form_with_annotations.py",
        "node_type": "file",
        "content": "import json\nimport sys\n\nfrom pypdf import PdfReader, PdfWriter\nfrom pypdf.annotations import FreeText\n\n\n\n\ndef transform_from_image_coords(bbox, image_width, image_height, pdf_width, pdf_height):\n    x_scale = pdf_width / image_width\n    y_scale = pdf_height / image_height\n\n    left = bbox[0] * x_scale\n    right = bbox[2] * x_scale\n\n    top = pdf_height - (bbox[1] * y_scale)\n    bottom = pdf_height - (bbox[3] * y_scale)\n\n    return left, bottom, right, top\n\n\ndef transform_from_pdf_coords(bbox, pdf_height):\n    left = bbox[0]\n    right = bbox[2]\n\n    pypdf_top = pdf_height - bbox[1]      \n    pypdf_bottom = pdf_height - bbox[3]   \n\n    return left, pypdf_bottom, right, pypdf_top\n\n\ndef fill_pdf_form(input_pdf_path, fields_json_path, output_pdf_path):\n    \n    with open(fields_json_path, \"r\") as f:\n        fields_data = json.load(f)\n    \n    reader = PdfReader(input_pdf_path)\n    writer = PdfWriter()\n    \n    writer.append(reader)\n    \n    pdf_dimensions = {}\n    for i, page in enumerate(reader.pages):\n        mediabox = page.mediabox\n        pdf_dimensions[i + 1] = [mediabox.width, mediabox.height]\n    \n    annotations = []\n    for field in fields_data[\"form_fields\"]:\n        page_num = field[\"page_number\"]\n\n        page_info = next(p for p in fields_data[\"pages\"] if p[\"page_number\"] == page_num)\n        pdf_width, pdf_height = pdf_dimensions[page_num]\n\n        if \"pdf_width\" in page_info:\n            transformed_entry_box = transform_from_pdf_coords(\n                field[\"entry_bounding_box\"],\n                float(pdf_height)\n            )\n        else:\n            image_width = page_info[\"image_width\"]\n            image_height = page_info[\"image_height\"]\n            transformed_entry_box = transform_from_image_coords(\n                field[\"entry_bounding_box\"],\n                image_width, image_height,\n                float(pdf_width), float(pdf_height)\n            )\n        \n        if \"entry_text\" not in field or \"text\" not in field[\"entry_text\"]:\n            continue\n        entry_text = field[\"entry_text\"]\n        text = entry_text[\"text\"]\n        if not text:\n            continue\n        \n        font_name = entry_text.get(\"font\", \"Arial\")\n        font_size = str(entry_text.get(\"font_size\", 14)) + \"pt\"\n        font_color = entry_text.get(\"font_color\", \"000000\")\n\n        annotation = FreeText(\n            text=text,\n            rect=transformed_entry_box,\n            font=font_name,\n            font_size=font_size,\n            font_color=font_color,\n            border_color=None,\n            background_color=None,\n        )\n        annotations.append(annotation)\n        writer.add_annotation(page_number=page_num - 1, annotation=annotation)\n        \n    with open(output_pdf_path, \"wb\") as output:\n        writer.write(output)\n    \n    print(f\"Successfully filled PDF form and saved to {output_pdf_path}\")\n    print(f\"Added {len(annotations)} text annotations\")\n\n\nif __name__ == \"__main__\":\n    if len(sys.argv) != 4:\n        print(\"Usage: fill_pdf_form_with_annotations.py [input pdf] [fields.json] [output pdf]\")\n        sys.exit(1)\n    input_pdf = sys.argv[1]\n    fields_json = sys.argv[2]\n    output_pdf = sys.argv[3]\n    \n    fill_pdf_form(input_pdf, fields_json, output_pdf)\n"
      }
    ]
  }
]

export default children
