// AUTO-GENERATED — DO NOT EDIT
// Source: https://github.com/anthropics/skills
import type { SkillTemplateNode } from '../types'

const children: SkillTemplateNode[] = [
  {
    "name": "SKILL.md",
    "node_type": "file",
    "content": "---\nname: slack-gif-creator\ndescription: Knowledge and utilities for creating animated GIFs optimized for Slack. Provides constraints, validation tools, and animation concepts. Use when users request animated GIFs for Slack like \"make me a GIF of X doing Y for Slack.\"\nlicense: Complete terms in LICENSE.txt\n---\n\n# Slack GIF Creator\n\nA toolkit providing utilities and knowledge for creating animated GIFs optimized for Slack.\n\n## Slack Requirements\n\n**Dimensions:**\n- Emoji GIFs: 128x128 (recommended)\n- Message GIFs: 480x480\n\n**Parameters:**\n- FPS: 10-30 (lower is smaller file size)\n- Colors: 48-128 (fewer = smaller file size)\n- Duration: Keep under 3 seconds for emoji GIFs\n\n## Core Workflow\n\n```python\nfrom core.gif_builder import GIFBuilder\nfrom PIL import Image, ImageDraw\n\n# 1. Create builder\nbuilder = GIFBuilder(width=128, height=128, fps=10)\n\n# 2. Generate frames\nfor i in range(12):\n    frame = Image.new('RGB', (128, 128), (240, 248, 255))\n    draw = ImageDraw.Draw(frame)\n\n    # Draw your animation using PIL primitives\n    # (circles, polygons, lines, etc.)\n\n    builder.add_frame(frame)\n\n# 3. Save with optimization\nbuilder.save('output.gif', num_colors=48, optimize_for_emoji=True)\n```\n\n## Drawing Graphics\n\n### Working with User-Uploaded Images\nIf a user uploads an image, consider whether they want to:\n- **Use it directly** (e.g., \"animate this\", \"split this into frames\")\n- **Use it as inspiration** (e.g., \"make something like this\")\n\nLoad and work with images using PIL:\n```python\nfrom PIL import Image\n\nuploaded = Image.open('file.png')\n# Use directly, or just as reference for colors/style\n```\n\n### Drawing from Scratch\nWhen drawing graphics from scratch, use PIL ImageDraw primitives:\n\n```python\nfrom PIL import ImageDraw\n\ndraw = ImageDraw.Draw(frame)\n\n# Circles/ovals\ndraw.ellipse([x1, y1, x2, y2], fill=(r, g, b), outline=(r, g, b), width=3)\n\n# Stars, triangles, any polygon\npoints = [(x1, y1), (x2, y2), (x3, y3), ...]\ndraw.polygon(points, fill=(r, g, b), outline=(r, g, b), width=3)\n\n# Lines\ndraw.line([(x1, y1), (x2, y2)], fill=(r, g, b), width=5)\n\n# Rectangles\ndraw.rectangle([x1, y1, x2, y2], fill=(r, g, b), outline=(r, g, b), width=3)\n```\n\n**Don't use:** Emoji fonts (unreliable across platforms) or assume pre-packaged graphics exist in this skill.\n\n### Making Graphics Look Good\n\nGraphics should look polished and creative, not basic. Here's how:\n\n**Use thicker lines** - Always set `width=2` or higher for outlines and lines. Thin lines (width=1) look choppy and amateurish.\n\n**Add visual depth**:\n- Use gradients for backgrounds (`create_gradient_background`)\n- Layer multiple shapes for complexity (e.g., a star with a smaller star inside)\n\n**Make shapes more interesting**:\n- Don't just draw a plain circle - add highlights, rings, or patterns\n- Stars can have glows (draw larger, semi-transparent versions behind)\n- Combine multiple shapes (stars + sparkles, circles + rings)\n\n**Pay attention to colors**:\n- Use vibrant, complementary colors\n- Add contrast (dark outlines on light shapes, light outlines on dark shapes)\n- Consider the overall composition\n\n**For complex shapes** (hearts, snowflakes, etc.):\n- Use combinations of polygons and ellipses\n- Calculate points carefully for symmetry\n- Add details (a heart can have a highlight curve, snowflakes have intricate branches)\n\nBe creative and detailed! A good Slack GIF should look polished, not like placeholder graphics.\n\n## Available Utilities\n\n### GIFBuilder (`core.gif_builder`)\nAssembles frames and optimizes for Slack:\n```python\nbuilder = GIFBuilder(width=128, height=128, fps=10)\nbuilder.add_frame(frame)  # Add PIL Image\nbuilder.add_frames(frames)  # Add list of frames\nbuilder.save('out.gif', num_colors=48, optimize_for_emoji=True, remove_duplicates=True)\n```\n\n### Validators (`core.validators`)\nCheck if GIF meets Slack requirements:\n```python\nfrom core.validators import validate_gif, is_slack_ready\n\n# Detailed validation\npasses, info = validate_gif('my.gif', is_emoji=True, verbose=True)\n\n# Quick check\nif is_slack_ready('my.gif'):\n    print(\"Ready!\")\n```\n\n### Easing Functions (`core.easing`)\nSmooth motion instead of linear:\n```python\nfrom core.easing import interpolate\n\n# Progress from 0.0 to 1.0\nt = i / (num_frames - 1)\n\n# Apply easing\ny = interpolate(start=0, end=400, t=t, easing='ease_out')\n\n# Available: linear, ease_in, ease_out, ease_in_out,\n#           bounce_out, elastic_out, back_out\n```\n\n### Frame Helpers (`core.frame_composer`)\nConvenience functions for common needs:\n```python\nfrom core.frame_composer import (\n    create_blank_frame,         # Solid color background\n    create_gradient_background,  # Vertical gradient\n    draw_circle,                # Helper for circles\n    draw_text,                  # Simple text rendering\n    draw_star                   # 5-pointed star\n)\n```\n\n## Animation Concepts\n\n### Shake/Vibrate\nOffset object position with oscillation:\n- Use `math.sin()` or `math.cos()` with frame index\n- Add small random variations for natural feel\n- Apply to x and/or y position\n\n### Pulse/Heartbeat\nScale object size rhythmically:\n- Use `math.sin(t * frequency * 2 * math.pi)` for smooth pulse\n- For heartbeat: two quick pulses then pause (adjust sine wave)\n- Scale between 0.8 and 1.2 of base size\n\n### Bounce\nObject falls and bounces:\n- Use `interpolate()` with `easing='bounce_out'` for landing\n- Use `easing='ease_in'` for falling (accelerating)\n- Apply gravity by increasing y velocity each frame\n\n### Spin/Rotate\nRotate object around center:\n- PIL: `image.rotate(angle, resample=Image.BICUBIC)`\n- For wobble: use sine wave for angle instead of linear\n\n### Fade In/Out\nGradually appear or disappear:\n- Create RGBA image, adjust alpha channel\n- Or use `Image.blend(image1, image2, alpha)`\n- Fade in: alpha from 0 to 1\n- Fade out: alpha from 1 to 0\n\n### Slide\nMove object from off-screen to position:\n- Start position: outside frame bounds\n- End position: target location\n- Use `interpolate()` with `easing='ease_out'` for smooth stop\n- For overshoot: use `easing='back_out'`\n\n### Zoom\nScale and position for zoom effect:\n- Zoom in: scale from 0.1 to 2.0, crop center\n- Zoom out: scale from 2.0 to 1.0\n- Can add motion blur for drama (PIL filter)\n\n### Explode/Particle Burst\nCreate particles radiating outward:\n- Generate particles with random angles and velocities\n- Update each particle: `x += vx`, `y += vy`\n- Add gravity: `vy += gravity_constant`\n- Fade out particles over time (reduce alpha)\n\n## Optimization Strategies\n\nOnly when asked to make the file size smaller, implement a few of the following methods:\n\n1. **Fewer frames** - Lower FPS (10 instead of 20) or shorter duration\n2. **Fewer colors** - `num_colors=48` instead of 128\n3. **Smaller dimensions** - 128x128 instead of 480x480\n4. **Remove duplicates** - `remove_duplicates=True` in save()\n5. **Emoji mode** - `optimize_for_emoji=True` auto-optimizes\n\n```python\n# Maximum optimization for emoji\nbuilder.save(\n    'emoji.gif',\n    num_colors=48,\n    optimize_for_emoji=True,\n    remove_duplicates=True\n)\n```\n\n## Philosophy\n\nThis skill provides:\n- **Knowledge**: Slack's requirements and animation concepts\n- **Utilities**: GIFBuilder, validators, easing functions\n- **Flexibility**: Create the animation logic using PIL primitives\n\nIt does NOT provide:\n- Rigid animation templates or pre-made functions\n- Emoji font rendering (unreliable across platforms)\n- A library of pre-packaged graphics built into the skill\n\n**Note on user uploads**: This skill doesn't include pre-built graphics, but if a user uploads an image, use PIL to load and work with it - interpret based on their request whether they want it used directly or just as inspiration.\n\nBe creative! Combine concepts (bouncing + rotating, pulsing + sliding, etc.) and use PIL's full capabilities.\n\n## Dependencies\n\n```bash\npip install pillow imageio numpy\n```\n"
  },
  {
    "name": "core",
    "node_type": "folder",
    "children": [
      {
        "name": "easing.py",
        "node_type": "file",
        "content": "#!/usr/bin/env python3\n\"\"\"\nEasing Functions - Timing functions for smooth animations.\n\nProvides various easing functions for natural motion and timing.\nAll functions take a value t (0.0 to 1.0) and return eased value (0.0 to 1.0).\n\"\"\"\n\nimport math\n\n\ndef linear(t: float) -> float:\n    \"\"\"Linear interpolation (no easing).\"\"\"\n    return t\n\n\ndef ease_in_quad(t: float) -> float:\n    \"\"\"Quadratic ease-in (slow start, accelerating).\"\"\"\n    return t * t\n\n\ndef ease_out_quad(t: float) -> float:\n    \"\"\"Quadratic ease-out (fast start, decelerating).\"\"\"\n    return t * (2 - t)\n\n\ndef ease_in_out_quad(t: float) -> float:\n    \"\"\"Quadratic ease-in-out (slow start and end).\"\"\"\n    if t < 0.5:\n        return 2 * t * t\n    return -1 + (4 - 2 * t) * t\n\n\ndef ease_in_cubic(t: float) -> float:\n    \"\"\"Cubic ease-in (slow start).\"\"\"\n    return t * t * t\n\n\ndef ease_out_cubic(t: float) -> float:\n    \"\"\"Cubic ease-out (fast start).\"\"\"\n    return (t - 1) * (t - 1) * (t - 1) + 1\n\n\ndef ease_in_out_cubic(t: float) -> float:\n    \"\"\"Cubic ease-in-out.\"\"\"\n    if t < 0.5:\n        return 4 * t * t * t\n    return (t - 1) * (2 * t - 2) * (2 * t - 2) + 1\n\n\ndef ease_in_bounce(t: float) -> float:\n    \"\"\"Bounce ease-in (bouncy start).\"\"\"\n    return 1 - ease_out_bounce(1 - t)\n\n\ndef ease_out_bounce(t: float) -> float:\n    \"\"\"Bounce ease-out (bouncy end).\"\"\"\n    if t < 1 / 2.75:\n        return 7.5625 * t * t\n    elif t < 2 / 2.75:\n        t -= 1.5 / 2.75\n        return 7.5625 * t * t + 0.75\n    elif t < 2.5 / 2.75:\n        t -= 2.25 / 2.75\n        return 7.5625 * t * t + 0.9375\n    else:\n        t -= 2.625 / 2.75\n        return 7.5625 * t * t + 0.984375\n\n\ndef ease_in_out_bounce(t: float) -> float:\n    \"\"\"Bounce ease-in-out.\"\"\"\n    if t < 0.5:\n        return ease_in_bounce(t * 2) * 0.5\n    return ease_out_bounce(t * 2 - 1) * 0.5 + 0.5\n\n\ndef ease_in_elastic(t: float) -> float:\n    \"\"\"Elastic ease-in (spring effect).\"\"\"\n    if t == 0 or t == 1:\n        return t\n    return -math.pow(2, 10 * (t - 1)) * math.sin((t - 1.1) * 5 * math.pi)\n\n\ndef ease_out_elastic(t: float) -> float:\n    \"\"\"Elastic ease-out (spring effect).\"\"\"\n    if t == 0 or t == 1:\n        return t\n    return math.pow(2, -10 * t) * math.sin((t - 0.1) * 5 * math.pi) + 1\n\n\ndef ease_in_out_elastic(t: float) -> float:\n    \"\"\"Elastic ease-in-out.\"\"\"\n    if t == 0 or t == 1:\n        return t\n    t = t * 2 - 1\n    if t < 0:\n        return -0.5 * math.pow(2, 10 * t) * math.sin((t - 0.1) * 5 * math.pi)\n    return math.pow(2, -10 * t) * math.sin((t - 0.1) * 5 * math.pi) * 0.5 + 1\n\n\n# Convenience mapping\nEASING_FUNCTIONS = {\n    \"linear\": linear,\n    \"ease_in\": ease_in_quad,\n    \"ease_out\": ease_out_quad,\n    \"ease_in_out\": ease_in_out_quad,\n    \"bounce_in\": ease_in_bounce,\n    \"bounce_out\": ease_out_bounce,\n    \"bounce\": ease_in_out_bounce,\n    \"elastic_in\": ease_in_elastic,\n    \"elastic_out\": ease_out_elastic,\n    \"elastic\": ease_in_out_elastic,\n}\n\n\ndef get_easing(name: str = \"linear\"):\n    \"\"\"Get easing function by name.\"\"\"\n    return EASING_FUNCTIONS.get(name, linear)\n\n\ndef interpolate(start: float, end: float, t: float, easing: str = \"linear\") -> float:\n    \"\"\"\n    Interpolate between two values with easing.\n\n    Args:\n        start: Start value\n        end: End value\n        t: Progress from 0.0 to 1.0\n        easing: Name of easing function\n\n    Returns:\n        Interpolated value\n    \"\"\"\n    ease_func = get_easing(easing)\n    eased_t = ease_func(t)\n    return start + (end - start) * eased_t\n\n\ndef ease_back_in(t: float) -> float:\n    \"\"\"Back ease-in (slight overshoot backward before forward motion).\"\"\"\n    c1 = 1.70158\n    c3 = c1 + 1\n    return c3 * t * t * t - c1 * t * t\n\n\ndef ease_back_out(t: float) -> float:\n    \"\"\"Back ease-out (overshoot forward then settle back).\"\"\"\n    c1 = 1.70158\n    c3 = c1 + 1\n    return 1 + c3 * pow(t - 1, 3) + c1 * pow(t - 1, 2)\n\n\ndef ease_back_in_out(t: float) -> float:\n    \"\"\"Back ease-in-out (overshoot at both ends).\"\"\"\n    c1 = 1.70158\n    c2 = c1 * 1.525\n    if t < 0.5:\n        return (pow(2 * t, 2) * ((c2 + 1) * 2 * t - c2)) / 2\n    return (pow(2 * t - 2, 2) * ((c2 + 1) * (t * 2 - 2) + c2) + 2) / 2\n\n\ndef apply_squash_stretch(\n    base_scale: tuple[float, float], intensity: float, direction: str = \"vertical\"\n) -> tuple[float, float]:\n    \"\"\"\n    Calculate squash and stretch scales for more dynamic animation.\n\n    Args:\n        base_scale: (width_scale, height_scale) base scales\n        intensity: Squash/stretch intensity (0.0-1.0)\n        direction: 'vertical', 'horizontal', or 'both'\n\n    Returns:\n        (width_scale, height_scale) with squash/stretch applied\n    \"\"\"\n    width_scale, height_scale = base_scale\n\n    if direction == \"vertical\":\n        # Compress vertically, expand horizontally (preserve volume)\n        height_scale *= 1 - intensity * 0.5\n        width_scale *= 1 + intensity * 0.5\n    elif direction == \"horizontal\":\n        # Compress horizontally, expand vertically\n        width_scale *= 1 - intensity * 0.5\n        height_scale *= 1 + intensity * 0.5\n    elif direction == \"both\":\n        # General squash (both dimensions)\n        width_scale *= 1 - intensity * 0.3\n        height_scale *= 1 - intensity * 0.3\n\n    return (width_scale, height_scale)\n\n\ndef calculate_arc_motion(\n    start: tuple[float, float], end: tuple[float, float], height: float, t: float\n) -> tuple[float, float]:\n    \"\"\"\n    Calculate position along a parabolic arc (natural motion path).\n\n    Args:\n        start: (x, y) starting position\n        end: (x, y) ending position\n        height: Arc height at midpoint (positive = upward)\n        t: Progress (0.0-1.0)\n\n    Returns:\n        (x, y) position along arc\n    \"\"\"\n    x1, y1 = start\n    x2, y2 = end\n\n    # Linear interpolation for x\n    x = x1 + (x2 - x1) * t\n\n    # Parabolic interpolation for y\n    # y = start + progress * (end - start) + arc_offset\n    # Arc offset peaks at t=0.5\n    arc_offset = 4 * height * t * (1 - t)\n    y = y1 + (y2 - y1) * t - arc_offset\n\n    return (x, y)\n\n\n# Add new easing functions to the convenience mapping\nEASING_FUNCTIONS.update(\n    {\n        \"back_in\": ease_back_in,\n        \"back_out\": ease_back_out,\n        \"back_in_out\": ease_back_in_out,\n        \"anticipate\": ease_back_in,  # Alias\n        \"overshoot\": ease_back_out,  # Alias\n    }\n)\n"
      },
      {
        "name": "frame_composer.py",
        "node_type": "file",
        "content": "#!/usr/bin/env python3\n\"\"\"\nFrame Composer - Utilities for composing visual elements into frames.\n\nProvides functions for drawing shapes, text, emojis, and compositing elements\ntogether to create animation frames.\n\"\"\"\n\nfrom typing import Optional\n\nimport numpy as np\nfrom PIL import Image, ImageDraw, ImageFont\n\n\ndef create_blank_frame(\n    width: int, height: int, color: tuple[int, int, int] = (255, 255, 255)\n) -> Image.Image:\n    \"\"\"\n    Create a blank frame with solid color background.\n\n    Args:\n        width: Frame width\n        height: Frame height\n        color: RGB color tuple (default: white)\n\n    Returns:\n        PIL Image\n    \"\"\"\n    return Image.new(\"RGB\", (width, height), color)\n\n\ndef draw_circle(\n    frame: Image.Image,\n    center: tuple[int, int],\n    radius: int,\n    fill_color: Optional[tuple[int, int, int]] = None,\n    outline_color: Optional[tuple[int, int, int]] = None,\n    outline_width: int = 1,\n) -> Image.Image:\n    \"\"\"\n    Draw a circle on a frame.\n\n    Args:\n        frame: PIL Image to draw on\n        center: (x, y) center position\n        radius: Circle radius\n        fill_color: RGB fill color (None for no fill)\n        outline_color: RGB outline color (None for no outline)\n        outline_width: Outline width in pixels\n\n    Returns:\n        Modified frame\n    \"\"\"\n    draw = ImageDraw.Draw(frame)\n    x, y = center\n    bbox = [x - radius, y - radius, x + radius, y + radius]\n    draw.ellipse(bbox, fill=fill_color, outline=outline_color, width=outline_width)\n    return frame\n\n\ndef draw_text(\n    frame: Image.Image,\n    text: str,\n    position: tuple[int, int],\n    color: tuple[int, int, int] = (0, 0, 0),\n    centered: bool = False,\n) -> Image.Image:\n    \"\"\"\n    Draw text on a frame.\n\n    Args:\n        frame: PIL Image to draw on\n        text: Text to draw\n        position: (x, y) position (top-left unless centered=True)\n        color: RGB text color\n        centered: If True, center text at position\n\n    Returns:\n        Modified frame\n    \"\"\"\n    draw = ImageDraw.Draw(frame)\n\n    # Uses Pillow's default font.\n    # If the font should be changed for the emoji, add additional logic here.\n    font = ImageFont.load_default()\n\n    if centered:\n        bbox = draw.textbbox((0, 0), text, font=font)\n        text_width = bbox[2] - bbox[0]\n        text_height = bbox[3] - bbox[1]\n        x = position[0] - text_width // 2\n        y = position[1] - text_height // 2\n        position = (x, y)\n\n    draw.text(position, text, fill=color, font=font)\n    return frame\n\n\ndef create_gradient_background(\n    width: int,\n    height: int,\n    top_color: tuple[int, int, int],\n    bottom_color: tuple[int, int, int],\n) -> Image.Image:\n    \"\"\"\n    Create a vertical gradient background.\n\n    Args:\n        width: Frame width\n        height: Frame height\n        top_color: RGB color at top\n        bottom_color: RGB color at bottom\n\n    Returns:\n        PIL Image with gradient\n    \"\"\"\n    frame = Image.new(\"RGB\", (width, height))\n    draw = ImageDraw.Draw(frame)\n\n    # Calculate color step for each row\n    r1, g1, b1 = top_color\n    r2, g2, b2 = bottom_color\n\n    for y in range(height):\n        # Interpolate color\n        ratio = y / height\n        r = int(r1 * (1 - ratio) + r2 * ratio)\n        g = int(g1 * (1 - ratio) + g2 * ratio)\n        b = int(b1 * (1 - ratio) + b2 * ratio)\n\n        # Draw horizontal line\n        draw.line([(0, y), (width, y)], fill=(r, g, b))\n\n    return frame\n\n\ndef draw_star(\n    frame: Image.Image,\n    center: tuple[int, int],\n    size: int,\n    fill_color: tuple[int, int, int],\n    outline_color: Optional[tuple[int, int, int]] = None,\n    outline_width: int = 1,\n) -> Image.Image:\n    \"\"\"\n    Draw a 5-pointed star.\n\n    Args:\n        frame: PIL Image to draw on\n        center: (x, y) center position\n        size: Star size (outer radius)\n        fill_color: RGB fill color\n        outline_color: RGB outline color (None for no outline)\n        outline_width: Outline width\n\n    Returns:\n        Modified frame\n    \"\"\"\n    import math\n\n    draw = ImageDraw.Draw(frame)\n    x, y = center\n\n    # Calculate star points\n    points = []\n    for i in range(10):\n        angle = (i * 36 - 90) * math.pi / 180  # 36 degrees per point, start at top\n        radius = size if i % 2 == 0 else size * 0.4  # Alternate between outer and inner\n        px = x + radius * math.cos(angle)\n        py = y + radius * math.sin(angle)\n        points.append((px, py))\n\n    # Draw star\n    draw.polygon(points, fill=fill_color, outline=outline_color, width=outline_width)\n\n    return frame\n"
      },
      {
        "name": "gif_builder.py",
        "node_type": "file",
        "content": "#!/usr/bin/env python3\n\"\"\"\nGIF Builder - Core module for assembling frames into GIFs optimized for Slack.\n\nThis module provides the main interface for creating GIFs from programmatically\ngenerated frames, with automatic optimization for Slack's requirements.\n\"\"\"\n\nfrom pathlib import Path\nfrom typing import Optional\n\nimport imageio.v3 as imageio\nimport numpy as np\nfrom PIL import Image\n\n\nclass GIFBuilder:\n    \"\"\"Builder for creating optimized GIFs from frames.\"\"\"\n\n    def __init__(self, width: int = 480, height: int = 480, fps: int = 15):\n        \"\"\"\n        Initialize GIF builder.\n\n        Args:\n            width: Frame width in pixels\n            height: Frame height in pixels\n            fps: Frames per second\n        \"\"\"\n        self.width = width\n        self.height = height\n        self.fps = fps\n        self.frames: list[np.ndarray] = []\n\n    def add_frame(self, frame: np.ndarray | Image.Image):\n        \"\"\"\n        Add a frame to the GIF.\n\n        Args:\n            frame: Frame as numpy array or PIL Image (will be converted to RGB)\n        \"\"\"\n        if isinstance(frame, Image.Image):\n            frame = np.array(frame.convert(\"RGB\"))\n\n        # Ensure frame is correct size\n        if frame.shape[:2] != (self.height, self.width):\n            pil_frame = Image.fromarray(frame)\n            pil_frame = pil_frame.resize(\n                (self.width, self.height), Image.Resampling.LANCZOS\n            )\n            frame = np.array(pil_frame)\n\n        self.frames.append(frame)\n\n    def add_frames(self, frames: list[np.ndarray | Image.Image]):\n        \"\"\"Add multiple frames at once.\"\"\"\n        for frame in frames:\n            self.add_frame(frame)\n\n    def optimize_colors(\n        self, num_colors: int = 128, use_global_palette: bool = True\n    ) -> list[np.ndarray]:\n        \"\"\"\n        Reduce colors in all frames using quantization.\n\n        Args:\n            num_colors: Target number of colors (8-256)\n            use_global_palette: Use a single palette for all frames (better compression)\n\n        Returns:\n            List of color-optimized frames\n        \"\"\"\n        optimized = []\n\n        if use_global_palette and len(self.frames) > 1:\n            # Create a global palette from all frames\n            # Sample frames to build palette\n            sample_size = min(5, len(self.frames))\n            sample_indices = [\n                int(i * len(self.frames) / sample_size) for i in range(sample_size)\n            ]\n            sample_frames = [self.frames[i] for i in sample_indices]\n\n            # Combine sample frames into a single image for palette generation\n            # Flatten each frame to get all pixels, then stack them\n            all_pixels = np.vstack(\n                [f.reshape(-1, 3) for f in sample_frames]\n            )  # (total_pixels, 3)\n\n            # Create a properly-shaped RGB image from the pixel data\n            # We'll make a roughly square image from all the pixels\n            total_pixels = len(all_pixels)\n            width = min(512, int(np.sqrt(total_pixels)))  # Reasonable width, max 512\n            height = (total_pixels + width - 1) // width  # Ceiling division\n\n            # Pad if necessary to fill the rectangle\n            pixels_needed = width * height\n            if pixels_needed > total_pixels:\n                padding = np.zeros((pixels_needed - total_pixels, 3), dtype=np.uint8)\n                all_pixels = np.vstack([all_pixels, padding])\n\n            # Reshape to proper RGB image format (H, W, 3)\n            img_array = (\n                all_pixels[:pixels_needed].reshape(height, width, 3).astype(np.uint8)\n            )\n            combined_img = Image.fromarray(img_array, mode=\"RGB\")\n\n            # Generate global palette\n            global_palette = combined_img.quantize(colors=num_colors, method=2)\n\n            # Apply global palette to all frames\n            for frame in self.frames:\n                pil_frame = Image.fromarray(frame)\n                quantized = pil_frame.quantize(palette=global_palette, dither=1)\n                optimized.append(np.array(quantized.convert(\"RGB\")))\n        else:\n            # Use per-frame quantization\n            for frame in self.frames:\n                pil_frame = Image.fromarray(frame)\n                quantized = pil_frame.quantize(colors=num_colors, method=2, dither=1)\n                optimized.append(np.array(quantized.convert(\"RGB\")))\n\n        return optimized\n\n    def deduplicate_frames(self, threshold: float = 0.9995) -> int:\n        \"\"\"\n        Remove duplicate or near-duplicate consecutive frames.\n\n        Args:\n            threshold: Similarity threshold (0.0-1.0). Higher = more strict (0.9995 = nearly identical).\n                      Use 0.9995+ to preserve subtle animations, 0.98 for aggressive removal.\n\n        Returns:\n            Number of frames removed\n        \"\"\"\n        if len(self.frames) < 2:\n            return 0\n\n        deduplicated = [self.frames[0]]\n        removed_count = 0\n\n        for i in range(1, len(self.frames)):\n            # Compare with previous frame\n            prev_frame = np.array(deduplicated[-1], dtype=np.float32)\n            curr_frame = np.array(self.frames[i], dtype=np.float32)\n\n            # Calculate similarity (normalized)\n            diff = np.abs(prev_frame - curr_frame)\n            similarity = 1.0 - (np.mean(diff) / 255.0)\n\n            # Keep frame if sufficiently different\n            # High threshold (0.9995+) means only remove nearly identical frames\n            if similarity < threshold:\n                deduplicated.append(self.frames[i])\n            else:\n                removed_count += 1\n\n        self.frames = deduplicated\n        return removed_count\n\n    def save(\n        self,\n        output_path: str | Path,\n        num_colors: int = 128,\n        optimize_for_emoji: bool = False,\n        remove_duplicates: bool = False,\n    ) -> dict:\n        \"\"\"\n        Save frames as optimized GIF for Slack.\n\n        Args:\n            output_path: Where to save the GIF\n            num_colors: Number of colors to use (fewer = smaller file)\n            optimize_for_emoji: If True, optimize for emoji size (128x128, fewer colors)\n            remove_duplicates: If True, remove duplicate consecutive frames (opt-in)\n\n        Returns:\n            Dictionary with file info (path, size, dimensions, frame_count)\n        \"\"\"\n        if not self.frames:\n            raise ValueError(\"No frames to save. Add frames with add_frame() first.\")\n\n        output_path = Path(output_path)\n\n        # Remove duplicate frames to reduce file size\n        if remove_duplicates:\n            removed = self.deduplicate_frames(threshold=0.9995)\n            if removed > 0:\n                print(\n                    f\"  Removed {removed} nearly identical frames (preserved subtle animations)\"\n                )\n\n        # Optimize for emoji if requested\n        if optimize_for_emoji:\n            if self.width > 128 or self.height > 128:\n                print(\n                    f\"  Resizing from {self.width}x{self.height} to 128x128 for emoji\"\n                )\n                self.width = 128\n                self.height = 128\n                # Resize all frames\n                resized_frames = []\n                for frame in self.frames:\n                    pil_frame = Image.fromarray(frame)\n                    pil_frame = pil_frame.resize((128, 128), Image.Resampling.LANCZOS)\n                    resized_frames.append(np.array(pil_frame))\n                self.frames = resized_frames\n            num_colors = min(num_colors, 48)  # More aggressive color limit for emoji\n\n            # More aggressive FPS reduction for emoji\n            if len(self.frames) > 12:\n                print(\n                    f\"  Reducing frames from {len(self.frames)} to ~12 for emoji size\"\n                )\n                # Keep every nth frame to get close to 12 frames\n                keep_every = max(1, len(self.frames) // 12)\n                self.frames = [\n                    self.frames[i] for i in range(0, len(self.frames), keep_every)\n                ]\n\n        # Optimize colors with global palette\n        optimized_frames = self.optimize_colors(num_colors, use_global_palette=True)\n\n        # Calculate frame duration in milliseconds\n        frame_duration = 1000 / self.fps\n\n        # Save GIF\n        imageio.imwrite(\n            output_path,\n            optimized_frames,\n            duration=frame_duration,\n            loop=0,  # Infinite loop\n        )\n\n        # Get file info\n        file_size_kb = output_path.stat().st_size / 1024\n        file_size_mb = file_size_kb / 1024\n\n        info = {\n            \"path\": str(output_path),\n            \"size_kb\": file_size_kb,\n            \"size_mb\": file_size_mb,\n            \"dimensions\": f\"{self.width}x{self.height}\",\n            \"frame_count\": len(optimized_frames),\n            \"fps\": self.fps,\n            \"duration_seconds\": len(optimized_frames) / self.fps,\n            \"colors\": num_colors,\n        }\n\n        # Print info\n        print(f\"\\n✓ GIF created successfully!\")\n        print(f\"  Path: {output_path}\")\n        print(f\"  Size: {file_size_kb:.1f} KB ({file_size_mb:.2f} MB)\")\n        print(f\"  Dimensions: {self.width}x{self.height}\")\n        print(f\"  Frames: {len(optimized_frames)} @ {self.fps} fps\")\n        print(f\"  Duration: {info['duration_seconds']:.1f}s\")\n        print(f\"  Colors: {num_colors}\")\n\n        # Size info\n        if optimize_for_emoji:\n            print(f\"  Optimized for emoji (128x128, reduced colors)\")\n        if file_size_mb > 1.0:\n            print(f\"\\n  Note: Large file size ({file_size_kb:.1f} KB)\")\n            print(\"  Consider: fewer frames, smaller dimensions, or fewer colors\")\n\n        return info\n\n    def clear(self):\n        \"\"\"Clear all frames (useful for creating multiple GIFs).\"\"\"\n        self.frames = []\n"
      },
      {
        "name": "validators.py",
        "node_type": "file",
        "content": "#!/usr/bin/env python3\n\"\"\"\nValidators - Check if GIFs meet Slack's requirements.\n\nThese validators help ensure your GIFs meet Slack's size and dimension constraints.\n\"\"\"\n\nfrom pathlib import Path\n\n\ndef validate_gif(\n    gif_path: str | Path, is_emoji: bool = True, verbose: bool = True\n) -> tuple[bool, dict]:\n    \"\"\"\n    Validate GIF for Slack (dimensions, size, frame count).\n\n    Args:\n        gif_path: Path to GIF file\n        is_emoji: True for emoji (128x128 recommended), False for message GIF\n        verbose: Print validation details\n\n    Returns:\n        Tuple of (passes: bool, results: dict with all details)\n    \"\"\"\n    from PIL import Image\n\n    gif_path = Path(gif_path)\n\n    if not gif_path.exists():\n        return False, {\"error\": f\"File not found: {gif_path}\"}\n\n    # Get file size\n    size_bytes = gif_path.stat().st_size\n    size_kb = size_bytes / 1024\n    size_mb = size_kb / 1024\n\n    # Get dimensions and frame info\n    try:\n        with Image.open(gif_path) as img:\n            width, height = img.size\n\n            # Count frames\n            frame_count = 0\n            try:\n                while True:\n                    img.seek(frame_count)\n                    frame_count += 1\n            except EOFError:\n                pass\n\n            # Get duration\n            try:\n                duration_ms = img.info.get(\"duration\", 100)\n                total_duration = (duration_ms * frame_count) / 1000\n                fps = frame_count / total_duration if total_duration > 0 else 0\n            except:\n                total_duration = None\n                fps = None\n\n    except Exception as e:\n        return False, {\"error\": f\"Failed to read GIF: {e}\"}\n\n    # Validate dimensions\n    if is_emoji:\n        optimal = width == height == 128\n        acceptable = width == height and 64 <= width <= 128\n        dim_pass = acceptable\n    else:\n        aspect_ratio = (\n            max(width, height) / min(width, height)\n            if min(width, height) > 0\n            else float(\"inf\")\n        )\n        dim_pass = aspect_ratio <= 2.0 and 320 <= min(width, height) <= 640\n\n    results = {\n        \"file\": str(gif_path),\n        \"passes\": dim_pass,\n        \"width\": width,\n        \"height\": height,\n        \"size_kb\": size_kb,\n        \"size_mb\": size_mb,\n        \"frame_count\": frame_count,\n        \"duration_seconds\": total_duration,\n        \"fps\": fps,\n        \"is_emoji\": is_emoji,\n        \"optimal\": optimal if is_emoji else None,\n    }\n\n    # Print if verbose\n    if verbose:\n        print(f\"\\nValidating {gif_path.name}:\")\n        print(\n            f\"  Dimensions: {width}x{height}\"\n            + (\n                f\" ({'optimal' if optimal else 'acceptable'})\"\n                if is_emoji and acceptable\n                else \"\"\n            )\n        )\n        print(\n            f\"  Size: {size_kb:.1f} KB\"\n            + (f\" ({size_mb:.2f} MB)\" if size_mb >= 1.0 else \"\")\n        )\n        print(\n            f\"  Frames: {frame_count}\"\n            + (f\" @ {fps:.1f} fps ({total_duration:.1f}s)\" if fps else \"\")\n        )\n\n        if not dim_pass:\n            print(\n                f\"  Note: {'Emoji should be 128x128' if is_emoji else 'Unusual dimensions for Slack'}\"\n            )\n\n        if size_mb > 5.0:\n            print(f\"  Note: Large file size - consider fewer frames/colors\")\n\n    return dim_pass, results\n\n\ndef is_slack_ready(\n    gif_path: str | Path, is_emoji: bool = True, verbose: bool = True\n) -> bool:\n    \"\"\"\n    Quick check if GIF is ready for Slack.\n\n    Args:\n        gif_path: Path to GIF file\n        is_emoji: True for emoji GIF, False for message GIF\n        verbose: Print feedback\n\n    Returns:\n        True if dimensions are acceptable\n    \"\"\"\n    passes, _ = validate_gif(gif_path, is_emoji, verbose)\n    return passes\n"
      }
    ]
  },
  {
    "name": "requirements.txt",
    "node_type": "file",
    "content": "pillow>=10.0.0\nimageio>=2.31.0\nimageio-ffmpeg>=0.4.9\nnumpy>=1.24.0"
  }
]

export default children
