// AUTO-GENERATED — DO NOT EDIT
// Source: https://github.com/anthropics/skills
import type { SkillTemplateNode } from '../types'

const children: SkillTemplateNode[] = [
  {
    "name": "SKILL.md",
    "node_type": "file",
    "content": "---\nname: webapp-testing\ndescription: Toolkit for interacting with and testing local web applications using Playwright. Supports verifying frontend functionality, debugging UI behavior, capturing browser screenshots, and viewing browser logs.\nlicense: Complete terms in LICENSE.txt\n---\n\n# Web Application Testing\n\nTo test local web applications, write native Python Playwright scripts.\n\n**Helper Scripts Available**:\n- `scripts/with_server.py` - Manages server lifecycle (supports multiple servers)\n\n**Always run scripts with `--help` first** to see usage. DO NOT read the source until you try running the script first and find that a customized solution is abslutely necessary. These scripts can be very large and thus pollute your context window. They exist to be called directly as black-box scripts rather than ingested into your context window.\n\n## Decision Tree: Choosing Your Approach\n\n```\nUser task → Is it static HTML?\n    ├─ Yes → Read HTML file directly to identify selectors\n    │         ├─ Success → Write Playwright script using selectors\n    │         └─ Fails/Incomplete → Treat as dynamic (below)\n    │\n    └─ No (dynamic webapp) → Is the server already running?\n        ├─ No → Run: python scripts/with_server.py --help\n        │        Then use the helper + write simplified Playwright script\n        │\n        └─ Yes → Reconnaissance-then-action:\n            1. Navigate and wait for networkidle\n            2. Take screenshot or inspect DOM\n            3. Identify selectors from rendered state\n            4. Execute actions with discovered selectors\n```\n\n## Example: Using with_server.py\n\nTo start a server, run `--help` first, then use the helper:\n\n**Single server:**\n```bash\npython scripts/with_server.py --server \"npm run dev\" --port 5173 -- python your_automation.py\n```\n\n**Multiple servers (e.g., backend + frontend):**\n```bash\npython scripts/with_server.py \\\n  --server \"cd backend && python server.py\" --port 3000 \\\n  --server \"cd frontend && npm run dev\" --port 5173 \\\n  -- python your_automation.py\n```\n\nTo create an automation script, include only Playwright logic (servers are managed automatically):\n```python\nfrom playwright.sync_api import sync_playwright\n\nwith sync_playwright() as p:\n    browser = p.chromium.launch(headless=True) # Always launch chromium in headless mode\n    page = browser.new_page()\n    page.goto('http://localhost:5173') # Server already running and ready\n    page.wait_for_load_state('networkidle') # CRITICAL: Wait for JS to execute\n    # ... your automation logic\n    browser.close()\n```\n\n## Reconnaissance-Then-Action Pattern\n\n1. **Inspect rendered DOM**:\n   ```python\n   page.screenshot(path='/tmp/inspect.png', full_page=True)\n   content = page.content()\n   page.locator('button').all()\n   ```\n\n2. **Identify selectors** from inspection results\n\n3. **Execute actions** using discovered selectors\n\n## Common Pitfall\n\n❌ **Don't** inspect the DOM before waiting for `networkidle` on dynamic apps\n✅ **Do** wait for `page.wait_for_load_state('networkidle')` before inspection\n\n## Best Practices\n\n- **Use bundled scripts as black boxes** - To accomplish a task, consider whether one of the scripts available in `scripts/` can help. These scripts handle common, complex workflows reliably without cluttering the context window. Use `--help` to see usage, then invoke directly. \n- Use `sync_playwright()` for synchronous scripts\n- Always close the browser when done\n- Use descriptive selectors: `text=`, `role=`, CSS selectors, or IDs\n- Add appropriate waits: `page.wait_for_selector()` or `page.wait_for_timeout()`\n\n## Reference Files\n\n- **examples/** - Examples showing common patterns:\n  - `element_discovery.py` - Discovering buttons, links, and inputs on a page\n  - `static_html_automation.py` - Using file:// URLs for local HTML\n  - `console_logging.py` - Capturing console logs during automation"
  },
  {
    "name": "examples",
    "node_type": "folder",
    "children": [
      {
        "name": "console_logging.py",
        "node_type": "file",
        "content": "from playwright.sync_api import sync_playwright\n\n# Example: Capturing console logs during browser automation\n\nurl = 'http://localhost:5173'  # Replace with your URL\n\nconsole_logs = []\n\nwith sync_playwright() as p:\n    browser = p.chromium.launch(headless=True)\n    page = browser.new_page(viewport={'width': 1920, 'height': 1080})\n\n    # Set up console log capture\n    def handle_console_message(msg):\n        console_logs.append(f\"[{msg.type}] {msg.text}\")\n        print(f\"Console: [{msg.type}] {msg.text}\")\n\n    page.on(\"console\", handle_console_message)\n\n    # Navigate to page\n    page.goto(url)\n    page.wait_for_load_state('networkidle')\n\n    # Interact with the page (triggers console logs)\n    page.click('text=Dashboard')\n    page.wait_for_timeout(1000)\n\n    browser.close()\n\n# Save console logs to file\nwith open('/mnt/user-data/outputs/console.log', 'w') as f:\n    f.write('\\n'.join(console_logs))\n\nprint(f\"\\nCaptured {len(console_logs)} console messages\")\nprint(f\"Logs saved to: /mnt/user-data/outputs/console.log\")"
      },
      {
        "name": "element_discovery.py",
        "node_type": "file",
        "content": "from playwright.sync_api import sync_playwright\n\n# Example: Discovering buttons and other elements on a page\n\nwith sync_playwright() as p:\n    browser = p.chromium.launch(headless=True)\n    page = browser.new_page()\n\n    # Navigate to page and wait for it to fully load\n    page.goto('http://localhost:5173')\n    page.wait_for_load_state('networkidle')\n\n    # Discover all buttons on the page\n    buttons = page.locator('button').all()\n    print(f\"Found {len(buttons)} buttons:\")\n    for i, button in enumerate(buttons):\n        text = button.inner_text() if button.is_visible() else \"[hidden]\"\n        print(f\"  [{i}] {text}\")\n\n    # Discover links\n    links = page.locator('a[href]').all()\n    print(f\"\\nFound {len(links)} links:\")\n    for link in links[:5]:  # Show first 5\n        text = link.inner_text().strip()\n        href = link.get_attribute('href')\n        print(f\"  - {text} -> {href}\")\n\n    # Discover input fields\n    inputs = page.locator('input, textarea, select').all()\n    print(f\"\\nFound {len(inputs)} input fields:\")\n    for input_elem in inputs:\n        name = input_elem.get_attribute('name') or input_elem.get_attribute('id') or \"[unnamed]\"\n        input_type = input_elem.get_attribute('type') or 'text'\n        print(f\"  - {name} ({input_type})\")\n\n    # Take screenshot for visual reference\n    page.screenshot(path='/tmp/page_discovery.png', full_page=True)\n    print(\"\\nScreenshot saved to /tmp/page_discovery.png\")\n\n    browser.close()"
      },
      {
        "name": "static_html_automation.py",
        "node_type": "file",
        "content": "from playwright.sync_api import sync_playwright\nimport os\n\n# Example: Automating interaction with static HTML files using file:// URLs\n\nhtml_file_path = os.path.abspath('path/to/your/file.html')\nfile_url = f'file://{html_file_path}'\n\nwith sync_playwright() as p:\n    browser = p.chromium.launch(headless=True)\n    page = browser.new_page(viewport={'width': 1920, 'height': 1080})\n\n    # Navigate to local HTML file\n    page.goto(file_url)\n\n    # Take screenshot\n    page.screenshot(path='/mnt/user-data/outputs/static_page.png', full_page=True)\n\n    # Interact with elements\n    page.click('text=Click Me')\n    page.fill('#name', 'John Doe')\n    page.fill('#email', 'john@example.com')\n\n    # Submit form\n    page.click('button[type=\"submit\"]')\n    page.wait_for_timeout(500)\n\n    # Take final screenshot\n    page.screenshot(path='/mnt/user-data/outputs/after_submit.png', full_page=True)\n\n    browser.close()\n\nprint(\"Static HTML automation completed!\")"
      }
    ]
  },
  {
    "name": "scripts",
    "node_type": "folder",
    "children": [
      {
        "name": "with_server.py",
        "node_type": "file",
        "content": "#!/usr/bin/env python3\n\"\"\"\nStart one or more servers, wait for them to be ready, run a command, then clean up.\n\nUsage:\n    # Single server\n    python scripts/with_server.py --server \"npm run dev\" --port 5173 -- python automation.py\n    python scripts/with_server.py --server \"npm start\" --port 3000 -- python test.py\n\n    # Multiple servers\n    python scripts/with_server.py \\\n      --server \"cd backend && python server.py\" --port 3000 \\\n      --server \"cd frontend && npm run dev\" --port 5173 \\\n      -- python test.py\n\"\"\"\n\nimport subprocess\nimport socket\nimport time\nimport sys\nimport argparse\n\ndef is_server_ready(port, timeout=30):\n    \"\"\"Wait for server to be ready by polling the port.\"\"\"\n    start_time = time.time()\n    while time.time() - start_time < timeout:\n        try:\n            with socket.create_connection(('localhost', port), timeout=1):\n                return True\n        except (socket.error, ConnectionRefusedError):\n            time.sleep(0.5)\n    return False\n\n\ndef main():\n    parser = argparse.ArgumentParser(description='Run command with one or more servers')\n    parser.add_argument('--server', action='append', dest='servers', required=True, help='Server command (can be repeated)')\n    parser.add_argument('--port', action='append', dest='ports', type=int, required=True, help='Port for each server (must match --server count)')\n    parser.add_argument('--timeout', type=int, default=30, help='Timeout in seconds per server (default: 30)')\n    parser.add_argument('command', nargs=argparse.REMAINDER, help='Command to run after server(s) ready')\n\n    args = parser.parse_args()\n\n    # Remove the '--' separator if present\n    if args.command and args.command[0] == '--':\n        args.command = args.command[1:]\n\n    if not args.command:\n        print(\"Error: No command specified to run\")\n        sys.exit(1)\n\n    # Parse server configurations\n    if len(args.servers) != len(args.ports):\n        print(\"Error: Number of --server and --port arguments must match\")\n        sys.exit(1)\n\n    servers = []\n    for cmd, port in zip(args.servers, args.ports):\n        servers.append({'cmd': cmd, 'port': port})\n\n    server_processes = []\n\n    try:\n        # Start all servers\n        for i, server in enumerate(servers):\n            print(f\"Starting server {i+1}/{len(servers)}: {server['cmd']}\")\n\n            # Use shell=True to support commands with cd and &&\n            process = subprocess.Popen(\n                server['cmd'],\n                shell=True,\n                stdout=subprocess.PIPE,\n                stderr=subprocess.PIPE\n            )\n            server_processes.append(process)\n\n            # Wait for this server to be ready\n            print(f\"Waiting for server on port {server['port']}...\")\n            if not is_server_ready(server['port'], timeout=args.timeout):\n                raise RuntimeError(f\"Server failed to start on port {server['port']} within {args.timeout}s\")\n\n            print(f\"Server ready on port {server['port']}\")\n\n        print(f\"\\nAll {len(servers)} server(s) ready\")\n\n        # Run the command\n        print(f\"Running: {' '.join(args.command)}\\n\")\n        result = subprocess.run(args.command)\n        sys.exit(result.returncode)\n\n    finally:\n        # Clean up all servers\n        print(f\"\\nStopping {len(server_processes)} server(s)...\")\n        for i, process in enumerate(server_processes):\n            try:\n                process.terminate()\n                process.wait(timeout=5)\n            except subprocess.TimeoutExpired:\n                process.kill()\n                process.wait()\n            print(f\"Server {i+1} stopped\")\n        print(\"All servers stopped\")\n\n\nif __name__ == '__main__':\n    main()"
      }
    ]
  }
]

export default children
