# -*- coding: utf-8 -*-

"""
Task Coach - Your friendly task manager
Copyright (C) 2004-2016 Task Coach developers <developers@taskcoach.org>

Task Coach is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Task Coach is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from taskcoachlib import meta, persistence, patterns, operating_system
from taskcoachlib.i18n import _
from taskcoachlib.widgets import GetPassword
from taskcoachlib.gui.dialog import BackupManagerDialog
import wx
import os
import gc
import sys
import codecs
import traceback

class IOController(object):
    """IOController is responsible for opening, closing, loading,
    saving, and exporting files. It also presents the necessary dialogs
    to let the user specify what file to load/save/etc."""

    def __init__(self, taskFile, messageCallback, settings):
        super().__init__()
        self.__taskFile = taskFile
        self.__messageCallback = messageCallback
        self.__settings = settings
        defaultPath = os.path.expanduser("~")
        self.__tskFileSaveDialogOpts = {
            "default_path": defaultPath,
            "default_extension": "tsk",
            "wildcard": _("%s files (*.tsk)|*.tsk|All files (*.*)|*")
            % meta.name,
        }
        self.__tskFileOpenDialogOpts = {
            "default_path": defaultPath,
            "default_extension": "tsk",
            "wildcard": _(
                "%s files (*.tsk)|*.tsk|Backup files (*.tsk.bak)|*.tsk.bak|"
                "All files (*.*)|*"
            )
            % meta.name,
        }
        self.__icsFileDialogOpts = {
            "default_path": defaultPath,
            "default_extension": "ics",
            "wildcard": _("iCalendar files (*.ics)|*.ics|All files (*.*)|*"),
        }
        self.__htmlFileDialogOpts = {
            "default_path": defaultPath,
            "default_extension": "html",
            "wildcard": _("HTML files (*.html)|*.html|All files (*.*)|*"),
        }
        self.__csvFileDialogOpts = {
            "default_path": defaultPath,
            "default_extension": "csv",
            "wildcard": _(
                "CSV files (*.csv)|*.csv|Text files (*.txt)|*.txt|"
                "All files (*.*)|*"
            ),
        }
        self.__todotxtFileDialogOpts = {
            "default_path": defaultPath,
            "default_extension": "txt",
            "wildcard": _("Todo.txt files (*.txt)|*.txt|All files (*.*)|*"),
        }
        self.__errorMessageOptions = dict(
            caption=_("%s file error") % meta.name, style=wx.ICON_ERROR
        )

    def need_save(self):
        return self.__taskFile.need_save()

    def changed_on_disk(self):
        return self.__taskFile.changed_on_disk()

    def has_deleted_items(self):
        return bool(
            [task for task in self.__taskFile.tasks() if task.isDeleted()]
            + [note for note in self.__taskFile.notes() if note.isDeleted()]
        )

    def purge_deleted_items(self):
        self.__taskFile.tasks().removeItems(
            [task for task in self.__taskFile.tasks() if task.isDeleted()]
        )
        self.__taskFile.notes().removeItems(
            [note for note in self.__taskFile.notes() if note.isDeleted()]
        )

    def open_after_start(self, commandLineArgs, earlyLockResult=None):
        """Open either the file specified on the command line, or the file
        the user was working on previously, or none at all.

        Args:
            commandLineArgs: Command line arguments
            earlyLockResult: Result from early lock check before main window:
                None = no file to open
                'ok' = file not locked, proceed normally
                'break' = file locked, user chose to break lock
                'skip' = file locked, user chose not to break (start fresh)
        """
        if commandLineArgs:
            filename = commandLineArgs[0]
        else:
            filename = self.__settings.get("file", "lastfile")
        if filename:
            # If early lock check was done, use its result
            if earlyLockResult == 'break':
                # User already confirmed breaking the lock
                wx.CallAfter(self.open, filename, breakLock=True)
            elif earlyLockResult == 'skip':
                # User chose not to break lock - start with no file (fresh)
                pass
            else:
                # Normal case - open file (will check lock again if needed)
                wx.CallAfter(self.open, filename)

    def open(
        self,
        filename=None,
        showerror=wx.MessageBox,
        fileExists=os.path.exists,
        breakLock=False,
        lock=True,
    ):
        if self.__taskFile.need_save():
            if not self.__save_unsaved_changes():
                return
        if not filename:
            filename = self.__ask_user_for_file(
                _("Open"), self.__tskFileOpenDialogOpts
            )
        if not filename:
            return
        self.__update_default_path(filename)
        if fileExists(filename):
            self.__close_unconditionally()
            self.__add_recent_file(filename)
            try:
                try:
                    self.__taskFile.load(
                        filename, lock=lock, breakLock=breakLock
                    )
                except Exception:
                    raise
            except persistence.LockTimeout:
                if breakLock:
                    if self.__ask_open_unlocked(filename):
                        self.open(filename, showerror, lock=False)
                elif self.__ask_break_lock(filename):
                    self.open(filename, showerror, breakLock=True)
                else:
                    return
            except persistence.LockFailed:
                if self.__ask_open_unlocked(filename):
                    self.open(filename, showerror, lock=False)
                else:
                    return
            except persistence.xml.reader.XMLReaderTooNewException:
                self.__show_too_new_error_message(filename, showerror)
                return
            except Exception:
                self.__show_generic_error_message(
                    filename, showerror, showBackups=True
                )
                return
            self.__messageCallback(
                _("Loaded %(nrtasks)d tasks from " "%(filename)s")
                % dict(
                    nrtasks=len(self.__taskFile.tasks()),
                    filename=self.__taskFile.filename(),
                )
            )
        else:
            errorMessage = (
                _("Cannot open %s because it doesn't exist") % filename
            )
            # Use CallAfter on Mac OS X because otherwise the app will hang:
            if operating_system.isMac():
                wx.CallAfter(
                    showerror, errorMessage, **self.__errorMessageOptions
                )
            else:
                showerror(errorMessage, **self.__errorMessageOptions)
            self.__remove_recent_file(filename)

    def merge(self, filename=None, showerror=wx.MessageBox):
        if not filename:
            filename = self.__ask_user_for_file(
                _("Merge"), self.__tskFileOpenDialogOpts
            )
        if filename:
            try:
                self.__taskFile.merge(filename)
            except persistence.LockTimeout:
                showerror(
                    _("Cannot open %(filename)s\nbecause it is locked.")
                    % dict(filename=filename),
                    **self.__errorMessageOptions
                )
                return
            except persistence.xml.reader.XMLReaderTooNewException:
                self.__show_too_new_error_message(filename, showerror)
                return
            except Exception:
                self.__show_generic_error_message(filename, showerror)
                return
            self.__messageCallback(
                _("Merged %(filename)s") % dict(filename=filename)
            )
            self.__add_recent_file(filename)

    def save(self, showerror=wx.MessageBox):
        if self.__taskFile.filename():
            if self._save_save(self.__taskFile, showerror):
                return True
            else:
                return self.save_as(showerror=showerror)
        elif not self.__taskFile.isEmpty():
            return self.save_as(showerror=showerror)  # Ask for filename
        else:
            return False

    def merge_disk_changes(self):
        self.__taskFile.merge_disk_changes()

    def save_as(
        self, filename=None, showerror=wx.MessageBox, fileExists=os.path.exists
    ):
        if not filename:
            filename = self.__ask_user_for_file(
                _("Save as"),
                self.__tskFileSaveDialogOpts,
                flag=wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT,
                fileExists=fileExists,
            )
            if not filename:
                return False  # User didn't enter a filename, cancel save
        if self._save_save(self.__taskFile, showerror, filename):
            return True
        else:
            return self.save_as(showerror=showerror)  # Try again

    def save_selection(
        self,
        tasks,
        filename=None,
        showerror=wx.MessageBox,
        TaskFileClass=persistence.TaskFile,
        fileExists=os.path.exists,
    ):
        if not filename:
            filename = self.__ask_user_for_file(
                _("Save selection"),
                self.__tskFileSaveDialogOpts,
                flag=wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT,
                fileExists=fileExists,
            )
            if not filename:
                return False  # User didn't enter a filename, cancel save
        selectionFile = self._create_selection_file(tasks, TaskFileClass)
        if self._save_save(selectionFile, showerror, filename):
            return True
        else:
            return self.save_selection(
                tasks, showerror=showerror, TaskFileClass=TaskFileClass
            )  # Try again

    def _create_selection_file(self, tasks, TaskFileClass):
        selectionFile = TaskFileClass()
        # Add the selected tasks:
        selectionFile.tasks().extend(tasks)
        # Include categories used by the selected tasks:
        allCategories = set()
        for task in tasks:
            allCategories.update(task.categories())
        # Also include parents of used categories, recursively:
        for category in allCategories.copy():
            allCategories.update(category.ancestors())
        selectionFile.categories().extend(allCategories)
        return selectionFile

    def _save_save(self, taskFile, showerror, filename=None):
        """Save the file and show an error message if saving fails."""
        try:
            if filename:
                taskFile.saveas(filename)
            else:
                filename = taskFile.filename()
                taskFile.save()
            self.__show_save_message(taskFile)
            self.__add_recent_file(filename)
            return True
        except persistence.LockTimeout:
            errorMessage = _(
                "Cannot save %s\nIt is locked by another instance " "of %s.\n"
            ) % (filename, meta.name)
            showerror(errorMessage, **self.__errorMessageOptions)
            return False
        except (OSError, IOError, persistence.LockFailed) as reason:
            errorMessage = _("Cannot save %s\n%s") % (
                filename,
                str(reason),
            )
            showerror(errorMessage, **self.__errorMessageOptions)
            return False

    def save_as_template(self, task):
        templates = persistence.TemplateList(
            self.__settings.pathToTemplatesDir()
        )
        templates.addTemplate(task)
        templates.save()

    def import_template(self, showerror=wx.MessageBox):
        filename = self.__ask_user_for_file(
            _("Import template"),
            fileDialogOpts={
                "default_extension": "tsktmpl",
                "wildcard": _("%s template files (*.tsktmpl)|" "*.tsktmpl")
                % meta.name,
            },
        )
        if filename:
            templates = persistence.TemplateList(
                self.__settings.pathToTemplatesDir()
            )
            try:
                templates.copyTemplate(filename)
            except Exception as reason:  # pylint: disable=W0703
                errorMessage = _("Cannot import template %s\n%s") % (
                    filename,
                    str(reason),
                )
                showerror(errorMessage, **self.__errorMessageOptions)

    def close(self, force=False):
        if self.__taskFile.need_save():
            if force:
                # No user interaction, since we're forced to close right now.
                if self.__taskFile.filename():
                    self._save_save(
                        self.__taskFile, lambda *args, **kwargs: None
                    )
                else:
                    pass  # No filename, we cannot ask, give up...
            else:
                if not self.__save_unsaved_changes():
                    return False
        self.__close_unconditionally()
        return True

    def export(
        self,
        title,
        fileDialogOpts,
        writerClass,
        viewer,
        selectionOnly,
        openfile=codecs.open,
        showerror=wx.MessageBox,
        filename=None,
        fileExists=os.path.exists,
        **kwargs
    ):
        filename = filename or self.__ask_user_for_file(
            title,
            fileDialogOpts,
            flag=wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT,
            fileExists=fileExists,
        )
        if filename:
            fd = self.__open_file_for_writing(filename, openfile, showerror)
            if fd is None:
                return False
            count = writerClass(fd, filename).write(
                viewer, self.__settings, selectionOnly, **kwargs
            )
            fd.close()
            self.__messageCallback(
                _("Exported %(count)d items to " "%(filename)s")
                % dict(count=count, filename=filename)
            )
            return True
        else:
            return False

    def export_as_html(
        self,
        viewer,
        selectionOnly=False,
        separateCSS=False,
        columns=None,
        openfile=codecs.open,
        showerror=wx.MessageBox,
        filename=None,
        fileExists=os.path.exists,
        defaultFilename=None,
    ):
        fileOpts = self.__htmlFileDialogOpts.copy()
        if defaultFilename:
            fileOpts["default_filename"] = defaultFilename
        return self.export(
            _("Export as HTML"),
            fileOpts,
            persistence.HTMLWriter,
            viewer,
            selectionOnly,
            openfile,
            showerror,
            filename,
            fileExists,
            separateCSS=separateCSS,
            columns=columns,
            taskFile=self.__taskFile,
        )

    def export_as_csv(
        self,
        viewer,
        selectionOnly=False,
        separateDateAndTimeColumns=False,
        columns=None,
        fileExists=os.path.exists,
        defaultFilename=None,
    ):
        fileOpts = self.__csvFileDialogOpts.copy()
        if defaultFilename:
            fileOpts["default_filename"] = defaultFilename
        return self.export(
            _("Export as CSV"),
            fileOpts,
            persistence.CSVWriter,
            viewer,
            selectionOnly,
            separateDateAndTimeColumns=separateDateAndTimeColumns,
            columns=columns,
            fileExists=fileExists,
            taskFile=self.__taskFile,
        )

    def export_as_icalendar(
        self, viewer, selectionOnly=False, selectedFields=None,
        defaultFilename=None, fileExists=os.path.exists
    ):
        # Use default filename if provided
        fileOpts = self.__icsFileDialogOpts.copy()
        if defaultFilename:
            fileOpts["default_filename"] = defaultFilename
        return self.export(
            _("Export as iCalendar"),
            fileOpts,
            persistence.iCalendarWriter,
            viewer,
            selectionOnly,
            fileExists=fileExists,
            selectedFields=selectedFields,
            taskFile=self.__taskFile,
        )

    def export_as_todo_txt(
        self, viewer, selectionOnly=False, fileExists=os.path.exists,
        defaultFilename=None,
    ):
        fileOpts = self.__todotxtFileDialogOpts.copy()
        if defaultFilename:
            fileOpts["default_filename"] = defaultFilename
        return self.export(
            _("Export as Todo.txt"),
            fileOpts,
            persistence.TodoTxtWriter,
            viewer,
            selectionOnly,
            fileExists=fileExists,
            taskFile=self.__taskFile,
        )

    def import_csv(self, **kwargs):
        persistence.CSVReader(
            self.__taskFile.tasks(), self.__taskFile.categories()
        ).read(**kwargs)

    def import_todo_txt(self, filename):
        persistence.TodoTxtReader(
            self.__taskFile.tasks(), self.__taskFile.categories()
        ).read(filename)

    def filename(self):
        return self.__taskFile.filename()

    def __open_file_for_writing(
        self, filename, openfile, showerror, mode="w", encoding="utf-8"
    ):
        try:
            return openfile(filename, mode, encoding)
        except IOError as reason:
            errorMessage = _("Cannot open %s\n%s") % (
                filename,
                str(reason),
            )
            showerror(errorMessage, **self.__errorMessageOptions)
            return None

    def __add_recent_file(self, fileName):
        recentFiles = self.__settings.getlist("file", "recentfiles")
        if fileName in recentFiles:
            recentFiles.remove(fileName)
        recentFiles.insert(0, fileName)
        maximumNumberOfRecentFiles = self.__settings.getint(
            "file", "maxrecentfiles"
        )
        recentFiles = recentFiles[:maximumNumberOfRecentFiles]
        self.__settings.setlist("file", "recentfiles", recentFiles)

    def __remove_recent_file(self, fileName):
        recentFiles = self.__settings.getlist("file", "recentfiles")
        if fileName in recentFiles:
            recentFiles.remove(fileName)
            self.__settings.setlist("file", "recentfiles", recentFiles)

    def __ask_user_for_file(
        self, title, fileDialogOpts, flag=wx.FD_OPEN, fileExists=os.path.exists
    ):
        filename = wx.FileSelector(
            title, flags=flag, **fileDialogOpts
        )  # pylint: disable=W0142
        if filename and (flag & wx.FD_SAVE):
            # On Ubuntu, the default extension is not added automatically to
            # a filename typed by the user. Add the extension if necessary.
            extension = os.path.extsep + fileDialogOpts["default_extension"]
            if not filename.endswith(extension):
                filename += extension
                if fileExists(filename):
                    return self.__ask_user_for_overwrite_confirmation(
                        filename, title, fileDialogOpts
                    )
        return filename

    def __ask_user_for_overwrite_confirmation(
        self, filename, title, fileDialogOpts
    ):
        result = wx.MessageBox(
            _("A file named %s already exists.\n" "Do you want to replace it?")
            % filename,
            title,
            style=wx.YES_NO | wx.CANCEL | wx.ICON_QUESTION | wx.NO_DEFAULT,
        )
        if result == wx.YES:
            extensions = {"Todo.txt": ".txt"}
            for auto in set(
                self.__settings.getlist("file", "autoimport")
                + self.__settings.getlist("file", "autoexport")
            ):
                autoName = os.path.splitext(filename)[0] + extensions[auto]
                if os.path.exists(autoName):
                    os.remove(autoName)
                if os.path.exists(autoName + "-meta"):
                    os.remove(autoName + "-meta")
            return filename
        elif result == wx.NO:
            return self.__ask_user_for_file(
                title, fileDialogOpts, flag=wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT
            )
        else:
            return None

    def __save_unsaved_changes(self):
        result = wx.MessageBox(
            _("You have unsaved changes.\n" "Save before closing?"),
            _("%s: save changes?") % meta.name,
            style=wx.YES_NO | wx.CANCEL | wx.ICON_QUESTION | wx.YES_DEFAULT,
        )
        if result == wx.YES:
            if not self.save():
                return False
        elif result == wx.CANCEL:
            return False
        return True

    def __ask_break_lock(self, filename):
        result = wx.MessageBox(
            _(
                """Cannot open %s because it is locked.

This means either that another instance of TaskCoach
is running and has this file opened, or that a previous
instance of Task Coach crashed. If no other instance is
running, you can safely break the lock.

Break the lock?"""
            )
            % filename,
            _("%s: file locked") % meta.name,
            style=wx.YES_NO | wx.ICON_QUESTION | wx.NO_DEFAULT,
        )
        return result == wx.YES

    def __ask_open_unlocked(self, filename):
        result = wx.MessageBox(
            _(
                "Cannot acquire a lock because locking is not "
                "supported\non the location of %s.\n"
                "Open %s unlocked?"
            )
            % (filename, filename),
            _("%s: file locked") % meta.name,
            style=wx.YES_NO | wx.ICON_QUESTION | wx.NO_DEFAULT,
        )
        return result == wx.YES

    def __close_unconditionally(self):
        self.__messageCallback(_("Closed %s") % self.__taskFile.filename())
        self.__taskFile.close()
        patterns.CommandHistory().clear()
        gc.collect()

    def __show_save_message(self, savedFile):
        self.__messageCallback(
            _("Saved %(nrtasks)d tasks to %(filename)s")
            % {
                "nrtasks": len(savedFile.tasks()),
                "filename": savedFile.filename(),
            }
        )

    def __show_too_new_error_message(self, filename, showerror):
        showerror(
            _(
                "Cannot open %(filename)s\n"
                "because it was created by a newer version of %(name)s.\n"
                "Please upgrade %(name)s."
            )
            % dict(filename=filename, name=meta.name),
            **self.__errorMessageOptions
        )

    def __show_generic_error_message(
        self, filename, showerror, showBackups=False
    ):
        sys.stderr.write("".join(traceback.format_exception(*sys.exc_info())))
        limitedException = "".join(
            traceback.format_exception(*sys.exc_info(), limit=10)
        )
        message = _("Error while reading %s:\n") % filename + limitedException
        man = persistence.BackupManifest(self.__settings)
        if showBackups and man.hasBackups(filename):
            message += "\n" + _(
                "The backup manager will now open to allow you to restore\nan older version of this file."
            )
        showerror(message, **self.__errorMessageOptions)

        if showBackups and man.hasBackups(filename):
            dlg = BackupManagerDialog(None, self.__settings, filename)
            try:
                if dlg.ShowModal() == wx.ID_OK:
                    wx.CallAfter(self.open, dlg.restoredFilename())
            finally:
                dlg.Destroy()

    def __update_default_path(self, filename):
        for options in [
            self.__tskFileOpenDialogOpts,
            self.__tskFileSaveDialogOpts,
            self.__csvFileDialogOpts,
            self.__icsFileDialogOpts,
            self.__htmlFileDialogOpts,
        ]:
            options["default_path"] = os.path.dirname(filename)
