"""
Task Coach - Your friendly task manager
Copyright (C) 2004-2016 Task Coach developers <developers@taskcoach.org>
Copyright (C) 2008 Rob McMullen <rob.mcmullen@gmail.com>

Task Coach is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Task Coach is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import wx
from taskcoachlib import operating_system
from taskcoachlib.gui.icons.icon_library import icon_catalog, LIST_ICON_SIZE
from taskcoachlib.gui.newid import IdProvider
from taskcoachlib.meta.debug import log_step


""" User interface commands (subclasses of UICommand) are actions that can
    be invoked by the user via the user interface (menu's, toolbar, etc.).
    See the Taskmaster pattern described here: 
    http://www.objectmentor.com/resources/articles/taskmast.pdf 
"""  # pylint: disable=W0105


class MenuItem(wx.MenuItem):
    """Menu item that knows its command and can update its own enabled state."""

    def __init__(self, command, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._command = command

    def update_state(self):
        enabled = bool(self._command.enabled(None))
        self.Enable(enabled)
        new_text = self._command.current_menu_text()
        if new_text is not None:
            try:
                self.SetItemLabel(new_text)
            except Exception:
                pass
        if enabled and self.IsCheckable():
            check = self._command.checked()
            if check is not None:
                self.Check(check)


class UICommand(object):
    """Base user interface command. An UICommand is some action that can be
    associated with menus and/or toolbars. It contains the menutext and
    helptext to be displayed and methods to attach the command to a menu
    or toolbar. Subclasses should implement doCommand() and optionally
    override enabled()."""

    def __init__(
        self,
        menuText="",
        helpText="",
        icon_id=None,
        kind=wx.ITEM_NORMAL,
        id=None,
        icon_id2=None,
        *args,
        **kwargs
    ):  # pylint: disable=W0622
        super().__init__()
        menuText = menuText or "<%s>" % _("None")
        self.menuText = menuText if "&" in menuText else "&" + menuText
        self.helpText = helpText
        self.icon_id = icon_id
        self.icon_id2 = icon_id2
        self.kind = kind
        self.id = IdProvider.get()
        self.toolbar = None
        self.menuItems = []  # uiCommands can be used in multiple menu's

    def __del__(self):
        IdProvider.put(self.id)

    def __eq__(self, other):
        return self is other

    def uniqueName(self):
        return self.__class__.__name__

    def accelerators(self):
        # The ENTER and NUMPAD_ENTER keys are treated differently between platforms...
        if "\t" in self.menuText and (
            "ENTER" in self.menuText or "RETURN" in self.menuText
        ):
            flags = wx.ACCEL_NORMAL
            for key in self.menuText.split("\t")[1].split("+"):
                if key == "Ctrl":
                    flags |= (
                        wx.ACCEL_CMD
                        if operating_system.isMac()
                        else wx.ACCEL_CTRL
                    )
                elif key in ["Shift", "Alt"]:
                    flags |= dict(Shift=wx.ACCEL_SHIFT, Alt=wx.ACCEL_ALT)[key]
                else:
                    assert key in ["ENTER", "RETURN"], key
            return [(flags, wx.WXK_NUMPAD_ENTER, self.id)]
        return []

    def addToMenu(self, menu, window, position=None, subMenu=None):
        menuItem = MenuItem(
            self, menu, self.id, self.menuText, self.helpText, self.kind,
            subMenu=subMenu
        )
        self.menuItems.append(menuItem)
        self.addBitmapToMenuItem(menuItem)
        if position is None:
            menu.Append(menuItem)
        else:
            menu.Insert(position, menuItem)
        self.bind(window, self.id)
        return self.id

    def addBitmapToMenuItem(self, menuItem):
        if (
            self.icon_id2
            and self.kind == wx.ITEM_CHECK
            and not operating_system.isGTK()
        ):
            bitmap1 = icon_catalog.get_bitmap(self.icon_id, LIST_ICON_SIZE)
            bitmap2 = icon_catalog.get_bitmap(self.icon_id2, LIST_ICON_SIZE)
            menuItem.SetBitmaps(bitmap1, bitmap2)
        elif self.kind == wx.ITEM_NORMAL:
            if self.icon_id is None:
                return  # No icon intended - correct, do nothing

            bitmap = icon_catalog.get_bitmap(self.icon_id, LIST_ICON_SIZE)
            if not bitmap.IsOk():
                # TRAP: icon_id given but invalid - this is an error
                log_step("ERROR: invalid icon '%s' for menu item '%s'" %
                         (self.icon_id, menuItem.GetItemLabelText()), prefix="ICON")
                return

            menuItem.SetBitmap(bitmap)

    def removeFromMenu(self, menu, window):
        for menuItem in self.menuItems:
            if menuItem.GetMenu() == menu:
                self.menuItems.remove(menuItem)
                menuId = menuItem.GetId()
                menu.Remove(menuId)
                break
        self.unbind(window, menuId)

    def appendToToolBar(self, toolbar):
        self.toolbar = toolbar
        bitmap = icon_catalog.get_bitmap(
            self.icon_id, toolbar.GetToolBitmapSize()[0]
        )
        toolbar.AddLabelTool(
            self.id,
            "",
            bitmap,
            wx.NullBitmap,
            self.kind,
            shortHelp=wx.MenuItem.GetLabelText(self.menuText),
            longHelp=self.helpText,
        )
        self.bind(toolbar, self.id)
        return self.id

    def bind(self, window, itemId):
        window.Bind(wx.EVT_MENU, self.onCommandActivate, id=itemId)

    def unbind(self, window, itemId):
        window.Unbind(wx.EVT_MENU, id=itemId)

    def onCommandActivate(self, event, *args, **kwargs):
        """For controls such as the ListCtrl and the TreeCtrl, activating
        the command is possible even when not enabled, so we need an
        explicit check here. Otherwise hitting return on an empty
        selection in the ListCtrl would bring up the TaskEditor."""
        if self.enabled(event):
            return self.doCommand(event, *args, **kwargs)

    def __call__(self, *args, **kwargs):
        return self.onCommandActivate(*args, **kwargs)

    def doCommand(self, event):
        raise NotImplementedError  # pragma: no cover

    def enabled(self, event):  # pylint: disable=W0613
        """Can be overridden in a subclass."""
        return True

    def current_menu_text(self):
        """Return updated menu text, or None to keep current. Override in
        subclasses whose menu text changes based on context (e.g. selection
        type)."""
        return None

    def checked(self):
        """Return True/False for checkable items, or None to skip.
        Override in subclasses with ITEM_CHECK kind."""
        return None

    def updateToolHelp(self):
        if not self.toolbar:
            return  # Not attached to a toolbar or it's hidden
        shortHelp = wx.MenuItem.GetLabelText(self.getMenuText())
        if shortHelp != self.toolbar.GetToolShortHelp(self.id):
            self.toolbar.SetToolShortHelp(self.id, shortHelp)
        longHelp = self.getHelpText()
        if longHelp != self.toolbar.GetToolLongHelp(self.id):
            self.toolbar.SetToolLongHelp(self.id, longHelp)

    def updateMenuText(self, menuText):
        self.menuText = menuText
        # SetItemLabel works on all platforms in modern wxPython 4.x
        # The old Windows-specific code that deleted/inserted menu items
        # was causing access violations when popup menus were displayed.
        for menuItem in self.menuItems:
            try:
                menuItem.SetItemLabel(menuText)
            except Exception:
                pass  # Ignore errors from deleted menu items

    def mainWindow(self):
        return wx.GetApp().TopWindow

    def getMenuText(self):
        return self.menuText

    def getHelpText(self):
        return self.helpText

