"""
Task Coach - Your friendly task manager
Copyright (C) 2004-2016 Task Coach developers <developers@taskcoach.org>

Task Coach is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Task Coach is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import os
from . import generator
from taskcoachlib import meta


css = (
    """/*
CSS file generated by %(name)s %(version)s for:
%%s. 
You can edit this file, it will not be overwritten by %(name)s.
*/
"""
    % meta.data.metaDict
    + generator.css
    + """
/* Other possibilities to tune the layout include:

   Change the styles for a header of a specific column, in this case the subject 
   column. Note how this style overrides the default style in the HTML file: 
   
   th.subject {
      text-align: center;
   }

   If the exported HTML file contains tasks it is possible to change the color 
   of completed (or overdue, duesoon, late, inactive, active) tasks like this:
   
   .completed {
       color: red;
   }
   
*/
"""
)


class _ViewerProxy(object):
    """Lightweight viewer proxy for 'All' exports when no real viewer is available."""

    def __init__(self, items, columns, title, isShowingTasks=False):
        self._items = items
        self._columns = columns
        self._title = title
        self._isShowingTasks = isShowingTasks

    def title(self):
        return self._title

    def visibleColumns(self):
        return self._columns

    def is_tree_viewer(self):
        return False

    def visibleItems(self):
        return self._items

    def isselected(self, item):
        return True

    def isSortable(self):
        return False

    def isSortedBy(self, name):
        return False

    def isShowingTasks(self):
        return self._isShowingTasks


class HTMLWriter(object):
    # Constants for "All" export options
    ALL_TASKS = "ALL_TASKS"
    ALL_EFFORTS = "ALL_EFFORTS"
    ALL_CATEGORIES = "ALL_CATEGORIES"
    ALL_NOTES = "ALL_NOTES"

    def __init__(self, fd, filename=None):
        self.__fd = fd
        self.__filename = filename
        self.__cssFilename = (
            os.path.splitext(filename)[0] + ".css" if filename else ""
        )

    def write(
        self,
        viewer,
        settings,
        selectionOnly=False,
        separateCSS=False,
        columns=None,
        taskFile=None,
    ):
        if isinstance(viewer, str) and viewer.startswith("ALL_"):
            viewer = self._createProxy(viewer, columns, taskFile)
            selectionOnly = False

        cssFilename = (
            os.path.basename(self.__cssFilename) if separateCSS else ""
        )
        htmlText, count = generator.viewer2html(
            viewer, settings, cssFilename, selectionOnly, columns
        )
        self.__fd.write(htmlText)
        if separateCSS:
            self._writeCSS()
        return count

    def _createProxy(self, viewerType, columns, taskFile):
        """Create a viewer proxy for 'All' exports."""
        items = []
        title = "Export"
        isShowingTasks = False
        if taskFile:
            if viewerType == self.ALL_TASKS:
                items = list(taskFile.tasks())
                title = "Tasks"
                isShowingTasks = True
            elif viewerType == self.ALL_EFFORTS:
                items = list(taskFile.efforts())
                title = "Efforts"
            elif viewerType == self.ALL_CATEGORIES:
                items = list(taskFile.categories())
                title = "Categories"
            elif viewerType == self.ALL_NOTES:
                items = list(taskFile.notes())
                title = "Notes"
        return _ViewerProxy(items, columns or [], title, isShowingTasks)

    def _writeCSS(self, open=open):  # pylint: disable=W0622
        if not self.__cssFilename or os.path.exists(self.__cssFilename):
            return
        try:
            fd = open(self.__cssFilename, "wb")
            fd.write(css % self.__filename)
            fd.close()
        except IOError:
            pass
