"""
Task Coach - Your friendly task manager
Copyright (C) 2004-2016 Task Coach developers <developers@taskcoach.org>

Task Coach is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Task Coach is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import wx, re, sre_constants
from taskcoachlib.gui.icons.icon_library import icon_catalog, LIST_ICON_SIZE
from taskcoachlib.widgets import tooltip
from taskcoachlib.i18n import _


class _SearchCtrlInner(tooltip.ToolTipMixin, wx.SearchCtrl):
    """Inner search control with all the search functionality.

    This is wrapped by SearchCtrl (wx.Panel) for Wayland-compatible popup positioning.
    """
    # Debounce delay in milliseconds - wait this long after user stops typing
    # before triggering the search. This prevents expensive operations on every keystroke.
    SEARCH_DEBOUNCE_DELAY_MS = 500

    def __init__(self, parent, wrapper, *args, **kwargs):
        self.__wrapper = wrapper  # The wrapping Panel for popup positioning
        self.__callback = kwargs.pop("callback")
        self.__matchCase = kwargs.pop("matchCase", False)
        self.__includeSubItems = kwargs.pop("includeSubItems", False)
        self.__searchDescription = kwargs.pop("searchDescription", False)
        self.__regularExpression = kwargs.pop("regularExpression", False)
        self.__bitmapSize = kwargs.pop("size", (LIST_ICON_SIZE, LIST_ICON_SIZE))
        self.__debounceDelay = kwargs.pop("debounceDelay", self.SEARCH_DEBOUNCE_DELAY_MS)
        value = kwargs.pop("value", "")
        super().__init__(parent, *args, **kwargs)
        self.SetSearchMenuBitmap(
            self.getBitmap("taskcoach_actions_magnifier_glass_dropdown_icon")
        )
        self.SetSearchBitmap(self.getBitmap("nuvola_apps_xmag"))
        self.SetCancelBitmap(self.getBitmap("nuvola_status_dialog-error"))
        self.__timer = wx.Timer(self)
        self.__recentSearches = []
        self.__maxRecentSearches = 5
        self.__tooltip = tooltip.SimpleToolTip(self)
        self.createMenu()
        self.bindEventHandlers()
        self.SetValue(value)

    def GetMainWindow(self):
        return self

    def getTextCtrl(self):
        textCtrl = [
            child
            for child in self.GetChildren()
            if isinstance(child, wx.TextCtrl)
        ]
        return textCtrl[0] if textCtrl else self

    def getBitmap(self, icon_id):
        return icon_catalog.get_bitmap(icon_id, self.__bitmapSize[0])

    def createMenu(self):
        # pylint: disable=W0201
        menu = wx.Menu()
        self.__matchCaseMenuItem = menu.AppendCheckItem(
            wx.ID_ANY, _("&Match case"), _("Match case when filtering")
        )
        self.__matchCaseMenuItem.Check(self.__matchCase)
        self.__includeSubItemsMenuItem = menu.AppendCheckItem(
            wx.ID_ANY,
            _("&Include sub items"),
            _("Include sub items of matching items in the search results"),
        )
        self.__includeSubItemsMenuItem.Check(self.__includeSubItems)
        self.__searchDescriptionMenuItem = menu.AppendCheckItem(
            wx.ID_ANY,
            _("&Search description too"),
            _("Search both subject and description"),
        )
        self.__searchDescriptionMenuItem.Check(self.__searchDescription)
        self.__regularExpressionMenuItem = menu.AppendCheckItem(
            wx.ID_ANY,
            _("&Regular Expression"),
            _("Consider search text as a regular expression"),
        )
        self.__regularExpressionMenuItem.Check(self.__regularExpression)
        self.SetMenu(menu)

    def _onSearchButton(self, event):  # pylint: disable=W0613
        """Handle search dropdown button click with Wayland-compatible positioning.

        On Wayland, popups position relative to their transient parent window.
        By calling PopupMenu on the wrapper Panel (not on self), we establish
        the Panel as the transient parent. Position arguments are irrelevant -
        Wayland's compositor handles placement.

        See bugs/ISSUE_159_SEARCH_DROPDOWN_POSITION.md for details.
        """
        self.HideTip()
        menu = self.GetMenu()
        if menu:
            self.__wrapper.PopupMenu(menu)
        # Don't Skip - we handle the menu ourselves

    def bindEventHandlers(self):
        # pylint: disable=W0142,W0612,W0201
        for args in [
            (wx.EVT_TIMER, self.onFind, self.__timer),
            (wx.EVT_TEXT_ENTER, self.onFind),
            (wx.EVT_TEXT, self.onFindLater),
            (wx.EVT_SEARCHCTRL_CANCEL_BTN, self.onCancel),
            (wx.EVT_SEARCHCTRL_SEARCH_BTN, self._onSearchButton),
            (wx.EVT_MENU, self.onMatchCaseMenuItem, self.__matchCaseMenuItem),
            (
                wx.EVT_MENU,
                self.onIncludeSubItemsMenuItem,
                self.__includeSubItemsMenuItem,
            ),
            (
                wx.EVT_MENU,
                self.onSearchDescriptionMenuItem,
                self.__searchDescriptionMenuItem,
            ),
            (
                wx.EVT_MENU,
                self.onRegularExpressionMenuItem,
                self.__regularExpressionMenuItem,
            ),
        ]:
            self.Bind(*args)
        # Precreate menu item ids for the recent searches and bind the event
        # handler for those menu item ids. It's no problem that the actual menu
        # items don't exist yet.
        self.__recentSearchMenuItemIds = [
            wx.NewId() for dummy in range(self.__maxRecentSearches)
        ]
        self.Bind(
            wx.EVT_MENU_RANGE,
            self.onRecentSearchMenuItem,
            id=self.__recentSearchMenuItemIds[0],
            id2=self.__recentSearchMenuItemIds[-1],
        )
        # Stop timer on window destruction to prevent crashes
        self.Bind(wx.EVT_WINDOW_DESTROY, self._onDestroy)

    def setMatchCase(self, matchCase):
        self.__matchCase = matchCase
        self.__matchCaseMenuItem.Check(matchCase)

    def setIncludeSubItems(self, includeSubItems):
        self.__includeSubItems = includeSubItems
        self.__includeSubItemsMenuItem.Check(includeSubItems)

    def setSearchDescription(self, searchDescription):
        self.__searchDescription = searchDescription
        self.__searchDescriptionMenuItem.Check(searchDescription)

    def setRegularExpression(self, regularExpression):
        self.__regularExpression = regularExpression
        self.__regularExpressionMenuItem.Check(regularExpression)

    def isValid(self):
        if self.__regularExpression:
            try:
                re.compile(self.GetValue())
            except sre_constants.error:
                return False
        return True

    def _onDestroy(self, event):
        """
        Automatically cleanup timer on window destruction.

        This is a critical safety mechanism to prevent timer-after-destruction crashes.
        The EVT_WINDOW_DESTROY binding ensures cleanup happens automatically,
        following wxPython best practices for timer lifecycle management.
        """
        if event.GetEventObject() == self:
            self.cleanup()
        event.Skip()

    def cleanup(self):
        """
        Stop the timer and clear callback to prevent crashes.

        Best practices implemented:
        1. Stop any running timer to prevent fire-after-destruction
        2. Replace callback with no-op to safely handle late events
        3. Can be called multiple times safely (idempotent)

        This prevents the NULL pointer crashes documented in PYTHON3_MIGRATION_NOTES.md
        """
        if self.__timer and self.__timer.IsRunning():
            self.__timer.Stop()
        # Replace callback with no-op lambda to safely handle any late timer events
        self.__callback = lambda *args, **kwargs: None

    def onFindLater(self, event):  # pylint: disable=W0613
        """
        Debounce search operations using a timer.

        This implements the "search-as-you-type" debouncing pattern:
        - Restarts the timer on each keystroke
        - Only triggers the actual search after user stops typing
        - Prevents expensive filtering operations on every character

        This is a best practice for search UX, used by Google, VS Code, etc.
        """
        self.__timer.Start(self.__debounceDelay, oneShot=True)

    def onFind(self, event):  # pylint: disable=W0613
        """
        Execute the actual search operation.

        This is called either:
        - After the debounce timer expires (user stopped typing)
        - Immediately when user presses Enter (EVT_TEXT_ENTER)
        - Immediately when changing search options via menu

        Best practice: Stop any pending timer to prevent double execution.
        """
        # Cancel any pending debounced search
        if self.__timer.IsRunning():
            self.__timer.Stop()
        if not self.IsEnabled():
            return
        if not self.isValid():
            self.__tooltip.SetData(
                [
                    (
                        None,
                        [
                            _("This is an invalid regular expression."),
                            _("Defaulting to substring search."),
                        ],
                    )
                ]
            )
            x, y = self.GetParent().ClientToScreen(*self.GetPosition())
            height = self.GetClientSize()[1]
            self.DoShowTip(x + 3, y + height + 4, self.__tooltip)
        else:
            self.HideTip()
        searchString = self.GetValue()
        if searchString:
            self.rememberSearchString(searchString)
        self.ShowCancelButton(bool(searchString))
        self.__callback(
            searchString,
            self.__matchCase,
            self.__includeSubItems,
            self.__searchDescription,
            self.__regularExpression,
        )

    def onCancel(self, event):
        """
        Handle search cancellation (clear button clicked).

        Best practice: Clear search immediately without debouncing.
        Users expect instant feedback when clearing a search.
        """
        self.SetValue("")
        self.onFind(event)  # Immediate search, not debounced
        event.Skip()

    def onMatchCaseMenuItem(self, event):
        self.__matchCase = self._isMenuItemChecked(event)
        self.onFind(event)
        # XXXFIXME: when skipping on OS X, we receive several events with different
        # IsChecked(), the last one being False. I can't reproduce this in a unit
        # test. Not skipping the event doesn't harm on other platforms (tested by
        # hand)

    def onIncludeSubItemsMenuItem(self, event):
        self.__includeSubItems = self._isMenuItemChecked(event)
        self.onFind(event)

    def onSearchDescriptionMenuItem(self, event):
        self.__searchDescription = self._isMenuItemChecked(event)
        self.onFind(event)

    def onRegularExpressionMenuItem(self, event):
        self.__regularExpression = self._isMenuItemChecked(event)
        self.onFind(event)

    def onRecentSearchMenuItem(self, event):
        self.SetValue(
            self.__recentSearches[
                event.GetId() - self.__recentSearchMenuItemIds[0]
            ]
        )
        self.onFind(event)
        # Don't call event.Skip(). It will result in this event handler being
        # called again with the next menu item since wxPython thinks the
        # event has not been dealt with (on Mac OS X at least).

    def rememberSearchString(self, searchString):
        if searchString in self.__recentSearches:
            self.__recentSearches.remove(searchString)
        self.__recentSearches.insert(0, searchString)
        if len(self.__recentSearches) > self.__maxRecentSearches:
            self.__recentSearches.pop()
        self.updateRecentSearches()

    def updateRecentSearches(self):
        menu = self.GetMenu()
        self.removeRecentSearches(menu)
        self.addRecentSearches(menu)

    def removeRecentSearches(self, menu):
        while menu.GetMenuItemCount() > 4:
            item = menu.FindItemByPosition(4)
            menu.DestroyItem(item)

    def addRecentSearches(self, menu):
        menu.AppendSeparator()
        item = menu.Append(wx.ID_ANY, _("Recent searches"))
        item.Enable(False)
        for index, searchString in enumerate(self.__recentSearches):
            menu.Append(self.__recentSearchMenuItemIds[index], searchString)

    def Enable(self, enable=True):  # pylint: disable=W0221
        """When wx.SearchCtrl is disabled it doesn't grey out the buttons,
        so we remove those."""
        self.SetValue("" if enable else _("Viewer not searchable"))
        super().Enable(enable)
        self.ShowCancelButton(enable and bool(self.GetValue()))
        self.ShowSearchButton(enable)

    def _isMenuItemChecked(self, event):
        # There's a bug in wxPython 2.8.3 on Windows XP that causes
        # event.IsChecked() to return the wrong value in the context menu.
        # The menu on the main window works fine. So we first try to access the
        # context menu to get the checked state from the menu item itself.
        # This will fail if the event is coming from the window, but in that
        # case we can event.IsChecked() expect to work so we use that.
        try:
            return (
                event.GetEventObject().FindItemById(event.GetId()).IsChecked()
            )
        except AttributeError:
            return event.IsChecked()

    def OnBeforeShowToolTip(self, x, y):
        return None


class SearchCtrl(wx.Panel):
    """SearchCtrl wrapped in a tight-fitting Panel for Wayland popup compatibility.

    On Wayland, popup menus must be positioned relative to their transient parent.
    By wrapping the SearchCtrl in a tight-fitting Panel and calling PopupMenu
    on the Panel, we establish the correct transient parent relationship.

    This is the modern best practice for Wayland-compatible popup positioning.
    See bugs/ISSUE_159_SEARCH_DROPDOWN_POSITION.md for details.
    """

    def __init__(self, parent, *args, **kwargs):
        # Extract style before passing to Panel (style is for inner SearchCtrl)
        style = kwargs.pop("style", 0)
        super().__init__(parent)

        # Create sizer and inner search control
        # Use proportion=0 to prevent Panel from stretching beyond SearchCtrl size
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.__searchCtrl = _SearchCtrlInner(self, self, *args, style=style, **kwargs)
        sizer.Add(self.__searchCtrl, 0, wx.EXPAND)
        self.SetSizer(sizer)
        # Fit Panel tightly to SearchCtrl for correct popup positioning
        self.Fit()

    # Delegate methods to inner search control
    def GetMainWindow(self):
        return self.__searchCtrl.GetMainWindow()

    def getTextCtrl(self):
        return self.__searchCtrl.getTextCtrl()

    def cleanup(self):
        return self.__searchCtrl.cleanup()

    def setMatchCase(self, matchCase):
        return self.__searchCtrl.setMatchCase(matchCase)

    def setIncludeSubItems(self, includeSubItems):
        return self.__searchCtrl.setIncludeSubItems(includeSubItems)

    def setSearchDescription(self, searchDescription):
        return self.__searchCtrl.setSearchDescription(searchDescription)

    def setRegularExpression(self, regularExpression):
        return self.__searchCtrl.setRegularExpression(regularExpression)

    def GetValue(self):
        return self.__searchCtrl.GetValue()

    def SetValue(self, value):
        return self.__searchCtrl.SetValue(value)

    def SetFocus(self):
        return self.__searchCtrl.SetFocus()

    def Enable(self, enable=True):
        super().Enable(enable)
        return self.__searchCtrl.Enable(enable)

    def IsEnabled(self):
        return self.__searchCtrl.IsEnabled()

    def SetMinSize(self, size):
        self.__searchCtrl.SetMinSize(size)
        super().SetMinSize(size)

    def GetMenu(self):
        return self.__searchCtrl.GetMenu()

    def PopupMenu(self):
        """Show the search options menu with Wayland-compatible positioning.

        This is called when user presses Ctrl-Down to drop down the menu.
        Delegates to inner control's _onSearchButton which handles positioning.
        """
        self.__searchCtrl._onSearchButton(None)
