/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/

using System;
using System.Collections.Generic;
using System.Linq;
using QuantConnect.Data;
using QuantConnect.Data.Market;
using QuantConnect.Interfaces;
using QuantConnect.Securities;

namespace QuantConnect.Algorithm.CSharp
{
    /// <summary>
    /// Regression algorithm illustrating the usage of the <see cref="QCAlgorithm.OptionChains(IEnumerable{Symbol})"/> method
    /// to get multiple option chains, which contains additional data besides the symbols, including prices, implied volatility and greeks.
    /// It also shows how this data can be used to filter the contracts based on certain criteria.
    /// </summary>
    public class OptionChainsMultipleFullDataRegressionAlgorithm : QCAlgorithm, IRegressionAlgorithmDefinition
    {
        private Symbol _googOptionContract;
        private Symbol _spxOptionContract;

        public override void Initialize()
        {
            SetStartDate(2015, 12, 24);
            SetEndDate(2015, 12, 24);
            SetCash(100000);

            var goog = AddEquity("GOOG").Symbol;
            var spx = AddIndex("SPX").Symbol;

            var chains = OptionChains(new[] { goog, spx });

            _googOptionContract = GetContract(chains, goog, TimeSpan.FromDays(10));
            _spxOptionContract = GetContract(chains, spx, TimeSpan.FromDays(60));

            AddOptionContract(_googOptionContract);
            AddIndexOptionContract(_spxOptionContract);
        }

        private Symbol GetContract(OptionChains chains, Symbol underlying, TimeSpan expirySpan)
        {
            return chains
                .Where(kvp => kvp.Key.Underlying == underlying)
                .Select(kvp => kvp.Value)
                .Single()
                // Get contracts expiring within a given span, with an implied volatility greater than 0.5 and a delta less than 0.5
                .Where(contractData => contractData.ID.Date - Time <= expirySpan &&
                    contractData.ImpliedVolatility > 0.5m &&
                    contractData.Greeks.Delta < 0.5m)
                // Get the contract with the latest expiration date
                .OrderByDescending(x => x.ID.Date)
                .First();
        }

        public override void OnData(Slice slice)
        {
            // Do some trading with the selected contract for sample purposes
            if (!Portfolio.Invested)
            {
                MarketOrder(_googOptionContract, 1);
            }
            else
            {
                Liquidate();
            }
        }

        /// <summary>
        /// This is used by the regression test system to indicate if the open source Lean repository has the required data to run this algorithm.
        /// </summary>
        public bool CanRunLocally { get; } = true;

        /// <summary>
        /// This is used by the regression test system to indicate which languages this algorithm is written in.
        /// </summary>
        public virtual List<Language> Languages { get; } = new() { Language.CSharp, Language.Python };

        /// <summary>
        /// Data Points count of all timeslices of algorithm
        /// </summary>
        public long DataPoints => 1059;

        /// <summary>
        /// Data Points count of the algorithm history
        /// </summary>
        public int AlgorithmHistoryDataPoints => 2;

        /// <summary>
        /// Final status of the algorithm
        /// </summary>
        public AlgorithmStatus AlgorithmStatus => AlgorithmStatus.Completed;

        /// <summary>
        /// This is used by the regression test system to indicate what the expected statistics are from running the algorithm
        /// </summary>
        public Dictionary<string, string> ExpectedStatistics => new Dictionary<string, string>
        {
            {"Total Orders", "210"},
            {"Average Win", "0%"},
            {"Average Loss", "0%"},
            {"Compounding Annual Return", "0%"},
            {"Drawdown", "0%"},
            {"Expectancy", "0"},
            {"Start Equity", "100000"},
            {"End Equity", "96041"},
            {"Net Profit", "0%"},
            {"Sharpe Ratio", "0"},
            {"Sortino Ratio", "0"},
            {"Probabilistic Sharpe Ratio", "0%"},
            {"Loss Rate", "0%"},
            {"Win Rate", "0%"},
            {"Profit-Loss Ratio", "0"},
            {"Alpha", "0"},
            {"Beta", "0"},
            {"Annual Standard Deviation", "0"},
            {"Annual Variance", "0"},
            {"Information Ratio", "0"},
            {"Tracking Error", "0"},
            {"Treynor Ratio", "0"},
            {"Total Fees", "$209.00"},
            {"Estimated Strategy Capacity", "$0"},
            {"Lowest Capacity Asset", "GOOCV W6U7PD1F77AE|GOOCV VP83T1ZUHROL"},
            {"Portfolio Turnover", "85.46%"},
            {"Drawdown Recovery", "0"},
            {"OrderListHash", "8ee1c7a1574ae0ad6f231ad0b7d15310"}
        };
    }
}
