﻿/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

using QuantConnect.Interfaces;

namespace QuantConnect.Orders
{
    /// <summary>
    /// Contains additional properties and settings for an order submitted to Binance brokerage
    /// </summary>
    public class dYdXOrderProperties : OrderProperties
    {
        /// <summary>
        /// This flag will ensure the order executes only as a maker (no fee) order.
        /// If part of the order results in taking liquidity rather than providing,
        /// it will be rejected and no part of the order will execute.
        /// Note: this flag is only applied to Limit orders.
        /// </summary>
        public bool PostOnly
        {
            get;
            set
            {
                if (value && IOC)
                {
                    throw new System.InvalidOperationException("Cannot set PostOnly when IOC is already set. Only one execution type can be active at a time.");
                }
                field = value;
            }
        }

        /// <summary>
        /// Enforces that an order only be placed
        /// on the book as a maker order. Note this means that validators will cancel
        /// any newly placed post only orders that would cross with other maker
        /// orders.
        /// </summary>
        public bool IOC
        {
            get;
            set
            {
                if (value && PostOnly)
                {
                    throw new System.InvalidOperationException("Cannot set IOC when PostOnly is already set. Only one execution type can be active at a time.");
                }
                field = value;
            }
        }

        /// <summary>
        /// The maximum amount of gas to use for the order.
        /// </summary>
        public ulong GasLimit { get; set; } = 1_000_000;

        /// <summary>
        /// If you send a reduce-only order, it will only trade if it decreases your position size.
        /// </summary>
        public bool ReduceOnly { get; set; }

        /// <summary>
        /// The block height at which the order expires.
        /// </summary>
        public uint GoodTilBlockOffset { get; set; } = 20;

        /// <summary>
        /// Returns a new instance clone of this object
        /// </summary>
        public override IOrderProperties Clone()
        {
            return (dYdXOrderProperties)MemberwiseClone();
        }
    }
}
