/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/

using Newtonsoft.Json;
using Newtonsoft.Json.Converters;

namespace QuantConnect.Packets
{
    /// <summary>
    /// Base class for packet messaging system
    /// </summary>
    public class Packet
    {
        /// <summary>
        /// Packet type defined by a string enum
        /// </summary>
        public PacketType Type { get; set; } = PacketType.None;

        /// <summary>
        /// User unique specific channel endpoint to send the packets
        /// </summary>
        public virtual string Channel { get; set; } = "";

        /// <summary>
        /// Initialize the base class and setup the packet type.
        /// </summary>
        /// <param name="type">PacketType for the class.</param>
        public Packet(PacketType type)
        {
            Channel = "";
            Type = type;
        }
    }

    /// <summary>
    /// Classifications of internal packet system
    /// </summary>
    [JsonConverter(typeof(StringEnumConverter))]
    public enum PacketType
    {
        /// <summary>
        /// Default, unset:
        /// </summary>
        None,

        /// <summary>
        /// Base type for backtest and live work
        /// </summary>
        AlgorithmNode,

        /// <summary>
        /// Autocomplete Work Packet
        /// </summary>
        AutocompleteWork,

        /// <summary>
        /// Result of the Autocomplete Job:
        /// </summary>
        AutocompleteResult,

        /// <summary>
        /// Controller->Backtest Node Packet:
        /// </summary>
        BacktestNode,

        /// <summary>
        /// Packet out of backtest node:
        /// </summary>
        BacktestResult,

        /// <summary>
        /// API-> Controller Work Packet:
        /// </summary>
        BacktestWork,

        /// <summary>
        /// Controller -> Live Node Packet:
        /// </summary>
        LiveNode,

        /// <summary>
        /// Live Node -> User Packet:
        /// </summary>
        LiveResult,

        /// <summary>
        /// API -> Controller Packet:
        /// </summary>
        LiveWork,

        /// <summary>
        /// Node -> User Algo Security Types
        /// </summary>
        SecurityTypes,

        /// <summary>
        /// Controller -> User Error in Backtest Settings:
        /// </summary>
        BacktestError,

        /// <summary>
        /// Nodes -> User Algorithm Status Packet:
        /// </summary>
        AlgorithmStatus,

        /// <summary>
        /// API -> Compiler Work Packet:
        /// </summary>
        BuildWork,

        /// <summary>
        /// Compiler -> User Build Success
        /// </summary>
        BuildSuccess,

        /// <summary>
        /// Compiler -> User, Compile Error
        /// </summary>
        BuildError,

        /// <summary>
        /// Node -> User Algorithm Runtime Error
        /// </summary>
        RuntimeError,

        /// <summary>
        /// Error is an internal handled error packet inside users algorithm
        /// </summary>
        HandledError,

        /// <summary>
        /// Nodes -> User Log Message
        /// </summary>
        Log,

        /// <summary>
        /// Nodes -> User Debug Message
        /// </summary>
        Debug,

        /// <summary>
        /// Nodes -> User, Order Update Event
        /// </summary>
        OrderEvent,

        /// <summary>
        /// Boolean true/false success
        /// </summary>
        Success,

        /// <summary>
        /// History live job packets
        /// </summary>
        History,

        /// <summary>
        /// Result from a command
        /// </summary>
        CommandResult,

        /// <summary>
        /// Hook from git hub
        /// </summary>
        GitHubHook,

        /// <summary>
        /// Documentation result from docs server
        /// </summary>
        DocumentationResult,

        /// <summary>
        /// Documentation request to the docs server
        /// </summary>
        Documentation,

        /// <summary>
        /// Debug packet generated by Lean
        /// </summary>
        SystemDebug,

        /// <summary>
        /// Packet containing insights generated by the algorithm
        /// </summary>
        AlphaResult,

        /// <summary>
        /// Alpha API -> Controller packet
        /// </summary>
        AlphaWork,

        /// <summary>
        /// Alpha Controller -> Alpha Node packet
        /// </summary>
        AlphaNode,

        /// <summary>
        /// Packet containing list of algorithms to run as a regression test
        /// </summary>
        RegressionAlgorithm,

        /// <summary>
        /// Packet containing a heartbeat
        /// </summary>
        AlphaHeartbeat,

        /// <summary>
        /// Used when debugging to send status updates
        /// </summary>
        DebuggingStatus,

        /// <summary>
        /// Optimization Node Packet:
        /// </summary>
        OptimizationNode,

        /// <summary>
        /// Optimization Estimate Packet:
        /// </summary>
        OptimizationEstimate,

        /// <summary>
        /// Optimization work status update
        /// </summary>
        OptimizationStatus,

        /// <summary>
        /// Optimization work result
        /// </summary>
        OptimizationResult,

        /// <summary>
        /// Aggregated packets
        /// </summary>
        Aggregated,

        /// <summary>
        /// Query the language model
        /// </summary>
        LanguageModelQuery,

        /// <summary>
        /// Send feedback to a language model response
        /// </summary>
        LanguageModelFeedback,

        /// <summary>
        /// The language models response
        /// </summary>
        LanguageModelResponse,

        /// <summary>
        /// Language model code analysis
        /// </summary>
        LanguageModelCodeAnalysis,

        /// <summary>
        /// Language model chat work
        /// </summary>
        LanguageModelChatWork,

        /// <summary>
        /// Language model chat response
        /// </summary>
        LanguageModelChatResponse,

        /// <summary>
        /// Algorithm name update
        /// </summary>
        AlgorithmNameUpdate,

        /// <summary>
        /// Algorithm tags update
        /// </summary>
        AlgorithmTagsUpdate,

        /// <summary>
        /// Research job packet
        /// </summary>
        ResearchNode,

        /// <summary>
        /// Organization update
        /// </summary>
        OrganizationUpdate,

        /// <summary>
        /// Compiler -> User Build Warnings
        /// </summary>
        BuildWarning,

        /// <summary>
        /// Language model function call related packet
        /// </summary>
        LanguageModelFunctionCall,

        /// <summary>
        /// Language model agent message
        /// </summary>
        LanguageModelAgentMessage,

        /// <summary>
        /// Agent job packet
        /// </summary>
        AgentNode
    }
}
