/* ----------------------------------------------------------------------
 * Project:      CMSIS DSP Library
 * Title:        arm_dct4_init_f32.c
 * Description:  Initialization function of DCT-4 & IDCT4 F32
 *
 * $Date:        27. January 2017
 * $Revision:    V.1.5.1
 *
 * Target Processor: Cortex-M cores
 * -------------------------------------------------------------------- */
/*
 * Copyright (C) 2010-2017 ARM Limited or its affiliates. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "arm_math.h"

/**
 * @ingroup DCT4_IDCT4
 */

/**
 * @addtogroup DCT4_IDCT4_Table DCT Type IV Tables
 * @{
 */

/*
* @brief  Weights Table
*/

/**
 * \par
 * Weights tables are generated using the formula : <pre>weights[n] = e^(-j*n*pi/(2*N))</pre>
 * \par
 * C command to generate the table
 * <pre>
 * for(i = 0; i< N; i++)
 * {
 *    weights[2*i]= cos(i*c);
 *    weights[(2*i)+1]= -sin(i * c);
 * } </pre>
 * \par
 * Where <code>N</code> is the Number of weights to be calculated and <code>c</code> is <code>pi/(2*N)</code>
 * \par
 * In the tables below the real and imaginary values are placed alternatively, hence the
 * array length is <code>2*N</code>.
 */

static const float32_t Weights_128[256] = {
	1.000000000000000000f, 0.000000000000000000f, 0.999924701839144500f,
	-0.012271538285719925f,
	0.999698818696204250f, -0.024541228522912288f, 0.999322384588349540f,
	-0.036807222941358832f,
	0.998795456205172410f, -0.049067674327418015f, 0.998118112900149180f,
	-0.061320736302208578f,
	0.997290456678690210f, -0.073564563599667426f, 0.996312612182778000f,
	-0.085797312344439894f,
	0.995184726672196930f, -0.098017140329560604f, 0.993906970002356060f,
	-0.110222207293883060f,
	0.992479534598709970f, -0.122410675199216200f, 0.990902635427780010f,
	-0.134580708507126170f,
	0.989176509964781010f, -0.146730474455361750f, 0.987301418157858430f,
	-0.158858143333861450f,
	0.985277642388941220f, -0.170961888760301220f, 0.983105487431216290f,
	-0.183039887955140950f,
	0.980785280403230430f, -0.195090322016128250f, 0.978317370719627650f,
	-0.207111376192218560f,
	0.975702130038528570f, -0.219101240156869800f, 0.972939952205560180f,
	-0.231058108280671110f,
	0.970031253194543970f, -0.242980179903263870f, 0.966976471044852070f,
	-0.254865659604514570f,
	0.963776065795439840f, -0.266712757474898370f, 0.960430519415565790f,
	-0.278519689385053060f,
	0.956940335732208820f, -0.290284677254462330f, 0.953306040354193860f,
	-0.302005949319228080f,
	0.949528180593036670f, -0.313681740398891520f, 0.945607325380521280f,
	-0.325310292162262930f,
	0.941544065183020810f, -0.336889853392220050f, 0.937339011912574960f,
	-0.348418680249434560f,
	0.932992798834738960f, -0.359895036534988110f, 0.928506080473215590f,
	-0.371317193951837540f,
	0.923879532511286740f, -0.382683432365089780f, 0.919113851690057770f,
	-0.393992040061048100f,
	0.914209755703530690f, -0.405241314004989860f, 0.909167983090522380f,
	-0.416429560097637150f,
	0.903989293123443340f, -0.427555093430282080f, 0.898674465693953820f,
	-0.438616238538527660f,
	0.893224301195515320f, -0.449611329654606540f, 0.887639620402853930f,
	-0.460538710958240010f,
	0.881921264348355050f, -0.471396736825997640f, 0.876070094195406600f,
	-0.482183772079122720f,
	0.870086991108711460f, -0.492898192229784040f, 0.863972856121586810f,
	-0.503538383725717580f,
	0.857728610000272120f, -0.514102744193221660f, 0.851355193105265200f,
	-0.524589682678468950f,
	0.844853565249707120f, -0.534997619887097150f, 0.838224705554838080f,
	-0.545324988422046460f,
	0.831469612302545240f, -0.555570233019602180f, 0.824589302785025290f,
	-0.565731810783613120f,
	0.817584813151583710f, -0.575808191417845340f, 0.810457198252594770f,
	-0.585797857456438860f,
	0.803207531480644940f, -0.595699304492433360f, 0.795836904608883570f,
	-0.605511041404325550f,
	0.788346427626606340f, -0.615231590580626820f, 0.780737228572094490f,
	-0.624859488142386340f,
	0.773010453362736990f, -0.634393284163645490f, 0.765167265622458960f,
	-0.643831542889791390f,
	0.757208846506484570f, -0.653172842953776760f, 0.749136394523459370f,
	-0.662415777590171780f,
	0.740951125354959110f, -0.671558954847018330f, 0.732654271672412820f,
	-0.680600997795453020f,
	0.724247082951467000f, -0.689540544737066830f, 0.715730825283818590f,
	-0.698376249408972920f,
	0.707106781186547570f, -0.707106781186547460f, 0.698376249408972920f,
	-0.715730825283818590f,
	0.689540544737066940f, -0.724247082951466890f, 0.680600997795453130f,
	-0.732654271672412820f,
	0.671558954847018330f, -0.740951125354959110f, 0.662415777590171780f,
	-0.749136394523459260f,
	0.653172842953776760f, -0.757208846506484460f, 0.643831542889791500f,
	-0.765167265622458960f,
	0.634393284163645490f, -0.773010453362736990f, 0.624859488142386450f,
	-0.780737228572094380f,
	0.615231590580626820f, -0.788346427626606230f, 0.605511041404325550f,
	-0.795836904608883460f,
	0.595699304492433470f, -0.803207531480644830f, 0.585797857456438860f,
	-0.810457198252594770f,
	0.575808191417845340f, -0.817584813151583710f, 0.565731810783613230f,
	-0.824589302785025290f,
	0.555570233019602290f, -0.831469612302545240f, 0.545324988422046460f,
	-0.838224705554837970f,
	0.534997619887097260f, -0.844853565249707010f, 0.524589682678468840f,
	-0.851355193105265200f,
	0.514102744193221660f, -0.857728610000272120f, 0.503538383725717580f,
	-0.863972856121586700f,
	0.492898192229784090f, -0.870086991108711350f, 0.482183772079122830f,
	-0.876070094195406600f,
	0.471396736825997810f, -0.881921264348354940f, 0.460538710958240010f,
	-0.887639620402853930f,
	0.449611329654606600f, -0.893224301195515320f, 0.438616238538527710f,
	-0.898674465693953820f,
	0.427555093430282200f, -0.903989293123443340f, 0.416429560097637320f,
	-0.909167983090522270f,
	0.405241314004989860f, -0.914209755703530690f, 0.393992040061048100f,
	-0.919113851690057770f,
	0.382683432365089840f, -0.923879532511286740f, 0.371317193951837600f,
	-0.928506080473215480f,
	0.359895036534988280f, -0.932992798834738850f, 0.348418680249434510f,
	-0.937339011912574960f,
	0.336889853392220050f, -0.941544065183020810f, 0.325310292162262980f,
	-0.945607325380521280f,
	0.313681740398891570f, -0.949528180593036670f, 0.302005949319228200f,
	-0.953306040354193750f,
	0.290284677254462330f, -0.956940335732208940f, 0.278519689385053060f,
	-0.960430519415565790f,
	0.266712757474898420f, -0.963776065795439840f, 0.254865659604514630f,
	-0.966976471044852070f,
	0.242980179903263980f, -0.970031253194543970f, 0.231058108280671280f,
	-0.972939952205560070f,
	0.219101240156869770f, -0.975702130038528570f, 0.207111376192218560f,
	-0.978317370719627650f,
	0.195090322016128330f, -0.980785280403230430f, 0.183039887955141060f,
	-0.983105487431216290f,
	0.170961888760301360f, -0.985277642388941220f, 0.158858143333861390f,
	-0.987301418157858430f,
	0.146730474455361750f, -0.989176509964781010f, 0.134580708507126220f,
	-0.990902635427780010f,
	0.122410675199216280f, -0.992479534598709970f, 0.110222207293883180f,
	-0.993906970002356060f,
	0.098017140329560770f, -0.995184726672196820f, 0.085797312344439880f,
	-0.996312612182778000f,
	0.073564563599667454f, -0.997290456678690210f, 0.061320736302208648f,
	-0.998118112900149180f,
	0.049067674327418126f, -0.998795456205172410f, 0.036807222941358991f,
	-0.999322384588349540f,
	0.024541228522912264f, -0.999698818696204250f, 0.012271538285719944f,
	-0.999924701839144500f
};

static const float32_t Weights_512[1024] = {
	1.000000000000000000f, 0.000000000000000000f, 0.999995293809576190f,
	-0.003067956762965976f,
	0.999981175282601110f, -0.006135884649154475f, 0.999957644551963900f,
	-0.009203754782059819f,
	0.999924701839144500f, -0.012271538285719925f, 0.999882347454212560f,
	-0.015339206284988100f,
	0.999830581795823400f, -0.018406729905804820f, 0.999769405351215280f,
	-0.021474080275469508f,
	0.999698818696204250f, -0.024541228522912288f, 0.999618822495178640f,
	-0.027608145778965740f,
	0.999529417501093140f, -0.030674803176636626f, 0.999430604555461730f,
	-0.033741171851377580f,
	0.999322384588349540f, -0.036807222941358832f, 0.999204758618363890f,
	-0.039872927587739811f,
	0.999077727752645360f, -0.042938256934940820f, 0.998941293186856870f,
	-0.046003182130914623f,
	0.998795456205172410f, -0.049067674327418015f, 0.998640218180265270f,
	-0.052131704680283324f,
	0.998475580573294770f, -0.055195244349689934f, 0.998301544933892890f,
	-0.058258264500435752f,
	0.998118112900149180f, -0.061320736302208578f, 0.997925286198596000f,
	-0.064382630929857465f,
	0.997723066644191640f, -0.067443919563664051f, 0.997511456140303450f,
	-0.070504573389613856f,
	0.997290456678690210f, -0.073564563599667426f, 0.997060070339482960f,
	-0.076623861392031492f,
	0.996820299291165670f, -0.079682437971430126f, 0.996571145790554840f,
	-0.082740264549375692f,
	0.996312612182778000f, -0.085797312344439894f, 0.996044700901251970f,
	-0.088853552582524600f,
	0.995767414467659820f, -0.091908956497132724f, 0.995480755491926940f,
	-0.094963495329638992f,
	0.995184726672196930f, -0.098017140329560604f, 0.994879330794805620f,
	-0.101069862754827820f,
	0.994564570734255420f, -0.104121633872054590f, 0.994240449453187900f,
	-0.107172424956808840f,
	0.993906970002356060f, -0.110222207293883060f, 0.993564135520595300f,
	-0.113270952177564350f,
	0.993211949234794500f, -0.116318630911904750f, 0.992850414459865100f,
	-0.119365214810991350f,
	0.992479534598709970f, -0.122410675199216200f, 0.992099313142191800f,
	-0.125454983411546230f,
	0.991709753669099530f, -0.128498110793793170f, 0.991310859846115440f,
	-0.131540028702883120f,
	0.990902635427780010f, -0.134580708507126170f, 0.990485084256457090f,
	-0.137620121586486040f,
	0.990058210262297120f, -0.140658239332849210f, 0.989622017463200890f,
	-0.143695033150294470f,
	0.989176509964781010f, -0.146730474455361750f, 0.988721691960323780f,
	-0.149764534677321510f,
	0.988257567730749460f, -0.152797185258443440f, 0.987784141644572180f,
	-0.155828397654265230f,
	0.987301418157858430f, -0.158858143333861450f, 0.986809401814185530f,
	-0.161886393780111830f,
	0.986308097244598670f, -0.164913120489969890f, 0.985797509167567480f,
	-0.167938294974731170f,
	0.985277642388941220f, -0.170961888760301220f, 0.984748501801904210f,
	-0.173983873387463820f,
	0.984210092386929030f, -0.177004220412148750f, 0.983662419211730250f,
	-0.180022901405699510f,
	0.983105487431216290f, -0.183039887955140950f, 0.982539302287441240f,
	-0.186055151663446630f,
	0.981963869109555240f, -0.189068664149806190f, 0.981379193313754560f,
	-0.192080397049892440f,
	0.980785280403230430f, -0.195090322016128250f, 0.980182135968117430f,
	-0.198098410717953560f,
	0.979569765685440520f, -0.201104634842091900f, 0.978948175319062200f,
	-0.204108966092816870f,
	0.978317370719627650f, -0.207111376192218560f, 0.977677357824509930f,
	-0.210111836880469610f,
	0.977028142657754390f, -0.213110319916091360f, 0.976369731330021140f,
	-0.216106797076219520f,
	0.975702130038528570f, -0.219101240156869800f, 0.975025345066994120f,
	-0.222093620973203510f,
	0.974339382785575860f, -0.225083911359792830f, 0.973644249650811980f,
	-0.228072083170885730f,
	0.972939952205560180f, -0.231058108280671110f, 0.972226497078936270f,
	-0.234041958583543430f,
	0.971503890986251780f, -0.237023605994367200f, 0.970772140728950350f,
	-0.240003022448741500f,
	0.970031253194543970f, -0.242980179903263870f, 0.969281235356548530f,
	-0.245955050335794590f,
	0.968522094274417380f, -0.248927605745720150f, 0.967753837093475510f,
	-0.251897818154216970f,
	0.966976471044852070f, -0.254865659604514570f, 0.966190003445412500f,
	-0.257831102162158990f,
	0.965394441697689400f, -0.260794117915275510f, 0.964589793289812760f,
	-0.263754678974831350f,
	0.963776065795439840f, -0.266712757474898370f, 0.962953266873683880f,
	-0.269668325572915090f,
	0.962121404269041580f, -0.272621355449948980f, 0.961280485811320640f,
	-0.275571819310958140f,
	0.960430519415565790f, -0.278519689385053060f, 0.959571513081984520f,
	-0.281464937925757940f,
	0.958703474895871600f, -0.284407537211271880f, 0.957826413027532910f,
	-0.287347459544729510f,
	0.956940335732208820f, -0.290284677254462330f, 0.956045251349996410f,
	-0.293219162694258630f,
	0.955141168305770780f, -0.296150888243623790f, 0.954228095109105670f,
	-0.299079826308040480f,
	0.953306040354193860f, -0.302005949319228080f, 0.952375012719765880f,
	-0.304929229735402370f,
	0.951435020969008340f, -0.307849640041534870f, 0.950486073949481700f,
	-0.310767152749611470f,
	0.949528180593036670f, -0.313681740398891520f, 0.948561349915730270f,
	-0.316593375556165850f,
	0.947585591017741090f, -0.319502030816015690f, 0.946600913083283530f,
	-0.322407678801069850f,
	0.945607325380521280f, -0.325310292162262930f, 0.944604837261480260f,
	-0.328209843579092500f,
	0.943593458161960390f, -0.331106305759876430f, 0.942573197601446870f,
	-0.333999651442009380f,
	0.941544065183020810f, -0.336889853392220050f, 0.940506070593268300f,
	-0.339776884406826850f,
	0.939459223602189920f, -0.342660717311994380f, 0.938403534063108060f,
	-0.345541324963989090f,
	0.937339011912574960f, -0.348418680249434560f, 0.936265667170278260f,
	-0.351292756085567090f,
	0.935183509938947610f, -0.354163525420490340f, 0.934092550404258980f,
	-0.357030961233429980f,
	0.932992798834738960f, -0.359895036534988110f, 0.931884265581668150f,
	-0.362755724367397230f,
	0.930766961078983710f, -0.365612997804773850f, 0.929640895843181330f,
	-0.368466829953372320f,
	0.928506080473215590f, -0.371317193951837540f, 0.927362525650401110f,
	-0.374164062971457930f,
	0.926210242138311380f, -0.377007410216418260f, 0.925049240782677580f,
	-0.379847208924051160f,
	0.923879532511286740f, -0.382683432365089780f, 0.922701128333878630f,
	-0.385516053843918850f,
	0.921514039342042010f, -0.388345046698826250f, 0.920318276709110590f,
	-0.391170384302253870f,
	0.919113851690057770f, -0.393992040061048100f, 0.917900775621390500f,
	-0.396809987416710310f,
	0.916679059921042700f, -0.399624199845646790f, 0.915448716088267830f,
	-0.402434650859418430f,
	0.914209755703530690f, -0.405241314004989860f, 0.912962190428398210f,
	-0.408044162864978690f,
	0.911706032005429880f, -0.410843171057903910f, 0.910441292258067250f,
	-0.413638312238434500f,
	0.909167983090522380f, -0.416429560097637150f, 0.907886116487666260f,
	-0.419216888363223910f,
	0.906595704514915330f, -0.422000270799799680f, 0.905296759318118820f,
	-0.424779681209108810f,
	0.903989293123443340f, -0.427555093430282080f, 0.902673318237258830f,
	-0.430326481340082610f,
	0.901348847046022030f, -0.433093818853151960f, 0.900015892016160280f,
	-0.435857079922255470f,
	0.898674465693953820f, -0.438616238538527660f, 0.897324580705418320f,
	-0.441371268731716670f,
	0.895966249756185220f, -0.444122144570429200f, 0.894599485631382700f,
	-0.446868840162374160f,
	0.893224301195515320f, -0.449611329654606540f, 0.891840709392342720f,
	-0.452349587233770890f,
	0.890448723244757880f, -0.455083587126343840f, 0.889048355854664570f,
	-0.457813303598877170f,
	0.887639620402853930f, -0.460538710958240010f, 0.886222530148880640f,
	-0.463259783551860150f,
	0.884797098430937790f, -0.465976495767966180f, 0.883363338665731580f,
	-0.468688822035827900f,
	0.881921264348355050f, -0.471396736825997640f, 0.880470889052160750f,
	-0.474100214650549970f,
	0.879012226428633530f, -0.476799230063322090f, 0.877545290207261350f,
	-0.479493757660153010f,
	0.876070094195406600f, -0.482183772079122720f, 0.874586652278176110f,
	-0.484869248000791060f,
	0.873094978418290090f, -0.487550160148436000f, 0.871595086655950980f,
	-0.490226483288291160f,
	0.870086991108711460f, -0.492898192229784040f, 0.868570705971340900f,
	-0.495565261825772540f,
	0.867046245515692650f, -0.498227666972781870f, 0.865513624090569090f,
	-0.500885382611240710f,
	0.863972856121586810f, -0.503538383725717580f, 0.862423956111040610f,
	-0.506186645345155230f,
	0.860866938637767310f, -0.508830142543106990f, 0.859301818357008470f,
	-0.511468850437970300f,
	0.857728610000272120f, -0.514102744193221660f, 0.856147328375194470f,
	-0.516731799017649870f,
	0.854557988365400530f, -0.519355990165589640f, 0.852960604930363630f,
	-0.521975292937154390f,
	0.851355193105265200f, -0.524589682678468950f, 0.849741768000852550f,
	-0.527199134781901280f,
	0.848120344803297230f, -0.529803624686294610f, 0.846490938774052130f,
	-0.532403127877197900f,
	0.844853565249707120f, -0.534997619887097150f, 0.843208239641845440f,
	-0.537587076295645390f,
	0.841554977436898440f, -0.540171472729892850f, 0.839893794195999520f,
	-0.542750784864515890f,
	0.838224705554838080f, -0.545324988422046460f, 0.836547727223512010f,
	-0.547894059173100190f,
	0.834862874986380010f, -0.550457972936604810f, 0.833170164701913190f,
	-0.553016705580027470f,
	0.831469612302545240f, -0.555570233019602180f, 0.829761233794523050f,
	-0.558118531220556100f,
	0.828045045257755800f, -0.560661576197336030f, 0.826321062845663530f,
	-0.563199344013834090f,
	0.824589302785025290f, -0.565731810783613120f, 0.822849781375826430f,
	-0.568258952670131490f,
	0.821102514991104650f, -0.570780745886967260f, 0.819347520076796900f,
	-0.573297166698042200f,
	0.817584813151583710f, -0.575808191417845340f, 0.815814410806733780f,
	-0.578313796411655590f,
	0.814036329705948410f, -0.580813958095764530f, 0.812250586585203880f,
	-0.583308652937698290f,
	0.810457198252594770f, -0.585797857456438860f, 0.808656181588174980f,
	-0.588281548222645220f,
	0.806847553543799330f, -0.590759701858874160f, 0.805031331142963660f,
	-0.593232295039799800f,
	0.803207531480644940f, -0.595699304492433360f, 0.801376171723140240f,
	-0.598160706996342270f,
	0.799537269107905010f, -0.600616479383868970f, 0.797690840943391160f,
	-0.603066598540348160f,
	0.795836904608883570f, -0.605511041404325550f, 0.793975477554337170f,
	-0.607949784967773630f,
	0.792106577300212390f, -0.610382806276309480f, 0.790230221437310030f,
	-0.612810082429409710f,
	0.788346427626606340f, -0.615231590580626820f, 0.786455213599085770f,
	-0.617647307937803870f,
	0.784556597155575240f, -0.620057211763289100f, 0.782650596166575730f,
	-0.622461279374149970f,
	0.780737228572094490f, -0.624859488142386340f, 0.778816512381475980f,
	-0.627251815495144080f,
	0.776888465673232440f, -0.629638238914926980f, 0.774953106594873930f,
	-0.632018735939809060f,
	0.773010453362736990f, -0.634393284163645490f, 0.771060524261813820f,
	-0.636761861236284200f,
	0.769103337645579700f, -0.639124444863775730f, 0.767138911935820400f,
	-0.641481012808583160f,
	0.765167265622458960f, -0.643831542889791390f, 0.763188417263381270f,
	-0.646176012983316280f,
	0.761202385484261780f, -0.648514401022112440f, 0.759209188978388070f,
	-0.650846684996380880f,
	0.757208846506484570f, -0.653172842953776760f, 0.755201376896536550f,
	-0.655492852999615350f,
	0.753186799043612520f, -0.657806693297078640f, 0.751165131909686480f,
	-0.660114342067420480f,
	0.749136394523459370f, -0.662415777590171780f, 0.747100605980180130f,
	-0.664710978203344790f,
	0.745057785441466060f, -0.666999922303637470f, 0.743007952135121720f,
	-0.669282588346636010f,
	0.740951125354959110f, -0.671558954847018330f, 0.738887324460615110f,
	-0.673829000378756040f,
	0.736816568877369900f, -0.676092703575315920f, 0.734738878095963500f,
	-0.678350043129861470f,
	0.732654271672412820f, -0.680600997795453020f, 0.730562769227827590f,
	-0.682845546385248080f,
	0.728464390448225200f, -0.685083667772700360f, 0.726359155084346010f,
	-0.687315340891759050f,
	0.724247082951467000f, -0.689540544737066830f, 0.722128193929215350f,
	-0.691759258364157750f,
	0.720002507961381650f, -0.693971460889654000f, 0.717870045055731710f,
	-0.696177131491462990f,
	0.715730825283818590f, -0.698376249408972920f, 0.713584868780793640f,
	-0.700568793943248340f,
	0.711432195745216430f, -0.702754744457225300f, 0.709272826438865690f,
	-0.704934080375904880f,
	0.707106781186547570f, -0.707106781186547460f, 0.704934080375904990f,
	-0.709272826438865580f,
	0.702754744457225300f, -0.711432195745216430f, 0.700568793943248450f,
	-0.713584868780793520f,
	0.698376249408972920f, -0.715730825283818590f, 0.696177131491462990f,
	-0.717870045055731710f,
	0.693971460889654000f, -0.720002507961381650f, 0.691759258364157750f,
	-0.722128193929215350f,
	0.689540544737066940f, -0.724247082951466890f, 0.687315340891759160f,
	-0.726359155084346010f,
	0.685083667772700360f, -0.728464390448225200f, 0.682845546385248080f,
	-0.730562769227827590f,
	0.680600997795453130f, -0.732654271672412820f, 0.678350043129861580f,
	-0.734738878095963390f,
	0.676092703575316030f, -0.736816568877369790f, 0.673829000378756150f,
	-0.738887324460615110f,
	0.671558954847018330f, -0.740951125354959110f, 0.669282588346636010f,
	-0.743007952135121720f,
	0.666999922303637470f, -0.745057785441465950f, 0.664710978203344900f,
	-0.747100605980180130f,
	0.662415777590171780f, -0.749136394523459260f, 0.660114342067420480f,
	-0.751165131909686370f,
	0.657806693297078640f, -0.753186799043612410f, 0.655492852999615460f,
	-0.755201376896536550f,
	0.653172842953776760f, -0.757208846506484460f, 0.650846684996380990f,
	-0.759209188978387960f,
	0.648514401022112550f, -0.761202385484261780f, 0.646176012983316390f,
	-0.763188417263381270f,
	0.643831542889791500f, -0.765167265622458960f, 0.641481012808583160f,
	-0.767138911935820400f,
	0.639124444863775730f, -0.769103337645579590f, 0.636761861236284200f,
	-0.771060524261813710f,
	0.634393284163645490f, -0.773010453362736990f, 0.632018735939809060f,
	-0.774953106594873820f,
	0.629638238914927100f, -0.776888465673232440f, 0.627251815495144190f,
	-0.778816512381475870f,
	0.624859488142386450f, -0.780737228572094380f, 0.622461279374150080f,
	-0.782650596166575730f,
	0.620057211763289210f, -0.784556597155575240f, 0.617647307937803980f,
	-0.786455213599085770f,
	0.615231590580626820f, -0.788346427626606230f, 0.612810082429409710f,
	-0.790230221437310030f,
	0.610382806276309480f, -0.792106577300212390f, 0.607949784967773740f,
	-0.793975477554337170f,
	0.605511041404325550f, -0.795836904608883460f, 0.603066598540348280f,
	-0.797690840943391040f,
	0.600616479383868970f, -0.799537269107905010f, 0.598160706996342380f,
	-0.801376171723140130f,
	0.595699304492433470f, -0.803207531480644830f, 0.593232295039799800f,
	-0.805031331142963660f,
	0.590759701858874280f, -0.806847553543799220f, 0.588281548222645330f,
	-0.808656181588174980f,
	0.585797857456438860f, -0.810457198252594770f, 0.583308652937698290f,
	-0.812250586585203880f,
	0.580813958095764530f, -0.814036329705948300f, 0.578313796411655590f,
	-0.815814410806733780f,
	0.575808191417845340f, -0.817584813151583710f, 0.573297166698042320f,
	-0.819347520076796900f,
	0.570780745886967370f, -0.821102514991104650f, 0.568258952670131490f,
	-0.822849781375826320f,
	0.565731810783613230f, -0.824589302785025290f, 0.563199344013834090f,
	-0.826321062845663420f,
	0.560661576197336030f, -0.828045045257755800f, 0.558118531220556100f,
	-0.829761233794523050f,
	0.555570233019602290f, -0.831469612302545240f, 0.553016705580027580f,
	-0.833170164701913190f,
	0.550457972936604810f, -0.834862874986380010f, 0.547894059173100190f,
	-0.836547727223511890f,
	0.545324988422046460f, -0.838224705554837970f, 0.542750784864516000f,
	-0.839893794195999410f,
	0.540171472729892970f, -0.841554977436898330f, 0.537587076295645510f,
	-0.843208239641845440f,
	0.534997619887097260f, -0.844853565249707010f, 0.532403127877198010f,
	-0.846490938774052020f,
	0.529803624686294830f, -0.848120344803297120f, 0.527199134781901390f,
	-0.849741768000852440f,
	0.524589682678468840f, -0.851355193105265200f, 0.521975292937154390f,
	-0.852960604930363630f,
	0.519355990165589530f, -0.854557988365400530f, 0.516731799017649980f,
	-0.856147328375194470f,
	0.514102744193221660f, -0.857728610000272120f, 0.511468850437970520f,
	-0.859301818357008360f,
	0.508830142543106990f, -0.860866938637767310f, 0.506186645345155450f,
	-0.862423956111040500f,
	0.503538383725717580f, -0.863972856121586700f, 0.500885382611240940f,
	-0.865513624090568980f,
	0.498227666972781870f, -0.867046245515692650f, 0.495565261825772490f,
	-0.868570705971340900f,
	0.492898192229784090f, -0.870086991108711350f, 0.490226483288291100f,
	-0.871595086655951090f,
	0.487550160148436050f, -0.873094978418290090f, 0.484869248000791120f,
	-0.874586652278176110f,
	0.482183772079122830f, -0.876070094195406600f, 0.479493757660153010f,
	-0.877545290207261240f,
	0.476799230063322250f, -0.879012226428633410f, 0.474100214650550020f,
	-0.880470889052160750f,
	0.471396736825997810f, -0.881921264348354940f, 0.468688822035827960f,
	-0.883363338665731580f,
	0.465976495767966130f, -0.884797098430937790f, 0.463259783551860260f,
	-0.886222530148880640f,
	0.460538710958240010f, -0.887639620402853930f, 0.457813303598877290f,
	-0.889048355854664570f,
	0.455083587126343840f, -0.890448723244757880f, 0.452349587233771000f,
	-0.891840709392342720f,
	0.449611329654606600f, -0.893224301195515320f, 0.446868840162374330f,
	-0.894599485631382580f,
	0.444122144570429260f, -0.895966249756185110f, 0.441371268731716620f,
	-0.897324580705418320f,
	0.438616238538527710f, -0.898674465693953820f, 0.435857079922255470f,
	-0.900015892016160280f,
	0.433093818853152010f, -0.901348847046022030f, 0.430326481340082610f,
	-0.902673318237258830f,
	0.427555093430282200f, -0.903989293123443340f, 0.424779681209108810f,
	-0.905296759318118820f,
	0.422000270799799790f, -0.906595704514915330f, 0.419216888363223960f,
	-0.907886116487666150f,
	0.416429560097637320f, -0.909167983090522270f, 0.413638312238434560f,
	-0.910441292258067140f,
	0.410843171057903910f, -0.911706032005429880f, 0.408044162864978740f,
	-0.912962190428398100f,
	0.405241314004989860f, -0.914209755703530690f, 0.402434650859418540f,
	-0.915448716088267830f,
	0.399624199845646790f, -0.916679059921042700f, 0.396809987416710420f,
	-0.917900775621390390f,
	0.393992040061048100f, -0.919113851690057770f, 0.391170384302253980f,
	-0.920318276709110480f,
	0.388345046698826300f, -0.921514039342041900f, 0.385516053843919020f,
	-0.922701128333878520f,
	0.382683432365089840f, -0.923879532511286740f, 0.379847208924051110f,
	-0.925049240782677580f,
	0.377007410216418310f, -0.926210242138311270f, 0.374164062971457990f,
	-0.927362525650401110f,
	0.371317193951837600f, -0.928506080473215480f, 0.368466829953372320f,
	-0.929640895843181330f,
	0.365612997804773960f, -0.930766961078983710f, 0.362755724367397230f,
	-0.931884265581668150f,
	0.359895036534988280f, -0.932992798834738850f, 0.357030961233430030f,
	-0.934092550404258870f,
	0.354163525420490510f, -0.935183509938947500f, 0.351292756085567150f,
	-0.936265667170278260f,
	0.348418680249434510f, -0.937339011912574960f, 0.345541324963989150f,
	-0.938403534063108060f,
	0.342660717311994380f, -0.939459223602189920f, 0.339776884406826960f,
	-0.940506070593268300f,
	0.336889853392220050f, -0.941544065183020810f, 0.333999651442009490f,
	-0.942573197601446870f,
	0.331106305759876430f, -0.943593458161960390f, 0.328209843579092660f,
	-0.944604837261480260f,
	0.325310292162262980f, -0.945607325380521280f, 0.322407678801070020f,
	-0.946600913083283530f,
	0.319502030816015750f, -0.947585591017741090f, 0.316593375556165850f,
	-0.948561349915730270f,
	0.313681740398891570f, -0.949528180593036670f, 0.310767152749611470f,
	-0.950486073949481700f,
	0.307849640041534980f, -0.951435020969008340f, 0.304929229735402430f,
	-0.952375012719765880f,
	0.302005949319228200f, -0.953306040354193750f, 0.299079826308040480f,
	-0.954228095109105670f,
	0.296150888243623960f, -0.955141168305770670f, 0.293219162694258680f,
	-0.956045251349996410f,
	0.290284677254462330f, -0.956940335732208940f, 0.287347459544729570f,
	-0.957826413027532910f,
	0.284407537211271820f, -0.958703474895871600f, 0.281464937925758050f,
	-0.959571513081984520f,
	0.278519689385053060f, -0.960430519415565790f, 0.275571819310958250f,
	-0.961280485811320640f,
	0.272621355449948980f, -0.962121404269041580f, 0.269668325572915200f,
	-0.962953266873683880f,
	0.266712757474898420f, -0.963776065795439840f, 0.263754678974831510f,
	-0.964589793289812650f,
	0.260794117915275570f, -0.965394441697689400f, 0.257831102162158930f,
	-0.966190003445412620f,
	0.254865659604514630f, -0.966976471044852070f, 0.251897818154216910f,
	-0.967753837093475510f,
	0.248927605745720260f, -0.968522094274417270f, 0.245955050335794590f,
	-0.969281235356548530f,
	0.242980179903263980f, -0.970031253194543970f, 0.240003022448741500f,
	-0.970772140728950350f,
	0.237023605994367340f, -0.971503890986251780f, 0.234041958583543460f,
	-0.972226497078936270f,
	0.231058108280671280f, -0.972939952205560070f, 0.228072083170885790f,
	-0.973644249650811870f,
	0.225083911359792780f, -0.974339382785575860f, 0.222093620973203590f,
	-0.975025345066994120f,
	0.219101240156869770f, -0.975702130038528570f, 0.216106797076219600f,
	-0.976369731330021140f,
	0.213110319916091360f, -0.977028142657754390f, 0.210111836880469720f,
	-0.977677357824509930f,
	0.207111376192218560f, -0.978317370719627650f, 0.204108966092817010f,
	-0.978948175319062200f,
	0.201104634842091960f, -0.979569765685440520f, 0.198098410717953730f,
	-0.980182135968117320f,
	0.195090322016128330f, -0.980785280403230430f, 0.192080397049892380f,
	-0.981379193313754560f,
	0.189068664149806280f, -0.981963869109555240f, 0.186055151663446630f,
	-0.982539302287441240f,
	0.183039887955141060f, -0.983105487431216290f, 0.180022901405699510f,
	-0.983662419211730250f,
	0.177004220412148860f, -0.984210092386929030f, 0.173983873387463850f,
	-0.984748501801904210f,
	0.170961888760301360f, -0.985277642388941220f, 0.167938294974731230f,
	-0.985797509167567370f,
	0.164913120489970090f, -0.986308097244598670f, 0.161886393780111910f,
	-0.986809401814185420f,
	0.158858143333861390f, -0.987301418157858430f, 0.155828397654265320f,
	-0.987784141644572180f,
	0.152797185258443410f, -0.988257567730749460f, 0.149764534677321620f,
	-0.988721691960323780f,
	0.146730474455361750f, -0.989176509964781010f, 0.143695033150294580f,
	-0.989622017463200780f,
	0.140658239332849240f, -0.990058210262297120f, 0.137620121586486180f,
	-0.990485084256456980f,
	0.134580708507126220f, -0.990902635427780010f, 0.131540028702883280f,
	-0.991310859846115440f,
	0.128498110793793220f, -0.991709753669099530f, 0.125454983411546210f,
	-0.992099313142191800f,
	0.122410675199216280f, -0.992479534598709970f, 0.119365214810991350f,
	-0.992850414459865100f,
	0.116318630911904880f, -0.993211949234794500f, 0.113270952177564360f,
	-0.993564135520595300f,
	0.110222207293883180f, -0.993906970002356060f, 0.107172424956808870f,
	-0.994240449453187900f,
	0.104121633872054730f, -0.994564570734255420f, 0.101069862754827880f,
	-0.994879330794805620f,
	0.098017140329560770f, -0.995184726672196820f, 0.094963495329639061f,
	-0.995480755491926940f,
	0.091908956497132696f, -0.995767414467659820f, 0.088853552582524684f,
	-0.996044700901251970f,
	0.085797312344439880f, -0.996312612182778000f, 0.082740264549375803f,
	-0.996571145790554840f,
	0.079682437971430126f, -0.996820299291165670f, 0.076623861392031617f,
	-0.997060070339482960f,
	0.073564563599667454f, -0.997290456678690210f, 0.070504573389614009f,
	-0.997511456140303450f,
	0.067443919563664106f, -0.997723066644191640f, 0.064382630929857410f,
	-0.997925286198596000f,
	0.061320736302208648f, -0.998118112900149180f, 0.058258264500435732f,
	-0.998301544933892890f,
	0.055195244349690031f, -0.998475580573294770f, 0.052131704680283317f,
	-0.998640218180265270f,
	0.049067674327418126f, -0.998795456205172410f, 0.046003182130914644f,
	-0.998941293186856870f,
	0.042938256934940959f, -0.999077727752645360f, 0.039872927587739845f,
	-0.999204758618363890f,
	0.036807222941358991f, -0.999322384588349540f, 0.033741171851377642f,
	-0.999430604555461730f,
	0.030674803176636581f, -0.999529417501093140f, 0.027608145778965820f,
	-0.999618822495178640f,
	0.024541228522912264f, -0.999698818696204250f, 0.021474080275469605f,
	-0.999769405351215280f,
	0.018406729905804820f, -0.999830581795823400f, 0.015339206284988220f,
	-0.999882347454212560f,
	0.012271538285719944f, -0.999924701839144500f, 0.009203754782059960f,
	-0.999957644551963900f,
	0.006135884649154515f, -0.999981175282601110f, 0.003067956762966138f,
	-0.999995293809576190f
};

static const float32_t Weights_2048[4096] = {
	1.000000000000000000f, 0.000000000000000000f, 0.999999705862882230f,
	-0.000766990318742704f,
	0.999998823451701880f, -0.001533980186284766f, 0.999997352766978210f,
	-0.002300969151425805f,
	0.999995293809576190f, -0.003067956762965976f, 0.999992646580707190f,
	-0.003834942569706228f,
	0.999989411081928400f, -0.004601926120448571f, 0.999985587315143200f,
	-0.005368906963996343f,
	0.999981175282601110f, -0.006135884649154475f, 0.999976174986897610f,
	-0.006902858724729756f,
	0.999970586430974140f, -0.007669828739531097f, 0.999964409618118280f,
	-0.008436794242369799f,
	0.999957644551963900f, -0.009203754782059819f, 0.999950291236490480f,
	-0.009970709907418031f,
	0.999942349676023910f, -0.010737659167264491f, 0.999933819875236000f,
	-0.011504602110422714f,
	0.999924701839144500f, -0.012271538285719925f, 0.999914995573113470f,
	-0.013038467241987334f,
	0.999904701082852900f, -0.013805388528060391f, 0.999893818374418490f,
	-0.014572301692779064f,
	0.999882347454212560f, -0.015339206284988100f, 0.999870288328982950f,
	-0.016106101853537287f,
	0.999857641005823860f, -0.016872987947281710f, 0.999844405492175240f,
	-0.017639864115082053f,
	0.999830581795823400f, -0.018406729905804820f, 0.999816169924900410f,
	-0.019173584868322623f,
	0.999801169887884260f, -0.019940428551514441f, 0.999785581693599210f,
	-0.020707260504265895f,
	0.999769405351215280f, -0.021474080275469508f, 0.999752640870248840f,
	-0.022240887414024961f,
	0.999735288260561680f, -0.023007681468839369f, 0.999717347532362190f,
	-0.023774461988827555f,
	0.999698818696204250f, -0.024541228522912288f, 0.999679701762987930f,
	-0.025307980620024571f,
	0.999659996743959220f, -0.026074717829103901f, 0.999639703650710200f,
	-0.026841439699098531f,
	0.999618822495178640f, -0.027608145778965740f, 0.999597353289648380f,
	-0.028374835617672099f,
	0.999575296046749220f, -0.029141508764193722f, 0.999552650779456990f,
	-0.029908164767516555f,
	0.999529417501093140f, -0.030674803176636626f, 0.999505596225325310f,
	-0.031441423540560301f,
	0.999481186966166950f, -0.032208025408304586f, 0.999456189737977340f,
	-0.032974608328897335f,
	0.999430604555461730f, -0.033741171851377580f, 0.999404431433671300f,
	-0.034507715524795750f,
	0.999377670388002850f, -0.035274238898213947f, 0.999350321434199440f,
	-0.036040741520706229f,
	0.999322384588349540f, -0.036807222941358832f, 0.999293859866887790f,
	-0.037573682709270494f,
	0.999264747286594420f, -0.038340120373552694f, 0.999235046864595850f,
	-0.039106535483329888f,
	0.999204758618363890f, -0.039872927587739811f, 0.999173882565716380f,
	-0.040639296235933736f,
	0.999142418724816910f, -0.041405640977076739f, 0.999110367114174890f,
	-0.042171961360347947f,
	0.999077727752645360f, -0.042938256934940820f, 0.999044500659429290f,
	-0.043704527250063421f,
	0.999010685854073380f, -0.044470771854938668f, 0.998976283356469820f,
	-0.045236990298804590f,
	0.998941293186856870f, -0.046003182130914623f, 0.998905715365818290f,
	-0.046769346900537863f,
	0.998869549914283560f, -0.047535484156959303f, 0.998832796853527990f,
	-0.048301593449480144f,
	0.998795456205172410f, -0.049067674327418015f, 0.998757527991183340f,
	-0.049833726340107277f,
	0.998719012233872940f, -0.050599749036899282f, 0.998679908955899090f,
	-0.051365741967162593f,
	0.998640218180265270f, -0.052131704680283324f, 0.998599939930320370f,
	-0.052897636725665324f,
	0.998559074229759310f, -0.053663537652730520f, 0.998517621102622210f,
	-0.054429407010919133f,
	0.998475580573294770f, -0.055195244349689934f, 0.998432952666508440f,
	-0.055961049218520569f,
	0.998389737407340160f, -0.056726821166907748f, 0.998345934821212370f,
	-0.057492559744367566f,
	0.998301544933892890f, -0.058258264500435752f, 0.998256567771495180f,
	-0.059023934984667931f,
	0.998211003360478190f, -0.059789570746639868f, 0.998164851727646240f,
	-0.060555171335947788f,
	0.998118112900149180f, -0.061320736302208578f, 0.998070786905482340f,
	-0.062086265195060088f,
	0.998022873771486240f, -0.062851757564161406f, 0.997974373526346990f,
	-0.063617212959193106f,
	0.997925286198596000f, -0.064382630929857465f, 0.997875611817110150f,
	-0.065148011025878833f,
	0.997825350411111640f, -0.065913352797003805f, 0.997774502010167820f,
	-0.066678655793001557f,
	0.997723066644191640f, -0.067443919563664051f, 0.997671044343441000f,
	-0.068209143658806329f,
	0.997618435138519550f, -0.068974327628266746f, 0.997565239060375750f,
	-0.069739471021907307f,
	0.997511456140303450f, -0.070504573389613856f, 0.997457086409941910f,
	-0.071269634281296401f,
	0.997402129901275300f, -0.072034653246889332f, 0.997346586646633230f,
	-0.072799629836351673f,
	0.997290456678690210f, -0.073564563599667426f, 0.997233740030466280f,
	-0.074329454086845756f,
	0.997176436735326190f, -0.075094300847921305f, 0.997118546826979980f,
	-0.075859103432954447f,
	0.997060070339482960f, -0.076623861392031492f, 0.997001007307235290f,
	-0.077388574275265049f,
	0.996941357764982160f, -0.078153241632794232f, 0.996881121747813850f,
	-0.078917863014784942f,
	0.996820299291165670f, -0.079682437971430126f, 0.996758890430818000f,
	-0.080446966052950014f,
	0.996696895202896060f, -0.081211446809592441f, 0.996634313643869900f,
	-0.081975879791633066f,
	0.996571145790554840f, -0.082740264549375692f, 0.996507391680110820f,
	-0.083504600633152432f,
	0.996443051350042630f, -0.084268887593324071f, 0.996378124838200210f,
	-0.085033124980280275f,
	0.996312612182778000f, -0.085797312344439894f, 0.996246513422315520f,
	-0.086561449236251170f,
	0.996179828595696980f, -0.087325535206192059f, 0.996112557742151130f,
	-0.088089569804770507f,
	0.996044700901251970f, -0.088853552582524600f, 0.995976258112917790f,
	-0.089617483090022959f,
	0.995907229417411720f, -0.090381360877864983f, 0.995837614855341610f,
	-0.091145185496681005f,
	0.995767414467659820f, -0.091908956497132724f, 0.995696628295663520f,
	-0.092672673429913310f,
	0.995625256380994310f, -0.093436335845747787f, 0.995553298765638470f,
	-0.094199943295393204f,
	0.995480755491926940f, -0.094963495329638992f, 0.995407626602534900f,
	-0.095726991499307162f,
	0.995333912140482280f, -0.096490431355252593f, 0.995259612149133390f,
	-0.097253814448363271f,
	0.995184726672196930f, -0.098017140329560604f, 0.995109255753726110f,
	-0.098780408549799623f,
	0.995033199438118630f, -0.099543618660069319f, 0.994956557770116380f,
	-0.100306770211392860f,
	0.994879330794805620f, -0.101069862754827820f, 0.994801518557617110f,
	-0.101832895841466530f,
	0.994723121104325700f, -0.102595869022436280f, 0.994644138481050710f,
	-0.103358781848899610f,
	0.994564570734255420f, -0.104121633872054590f, 0.994484417910747600f,
	-0.104884424643134970f,
	0.994403680057679100f, -0.105647153713410620f, 0.994322357222545810f,
	-0.106409820634187680f,
	0.994240449453187900f, -0.107172424956808840f, 0.994157956797789730f,
	-0.107934966232653650f,
	0.994074879304879370f, -0.108697444013138720f, 0.993991217023329380f,
	-0.109459857849717980f,
	0.993906970002356060f, -0.110222207293883060f, 0.993822138291519660f,
	-0.110984491897163390f,
	0.993736721940724600f, -0.111746711211126590f, 0.993650721000219120f,
	-0.112508864787378690f,
	0.993564135520595300f, -0.113270952177564350f, 0.993476965552789190f,
	-0.114032972933367200f,
	0.993389211148080650f, -0.114794926606510080f, 0.993300872358093280f,
	-0.115556812748755260f,
	0.993211949234794500f, -0.116318630911904750f, 0.993122441830495580f,
	-0.117080380647800590f,
	0.993032350197851410f, -0.117842061508324980f, 0.992941674389860470f,
	-0.118603673045400720f,
	0.992850414459865100f, -0.119365214810991350f, 0.992758570461551140f,
	-0.120126686357101500f,
	0.992666142448948020f, -0.120888087235777080f, 0.992573130476428810f,
	-0.121649416999105530f,
	0.992479534598709970f, -0.122410675199216200f, 0.992385354870851670f,
	-0.123171861388280480f,
	0.992290591348257370f, -0.123932975118512160f, 0.992195244086673920f,
	-0.124694015942167640f,
	0.992099313142191800f, -0.125454983411546230f, 0.992002798571244520f,
	-0.126215877078990350f,
	0.991905700430609330f, -0.126976696496885870f, 0.991808018777406430f,
	-0.127737441217662310f,
	0.991709753669099530f, -0.128498110793793170f, 0.991610905163495370f,
	-0.129258704777796140f,
	0.991511473318743900f, -0.130019222722233350f, 0.991411458193338540f,
	-0.130779664179711710f,
	0.991310859846115440f, -0.131540028702883120f, 0.991209678336254060f,
	-0.132300315844444650f,
	0.991107913723276890f, -0.133060525157139060f, 0.991005566067049370f,
	-0.133820656193754720f,
	0.990902635427780010f, -0.134580708507126170f, 0.990799121866020370f,
	-0.135340681650134210f,
	0.990695025442664630f, -0.136100575175706200f, 0.990590346218950150f,
	-0.136860388636816380f,
	0.990485084256457090f, -0.137620121586486040f, 0.990379239617108160f,
	-0.138379773577783890f,
	0.990272812363169110f, -0.139139344163826200f, 0.990165802557248400f,
	-0.139898832897777210f,
	0.990058210262297120f, -0.140658239332849210f, 0.989950035541608990f,
	-0.141417563022303020f,
	0.989841278458820530f, -0.142176803519448030f, 0.989731939077910570f,
	-0.142935960377642670f,
	0.989622017463200890f, -0.143695033150294470f, 0.989511513679355190f,
	-0.144454021390860470f,
	0.989400427791380380f, -0.145212924652847460f, 0.989288759864625170f,
	-0.145971742489812210f,
	0.989176509964781010f, -0.146730474455361750f, 0.989063678157881540f,
	-0.147489120103153570f,
	0.988950264510302990f, -0.148247678986896030f, 0.988836269088763540f,
	-0.149006150660348450f,
	0.988721691960323780f, -0.149764534677321510f, 0.988606533192386450f,
	-0.150522830591677400f,
	0.988490792852696590f, -0.151281037957330220f, 0.988374471009341280f,
	-0.152039156328246050f,
	0.988257567730749460f, -0.152797185258443440f, 0.988140083085692570f,
	-0.153555124301993450f,
	0.988022017143283530f, -0.154312973013020100f, 0.987903369972977790f,
	-0.155070730945700510f,
	0.987784141644572180f, -0.155828397654265230f, 0.987664332228205710f,
	-0.156585972692998430f,
	0.987543941794359230f, -0.157343455616238250f, 0.987422970413855410f,
	-0.158100845978376980f,
	0.987301418157858430f, -0.158858143333861450f, 0.987179285097874340f,
	-0.159615347237193060f,
	0.987056571305750970f, -0.160372457242928280f, 0.986933276853677710f,
	-0.161129472905678810f,
	0.986809401814185530f, -0.161886393780111830f, 0.986684946260146690f,
	-0.162643219420950310f,
	0.986559910264775410f, -0.163399949382973230f, 0.986434293901627180f,
	-0.164156583221015810f,
	0.986308097244598670f, -0.164913120489969890f, 0.986181320367928270f,
	-0.165669560744784120f,
	0.986053963346195440f, -0.166425903540464100f, 0.985926026254321130f,
	-0.167182148432072940f,
	0.985797509167567480f, -0.167938294974731170f, 0.985668412161537550f,
	-0.168694342723617330f,
	0.985538735312176060f, -0.169450291233967960f, 0.985408478695768420f,
	-0.170206140061078070f,
	0.985277642388941220f, -0.170961888760301220f, 0.985146226468662230f,
	-0.171717536887049970f,
	0.985014231012239840f, -0.172473083996795950f, 0.984881656097323700f,
	-0.173228529645070320f,
	0.984748501801904210f, -0.173983873387463820f, 0.984614768204312600f,
	-0.174739114779627200f,
	0.984480455383220930f, -0.175494253377271430f, 0.984345563417641900f,
	-0.176249288736167880f,
	0.984210092386929030f, -0.177004220412148750f, 0.984074042370776450f,
	-0.177759047961107170f,
	0.983937413449218920f, -0.178513770938997510f, 0.983800205702631600f,
	-0.179268388901835750f,
	0.983662419211730250f, -0.180022901405699510f, 0.983524054057571260f,
	-0.180777308006728590f,
	0.983385110321551180f, -0.181531608261124970f, 0.983245588085407070f,
	-0.182285801725153300f,
	0.983105487431216290f, -0.183039887955140950f, 0.982964808441396440f,
	-0.183793866507478450f,
	0.982823551198705240f, -0.184547736938619620f, 0.982681715786240860f,
	-0.185301498805081900f,
	0.982539302287441240f, -0.186055151663446630f, 0.982396310786084690f,
	-0.186808695070359270f,
	0.982252741366289370f, -0.187562128582529600f, 0.982108594112513610f,
	-0.188315451756732120f,
	0.981963869109555240f, -0.189068664149806190f, 0.981818566442552500f,
	-0.189821765318656410f,
	0.981672686196983110f, -0.190574754820252740f, 0.981526228458664770f,
	-0.191327632211630900f,
	0.981379193313754560f, -0.192080397049892440f, 0.981231580848749730f,
	-0.192833048892205230f,
	0.981083391150486710f, -0.193585587295803610f, 0.980934624306141640f,
	-0.194338011817988600f,
	0.980785280403230430f, -0.195090322016128250f, 0.980635359529608120f,
	-0.195842517447657850f,
	0.980484861773469380f, -0.196594597670080220f, 0.980333787223347960f,
	-0.197346562240965920f,
	0.980182135968117430f, -0.198098410717953560f, 0.980029908096990090f,
	-0.198850142658750090f,
	0.979877103699517640f, -0.199601757621130970f, 0.979723722865591170f,
	-0.200353255162940450f,
	0.979569765685440520f, -0.201104634842091900f, 0.979415232249634780f,
	-0.201855896216568050f,
	0.979260122649082020f, -0.202607038844421130f, 0.979104436975029250f,
	-0.203358062283773320f,
	0.978948175319062200f, -0.204108966092816870f, 0.978791337773105670f,
	-0.204859749829814420f,
	0.978633924429423210f, -0.205610413053099240f, 0.978475935380616830f,
	-0.206360955321075510f,
	0.978317370719627650f, -0.207111376192218560f, 0.978158230539735050f,
	-0.207861675225075070f,
	0.977998514934557140f, -0.208611851978263490f, 0.977838223998050430f,
	-0.209361906010474160f,
	0.977677357824509930f, -0.210111836880469610f, 0.977515916508569280f,
	-0.210861644147084860f,
	0.977353900145199960f, -0.211611327369227550f, 0.977191308829712280f,
	-0.212360886105878420f,
	0.977028142657754390f, -0.213110319916091360f, 0.976864401725312640f,
	-0.213859628358993750f,
	0.976700086128711840f, -0.214608810993786760f, 0.976535195964614470f,
	-0.215357867379745550f,
	0.976369731330021140f, -0.216106797076219520f, 0.976203692322270560f,
	-0.216855599642632620f,
	0.976037079039039020f, -0.217604274638483640f, 0.975869891578341030f,
	-0.218352821623346320f,
	0.975702130038528570f, -0.219101240156869800f, 0.975533794518291360f,
	-0.219849529798778700f,
	0.975364885116656980f, -0.220597690108873510f, 0.975195401932990370f,
	-0.221345720647030810f,
	0.975025345066994120f, -0.222093620973203510f, 0.974854714618708430f,
	-0.222841390647421120f,
	0.974683510688510670f, -0.223589029229789990f, 0.974511733377115720f,
	-0.224336536280493600f,
	0.974339382785575860f, -0.225083911359792830f, 0.974166459015280320f,
	-0.225831154028026170f,
	0.973992962167955830f, -0.226578263845610000f, 0.973818892345666100f,
	-0.227325240373038860f,
	0.973644249650811980f, -0.228072083170885730f, 0.973469034186131070f,
	-0.228818791799802220f,
	0.973293246054698250f, -0.229565365820518870f, 0.973116885359925130f,
	-0.230311804793845440f,
	0.972939952205560180f, -0.231058108280671110f, 0.972762446695688570f,
	-0.231804275841964780f,
	0.972584368934732210f, -0.232550307038775240f, 0.972405719027449770f,
	-0.233296201432231590f,
	0.972226497078936270f, -0.234041958583543430f, 0.972046703194623500f,
	-0.234787578054000970f,
	0.971866337480279400f, -0.235533059404975490f, 0.971685400042008540f,
	-0.236278402197919570f,
	0.971503890986251780f, -0.237023605994367200f, 0.971321810419786160f,
	-0.237768670355934190f,
	0.971139158449725090f, -0.238513594844318420f, 0.970955935183517970f,
	-0.239258379021299980f,
	0.970772140728950350f, -0.240003022448741500f, 0.970587775194143630f,
	-0.240747524688588430f,
	0.970402838687555500f, -0.241491885302869330f, 0.970217331317979160f,
	-0.242236103853696010f,
	0.970031253194543970f, -0.242980179903263870f, 0.969844604426714830f,
	-0.243724113013852160f,
	0.969657385124292450f, -0.244467902747824150f, 0.969469595397413060f,
	-0.245211548667627540f,
	0.969281235356548530f, -0.245955050335794590f, 0.969092305112506210f,
	-0.246698407314942410f,
	0.968902804776428870f, -0.247441619167773270f, 0.968712734459794780f,
	-0.248184685457074780f,
	0.968522094274417380f, -0.248927605745720150f, 0.968330884332445190f,
	-0.249670379596668570f,
	0.968139104746362440f, -0.250413006572965220f, 0.967946755628987800f,
	-0.251155486237741920f,
	0.967753837093475510f, -0.251897818154216970f, 0.967560349253314360f,
	-0.252640001885695520f,
	0.967366292222328510f, -0.253382036995570160f, 0.967171666114676640f,
	-0.254123923047320620f,
	0.966976471044852070f, -0.254865659604514570f, 0.966780707127683270f,
	-0.255607246230807380f,
	0.966584374478333120f, -0.256348682489942910f, 0.966387473212298900f,
	-0.257089967945753120f,
	0.966190003445412500f, -0.257831102162158990f, 0.965991965293840570f,
	-0.258572084703170340f,
	0.965793358874083680f, -0.259312915132886230f, 0.965594184302976830f,
	-0.260053593015495190f,
	0.965394441697689400f, -0.260794117915275510f, 0.965194131175724720f,
	-0.261534489396595520f,
	0.964993252854920320f, -0.262274707023913590f, 0.964791806853447900f,
	-0.263014770361779000f,
	0.964589793289812760f, -0.263754678974831350f, 0.964387212282854290f,
	-0.264494432427801630f,
	0.964184063951745830f, -0.265234030285511790f, 0.963980348415994110f,
	-0.265973472112875590f,
	0.963776065795439840f, -0.266712757474898370f, 0.963571216210257320f,
	-0.267451885936677620f,
	0.963365799780954050f, -0.268190857063403180f, 0.963159816628371360f,
	-0.268929670420357260f,
	0.962953266873683880f, -0.269668325572915090f, 0.962746150638399410f,
	-0.270406822086544820f,
	0.962538468044359160f, -0.271145159526808010f, 0.962330219213737400f,
	-0.271883337459359720f,
	0.962121404269041580f, -0.272621355449948980f, 0.961912023333112210f,
	-0.273359213064418680f,
	0.961702076529122540f, -0.274096909868706380f, 0.961491563980579000f,
	-0.274834445428843940f,
	0.961280485811320640f, -0.275571819310958140f, 0.961068842145519350f,
	-0.276309031081271080f,
	0.960856633107679660f, -0.277046080306099900f, 0.960643858822638590f,
	-0.277782966551857690f,
	0.960430519415565790f, -0.278519689385053060f, 0.960216615011963430f,
	-0.279256248372291180f,
	0.960002145737665960f, -0.279992643080273220f, 0.959787111718839900f,
	-0.280728873075797190f,
	0.959571513081984520f, -0.281464937925757940f, 0.959355349953930790f,
	-0.282200837197147560f,
	0.959138622461841890f, -0.282936570457055390f, 0.958921330733213170f,
	-0.283672137272668430f,
	0.958703474895871600f, -0.284407537211271880f, 0.958485055077976100f,
	-0.285142769840248670f,
	0.958266071408017670f, -0.285877834727080620f, 0.958046524014818600f,
	-0.286612731439347790f,
	0.957826413027532910f, -0.287347459544729510f, 0.957605738575646350f,
	-0.288082018611004130f,
	0.957384500788975860f, -0.288816408206049480f, 0.957162699797670210f,
	-0.289550627897843030f,
	0.956940335732208820f, -0.290284677254462330f, 0.956717408723403050f,
	-0.291018555844085090f,
	0.956493918902395100f, -0.291752263234989260f, 0.956269866400658030f,
	-0.292485798995553880f,
	0.956045251349996410f, -0.293219162694258630f, 0.955820073882545420f,
	-0.293952353899684660f,
	0.955594334130771110f, -0.294685372180514330f, 0.955368032227470350f,
	-0.295418217105532010f,
	0.955141168305770780f, -0.296150888243623790f, 0.954913742499130520f,
	-0.296883385163778270f,
	0.954685754941338340f, -0.297615707435086200f, 0.954457205766513490f,
	-0.298347854626741400f,
	0.954228095109105670f, -0.299079826308040480f, 0.953998423103894490f,
	-0.299811622048383350f,
	0.953768189885990330f, -0.300543241417273450f, 0.953537395590833280f,
	-0.301274683984317950f,
	0.953306040354193860f, -0.302005949319228080f, 0.953074124312172200f,
	-0.302737036991819140f,
	0.952841647601198720f, -0.303467946572011320f, 0.952608610358033350f,
	-0.304198677629829110f,
	0.952375012719765880f, -0.304929229735402370f, 0.952140854823815830f,
	-0.305659602458966120f,
	0.951906136807932350f, -0.306389795370860920f, 0.951670858810193860f,
	-0.307119808041533100f,
	0.951435020969008340f, -0.307849640041534870f, 0.951198623423113230f,
	-0.308579290941525090f,
	0.950961666311575080f, -0.309308760312268730f, 0.950724149773789610f,
	-0.310038047724637890f,
	0.950486073949481700f, -0.310767152749611470f, 0.950247438978705230f,
	-0.311496074958275910f,
	0.950008245001843000f, -0.312224813921824880f, 0.949768492159606680f,
	-0.312953369211560200f,
	0.949528180593036670f, -0.313681740398891520f, 0.949287310443502120f,
	-0.314409927055336660f,
	0.949045881852700560f, -0.315137928752522440f, 0.948803894962658490f,
	-0.315865745062183960f,
	0.948561349915730270f, -0.316593375556165850f, 0.948318246854599090f,
	-0.317320819806421740f,
	0.948074585922276230f, -0.318048077385014950f, 0.947830367262101010f,
	-0.318775147864118480f,
	0.947585591017741090f, -0.319502030816015690f, 0.947340257333192050f,
	-0.320228725813099860f,
	0.947094366352777220f, -0.320955232427875210f, 0.946847918221148000f,
	-0.321681550232956580f,
	0.946600913083283530f, -0.322407678801069850f, 0.946353351084490590f,
	-0.323133617705052330f,
	0.946105232370403450f, -0.323859366517852850f, 0.945856557086983910f,
	-0.324584924812532150f,
	0.945607325380521280f, -0.325310292162262930f, 0.945357537397632290f,
	-0.326035468140330240f,
	0.945107193285260610f, -0.326760452320131730f, 0.944856293190677210f,
	-0.327485244275178000f,
	0.944604837261480260f, -0.328209843579092500f, 0.944352825645594750f,
	-0.328934249805612200f,
	0.944100258491272660f, -0.329658462528587490f, 0.943847135947092690f,
	-0.330382481321982780f,
	0.943593458161960390f, -0.331106305759876430f, 0.943339225285107720f,
	-0.331829935416461110f,
	0.943084437466093490f, -0.332553369866044220f, 0.942829094854802710f,
	-0.333276608683047930f,
	0.942573197601446870f, -0.333999651442009380f, 0.942316745856563780f,
	-0.334722497717581220f,
	0.942059739771017310f, -0.335445147084531600f, 0.941802179495997650f,
	-0.336167599117744520f,
	0.941544065183020810f, -0.336889853392220050f, 0.941285396983928660f,
	-0.337611909483074620f,
	0.941026175050889260f, -0.338333766965541130f, 0.940766399536396070f,
	-0.339055425414969640f,
	0.940506070593268300f, -0.339776884406826850f, 0.940245188374650880f,
	-0.340498143516697160f,
	0.939983753034014050f, -0.341219202320282360f, 0.939721764725153340f,
	-0.341940060393402190f,
	0.939459223602189920f, -0.342660717311994380f, 0.939196129819569900f,
	-0.343381172652115040f,
	0.938932483532064600f, -0.344101425989938810f, 0.938668284894770170f,
	-0.344821476901759290f,
	0.938403534063108060f, -0.345541324963989090f, 0.938138231192824360f,
	-0.346260969753160010f,
	0.937872376439989890f, -0.346980410845923680f, 0.937605969960999990f,
	-0.347699647819051380f,
	0.937339011912574960f, -0.348418680249434560f, 0.937071502451759190f,
	-0.349137507714084970f,
	0.936803441735921560f, -0.349856129790134920f, 0.936534829922755500f,
	-0.350574546054837510f,
	0.936265667170278260f, -0.351292756085567090f, 0.935995953636831410f,
	-0.352010759459819080f,
	0.935725689481080370f, -0.352728555755210730f, 0.935454874862014620f,
	-0.353446144549480810f,
	0.935183509938947610f, -0.354163525420490340f, 0.934911594871516090f,
	-0.354880697946222790f,
	0.934639129819680780f, -0.355597661704783850f, 0.934366114943725790f,
	-0.356314416274402410f,
	0.934092550404258980f, -0.357030961233429980f, 0.933818436362210960f,
	-0.357747296160341900f,
	0.933543772978836170f, -0.358463420633736540f, 0.933268560415712050f,
	-0.359179334232336500f,
	0.932992798834738960f, -0.359895036534988110f, 0.932716488398140250f,
	-0.360610527120662270f,
	0.932439629268462360f, -0.361325805568454280f, 0.932162221608574430f,
	-0.362040871457584180f,
	0.931884265581668150f, -0.362755724367397230f, 0.931605761351257830f,
	-0.363470363877363760f,
	0.931326709081180430f, -0.364184789567079890f, 0.931047108935595280f,
	-0.364899001016267320f,
	0.930766961078983710f, -0.365612997804773850f, 0.930486265676149780f,
	-0.366326779512573590f,
	0.930205022892219070f, -0.367040345719767180f, 0.929923232892639670f,
	-0.367753696006581980f,
	0.929640895843181330f, -0.368466829953372320f, 0.929358011909935500f,
	-0.369179747140620020f,
	0.929074581259315860f, -0.369892447148934100f, 0.928790604058057020f,
	-0.370604929559051670f,
	0.928506080473215590f, -0.371317193951837540f, 0.928221010672169440f,
	-0.372029239908285010f,
	0.927935394822617890f, -0.372741067009515760f, 0.927649233092581180f,
	-0.373452674836780300f,
	0.927362525650401110f, -0.374164062971457930f, 0.927075272664740100f,
	-0.374875230995057540f,
	0.926787474304581750f, -0.375586178489217220f, 0.926499130739230510f,
	-0.376296905035704790f,
	0.926210242138311380f, -0.377007410216418260f, 0.925920808671770070f,
	-0.377717693613385640f,
	0.925630830509872720f, -0.378427754808765560f, 0.925340307823206310f,
	-0.379137593384847320f,
	0.925049240782677580f, -0.379847208924051160f, 0.924757629559513910f,
	-0.380556601008928520f,
	0.924465474325262600f, -0.381265769222162380f, 0.924172775251791200f,
	-0.381974713146567220f,
	0.923879532511286740f, -0.382683432365089780f, 0.923585746276256670f,
	-0.383391926460808660f,
	0.923291416719527640f, -0.384100195016935040f, 0.922996544014246250f,
	-0.384808237616812880f,
	0.922701128333878630f, -0.385516053843918850f, 0.922405169852209880f,
	-0.386223643281862980f,
	0.922108668743345180f, -0.386931005514388580f, 0.921811625181708120f,
	-0.387638140125372730f,
	0.921514039342042010f, -0.388345046698826250f, 0.921215911399408730f,
	-0.389051724818894380f,
	0.920917241529189520f, -0.389758174069856410f, 0.920618029907083970f,
	-0.390464394036126590f,
	0.920318276709110590f, -0.391170384302253870f, 0.920017982111606570f,
	-0.391876144452922350f,
	0.919717146291227360f, -0.392581674072951470f, 0.919415769424947070f,
	-0.393286972747296400f,
	0.919113851690057770f, -0.393992040061048100f, 0.918811393264170050f,
	-0.394696875599433560f,
	0.918508394325212250f, -0.395401478947816350f, 0.918204855051430900f,
	-0.396105849691696270f,
	0.917900775621390500f, -0.396809987416710310f, 0.917596156213972950f,
	-0.397513891708632330f,
	0.917290997008377910f, -0.398217562153373560f, 0.916985298184123000f,
	-0.398920998336982910f,
	0.916679059921042700f, -0.399624199845646790f, 0.916372282399289140f,
	-0.400327166265690090f,
	0.916064965799331720f, -0.401029897183575620f, 0.915757110301956720f,
	-0.401732392185905010f,
	0.915448716088267830f, -0.402434650859418430f, 0.915139783339685260f,
	-0.403136672790995300f,
	0.914830312237946200f, -0.403838457567654070f, 0.914520302965104450f,
	-0.404540004776553000f,
	0.914209755703530690f, -0.405241314004989860f, 0.913898670635911680f,
	-0.405942384840402510f,
	0.913587047945250810f, -0.406643216870369030f, 0.913274887814867760f,
	-0.407343809682607970f,
	0.912962190428398210f, -0.408044162864978690f, 0.912648955969793900f,
	-0.408744276005481360f,
	0.912335184623322750f, -0.409444148692257590f, 0.912020876573568340f,
	-0.410143780513590240f,
	0.911706032005429880f, -0.410843171057903910f, 0.911390651104122430f,
	-0.411542319913765220f,
	0.911074734055176360f, -0.412241226669882890f, 0.910758281044437570f,
	-0.412939890915108080f,
	0.910441292258067250f, -0.413638312238434500f, 0.910123767882541680f,
	-0.414336490228999100f,
	0.909805708104652220f, -0.415034424476081630f, 0.909487113111505430f,
	-0.415732114569105360f,
	0.909167983090522380f, -0.416429560097637150f, 0.908848318229439120f,
	-0.417126760651387870f,
	0.908528118716306120f, -0.417823715820212270f, 0.908207384739488700f,
	-0.418520425194109700f,
	0.907886116487666260f, -0.419216888363223910f, 0.907564314149832630f,
	-0.419913104917843620f,
	0.907241977915295820f, -0.420609074448402510f, 0.906919107973678140f,
	-0.421304796545479640f,
	0.906595704514915330f, -0.422000270799799680f, 0.906271767729257660f,
	-0.422695496802232950f,
	0.905947297807268460f, -0.423390474143796050f, 0.905622294939825270f,
	-0.424085202415651560f,
	0.905296759318118820f, -0.424779681209108810f, 0.904970691133653250f,
	-0.425473910115623800f,
	0.904644090578246240f, -0.426167888726799620f, 0.904316957844028320f,
	-0.426861616634386430f,
	0.903989293123443340f, -0.427555093430282080f, 0.903661096609247980f,
	-0.428248318706531960f,
	0.903332368494511820f, -0.428941292055329490f, 0.903003108972617150f,
	-0.429634013069016380f,
	0.902673318237258830f, -0.430326481340082610f, 0.902342996482444200f,
	-0.431018696461167030f,
	0.902012143902493180f, -0.431710658025057260f, 0.901680760692037730f,
	-0.432402365624690140f,
	0.901348847046022030f, -0.433093818853151960f, 0.901016403159702330f,
	-0.433785017303678520f,
	0.900683429228646970f, -0.434475960569655650f, 0.900349925448735600f,
	-0.435166648244619260f,
	0.900015892016160280f, -0.435857079922255470f, 0.899681329127423930f,
	-0.436547255196401200f,
	0.899346236979341570f, -0.437237173661044090f, 0.899010615769039070f,
	-0.437926834910322860f,
	0.898674465693953820f, -0.438616238538527660f, 0.898337786951834310f,
	-0.439305384140099950f,
	0.898000579740739880f, -0.439994271309633260f, 0.897662844259040860f,
	-0.440682899641872900f,
	0.897324580705418320f, -0.441371268731716670f, 0.896985789278863970f,
	-0.442059378174214700f,
	0.896646470178680150f, -0.442747227564570020f, 0.896306623604479550f,
	-0.443434816498138480f,
	0.895966249756185220f, -0.444122144570429200f, 0.895625348834030110f,
	-0.444809211377104880f,
	0.895283921038557580f, -0.445496016513981740f, 0.894941966570620750f,
	-0.446182559577030070f,
	0.894599485631382700f, -0.446868840162374160f, 0.894256478422316040f,
	-0.447554857866293010f,
	0.893912945145203250f, -0.448240612285219890f, 0.893568886002135910f,
	-0.448926103015743260f,
	0.893224301195515320f, -0.449611329654606540f, 0.892879190928051680f,
	-0.450296291798708610f,
	0.892533555402764580f, -0.450980989045103860f, 0.892187394822982480f,
	-0.451665420991002490f,
	0.891840709392342720f, -0.452349587233770890f, 0.891493499314791380f,
	-0.453033487370931580f,
	0.891145764794583180f, -0.453717121000163870f, 0.890797506036281490f,
	-0.454400487719303580f,
	0.890448723244757880f, -0.455083587126343840f, 0.890099416625192320f,
	-0.455766418819434640f,
	0.889749586383072780f, -0.456448982396883920f, 0.889399232724195520f,
	-0.457131277457156980f,
	0.889048355854664570f, -0.457813303598877170f, 0.888696955980891600f,
	-0.458495060420826270f,
	0.888345033309596350f, -0.459176547521944090f, 0.887992588047805560f,
	-0.459857764501329540f,
	0.887639620402853930f, -0.460538710958240010f, 0.887286130582383150f,
	-0.461219386492092380f,
	0.886932118794342190f, -0.461899790702462730f, 0.886577585246987040f,
	-0.462579923189086810f,
	0.886222530148880640f, -0.463259783551860150f, 0.885866953708892790f,
	-0.463939371390838520f,
	0.885510856136199950f, -0.464618686306237820f, 0.885154237640285110f,
	-0.465297727898434600f,
	0.884797098430937790f, -0.465976495767966180f, 0.884439438718253810f,
	-0.466654989515530920f,
	0.884081258712634990f, -0.467333208741988420f, 0.883722558624789660f,
	-0.468011153048359830f,
	0.883363338665731580f, -0.468688822035827900f, 0.883003599046780830f,
	-0.469366215305737520f,
	0.882643339979562790f, -0.470043332459595620f, 0.882282561676008710f,
	-0.470720173099071600f,
	0.881921264348355050f, -0.471396736825997640f, 0.881559448209143780f,
	-0.472073023242368660f,
	0.881197113471222090f, -0.472749031950342790f, 0.880834260347742040f,
	-0.473424762552241530f,
	0.880470889052160750f, -0.474100214650549970f, 0.880106999798240360f,
	-0.474775387847917120f,
	0.879742592800047410f, -0.475450281747155870f, 0.879377668271953290f,
	-0.476124895951243580f,
	0.879012226428633530f, -0.476799230063322090f, 0.878646267485068130f,
	-0.477473283686698060f,
	0.878279791656541580f, -0.478147056424843010f, 0.877912799158641840f,
	-0.478820547881393890f,
	0.877545290207261350f, -0.479493757660153010f, 0.877177265018595940f,
	-0.480166685365088390f,
	0.876808723809145650f, -0.480839330600333960f, 0.876439666795713610f,
	-0.481511692970189860f,
	0.876070094195406600f, -0.482183772079122720f, 0.875700006225634600f,
	-0.482855567531765670f,
	0.875329403104110890f, -0.483527078932918740f, 0.874958285048851650f,
	-0.484198305887549030f,
	0.874586652278176110f, -0.484869248000791060f, 0.874214505010706300f,
	-0.485539904877946960f,
	0.873841843465366860f, -0.486210276124486420f, 0.873468667861384880f,
	-0.486880361346047340f,
	0.873094978418290090f, -0.487550160148436000f, 0.872720775355914300f,
	-0.488219672137626790f,
	0.872346058894391540f, -0.488888896919763170f, 0.871970829254157810f,
	-0.489557834101157440f,
	0.871595086655950980f, -0.490226483288291160f, 0.871218831320811020f,
	-0.490894844087815090f,
	0.870842063470078980f, -0.491562916106549900f, 0.870464783325397670f,
	-0.492230698951486020f,
	0.870086991108711460f, -0.492898192229784040f, 0.869708687042265670f,
	-0.493565395548774770f,
	0.869329871348606840f, -0.494232308515959670f, 0.868950544250582380f,
	-0.494898930739011260f,
	0.868570705971340900f, -0.495565261825772540f, 0.868190356734331310f,
	-0.496231301384258250f,
	0.867809496763303320f, -0.496897049022654470f, 0.867428126282306920f,
	-0.497562504349319150f,
	0.867046245515692650f, -0.498227666972781870f, 0.866663854688111130f,
	-0.498892536501744590f,
	0.866280954024512990f, -0.499557112545081840f, 0.865897543750148820f,
	-0.500221394711840680f,
	0.865513624090569090f, -0.500885382611240710f, 0.865129195271623800f,
	-0.501549075852675390f,
	0.864744257519462380f, -0.502212474045710790f, 0.864358811060534030f,
	-0.502875576800086990f,
	0.863972856121586810f, -0.503538383725717580f, 0.863586392929668100f,
	-0.504200894432690340f,
	0.863199421712124160f, -0.504863108531267590f, 0.862811942696600330f,
	-0.505525025631885390f,
	0.862423956111040610f, -0.506186645345155230f, 0.862035462183687210f,
	-0.506847967281863210f,
	0.861646461143081300f, -0.507508991052970870f, 0.861256953218062170f,
	-0.508169716269614600f,
	0.860866938637767310f, -0.508830142543106990f, 0.860476417631632070f,
	-0.509490269484936360f,
	0.860085390429390140f, -0.510150096706766810f, 0.859693857261072610f,
	-0.510809623820439040f,
	0.859301818357008470f, -0.511468850437970300f, 0.858909273947823900f,
	-0.512127776171554690f,
	0.858516224264442740f, -0.512786400633562960f, 0.858122669538086140f,
	-0.513444723436543460f,
	0.857728610000272120f, -0.514102744193221660f, 0.857334045882815590f,
	-0.514760462516501200f,
	0.856938977417828760f, -0.515417878019462930f, 0.856543404837719960f,
	-0.516074990315366630f,
	0.856147328375194470f, -0.516731799017649870f, 0.855750748263253920f,
	-0.517388303739929060f,
	0.855353664735196030f, -0.518044504095999340f, 0.854956078024614930f,
	-0.518700399699834950f,
	0.854557988365400530f, -0.519355990165589640f, 0.854159395991738850f,
	-0.520011275107596040f,
	0.853760301138111410f, -0.520666254140367160f, 0.853360704039295430f,
	-0.521320926878595660f,
	0.852960604930363630f, -0.521975292937154390f, 0.852560004046684080f,
	-0.522629351931096610f,
	0.852158901623919830f, -0.523283103475656430f, 0.851757297898029120f,
	-0.523936547186248600f,
	0.851355193105265200f, -0.524589682678468950f, 0.850952587482175730f,
	-0.525242509568094710f,
	0.850549481265603480f, -0.525895027471084630f, 0.850145874692685210f,
	-0.526547236003579440f,
	0.849741768000852550f, -0.527199134781901280f, 0.849337161427830780f,
	-0.527850723422555230f,
	0.848932055211639610f, -0.528502001542228480f, 0.848526449590592650f,
	-0.529152968757790610f,
	0.848120344803297230f, -0.529803624686294610f, 0.847713741088654380f,
	-0.530453968944976320f,
	0.847306638685858320f, -0.531104001151255000f, 0.846899037834397240f,
	-0.531753720922733320f,
	0.846490938774052130f, -0.532403127877197900f, 0.846082341744897050f,
	-0.533052221632619450f,
	0.845673246987299070f, -0.533701001807152960f, 0.845263654741918220f,
	-0.534349468019137520f,
	0.844853565249707120f, -0.534997619887097150f, 0.844442978751910660f,
	-0.535645457029741090f,
	0.844031895490066410f, -0.536292979065963180f, 0.843620315706004150f,
	-0.536940185614842910f,
	0.843208239641845440f, -0.537587076295645390f, 0.842795667540004120f,
	-0.538233650727821700f,
	0.842382599643185850f, -0.538879908531008420f, 0.841969036194387680f,
	-0.539525849325028890f,
	0.841554977436898440f, -0.540171472729892850f, 0.841140423614298080f,
	-0.540816778365796670f,
	0.840725374970458070f, -0.541461765853123440f, 0.840309831749540770f,
	-0.542106434812443920f,
	0.839893794195999520f, -0.542750784864515890f, 0.839477262554578550f,
	-0.543394815630284800f,
	0.839060237070312740f, -0.544038526730883820f, 0.838642717988527300f,
	-0.544681917787634530f,
	0.838224705554838080f, -0.545324988422046460f, 0.837806200015150940f,
	-0.545967738255817570f,
	0.837387201615661940f, -0.546610166910834860f, 0.836967710602857020f,
	-0.547252274009174090f,
	0.836547727223512010f, -0.547894059173100190f, 0.836127251724692270f,
	-0.548535522025067390f,
	0.835706284353752600f, -0.549176662187719660f, 0.835284825358337370f,
	-0.549817479283890910f,
	0.834862874986380010f, -0.550457972936604810f, 0.834440433486103190f,
	-0.551098142769075430f,
	0.834017501106018130f, -0.551737988404707340f, 0.833594078094925140f,
	-0.552377509467096070f,
	0.833170164701913190f, -0.553016705580027470f, 0.832745761176359460f,
	-0.553655576367479310f,
	0.832320867767929680f, -0.554294121453620000f, 0.831895484726577590f,
	-0.554932340462810370f,
	0.831469612302545240f, -0.555570233019602180f, 0.831043250746362320f,
	-0.556207798748739930f,
	0.830616400308846310f, -0.556845037275160100f, 0.830189061241102370f,
	-0.557481948223991550f,
	0.829761233794523050f, -0.558118531220556100f, 0.829332918220788250f,
	-0.558754785890368310f,
	0.828904114771864870f, -0.559390711859136140f, 0.828474823700007130f,
	-0.560026308752760380f,
	0.828045045257755800f, -0.560661576197336030f, 0.827614779697938400f,
	-0.561296513819151470f,
	0.827184027273669130f, -0.561931121244689470f, 0.826752788238348520f,
	-0.562565398100626560f,
	0.826321062845663530f, -0.563199344013834090f, 0.825888851349586780f,
	-0.563832958611378170f,
	0.825456154004377550f, -0.564466241520519500f, 0.825022971064580220f,
	-0.565099192368713980f,
	0.824589302785025290f, -0.565731810783613120f, 0.824155149420828570f,
	-0.566364096393063840f,
	0.823720511227391430f, -0.566996048825108680f, 0.823285388460400110f,
	-0.567627667707986230f,
	0.822849781375826430f, -0.568258952670131490f, 0.822413690229926390f,
	-0.568889903340175860f,
	0.821977115279241550f, -0.569520519346947140f, 0.821540056780597610f,
	-0.570150800319470300f,
	0.821102514991104650f, -0.570780745886967260f, 0.820664490168157460f,
	-0.571410355678857230f,
	0.820225982569434690f, -0.572039629324757050f, 0.819786992452898990f,
	-0.572668566454481160f,
	0.819347520076796900f, -0.573297166698042200f, 0.818907565699658950f,
	-0.573925429685650750f,
	0.818467129580298660f, -0.574553355047715760f, 0.818026211977813440f,
	-0.575180942414845080f,
	0.817584813151583710f, -0.575808191417845340f, 0.817142933361272970f,
	-0.576435101687721830f,
	0.816700572866827850f, -0.577061672855679440f, 0.816257731928477390f,
	-0.577687904553122800f,
	0.815814410806733780f, -0.578313796411655590f, 0.815370609762391290f,
	-0.578939348063081780f,
	0.814926329056526620f, -0.579564559139405630f, 0.814481568950498610f,
	-0.580189429272831680f,
	0.814036329705948410f, -0.580813958095764530f, 0.813590611584798510f,
	-0.581438145240810170f,
	0.813144414849253590f, -0.582061990340775440f, 0.812697739761799490f,
	-0.582685493028668460f,
	0.812250586585203880f, -0.583308652937698290f, 0.811802955582515470f,
	-0.583931469701276180f,
	0.811354847017063730f, -0.584553942953015330f, 0.810906261152459670f,
	-0.585176072326730410f,
	0.810457198252594770f, -0.585797857456438860f, 0.810007658581641140f,
	-0.586419297976360500f,
	0.809557642404051260f, -0.587040393520917970f, 0.809107149984558240f,
	-0.587661143724736660f,
	0.808656181588174980f, -0.588281548222645220f, 0.808204737480194720f,
	-0.588901606649675720f,
	0.807752817926190360f, -0.589521318641063940f, 0.807300423192014450f,
	-0.590140683832248820f,
	0.806847553543799330f, -0.590759701858874160f, 0.806394209247956240f,
	-0.591378372356787580f,
	0.805940390571176280f, -0.591996694962040990f, 0.805486097780429230f,
	-0.592614669310891130f,
	0.805031331142963660f, -0.593232295039799800f, 0.804576090926307110f,
	-0.593849571785433630f,
	0.804120377398265810f, -0.594466499184664430f, 0.803664190826924090f,
	-0.595083076874569960f,
	0.803207531480644940f, -0.595699304492433360f, 0.802750399628069160f,
	-0.596315181675743710f,
	0.802292795538115720f, -0.596930708062196500f, 0.801834719479981310f,
	-0.597545883289693160f,
	0.801376171723140240f, -0.598160706996342270f, 0.800917152537344300f,
	-0.598775178820458720f,
	0.800457662192622820f, -0.599389298400564540f, 0.799997700959281910f,
	-0.600003065375388940f,
	0.799537269107905010f, -0.600616479383868970f, 0.799076366909352350f,
	-0.601229540065148500f,
	0.798614994634760820f, -0.601842247058580030f, 0.798153152555543750f,
	-0.602454600003723750f,
	0.797690840943391160f, -0.603066598540348160f, 0.797228060070268810f,
	-0.603678242308430370f,
	0.796764810208418830f, -0.604289530948155960f, 0.796301091630359110f,
	-0.604900464099919820f,
	0.795836904608883570f, -0.605511041404325550f, 0.795372249417061310f,
	-0.606121262502186120f,
	0.794907126328237010f, -0.606731127034524480f, 0.794441535616030590f,
	-0.607340634642572930f,
	0.793975477554337170f, -0.607949784967773630f, 0.793508952417326660f,
	-0.608558577651779450f,
	0.793041960479443640f, -0.609167012336453210f, 0.792574502015407690f,
	-0.609775088663868430f,
	0.792106577300212390f, -0.610382806276309480f, 0.791638186609125880f,
	-0.610990164816271660f,
	0.791169330217690200f, -0.611597163926461910f, 0.790700008401721610f,
	-0.612203803249797950f,
	0.790230221437310030f, -0.612810082429409710f, 0.789759969600819070f,
	-0.613416001108638590f,
	0.789289253168885650f, -0.614021558931038380f, 0.788818072418420280f,
	-0.614626755540375050f,
	0.788346427626606340f, -0.615231590580626820f, 0.787874319070900220f,
	-0.615836063695985090f,
	0.787401747029031430f, -0.616440174530853650f, 0.786928711779001810f,
	-0.617043922729849760f,
	0.786455213599085770f, -0.617647307937803870f, 0.785981252767830150f,
	-0.618250329799760250f,
	0.785506829564053930f, -0.618852987960976320f, 0.785031944266848080f,
	-0.619455282066924020f,
	0.784556597155575240f, -0.620057211763289100f, 0.784080788509869950f,
	-0.620658776695972140f,
	0.783604518609638200f, -0.621259976511087550f, 0.783127787735057310f,
	-0.621860810854965360f,
	0.782650596166575730f, -0.622461279374149970f, 0.782172944184913010f,
	-0.623061381715401260f,
	0.781694832071059390f, -0.623661117525694530f, 0.781216260106276090f,
	-0.624260486452220650f,
	0.780737228572094490f, -0.624859488142386340f, 0.780257737750316590f,
	-0.625458122243814360f,
	0.779777787923014550f, -0.626056388404343520f, 0.779297379372530300f,
	-0.626654286272029350f,
	0.778816512381475980f, -0.627251815495144080f, 0.778335187232733210f,
	-0.627848975722176460f,
	0.777853404209453150f, -0.628445766601832710f, 0.777371163595056310f,
	-0.629042187783036000f,
	0.776888465673232440f, -0.629638238914926980f, 0.776405310727940390f,
	-0.630233919646864370f,
	0.775921699043407690f, -0.630829229628424470f, 0.775437630904130540f,
	-0.631424168509401860f,
	0.774953106594873930f, -0.632018735939809060f, 0.774468126400670860f,
	-0.632612931569877410f,
	0.773982690606822900f, -0.633206755050057190f, 0.773496799498899050f,
	-0.633800206031017280f,
	0.773010453362736990f, -0.634393284163645490f, 0.772523652484441330f,
	-0.634985989099049460f,
	0.772036397150384520f, -0.635578320488556110f, 0.771548687647206300f,
	-0.636170277983712170f,
	0.771060524261813820f, -0.636761861236284200f, 0.770571907281380810f,
	-0.637353069898259130f,
	0.770082836993347900f, -0.637943903621844060f, 0.769593313685422940f,
	-0.638534362059466790f,
	0.769103337645579700f, -0.639124444863775730f, 0.768612909162058380f,
	-0.639714151687640450f,
	0.768122028523365420f, -0.640303482184151670f, 0.767630696018273380f,
	-0.640892436006621380f,
	0.767138911935820400f, -0.641481012808583160f, 0.766646676565310380f,
	-0.642069212243792540f,
	0.766153990196312920f, -0.642657033966226860f, 0.765660853118662500f,
	-0.643244477630085850f,
	0.765167265622458960f, -0.643831542889791390f, 0.764673227998067140f,
	-0.644418229399988380f,
	0.764178740536116670f, -0.645004536815543930f, 0.763683803527501870f,
	-0.645590464791548690f,
	0.763188417263381270f, -0.646176012983316280f, 0.762692582035177980f,
	-0.646761181046383920f,
	0.762196298134578900f, -0.647345968636512060f, 0.761699565853535380f,
	-0.647930375409685340f,
	0.761202385484261780f, -0.648514401022112440f, 0.760704757319236920f,
	-0.649098045130225950f,
	0.760206681651202420f, -0.649681307390683190f, 0.759708158773163440f,
	-0.650264187460365850f,
	0.759209188978388070f, -0.650846684996380880f, 0.758709772560407390f,
	-0.651428799656059820f,
	0.758209909813015280f, -0.652010531096959500f, 0.757709601030268080f,
	-0.652591878976862440f,
	0.757208846506484570f, -0.653172842953776760f, 0.756707646536245670f,
	-0.653753422685936060f,
	0.756206001414394540f, -0.654333617831800440f, 0.755703911436035880f,
	-0.654913428050056030f,
	0.755201376896536550f, -0.655492852999615350f, 0.754698398091524500f,
	-0.656071892339617600f,
	0.754194975316889170f, -0.656650545729428940f, 0.753691108868781210f,
	-0.657228812828642540f,
	0.753186799043612520f, -0.657806693297078640f, 0.752682046138055340f,
	-0.658384186794785050f,
	0.752176850449042810f, -0.658961292982037320f, 0.751671212273768430f,
	-0.659538011519338660f,
	0.751165131909686480f, -0.660114342067420480f, 0.750658609654510700f,
	-0.660690284287242300f,
	0.750151645806215070f, -0.661265837839992270f, 0.749644240663033480f,
	-0.661841002387086870f,
	0.749136394523459370f, -0.662415777590171780f, 0.748628107686245440f,
	-0.662990163111121470f,
	0.748119380450403600f, -0.663564158612039770f, 0.747610213115205150f,
	-0.664137763755260010f,
	0.747100605980180130f, -0.664710978203344790f, 0.746590559345117310f,
	-0.665283801619087180f,
	0.746080073510063780f, -0.665856233665509720f, 0.745569148775325430f,
	-0.666428274005865240f,
	0.745057785441466060f, -0.666999922303637470f, 0.744545983809307370f,
	-0.667571178222540310f,
	0.744033744179929290f, -0.668142041426518450f, 0.743521066854669120f,
	-0.668712511579747980f,
	0.743007952135121720f, -0.669282588346636010f, 0.742494400323139180f,
	-0.669852271391821020f,
	0.741980411720831070f, -0.670421560380173090f, 0.741465986630563290f,
	-0.670990454976794220f,
	0.740951125354959110f, -0.671558954847018330f, 0.740435828196898020f,
	-0.672127059656411730f,
	0.739920095459516200f, -0.672694769070772860f, 0.739403927446205760f,
	-0.673262082756132970f,
	0.738887324460615110f, -0.673829000378756040f, 0.738370286806648620f,
	-0.674395521605139050f,
	0.737852814788465980f, -0.674961646102011930f, 0.737334908710482910f,
	-0.675527373536338520f,
	0.736816568877369900f, -0.676092703575315920f, 0.736297795594053170f,
	-0.676657635886374950f,
	0.735778589165713590f, -0.677222170137180330f, 0.735258949897786840f,
	-0.677786305995631500f,
	0.734738878095963500f, -0.678350043129861470f, 0.734218374066188280f,
	-0.678913381208238410f,
	0.733697438114660370f, -0.679476319899364970f, 0.733176070547832740f,
	-0.680038858872078930f,
	0.732654271672412820f, -0.680600997795453020f, 0.732132041795361290f,
	-0.681162736338795430f,
	0.731609381223892630f, -0.681724074171649710f, 0.731086290265474340f,
	-0.682285010963795570f,
	0.730562769227827590f, -0.682845546385248080f, 0.730038818418926260f,
	-0.683405680106258680f,
	0.729514438146997010f, -0.683965411797315400f, 0.728989628720519420f,
	-0.684524741129142300f,
	0.728464390448225200f, -0.685083667772700360f, 0.727938723639098620f,
	-0.685642191399187470f,
	0.727412628602375770f, -0.686200311680038590f, 0.726886105647544970f,
	-0.686758028286925890f,
	0.726359155084346010f, -0.687315340891759050f, 0.725831777222770370f,
	-0.687872249166685550f,
	0.725303972373060770f, -0.688428752784090440f, 0.724775740845711280f,
	-0.688984851416597040f,
	0.724247082951467000f, -0.689540544737066830f, 0.723717999001323500f,
	-0.690095832418599950f,
	0.723188489306527460f, -0.690650714134534600f, 0.722658554178575610f,
	-0.691205189558448450f,
	0.722128193929215350f, -0.691759258364157750f, 0.721597408870443770f,
	-0.692312920225718220f,
	0.721066199314508110f, -0.692866174817424630f, 0.720534565573905270f,
	-0.693419021813811760f,
	0.720002507961381650f, -0.693971460889654000f, 0.719470026789932990f,
	-0.694523491719965520f,
	0.718937122372804490f, -0.695075113980000880f, 0.718403795023489830f,
	-0.695626327345254870f,
	0.717870045055731710f, -0.696177131491462990f, 0.717335872783521730f,
	-0.696727526094601200f,
	0.716801278521099540f, -0.697277510830886520f, 0.716266262582953120f,
	-0.697827085376777290f,
	0.715730825283818590f, -0.698376249408972920f, 0.715194966938680120f,
	-0.698925002604414150f,
	0.714658687862769090f, -0.699473344640283770f, 0.714121988371564820f,
	-0.700021275194006250f,
	0.713584868780793640f, -0.700568793943248340f, 0.713047329406429340f,
	-0.701115900565918660f,
	0.712509370564692320f, -0.701662594740168450f, 0.711970992572050100f,
	-0.702208876144391870f,
	0.711432195745216430f, -0.702754744457225300f, 0.710892980401151680f,
	-0.703300199357548730f,
	0.710353346857062420f, -0.703845240524484940f, 0.709813295430400840f,
	-0.704389867637400410f,
	0.709272826438865690f, -0.704934080375904880f, 0.708731940200400650f,
	-0.705477878419852100f,
	0.708190637033195400f, -0.706021261449339740f, 0.707648917255684350f,
	-0.706564229144709510f,
	0.707106781186547570f, -0.707106781186547460f, 0.706564229144709620f,
	-0.707648917255684350f,
	0.706021261449339740f, -0.708190637033195290f, 0.705477878419852210f,
	-0.708731940200400650f,
	0.704934080375904990f, -0.709272826438865580f, 0.704389867637400410f,
	-0.709813295430400840f,
	0.703845240524484940f, -0.710353346857062310f, 0.703300199357548730f,
	-0.710892980401151680f,
	0.702754744457225300f, -0.711432195745216430f, 0.702208876144391870f,
	-0.711970992572049990f,
	0.701662594740168570f, -0.712509370564692320f, 0.701115900565918660f,
	-0.713047329406429230f,
	0.700568793943248450f, -0.713584868780793520f, 0.700021275194006360f,
	-0.714121988371564710f,
	0.699473344640283770f, -0.714658687862768980f, 0.698925002604414150f,
	-0.715194966938680010f,
	0.698376249408972920f, -0.715730825283818590f, 0.697827085376777290f,
	-0.716266262582953120f,
	0.697277510830886630f, -0.716801278521099540f, 0.696727526094601200f,
	-0.717335872783521730f,
	0.696177131491462990f, -0.717870045055731710f, 0.695626327345254870f,
	-0.718403795023489720f,
	0.695075113980000880f, -0.718937122372804380f, 0.694523491719965520f,
	-0.719470026789932990f,
	0.693971460889654000f, -0.720002507961381650f, 0.693419021813811880f,
	-0.720534565573905270f,
	0.692866174817424740f, -0.721066199314508110f, 0.692312920225718220f,
	-0.721597408870443660f,
	0.691759258364157750f, -0.722128193929215350f, 0.691205189558448450f,
	-0.722658554178575610f,
	0.690650714134534720f, -0.723188489306527350f, 0.690095832418599950f,
	-0.723717999001323390f,
	0.689540544737066940f, -0.724247082951466890f, 0.688984851416597150f,
	-0.724775740845711280f,
	0.688428752784090550f, -0.725303972373060660f, 0.687872249166685550f,
	-0.725831777222770370f,
	0.687315340891759160f, -0.726359155084346010f, 0.686758028286925890f,
	-0.726886105647544970f,
	0.686200311680038700f, -0.727412628602375770f, 0.685642191399187470f,
	-0.727938723639098620f,
	0.685083667772700360f, -0.728464390448225200f, 0.684524741129142300f,
	-0.728989628720519310f,
	0.683965411797315510f, -0.729514438146996900f, 0.683405680106258790f,
	-0.730038818418926150f,
	0.682845546385248080f, -0.730562769227827590f, 0.682285010963795570f,
	-0.731086290265474230f,
	0.681724074171649820f, -0.731609381223892520f, 0.681162736338795430f,
	-0.732132041795361290f,
	0.680600997795453130f, -0.732654271672412820f, 0.680038858872079040f,
	-0.733176070547832740f,
	0.679476319899365080f, -0.733697438114660260f, 0.678913381208238410f,
	-0.734218374066188170f,
	0.678350043129861580f, -0.734738878095963390f, 0.677786305995631500f,
	-0.735258949897786730f,
	0.677222170137180450f, -0.735778589165713480f, 0.676657635886374950f,
	-0.736297795594053060f,
	0.676092703575316030f, -0.736816568877369790f, 0.675527373536338630f,
	-0.737334908710482790f,
	0.674961646102012040f, -0.737852814788465980f, 0.674395521605139050f,
	-0.738370286806648510f,
	0.673829000378756150f, -0.738887324460615110f, 0.673262082756132970f,
	-0.739403927446205760f,
	0.672694769070772970f, -0.739920095459516090f, 0.672127059656411840f,
	-0.740435828196898020f,
	0.671558954847018330f, -0.740951125354959110f, 0.670990454976794220f,
	-0.741465986630563290f,
	0.670421560380173090f, -0.741980411720830960f, 0.669852271391821130f,
	-0.742494400323139180f,
	0.669282588346636010f, -0.743007952135121720f, 0.668712511579748090f,
	-0.743521066854669120f,
	0.668142041426518560f, -0.744033744179929180f, 0.667571178222540310f,
	-0.744545983809307250f,
	0.666999922303637470f, -0.745057785441465950f, 0.666428274005865350f,
	-0.745569148775325430f,
	0.665856233665509720f, -0.746080073510063780f, 0.665283801619087180f,
	-0.746590559345117310f,
	0.664710978203344900f, -0.747100605980180130f, 0.664137763755260010f,
	-0.747610213115205150f,
	0.663564158612039880f, -0.748119380450403490f, 0.662990163111121470f,
	-0.748628107686245330f,
	0.662415777590171780f, -0.749136394523459260f, 0.661841002387086870f,
	-0.749644240663033480f,
	0.661265837839992270f, -0.750151645806214960f, 0.660690284287242300f,
	-0.750658609654510590f,
	0.660114342067420480f, -0.751165131909686370f, 0.659538011519338770f,
	-0.751671212273768430f,
	0.658961292982037320f, -0.752176850449042700f, 0.658384186794785050f,
	-0.752682046138055230f,
	0.657806693297078640f, -0.753186799043612410f, 0.657228812828642650f,
	-0.753691108868781210f,
	0.656650545729429050f, -0.754194975316889170f, 0.656071892339617710f,
	-0.754698398091524390f,
	0.655492852999615460f, -0.755201376896536550f, 0.654913428050056150f,
	-0.755703911436035880f,
	0.654333617831800550f, -0.756206001414394540f, 0.653753422685936170f,
	-0.756707646536245670f,
	0.653172842953776760f, -0.757208846506484460f, 0.652591878976862550f,
	-0.757709601030268080f,
	0.652010531096959500f, -0.758209909813015280f, 0.651428799656059820f,
	-0.758709772560407390f,
	0.650846684996380990f, -0.759209188978387960f, 0.650264187460365960f,
	-0.759708158773163440f,
	0.649681307390683190f, -0.760206681651202420f, 0.649098045130226060f,
	-0.760704757319236920f,
	0.648514401022112550f, -0.761202385484261780f, 0.647930375409685460f,
	-0.761699565853535270f,
	0.647345968636512060f, -0.762196298134578900f, 0.646761181046383920f,
	-0.762692582035177870f,
	0.646176012983316390f, -0.763188417263381270f, 0.645590464791548800f,
	-0.763683803527501870f,
	0.645004536815544040f, -0.764178740536116670f, 0.644418229399988380f,
	-0.764673227998067140f,
	0.643831542889791500f, -0.765167265622458960f, 0.643244477630085850f,
	-0.765660853118662390f,
	0.642657033966226860f, -0.766153990196312810f, 0.642069212243792540f,
	-0.766646676565310380f,
	0.641481012808583160f, -0.767138911935820400f, 0.640892436006621380f,
	-0.767630696018273270f,
	0.640303482184151670f, -0.768122028523365310f, 0.639714151687640450f,
	-0.768612909162058270f,
	0.639124444863775730f, -0.769103337645579590f, 0.638534362059466790f,
	-0.769593313685422940f,
	0.637943903621844170f, -0.770082836993347900f, 0.637353069898259130f,
	-0.770571907281380700f,
	0.636761861236284200f, -0.771060524261813710f, 0.636170277983712170f,
	-0.771548687647206300f,
	0.635578320488556230f, -0.772036397150384410f, 0.634985989099049460f,
	-0.772523652484441330f,
	0.634393284163645490f, -0.773010453362736990f, 0.633800206031017280f,
	-0.773496799498899050f,
	0.633206755050057190f, -0.773982690606822790f, 0.632612931569877520f,
	-0.774468126400670860f,
	0.632018735939809060f, -0.774953106594873820f, 0.631424168509401860f,
	-0.775437630904130430f,
	0.630829229628424470f, -0.775921699043407580f, 0.630233919646864480f,
	-0.776405310727940390f,
	0.629638238914927100f, -0.776888465673232440f, 0.629042187783036000f,
	-0.777371163595056200f,
	0.628445766601832710f, -0.777853404209453040f, 0.627848975722176570f,
	-0.778335187232733090f,
	0.627251815495144190f, -0.778816512381475870f, 0.626654286272029460f,
	-0.779297379372530300f,
	0.626056388404343520f, -0.779777787923014440f, 0.625458122243814360f,
	-0.780257737750316590f,
	0.624859488142386450f, -0.780737228572094380f, 0.624260486452220650f,
	-0.781216260106276090f,
	0.623661117525694640f, -0.781694832071059390f, 0.623061381715401370f,
	-0.782172944184912900f,
	0.622461279374150080f, -0.782650596166575730f, 0.621860810854965360f,
	-0.783127787735057310f,
	0.621259976511087660f, -0.783604518609638200f, 0.620658776695972140f,
	-0.784080788509869950f,
	0.620057211763289210f, -0.784556597155575240f, 0.619455282066924020f,
	-0.785031944266848080f,
	0.618852987960976320f, -0.785506829564053930f, 0.618250329799760250f,
	-0.785981252767830150f,
	0.617647307937803980f, -0.786455213599085770f, 0.617043922729849760f,
	-0.786928711779001700f,
	0.616440174530853650f, -0.787401747029031320f, 0.615836063695985090f,
	-0.787874319070900110f,
	0.615231590580626820f, -0.788346427626606230f, 0.614626755540375050f,
	-0.788818072418420170f,
	0.614021558931038490f, -0.789289253168885650f, 0.613416001108638590f,
	-0.789759969600819070f,
	0.612810082429409710f, -0.790230221437310030f, 0.612203803249798060f,
	-0.790700008401721610f,
	0.611597163926462020f, -0.791169330217690090f, 0.610990164816271770f,
	-0.791638186609125770f,
	0.610382806276309480f, -0.792106577300212390f, 0.609775088663868430f,
	-0.792574502015407580f,
	0.609167012336453210f, -0.793041960479443640f, 0.608558577651779450f,
	-0.793508952417326660f,
	0.607949784967773740f, -0.793975477554337170f, 0.607340634642572930f,
	-0.794441535616030590f,
	0.606731127034524480f, -0.794907126328237010f, 0.606121262502186230f,
	-0.795372249417061190f,
	0.605511041404325550f, -0.795836904608883460f, 0.604900464099919930f,
	-0.796301091630359110f,
	0.604289530948156070f, -0.796764810208418720f, 0.603678242308430370f,
	-0.797228060070268700f,
	0.603066598540348280f, -0.797690840943391040f, 0.602454600003723860f,
	-0.798153152555543750f,
	0.601842247058580030f, -0.798614994634760820f, 0.601229540065148620f,
	-0.799076366909352350f,
	0.600616479383868970f, -0.799537269107905010f, 0.600003065375389060f,
	-0.799997700959281910f,
	0.599389298400564540f, -0.800457662192622710f, 0.598775178820458720f,
	-0.800917152537344300f,
	0.598160706996342380f, -0.801376171723140130f, 0.597545883289693270f,
	-0.801834719479981310f,
	0.596930708062196500f, -0.802292795538115720f, 0.596315181675743820f,
	-0.802750399628069160f,
	0.595699304492433470f, -0.803207531480644830f, 0.595083076874569960f,
	-0.803664190826924090f,
	0.594466499184664540f, -0.804120377398265700f, 0.593849571785433630f,
	-0.804576090926307000f,
	0.593232295039799800f, -0.805031331142963660f, 0.592614669310891130f,
	-0.805486097780429120f,
	0.591996694962040990f, -0.805940390571176280f, 0.591378372356787580f,
	-0.806394209247956240f,
	0.590759701858874280f, -0.806847553543799220f, 0.590140683832248940f,
	-0.807300423192014450f,
	0.589521318641063940f, -0.807752817926190360f, 0.588901606649675840f,
	-0.808204737480194720f,
	0.588281548222645330f, -0.808656181588174980f, 0.587661143724736770f,
	-0.809107149984558130f,
	0.587040393520918080f, -0.809557642404051260f, 0.586419297976360500f,
	-0.810007658581641140f,
	0.585797857456438860f, -0.810457198252594770f, 0.585176072326730410f,
	-0.810906261152459670f,
	0.584553942953015330f, -0.811354847017063730f, 0.583931469701276300f,
	-0.811802955582515360f,
	0.583308652937698290f, -0.812250586585203880f, 0.582685493028668460f,
	-0.812697739761799490f,
	0.582061990340775550f, -0.813144414849253590f, 0.581438145240810280f,
	-0.813590611584798510f,
	0.580813958095764530f, -0.814036329705948300f, 0.580189429272831680f,
	-0.814481568950498610f,
	0.579564559139405740f, -0.814926329056526620f, 0.578939348063081890f,
	-0.815370609762391290f,
	0.578313796411655590f, -0.815814410806733780f, 0.577687904553122800f,
	-0.816257731928477390f,
	0.577061672855679550f, -0.816700572866827850f, 0.576435101687721830f,
	-0.817142933361272970f,
	0.575808191417845340f, -0.817584813151583710f, 0.575180942414845190f,
	-0.818026211977813440f,
	0.574553355047715760f, -0.818467129580298660f, 0.573925429685650750f,
	-0.818907565699658950f,
	0.573297166698042320f, -0.819347520076796900f, 0.572668566454481160f,
	-0.819786992452898990f,
	0.572039629324757050f, -0.820225982569434690f, 0.571410355678857340f,
	-0.820664490168157460f,
	0.570780745886967370f, -0.821102514991104650f, 0.570150800319470300f,
	-0.821540056780597610f,
	0.569520519346947250f, -0.821977115279241550f, 0.568889903340175970f,
	-0.822413690229926390f,
	0.568258952670131490f, -0.822849781375826320f, 0.567627667707986230f,
	-0.823285388460400110f,
	0.566996048825108680f, -0.823720511227391320f, 0.566364096393063950f,
	-0.824155149420828570f,
	0.565731810783613230f, -0.824589302785025290f, 0.565099192368714090f,
	-0.825022971064580220f,
	0.564466241520519500f, -0.825456154004377440f, 0.563832958611378170f,
	-0.825888851349586780f,
	0.563199344013834090f, -0.826321062845663420f, 0.562565398100626560f,
	-0.826752788238348520f,
	0.561931121244689470f, -0.827184027273669020f, 0.561296513819151470f,
	-0.827614779697938400f,
	0.560661576197336030f, -0.828045045257755800f, 0.560026308752760380f,
	-0.828474823700007130f,
	0.559390711859136140f, -0.828904114771864870f, 0.558754785890368310f,
	-0.829332918220788250f,
	0.558118531220556100f, -0.829761233794523050f, 0.557481948223991660f,
	-0.830189061241102370f,
	0.556845037275160100f, -0.830616400308846200f, 0.556207798748739930f,
	-0.831043250746362320f,
	0.555570233019602290f, -0.831469612302545240f, 0.554932340462810370f,
	-0.831895484726577590f,
	0.554294121453620110f, -0.832320867767929680f, 0.553655576367479310f,
	-0.832745761176359460f,
	0.553016705580027580f, -0.833170164701913190f, 0.552377509467096070f,
	-0.833594078094925140f,
	0.551737988404707450f, -0.834017501106018130f, 0.551098142769075430f,
	-0.834440433486103190f,
	0.550457972936604810f, -0.834862874986380010f, 0.549817479283891020f,
	-0.835284825358337370f,
	0.549176662187719770f, -0.835706284353752600f, 0.548535522025067390f,
	-0.836127251724692160f,
	0.547894059173100190f, -0.836547727223511890f, 0.547252274009174090f,
	-0.836967710602857020f,
	0.546610166910834860f, -0.837387201615661940f, 0.545967738255817680f,
	-0.837806200015150940f,
	0.545324988422046460f, -0.838224705554837970f, 0.544681917787634530f,
	-0.838642717988527300f,
	0.544038526730883930f, -0.839060237070312630f, 0.543394815630284800f,
	-0.839477262554578550f,
	0.542750784864516000f, -0.839893794195999410f, 0.542106434812444030f,
	-0.840309831749540770f,
	0.541461765853123560f, -0.840725374970458070f, 0.540816778365796670f,
	-0.841140423614298080f,
	0.540171472729892970f, -0.841554977436898330f, 0.539525849325029010f,
	-0.841969036194387680f,
	0.538879908531008420f, -0.842382599643185960f, 0.538233650727821700f,
	-0.842795667540004120f,
	0.537587076295645510f, -0.843208239641845440f, 0.536940185614843020f,
	-0.843620315706004040f,
	0.536292979065963180f, -0.844031895490066410f, 0.535645457029741090f,
	-0.844442978751910660f,
	0.534997619887097260f, -0.844853565249707010f, 0.534349468019137520f,
	-0.845263654741918220f,
	0.533701001807152960f, -0.845673246987299070f, 0.533052221632619670f,
	-0.846082341744896940f,
	0.532403127877198010f, -0.846490938774052020f, 0.531753720922733320f,
	-0.846899037834397350f,
	0.531104001151255000f, -0.847306638685858320f, 0.530453968944976320f,
	-0.847713741088654270f,
	0.529803624686294830f, -0.848120344803297120f, 0.529152968757790720f,
	-0.848526449590592650f,
	0.528502001542228480f, -0.848932055211639610f, 0.527850723422555460f,
	-0.849337161427830670f,
	0.527199134781901390f, -0.849741768000852440f, 0.526547236003579330f,
	-0.850145874692685210f,
	0.525895027471084740f, -0.850549481265603370f, 0.525242509568094710f,
	-0.850952587482175730f,
	0.524589682678468840f, -0.851355193105265200f, 0.523936547186248600f,
	-0.851757297898029120f,
	0.523283103475656430f, -0.852158901623919830f, 0.522629351931096720f,
	-0.852560004046683970f,
	0.521975292937154390f, -0.852960604930363630f, 0.521320926878595550f,
	-0.853360704039295430f,
	0.520666254140367270f, -0.853760301138111300f, 0.520011275107596040f,
	-0.854159395991738730f,
	0.519355990165589530f, -0.854557988365400530f, 0.518700399699835170f,
	-0.854956078024614820f,
	0.518044504095999340f, -0.855353664735196030f, 0.517388303739929060f,
	-0.855750748263253920f,
	0.516731799017649980f, -0.856147328375194470f, 0.516074990315366630f,
	-0.856543404837719960f,
	0.515417878019463150f, -0.856938977417828650f, 0.514760462516501200f,
	-0.857334045882815590f,
	0.514102744193221660f, -0.857728610000272120f, 0.513444723436543570f,
	-0.858122669538086020f,
	0.512786400633563070f, -0.858516224264442740f, 0.512127776171554690f,
	-0.858909273947823900f,
	0.511468850437970520f, -0.859301818357008360f, 0.510809623820439040f,
	-0.859693857261072610f,
	0.510150096706766700f, -0.860085390429390140f, 0.509490269484936360f,
	-0.860476417631632070f,
	0.508830142543106990f, -0.860866938637767310f, 0.508169716269614710f,
	-0.861256953218062060f,
	0.507508991052970870f, -0.861646461143081300f, 0.506847967281863320f,
	-0.862035462183687210f,
	0.506186645345155450f, -0.862423956111040500f, 0.505525025631885510f,
	-0.862811942696600330f,
	0.504863108531267480f, -0.863199421712124160f, 0.504200894432690560f,
	-0.863586392929667990f,
	0.503538383725717580f, -0.863972856121586700f, 0.502875576800086880f,
	-0.864358811060534030f,
	0.502212474045710900f, -0.864744257519462380f, 0.501549075852675390f,
	-0.865129195271623690f,
	0.500885382611240940f, -0.865513624090568980f, 0.500221394711840680f,
	-0.865897543750148820f,
	0.499557112545081890f, -0.866280954024512990f, 0.498892536501744750f,
	-0.866663854688111020f,
	0.498227666972781870f, -0.867046245515692650f, 0.497562504349319090f,
	-0.867428126282306920f,
	0.496897049022654640f, -0.867809496763303210f, 0.496231301384258310f,
	-0.868190356734331310f,
	0.495565261825772490f, -0.868570705971340900f, 0.494898930739011310f,
	-0.868950544250582380f,
	0.494232308515959730f, -0.869329871348606730f, 0.493565395548774880f,
	-0.869708687042265560f,
	0.492898192229784090f, -0.870086991108711350f, 0.492230698951486080f,
	-0.870464783325397670f,
	0.491562916106550060f, -0.870842063470078860f, 0.490894844087815140f,
	-0.871218831320810900f,
	0.490226483288291100f, -0.871595086655951090f, 0.489557834101157550f,
	-0.871970829254157700f,
	0.488888896919763230f, -0.872346058894391540f, 0.488219672137626740f,
	-0.872720775355914300f,
	0.487550160148436050f, -0.873094978418290090f, 0.486880361346047400f,
	-0.873468667861384880f,
	0.486210276124486530f, -0.873841843465366750f, 0.485539904877947020f,
	-0.874214505010706300f,
	0.484869248000791120f, -0.874586652278176110f, 0.484198305887549140f,
	-0.874958285048851540f,
	0.483527078932918740f, -0.875329403104110780f, 0.482855567531765670f,
	-0.875700006225634600f,
	0.482183772079122830f, -0.876070094195406600f, 0.481511692970189920f,
	-0.876439666795713610f,
	0.480839330600333900f, -0.876808723809145760f, 0.480166685365088440f,
	-0.877177265018595940f,
	0.479493757660153010f, -0.877545290207261240f, 0.478820547881394050f,
	-0.877912799158641730f,
	0.478147056424843120f, -0.878279791656541460f, 0.477473283686698060f,
	-0.878646267485068130f,
	0.476799230063322250f, -0.879012226428633410f, 0.476124895951243630f,
	-0.879377668271953180f,
	0.475450281747155870f, -0.879742592800047410f, 0.474775387847917230f,
	-0.880106999798240360f,
	0.474100214650550020f, -0.880470889052160750f, 0.473424762552241530f,
	-0.880834260347742040f,
	0.472749031950342900f, -0.881197113471221980f, 0.472073023242368660f,
	-0.881559448209143780f,
	0.471396736825997810f, -0.881921264348354940f, 0.470720173099071710f,
	-0.882282561676008600f,
	0.470043332459595620f, -0.882643339979562790f, 0.469366215305737630f,
	-0.883003599046780720f,
	0.468688822035827960f, -0.883363338665731580f, 0.468011153048359830f,
	-0.883722558624789660f,
	0.467333208741988530f, -0.884081258712634990f, 0.466654989515530970f,
	-0.884439438718253700f,
	0.465976495767966130f, -0.884797098430937790f, 0.465297727898434650f,
	-0.885154237640285110f,
	0.464618686306237820f, -0.885510856136199950f, 0.463939371390838460f,
	-0.885866953708892790f,
	0.463259783551860260f, -0.886222530148880640f, 0.462579923189086810f,
	-0.886577585246987040f,
	0.461899790702462840f, -0.886932118794342080f, 0.461219386492092430f,
	-0.887286130582383150f,
	0.460538710958240010f, -0.887639620402853930f, 0.459857764501329650f,
	-0.887992588047805560f,
	0.459176547521944150f, -0.888345033309596240f, 0.458495060420826220f,
	-0.888696955980891710f,
	0.457813303598877290f, -0.889048355854664570f, 0.457131277457156980f,
	-0.889399232724195520f,
	0.456448982396883860f, -0.889749586383072890f, 0.455766418819434750f,
	-0.890099416625192210f,
	0.455083587126343840f, -0.890448723244757880f, 0.454400487719303750f,
	-0.890797506036281490f,
	0.453717121000163930f, -0.891145764794583180f, 0.453033487370931580f,
	-0.891493499314791380f,
	0.452349587233771000f, -0.891840709392342720f, 0.451665420991002540f,
	-0.892187394822982480f,
	0.450980989045103810f, -0.892533555402764690f, 0.450296291798708730f,
	-0.892879190928051680f,
	0.449611329654606600f, -0.893224301195515320f, 0.448926103015743260f,
	-0.893568886002136020f,
	0.448240612285220000f, -0.893912945145203250f, 0.447554857866293010f,
	-0.894256478422316040f,
	0.446868840162374330f, -0.894599485631382580f, 0.446182559577030120f,
	-0.894941966570620750f,
	0.445496016513981740f, -0.895283921038557580f, 0.444809211377105000f,
	-0.895625348834030000f,
	0.444122144570429260f, -0.895966249756185110f, 0.443434816498138430f,
	-0.896306623604479660f,
	0.442747227564570130f, -0.896646470178680150f, 0.442059378174214760f,
	-0.896985789278863970f,
	0.441371268731716620f, -0.897324580705418320f, 0.440682899641873020f,
	-0.897662844259040750f,
	0.439994271309633260f, -0.898000579740739880f, 0.439305384140100060f,
	-0.898337786951834190f,
	0.438616238538527710f, -0.898674465693953820f, 0.437926834910322860f,
	-0.899010615769039070f,
	0.437237173661044200f, -0.899346236979341460f, 0.436547255196401250f,
	-0.899681329127423930f,
	0.435857079922255470f, -0.900015892016160280f, 0.435166648244619370f,
	-0.900349925448735600f,
	0.434475960569655710f, -0.900683429228646860f, 0.433785017303678520f,
	-0.901016403159702330f,
	0.433093818853152010f, -0.901348847046022030f, 0.432402365624690140f,
	-0.901680760692037730f,
	0.431710658025057370f, -0.902012143902493070f, 0.431018696461167080f,
	-0.902342996482444200f,
	0.430326481340082610f, -0.902673318237258830f, 0.429634013069016500f,
	-0.903003108972617040f,
	0.428941292055329550f, -0.903332368494511820f, 0.428248318706531910f,
	-0.903661096609247980f,
	0.427555093430282200f, -0.903989293123443340f, 0.426861616634386490f,
	-0.904316957844028320f,
	0.426167888726799620f, -0.904644090578246240f, 0.425473910115623910f,
	-0.904970691133653250f,
	0.424779681209108810f, -0.905296759318118820f, 0.424085202415651670f,
	-0.905622294939825160f,
	0.423390474143796100f, -0.905947297807268460f, 0.422695496802232950f,
	-0.906271767729257660f,
	0.422000270799799790f, -0.906595704514915330f, 0.421304796545479700f,
	-0.906919107973678030f,
	0.420609074448402510f, -0.907241977915295930f, 0.419913104917843730f,
	-0.907564314149832520f,
	0.419216888363223960f, -0.907886116487666150f, 0.418520425194109700f,
	-0.908207384739488700f,
	0.417823715820212380f, -0.908528118716306120f, 0.417126760651387870f,
	-0.908848318229439120f,
	0.416429560097637320f, -0.909167983090522270f, 0.415732114569105420f,
	-0.909487113111505430f,
	0.415034424476081630f, -0.909805708104652220f, 0.414336490228999210f,
	-0.910123767882541570f,
	0.413638312238434560f, -0.910441292258067140f, 0.412939890915108020f,
	-0.910758281044437570f,
	0.412241226669883000f, -0.911074734055176250f, 0.411542319913765280f,
	-0.911390651104122320f,
	0.410843171057903910f, -0.911706032005429880f, 0.410143780513590350f,
	-0.912020876573568230f,
	0.409444148692257590f, -0.912335184623322750f, 0.408744276005481520f,
	-0.912648955969793900f,
	0.408044162864978740f, -0.912962190428398100f, 0.407343809682607970f,
	-0.913274887814867760f,
	0.406643216870369140f, -0.913587047945250810f, 0.405942384840402570f,
	-0.913898670635911680f,
	0.405241314004989860f, -0.914209755703530690f, 0.404540004776553110f,
	-0.914520302965104450f,
	0.403838457567654130f, -0.914830312237946090f, 0.403136672790995240f,
	-0.915139783339685260f,
	0.402434650859418540f, -0.915448716088267830f, 0.401732392185905010f,
	-0.915757110301956720f,
	0.401029897183575790f, -0.916064965799331610f, 0.400327166265690150f,
	-0.916372282399289140f,
	0.399624199845646790f, -0.916679059921042700f, 0.398920998336983020f,
	-0.916985298184122890f,
	0.398217562153373620f, -0.917290997008377910f, 0.397513891708632330f,
	-0.917596156213972950f,
	0.396809987416710420f, -0.917900775621390390f, 0.396105849691696320f,
	-0.918204855051430900f,
	0.395401478947816300f, -0.918508394325212250f, 0.394696875599433670f,
	-0.918811393264169940f,
	0.393992040061048100f, -0.919113851690057770f, 0.393286972747296570f,
	-0.919415769424946960f,
	0.392581674072951530f, -0.919717146291227360f, 0.391876144452922350f,
	-0.920017982111606570f,
	0.391170384302253980f, -0.920318276709110480f, 0.390464394036126650f,
	-0.920618029907083860f,
	0.389758174069856410f, -0.920917241529189520f, 0.389051724818894500f,
	-0.921215911399408730f,
	0.388345046698826300f, -0.921514039342041900f, 0.387638140125372680f,
	-0.921811625181708120f,
	0.386931005514388690f, -0.922108668743345070f, 0.386223643281862980f,
	-0.922405169852209880f,
	0.385516053843919020f, -0.922701128333878520f, 0.384808237616812930f,
	-0.922996544014246250f,
	0.384100195016935040f, -0.923291416719527640f, 0.383391926460808770f,
	-0.923585746276256560f,
	0.382683432365089840f, -0.923879532511286740f, 0.381974713146567220f,
	-0.924172775251791200f,
	0.381265769222162490f, -0.924465474325262600f, 0.380556601008928570f,
	-0.924757629559513910f,
	0.379847208924051110f, -0.925049240782677580f, 0.379137593384847430f,
	-0.925340307823206200f,
	0.378427754808765620f, -0.925630830509872720f, 0.377717693613385810f,
	-0.925920808671769960f,
	0.377007410216418310f, -0.926210242138311270f, 0.376296905035704790f,
	-0.926499130739230510f,
	0.375586178489217330f, -0.926787474304581750f, 0.374875230995057600f,
	-0.927075272664740100f,
	0.374164062971457990f, -0.927362525650401110f, 0.373452674836780410f,
	-0.927649233092581180f,
	0.372741067009515810f, -0.927935394822617890f, 0.372029239908284960f,
	-0.928221010672169440f,
	0.371317193951837600f, -0.928506080473215480f, 0.370604929559051670f,
	-0.928790604058057020f,
	0.369892447148934270f, -0.929074581259315750f, 0.369179747140620070f,
	-0.929358011909935500f,
	0.368466829953372320f, -0.929640895843181330f, 0.367753696006582090f,
	-0.929923232892639560f,
	0.367040345719767240f, -0.930205022892219070f, 0.366326779512573590f,
	-0.930486265676149780f,
	0.365612997804773960f, -0.930766961078983710f, 0.364899001016267380f,
	-0.931047108935595170f,
	0.364184789567079840f, -0.931326709081180430f, 0.363470363877363870f,
	-0.931605761351257830f,
	0.362755724367397230f, -0.931884265581668150f, 0.362040871457584350f,
	-0.932162221608574320f,
	0.361325805568454340f, -0.932439629268462360f, 0.360610527120662270f,
	-0.932716488398140250f,
	0.359895036534988280f, -0.932992798834738850f, 0.359179334232336560f,
	-0.933268560415712050f,
	0.358463420633736540f, -0.933543772978836170f, 0.357747296160342010f,
	-0.933818436362210960f,
	0.357030961233430030f, -0.934092550404258870f, 0.356314416274402360f,
	-0.934366114943725900f,
	0.355597661704783960f, -0.934639129819680780f, 0.354880697946222790f,
	-0.934911594871516090f,
	0.354163525420490510f, -0.935183509938947500f, 0.353446144549480870f,
	-0.935454874862014620f,
	0.352728555755210730f, -0.935725689481080370f, 0.352010759459819240f,
	-0.935995953636831300f,
	0.351292756085567150f, -0.936265667170278260f, 0.350574546054837570f,
	-0.936534829922755500f,
	0.349856129790135030f, -0.936803441735921560f, 0.349137507714085030f,
	-0.937071502451759190f,
	0.348418680249434510f, -0.937339011912574960f, 0.347699647819051490f,
	-0.937605969960999990f,
	0.346980410845923680f, -0.937872376439989890f, 0.346260969753160170f,
	-0.938138231192824360f,
	0.345541324963989150f, -0.938403534063108060f, 0.344821476901759290f,
	-0.938668284894770170f,
	0.344101425989938980f, -0.938932483532064490f, 0.343381172652115100f,
	-0.939196129819569900f,
	0.342660717311994380f, -0.939459223602189920f, 0.341940060393402300f,
	-0.939721764725153340f,
	0.341219202320282410f, -0.939983753034013940f, 0.340498143516697100f,
	-0.940245188374650880f,
	0.339776884406826960f, -0.940506070593268300f, 0.339055425414969640f,
	-0.940766399536396070f,
	0.338333766965541290f, -0.941026175050889260f, 0.337611909483074680f,
	-0.941285396983928660f,
	0.336889853392220050f, -0.941544065183020810f, 0.336167599117744690f,
	-0.941802179495997650f,
	0.335445147084531660f, -0.942059739771017310f, 0.334722497717581220f,
	-0.942316745856563780f,
	0.333999651442009490f, -0.942573197601446870f, 0.333276608683047980f,
	-0.942829094854802710f,
	0.332553369866044220f, -0.943084437466093490f, 0.331829935416461220f,
	-0.943339225285107720f,
	0.331106305759876430f, -0.943593458161960390f, 0.330382481321982950f,
	-0.943847135947092690f,
	0.329658462528587550f, -0.944100258491272660f, 0.328934249805612200f,
	-0.944352825645594750f,
	0.328209843579092660f, -0.944604837261480260f, 0.327485244275178060f,
	-0.944856293190677210f,
	0.326760452320131790f, -0.945107193285260610f, 0.326035468140330350f,
	-0.945357537397632290f,
	0.325310292162262980f, -0.945607325380521280f, 0.324584924812532150f,
	-0.945856557086983910f,
	0.323859366517852960f, -0.946105232370403340f, 0.323133617705052330f,
	-0.946353351084490590f,
	0.322407678801070020f, -0.946600913083283530f, 0.321681550232956640f,
	-0.946847918221148000f,
	0.320955232427875210f, -0.947094366352777220f, 0.320228725813100020f,
	-0.947340257333191940f,
	0.319502030816015750f, -0.947585591017741090f, 0.318775147864118480f,
	-0.947830367262101010f,
	0.318048077385015060f, -0.948074585922276230f, 0.317320819806421790f,
	-0.948318246854599090f,
	0.316593375556165850f, -0.948561349915730270f, 0.315865745062184070f,
	-0.948803894962658380f,
	0.315137928752522440f, -0.949045881852700560f, 0.314409927055336820f,
	-0.949287310443502010f,
	0.313681740398891570f, -0.949528180593036670f, 0.312953369211560200f,
	-0.949768492159606680f,
	0.312224813921825050f, -0.950008245001843000f, 0.311496074958275970f,
	-0.950247438978705230f,
	0.310767152749611470f, -0.950486073949481700f, 0.310038047724638000f,
	-0.950724149773789610f,
	0.309308760312268780f, -0.950961666311575080f, 0.308579290941525030f,
	-0.951198623423113230f,
	0.307849640041534980f, -0.951435020969008340f, 0.307119808041533100f,
	-0.951670858810193860f,
	0.306389795370861080f, -0.951906136807932230f, 0.305659602458966230f,
	-0.952140854823815830f,
	0.304929229735402430f, -0.952375012719765880f, 0.304198677629829270f,
	-0.952608610358033240f,
	0.303467946572011370f, -0.952841647601198720f, 0.302737036991819140f,
	-0.953074124312172200f,
	0.302005949319228200f, -0.953306040354193750f, 0.301274683984318000f,
	-0.953537395590833280f,
	0.300543241417273400f, -0.953768189885990330f, 0.299811622048383460f,
	-0.953998423103894490f,
	0.299079826308040480f, -0.954228095109105670f, 0.298347854626741570f,
	-0.954457205766513490f,
	0.297615707435086310f, -0.954685754941338340f, 0.296883385163778270f,
	-0.954913742499130520f,
	0.296150888243623960f, -0.955141168305770670f, 0.295418217105532070f,
	-0.955368032227470240f,
	0.294685372180514330f, -0.955594334130771110f, 0.293952353899684770f,
	-0.955820073882545420f,
	0.293219162694258680f, -0.956045251349996410f, 0.292485798995553830f,
	-0.956269866400658140f,
	0.291752263234989370f, -0.956493918902394990f, 0.291018555844085090f,
	-0.956717408723403050f,
	0.290284677254462330f, -0.956940335732208940f, 0.289550627897843140f,
	-0.957162699797670100f,
	0.288816408206049480f, -0.957384500788975860f, 0.288082018611004300f,
	-0.957605738575646240f,
	0.287347459544729570f, -0.957826413027532910f, 0.286612731439347790f,
	-0.958046524014818600f,
	0.285877834727080730f, -0.958266071408017670f, 0.285142769840248720f,
	-0.958485055077976100f,
	0.284407537211271820f, -0.958703474895871600f, 0.283672137272668550f,
	-0.958921330733213060f,
	0.282936570457055390f, -0.959138622461841890f, 0.282200837197147500f,
	-0.959355349953930790f,
	0.281464937925758050f, -0.959571513081984520f, 0.280728873075797190f,
	-0.959787111718839900f,
	0.279992643080273380f, -0.960002145737665850f, 0.279256248372291240f,
	-0.960216615011963430f,
	0.278519689385053060f, -0.960430519415565790f, 0.277782966551857800f,
	-0.960643858822638470f,
	0.277046080306099950f, -0.960856633107679660f, 0.276309031081271030f,
	-0.961068842145519350f,
	0.275571819310958250f, -0.961280485811320640f, 0.274834445428843940f,
	-0.961491563980579000f,
	0.274096909868706330f, -0.961702076529122540f, 0.273359213064418790f,
	-0.961912023333112100f,
	0.272621355449948980f, -0.962121404269041580f, 0.271883337459359890f,
	-0.962330219213737400f,
	0.271145159526808070f, -0.962538468044359160f, 0.270406822086544820f,
	-0.962746150638399410f,
	0.269668325572915200f, -0.962953266873683880f, 0.268929670420357310f,
	-0.963159816628371360f,
	0.268190857063403180f, -0.963365799780954050f, 0.267451885936677740f,
	-0.963571216210257210f,
	0.266712757474898420f, -0.963776065795439840f, 0.265973472112875530f,
	-0.963980348415994110f,
	0.265234030285511900f, -0.964184063951745720f, 0.264494432427801630f,
	-0.964387212282854290f,
	0.263754678974831510f, -0.964589793289812650f, 0.263014770361779060f,
	-0.964791806853447900f,
	0.262274707023913590f, -0.964993252854920320f, 0.261534489396595630f,
	-0.965194131175724720f,
	0.260794117915275570f, -0.965394441697689400f, 0.260053593015495130f,
	-0.965594184302976830f,
	0.259312915132886350f, -0.965793358874083570f, 0.258572084703170390f,
	-0.965991965293840570f,
	0.257831102162158930f, -0.966190003445412620f, 0.257089967945753230f,
	-0.966387473212298790f,
	0.256348682489942910f, -0.966584374478333120f, 0.255607246230807550f,
	-0.966780707127683270f,
	0.254865659604514630f, -0.966976471044852070f, 0.254123923047320620f,
	-0.967171666114676640f,
	0.253382036995570270f, -0.967366292222328510f, 0.252640001885695580f,
	-0.967560349253314360f,
	0.251897818154216910f, -0.967753837093475510f, 0.251155486237742030f,
	-0.967946755628987800f,
	0.250413006572965280f, -0.968139104746362330f, 0.249670379596668520f,
	-0.968330884332445300f,
	0.248927605745720260f, -0.968522094274417270f, 0.248184685457074780f,
	-0.968712734459794780f,
	0.247441619167773440f, -0.968902804776428870f, 0.246698407314942500f,
	-0.969092305112506100f,
	0.245955050335794590f, -0.969281235356548530f, 0.245211548667627680f,
	-0.969469595397412950f,
	0.244467902747824210f, -0.969657385124292450f, 0.243724113013852130f,
	-0.969844604426714830f,
	0.242980179903263980f, -0.970031253194543970f, 0.242236103853696070f,
	-0.970217331317979160f,
	0.241491885302869300f, -0.970402838687555500f, 0.240747524688588540f,
	-0.970587775194143630f,
	0.240003022448741500f, -0.970772140728950350f, 0.239258379021300120f,
	-0.970955935183517970f,
	0.238513594844318500f, -0.971139158449725090f, 0.237768670355934210f,
	-0.971321810419786160f,
	0.237023605994367340f, -0.971503890986251780f, 0.236278402197919620f,
	-0.971685400042008540f,
	0.235533059404975460f, -0.971866337480279400f, 0.234787578054001080f,
	-0.972046703194623500f,
	0.234041958583543460f, -0.972226497078936270f, 0.233296201432231560f,
	-0.972405719027449770f,
	0.232550307038775330f, -0.972584368934732210f, 0.231804275841964780f,
	-0.972762446695688570f,
	0.231058108280671280f, -0.972939952205560070f, 0.230311804793845530f,
	-0.973116885359925130f,
	0.229565365820518870f, -0.973293246054698250f, 0.228818791799802360f,
	-0.973469034186130950f,
	0.228072083170885790f, -0.973644249650811870f, 0.227325240373038830f,
	-0.973818892345666100f,
	0.226578263845610110f, -0.973992962167955830f, 0.225831154028026200f,
	-0.974166459015280320f,
	0.225083911359792780f, -0.974339382785575860f, 0.224336536280493690f,
	-0.974511733377115720f,
	0.223589029229790020f, -0.974683510688510670f, 0.222841390647421280f,
	-0.974854714618708430f,
	0.222093620973203590f, -0.975025345066994120f, 0.221345720647030810f,
	-0.975195401932990370f,
	0.220597690108873650f, -0.975364885116656870f, 0.219849529798778750f,
	-0.975533794518291360f,
	0.219101240156869770f, -0.975702130038528570f, 0.218352821623346430f,
	-0.975869891578341030f,
	0.217604274638483670f, -0.976037079039039020f, 0.216855599642632570f,
	-0.976203692322270560f,
	0.216106797076219600f, -0.976369731330021140f, 0.215357867379745550f,
	-0.976535195964614470f,
	0.214608810993786920f, -0.976700086128711840f, 0.213859628358993830f,
	-0.976864401725312640f,
	0.213110319916091360f, -0.977028142657754390f, 0.212360886105878580f,
	-0.977191308829712280f,
	0.211611327369227610f, -0.977353900145199960f, 0.210861644147084830f,
	-0.977515916508569280f,
	0.210111836880469720f, -0.977677357824509930f, 0.209361906010474190f,
	-0.977838223998050430f,
	0.208611851978263460f, -0.977998514934557140f, 0.207861675225075150f,
	-0.978158230539735050f,
	0.207111376192218560f, -0.978317370719627650f, 0.206360955321075680f,
	-0.978475935380616830f,
	0.205610413053099320f, -0.978633924429423100f, 0.204859749829814420f,
	-0.978791337773105670f,
	0.204108966092817010f, -0.978948175319062200f, 0.203358062283773370f,
	-0.979104436975029250f,
	0.202607038844421110f, -0.979260122649082020f, 0.201855896216568160f,
	-0.979415232249634780f,
	0.201104634842091960f, -0.979569765685440520f, 0.200353255162940420f,
	-0.979723722865591170f,
	0.199601757621131050f, -0.979877103699517640f, 0.198850142658750120f,
	-0.980029908096989980f,
	0.198098410717953730f, -0.980182135968117320f, 0.197346562240966000f,
	-0.980333787223347960f,
	0.196594597670080220f, -0.980484861773469380f, 0.195842517447657990f,
	-0.980635359529608120f,
	0.195090322016128330f, -0.980785280403230430f, 0.194338011817988600f,
	-0.980934624306141640f,
	0.193585587295803750f, -0.981083391150486590f, 0.192833048892205290f,
	-0.981231580848749730f,
	0.192080397049892380f, -0.981379193313754560f, 0.191327632211630990f,
	-0.981526228458664660f,
	0.190574754820252800f, -0.981672686196983110f, 0.189821765318656580f,
	-0.981818566442552500f,
	0.189068664149806280f, -0.981963869109555240f, 0.188315451756732120f,
	-0.982108594112513610f,
	0.187562128582529740f, -0.982252741366289370f, 0.186808695070359330f,
	-0.982396310786084690f,
	0.186055151663446630f, -0.982539302287441240f, 0.185301498805082040f,
	-0.982681715786240860f,
	0.184547736938619640f, -0.982823551198705240f, 0.183793866507478390f,
	-0.982964808441396440f,
	0.183039887955141060f, -0.983105487431216290f, 0.182285801725153320f,
	-0.983245588085407070f,
	0.181531608261125130f, -0.983385110321551180f, 0.180777308006728670f,
	-0.983524054057571260f,
	0.180022901405699510f, -0.983662419211730250f, 0.179268388901835880f,
	-0.983800205702631490f,
	0.178513770938997590f, -0.983937413449218920f, 0.177759047961107140f,
	-0.984074042370776450f,
	0.177004220412148860f, -0.984210092386929030f, 0.176249288736167940f,
	-0.984345563417641900f,
	0.175494253377271400f, -0.984480455383220930f, 0.174739114779627310f,
	-0.984614768204312600f,
	0.173983873387463850f, -0.984748501801904210f, 0.173228529645070490f,
	-0.984881656097323700f,
	0.172473083996796030f, -0.985014231012239840f, 0.171717536887049970f,
	-0.985146226468662230f,
	0.170961888760301360f, -0.985277642388941220f, 0.170206140061078120f,
	-0.985408478695768420f,
	0.169450291233967930f, -0.985538735312176060f, 0.168694342723617440f,
	-0.985668412161537550f,
	0.167938294974731230f, -0.985797509167567370f, 0.167182148432072880f,
	-0.985926026254321130f,
	0.166425903540464220f, -0.986053963346195440f, 0.165669560744784140f,
	-0.986181320367928270f,
	0.164913120489970090f, -0.986308097244598670f, 0.164156583221015890f,
	-0.986434293901627070f,
	0.163399949382973230f, -0.986559910264775410f, 0.162643219420950450f,
	-0.986684946260146690f,
	0.161886393780111910f, -0.986809401814185420f, 0.161129472905678780f,
	-0.986933276853677710f,
	0.160372457242928400f, -0.987056571305750970f, 0.159615347237193090f,
	-0.987179285097874340f,
	0.158858143333861390f, -0.987301418157858430f, 0.158100845978377090f,
	-0.987422970413855410f,
	0.157343455616238280f, -0.987543941794359230f, 0.156585972692998590f,
	-0.987664332228205710f,
	0.155828397654265320f, -0.987784141644572180f, 0.155070730945700510f,
	-0.987903369972977790f,
	0.154312973013020240f, -0.988022017143283530f, 0.153555124301993500f,
	-0.988140083085692570f,
	0.152797185258443410f, -0.988257567730749460f, 0.152039156328246160f,
	-0.988374471009341280f,
	0.151281037957330250f, -0.988490792852696590f, 0.150522830591677370f,
	-0.988606533192386450f,
	0.149764534677321620f, -0.988721691960323780f, 0.149006150660348470f,
	-0.988836269088763540f,
	0.148247678986896200f, -0.988950264510302990f, 0.147489120103153680f,
	-0.989063678157881540f,
	0.146730474455361750f, -0.989176509964781010f, 0.145971742489812370f,
	-0.989288759864625170f,
	0.145212924652847520f, -0.989400427791380380f, 0.144454021390860440f,
	-0.989511513679355190f,
	0.143695033150294580f, -0.989622017463200780f, 0.142935960377642700f,
	-0.989731939077910570f,
	0.142176803519448000f, -0.989841278458820530f, 0.141417563022303130f,
	-0.989950035541608990f,
	0.140658239332849240f, -0.990058210262297120f, 0.139898832897777380f,
	-0.990165802557248400f,
	0.139139344163826280f, -0.990272812363169110f, 0.138379773577783890f,
	-0.990379239617108160f,
	0.137620121586486180f, -0.990485084256456980f, 0.136860388636816430f,
	-0.990590346218950150f,
	0.136100575175706200f, -0.990695025442664630f, 0.135340681650134330f,
	-0.990799121866020370f,
	0.134580708507126220f, -0.990902635427780010f, 0.133820656193754690f,
	-0.991005566067049370f,
	0.133060525157139180f, -0.991107913723276780f, 0.132300315844444680f,
	-0.991209678336254060f,
	0.131540028702883280f, -0.991310859846115440f, 0.130779664179711790f,
	-0.991411458193338540f,
	0.130019222722233350f, -0.991511473318743900f, 0.129258704777796270f,
	-0.991610905163495370f,
	0.128498110793793220f, -0.991709753669099530f, 0.127737441217662280f,
	-0.991808018777406430f,
	0.126976696496885980f, -0.991905700430609330f, 0.126215877078990400f,
	-0.992002798571244520f,
	0.125454983411546210f, -0.992099313142191800f, 0.124694015942167770f,
	-0.992195244086673920f,
	0.123932975118512200f, -0.992290591348257370f, 0.123171861388280650f,
	-0.992385354870851670f,
	0.122410675199216280f, -0.992479534598709970f, 0.121649416999105540f,
	-0.992573130476428810f,
	0.120888087235777220f, -0.992666142448948020f, 0.120126686357101580f,
	-0.992758570461551140f,
	0.119365214810991350f, -0.992850414459865100f, 0.118603673045400840f,
	-0.992941674389860470f,
	0.117842061508325020f, -0.993032350197851410f, 0.117080380647800550f,
	-0.993122441830495580f,
	0.116318630911904880f, -0.993211949234794500f, 0.115556812748755290f,
	-0.993300872358093280f,
	0.114794926606510250f, -0.993389211148080650f, 0.114032972933367300f,
	-0.993476965552789190f,
	0.113270952177564360f, -0.993564135520595300f, 0.112508864787378830f,
	-0.993650721000219120f,
	0.111746711211126660f, -0.993736721940724600f, 0.110984491897163380f,
	-0.993822138291519660f,
	0.110222207293883180f, -0.993906970002356060f, 0.109459857849718030f,
	-0.993991217023329380f,
	0.108697444013138670f, -0.994074879304879370f, 0.107934966232653760f,
	-0.994157956797789730f,
	0.107172424956808870f, -0.994240449453187900f, 0.106409820634187840f,
	-0.994322357222545810f,
	0.105647153713410700f, -0.994403680057679100f, 0.104884424643134970f,
	-0.994484417910747600f,
	0.104121633872054730f, -0.994564570734255420f, 0.103358781848899700f,
	-0.994644138481050710f,
	0.102595869022436280f, -0.994723121104325700f, 0.101832895841466670f,
	-0.994801518557617110f,
	0.101069862754827880f, -0.994879330794805620f, 0.100306770211392820f,
	-0.994956557770116380f,
	0.099543618660069444f, -0.995033199438118630f, 0.098780408549799664f,
	-0.995109255753726110f,
	0.098017140329560770f, -0.995184726672196820f, 0.097253814448363354f,
	-0.995259612149133390f,
	0.096490431355252607f, -0.995333912140482280f, 0.095726991499307315f,
	-0.995407626602534900f,
	0.094963495329639061f, -0.995480755491926940f, 0.094199943295393190f,
	-0.995553298765638470f,
	0.093436335845747912f, -0.995625256380994310f, 0.092672673429913366f,
	-0.995696628295663520f,
	0.091908956497132696f, -0.995767414467659820f, 0.091145185496681130f,
	-0.995837614855341610f,
	0.090381360877865011f, -0.995907229417411720f, 0.089617483090022917f,
	-0.995976258112917790f,
	0.088853552582524684f, -0.996044700901251970f, 0.088089569804770507f,
	-0.996112557742151130f,
	0.087325535206192226f, -0.996179828595696870f, 0.086561449236251239f,
	-0.996246513422315520f,
	0.085797312344439880f, -0.996312612182778000f, 0.085033124980280414f,
	-0.996378124838200210f,
	0.084268887593324127f, -0.996443051350042630f, 0.083504600633152404f,
	-0.996507391680110820f,
	0.082740264549375803f, -0.996571145790554840f, 0.081975879791633108f,
	-0.996634313643869900f,
	0.081211446809592386f, -0.996696895202896060f, 0.080446966052950097f,
	-0.996758890430818000f,
	0.079682437971430126f, -0.996820299291165670f, 0.078917863014785095f,
	-0.996881121747813850f,
	0.078153241632794315f, -0.996941357764982160f, 0.077388574275265049f,
	-0.997001007307235290f,
	0.076623861392031617f, -0.997060070339482960f, 0.075859103432954503f,
	-0.997118546826979980f,
	0.075094300847921291f, -0.997176436735326190f, 0.074329454086845867f,
	-0.997233740030466160f,
	0.073564563599667454f, -0.997290456678690210f, 0.072799629836351618f,
	-0.997346586646633230f,
	0.072034653246889416f, -0.997402129901275300f, 0.071269634281296415f,
	-0.997457086409941910f,
	0.070504573389614009f, -0.997511456140303450f, 0.069739471021907376f,
	-0.997565239060375750f,
	0.068974327628266732f, -0.997618435138519550f, 0.068209143658806454f,
	-0.997671044343441000f,
	0.067443919563664106f, -0.997723066644191640f, 0.066678655793001543f,
	-0.997774502010167820f,
	0.065913352797003930f, -0.997825350411111640f, 0.065148011025878860f,
	-0.997875611817110150f,
	0.064382630929857410f, -0.997925286198596000f, 0.063617212959193190f,
	-0.997974373526346990f,
	0.062851757564161420f, -0.998022873771486240f, 0.062086265195060247f,
	-0.998070786905482340f,
	0.061320736302208648f, -0.998118112900149180f, 0.060555171335947781f,
	-0.998164851727646240f,
	0.059789570746640007f, -0.998211003360478190f, 0.059023934984667986f,
	-0.998256567771495180f,
	0.058258264500435732f, -0.998301544933892890f, 0.057492559744367684f,
	-0.998345934821212370f,
	0.056726821166907783f, -0.998389737407340160f, 0.055961049218520520f,
	-0.998432952666508440f,
	0.055195244349690031f, -0.998475580573294770f, 0.054429407010919147f,
	-0.998517621102622210f,
	0.053663537652730679f, -0.998559074229759310f, 0.052897636725665401f,
	-0.998599939930320370f,
	0.052131704680283317f, -0.998640218180265270f, 0.051365741967162731f,
	-0.998679908955899090f,
	0.050599749036899337f, -0.998719012233872940f, 0.049833726340107257f,
	-0.998757527991183340f,
	0.049067674327418126f, -0.998795456205172410f, 0.048301593449480172f,
	-0.998832796853527990f,
	0.047535484156959261f, -0.998869549914283560f, 0.046769346900537960f,
	-0.998905715365818290f,
	0.046003182130914644f, -0.998941293186856870f, 0.045236990298804750f,
	-0.998976283356469820f,
	0.044470771854938744f, -0.999010685854073380f, 0.043704527250063421f,
	-0.999044500659429290f,
	0.042938256934940959f, -0.999077727752645360f, 0.042171961360348002f,
	-0.999110367114174890f,
	0.041405640977076712f, -0.999142418724816910f, 0.040639296235933854f,
	-0.999173882565716380f,
	0.039872927587739845f, -0.999204758618363890f, 0.039106535483329839f,
	-0.999235046864595850f,
	0.038340120373552791f, -0.999264747286594420f, 0.037573682709270514f,
	-0.999293859866887790f,
	0.036807222941358991f, -0.999322384588349540f, 0.036040741520706299f,
	-0.999350321434199440f,
	0.035274238898213947f, -0.999377670388002850f, 0.034507715524795889f,
	-0.999404431433671300f,
	0.033741171851377642f, -0.999430604555461730f, 0.032974608328897315f,
	-0.999456189737977340f,
	0.032208025408304704f, -0.999481186966166950f, 0.031441423540560343f,
	-0.999505596225325310f,
	0.030674803176636581f, -0.999529417501093140f, 0.029908164767516655f,
	-0.999552650779456990f,
	0.029141508764193740f, -0.999575296046749220f, 0.028374835617672258f,
	-0.999597353289648380f,
	0.027608145778965820f, -0.999618822495178640f, 0.026841439699098527f,
	-0.999639703650710200f,
	0.026074717829104040f, -0.999659996743959220f, 0.025307980620024630f,
	-0.999679701762987930f,
	0.024541228522912264f, -0.999698818696204250f, 0.023774461988827676f,
	-0.999717347532362190f,
	0.023007681468839410f, -0.999735288260561680f, 0.022240887414024919f,
	-0.999752640870248840f,
	0.021474080275469605f, -0.999769405351215280f, 0.020707260504265912f,
	-0.999785581693599210f,
	0.019940428551514598f, -0.999801169887884260f, 0.019173584868322699f,
	-0.999816169924900410f,
	0.018406729905804820f, -0.999830581795823400f, 0.017639864115082195f,
	-0.999844405492175240f,
	0.016872987947281773f, -0.999857641005823860f, 0.016106101853537263f,
	-0.999870288328982950f,
	0.015339206284988220f, -0.999882347454212560f, 0.014572301692779104f,
	-0.999893818374418490f,
	0.013805388528060349f, -0.999904701082852900f, 0.013038467241987433f,
	-0.999914995573113470f,
	0.012271538285719944f, -0.999924701839144500f, 0.011504602110422875f,
	-0.999933819875236000f,
	0.010737659167264572f, -0.999942349676023910f, 0.009970709907418029f,
	-0.999950291236490480f,
	0.009203754782059960f, -0.999957644551963900f, 0.008436794242369860f,
	-0.999964409618118280f,
	0.007669828739531077f, -0.999970586430974140f, 0.006902858724729877f,
	-0.999976174986897610f,
	0.006135884649154515f, -0.999981175282601110f, 0.005368906963996303f,
	-0.999985587315143200f,
	0.004601926120448672f, -0.999989411081928400f, 0.003834942569706248f,
	-0.999992646580707190f,
	0.003067956762966138f, -0.999995293809576190f, 0.002300969151425887f,
	-0.999997352766978210f,
	0.001533980186284766f, -0.999998823451701880f, 0.000766990318742846f,
	-0.999999705862882230f
};

static const float32_t Weights_8192[16384] = {
	1.000000000000000000f, -0.000000000000000000f, 0.999999981616429330f,
	-0.000191747597310703f,
	0.999999926465717890f, -0.000383495187571396f, 0.999999834547867670f,
	-0.000575242763732066f,
	0.999999705862882230f, -0.000766990318742704f, 0.999999540410766110f,
	-0.000958737845553301f,
	0.999999338191525530f, -0.001150485337113849f, 0.999999099205167830f,
	-0.001342232786374338f,
	0.999998823451701880f, -0.001533980186284766f, 0.999998510931137790f,
	-0.001725727529795126f,
	0.999998161643486980f, -0.001917474809855419f, 0.999997775588762350f,
	-0.002109222019415644f,
	0.999997352766978210f, -0.002300969151425805f, 0.999996893178149880f,
	-0.002492716198835908f,
	0.999996396822294350f, -0.002684463154595962f, 0.999995863699429940f,
	-0.002876210011655979f,
	0.999995293809576190f, -0.003067956762965976f, 0.999994687152754080f,
	-0.003259703401475973f,
	0.999994043728985820f, -0.003451449920135994f, 0.999993363538295150f,
	-0.003643196311896068f,
	0.999992646580707190f, -0.003834942569706228f, 0.999991892856248010f,
	-0.004026688686516512f,
	0.999991102364945590f, -0.004218434655276963f, 0.999990275106828920f,
	-0.004410180468937631f,
	0.999989411081928400f, -0.004601926120448571f, 0.999988510290275690f,
	-0.004793671602759841f,
	0.999987572731904080f, -0.004985416908821511f, 0.999986598406848000f,
	-0.005177162031583651f,
	0.999985587315143200f, -0.005368906963996343f, 0.999984539456826970f,
	-0.005560651699009674f,
	0.999983454831937730f, -0.005752396229573736f, 0.999982333440515350f,
	-0.005944140548638633f,
	0.999981175282601110f, -0.006135884649154475f, 0.999979980358237650f,
	-0.006327628524071378f,
	0.999978748667468830f, -0.006519372166339468f, 0.999977480210339940f,
	-0.006711115568908879f,
	0.999976174986897610f, -0.006902858724729756f, 0.999974832997189810f,
	-0.007094601626752250f,
	0.999973454241265940f, -0.007286344267926521f, 0.999972038719176730f,
	-0.007478086641202744f,
	0.999970586430974140f, -0.007669828739531097f, 0.999969097376711580f,
	-0.007861570555861772f,
	0.999967571556443780f, -0.008053312083144972f, 0.999966008970226920f,
	-0.008245053314330906f,
	0.999964409618118280f, -0.008436794242369799f, 0.999962773500176930f,
	-0.008628534860211886f,
	0.999961100616462820f, -0.008820275160807412f, 0.999959390967037450f,
	-0.009012015137106633f,
	0.999957644551963900f, -0.009203754782059819f, 0.999955861371306100f,
	-0.009395494088617252f,
	0.999954041425129780f, -0.009587233049729225f, 0.999952184713501780f,
	-0.009778971658346044f,
	0.999950291236490480f, -0.009970709907418031f, 0.999948360994165400f,
	-0.010162447789895513f,
	0.999946393986597460f, -0.010354185298728842f, 0.999944390213859060f,
	-0.010545922426868378f,
	0.999942349676023910f, -0.010737659167264491f, 0.999940272373166960f,
	-0.010929395512867571f,
	0.999938158305364590f, -0.011121131456628021f, 0.999936007472694620f,
	-0.011312866991496258f,
	0.999933819875236000f, -0.011504602110422714f, 0.999931595513069200f,
	-0.011696336806357838f,
	0.999929334386276070f, -0.011888071072252092f, 0.999927036494939640f,
	-0.012079804901055957f,
	0.999924701839144500f, -0.012271538285719925f, 0.999922330418976490f,
	-0.012463271219194511f,
	0.999919922234522750f, -0.012655003694430242f, 0.999917477285871770f,
	-0.012846735704377662f,
	0.999914995573113470f, -0.013038467241987334f, 0.999912477096339240f,
	-0.013230198300209835f,
	0.999909921855641540f, -0.013421928871995765f, 0.999907329851114300f,
	-0.013613658950295740f,
	0.999904701082852900f, -0.013805388528060391f, 0.999902035550953920f,
	-0.013997117598240367f,
	0.999899333255515390f, -0.014188846153786345f, 0.999896594196636680f,
	-0.014380574187649006f,
	0.999893818374418490f, -0.014572301692779064f, 0.999891005788962950f,
	-0.014764028662127246f,
	0.999888156440373320f, -0.014955755088644296f, 0.999885270328754520f,
	-0.015147480965280987f,
	0.999882347454212560f, -0.015339206284988100f, 0.999879387816854930f,
	-0.015530931040716447f,
	0.999876391416790410f, -0.015722655225416857f, 0.999873358254129260f,
	-0.015914378832040183f,
	0.999870288328982950f, -0.016106101853537287f, 0.999867181641464380f,
	-0.016297824282859065f,
	0.999864038191687680f, -0.016489546112956437f, 0.999860857979768540f,
	-0.016681267336780332f,
	0.999857641005823860f, -0.016872987947281710f, 0.999854387269971890f,
	-0.017064707937411563f,
	0.999851096772332190f, -0.017256427300120877f, 0.999847769513025900f,
	-0.017448146028360693f,
	0.999844405492175240f, -0.017639864115082053f, 0.999841004709904000f,
	-0.017831581553236039f,
	0.999837567166337090f, -0.018023298335773746f, 0.999834092861600960f,
	-0.018215014455646290f,
	0.999830581795823400f, -0.018406729905804820f, 0.999827033969133420f,
	-0.018598444679200511f,
	0.999823449381661570f, -0.018790158768784555f, 0.999819828033539420f,
	-0.018981872167508178f,
	0.999816169924900410f, -0.019173584868322623f, 0.999812475055878780f,
	-0.019365296864179156f,
	0.999808743426610520f, -0.019557008148029083f, 0.999804975037232870f,
	-0.019748718712823729f,
	0.999801169887884260f, -0.019940428551514441f, 0.999797327978704690f,
	-0.020132137657052594f,
	0.999793449309835270f, -0.020323846022389593f, 0.999789533881418780f,
	-0.020515553640476875f,
	0.999785581693599210f, -0.020707260504265895f, 0.999781592746521670f,
	-0.020898966606708137f,
	0.999777567040332940f, -0.021090671940755121f, 0.999773504575180990f,
	-0.021282376499358387f,
	0.999769405351215280f, -0.021474080275469508f, 0.999765269368586450f,
	-0.021665783262040078f,
	0.999761096627446610f, -0.021857485452021735f, 0.999756887127949080f,
	-0.022049186838366135f,
	0.999752640870248840f, -0.022240887414024961f, 0.999748357854501780f,
	-0.022432587171949934f,
	0.999744038080865430f, -0.022624286105092803f, 0.999739681549498660f,
	-0.022815984206405345f,
	0.999735288260561680f, -0.023007681468839369f, 0.999730858214216030f,
	-0.023199377885346720f,
	0.999726391410624470f, -0.023391073448879258f, 0.999721887849951310f,
	-0.023582768152388894f,
	0.999717347532362190f, -0.023774461988827555f, 0.999712770458023870f,
	-0.023966154951147210f,
	0.999708156627104880f, -0.024157847032299864f, 0.999703506039774650f,
	-0.024349538225237534f,
	0.999698818696204250f, -0.024541228522912288f, 0.999694094596566000f,
	-0.024732917918276223f,
	0.999689333741033640f, -0.024924606404281468f, 0.999684536129782140f,
	-0.025116293973880186f,
	0.999679701762987930f, -0.025307980620024571f, 0.999674830640828740f,
	-0.025499666335666853f,
	0.999669922763483760f, -0.025691351113759295f, 0.999664978131133310f,
	-0.025883034947254198f,
	0.999659996743959220f, -0.026074717829103901f, 0.999654978602144690f,
	-0.026266399752260760f,
	0.999649923705874240f, -0.026458080709677187f, 0.999644832055333610f,
	-0.026649760694305618f,
	0.999639703650710200f, -0.026841439699098531f, 0.999634538492192300f,
	-0.027033117717008431f,
	0.999629336579970110f, -0.027224794740987875f, 0.999624097914234570f,
	-0.027416470763989436f,
	0.999618822495178640f, -0.027608145778965740f, 0.999613510322995950f,
	-0.027799819778869445f,
	0.999608161397882110f, -0.027991492756653243f, 0.999602775720033530f,
	-0.028183164705269874f,
	0.999597353289648380f, -0.028374835617672099f, 0.999591894106925950f,
	-0.028566505486812728f,
	0.999586398172067070f, -0.028758174305644615f, 0.999580865485273700f,
	-0.028949842067120635f,
	0.999575296046749220f, -0.029141508764193722f, 0.999569689856698580f,
	-0.029333174389816835f,
	0.999564046915327740f, -0.029524838936942976f, 0.999558367222844300f,
	-0.029716502398525191f,
	0.999552650779456990f, -0.029908164767516555f, 0.999546897585375960f,
	-0.030099826036870198f,
	0.999541107640812940f, -0.030291486199539284f, 0.999535280945980540f,
	-0.030483145248477009f,
	0.999529417501093140f, -0.030674803176636626f, 0.999523517306366350f,
	-0.030866459976971412f,
	0.999517580362016990f, -0.031058115642434700f, 0.999511606668263440f,
	-0.031249770165979861f,
	0.999505596225325310f, -0.031441423540560301f, 0.999499549033423640f,
	-0.031633075759129478f,
	0.999493465092780590f, -0.031824726814640887f, 0.999487344403620080f,
	-0.032016376700048060f,
	0.999481186966166950f, -0.032208025408304586f, 0.999474992780647780f,
	-0.032399672932364086f,
	0.999468761847290050f, -0.032591319265180226f, 0.999462494166323160f,
	-0.032782964399706724f,
	0.999456189737977340f, -0.032974608328897335f, 0.999449848562484530f,
	-0.033166251045705857f,
	0.999443470640077770f, -0.033357892543086139f, 0.999437055970991530f,
	-0.033549532813992068f,
	0.999430604555461730f, -0.033741171851377580f, 0.999424116393725640f,
	-0.033932809648196664f,
	0.999417591486021720f, -0.034124446197403326f, 0.999411029832589780f,
	-0.034316081491951651f,
	0.999404431433671300f, -0.034507715524795750f, 0.999397796289508640f,
	-0.034699348288889799f,
	0.999391124400346050f, -0.034890979777188004f, 0.999384415766428560f,
	-0.035082609982644619f,
	0.999377670388002850f, -0.035274238898213947f, 0.999370888265317170f,
	-0.035465866516850353f,
	0.999364069398620550f, -0.035657492831508222f, 0.999357213788164000f,
	-0.035849117835142018f,
	0.999350321434199440f, -0.036040741520706229f, 0.999343392336980220f,
	-0.036232363881155395f,
	0.999336426496761240f, -0.036423984909444110f, 0.999329423913798420f,
	-0.036615604598527030f,
	0.999322384588349540f, -0.036807222941358832f, 0.999315308520673070f,
	-0.036998839930894263f,
	0.999308195711029470f, -0.037190455560088119f, 0.999301046159680070f,
	-0.037382069821895229f,
	0.999293859866887790f, -0.037573682709270494f, 0.999286636832916740f,
	-0.037765294215168860f,
	0.999279377058032710f, -0.037956904332545310f, 0.999272080542502610f,
	-0.038148513054354891f,
	0.999264747286594420f, -0.038340120373552694f, 0.999257377290578060f,
	-0.038531726283093870f,
	0.999249970554724420f, -0.038723330775933623f, 0.999242527079305830f,
	-0.038914933845027193f,
	0.999235046864595850f, -0.039106535483329888f, 0.999227529910869610f,
	-0.039298135683797059f,
	0.999219976218403530f, -0.039489734439384118f, 0.999212385787475290f,
	-0.039681331743046527f,
	0.999204758618363890f, -0.039872927587739811f, 0.999197094711349880f,
	-0.040064521966419520f,
	0.999189394066714920f, -0.040256114872041282f, 0.999181656684742350f,
	-0.040447706297560782f,
	0.999173882565716380f, -0.040639296235933736f, 0.999166071709923000f,
	-0.040830884680115948f,
	0.999158224117649430f, -0.041022471623063238f, 0.999150339789184110f,
	-0.041214057057731519f,
	0.999142418724816910f, -0.041405640977076739f, 0.999134460924839150f,
	-0.041597223374054894f,
	0.999126466389543390f, -0.041788804241622061f, 0.999118435119223490f,
	-0.041980383572734356f,
	0.999110367114174890f, -0.042171961360347947f, 0.999102262374694130f,
	-0.042363537597419072f,
	0.999094120901079070f, -0.042555112276904020f, 0.999085942693629270f,
	-0.042746685391759132f,
	0.999077727752645360f, -0.042938256934940820f, 0.999069476078429330f,
	-0.043129826899405546f,
	0.999061187671284600f, -0.043321395278109825f, 0.999052862531515930f,
	-0.043512962064010237f,
	0.999044500659429290f, -0.043704527250063421f, 0.999036102055332330f,
	-0.043896090829226068f,
	0.999027666719533690f, -0.044087652794454944f, 0.999019194652343460f,
	-0.044279213138706849f,
	0.999010685854073380f, -0.044470771854938668f, 0.999002140325035980f,
	-0.044662328936107325f,
	0.998993558065545680f, -0.044853884375169815f, 0.998984939075918010f,
	-0.045045438165083197f,
	0.998976283356469820f, -0.045236990298804590f, 0.998967590907519300f,
	-0.045428540769291155f,
	0.998958861729386080f, -0.045620089569500144f, 0.998950095822391250f,
	-0.045811636692388844f,
	0.998941293186856870f, -0.046003182130914623f, 0.998932453823106690f,
	-0.046194725878034908f,
	0.998923577731465780f, -0.046386267926707157f, 0.998914664912260440f,
	-0.046577808269888943f,
	0.998905715365818290f, -0.046769346900537863f, 0.998896729092468410f,
	-0.046960883811611592f,
	0.998887706092541290f, -0.047152418996067869f, 0.998878646366368690f,
	-0.047343952446864478f,
	0.998869549914283560f, -0.047535484156959303f, 0.998860416736620520f,
	-0.047727014119310254f,
	0.998851246833715180f, -0.047918542326875327f, 0.998842040205904840f,
	-0.048110068772612591f,
	0.998832796853527990f, -0.048301593449480144f, 0.998823516776924490f,
	-0.048493116350436176f,
	0.998814199976435390f, -0.048684637468438943f, 0.998804846452403420f,
	-0.048876156796446760f,
	0.998795456205172410f, -0.049067674327418015f, 0.998786029235087640f,
	-0.049259190054311140f,
	0.998776565542495610f, -0.049450703970084664f, 0.998767065127744380f,
	-0.049642216067697156f,
	0.998757527991183340f, -0.049833726340107277f, 0.998747954133162860f,
	-0.050025234780273729f,
	0.998738343554035230f, -0.050216741381155311f, 0.998728696254153720f,
	-0.050408246135710856f,
	0.998719012233872940f, -0.050599749036899282f, 0.998709291493549030f,
	-0.050791250077679581f,
	0.998699534033539280f, -0.050982749251010803f, 0.998689739854202620f,
	-0.051174246549852080f,
	0.998679908955899090f, -0.051365741967162593f, 0.998670041338990070f,
	-0.051557235495901611f,
	0.998660137003838490f, -0.051748727129028456f, 0.998650195950808280f,
	-0.051940216859502536f,
	0.998640218180265270f, -0.052131704680283324f, 0.998630203692576050f,
	-0.052323190584330347f,
	0.998620152488108870f, -0.052514674564603223f, 0.998610064567233340f,
	-0.052706156614061632f,
	0.998599939930320370f, -0.052897636725665324f, 0.998589778577742230f,
	-0.053089114892374133f,
	0.998579580509872500f, -0.053280591107147945f, 0.998569345727086110f,
	-0.053472065362946727f,
	0.998559074229759310f, -0.053663537652730520f, 0.998548766018269920f,
	-0.053855007969459440f,
	0.998538421092996730f, -0.054046476306093660f, 0.998528039454320230f,
	-0.054237942655593452f,
	0.998517621102622210f, -0.054429407010919133f, 0.998507166038285490f,
	-0.054620869365031105f,
	0.998496674261694640f, -0.054812329710889854f, 0.998486145773235360f,
	-0.055003788041455920f,
	0.998475580573294770f, -0.055195244349689934f, 0.998464978662261250f,
	-0.055386698628552597f,
	0.998454340040524800f, -0.055578150871004678f, 0.998443664708476340f,
	-0.055769601070007030f,
	0.998432952666508440f, -0.055961049218520569f, 0.998422203915015020f,
	-0.056152495309506292f,
	0.998411418454391300f, -0.056343939335925290f, 0.998400596285033640f,
	-0.056535381290738700f,
	0.998389737407340160f, -0.056726821166907748f, 0.998378841821709990f,
	-0.056918258957393740f,
	0.998367909528543820f, -0.057109694655158062f, 0.998356940528243420f,
	-0.057301128253162158f,
	0.998345934821212370f, -0.057492559744367566f, 0.998334892407855000f,
	-0.057683989121735904f,
	0.998323813288577560f, -0.057875416378228857f, 0.998312697463787260f,
	-0.058066841506808194f,
	0.998301544933892890f, -0.058258264500435752f, 0.998290355699304350f,
	-0.058449685352073476f,
	0.998279129760433200f, -0.058641104054683341f, 0.998267867117692110f,
	-0.058832520601227435f,
	0.998256567771495180f, -0.059023934984667931f, 0.998245231722257880f,
	-0.059215347197967061f,
	0.998233858970396850f, -0.059406757234087150f, 0.998222449516330550f,
	-0.059598165085990591f,
	0.998211003360478190f, -0.059789570746639868f, 0.998199520503260660f,
	-0.059980974208997548f,
	0.998188000945100300f, -0.060172375466026259f, 0.998176444686420530f,
	-0.060363774510688743f,
	0.998164851727646240f, -0.060555171335947788f, 0.998153222069203760f,
	-0.060746565934766288f,
	0.998141555711520520f, -0.060937958300107203f, 0.998129852655025630f,
	-0.061129348424933588f,
	0.998118112900149180f, -0.061320736302208578f, 0.998106336447323050f,
	-0.061512121924895378f,
	0.998094523296980010f, -0.061703505285957298f, 0.998082673449554590f,
	-0.061894886378357716f,
	0.998070786905482340f, -0.062086265195060088f, 0.998058863665200250f,
	-0.062277641729027972f,
	0.998046903729146840f, -0.062469015973224996f, 0.998034907097761770f,
	-0.062660387920614874f,
	0.998022873771486240f, -0.062851757564161406f, 0.998010803750762450f,
	-0.063043124896828492f,
	0.997998697036034390f, -0.063234489911580066f, 0.997986553627747020f,
	-0.063425852601380228f,
	0.997974373526346990f, -0.063617212959193106f, 0.997962156732281950f,
	-0.063808570977982898f,
	0.997949903246001190f, -0.063999926650713940f, 0.997937613067955250f,
	-0.064191279970350637f,
	0.997925286198596000f, -0.064382630929857465f, 0.997912922638376610f,
	-0.064573979522198982f,
	0.997900522387751620f, -0.064765325740339885f, 0.997888085447177110f,
	-0.064956669577244872f,
	0.997875611817110150f, -0.065148011025878833f, 0.997863101498009500f,
	-0.065339350079206632f,
	0.997850554490335110f, -0.065530686730193327f, 0.997837970794548280f,
	-0.065722020971803990f,
	0.997825350411111640f, -0.065913352797003805f, 0.997812693340489280f,
	-0.066104682198758077f,
	0.997799999583146470f, -0.066296009170032130f, 0.997787269139549960f,
	-0.066487333703791451f,
	0.997774502010167820f, -0.066678655793001557f, 0.997761698195469560f,
	-0.066869975430628115f,
	0.997748857695925690f, -0.067061292609636822f, 0.997735980512008620f,
	-0.067252607322993499f,
	0.997723066644191640f, -0.067443919563664051f, 0.997710116092949570f,
	-0.067635229324614479f,
	0.997697128858758500f, -0.067826536598810869f, 0.997684104942096030f,
	-0.068017841379219388f,
	0.997671044343441000f, -0.068209143658806329f, 0.997657947063273710f,
	-0.068400443430538013f,
	0.997644813102075420f, -0.068591740687380942f, 0.997631642460329320f,
	-0.068783035422301630f,
	0.997618435138519550f, -0.068974327628266746f, 0.997605191137131640f,
	-0.069165617298242985f,
	0.997591910456652630f, -0.069356904425197208f, 0.997578593097570800f,
	-0.069548189002096306f,
	0.997565239060375750f, -0.069739471021907307f, 0.997551848345558430f,
	-0.069930750477597309f,
	0.997538420953611340f, -0.070122027362133521f, 0.997524956885027960f,
	-0.070313301668483250f,
	0.997511456140303450f, -0.070504573389613856f, 0.997497918719934210f,
	-0.070695842518492855f,
	0.997484344624417930f, -0.070887109048087801f, 0.997470733854253670f,
	-0.071078372971366405f,
	0.997457086409941910f, -0.071269634281296401f, 0.997443402291984360f,
	-0.071460892970845680f,
	0.997429681500884180f, -0.071652149032982212f, 0.997415924037145960f,
	-0.071843402460674027f,
	0.997402129901275300f, -0.072034653246889332f, 0.997388299093779460f,
	-0.072225901384596322f,
	0.997374431615167150f, -0.072417146866763413f, 0.997360527465947940f,
	-0.072608389686358993f,
	0.997346586646633230f, -0.072799629836351673f, 0.997332609157735470f,
	-0.072990867309710036f,
	0.997318594999768600f, -0.073182102099402888f, 0.997304544173247990f,
	-0.073373334198399032f,
	0.997290456678690210f, -0.073564563599667426f, 0.997276332516613180f,
	-0.073755790296177098f,
	0.997262171687536170f, -0.073947014280897200f, 0.997247974191979860f,
	-0.074138235546796979f,
	0.997233740030466280f, -0.074329454086845756f, 0.997219469203518670f,
	-0.074520669894013000f,
	0.997205161711661850f, -0.074711882961268211f, 0.997190817555421940f,
	-0.074903093281581082f,
	0.997176436735326190f, -0.075094300847921305f, 0.997162019251903290f,
	-0.075285505653258769f,
	0.997147565105683480f, -0.075476707690563388f, 0.997133074297198110f,
	-0.075667906952805231f,
	0.997118546826979980f, -0.075859103432954447f, 0.997103982695563330f,
	-0.076050297123981259f,
	0.997089381903483400f, -0.076241488018856066f, 0.997074744451277310f,
	-0.076432676110549283f,
	0.997060070339482960f, -0.076623861392031492f, 0.997045359568640040f,
	-0.076815043856273343f,
	0.997030612139289450f, -0.077006223496245640f, 0.997015828051973310f,
	-0.077197400304919200f,
	0.997001007307235290f, -0.077388574275265049f, 0.996986149905620180f,
	-0.077579745400254224f,
	0.996971255847674320f, -0.077770913672857947f, 0.996956325133945280f,
	-0.077962079086047492f,
	0.996941357764982160f, -0.078153241632794232f, 0.996926353741335090f,
	-0.078344401306069705f,
	0.996911313063555740f, -0.078535558098845479f, 0.996896235732197210f,
	-0.078726712004093299f,
	0.996881121747813850f, -0.078917863014784942f, 0.996865971110961310f,
	-0.079109011123892375f,
	0.996850783822196610f, -0.079300156324387597f, 0.996835559882078170f,
	-0.079491298609242769f,
	0.996820299291165670f, -0.079682437971430126f, 0.996805002050020430f,
	-0.079873574403921996f,
	0.996789668159204560f, -0.080064707899690890f, 0.996774297619282050f,
	-0.080255838451709319f,
	0.996758890430818000f, -0.080446966052950014f, 0.996743446594378860f,
	-0.080638090696385709f,
	0.996727966110532490f, -0.080829212374989329f, 0.996712448979848010f,
	-0.081020331081733857f,
	0.996696895202896060f, -0.081211446809592441f, 0.996681304780248300f,
	-0.081402559551538245f,
	0.996665677712478160f, -0.081593669300544652f, 0.996650014000160070f,
	-0.081784776049585076f,
	0.996634313643869900f, -0.081975879791633066f, 0.996618576644185070f,
	-0.082166980519662314f,
	0.996602803001684130f, -0.082358078226646536f, 0.996586992716946950f,
	-0.082549172905559673f,
	0.996571145790554840f, -0.082740264549375692f, 0.996555262223090540f,
	-0.082931353151068699f,
	0.996539342015137940f, -0.083122438703612911f, 0.996523385167282450f,
	-0.083313521199982685f,
	0.996507391680110820f, -0.083504600633152432f, 0.996491361554210920f,
	-0.083695676996096716f,
	0.996475294790172160f, -0.083886750281790226f, 0.996459191388585410f,
	-0.084077820483207694f,
	0.996443051350042630f, -0.084268887593324071f, 0.996426874675137240f,
	-0.084459951605114325f,
	0.996410661364464100f, -0.084651012511553617f, 0.996394411418619290f,
	-0.084842070305617134f,
	0.996378124838200210f, -0.085033124980280275f, 0.996361801623805720f,
	-0.085224176528518478f,
	0.996345441776035900f, -0.085415224943307333f, 0.996329045295492380f,
	-0.085606270217622529f,
	0.996312612182778000f, -0.085797312344439894f, 0.996296142438496850f,
	-0.085988351316735337f,
	0.996279636063254650f, -0.086179387127484894f, 0.996263093057658140f,
	-0.086370419769664752f,
	0.996246513422315520f, -0.086561449236251170f, 0.996229897157836500f,
	-0.086752475520220543f,
	0.996213244264832040f, -0.086943498614549378f, 0.996196554743914220f,
	-0.087134518512214307f,
	0.996179828595696980f, -0.087325535206192059f, 0.996163065820794950f,
	-0.087516548689459531f,
	0.996146266419824620f, -0.087707558954993659f, 0.996129430393403740f,
	-0.087898565995771588f,
	0.996112557742151130f, -0.088089569804770507f, 0.996095648466687300f,
	-0.088280570374967740f,
	0.996078702567633980f, -0.088471567699340767f, 0.996061720045614000f,
	-0.088662561770867149f,
	0.996044700901251970f, -0.088853552582524600f, 0.996027645135173610f,
	-0.089044540127290892f,
	0.996010552748005870f, -0.089235524398144014f, 0.995993423740377360f,
	-0.089426505388061961f,
	0.995976258112917790f, -0.089617483090022959f, 0.995959055866258320f,
	-0.089808457497005278f,
	0.995941817001031350f, -0.089999428601987341f, 0.995924541517870800f,
	-0.090190396397947695f,
	0.995907229417411720f, -0.090381360877864983f, 0.995889880700290720f,
	-0.090572322034717989f,
	0.995872495367145730f, -0.090763279861485621f, 0.995855073418615790f,
	-0.090954234351146926f,
	0.995837614855341610f, -0.091145185496681005f, 0.995820119677964910f,
	-0.091336133291067184f,
	0.995802587887129160f, -0.091527077727284828f, 0.995785019483478750f,
	-0.091718018798313455f,
	0.995767414467659820f, -0.091908956497132724f, 0.995749772840319510f,
	-0.092099890816722388f,
	0.995732094602106430f, -0.092290821750062355f, 0.995714379753670610f,
	-0.092481749290132600f,
	0.995696628295663520f, -0.092672673429913310f, 0.995678840228737540f,
	-0.092863594162384724f,
	0.995661015553546910f, -0.093054511480527249f, 0.995643154270746900f,
	-0.093245425377321375f,
	0.995625256380994310f, -0.093436335845747787f, 0.995607321884947050f,
	-0.093627242878787195f,
	0.995589350783264600f, -0.093818146469420549f, 0.995571343076607770f,
	-0.094009046610628838f,
	0.995553298765638470f, -0.094199943295393204f, 0.995535217851020390f,
	-0.094390836516694943f,
	0.995517100333418110f, -0.094581726267515445f, 0.995498946213497770f,
	-0.094772612540836243f,
	0.995480755491926940f, -0.094963495329638992f, 0.995462528169374420f,
	-0.095154374626905486f,
	0.995444264246510340f, -0.095345250425617617f, 0.995425963724006160f,
	-0.095536122718757471f,
	0.995407626602534900f, -0.095726991499307162f, 0.995389252882770690f,
	-0.095917856760249040f,
	0.995370842565388990f, -0.096108718494565509f, 0.995352395651066810f,
	-0.096299576695239128f,
	0.995333912140482280f, -0.096490431355252593f, 0.995315392034315070f,
	-0.096681282467588725f,
	0.995296835333246090f, -0.096872130025230471f, 0.995278242037957670f,
	-0.097062974021160917f,
	0.995259612149133390f, -0.097253814448363271f, 0.995240945667458130f,
	-0.097444651299820870f,
	0.995222242593618360f, -0.097635484568517200f, 0.995203502928301510f,
	-0.097826314247435861f,
	0.995184726672196930f, -0.098017140329560604f, 0.995165913825994620f,
	-0.098207962807875276f,
	0.995147064390386470f, -0.098398781675363881f, 0.995128178366065490f,
	-0.098589596925010584f,
	0.995109255753726110f, -0.098780408549799623f, 0.995090296554064000f,
	-0.098971216542715429f,
	0.995071300767776170f, -0.099162020896742503f, 0.995052268395561050f,
	-0.099352821604865540f,
	0.995033199438118630f, -0.099543618660069319f, 0.995014093896149700f,
	-0.099734412055338825f,
	0.994994951770357020f, -0.099925201783659073f, 0.994975773061444140f,
	-0.100115987838015310f,
	0.994956557770116380f, -0.100306770211392860f, 0.994937305897080070f,
	-0.100497548896777200f,
	0.994918017443043200f, -0.100688323887153960f, 0.994898692408714870f,
	-0.100879095175508860f,
	0.994879330794805620f, -0.101069862754827820f, 0.994859932602027320f,
	-0.101260626618096830f,
	0.994840497831093180f, -0.101451386758302080f, 0.994821026482717860f,
	-0.101642143168429830f,
	0.994801518557617110f, -0.101832895841466530f, 0.994781974056508260f,
	-0.102023644770398740f,
	0.994762392980109930f, -0.102214389948213210f, 0.994742775329142010f,
	-0.102405131367896720f,
	0.994723121104325700f, -0.102595869022436280f, 0.994703430306383860f,
	-0.102786602904819040f,
	0.994683702936040250f, -0.102977333008032220f, 0.994663938994020390f,
	-0.103168059325063230f,
	0.994644138481050710f, -0.103358781848899610f, 0.994624301397859400f,
	-0.103549500572529070f,
	0.994604427745175660f, -0.103740215488939370f, 0.994584517523730340f,
	-0.103930926591118510f,
	0.994564570734255420f, -0.104121633872054590f, 0.994544587377484300f,
	-0.104312337324735800f,
	0.994524567454151740f, -0.104503036942150570f, 0.994504510964993700f,
	-0.104693732717287390f,
	0.994484417910747600f, -0.104884424643134970f, 0.994464288292152390f,
	-0.105075112712682040f,
	0.994444122109948040f, -0.105265796918917600f, 0.994423919364875950f,
	-0.105456477254830710f,
	0.994403680057679100f, -0.105647153713410620f, 0.994383404189101430f,
	-0.105837826287646670f,
	0.994363091759888570f, -0.106028494970528410f, 0.994342742770787270f,
	-0.106219159755045480f,
	0.994322357222545810f, -0.106409820634187680f, 0.994301935115913580f,
	-0.106600477600944960f,
	0.994281476451641550f, -0.106791130648307390f, 0.994260981230481790f,
	-0.106981779769265230f,
	0.994240449453187900f, -0.107172424956808840f, 0.994219881120514960f,
	-0.107363066203928760f,
	0.994199276233218910f, -0.107553703503615620f, 0.994178634792057590f,
	-0.107744336848860280f,
	0.994157956797789730f, -0.107934966232653650f, 0.994137242251175720f,
	-0.108125591647986870f,
	0.994116491152977070f, -0.108316213087851170f, 0.994095703503956930f,
	-0.108506830545237920f,
	0.994074879304879370f, -0.108697444013138720f, 0.994054018556510210f,
	-0.108888053484545190f,
	0.994033121259616400f, -0.109078658952449240f, 0.994012187414966220f,
	-0.109269260409842780f,
	0.993991217023329380f, -0.109459857849717980f, 0.993970210085476920f,
	-0.109650451265067100f,
	0.993949166602181130f, -0.109841040648882600f, 0.993928086574215830f,
	-0.110031625994157000f,
	0.993906970002356060f, -0.110222207293883060f, 0.993885816887378090f,
	-0.110412784541053630f,
	0.993864627230059750f, -0.110603357728661730f, 0.993843401031180180f,
	-0.110793926849700560f,
	0.993822138291519660f, -0.110984491897163390f, 0.993800839011860120f,
	-0.111175052864043720f,
	0.993779503192984580f, -0.111365609743335160f, 0.993758130835677430f,
	-0.111556162528031480f,
	0.993736721940724600f, -0.111746711211126590f, 0.993715276508913230f,
	-0.111937255785614570f,
	0.993693794541031790f, -0.112127796244489640f, 0.993672276037870010f,
	-0.112318332580746170f,
	0.993650721000219120f, -0.112508864787378690f, 0.993629129428871720f,
	-0.112699392857381860f,
	0.993607501324621610f, -0.112889916783750520f, 0.993585836688263950f,
	-0.113080436559479620f,
	0.993564135520595300f, -0.113270952177564350f, 0.993542397822413600f,
	-0.113461463630999950f,
	0.993520623594518090f, -0.113651970912781870f, 0.993498812837709360f,
	-0.113842474015905710f,
	0.993476965552789190f, -0.114032972933367200f, 0.993455081740560960f,
	-0.114223467658162260f,
	0.993433161401829360f, -0.114413958183286920f, 0.993411204537400060f,
	-0.114604444501737420f,
	0.993389211148080650f, -0.114794926606510080f, 0.993367181234679600f,
	-0.114985404490601460f,
	0.993345114798006910f, -0.115175878147008190f, 0.993323011838873950f,
	-0.115366347568727140f,
	0.993300872358093280f, -0.115556812748755260f, 0.993278696356479030f,
	-0.115747273680089720f,
	0.993256483834846440f, -0.115937730355727780f, 0.993234234794012290f,
	-0.116128182768666930f,
	0.993211949234794500f, -0.116318630911904750f, 0.993189627158012620f,
	-0.116509074778439040f,
	0.993167268564487230f, -0.116699514361267690f, 0.993144873455040430f,
	-0.116889949653388780f,
	0.993122441830495580f, -0.117080380647800590f, 0.993099973691677570f,
	-0.117270807337501460f,
	0.993077469039412300f, -0.117461229715489990f, 0.993054927874527320f,
	-0.117651647774764860f,
	0.993032350197851410f, -0.117842061508324980f, 0.993009736010214580f,
	-0.118032470909169340f,
	0.992987085312448390f, -0.118222875970297170f, 0.992964398105385610f,
	-0.118413276684707790f,
	0.992941674389860470f, -0.118603673045400720f, 0.992918914166708300f,
	-0.118794065045375640f,
	0.992896117436765980f, -0.118984452677632340f, 0.992873284200871730f,
	-0.119174835935170880f,
	0.992850414459865100f, -0.119365214810991350f, 0.992827508214586760f,
	-0.119555589298094110f,
	0.992804565465879140f, -0.119745959389479600f, 0.992781586214585570f,
	-0.119936325078148470f,
	0.992758570461551140f, -0.120126686357101500f, 0.992735518207621850f,
	-0.120317043219339680f,
	0.992712429453645460f, -0.120507395657864130f, 0.992689304200470750f,
	-0.120697743665676110f,
	0.992666142448948020f, -0.120888087235777080f, 0.992642944199928820f,
	-0.121078426361168640f,
	0.992619709454266140f, -0.121268761034852600f, 0.992596438212814290f,
	-0.121459091249830840f,
	0.992573130476428810f, -0.121649416999105530f, 0.992549786245966680f,
	-0.121839738275678890f,
	0.992526405522286100f, -0.122030055072553360f, 0.992502988306246950f,
	-0.122220367382731540f,
	0.992479534598709970f, -0.122410675199216200f, 0.992456044400537700f,
	-0.122600978515010240f,
	0.992432517712593660f, -0.122791277323116770f, 0.992408954535742850f,
	-0.122981571616539050f,
	0.992385354870851670f, -0.123171861388280480f, 0.992361718718787870f,
	-0.123362146631344680f,
	0.992338046080420420f, -0.123552427338735370f, 0.992314336956619640f,
	-0.123742703503456510f,
	0.992290591348257370f, -0.123932975118512160f, 0.992266809256206580f,
	-0.124123242176906600f,
	0.992242990681341700f, -0.124313504671644230f, 0.992219135624538450f,
	-0.124503762595729660f,
	0.992195244086673920f, -0.124694015942167640f, 0.992171316068626520f,
	-0.124884264703963130f,
	0.992147351571276090f, -0.125074508874121170f, 0.992123350595503720f,
	-0.125264748445647060f,
	0.992099313142191800f, -0.125454983411546230f, 0.992075239212224070f,
	-0.125645213764824290f,
	0.992051128806485720f, -0.125835439498487000f, 0.992026981925863360f,
	-0.126025660605540320f,
	0.992002798571244520f, -0.126215877078990350f, 0.991978578743518580f,
	-0.126406088911843380f,
	0.991954322443575950f, -0.126596296097105850f, 0.991930029672308480f,
	-0.126786498627784410f,
	0.991905700430609330f, -0.126976696496885870f, 0.991881334719373010f,
	-0.127166889697417160f,
	0.991856932539495470f, -0.127357078222385400f, 0.991832493891873780f,
	-0.127547262064797970f,
	0.991808018777406430f, -0.127737441217662310f, 0.991783507196993490f,
	-0.127927615673986080f,
	0.991758959151536110f, -0.128117785426777130f, 0.991734374641936810f,
	-0.128307950469043420f,
	0.991709753669099530f, -0.128498110793793170f, 0.991685096233929420f,
	-0.128688266394034690f,
	0.991660402337333210f, -0.128878417262776550f, 0.991635671980218740f,
	-0.129068563393027410f,
	0.991610905163495370f, -0.129258704777796140f, 0.991586101888073500f,
	-0.129448841410091780f,
	0.991561262154865290f, -0.129638973282923560f, 0.991536385964783880f,
	-0.129829100389300930f,
	0.991511473318743900f, -0.130019222722233350f, 0.991486524217661480f,
	-0.130209340274730630f,
	0.991461538662453790f, -0.130399453039802690f, 0.991436516654039420f,
	-0.130589561010459650f,
	0.991411458193338540f, -0.130779664179711710f, 0.991386363281272280f,
	-0.130969762540569380f,
	0.991361231918763460f, -0.131159856086043270f, 0.991336064106736140f,
	-0.131349944809144190f,
	0.991310859846115440f, -0.131540028702883120f, 0.991285619137828200f,
	-0.131730107760271160f,
	0.991260341982802440f, -0.131920181974319790f, 0.991235028381967420f,
	-0.132110251338040360f,
	0.991209678336254060f, -0.132300315844444650f, 0.991184291846594180f,
	-0.132490375486544550f,
	0.991158868913921350f, -0.132680430257352070f, 0.991133409539170170f,
	-0.132870480149879430f,
	0.991107913723276890f, -0.133060525157139060f, 0.991082381467178640f,
	-0.133250565272143570f,
	0.991056812771814340f, -0.133440600487905680f, 0.991031207638124130f,
	-0.133630630797438340f,
	0.991005566067049370f, -0.133820656193754720f, 0.990979888059532740f,
	-0.134010676669868130f,
	0.990954173616518500f, -0.134200692218792020f, 0.990928422738951990f,
	-0.134390702833540070f,
	0.990902635427780010f, -0.134580708507126170f, 0.990876811683950700f,
	-0.134770709232564350f,
	0.990850951508413620f, -0.134960705002868750f, 0.990825054902119470f,
	-0.135150695811053850f,
	0.990799121866020370f, -0.135340681650134210f, 0.990773152401069780f,
	-0.135530662513124590f,
	0.990747146508222710f, -0.135720638393039910f, 0.990721104188435180f,
	-0.135910609282895330f,
	0.990695025442664630f, -0.136100575175706200f, 0.990668910271870100f,
	-0.136290536064487960f,
	0.990642758677011570f, -0.136480491942256280f, 0.990616570659050620f,
	-0.136670442802027090f,
	0.990590346218950150f, -0.136860388636816380f, 0.990564085357674370f,
	-0.137050329439640410f,
	0.990537788076188750f, -0.137240265203515590f, 0.990511454375460290f,
	-0.137430195921458550f,
	0.990485084256457090f, -0.137620121586486040f, 0.990458677720148620f,
	-0.137810042191615080f,
	0.990432234767505970f, -0.137999957729862790f, 0.990405755399501260f,
	-0.138189868194246560f,
	0.990379239617108160f, -0.138379773577783890f, 0.990352687421301450f,
	-0.138569673873492500f,
	0.990326098813057330f, -0.138759569074390350f, 0.990299473793353590f,
	-0.138949459173495490f,
	0.990272812363169110f, -0.139139344163826200f, 0.990246114523483990f,
	-0.139329224038400980f,
	0.990219380275280000f, -0.139519098790238490f, 0.990192609619540030f,
	-0.139708968412357550f,
	0.990165802557248400f, -0.139898832897777210f, 0.990138959089390650f,
	-0.140088692239516670f,
	0.990112079216953770f, -0.140278546430595420f, 0.990085162940925970f,
	-0.140468395464033000f,
	0.990058210262297120f, -0.140658239332849210f, 0.990031221182058000f,
	-0.140848078030064080f,
	0.990004195701200910f, -0.141037911548697710f, 0.989977133820719610f,
	-0.141227739881770510f,
	0.989950035541608990f, -0.141417563022303020f, 0.989922900864865450f,
	-0.141607380963316020f,
	0.989895729791486660f, -0.141797193697830390f, 0.989868522322471580f,
	-0.141987001218867290f,
	0.989841278458820530f, -0.142176803519448030f, 0.989813998201535260f,
	-0.142366600592594180f,
	0.989786681551618640f, -0.142556392431327340f, 0.989759328510075200f,
	-0.142746179028669460f,
	0.989731939077910570f, -0.142935960377642670f, 0.989704513256131850f,
	-0.143125736471269190f,
	0.989677051045747210f, -0.143315507302571500f, 0.989649552447766530f,
	-0.143505272864572290f,
	0.989622017463200890f, -0.143695033150294470f, 0.989594446093062460f,
	-0.143884788152760980f,
	0.989566838338365120f, -0.144074537864995160f, 0.989539194200123930f,
	-0.144264282280020440f,
	0.989511513679355190f, -0.144454021390860470f, 0.989483796777076760f,
	-0.144643755190539040f,
	0.989456043494307710f, -0.144833483672080210f, 0.989428253832068230f,
	-0.145023206828508220f,
	0.989400427791380380f, -0.145212924652847460f, 0.989372565373267010f,
	-0.145402637138122570f,
	0.989344666578752640f, -0.145592344277358340f, 0.989316731408863000f,
	-0.145782046063579860f,
	0.989288759864625170f, -0.145971742489812210f, 0.989260751947067640f,
	-0.146161433549080900f,
	0.989232707657220050f, -0.146351119234411460f, 0.989204626996113780f,
	-0.146540799538829760f,
	0.989176509964781010f, -0.146730474455361750f, 0.989148356564255590f,
	-0.146920143977033620f,
	0.989120166795572690f, -0.147109808096871820f, 0.989091940659768800f,
	-0.147299466807902850f,
	0.989063678157881540f, -0.147489120103153570f, 0.989035379290950310f,
	-0.147678767975650970f,
	0.989007044060015270f, -0.147868410418422220f, 0.988978672466118480f,
	-0.148058047424494720f,
	0.988950264510302990f, -0.148247678986896030f, 0.988921820193613190f,
	-0.148437305098653970f,
	0.988893339517095130f, -0.148626925752796540f, 0.988864822481795640f,
	-0.148816540942351920f,
	0.988836269088763540f, -0.149006150660348450f, 0.988807679339048450f,
	-0.149195754899814820f,
	0.988779053233701520f, -0.149385353653779720f, 0.988750390773775360f,
	-0.149574946915272230f,
	0.988721691960323780f, -0.149764534677321510f, 0.988692956794401940f,
	-0.149954116932956960f,
	0.988664185277066230f, -0.150143693675208190f, 0.988635377409374790f,
	-0.150333264897105000f,
	0.988606533192386450f, -0.150522830591677400f, 0.988577652627162020f,
	-0.150712390751955610f,
	0.988548735714763200f, -0.150901945370970040f, 0.988519782456253270f,
	-0.151091494441751300f,
	0.988490792852696590f, -0.151281037957330220f, 0.988461766905159300f,
	-0.151470575910737810f,
	0.988432704614708340f, -0.151660108295005310f, 0.988403605982412390f,
	-0.151849635103164180f,
	0.988374471009341280f, -0.152039156328246050f, 0.988345299696566150f,
	-0.152228671963282740f,
	0.988316092045159690f, -0.152418182001306330f, 0.988286848056195820f,
	-0.152607686435349050f,
	0.988257567730749460f, -0.152797185258443440f, 0.988228251069897420f,
	-0.152986678463622040f,
	0.988198898074717610f, -0.153176166043917840f, 0.988169508746289060f,
	-0.153365647992363880f,
	0.988140083085692570f, -0.153555124301993450f, 0.988110621094009820f,
	-0.153744594965840030f,
	0.988081122772324070f, -0.153934059976937350f, 0.988051588121720110f,
	-0.154123519328319360f,
	0.988022017143283530f, -0.154312973013020100f, 0.987992409838101880f,
	-0.154502421024073940f,
	0.987962766207263420f, -0.154691863354515430f, 0.987933086251858380f,
	-0.154881299997379320f,
	0.987903369972977790f, -0.155070730945700510f, 0.987873617371714200f,
	-0.155260156192514240f,
	0.987843828449161740f, -0.155449575730855850f, 0.987814003206415550f,
	-0.155638989553760900f,
	0.987784141644572180f, -0.155828397654265230f, 0.987754243764729530f,
	-0.156017800025404800f,
	0.987724309567986960f, -0.156207196660215900f, 0.987694339055445130f,
	-0.156396587551734880f,
	0.987664332228205710f, -0.156585972692998430f, 0.987634289087372160f,
	-0.156775352077043350f,
	0.987604209634049160f, -0.156964725696906780f, 0.987574093869342360f,
	-0.157154093545625900f,
	0.987543941794359230f, -0.157343455616238250f, 0.987513753410208420f,
	-0.157532811901781530f,
	0.987483528717999710f, -0.157722162395293630f, 0.987453267718844560f,
	-0.157911507089812660f,
	0.987422970413855410f, -0.158100845978376980f, 0.987392636804146240f,
	-0.158290179054025180f,
	0.987362266890832400f, -0.158479506309795960f, 0.987331860675030430f,
	-0.158668827738728310f,
	0.987301418157858430f, -0.158858143333861450f, 0.987270939340435420f,
	-0.159047453088234760f,
	0.987240424223882250f, -0.159236756994887850f, 0.987209872809320820f,
	-0.159426055046860580f,
	0.987179285097874340f, -0.159615347237193060f, 0.987148661090667570f,
	-0.159804633558925440f,
	0.987118000788826280f, -0.159993914005098270f, 0.987087304193477900f,
	-0.160183188568752220f,
	0.987056571305750970f, -0.160372457242928280f, 0.987025802126775600f,
	-0.160561720020667490f,
	0.986994996657682980f, -0.160750976895011220f, 0.986964154899605650f,
	-0.160940227859001080f,
	0.986933276853677710f, -0.161129472905678810f, 0.986902362521034470f,
	-0.161318712028086400f,
	0.986871411902812470f, -0.161507945219266120f, 0.986840425000149680f,
	-0.161697172472260400f,
	0.986809401814185530f, -0.161886393780111830f, 0.986778342346060430f,
	-0.162075609135863330f,
	0.986747246596916590f, -0.162264818532558000f, 0.986716114567897100f,
	-0.162454021963239190f,
	0.986684946260146690f, -0.162643219420950310f, 0.986653741674811350f,
	-0.162832410898735210f,
	0.986622500813038480f, -0.163021596389637840f, 0.986591223675976400f,
	-0.163210775886702380f,
	0.986559910264775410f, -0.163399949382973230f, 0.986528560580586690f,
	-0.163589116871495020f,
	0.986497174624562880f, -0.163778278345312670f, 0.986465752397857940f,
	-0.163967433797471170f,
	0.986434293901627180f, -0.164156583221015810f, 0.986402799137027220f,
	-0.164345726608992190f,
	0.986371268105216030f, -0.164534863954446000f, 0.986339700807353000f,
	-0.164723995250423170f,
	0.986308097244598670f, -0.164913120489969890f, 0.986276457418115090f,
	-0.165102239666132660f,
	0.986244781329065460f, -0.165291352771958000f, 0.986213068978614490f,
	-0.165480459800492780f,
	0.986181320367928270f, -0.165669560744784120f, 0.986149535498173860f,
	-0.165858655597879300f,
	0.986117714370520090f, -0.166047744352825790f, 0.986085856986136820f,
	-0.166236827002671420f,
	0.986053963346195440f, -0.166425903540464100f, 0.986022033451868560f,
	-0.166614973959252090f,
	0.985990067304330140f, -0.166804038252083730f, 0.985958064904755460f,
	-0.166993096412007710f,
	0.985926026254321130f, -0.167182148432072940f, 0.985893951354205210f,
	-0.167371194305328430f,
	0.985861840205586980f, -0.167560234024823560f, 0.985829692809647050f,
	-0.167749267583607890f,
	0.985797509167567480f, -0.167938294974731170f, 0.985765289280531310f,
	-0.168127316191243410f,
	0.985733033149723490f, -0.168316331226194830f, 0.985700740776329850f,
	-0.168505340072635900f,
	0.985668412161537550f, -0.168694342723617330f, 0.985636047306535420f,
	-0.168883339172189980f,
	0.985603646212513400f, -0.169072329411405010f, 0.985571208880662740f,
	-0.169261313434313830f,
	0.985538735312176060f, -0.169450291233967960f, 0.985506225508247290f,
	-0.169639262803419290f,
	0.985473679470071810f, -0.169828228135719850f, 0.985441097198846210f,
	-0.170017187223921950f,
	0.985408478695768420f, -0.170206140061078070f, 0.985375823962037710f,
	-0.170395086640240940f,
	0.985343132998854790f, -0.170584026954463590f, 0.985310405807421570f,
	-0.170772960996799230f,
	0.985277642388941220f, -0.170961888760301220f, 0.985244842744618540f,
	-0.171150810238023280f,
	0.985212006875659350f, -0.171339725423019310f, 0.985179134783271130f,
	-0.171528634308343420f,
	0.985146226468662230f, -0.171717536887049970f, 0.985113281933042710f,
	-0.171906433152193530f,
	0.985080301177623800f, -0.172095323096829010f, 0.985047284203618200f,
	-0.172284206714011370f,
	0.985014231012239840f, -0.172473083996795950f, 0.984981141604703960f,
	-0.172661954938238270f,
	0.984948015982227030f, -0.172850819531394080f, 0.984914854146027200f,
	-0.173039677769319360f,
	0.984881656097323700f, -0.173228529645070320f, 0.984848421837337010f,
	-0.173417375151703470f,
	0.984815151367289140f, -0.173606214282275410f, 0.984781844688403350f,
	-0.173795047029843160f,
	0.984748501801904210f, -0.173983873387463820f, 0.984715122709017620f,
	-0.174172693348194820f,
	0.984681707410970940f, -0.174361506905093750f, 0.984648255908992630f,
	-0.174550314051218510f,
	0.984614768204312600f, -0.174739114779627200f, 0.984581244298162180f,
	-0.174927909083378160f,
	0.984547684191773960f, -0.175116696955529920f, 0.984514087886381840f,
	-0.175305478389141320f,
	0.984480455383220930f, -0.175494253377271430f, 0.984446786683527920f,
	-0.175683021912979490f,
	0.984413081788540700f, -0.175871783989325040f, 0.984379340699498510f,
	-0.176060539599367820f,
	0.984345563417641900f, -0.176249288736167880f, 0.984311749944212780f,
	-0.176438031392785410f,
	0.984277900280454370f, -0.176626767562280880f, 0.984244014427611110f,
	-0.176815497237715000f,
	0.984210092386929030f, -0.177004220412148750f, 0.984176134159655320f,
	-0.177192937078643280f,
	0.984142139747038570f, -0.177381647230260040f, 0.984108109150328540f,
	-0.177570350860060710f,
	0.984074042370776450f, -0.177759047961107170f, 0.984039939409634970f,
	-0.177947738526461560f,
	0.984005800268157870f, -0.178136422549186300f, 0.983971624947600270f,
	-0.178325100022344000f,
	0.983937413449218920f, -0.178513770938997510f, 0.983903165774271500f,
	-0.178702435292209970f,
	0.983868881924017220f, -0.178891093075044720f, 0.983834561899716630f,
	-0.179079744280565390f,
	0.983800205702631600f, -0.179268388901835750f, 0.983765813334025240f,
	-0.179457026931919890f,
	0.983731384795162090f, -0.179645658363882160f, 0.983696920087308140f,
	-0.179834283190787090f,
	0.983662419211730250f, -0.180022901405699510f, 0.983627882169697210f,
	-0.180211513001684450f,
	0.983593308962478650f, -0.180400117971807240f, 0.983558699591345900f,
	-0.180588716309133340f,
	0.983524054057571260f, -0.180777308006728590f, 0.983489372362428730f,
	-0.180965893057658980f,
	0.983454654507193270f, -0.181154471454990810f, 0.983419900493141540f,
	-0.181343043191790540f,
	0.983385110321551180f, -0.181531608261124970f, 0.983350283993701500f,
	-0.181720166656061110f,
	0.983315421510872810f, -0.181908718369666160f, 0.983280522874346970f,
	-0.182097263395007650f,
	0.983245588085407070f, -0.182285801725153300f, 0.983210617145337640f,
	-0.182474333353171120f,
	0.983175610055424420f, -0.182662858272129270f, 0.983140566816954500f,
	-0.182851376475096330f,
	0.983105487431216290f, -0.183039887955140950f, 0.983070371899499640f,
	-0.183228392705332140f,
	0.983035220223095640f, -0.183416890718739100f, 0.983000032403296590f,
	-0.183605381988431270f,
	0.982964808441396440f, -0.183793866507478450f, 0.982929548338690170f,
	-0.183982344268950520f,
	0.982894252096474070f, -0.184170815265917720f, 0.982858919716046110f,
	-0.184359279491450510f,
	0.982823551198705240f, -0.184547736938619620f, 0.982788146545751970f,
	-0.184736187600495950f,
	0.982752705758487830f, -0.184924631470150790f, 0.982717228838215990f,
	-0.185113068540655540f,
	0.982681715786240860f, -0.185301498805081900f, 0.982646166603868050f,
	-0.185489922256501880f,
	0.982610581292404750f, -0.185678338887987630f, 0.982574959853159240f,
	-0.185866748692611660f,
	0.982539302287441240f, -0.186055151663446630f, 0.982503608596561830f,
	-0.186243547793565560f,
	0.982467878781833170f, -0.186431937076041610f, 0.982432112844569110f,
	-0.186620319503948280f,
	0.982396310786084690f, -0.186808695070359270f, 0.982360472607696210f,
	-0.186997063768348540f,
	0.982324598310721280f, -0.187185425590990330f, 0.982288687896478830f,
	-0.187373780531359110f,
	0.982252741366289370f, -0.187562128582529600f, 0.982216758721474510f,
	-0.187750469737576780f,
	0.982180739963357090f, -0.187938803989575910f, 0.982144685093261580f,
	-0.188127131331602420f,
	0.982108594112513610f, -0.188315451756732120f, 0.982072467022440000f,
	-0.188503765258040940f,
	0.982036303824369020f, -0.188692071828605230f, 0.982000104519630490f,
	-0.188880371461501380f,
	0.981963869109555240f, -0.189068664149806190f, 0.981927597595475540f,
	-0.189256949886596750f,
	0.981891289978725100f, -0.189445228664950230f, 0.981854946260638630f,
	-0.189633500477944190f,
	0.981818566442552500f, -0.189821765318656410f, 0.981782150525804310f,
	-0.190010023180164990f,
	0.981745698511732990f, -0.190198274055548150f, 0.981709210401678800f,
	-0.190386517937884470f,
	0.981672686196983110f, -0.190574754820252740f, 0.981636125898989080f,
	-0.190762984695732110f,
	0.981599529509040720f, -0.190951207557401800f, 0.981562897028483650f,
	-0.191139423398341450f,
	0.981526228458664770f, -0.191327632211630900f, 0.981489523800932130f,
	-0.191515833990350210f,
	0.981452783056635520f, -0.191704028727579800f, 0.981416006227125550f,
	-0.191892216416400220f,
	0.981379193313754560f, -0.192080397049892440f, 0.981342344317876040f,
	-0.192268570621137500f,
	0.981305459240844670f, -0.192456737123216840f, 0.981268538084016710f,
	-0.192644896549212100f,
	0.981231580848749730f, -0.192833048892205230f, 0.981194587536402320f,
	-0.193021194145278380f,
	0.981157558148334830f, -0.193209332301513960f, 0.981120492685908730f,
	-0.193397463353994740f,
	0.981083391150486710f, -0.193585587295803610f, 0.981046253543432780f,
	-0.193773704120023820f,
	0.981009079866112630f, -0.193961813819738840f, 0.980971870119892840f,
	-0.194149916388032450f,
	0.980934624306141640f, -0.194338011817988600f, 0.980897342426228390f,
	-0.194526100102691610f,
	0.980860024481523870f, -0.194714181235225960f, 0.980822670473400100f,
	-0.194902255208676520f,
	0.980785280403230430f, -0.195090322016128250f, 0.980747854272389750f,
	-0.195278381650666550f,
	0.980710392082253970f, -0.195466434105376980f, 0.980672893834200530f,
	-0.195654479373345370f,
	0.980635359529608120f, -0.195842517447657850f, 0.980597789169856850f,
	-0.196030548321400790f,
	0.980560182756327840f, -0.196218571987660880f, 0.980522540290404090f,
	-0.196406588439524970f,
	0.980484861773469380f, -0.196594597670080220f, 0.980447147206909060f,
	-0.196782599672414100f,
	0.980409396592109910f, -0.196970594439614340f, 0.980371609930459800f,
	-0.197158581964768880f,
	0.980333787223347960f, -0.197346562240965920f, 0.980295928472165290f,
	-0.197534535261294030f,
	0.980258033678303550f, -0.197722501018841920f, 0.980220102843156080f,
	-0.197910459506698670f,
	0.980182135968117430f, -0.198098410717953560f, 0.980144133054583590f,
	-0.198286354645696220f,
	0.980106094103951770f, -0.198474291283016390f, 0.980068019117620650f,
	-0.198662220623004200f,
	0.980029908096990090f, -0.198850142658750090f, 0.979991761043461200f,
	-0.199038057383344680f,
	0.979953577958436740f, -0.199225964789878830f, 0.979915358843320480f,
	-0.199413864871443770f,
	0.979877103699517640f, -0.199601757621130970f, 0.979838812528434740f,
	-0.199789643032032090f,
	0.979800485331479790f, -0.199977521097239150f, 0.979762122110061750f,
	-0.200165391809844440f,
	0.979723722865591170f, -0.200353255162940450f, 0.979685287599479930f,
	-0.200541111149619980f,
	0.979646816313141210f, -0.200728959762976140f, 0.979608309007989450f,
	-0.200916800996102230f,
	0.979569765685440520f, -0.201104634842091900f, 0.979531186346911500f,
	-0.201292461294039020f,
	0.979492570993820810f, -0.201480280345037730f, 0.979453919627588210f,
	-0.201668091988182530f,
	0.979415232249634780f, -0.201855896216568050f, 0.979376508861383170f,
	-0.202043693023289260f,
	0.979337749464256780f, -0.202231482401441450f, 0.979298954059681040f,
	-0.202419264344120160f,
	0.979260122649082020f, -0.202607038844421130f, 0.979221255233887700f,
	-0.202794805895440440f,
	0.979182351815526930f, -0.202982565490274440f, 0.979143412395430230f,
	-0.203170317622019790f,
	0.979104436975029250f, -0.203358062283773320f, 0.979065425555756930f,
	-0.203545799468632190f,
	0.979026378139047580f, -0.203733529169693920f, 0.978987294726337050f,
	-0.203921251380056120f,
	0.978948175319062200f, -0.204108966092816870f, 0.978909019918661310f,
	-0.204296673301074370f,
	0.978869828526574120f, -0.204484372997927240f, 0.978830601144241470f,
	-0.204672065176474210f,
	0.978791337773105670f, -0.204859749829814420f, 0.978752038414610340f,
	-0.205047426951047250f,
	0.978712703070200420f, -0.205235096533272350f, 0.978673331741322210f,
	-0.205422758569589610f,
	0.978633924429423210f, -0.205610413053099240f, 0.978594481135952270f,
	-0.205798059976901790f,
	0.978555001862359550f, -0.205985699334097910f, 0.978515486610096910f,
	-0.206173331117788710f,
	0.978475935380616830f, -0.206360955321075510f, 0.978436348175373730f,
	-0.206548571937059890f,
	0.978396724995823090f, -0.206736180958843690f, 0.978357065843421640f,
	-0.206923782379529100f,
	0.978317370719627650f, -0.207111376192218560f, 0.978277639625900530f,
	-0.207298962390014750f,
	0.978237872563701090f, -0.207486540966020650f, 0.978198069534491400f,
	-0.207674111913339570f,
	0.978158230539735050f, -0.207861675225075070f, 0.978118355580896660f,
	-0.208049230894330940f,
	0.978078444659442380f, -0.208236778914211330f, 0.978038497776839600f,
	-0.208424319277820600f,
	0.977998514934557140f, -0.208611851978263490f, 0.977958496134064830f,
	-0.208799377008644900f,
	0.977918441376834370f, -0.208986894362070070f, 0.977878350664338150f,
	-0.209174404031644580f,
	0.977838223998050430f, -0.209361906010474160f, 0.977798061379446360f,
	-0.209549400291664940f,
	0.977757862810002760f, -0.209736886868323290f, 0.977717628291197460f,
	-0.209924365733555880f,
	0.977677357824509930f, -0.210111836880469610f, 0.977637051411420770f,
	-0.210299300302171730f,
	0.977596709053411890f, -0.210486755991769720f, 0.977556330751966460f,
	-0.210674203942371440f,
	0.977515916508569280f, -0.210861644147084860f, 0.977475466324706170f,
	-0.211049076599018390f,
	0.977434980201864260f, -0.211236501291280710f, 0.977394458141532250f,
	-0.211423918216980670f,
	0.977353900145199960f, -0.211611327369227550f, 0.977313306214358750f,
	-0.211798728741130840f,
	0.977272676350500860f, -0.211986122325800330f, 0.977232010555120320f,
	-0.212173508116346080f,
	0.977191308829712280f, -0.212360886105878420f, 0.977150571175773200f,
	-0.212548256287508060f,
	0.977109797594800880f, -0.212735618654345930f, 0.977068988088294450f,
	-0.212922973199503180f,
	0.977028142657754390f, -0.213110319916091360f, 0.976987261304682390f,
	-0.213297658797222320f,
	0.976946344030581670f, -0.213484989836008050f, 0.976905390836956490f,
	-0.213672313025560970f,
	0.976864401725312640f, -0.213859628358993750f, 0.976823376697157240f,
	-0.214046935829419360f,
	0.976782315753998650f, -0.214234235429950990f, 0.976741218897346550f,
	-0.214421527153702160f,
	0.976700086128711840f, -0.214608810993786760f, 0.976658917449606980f,
	-0.214796086943318860f,
	0.976617712861545640f, -0.214983354995412820f, 0.976576472366042610f,
	-0.215170615143183390f,
	0.976535195964614470f, -0.215357867379745550f, 0.976493883658778650f,
	-0.215545111698214500f,
	0.976452535450054060f, -0.215732348091705880f, 0.976411151339961040f,
	-0.215919576553335490f,
	0.976369731330021140f, -0.216106797076219520f, 0.976328275421757260f,
	-0.216294009653474340f,
	0.976286783616693630f, -0.216481214278216730f, 0.976245255916355800f,
	-0.216668410943563730f,
	0.976203692322270560f, -0.216855599642632620f, 0.976162092835966110f,
	-0.217042780368540990f,
	0.976120457458971910f, -0.217229953114406790f, 0.976078786192818850f,
	-0.217417117873348190f,
	0.976037079039039020f, -0.217604274638483640f, 0.975995335999165990f,
	-0.217791423402931950f,
	0.975953557074734300f, -0.217978564159812200f, 0.975911742267280170f,
	-0.218165696902243800f,
	0.975869891578341030f, -0.218352821623346320f, 0.975828005009455660f,
	-0.218539938316239770f,
	0.975786082562163930f, -0.218727046974044440f, 0.975744124238007270f,
	-0.218914147589880840f,
	0.975702130038528570f, -0.219101240156869800f, 0.975660099965271590f,
	-0.219288324668132470f,
	0.975618034019781750f, -0.219475401116790310f, 0.975575932203605720f,
	-0.219662469495965050f,
	0.975533794518291360f, -0.219849529798778700f, 0.975491620965388110f,
	-0.220036582018353580f,
	0.975449411546446380f, -0.220223626147812380f, 0.975407166263018270f,
	-0.220410662180277940f,
	0.975364885116656980f, -0.220597690108873510f, 0.975322568108916930f,
	-0.220784709926722610f,
	0.975280215241354220f, -0.220971721626949110f, 0.975237826515525820f,
	-0.221158725202677010f,
	0.975195401932990370f, -0.221345720647030810f, 0.975152941495307620f,
	-0.221532707953135230f,
	0.975110445204038890f, -0.221719687114115220f, 0.975067913060746470f,
	-0.221906658123096100f,
	0.975025345066994120f, -0.222093620973203510f, 0.974982741224347140f,
	-0.222280575657563370f,
	0.974940101534371830f, -0.222467522169301880f, 0.974897425998635820f,
	-0.222654460501545500f,
	0.974854714618708430f, -0.222841390647421120f, 0.974811967396159830f,
	-0.223028312600055820f,
	0.974769184332561770f, -0.223215226352576980f, 0.974726365429487320f,
	-0.223402131898112370f,
	0.974683510688510670f, -0.223589029229789990f, 0.974640620111207560f,
	-0.223775918340738150f,
	0.974597693699155050f, -0.223962799224085460f, 0.974554731453931230f,
	-0.224149671872960870f,
	0.974511733377115720f, -0.224336536280493600f, 0.974468699470289580f,
	-0.224523392439813170f,
	0.974425629735034990f, -0.224710240344049430f, 0.974382524172935470f,
	-0.224897079986332490f,
	0.974339382785575860f, -0.225083911359792830f, 0.974296205574542440f,
	-0.225270734457561160f,
	0.974252992541422500f, -0.225457549272768540f, 0.974209743687805220f,
	-0.225644355798546330f,
	0.974166459015280320f, -0.225831154028026170f, 0.974123138525439640f,
	-0.226017943954340020f,
	0.974079782219875680f, -0.226204725570620190f, 0.974036390100182610f,
	-0.226391498869999240f,
	0.973992962167955830f, -0.226578263845610000f, 0.973949498424792170f,
	-0.226765020490585690f,
	0.973905998872289570f, -0.226951768798059810f, 0.973862463512047300f,
	-0.227138508761166170f,
	0.973818892345666100f, -0.227325240373038860f, 0.973775285374748110f,
	-0.227511963626812280f,
	0.973731642600896400f, -0.227698678515621170f, 0.973687964025715670f,
	-0.227885385032600530f,
	0.973644249650811980f, -0.228072083170885730f, 0.973600499477792370f,
	-0.228258772923612380f,
	0.973556713508265560f, -0.228445454283916470f, 0.973512891743841370f,
	-0.228632127244934230f,
	0.973469034186131070f, -0.228818791799802220f, 0.973425140836747030f,
	-0.229005447941657340f,
	0.973381211697303290f, -0.229192095663636770f, 0.973337246769414910f,
	-0.229378734958878010f,
	0.973293246054698250f, -0.229565365820518870f, 0.973249209554771230f,
	-0.229751988241697490f,
	0.973205137271252800f, -0.229938602215552210f, 0.973161029205763530f,
	-0.230125207735221850f,
	0.973116885359925130f, -0.230311804793845440f, 0.973072705735360530f,
	-0.230498393384562350f,
	0.973028490333694210f, -0.230684973500512200f, 0.972984239156551740f,
	-0.230871545134835020f,
	0.972939952205560180f, -0.231058108280671110f, 0.972895629482347760f,
	-0.231244662931161050f,
	0.972851270988544180f, -0.231431209079445750f, 0.972806876725780370f,
	-0.231617746718666470f,
	0.972762446695688570f, -0.231804275841964780f, 0.972717980899902250f,
	-0.231990796442482440f,
	0.972673479340056430f, -0.232177308513361710f, 0.972628942017787270f,
	-0.232363812047745030f,
	0.972584368934732210f, -0.232550307038775240f, 0.972539760092530180f,
	-0.232736793479595390f,
	0.972495115492821190f, -0.232923271363348980f, 0.972450435137246830f,
	-0.233109740683179690f,
	0.972405719027449770f, -0.233296201432231590f, 0.972360967165074140f,
	-0.233482653603649090f,
	0.972316179551765300f, -0.233669097190576820f, 0.972271356189170040f,
	-0.233855532186159840f,
	0.972226497078936270f, -0.234041958583543430f, 0.972181602222713440f,
	-0.234228376375873210f,
	0.972136671622152230f, -0.234414785556295160f, 0.972091705278904430f,
	-0.234601186117955550f,
	0.972046703194623500f, -0.234787578054000970f, 0.972001665370963890f,
	-0.234973961357578250f,
	0.971956591809581720f, -0.235160336021834730f, 0.971911482512134000f,
	-0.235346702039917840f,
	0.971866337480279400f, -0.235533059404975490f, 0.971821156715677700f,
	-0.235719408110155820f,
	0.971775940219990140f, -0.235905748148607370f, 0.971730687994879160f,
	-0.236092079513478910f,
	0.971685400042008540f, -0.236278402197919570f, 0.971640076363043390f,
	-0.236464716195078780f,
	0.971594716959650160f, -0.236651021498106380f, 0.971549321833496630f,
	-0.236837318100152380f,
	0.971503890986251780f, -0.237023605994367200f, 0.971458424419585960f,
	-0.237209885173901600f,
	0.971412922135170940f, -0.237396155631906610f, 0.971367384134679490f,
	-0.237582417361533570f,
	0.971321810419786160f, -0.237768670355934190f, 0.971276200992166490f,
	-0.237954914608260540f,
	0.971230555853497380f, -0.238141150111664840f, 0.971184875005457030f,
	-0.238327376859299810f,
	0.971139158449725090f, -0.238513594844318420f, 0.971093406187982460f,
	-0.238699804059873980f,
	0.971047618221911100f, -0.238886004499120040f, 0.971001794553194690f,
	-0.239072196155210610f,
	0.970955935183517970f, -0.239258379021299980f, 0.970910040114567050f,
	-0.239444553090542630f,
	0.970864109348029470f, -0.239630718356093560f, 0.970818142885593870f,
	-0.239816874811108000f,
	0.970772140728950350f, -0.240003022448741500f, 0.970726102879790110f,
	-0.240189161262149900f,
	0.970680029339806130f, -0.240375291244489450f, 0.970633920110692160f,
	-0.240561412388916650f,
	0.970587775194143630f, -0.240747524688588430f, 0.970541594591857070f,
	-0.240933628136661910f,
	0.970495378305530560f, -0.241119722726294590f, 0.970449126336863090f,
	-0.241305808450644370f,
	0.970402838687555500f, -0.241491885302869330f, 0.970356515359309450f,
	-0.241677953276128010f,
	0.970310156353828110f, -0.241864012363579180f, 0.970263761672816140f,
	-0.242050062558382070f,
	0.970217331317979160f, -0.242236103853696010f, 0.970170865291024480f,
	-0.242422136242680890f,
	0.970124363593660280f, -0.242608159718496810f, 0.970077826227596420f,
	-0.242794174274304220f,
	0.970031253194543970f, -0.242980179903263870f, 0.969984644496215240f,
	-0.243166176598536900f,
	0.969938000134323960f, -0.243352164353284740f, 0.969891320110585100f,
	-0.243538143160669130f,
	0.969844604426714830f, -0.243724113013852160f, 0.969797853084430890f,
	-0.243910073905996260f,
	0.969751066085452140f, -0.244096025830264210f, 0.969704243431498860f,
	-0.244281968779819030f,
	0.969657385124292450f, -0.244467902747824150f, 0.969610491165555870f,
	-0.244653827727443320f,
	0.969563561557013180f, -0.244839743711840670f, 0.969516596300390000f,
	-0.245025650694180470f,
	0.969469595397413060f, -0.245211548667627540f, 0.969422558849810320f,
	-0.245397437625346960f,
	0.969375486659311280f, -0.245583317560504060f, 0.969328378827646660f,
	-0.245769188466264580f,
	0.969281235356548530f, -0.245955050335794590f, 0.969234056247750050f,
	-0.246140903162260530f,
	0.969186841502985950f, -0.246326746938829030f, 0.969139591123992280f,
	-0.246512581658667210f,
	0.969092305112506210f, -0.246698407314942410f, 0.969044983470266240f,
	-0.246884223900822430f,
	0.968997626199012420f, -0.247070031409475250f, 0.968950233300485800f,
	-0.247255829834069300f,
	0.968902804776428870f, -0.247441619167773270f, 0.968855340628585580f,
	-0.247627399403756280f,
	0.968807840858700970f, -0.247813170535187670f, 0.968760305468521430f,
	-0.247998932555237110f,
	0.968712734459794780f, -0.248184685457074780f, 0.968665127834270060f,
	-0.248370429233870980f,
	0.968617485593697540f, -0.248556163878796560f, 0.968569807739828930f,
	-0.248741889385022480f,
	0.968522094274417380f, -0.248927605745720150f, 0.968474345199216820f,
	-0.249113312954061360f,
	0.968426560515983190f, -0.249299011003218190f, 0.968378740226473300f,
	-0.249484699886362960f,
	0.968330884332445190f, -0.249670379596668550f, 0.968282992835658660f,
	-0.249856050127307990f,
	0.968235065737874320f, -0.250041711471454650f, 0.968187103040854420f,
	-0.250227363622282370f,
	0.968139104746362440f, -0.250413006572965220f, 0.968091070856162970f,
	-0.250598640316677670f,
	0.968043001372022260f, -0.250784264846594500f, 0.967994896295707670f,
	-0.250969880155890720f,
	0.967946755628987800f, -0.251155486237741920f, 0.967898579373632660f,
	-0.251341083085323880f,
	0.967850367531413620f, -0.251526670691812610f, 0.967802120104103270f,
	-0.251712249050384700f,
	0.967753837093475510f, -0.251897818154216970f, 0.967705518501305480f,
	-0.252083377996486450f,
	0.967657164329369880f, -0.252268928570370810f, 0.967608774579446500f,
	-0.252454469869047740f,
	0.967560349253314360f, -0.252640001885695520f, 0.967511888352754150f,
	-0.252825524613492610f,
	0.967463391879547550f, -0.253011038045617860f, 0.967414859835477480f,
	-0.253196542175250560f,
	0.967366292222328510f, -0.253382036995570160f, 0.967317689041886310f,
	-0.253567522499756560f,
	0.967269050295937790f, -0.253752998680989990f, 0.967220375986271420f,
	-0.253938465532451090f,
	0.967171666114676640f, -0.254123923047320620f, 0.967122920682944360f,
	-0.254309371218780000f,
	0.967074139692867040f, -0.254494810040010730f, 0.967025323146238010f,
	-0.254680239504194830f,
	0.966976471044852070f, -0.254865659604514570f, 0.966927583390505660f,
	-0.255051070334152470f,
	0.966878660184995910f, -0.255236471686291710f, 0.966829701430121810f,
	-0.255421863654115460f,
	0.966780707127683270f, -0.255607246230807380f, 0.966731677279481840f,
	-0.255792619409551610f,
	0.966682611887320080f, -0.255977983183532430f, 0.966633510953002100f,
	-0.256163337545934460f,
	0.966584374478333120f, -0.256348682489942910f, 0.966535202465119700f,
	-0.256534018008743040f,
	0.966485994915169840f, -0.256719344095520660f, 0.966436751830292650f,
	-0.256904660743461910f,
	0.966387473212298900f, -0.257089967945753120f, 0.966338159063000130f,
	-0.257275265695581120f,
	0.966288809384209690f, -0.257460553986133100f, 0.966239424177741890f,
	-0.257645832810596390f,
	0.966190003445412500f, -0.257831102162158990f, 0.966140547189038750f,
	-0.258016362034009020f,
	0.966091055410438830f, -0.258201612419334870f, 0.966041528111432400f,
	-0.258386853311325600f,
	0.965991965293840570f, -0.258572084703170340f, 0.965942366959485540f,
	-0.258757306588058680f,
	0.965892733110190860f, -0.258942518959180520f, 0.965843063747781510f,
	-0.259127721809726150f,
	0.965793358874083680f, -0.259312915132886230f, 0.965743618490924830f,
	-0.259498098921851660f,
	0.965693842600133690f, -0.259683273169813770f, 0.965644031203540590f,
	-0.259868437869964270f,
	0.965594184302976830f, -0.260053593015495190f, 0.965544301900275180f,
	-0.260238738599598840f,
	0.965494383997269500f, -0.260423874615468010f, 0.965444430595795430f,
	-0.260609001056295750f,
	0.965394441697689400f, -0.260794117915275510f, 0.965344417304789370f,
	-0.260979225185601070f,
	0.965294357418934660f, -0.261164322860466480f, 0.965244262041965780f,
	-0.261349410933066350f,
	0.965194131175724720f, -0.261534489396595520f, 0.965143964822054450f,
	-0.261719558244249030f,
	0.965093762982799590f, -0.261904617469222610f, 0.965043525659805890f,
	-0.262089667064712040f,
	0.964993252854920320f, -0.262274707023913590f, 0.964942944569991410f,
	-0.262459737340023980f,
	0.964892600806868890f, -0.262644758006240040f, 0.964842221567403620f,
	-0.262829769015759160f,
	0.964791806853447900f, -0.263014770361779000f, 0.964741356666855340f,
	-0.263199762037497560f,
	0.964690871009481030f, -0.263384744036113280f, 0.964640349883180930f,
	-0.263569716350824880f,
	0.964589793289812760f, -0.263754678974831350f, 0.964539201231235150f,
	-0.263939631901332350f,
	0.964488573709308410f, -0.264124575123527550f, 0.964437910725893910f,
	-0.264309508634617110f,
	0.964387212282854290f, -0.264494432427801630f, 0.964336478382053720f,
	-0.264679346496281890f,
	0.964285709025357480f, -0.264864250833259260f, 0.964234904214632200f,
	-0.265049145431935250f,
	0.964184063951745830f, -0.265234030285511790f, 0.964133188238567640f,
	-0.265418905387191260f,
	0.964082277076968140f, -0.265603770730176330f, 0.964031330468819280f,
	-0.265788626307669920f,
	0.963980348415994110f, -0.265973472112875590f, 0.963929330920367140f,
	-0.266158308138996990f,
	0.963878277983814200f, -0.266343134379238180f, 0.963827189608212340f,
	-0.266527950826803690f,
	0.963776065795439840f, -0.266712757474898370f, 0.963724906547376530f,
	-0.266897554316727350f,
	0.963673711865903230f, -0.267082341345496300f, 0.963622481752902220f,
	-0.267267118554410930f,
	0.963571216210257320f, -0.267451885936677620f, 0.963519915239853140f,
	-0.267636643485503090f,
	0.963468578843575950f, -0.267821391194094150f, 0.963417207023313350f,
	-0.268006129055658290f,
	0.963365799780954050f, -0.268190857063403180f, 0.963314357118388200f,
	-0.268375575210536900f,
	0.963262879037507070f, -0.268560283490267890f, 0.963211365540203480f,
	-0.268744981895804980f,
	0.963159816628371360f, -0.268929670420357260f, 0.963108232303906190f,
	-0.269114349057134380f,
	0.963056612568704340f, -0.269299017799346120f, 0.963004957424663850f,
	-0.269483676640202840f,
	0.962953266873683880f, -0.269668325572915090f, 0.962901540917665000f,
	-0.269852964590693860f,
	0.962849779558509030f, -0.270037593686750570f, 0.962797982798119010f,
	-0.270222212854296870f,
	0.962746150638399410f, -0.270406822086544820f, 0.962694283081255930f,
	-0.270591421376706940f,
	0.962642380128595710f, -0.270776010717996010f, 0.962590441782326890f,
	-0.270960590103625170f,
	0.962538468044359160f, -0.271145159526808010f, 0.962486458916603450f,
	-0.271329718980758420f,
	0.962434414400972100f, -0.271514268458690700f, 0.962382334499378380f,
	-0.271698807953819510f,
	0.962330219213737400f, -0.271883337459359720f, 0.962278068545965090f,
	-0.272067856968526920f,
	0.962225882497979020f, -0.272252366474536710f, 0.962173661071697880f,
	-0.272436865970605240f,
	0.962121404269041580f, -0.272621355449948980f, 0.962069112091931580f,
	-0.272805834905784810f,
	0.962016784542290560f, -0.272990304331329920f, 0.961964421622042320f,
	-0.273174763719801930f,
	0.961912023333112210f, -0.273359213064418680f, 0.961859589677426570f,
	-0.273543652358398730f,
	0.961807120656913540f, -0.273728081594960540f, 0.961754616273502010f,
	-0.273912500767323260f,
	0.961702076529122540f, -0.274096909868706380f, 0.961649501425706820f,
	-0.274281308892329660f,
	0.961596890965187860f, -0.274465697831413220f, 0.961544245149499990f,
	-0.274650076679177680f,
	0.961491563980579000f, -0.274834445428843940f, 0.961438847460361680f,
	-0.275018804073633220f,
	0.961386095590786250f, -0.275203152606767310f, 0.961333308373792270f,
	-0.275387491021468140f,
	0.961280485811320640f, -0.275571819310958140f, 0.961227627905313460f,
	-0.275756137468460120f,
	0.961174734657714080f, -0.275940445487197150f, 0.961121806070467380f,
	-0.276124743360392830f,
	0.961068842145519350f, -0.276309031081271080f, 0.961015842884817230f,
	-0.276493308643055990f,
	0.960962808290309780f, -0.276677576038972420f, 0.960909738363946770f,
	-0.276861833262245280f,
	0.960856633107679660f, -0.277046080306099900f, 0.960803492523460760f,
	-0.277230317163762170f,
	0.960750316613243950f, -0.277414543828458090f, 0.960697105378984450f,
	-0.277598760293414290f,
	0.960643858822638590f, -0.277782966551857690f, 0.960590576946164120f,
	-0.277967162597015370f,
	0.960537259751520050f, -0.278151348422115090f, 0.960483907240666790f,
	-0.278335524020384920f,
	0.960430519415565790f, -0.278519689385053060f, 0.960377096278180130f,
	-0.278703844509348490f,
	0.960323637830473920f, -0.278887989386500280f, 0.960270144074412800f,
	-0.279072124009737800f,
	0.960216615011963430f, -0.279256248372291180f, 0.960163050645094000f,
	-0.279440362467390510f,
	0.960109450975773940f, -0.279624466288266590f, 0.960055816005973890f,
	-0.279808559828150390f,
	0.960002145737665960f, -0.279992643080273220f, 0.959948440172823210f,
	-0.280176716037866980f,
	0.959894699313420530f, -0.280360778694163810f, 0.959840923161433770f,
	-0.280544831042396250f,
	0.959787111718839900f, -0.280728873075797190f, 0.959733264987617680f,
	-0.280912904787600000f,
	0.959679382969746750f, -0.281096926171038260f, 0.959625465667208190f,
	-0.281280937219346110f,
	0.959571513081984520f, -0.281464937925757940f, 0.959517525216059260f,
	-0.281648928283508630f,
	0.959463502071417510f, -0.281832908285833350f, 0.959409443650045550f,
	-0.282016877925967640f,
	0.959355349953930790f, -0.282200837197147560f, 0.959301220985062210f,
	-0.282384786092609360f,
	0.959247056745430090f, -0.282568724605589740f, 0.959192857237025740f,
	-0.282752652729325930f,
	0.959138622461841890f, -0.282936570457055390f, 0.959084352421872730f,
	-0.283120477782015820f,
	0.959030047119113660f, -0.283304374697445740f, 0.958975706555561080f,
	-0.283488261196583550f,
	0.958921330733213170f, -0.283672137272668430f, 0.958866919654069010f,
	-0.283856002918939750f,
	0.958812473320129310f, -0.284039858128637190f, 0.958757991733395710f,
	-0.284223702895001040f,
	0.958703474895871600f, -0.284407537211271880f, 0.958648922809561150f,
	-0.284591361070690440f,
	0.958594335476470220f, -0.284775174466498300f, 0.958539712898605730f,
	-0.284958977391937040f,
	0.958485055077976100f, -0.285142769840248670f, 0.958430362016590930f,
	-0.285326551804675870f,
	0.958375633716461170f, -0.285510323278461260f, 0.958320870179598880f,
	-0.285694084254848320f,
	0.958266071408017670f, -0.285877834727080620f, 0.958211237403732260f,
	-0.286061574688402040f,
	0.958156368168758820f, -0.286245304132057120f, 0.958101463705114730f,
	-0.286429023051290700f,
	0.958046524014818600f, -0.286612731439347790f, 0.957991549099890370f,
	-0.286796429289474080f,
	0.957936538962351420f, -0.286980116594915570f, 0.957881493604224370f,
	-0.287163793348918390f,
	0.957826413027532910f, -0.287347459544729510f, 0.957771297234302320f,
	-0.287531115175595930f,
	0.957716146226558870f, -0.287714760234765170f, 0.957660960006330610f,
	-0.287898394715485170f,
	0.957605738575646350f, -0.288082018611004130f, 0.957550481936536470f,
	-0.288265631914570770f,
	0.957495190091032570f, -0.288449234619434220f, 0.957439863041167680f,
	-0.288632826718843830f,
	0.957384500788975860f, -0.288816408206049480f, 0.957329103336492790f,
	-0.288999979074301420f,
	0.957273670685755200f, -0.289183539316850200f, 0.957218202838801210f,
	-0.289367088926947010f,
	0.957162699797670210f, -0.289550627897843030f, 0.957107161564402790f,
	-0.289734156222790250f,
	0.957051588141040970f, -0.289917673895040750f, 0.956995979529628230f,
	-0.290101180907847090f,
	0.956940335732208820f, -0.290284677254462330f, 0.956884656750828900f,
	-0.290468162928139820f,
	0.956828942587535370f, -0.290651637922133220f, 0.956773193244376930f,
	-0.290835102229696830f,
	0.956717408723403050f, -0.291018555844085090f, 0.956661589026665090f,
	-0.291201998758552900f,
	0.956605734156215080f, -0.291385430966355660f, 0.956549844114106820f,
	-0.291568852460749040f,
	0.956493918902395100f, -0.291752263234989260f, 0.956437958523136180f,
	-0.291935663282332780f,
	0.956381962978387730f, -0.292119052596036380f, 0.956325932270208230f,
	-0.292302431169357560f,
	0.956269866400658030f, -0.292485798995553880f, 0.956213765371798470f,
	-0.292669156067883460f,
	0.956157629185692140f, -0.292852502379604810f, 0.956101457844403040f,
	-0.293035837923976810f,
	0.956045251349996410f, -0.293219162694258630f, 0.955989009704538930f,
	-0.293402476683710110f,
	0.955932732910098280f, -0.293585779885591200f, 0.955876420968743590f,
	-0.293769072293162400f,
	0.955820073882545420f, -0.293952353899684660f, 0.955763691653575440f,
	-0.294135624698419030f,
	0.955707274283906560f, -0.294318884682627400f, 0.955650821775613330f,
	-0.294502133845571670f,
	0.955594334130771110f, -0.294685372180514330f, 0.955537811351456880f,
	-0.294868599680718270f,
	0.955481253439748770f, -0.295051816339446720f, 0.955424660397726330f,
	-0.295235022149963220f,
	0.955368032227470350f, -0.295418217105532010f, 0.955311368931062720f,
	-0.295601401199417360f,
	0.955254670510586990f, -0.295784574424884260f, 0.955197936968127710f,
	-0.295967736775197890f,
	0.955141168305770780f, -0.296150888243623790f, 0.955084364525603410f,
	-0.296334028823428190f,
	0.955027525629714160f, -0.296517158507877470f, 0.954970651620192790f,
	-0.296700277290238350f,
	0.954913742499130520f, -0.296883385163778270f, 0.954856798268619580f,
	-0.297066482121764730f,
	0.954799818930753720f, -0.297249568157465840f, 0.954742804487627940f,
	-0.297432643264150030f,
	0.954685754941338340f, -0.297615707435086200f, 0.954628670293982680f,
	-0.297798760663543550f,
	0.954571550547659630f, -0.297981802942791810f, 0.954514395704469500f,
	-0.298164834266100850f,
	0.954457205766513490f, -0.298347854626741400f, 0.954399980735894490f,
	-0.298530864017984120f,
	0.954342720614716480f, -0.298713862433100330f, 0.954285425405084650f,
	-0.298896849865361800f,
	0.954228095109105670f, -0.299079826308040480f, 0.954170729728887280f,
	-0.299262791754408840f,
	0.954113329266538800f, -0.299445746197739890f, 0.954055893724170660f,
	-0.299628689631306790f,
	0.953998423103894490f, -0.299811622048383350f, 0.953940917407823500f,
	-0.299994543442243580f,
	0.953883376638071770f, -0.300177453806161950f, 0.953825800796755050f,
	-0.300360353133413530f,
	0.953768189885990330f, -0.300543241417273450f, 0.953710543907895670f,
	-0.300726118651017500f,
	0.953652862864590500f, -0.300908984827921890f, 0.953595146758195680f,
	-0.301091839941263100f,
	0.953537395590833280f, -0.301274683984317950f, 0.953479609364626610f,
	-0.301457516950363940f,
	0.953421788081700310f, -0.301640338832678770f, 0.953363931744180330f,
	-0.301823149624540650f,
	0.953306040354193860f, -0.302005949319228080f, 0.953248113913869320f,
	-0.302188737910019990f,
	0.953190152425336670f, -0.302371515390195970f, 0.953132155890726750f,
	-0.302554281753035610f,
	0.953074124312172200f, -0.302737036991819140f, 0.953016057691806530f,
	-0.302919781099827310f,
	0.952957956031764700f, -0.303102514070341060f, 0.952899819334182880f,
	-0.303285235896641750f,
	0.952841647601198720f, -0.303467946572011320f, 0.952783440834950920f,
	-0.303650646089731910f,
	0.952725199037579570f, -0.303833334443086360f, 0.952666922211226170f,
	-0.304016011625357570f,
	0.952608610358033350f, -0.304198677629829110f, 0.952550263480144930f,
	-0.304381332449784880f,
	0.952491881579706320f, -0.304563976078509100f, 0.952433464658864030f,
	-0.304746608509286530f,
	0.952375012719765880f, -0.304929229735402370f, 0.952316525764560940f,
	-0.305111839750142110f,
	0.952258003795399600f, -0.305294438546791670f, 0.952199446814433580f,
	-0.305477026118637420f,
	0.952140854823815830f, -0.305659602458966120f, 0.952082227825700620f,
	-0.305842167561065080f,
	0.952023565822243570f, -0.306024721418221790f, 0.951964868815601380f,
	-0.306207264023724220f,
	0.951906136807932350f, -0.306389795370860920f, 0.951847369801395620f,
	-0.306572315452920740f,
	0.951788567798152130f, -0.306754824263192780f, 0.951729730800363830f,
	-0.306937321794966910f,
	0.951670858810193860f, -0.307119808041533100f, 0.951611951829806850f,
	-0.307302282996181790f,
	0.951553009861368590f, -0.307484746652204100f, 0.951494032907046370f,
	-0.307667199002891190f,
	0.951435020969008340f, -0.307849640041534870f, 0.951375974049424420f,
	-0.308032069761427330f,
	0.951316892150465550f, -0.308214488155861050f, 0.951257775274304000f,
	-0.308396895218129190f,
	0.951198623423113230f, -0.308579290941525090f, 0.951139436599068190f,
	-0.308761675319342450f,
	0.951080214804345010f, -0.308944048344875710f, 0.951020958041121080f,
	-0.309126410011419440f,
	0.950961666311575080f, -0.309308760312268730f, 0.950902339617887060f,
	-0.309491099240719100f,
	0.950842977962238160f, -0.309673426790066380f, 0.950783581346811070f,
	-0.309855742953607070f,
	0.950724149773789610f, -0.310038047724637890f, 0.950664683245358910f,
	-0.310220341096455850f,
	0.950605181763705340f, -0.310402623062358720f, 0.950545645331016600f,
	-0.310584893615644450f,
	0.950486073949481700f, -0.310767152749611470f, 0.950426467621290900f,
	-0.310949400457558640f,
	0.950366826348635780f, -0.311131636732785270f, 0.950307150133709260f,
	-0.311313861568590920f,
	0.950247438978705230f, -0.311496074958275910f, 0.950187692885819280f,
	-0.311678276895140550f,
	0.950127911857248100f, -0.311860467372486020f, 0.950068095895189590f,
	-0.312042646383613510f,
	0.950008245001843000f, -0.312224813921824880f, 0.949948359179409010f,
	-0.312406969980422440f,
	0.949888438430089300f, -0.312589114552708710f, 0.949828482756087110f,
	-0.312771247631986770f,
	0.949768492159606680f, -0.312953369211560200f, 0.949708466642853800f,
	-0.313135479284732840f,
	0.949648406208035480f, -0.313317577844809010f, 0.949588310857359950f,
	-0.313499664885093510f,
	0.949528180593036670f, -0.313681740398891520f, 0.949468015417276550f,
	-0.313863804379508500f,
	0.949407815332291570f, -0.314045856820250710f, 0.949347580340295210f,
	-0.314227897714424440f,
	0.949287310443502120f, -0.314409927055336660f, 0.949227005644128210f,
	-0.314591944836294660f,
	0.949166665944390700f, -0.314773951050606070f, 0.949106291346508260f,
	-0.314955945691579140f,
	0.949045881852700560f, -0.315137928752522440f, 0.948985437465188710f,
	-0.315319900226744890f,
	0.948924958186195160f, -0.315501860107555990f, 0.948864444017943340f,
	-0.315683808388265650f,
	0.948803894962658490f, -0.315865745062183960f, 0.948743311022566480f,
	-0.316047670122621860f,
	0.948682692199895090f, -0.316229583562890330f, 0.948622038496872990f,
	-0.316411485376300980f,
	0.948561349915730270f, -0.316593375556165850f, 0.948500626458698260f,
	-0.316775254095797270f,
	0.948439868128009620f, -0.316957120988508150f, 0.948379074925898120f,
	-0.317138976227611780f,
	0.948318246854599090f, -0.317320819806421740f, 0.948257383916349060f,
	-0.317502651718252260f,
	0.948196486113385580f, -0.317684471956417970f, 0.948135553447947980f,
	-0.317866280514233660f,
	0.948074585922276230f, -0.318048077385014950f, 0.948013583538612200f,
	-0.318229862562077530f,
	0.947952546299198670f, -0.318411636038737790f, 0.947891474206279840f,
	-0.318593397808312420f,
	0.947830367262101010f, -0.318775147864118480f, 0.947769225468909180f,
	-0.318956886199473650f,
	0.947708048828952100f, -0.319138612807695900f, 0.947646837344479300f,
	-0.319320327682103610f,
	0.947585591017741090f, -0.319502030816015690f, 0.947524309850989570f,
	-0.319683722202751430f,
	0.947462993846477700f, -0.319865401835630500f, 0.947401643006459900f,
	-0.320047069707973140f,
	0.947340257333192050f, -0.320228725813099860f, 0.947278836828930880f,
	-0.320410370144331820f,
	0.947217381495934820f, -0.320592002694990330f, 0.947155891336463270f,
	-0.320773623458397330f,
	0.947094366352777220f, -0.320955232427875210f, 0.947032806547138620f,
	-0.321136829596746660f,
	0.946971211921810880f, -0.321318414958334850f, 0.946909582479058760f,
	-0.321499988505963510f,
	0.946847918221148000f, -0.321681550232956580f, 0.946786219150346000f,
	-0.321863100132638580f,
	0.946724485268921170f, -0.322044638198334510f, 0.946662716579143360f,
	-0.322226164423369600f,
	0.946600913083283530f, -0.322407678801069850f, 0.946539074783614100f,
	-0.322589181324761330f,
	0.946477201682408680f, -0.322770671987770710f, 0.946415293781942110f,
	-0.322952150783425260f,
	0.946353351084490590f, -0.323133617705052330f, 0.946291373592331620f,
	-0.323315072745979980f,
	0.946229361307743820f, -0.323496515899536710f, 0.946167314233007370f,
	-0.323677947159051240f,
	0.946105232370403450f, -0.323859366517852850f, 0.946043115722214560f,
	-0.324040773969271450f,
	0.945980964290724760f, -0.324222169506636960f, 0.945918778078219110f,
	-0.324403553123280230f,
	0.945856557086983910f, -0.324584924812532150f, 0.945794301319306970f,
	-0.324766284567724220f,
	0.945732010777477150f, -0.324947632382188430f, 0.945669685463784710f,
	-0.325128968249257080f,
	0.945607325380521280f, -0.325310292162262930f, 0.945544930529979680f,
	-0.325491604114539310f,
	0.945482500914453740f, -0.325672904099419850f, 0.945420036536239070f,
	-0.325854192110238580f,
	0.945357537397632290f, -0.326035468140330240f, 0.945295003500931210f,
	-0.326216732183029710f,
	0.945232434848435000f, -0.326397984231672490f, 0.945169831442444150f,
	-0.326579224279594400f,
	0.945107193285260610f, -0.326760452320131730f, 0.945044520379187070f,
	-0.326941668346621420f,
	0.944981812726528150f, -0.327122872352400510f, 0.944919070329589220f,
	-0.327304064330806670f,
	0.944856293190677210f, -0.327485244275178000f, 0.944793481312100280f,
	-0.327666412178853120f,
	0.944730634696167800f, -0.327847568035170840f, 0.944667753345190490f,
	-0.328028711837470680f,
	0.944604837261480260f, -0.328209843579092500f, 0.944541886447350490f,
	-0.328390963253376580f,
	0.944478900905115550f, -0.328572070853663740f, 0.944415880637091250f,
	-0.328753166373294990f,
	0.944352825645594750f, -0.328934249805612200f, 0.944289735932944410f,
	-0.329115321143957250f,
	0.944226611501459810f, -0.329296380381672750f, 0.944163452353461770f,
	-0.329477427512101740f,
	0.944100258491272660f, -0.329658462528587490f, 0.944037029917215830f,
	-0.329839485424473940f,
	0.943973766633615980f, -0.330020496193105420f, 0.943910468642799150f,
	-0.330201494827826570f,
	0.943847135947092690f, -0.330382481321982780f, 0.943783768548825060f,
	-0.330563455668919540f,
	0.943720366450326200f, -0.330744417861982890f, 0.943656929653927220f,
	-0.330925367894519540f,
	0.943593458161960390f, -0.331106305759876430f, 0.943529951976759480f,
	-0.331287231451400820f,
	0.943466411100659320f, -0.331468144962440870f, 0.943402835535996240f,
	-0.331649046286344670f,
	0.943339225285107720f, -0.331829935416461110f, 0.943275580350332540f,
	-0.332010812346139380f,
	0.943211900734010620f, -0.332191677068729150f, 0.943148186438483420f,
	-0.332372529577580620f,
	0.943084437466093490f, -0.332553369866044220f, 0.943020653819184650f,
	-0.332734197927471050f,
	0.942956835500102120f, -0.332915013755212650f, 0.942892982511192250f,
	-0.333095817342620780f,
	0.942829094854802710f, -0.333276608683047930f, 0.942765172533282510f,
	-0.333457387769846850f,
	0.942701215548981900f, -0.333638154596370860f, 0.942637223904252530f,
	-0.333818909155973620f,
	0.942573197601446870f, -0.333999651442009380f, 0.942509136642919240f,
	-0.334180381447832690f,
	0.942445041031024890f, -0.334361099166798740f, 0.942380910768120470f,
	-0.334541804592262900f,
	0.942316745856563780f, -0.334722497717581220f, 0.942252546298714020f,
	-0.334903178536110180f,
	0.942188312096931770f, -0.335083847041206580f, 0.942124043253578570f,
	-0.335264503226227810f,
	0.942059739771017310f, -0.335445147084531600f, 0.941995401651612550f,
	-0.335625778609476290f,
	0.941931028897729620f, -0.335806397794420450f, 0.941866621511735280f,
	-0.335987004632723350f,
	0.941802179495997650f, -0.336167599117744520f, 0.941737702852886160f,
	-0.336348181242844050f,
	0.941673191584771360f, -0.336528751001382410f, 0.941608645694025250f,
	-0.336709308386720580f,
	0.941544065183020810f, -0.336889853392220050f, 0.941479450054132580f,
	-0.337070386011242620f,
	0.941414800309736340f, -0.337250906237150590f, 0.941350115952208970f,
	-0.337431414063306840f,
	0.941285396983928660f, -0.337611909483074620f, 0.941220643407275180f,
	-0.337792392489817460f,
	0.941155855224629190f, -0.337972863076899720f, 0.941091032438372780f,
	-0.338153321237685930f,
	0.941026175050889260f, -0.338333766965541130f, 0.940961283064563280f,
	-0.338514200253830940f,
	0.940896356481780830f, -0.338694621095921190f, 0.940831395304928870f,
	-0.338875029485178450f,
	0.940766399536396070f, -0.339055425414969640f, 0.940701369178571940f,
	-0.339235808878661950f,
	0.940636304233847590f, -0.339416179869623360f, 0.940571204704615190f,
	-0.339596538381222110f,
	0.940506070593268300f, -0.339776884406826850f, 0.940440901902201750f,
	-0.339957217939806880f,
	0.940375698633811540f, -0.340137538973531720f, 0.940310460790495070f,
	-0.340317847501371670f,
	0.940245188374650880f, -0.340498143516697160f, 0.940179881388678920f,
	-0.340678427012879200f,
	0.940114539834980280f, -0.340858697983289440f, 0.940049163715957370f,
	-0.341038956421299720f,
	0.939983753034014050f, -0.341219202320282360f, 0.939918307791555050f,
	-0.341399435673610420f,
	0.939852827990986680f, -0.341579656474657160f, 0.939787313634716570f,
	-0.341759864716796310f,
	0.939721764725153340f, -0.341940060393402190f, 0.939656181264707180f,
	-0.342120243497849530f,
	0.939590563255789270f, -0.342300414023513520f, 0.939524910700812230f,
	-0.342480571963769800f,
	0.939459223602189920f, -0.342660717311994380f, 0.939393501962337510f,
	-0.342840850061563950f,
	0.939327745783671400f, -0.343020970205855540f, 0.939261955068609210f,
	-0.343201077738246540f,
	0.939196129819569900f, -0.343381172652115040f, 0.939130270038973650f,
	-0.343561254940839390f,
	0.939064375729241950f, -0.343741324597798490f, 0.938998446892797540f,
	-0.343921381616371700f,
	0.938932483532064600f, -0.344101425989938810f, 0.938866485649468060f,
	-0.344281457711880180f,
	0.938800453247434770f, -0.344461476775576540f, 0.938734386328392460f,
	-0.344641483174408960f,
	0.938668284894770170f, -0.344821476901759290f, 0.938602148948998400f,
	-0.345001457951009670f,
	0.938535978493508560f, -0.345181426315542550f, 0.938469773530733800f,
	-0.345361381988741220f,
	0.938403534063108060f, -0.345541324963989090f, 0.938337260093066950f,
	-0.345721255234670120f,
	0.938270951623047190f, -0.345901172794168990f, 0.938204608655486490f,
	-0.346081077635870430f,
	0.938138231192824360f, -0.346260969753160010f, 0.938071819237501270f,
	-0.346440849139423520f,
	0.938005372791958840f, -0.346620715788047320f, 0.937938891858640320f,
	-0.346800569692418290f,
	0.937872376439989890f, -0.346980410845923680f, 0.937805826538453120f,
	-0.347160239241951160f,
	0.937739242156476970f, -0.347340054873889140f, 0.937672623296509470f,
	-0.347519857735126110f,
	0.937605969960999990f, -0.347699647819051380f, 0.937539282152399230f,
	-0.347879425119054510f,
	0.937472559873159250f, -0.348059189628525610f, 0.937405803125732960f,
	-0.348238941340855260f,
	0.937339011912574960f, -0.348418680249434560f, 0.937272186236140950f,
	-0.348598406347654930f,
	0.937205326098887960f, -0.348778119628908420f, 0.937138431503274140f,
	-0.348957820086587490f,
	0.937071502451759190f, -0.349137507714084970f, 0.937004538946803690f,
	-0.349317182504794380f,
	0.936937540990869900f, -0.349496844452109550f, 0.936870508586420960f,
	-0.349676493549424760f,
	0.936803441735921560f, -0.349856129790134920f, 0.936736340441837620f,
	-0.350035753167635240f,
	0.936669204706636170f, -0.350215363675321580f, 0.936602034532785570f,
	-0.350394961306590150f,
	0.936534829922755500f, -0.350574546054837510f, 0.936467590879016990f,
	-0.350754117913461060f,
	0.936400317404042060f, -0.350933676875858360f, 0.936333009500304180f,
	-0.351113222935427460f,
	0.936265667170278260f, -0.351292756085567090f, 0.936198290416440090f,
	-0.351472276319676310f,
	0.936130879241267030f, -0.351651783631154570f, 0.936063433647237540f,
	-0.351831278013402030f,
	0.935995953636831410f, -0.352010759459819080f, 0.935928439212529660f,
	-0.352190227963806830f,
	0.935860890376814640f, -0.352369683518766630f, 0.935793307132169900f,
	-0.352549126118100460f,
	0.935725689481080370f, -0.352728555755210730f, 0.935658037426032040f,
	-0.352907972423500250f,
	0.935590350969512370f, -0.353087376116372480f, 0.935522630114009930f,
	-0.353266766827231240f,
	0.935454874862014620f, -0.353446144549480810f, 0.935387085216017770f,
	-0.353625509276525970f,
	0.935319261178511610f, -0.353804861001772050f, 0.935251402751989920f,
	-0.353984199718624770f,
	0.935183509938947610f, -0.354163525420490340f, 0.935115582741880890f,
	-0.354342838100775550f,
	0.935047621163287430f, -0.354522137752887430f, 0.934979625205665800f,
	-0.354701424370233830f,
	0.934911594871516090f, -0.354880697946222790f, 0.934843530163339540f,
	-0.355059958474262860f,
	0.934775431083638700f, -0.355239205947763310f, 0.934707297634917440f,
	-0.355418440360133650f,
	0.934639129819680780f, -0.355597661704783850f, 0.934570927640435030f,
	-0.355776869975124640f,
	0.934502691099687870f, -0.355956065164566850f, 0.934434420199948050f,
	-0.356135247266522130f,
	0.934366114943725790f, -0.356314416274402410f, 0.934297775333532530f,
	-0.356493572181620090f,
	0.934229401371880820f, -0.356672714981588260f, 0.934160993061284530f,
	-0.356851844667720300f,
	0.934092550404258980f, -0.357030961233429980f, 0.934024073403320390f,
	-0.357210064672131960f,
	0.933955562060986730f, -0.357389154977240940f, 0.933887016379776890f,
	-0.357568232142172260f,
	0.933818436362210960f, -0.357747296160341900f, 0.933749822010810580f,
	-0.357926347025166010f,
	0.933681173328098410f, -0.358105384730061590f, 0.933612490316598540f,
	-0.358284409268445850f,
	0.933543772978836170f, -0.358463420633736540f, 0.933475021317337950f,
	-0.358642418819351990f,
	0.933406235334631520f, -0.358821403818710860f, 0.933337415033246190f,
	-0.359000375625232460f,
	0.933268560415712050f, -0.359179334232336500f, 0.933199671484560730f,
	-0.359358279633443130f,
	0.933130748242325230f, -0.359537211821973070f, 0.933061790691539380f,
	-0.359716130791347570f,
	0.932992798834738960f, -0.359895036534988110f, 0.932923772674460140f,
	-0.360073929046317020f,
	0.932854712213241120f, -0.360252808318756890f, 0.932785617453621100f,
	-0.360431674345730700f,
	0.932716488398140250f, -0.360610527120662270f, 0.932647325049340450f,
	-0.360789366636975580f,
	0.932578127409764420f, -0.360968192888095230f, 0.932508895481956590f,
	-0.361147005867446250f,
	0.932439629268462360f, -0.361325805568454280f, 0.932370328771828460f,
	-0.361504591984545260f,
	0.932300993994602760f, -0.361683365109145840f, 0.932231624939334540f,
	-0.361862124935682980f,
	0.932162221608574430f, -0.362040871457584180f, 0.932092784004874050f,
	-0.362219604668277460f,
	0.932023312130786490f, -0.362398324561191310f, 0.931953805988866010f,
	-0.362577031129754760f,
	0.931884265581668150f, -0.362755724367397230f, 0.931814690911749730f,
	-0.362934404267548640f,
	0.931745081981668720f, -0.363113070823639470f, 0.931675438793984620f,
	-0.363291724029100760f,
	0.931605761351257830f, -0.363470363877363760f, 0.931536049656050300f,
	-0.363648990361860550f,
	0.931466303710925090f, -0.363827603476023500f, 0.931396523518446600f,
	-0.364006203213285470f,
	0.931326709081180430f, -0.364184789567079890f, 0.931256860401693420f,
	-0.364363362530840620f,
	0.931186977482553750f, -0.364541922098002120f, 0.931117060326330790f,
	-0.364720468261999280f,
	0.931047108935595280f, -0.364899001016267320f, 0.930977123312918930f,
	-0.365077520354242180f,
	0.930907103460875130f, -0.365256026269360320f, 0.930837049382038150f,
	-0.365434518755058390f,
	0.930766961078983710f, -0.365612997804773850f, 0.930696838554288860f,
	-0.365791463411944570f,
	0.930626681810531760f, -0.365969915570008740f, 0.930556490850291800f,
	-0.366148354272405330f,
	0.930486265676149780f, -0.366326779512573590f, 0.930416006290687550f,
	-0.366505191283953370f,
	0.930345712696488470f, -0.366683589579984930f, 0.930275384896137150f,
	-0.366861974394109060f,
	0.930205022892219070f, -0.367040345719767180f, 0.930134626687321390f,
	-0.367218703550400980f,
	0.930064196284032360f, -0.367397047879452710f, 0.929993731684941480f,
	-0.367575378700365330f,
	0.929923232892639670f, -0.367753696006581980f, 0.929852699909718750f,
	-0.367931999791546450f,
	0.929782132738772190f, -0.368110290048703050f, 0.929711531382394370f,
	-0.368288566771496570f,
	0.929640895843181330f, -0.368466829953372320f, 0.929570226123729860f,
	-0.368645079587776040f,
	0.929499522226638560f, -0.368823315668153910f, 0.929428784154506800f,
	-0.369001538187952780f,
	0.929358011909935500f, -0.369179747140620020f, 0.929287205495526790f,
	-0.369357942519603130f,
	0.929216364913884040f, -0.369536124318350650f, 0.929145490167611720f,
	-0.369714292530311240f,
	0.929074581259315860f, -0.369892447148934100f, 0.929003638191603360f,
	-0.370070588167669080f,
	0.928932660967082820f, -0.370248715579966360f, 0.928861649588363700f,
	-0.370426829379276790f,
	0.928790604058057020f, -0.370604929559051670f, 0.928719524378774810f,
	-0.370783016112742560f,
	0.928648410553130520f, -0.370961089033801980f, 0.928577262583738850f,
	-0.371139148315682570f,
	0.928506080473215590f, -0.371317193951837540f, 0.928434864224177980f,
	-0.371495225935720760f,
	0.928363613839244370f, -0.371673244260786520f, 0.928292329321034670f,
	-0.371851248920489490f,
	0.928221010672169440f, -0.372029239908285010f, 0.928149657895271150f,
	-0.372207217217628840f,
	0.928078270992963140f, -0.372385180841977360f, 0.928006849967869970f,
	-0.372563130774787250f,
	0.927935394822617890f, -0.372741067009515760f, 0.927863905559833780f,
	-0.372918989539620830f,
	0.927792382182146320f, -0.373096898358560640f, 0.927720824692185200f,
	-0.373274793459793970f,
	0.927649233092581180f, -0.373452674836780300f, 0.927577607385966730f,
	-0.373630542482979280f,
	0.927505947574975180f, -0.373808396391851210f, 0.927434253662241300f,
	-0.373986236556857030f,
	0.927362525650401110f, -0.374164062971457930f, 0.927290763542091720f,
	-0.374341875629115920f,
	0.927218967339951790f, -0.374519674523293210f, 0.927147137046620880f,
	-0.374697459647452600f,
	0.927075272664740100f, -0.374875230995057540f, 0.927003374196951670f,
	-0.375052988559571920f,
	0.926931441645899130f, -0.375230732334459920f, 0.926859475014227160f,
	-0.375408462313186590f,
	0.926787474304581750f, -0.375586178489217220f, 0.926715439519610330f,
	-0.375763880856017700f,
	0.926643370661961230f, -0.375941569407054420f, 0.926571267734284330f,
	-0.376119244135794340f,
	0.926499130739230510f, -0.376296905035704790f, 0.926426959679452210f,
	-0.376474552100253770f,
	0.926354754557602860f, -0.376652185322909560f, 0.926282515376337210f,
	-0.376829804697141280f,
	0.926210242138311380f, -0.377007410216418260f, 0.926137934846182560f,
	-0.377185001874210450f,
	0.926065593502609310f, -0.377362579663988340f, 0.925993218110251480f,
	-0.377540143579222940f,
	0.925920808671770070f, -0.377717693613385640f, 0.925848365189827270f,
	-0.377895229759948490f,
	0.925775887667086740f, -0.378072752012383990f, 0.925703376106213230f,
	-0.378250260364165200f,
	0.925630830509872720f, -0.378427754808765560f, 0.925558250880732740f,
	-0.378605235339659120f,
	0.925485637221461490f, -0.378782701950320540f, 0.925412989534729060f,
	-0.378960154634224720f,
	0.925340307823206310f, -0.379137593384847320f, 0.925267592089565660f,
	-0.379315018195664430f,
	0.925194842336480530f, -0.379492429060152630f, 0.925122058566625880f,
	-0.379669825971788940f,
	0.925049240782677580f, -0.379847208924051160f, 0.924976388987313160f,
	-0.380024577910417270f,
	0.924903503183210910f, -0.380201932924366050f, 0.924830583373050800f,
	-0.380379273959376600f,
	0.924757629559513910f, -0.380556601008928520f, 0.924684641745282420f,
	-0.380733914066502140f,
	0.924611619933039970f, -0.380911213125578070f, 0.924538564125471420f,
	-0.381088498179637520f,
	0.924465474325262600f, -0.381265769222162380f, 0.924392350535101050f,
	-0.381443026246634730f,
	0.924319192757675160f, -0.381620269246537360f, 0.924246000995674890f,
	-0.381797498215353640f,
	0.924172775251791200f, -0.381974713146567220f, 0.924099515528716280f,
	-0.382151914033662610f,
	0.924026221829143850f, -0.382329100870124510f, 0.923952894155768640f,
	-0.382506273649438230f,
	0.923879532511286740f, -0.382683432365089780f, 0.923806136898395410f,
	-0.382860577010565420f,
	0.923732707319793290f, -0.383037707579352020f, 0.923659243778179980f,
	-0.383214824064937180f,
	0.923585746276256670f, -0.383391926460808660f, 0.923512214816725630f,
	-0.383569014760454910f,
	0.923438649402290370f, -0.383746088957365010f, 0.923365050035655720f,
	-0.383923149045028390f,
	0.923291416719527640f, -0.384100195016935040f, 0.923217749456613500f,
	-0.384277226866575510f,
	0.923144048249621930f, -0.384454244587440820f, 0.923070313101262420f,
	-0.384631248173022580f,
	0.922996544014246250f, -0.384808237616812880f, 0.922922740991285680f,
	-0.384985212912304200f,
	0.922848904035094120f, -0.385162174052989860f, 0.922775033148386380f,
	-0.385339121032363340f,
	0.922701128333878630f, -0.385516053843918850f, 0.922627189594287910f,
	-0.385692972481151140f,
	0.922553216932332830f, -0.385869876937555310f, 0.922479210350733210f,
	-0.386046767206627170f,
	0.922405169852209880f, -0.386223643281862980f, 0.922331095439485440f,
	-0.386400505156759440f,
	0.922256987115283030f, -0.386577352824813920f, 0.922182844882327600f,
	-0.386754186279524180f,
	0.922108668743345180f, -0.386931005514388580f, 0.922034458701062820f,
	-0.387107810522905990f,
	0.921960214758209220f, -0.387284601298575840f, 0.921885936917513970f,
	-0.387461377834897870f,
	0.921811625181708120f, -0.387638140125372730f, 0.921737279553523910f,
	-0.387814888163501180f,
	0.921662900035694730f, -0.387991621942784860f, 0.921588486630955490f,
	-0.388168341456725740f,
	0.921514039342042010f, -0.388345046698826250f, 0.921439558171691430f,
	-0.388521737662589570f,
	0.921365043122642340f, -0.388698414341519190f, 0.921290494197634540f,
	-0.388875076729119250f,
	0.921215911399408730f, -0.389051724818894380f, 0.921141294730707270f,
	-0.389228358604349730f,
	0.921066644194273640f, -0.389404978078990940f, 0.920991959792852310f,
	-0.389581583236324300f,
	0.920917241529189520f, -0.389758174069856410f, 0.920842489406032190f,
	-0.389934750573094730f,
	0.920767703426128790f, -0.390111312739546910f, 0.920692883592229120f,
	-0.390287860562721190f,
	0.920618029907083970f, -0.390464394036126590f, 0.920543142373445480f,
	-0.390640913153272430f,
	0.920468220994067110f, -0.390817417907668500f, 0.920393265771703550f,
	-0.390993908292825380f,
	0.920318276709110590f, -0.391170384302253870f, 0.920243253809045370f,
	-0.391346845929465560f,
	0.920168197074266340f, -0.391523293167972410f, 0.920093106507533180f,
	-0.391699726011286940f,
	0.920017982111606570f, -0.391876144452922350f, 0.919942823889248640f,
	-0.392052548486392090f,
	0.919867631843222950f, -0.392228938105210310f, 0.919792405976293860f,
	-0.392405313302891690f,
	0.919717146291227360f, -0.392581674072951470f, 0.919641852790790470f,
	-0.392758020408905280f,
	0.919566525477751530f, -0.392934352304269490f, 0.919491164354880100f,
	-0.393110669752560760f,
	0.919415769424947070f, -0.393286972747296400f, 0.919340340690724340f,
	-0.393463261281994330f,
	0.919264878154985370f, -0.393639535350172880f, 0.919189381820504470f,
	-0.393815794945351020f,
	0.919113851690057770f, -0.393992040061048100f, 0.919038287766422050f,
	-0.394168270690784080f,
	0.918962690052375630f, -0.394344486828079600f, 0.918887058550697970f,
	-0.394520688466455600f,
	0.918811393264170050f, -0.394696875599433560f, 0.918735694195573550f,
	-0.394873048220535760f,
	0.918659961347691900f, -0.395049206323284770f, 0.918584194723309540f,
	-0.395225349901203670f,
	0.918508394325212250f, -0.395401478947816350f, 0.918432560156186910f,
	-0.395577593456646840f,
	0.918356692219021720f, -0.395753693421220080f, 0.918280790516506130f,
	-0.395929778835061250f,
	0.918204855051430900f, -0.396105849691696270f, 0.918128885826588030f,
	-0.396281905984651520f,
	0.918052882844770380f, -0.396457947707453910f, 0.917976846108772730f,
	-0.396633974853630830f,
	0.917900775621390500f, -0.396809987416710310f, 0.917824671385420570f,
	-0.396985985390220900f,
	0.917748533403661250f, -0.397161968767691610f, 0.917672361678911860f,
	-0.397337937542652060f,
	0.917596156213972950f, -0.397513891708632330f, 0.917519917011646260f,
	-0.397689831259163180f,
	0.917443644074735220f, -0.397865756187775750f, 0.917367337406043930f,
	-0.398041666488001770f,
	0.917290997008377910f, -0.398217562153373560f, 0.917214622884544250f,
	-0.398393443177423980f,
	0.917138215037350710f, -0.398569309553686300f, 0.917061773469606820f,
	-0.398745161275694430f,
	0.916985298184123000f, -0.398920998336982910f, 0.916908789183710990f,
	-0.399096820731086540f,
	0.916832246471183890f, -0.399272628451540990f, 0.916755670049355990f,
	-0.399448421491882140f,
	0.916679059921042700f, -0.399624199845646790f, 0.916602416089060790f,
	-0.399799963506371980f,
	0.916525738556228210f, -0.399975712467595330f, 0.916449027325364150f,
	-0.400151446722855130f,
	0.916372282399289140f, -0.400327166265690090f, 0.916295503780824800f,
	-0.400502871089639500f,
	0.916218691472794220f, -0.400678561188243240f, 0.916141845478021350f,
	-0.400854236555041650f,
	0.916064965799331720f, -0.401029897183575620f, 0.915988052439551950f,
	-0.401205543067386710f,
	0.915911105401509880f, -0.401381174200016790f, 0.915834124688034710f,
	-0.401556790575008540f,
	0.915757110301956720f, -0.401732392185905010f, 0.915680062246107650f,
	-0.401907979026249700f,
	0.915602980523320230f, -0.402083551089586990f, 0.915525865136428530f,
	-0.402259108369461490f,
	0.915448716088267830f, -0.402434650859418430f, 0.915371533381674760f,
	-0.402610178553003680f,
	0.915294317019487050f, -0.402785691443763530f, 0.915217067004543860f,
	-0.402961189525244900f,
	0.915139783339685260f, -0.403136672790995300f, 0.915062466027752760f,
	-0.403312141234562550f,
	0.914985115071589310f, -0.403487594849495310f, 0.914907730474038730f,
	-0.403663033629342640f,
	0.914830312237946200f, -0.403838457567654070f, 0.914752860366158220f,
	-0.404013866657979890f,
	0.914675374861522390f, -0.404189260893870690f, 0.914597855726887790f,
	-0.404364640268877810f,
	0.914520302965104450f, -0.404540004776553000f, 0.914442716579023870f,
	-0.404715354410448650f,
	0.914365096571498560f, -0.404890689164117580f, 0.914287442945382440f,
	-0.405066009031113340f,
	0.914209755703530690f, -0.405241314004989860f, 0.914132034848799460f,
	-0.405416604079301630f,
	0.914054280384046570f, -0.405591879247603870f, 0.913976492312130630f,
	-0.405767139503452060f,
	0.913898670635911680f, -0.405942384840402510f, 0.913820815358251100f,
	-0.406117615252011840f,
	0.913742926482011390f, -0.406292830731837360f, 0.913665004010056350f,
	-0.406468031273437000f,
	0.913587047945250810f, -0.406643216870369030f, 0.913509058290461140f,
	-0.406818387516192310f,
	0.913431035048554720f, -0.406993543204466510f, 0.913352978222400250f,
	-0.407168683928751550f,
	0.913274887814867760f, -0.407343809682607970f, 0.913196763828828200f,
	-0.407518920459596920f,
	0.913118606267154240f, -0.407694016253280110f, 0.913040415132719160f,
	-0.407869097057219800f,
	0.912962190428398210f, -0.408044162864978690f, 0.912883932157067200f,
	-0.408219213670120100f,
	0.912805640321603500f, -0.408394249466208000f, 0.912727314924885900f,
	-0.408569270246806780f,
	0.912648955969793900f, -0.408744276005481360f, 0.912570563459208730f,
	-0.408919266735797430f,
	0.912492137396012650f, -0.409094242431320980f, 0.912413677783089020f,
	-0.409269203085618590f,
	0.912335184623322750f, -0.409444148692257590f, 0.912256657919599760f,
	-0.409619079244805670f,
	0.912178097674807180f, -0.409793994736831150f, 0.912099503891833470f,
	-0.409968895161902880f,
	0.912020876573568340f, -0.410143780513590240f, 0.911942215722902570f,
	-0.410318650785463260f,
	0.911863521342728520f, -0.410493505971092410f, 0.911784793435939430f,
	-0.410668346064048730f,
	0.911706032005429880f, -0.410843171057903910f, 0.911627237054095650f,
	-0.411017980946230210f,
	0.911548408584833990f, -0.411192775722600160f, 0.911469546600543020f,
	-0.411367555380587220f,
	0.911390651104122430f, -0.411542319913765220f, 0.911311722098472780f,
	-0.411717069315708560f,
	0.911232759586496190f, -0.411891803579992170f, 0.911153763571095900f,
	-0.412066522700191560f,
	0.911074734055176360f, -0.412241226669882890f, 0.910995671041643140f,
	-0.412415915482642730f,
	0.910916574533403360f, -0.412590589132048210f, 0.910837444533365010f,
	-0.412765247611677270f,
	0.910758281044437570f, -0.412939890915108080f, 0.910679084069531570f,
	-0.413114519035919450f,
	0.910599853611558930f, -0.413289131967690960f, 0.910520589673432750f,
	-0.413463729704002410f,
	0.910441292258067250f, -0.413638312238434500f, 0.910361961368377990f,
	-0.413812879564568300f,
	0.910282597007281760f, -0.413987431675985400f, 0.910203199177696540f,
	-0.414161968566268080f,
	0.910123767882541680f, -0.414336490228999100f, 0.910044303124737500f,
	-0.414510996657761750f,
	0.909964804907205660f, -0.414685487846140010f, 0.909885273232869160f,
	-0.414859963787718330f,
	0.909805708104652220f, -0.415034424476081630f, 0.909726109525480160f,
	-0.415208869904815590f,
	0.909646477498279540f, -0.415383300067506230f, 0.909566812025978330f,
	-0.415557714957740410f,
	0.909487113111505430f, -0.415732114569105360f, 0.909407380757791260f,
	-0.415906498895188770f,
	0.909327614967767260f, -0.416080867929579210f, 0.909247815744366310f,
	-0.416255221665865480f,
	0.909167983090522380f, -0.416429560097637150f, 0.909088117009170580f,
	-0.416603883218484350f,
	0.909008217503247450f, -0.416778191021997650f, 0.908928284575690640f,
	-0.416952483501768170f,
	0.908848318229439120f, -0.417126760651387870f, 0.908768318467432890f,
	-0.417301022464448890f,
	0.908688285292613360f, -0.417475268934544290f, 0.908608218707923190f,
	-0.417649500055267410f,
	0.908528118716306120f, -0.417823715820212270f, 0.908447985320707250f,
	-0.417997916222973550f,
	0.908367818524072890f, -0.418172101257146320f, 0.908287618329350450f,
	-0.418346270916326260f,
	0.908207384739488700f, -0.418520425194109700f, 0.908127117757437600f,
	-0.418694564084093560f,
	0.908046817386148340f, -0.418868687579875050f, 0.907966483628573350f,
	-0.419042795675052370f,
	0.907886116487666260f, -0.419216888363223910f, 0.907805715966381930f,
	-0.419390965637988890f,
	0.907725282067676440f, -0.419565027492946880f, 0.907644814794507200f,
	-0.419739073921698180f,
	0.907564314149832630f, -0.419913104917843620f, 0.907483780136612570f,
	-0.420087120474984530f,
	0.907403212757808110f, -0.420261120586722880f, 0.907322612016381420f,
	-0.420435105246661170f,
	0.907241977915295820f, -0.420609074448402510f, 0.907161310457516250f,
	-0.420783028185550520f,
	0.907080609646008450f, -0.420956966451709440f, 0.906999875483739610f,
	-0.421130889240483970f,
	0.906919107973678140f, -0.421304796545479640f, 0.906838307118793430f,
	-0.421478688360302280f,
	0.906757472922056550f, -0.421652564678558330f, 0.906676605386439460f,
	-0.421826425493854910f,
	0.906595704514915330f, -0.422000270799799680f, 0.906514770310458800f,
	-0.422174100590000770f,
	0.906433802776045460f, -0.422347914858067050f, 0.906352801914652400f,
	-0.422521713597607820f,
	0.906271767729257660f, -0.422695496802232950f, 0.906190700222840650f,
	-0.422869264465553060f,
	0.906109599398381980f, -0.423043016581179040f, 0.906028465258863600f,
	-0.423216753142722610f,
	0.905947297807268460f, -0.423390474143796050f, 0.905866097046580940f,
	-0.423564179578011960f,
	0.905784862979786550f, -0.423737869438983840f, 0.905703595609872010f,
	-0.423911543720325580f,
	0.905622294939825270f, -0.424085202415651560f, 0.905540960972635590f,
	-0.424258845518576950f,
	0.905459593711293250f, -0.424432473022717420f, 0.905378193158790090f,
	-0.424606084921689110f,
	0.905296759318118820f, -0.424779681209108810f, 0.905215292192273590f,
	-0.424953261878593890f,
	0.905133791784249690f, -0.425126826923762360f, 0.905052258097043590f,
	-0.425300376338232640f,
	0.904970691133653250f, -0.425473910115623800f, 0.904889090897077470f,
	-0.425647428249555590f,
	0.904807457390316540f, -0.425820930733648240f, 0.904725790616371930f,
	-0.425994417561522400f,
	0.904644090578246240f, -0.426167888726799620f, 0.904562357278943300f,
	-0.426341344223101830f,
	0.904480590721468250f, -0.426514784044051520f, 0.904398790908827350f,
	-0.426688208183271860f,
	0.904316957844028320f, -0.426861616634386430f, 0.904235091530079750f,
	-0.427035009391019680f,
	0.904153191969991780f, -0.427208386446796320f, 0.904071259166775440f,
	-0.427381747795341770f,
	0.903989293123443340f, -0.427555093430282080f, 0.903907293843009050f,
	-0.427728423345243800f,
	0.903825261328487510f, -0.427901737533854080f, 0.903743195582894620f,
	-0.428075035989740730f,
	0.903661096609247980f, -0.428248318706531960f, 0.903578964410566070f,
	-0.428421585677856650f,
	0.903496798989868450f, -0.428594836897344400f, 0.903414600350176290f,
	-0.428768072358625070f,
	0.903332368494511820f, -0.428941292055329490f, 0.903250103425898400f,
	-0.429114495981088750f,
	0.903167805147360720f, -0.429287684129534610f, 0.903085473661924600f,
	-0.429460856494299490f,
	0.903003108972617150f, -0.429634013069016380f, 0.902920711082466740f,
	-0.429807153847318710f,
	0.902838279994502830f, -0.429980278822840620f, 0.902755815711756120f,
	-0.430153387989216870f,
	0.902673318237258830f, -0.430326481340082610f, 0.902590787574043870f,
	-0.430499558869073820f,
	0.902508223725145940f, -0.430672620569826800f, 0.902425626693600380f,
	-0.430845666435978660f,
	0.902342996482444200f, -0.431018696461167030f, 0.902260333094715540f,
	-0.431191710639029950f,
	0.902177636533453620f, -0.431364708963206330f, 0.902094906801698900f,
	-0.431537691427335500f,
	0.902012143902493180f, -0.431710658025057260f, 0.901929347838879460f,
	-0.431883608750012250f,
	0.901846518613901750f, -0.432056543595841500f, 0.901763656230605730f,
	-0.432229462556186720f,
	0.901680760692037730f, -0.432402365624690140f, 0.901597832001245660f,
	-0.432575252794994650f,
	0.901514870161278740f, -0.432748124060743700f, 0.901431875175186970f,
	-0.432920979415581280f,
	0.901348847046022030f, -0.433093818853151960f, 0.901265785776836580f,
	-0.433266642367100940f,
	0.901182691370684520f, -0.433439449951074090f, 0.901099563830620950f,
	-0.433612241598717580f,
	0.901016403159702330f, -0.433785017303678520f, 0.900933209360986200f,
	-0.433957777059604420f,
	0.900849982437531450f, -0.434130520860143310f, 0.900766722392397860f,
	-0.434303248698943990f,
	0.900683429228646970f, -0.434475960569655650f, 0.900600102949340900f,
	-0.434648656465928320f,
	0.900516743557543520f, -0.434821336381412290f, 0.900433351056319830f,
	-0.434994000309758710f,
	0.900349925448735600f, -0.435166648244619260f, 0.900266466737858480f,
	-0.435339280179646070f,
	0.900182974926756810f, -0.435511896108492000f, 0.900099450018500450f,
	-0.435684496024810460f,
	0.900015892016160280f, -0.435857079922255470f, 0.899932300922808510f,
	-0.436029647794481560f,
	0.899848676741518580f, -0.436202199635143950f, 0.899765019475365140f,
	-0.436374735437898340f,
	0.899681329127423930f, -0.436547255196401200f, 0.899597605700772180f,
	-0.436719758904309360f,
	0.899513849198487980f, -0.436892246555280360f, 0.899430059623650860f,
	-0.437064718142972370f,
	0.899346236979341570f, -0.437237173661044090f, 0.899262381268642000f,
	-0.437409613103154790f,
	0.899178492494635330f, -0.437582036462964400f, 0.899094570660405770f,
	-0.437754443734133410f,
	0.899010615769039070f, -0.437926834910322860f, 0.898926627823621870f,
	-0.438099209985194470f,
	0.898842606827242370f, -0.438271568952410430f, 0.898758552782989440f,
	-0.438443911805633690f,
	0.898674465693953820f, -0.438616238538527660f, 0.898590345563227030f,
	-0.438788549144756290f,
	0.898506192393901950f, -0.438960843617984320f, 0.898422006189072530f,
	-0.439133121951876930f,
	0.898337786951834310f, -0.439305384140099950f, 0.898253534685283570f,
	-0.439477630176319800f,
	0.898169249392518080f, -0.439649860054203480f, 0.898084931076636780f,
	-0.439822073767418500f,
	0.898000579740739880f, -0.439994271309633260f, 0.897916195387928660f,
	-0.440166452674516320f,
	0.897831778021305650f, -0.440338617855737250f, 0.897747327643974690f,
	-0.440510766846965940f,
	0.897662844259040860f, -0.440682899641872900f, 0.897578327869610230f,
	-0.440855016234129430f,
	0.897493778478790310f, -0.441027116617407230f, 0.897409196089689720f,
	-0.441199200785378660f,
	0.897324580705418320f, -0.441371268731716670f, 0.897239932329087160f,
	-0.441543320450094870f,
	0.897155250963808550f, -0.441715355934187310f, 0.897070536612695870f,
	-0.441887375177668850f,
	0.896985789278863970f, -0.442059378174214700f, 0.896901008965428790f,
	-0.442231364917500980f,
	0.896816195675507300f, -0.442403335401204080f, 0.896731349412217880f,
	-0.442575289619001170f,
	0.896646470178680150f, -0.442747227564570020f, 0.896561557978014960f,
	-0.442919149231588980f,
	0.896476612813344120f, -0.443091054613736880f, 0.896391634687790820f,
	-0.443262943704693320f,
	0.896306623604479550f, -0.443434816498138480f, 0.896221579566536030f,
	-0.443606672987752970f,
	0.896136502577086770f, -0.443778513167218220f, 0.896051392639260150f,
	-0.443950337030216140f,
	0.895966249756185220f, -0.444122144570429200f, 0.895881073930992370f,
	-0.444293935781540580f,
	0.895795865166813530f, -0.444465710657234000f, 0.895710623466781320f,
	-0.444637469191193790f,
	0.895625348834030110f, -0.444809211377104880f, 0.895540041271694950f,
	-0.444980937208652730f,
	0.895454700782912450f, -0.445152646679523640f, 0.895369327370820310f,
	-0.445324339783404190f,
	0.895283921038557580f, -0.445496016513981740f, 0.895198481789264200f,
	-0.445667676864944300f,
	0.895113009626081760f, -0.445839320829980290f, 0.895027504552152630f,
	-0.446010948402778940f,
	0.894941966570620750f, -0.446182559577030070f, 0.894856395684631050f,
	-0.446354154346423840f,
	0.894770791897329550f, -0.446525732704651350f, 0.894685155211863980f,
	-0.446697294645404090f,
	0.894599485631382700f, -0.446868840162374160f, 0.894513783159035620f,
	-0.447040369249254440f,
	0.894428047797973800f, -0.447211881899738320f, 0.894342279551349480f,
	-0.447383378107519600f,
	0.894256478422316040f, -0.447554857866293010f, 0.894170644414028270f,
	-0.447726321169753580f,
	0.894084777529641990f, -0.447897768011597310f, 0.893998877772314240f,
	-0.448069198385520400f,
	0.893912945145203250f, -0.448240612285219890f, 0.893826979651468620f,
	-0.448412009704393430f,
	0.893740981294271040f, -0.448583390636739240f, 0.893654950076772540f,
	-0.448754755075955970f,
	0.893568886002135910f, -0.448926103015743260f, 0.893482789073525850f,
	-0.449097434449801050f,
	0.893396659294107720f, -0.449268749371829920f, 0.893310496667048200f,
	-0.449440047775531150f,
	0.893224301195515320f, -0.449611329654606540f, 0.893138072882678320f,
	-0.449782595002758690f,
	0.893051811731707450f, -0.449953843813690520f, 0.892965517745774370f,
	-0.450125076081105690f,
	0.892879190928051680f, -0.450296291798708610f, 0.892792831281713610f,
	-0.450467490960204110f,
	0.892706438809935390f, -0.450638673559297600f, 0.892620013515893150f,
	-0.450809839589695280f,
	0.892533555402764580f, -0.450980989045103860f, 0.892447064473728680f,
	-0.451152121919230600f,
	0.892360540731965360f, -0.451323238205783520f, 0.892273984180655840f,
	-0.451494337898471100f,
	0.892187394822982480f, -0.451665420991002490f, 0.892100772662129060f,
	-0.451836487477087490f,
	0.892014117701280470f, -0.452007537350436420f, 0.891927429943622510f,
	-0.452178570604760350f,
	0.891840709392342720f, -0.452349587233770890f, 0.891753956050629460f,
	-0.452520587231180050f,
	0.891667169921672280f, -0.452691570590700920f, 0.891580351008662290f,
	-0.452862537306046750f,
	0.891493499314791380f, -0.453033487370931580f, 0.891406614843252900f,
	-0.453204420779070190f,
	0.891319697597241390f, -0.453375337524177750f, 0.891232747579952520f,
	-0.453546237599970090f,
	0.891145764794583180f, -0.453717121000163870f, 0.891058749244331590f,
	-0.453887987718476050f,
	0.890971700932396860f, -0.454058837748624430f, 0.890884619861979530f,
	-0.454229671084327320f,
	0.890797506036281490f, -0.454400487719303580f, 0.890710359458505630f,
	-0.454571287647272950f,
	0.890623180131855930f, -0.454742070861955450f, 0.890535968059537830f,
	-0.454912837357071940f,
	0.890448723244757880f, -0.455083587126343840f, 0.890361445690723840f,
	-0.455254320163493100f,
	0.890274135400644600f, -0.455425036462242360f, 0.890186792377730240f,
	-0.455595736016314980f,
	0.890099416625192320f, -0.455766418819434640f, 0.890012008146243260f,
	-0.455937084865326030f,
	0.889924566944096720f, -0.456107734147714110f, 0.889837093021967900f,
	-0.456278366660324620f,
	0.889749586383072780f, -0.456448982396883920f, 0.889662047030628900f,
	-0.456619581351118910f,
	0.889574474967854580f, -0.456790163516757160f, 0.889486870197969900f,
	-0.456960728887526980f,
	0.889399232724195520f, -0.457131277457156980f, 0.889311562549753850f,
	-0.457301809219376630f,
	0.889223859677868210f, -0.457472324167916060f, 0.889136124111763240f,
	-0.457642822296505770f,
	0.889048355854664570f, -0.457813303598877170f, 0.888960554909799310f,
	-0.457983768068762120f,
	0.888872721280395630f, -0.458154215699893060f, 0.888784854969682850f,
	-0.458324646486003240f,
	0.888696955980891600f, -0.458495060420826270f, 0.888609024317253860f,
	-0.458665457498096560f,
	0.888521059982002260f, -0.458835837711549120f, 0.888433062978371320f,
	-0.459006201054919630f,
	0.888345033309596350f, -0.459176547521944090f, 0.888256970978913870f,
	-0.459346877106359630f,
	0.888168875989561730f, -0.459517189801903480f, 0.888080748344778900f,
	-0.459687485602313870f,
	0.887992588047805560f, -0.459857764501329540f, 0.887904395101883240f,
	-0.460028026492689650f,
	0.887816169510254440f, -0.460198271570134320f, 0.887727911276163020f,
	-0.460368499727404010f,
	0.887639620402853930f, -0.460538710958240010f, 0.887551296893573370f,
	-0.460708905256384080f,
	0.887462940751568840f, -0.460879082615578690f, 0.887374551980088850f,
	-0.461049243029566900f,
	0.887286130582383150f, -0.461219386492092380f, 0.887197676561702900f,
	-0.461389512996899450f,
	0.887109189921300170f, -0.461559622537733080f, 0.887020670664428360f,
	-0.461729715108338770f,
	0.886932118794342190f, -0.461899790702462730f, 0.886843534314297410f,
	-0.462069849313851750f,
	0.886754917227550840f, -0.462239890936253340f, 0.886666267537361000f,
	-0.462409915563415430f,
	0.886577585246987040f, -0.462579923189086810f, 0.886488870359689600f,
	-0.462749913807016740f,
	0.886400122878730600f, -0.462919887410955080f, 0.886311342807372780f,
	-0.463089843994652530f,
	0.886222530148880640f, -0.463259783551860150f, 0.886133684906519340f,
	-0.463429706076329830f,
	0.886044807083555600f, -0.463599611561814010f, 0.885955896683257030f,
	-0.463769500002065630f,
	0.885866953708892790f, -0.463939371390838520f, 0.885777978163732940f,
	-0.464109225721886950f,
	0.885688970051048960f, -0.464279062988965760f, 0.885599929374113360f,
	-0.464448883185830660f,
	0.885510856136199950f, -0.464618686306237820f, 0.885421750340583680f,
	-0.464788472343943990f,
	0.885332611990540590f, -0.464958241292706690f, 0.885243441089348270f,
	-0.465127993146283950f,
	0.885154237640285110f, -0.465297727898434600f, 0.885065001646630930f,
	-0.465467445542917800f,
	0.884975733111666660f, -0.465637146073493660f, 0.884886432038674560f,
	-0.465806829483922710f,
	0.884797098430937790f, -0.465976495767966180f, 0.884707732291741040f,
	-0.466146144919385890f,
	0.884618333624369920f, -0.466315776931944430f, 0.884528902432111460f,
	-0.466485391799404900f,
	0.884439438718253810f, -0.466654989515530920f, 0.884349942486086120f,
	-0.466824570074086950f,
	0.884260413738899190f, -0.466994133468838000f, 0.884170852479984500f,
	-0.467163679693549770f,
	0.884081258712634990f, -0.467333208741988420f, 0.883991632440144890f,
	-0.467502720607920920f,
	0.883901973665809470f, -0.467672215285114770f, 0.883812282392925090f,
	-0.467841692767338170f,
	0.883722558624789660f, -0.468011153048359830f, 0.883632802364701870f,
	-0.468180596121949290f,
	0.883543013615961880f, -0.468350021981876530f, 0.883453192381870920f,
	-0.468519430621912310f,
	0.883363338665731580f, -0.468688822035827900f, 0.883273452470847430f,
	-0.468858196217395330f,
	0.883183533800523390f, -0.469027553160387130f, 0.883093582658065370f,
	-0.469196892858576580f,
	0.883003599046780830f, -0.469366215305737520f, 0.882913582969978020f,
	-0.469535520495644450f,
	0.882823534430966620f, -0.469704808422072460f, 0.882733453433057650f,
	-0.469874079078797360f,
	0.882643339979562790f, -0.470043332459595620f, 0.882553194073795510f,
	-0.470212568558244170f,
	0.882463015719070150f, -0.470381787368520650f, 0.882372804918702290f,
	-0.470550988884203550f,
	0.882282561676008710f, -0.470720173099071600f, 0.882192285994307430f,
	-0.470889340006904520f,
	0.882101977876917580f, -0.471058489601482500f, 0.882011637327159590f,
	-0.471227621876586340f,
	0.881921264348355050f, -0.471396736825997640f, 0.881830858943826620f,
	-0.471565834443498420f,
	0.881740421116898320f, -0.471734914722871430f, 0.881649950870895260f,
	-0.471903977657900210f,
	0.881559448209143780f, -0.472073023242368660f, 0.881468913134971440f,
	-0.472242051470061490f,
	0.881378345651706920f, -0.472411062334764040f, 0.881287745762680100f,
	-0.472580055830262250f,
	0.881197113471222090f, -0.472749031950342790f, 0.881106448780665130f,
	-0.472917990688792760f,
	0.881015751694342870f, -0.473086932039400050f, 0.880925022215589880f,
	-0.473255855995953320f,
	0.880834260347742040f, -0.473424762552241530f, 0.880743466094136340f,
	-0.473593651702054530f,
	0.880652639458111010f, -0.473762523439182850f, 0.880561780443005700f,
	-0.473931377757417450f,
	0.880470889052160750f, -0.474100214650549970f, 0.880379965288918150f,
	-0.474269034112372980f,
	0.880289009156621010f, -0.474437836136679230f, 0.880198020658613190f,
	-0.474606620717262560f,
	0.880106999798240360f, -0.474775387847917120f, 0.880015946578849070f,
	-0.474944137522437800f,
	0.879924861003786860f, -0.475112869734620300f, 0.879833743076402940f,
	-0.475281584478260740f,
	0.879742592800047410f, -0.475450281747155870f, 0.879651410178071580f,
	-0.475618961535103300f,
	0.879560195213827890f, -0.475787623835901120f, 0.879468947910670210f,
	-0.475956268643348060f,
	0.879377668271953290f, -0.476124895951243580f, 0.879286356301033250f,
	-0.476293505753387690f,
	0.879195012001267480f, -0.476462098043581190f, 0.879103635376014330f,
	-0.476630672815625320f,
	0.879012226428633530f, -0.476799230063322090f, 0.878920785162485840f,
	-0.476967769780474170f,
	0.878829311580933360f, -0.477136291960884810f, 0.878737805687339390f,
	-0.477304796598357890f,
	0.878646267485068130f, -0.477473283686698060f, 0.878554696977485450f,
	-0.477641753219710470f,
	0.878463094167957870f, -0.477810205191200990f, 0.878371459059853480f,
	-0.477978639594976160f,
	0.878279791656541580f, -0.478147056424843010f, 0.878188091961392250f,
	-0.478315455674609480f,
	0.878096359977777130f, -0.478483837338083970f, 0.878004595709069080f,
	-0.478652201409075500f,
	0.877912799158641840f, -0.478820547881393890f, 0.877820970329870500f,
	-0.478988876748849490f,
	0.877729109226131570f, -0.479157188005253310f, 0.877637215850802230f,
	-0.479325481644417070f,
	0.877545290207261350f, -0.479493757660153010f, 0.877453332298888560f,
	-0.479662016046274180f,
	0.877361342129065140f, -0.479830256796594190f, 0.877269319701173170f,
	-0.479998479904927280f,
	0.877177265018595940f, -0.480166685365088390f, 0.877085178084718420f,
	-0.480334873170893020f,
	0.876993058902925890f, -0.480503043316157510f, 0.876900907476605650f,
	-0.480671195794698640f,
	0.876808723809145650f, -0.480839330600333960f, 0.876716507903935400f,
	-0.481007447726881590f,
	0.876624259764365310f, -0.481175547168160300f, 0.876531979393827100f,
	-0.481343628917989710f,
	0.876439666795713610f, -0.481511692970189860f, 0.876347321973419020f,
	-0.481679739318581490f,
	0.876254944930338510f, -0.481847767956986030f, 0.876162535669868460f,
	-0.482015778879225590f,
	0.876070094195406600f, -0.482183772079122720f, 0.875977620510351770f,
	-0.482351747550500980f,
	0.875885114618103810f, -0.482519705287184350f, 0.875792576522063880f,
	-0.482687645282997460f,
	0.875700006225634600f, -0.482855567531765670f, 0.875607403732219350f,
	-0.483023472027314880f,
	0.875514769045222850f, -0.483191358763471860f, 0.875422102168050940f,
	-0.483359227734063810f,
	0.875329403104110890f, -0.483527078932918740f, 0.875236671856810870f,
	-0.483694912353865140f,
	0.875143908429560360f, -0.483862727990732270f, 0.875051112825769970f,
	-0.484030525837350010f,
	0.874958285048851650f, -0.484198305887549030f, 0.874865425102218320f,
	-0.484366068135160420f,
	0.874772532989284150f, -0.484533812574016180f, 0.874679608713464510f,
	-0.484701539197948670f,
	0.874586652278176110f, -0.484869248000791060f, 0.874493663686836560f,
	-0.485036938976377290f,
	0.874400642942864790f, -0.485204612118541820f, 0.874307590049680950f,
	-0.485372267421119770f,
	0.874214505010706300f, -0.485539904877946960f, 0.874121387829363330f,
	-0.485707524482859750f,
	0.874028238509075740f, -0.485875126229695250f, 0.873935057053268240f,
	-0.486042710112291330f,
	0.873841843465366860f, -0.486210276124486420f, 0.873748597748798870f,
	-0.486377824260119440f,
	0.873655319906992630f, -0.486545354513030270f, 0.873562009943377850f,
	-0.486712866877059170f,
	0.873468667861384880f, -0.486880361346047340f, 0.873375293664446000f,
	-0.487047837913836380f,
	0.873281887355994210f, -0.487215296574268760f, 0.873188448939463790f,
	-0.487382737321187360f,
	0.873094978418290090f, -0.487550160148436000f, 0.873001475795909920f,
	-0.487717565049858800f,
	0.872907941075761080f, -0.487884952019301040f, 0.872814374261282390f,
	-0.488052321050608250f,
	0.872720775355914300f, -0.488219672137626790f, 0.872627144363097960f,
	-0.488387005274203530f,
	0.872533481286276170f, -0.488554320454186180f, 0.872439786128892280f,
	-0.488721617671423080f,
	0.872346058894391540f, -0.488888896919763170f, 0.872252299586219860f,
	-0.489056158193056030f,
	0.872158508207824480f, -0.489223401485151980f, 0.872064684762653860f,
	-0.489390626789901920f,
	0.871970829254157810f, -0.489557834101157440f, 0.871876941685786890f,
	-0.489725023412770910f,
	0.871783022060993120f, -0.489892194718595190f, 0.871689070383229740f,
	-0.490059348012483850f,
	0.871595086655950980f, -0.490226483288291160f, 0.871501070882612530f,
	-0.490393600539871970f,
	0.871407023066670950f, -0.490560699761082020f, 0.871312943211584030f,
	-0.490727780945777400f,
	0.871218831320811020f, -0.490894844087815090f, 0.871124687397811900f,
	-0.491061889181052650f,
	0.871030511446048260f, -0.491228916219348280f, 0.870936303468982760f,
	-0.491395925196560780f,
	0.870842063470078980f, -0.491562916106549900f, 0.870747791452801790f,
	-0.491729888943175760f,
	0.870653487420617430f, -0.491896843700299290f, 0.870559151376993250f,
	-0.492063780371782000f,
	0.870464783325397670f, -0.492230698951486020f, 0.870370383269300270f,
	-0.492397599433274380f,
	0.870275951212171940f, -0.492564481811010590f, 0.870181487157484560f,
	-0.492731346078558840f,
	0.870086991108711460f, -0.492898192229784040f, 0.869992463069326870f,
	-0.493065020258551700f,
	0.869897903042806340f, -0.493231830158727900f, 0.869803311032626650f,
	-0.493398621924179770f,
	0.869708687042265670f, -0.493565395548774770f, 0.869614031075202300f,
	-0.493732151026381020f,
	0.869519343134916860f, -0.493898888350867480f, 0.869424623224890890f,
	-0.494065607516103570f,
	0.869329871348606840f, -0.494232308515959670f, 0.869235087509548370f,
	-0.494398991344306650f,
	0.869140271711200560f, -0.494565655995015950f, 0.869045423957049530f,
	-0.494732302461959870f,
	0.868950544250582380f, -0.494898930739011260f, 0.868855632595287860f,
	-0.495065540820043560f,
	0.868760688994655310f, -0.495232132698931180f, 0.868665713452175690f,
	-0.495398706369549020f,
	0.868570705971340900f, -0.495565261825772540f, 0.868475666555644120f,
	-0.495731799061477960f,
	0.868380595208579800f, -0.495898318070542190f, 0.868285491933643350f,
	-0.496064818846842890f,
	0.868190356734331310f, -0.496231301384258250f, 0.868095189614141670f,
	-0.496397765676667160f,
	0.867999990576573510f, -0.496564211717949290f, 0.867904759625126920f,
	-0.496730639501984760f,
	0.867809496763303320f, -0.496897049022654470f, 0.867714201994605140f,
	-0.497063440273840250f,
	0.867618875322536230f, -0.497229813249424220f, 0.867523516750601460f,
	-0.497396167943289280f,
	0.867428126282306920f, -0.497562504349319150f, 0.867332703921159800f,
	-0.497728822461397940f,
	0.867237249670668400f, -0.497895122273410870f, 0.867141763534342470f,
	-0.498061403779243410f,
	0.867046245515692650f, -0.498227666972781870f, 0.866950695618230900f,
	-0.498393911847913210f,
	0.866855113845470430f, -0.498560138398525140f, 0.866759500200925400f,
	-0.498726346618505900f,
	0.866663854688111130f, -0.498892536501744590f, 0.866568177310544470f,
	-0.499058708042130870f,
	0.866472468071743050f, -0.499224861233555080f, 0.866376726975225830f,
	-0.499390996069908170f,
	0.866280954024512990f, -0.499557112545081840f, 0.866185149223125840f,
	-0.499723210652968540f,
	0.866089312574586770f, -0.499889290387461330f, 0.865993444082419520f,
	-0.500055351742453860f,
	0.865897543750148820f, -0.500221394711840680f, 0.865801611581300760f,
	-0.500387419289516580f,
	0.865705647579402380f, -0.500553425469377420f, 0.865609651747981990f,
	-0.500719413245319880f,
	0.865513624090569090f, -0.500885382611240710f, 0.865417564610694410f,
	-0.501051333561038040f,
	0.865321473311889800f, -0.501217266088609950f, 0.865225350197688200f,
	-0.501383180187855770f,
	0.865129195271623800f, -0.501549075852675390f, 0.865033008537231860f,
	-0.501714953076969120f,
	0.864936789998049020f, -0.501880811854638290f, 0.864840539657612870f,
	-0.502046652179584660f,
	0.864744257519462380f, -0.502212474045710790f, 0.864647943587137480f,
	-0.502378277446919760f,
	0.864551597864179340f, -0.502544062377115690f, 0.864455220354130360f,
	-0.502709828830202990f,
	0.864358811060534030f, -0.502875576800086990f, 0.864262369986934950f,
	-0.503041306280673450f,
	0.864165897136879300f, -0.503207017265868920f, 0.864069392513913790f,
	-0.503372709749581040f,
	0.863972856121586810f, -0.503538383725717580f, 0.863876287963447510f,
	-0.503704039188187070f,
	0.863779688043046720f, -0.503869676130898950f, 0.863683056363935830f,
	-0.504035294547763190f,
	0.863586392929668100f, -0.504200894432690340f, 0.863489697743797140f,
	-0.504366475779592040f,
	0.863392970809878420f, -0.504532038582380270f, 0.863296212131468230f,
	-0.504697582834967570f,
	0.863199421712124160f, -0.504863108531267590f, 0.863102599555404910f,
	-0.505028615665194080f,
	0.863005745664870320f, -0.505194104230662240f, 0.862908860044081400f,
	-0.505359574221587280f,
	0.862811942696600330f, -0.505525025631885390f, 0.862714993625990690f,
	-0.505690458455473450f,
	0.862618012835816740f, -0.505855872686268860f, 0.862521000329644520f,
	-0.506021268318189720f,
	0.862423956111040610f, -0.506186645345155230f, 0.862326880183573060f,
	-0.506352003761084800f,
	0.862229772550811240f, -0.506517343559898530f, 0.862132633216325380f,
	-0.506682664735517600f,
	0.862035462183687210f, -0.506847967281863210f, 0.861938259456469290f,
	-0.507013251192858230f,
	0.861841025038245330f, -0.507178516462425180f, 0.861743758932590700f,
	-0.507343763084487920f,
	0.861646461143081300f, -0.507508991052970870f, 0.861549131673294720f,
	-0.507674200361798890f,
	0.861451770526809320f, -0.507839391004897720f, 0.861354377707204910f,
	-0.508004562976194010f,
	0.861256953218062170f, -0.508169716269614600f, 0.861159497062963350f,
	-0.508334850879087360f,
	0.861062009245491480f, -0.508499966798540930f, 0.860964489769231010f,
	-0.508665064021904030f,
	0.860866938637767310f, -0.508830142543106990f, 0.860769355854687170f,
	-0.508995202356080090f,
	0.860671741423578380f, -0.509160243454754640f, 0.860574095348029980f,
	-0.509325265833062480f,
	0.860476417631632070f, -0.509490269484936360f, 0.860378708277976130f,
	-0.509655254404309250f,
	0.860280967290654510f, -0.509820220585115450f, 0.860183194673260990f,
	-0.509985168021289460f,
	0.860085390429390140f, -0.510150096706766810f, 0.859987554562638200f,
	-0.510315006635483240f,
	0.859889687076602290f, -0.510479897801375700f, 0.859791787974880650f,
	-0.510644770198381610f,
	0.859693857261072610f, -0.510809623820439040f, 0.859595894938779080f,
	-0.510974458661486830f,
	0.859497901011601730f, -0.511139274715464390f, 0.859399875483143450f,
	-0.511304071976312000f,
	0.859301818357008470f, -0.511468850437970300f, 0.859203729636801920f,
	-0.511633610094381240f,
	0.859105609326130450f, -0.511798350939486890f, 0.859007457428601520f,
	-0.511963072967230200f,
	0.858909273947823900f, -0.512127776171554690f, 0.858811058887407610f,
	-0.512292460546404870f,
	0.858712812250963520f, -0.512457126085725690f, 0.858614534042104190f,
	-0.512621772783462990f,
	0.858516224264442740f, -0.512786400633562960f, 0.858417882921593930f,
	-0.512951009629972980f,
	0.858319510017173440f, -0.513115599766640560f, 0.858221105554798250f,
	-0.513280171037514220f,
	0.858122669538086140f, -0.513444723436543460f, 0.858024201970656540f,
	-0.513609256957677780f,
	0.857925702856129790f, -0.513773771594868030f, 0.857827172198127430f,
	-0.513938267342065380f,
	0.857728610000272120f, -0.514102744193221660f, 0.857630016266187620f,
	-0.514267202142289710f,
	0.857531390999499150f, -0.514431641183222820f, 0.857432734203832700f,
	-0.514596061309975040f,
	0.857334045882815590f, -0.514760462516501200f, 0.857235326040076460f,
	-0.514924844796756490f,
	0.857136574679244980f, -0.515089208144697160f, 0.857037791803951680f,
	-0.515253552554280180f,
	0.856938977417828760f, -0.515417878019462930f, 0.856840131524509220f,
	-0.515582184534203790f,
	0.856741254127627470f, -0.515746472092461380f, 0.856642345230818840f,
	-0.515910740688195650f,
	0.856543404837719960f, -0.516074990315366630f, 0.856444432951968590f,
	-0.516239220967935510f,
	0.856345429577203610f, -0.516403432639863990f, 0.856246394717065210f,
	-0.516567625325114350f,
	0.856147328375194470f, -0.516731799017649870f, 0.856048230555233940f,
	-0.516895953711434150f,
	0.855949101260826910f, -0.517060089400431910f, 0.855849940495618240f,
	-0.517224206078608310f,
	0.855750748263253920f, -0.517388303739929060f, 0.855651524567380690f,
	-0.517552382378360880f,
	0.855552269411646860f, -0.517716441987871150f, 0.855452982799701830f,
	-0.517880482562427690f,
	0.855353664735196030f, -0.518044504095999340f, 0.855254315221780970f,
	-0.518208506582555460f,
	0.855154934263109620f, -0.518372490016066110f, 0.855055521862835950f,
	-0.518536454390502220f,
	0.854956078024614930f, -0.518700399699834950f, 0.854856602752102850f,
	-0.518864325938036890f,
	0.854757096048957220f, -0.519028233099080860f, 0.854657557918836460f,
	-0.519192121176940250f,
	0.854557988365400530f, -0.519355990165589640f, 0.854458387392310170f,
	-0.519519840059003760f,
	0.854358755003227440f, -0.519683670851158410f, 0.854259091201815530f,
	-0.519847482536030190f,
	0.854159395991738850f, -0.520011275107596040f, 0.854059669376662780f,
	-0.520175048559833760f,
	0.853959911360254180f, -0.520338802886721960f, 0.853860121946180770f,
	-0.520502538082239670f,
	0.853760301138111410f, -0.520666254140367160f, 0.853660448939716380f,
	-0.520829951055084670f,
	0.853560565354666840f, -0.520993628820373920f, 0.853460650386635320f,
	-0.521157287430216610f,
	0.853360704039295430f, -0.521320926878595660f, 0.853260726316321880f,
	-0.521484547159494330f,
	0.853160717221390420f, -0.521648148266897090f, 0.853060676758178320f,
	-0.521811730194788550f,
	0.852960604930363630f, -0.521975292937154390f, 0.852860501741625750f,
	-0.522138836487980760f,
	0.852760367195645300f, -0.522302360841254590f, 0.852660201296103760f,
	-0.522465865990963780f,
	0.852560004046684080f, -0.522629351931096610f, 0.852459775451070100f,
	-0.522792818655642090f,
	0.852359515512947090f, -0.522956266158590140f, 0.852259224236001090f,
	-0.523119694433931250f,
	0.852158901623919830f, -0.523283103475656430f, 0.852058547680391690f,
	-0.523446493277757830f,
	0.851958162409106380f, -0.523609863834227920f, 0.851857745813754840f,
	-0.523773215139060170f,
	0.851757297898029120f, -0.523936547186248600f, 0.851656818665622370f,
	-0.524099859969787700f,
	0.851556308120228980f, -0.524263153483673360f, 0.851455766265544310f,
	-0.524426427721901400f,
	0.851355193105265200f, -0.524589682678468950f, 0.851254588643089120f,
	-0.524752918347373360f,
	0.851153952882715340f, -0.524916134722613000f, 0.851053285827843790f,
	-0.525079331798186780f,
	0.850952587482175730f, -0.525242509568094710f, 0.850851857849413530f,
	-0.525405668026336930f,
	0.850751096933260790f, -0.525568807166914680f, 0.850650304737422090f,
	-0.525731926983829760f,
	0.850549481265603480f, -0.525895027471084630f, 0.850448626521511760f,
	-0.526058108622682760f,
	0.850347740508854980f, -0.526221170432628060f, 0.850246823231342710f,
	-0.526384212894925100f,
	0.850145874692685210f, -0.526547236003579440f, 0.850044894896594180f,
	-0.526710239752597010f,
	0.849943883846782210f, -0.526873224135984590f, 0.849842841546963320f,
	-0.527036189147750080f,
	0.849741768000852550f, -0.527199134781901280f, 0.849640663212165910f,
	-0.527362061032447540f,
	0.849539527184620890f, -0.527524967893398200f, 0.849438359921936060f,
	-0.527687855358763720f,
	0.849337161427830780f, -0.527850723422555230f, 0.849235931706025960f,
	-0.528013572078784630f,
	0.849134670760243630f, -0.528176401321464370f, 0.849033378594206800f,
	-0.528339211144607690f,
	0.848932055211639610f, -0.528502001542228480f, 0.848830700616267530f,
	-0.528664772508341320f,
	0.848729314811817130f, -0.528827524036961870f, 0.848627897802015860f,
	-0.528990256122106040f,
	0.848526449590592650f, -0.529152968757790610f, 0.848424970181277600f,
	-0.529315661938033260f,
	0.848323459577801640f, -0.529478335656851980f, 0.848221917783896990f,
	-0.529640989908265910f,
	0.848120344803297230f, -0.529803624686294610f, 0.848018740639736810f,
	-0.529966239984958620f,
	0.847917105296951410f, -0.530128835798278960f, 0.847815438778677930f,
	-0.530291412120277310f,
	0.847713741088654380f, -0.530453968944976320f, 0.847612012230619660f,
	-0.530616506266399330f,
	0.847510252208314330f, -0.530779024078570140f, 0.847408461025479730f,
	-0.530941522375513620f,
	0.847306638685858320f, -0.531104001151255000f, 0.847204785193194090f,
	-0.531266460399820390f,
	0.847102900551231500f, -0.531428900115236800f, 0.847000984763716880f,
	-0.531591320291531670f,
	0.846899037834397240f, -0.531753720922733320f, 0.846797059767020910f,
	-0.531916102002870650f,
	0.846695050565337450f, -0.532078463525973540f, 0.846593010233097190f,
	-0.532240805486072220f,
	0.846490938774052130f, -0.532403127877197900f, 0.846388836191954930f,
	-0.532565430693382580f,
	0.846286702490559710f, -0.532727713928658810f, 0.846184537673621560f,
	-0.532889977577059800f,
	0.846082341744897050f, -0.533052221632619450f, 0.845980114708143270f,
	-0.533214446089372960f,
	0.845877856567119000f, -0.533376650941355330f, 0.845775567325584010f,
	-0.533538836182603120f,
	0.845673246987299070f, -0.533701001807152960f, 0.845570895556026270f,
	-0.533863147809042650f,
	0.845468513035528830f, -0.534025274182310380f, 0.845366099429570970f,
	-0.534187380920995380f,
	0.845263654741918220f, -0.534349468019137520f, 0.845161178976337140f,
	-0.534511535470777120f,
	0.845058672136595470f, -0.534673583269955510f, 0.844956134226462210f,
	-0.534835611410714560f,
	0.844853565249707120f, -0.534997619887097150f, 0.844750965210101510f,
	-0.535159608693146600f,
	0.844648334111417820f, -0.535321577822907120f, 0.844545671957429240f,
	-0.535483527270423370f,
	0.844442978751910660f, -0.535645457029741090f, 0.844340254498637590f,
	-0.535807367094906390f,
	0.844237499201387020f, -0.535969257459966710f, 0.844134712863936930f,
	-0.536131128118969460f,
	0.844031895490066410f, -0.536292979065963180f, 0.843929047083555870f,
	-0.536454810294997090f,
	0.843826167648186740f, -0.536616621800121040f, 0.843723257187741660f,
	-0.536778413575385920f,
	0.843620315706004150f, -0.536940185614842910f, 0.843517343206759200f,
	-0.537101937912544130f,
	0.843414339693792760f, -0.537263670462542530f, 0.843311305170892140f,
	-0.537425383258891550f,
	0.843208239641845440f, -0.537587076295645390f, 0.843105143110442160f,
	-0.537748749566859360f,
	0.843002015580472940f, -0.537910403066588880f, 0.842898857055729310f,
	-0.538072036788890600f,
	0.842795667540004120f, -0.538233650727821700f, 0.842692447037091670f,
	-0.538395244877439950f,
	0.842589195550786710f, -0.538556819231804100f, 0.842485913084885630f,
	-0.538718373784973560f,
	0.842382599643185850f, -0.538879908531008420f, 0.842279255229485990f,
	-0.539041423463969440f,
	0.842175879847585570f, -0.539202918577918240f, 0.842072473501285560f,
	-0.539364393866917040f,
	0.841969036194387680f, -0.539525849325028890f, 0.841865567930695340f,
	-0.539687284946317570f,
	0.841762068714012490f, -0.539848700724847590f, 0.841658538548144760f,
	-0.540010096654684020f,
	0.841554977436898440f, -0.540171472729892850f, 0.841451385384081260f,
	-0.540332828944540710f,
	0.841347762393501950f, -0.540494165292695230f, 0.841244108468970580f,
	-0.540655481768424150f,
	0.841140423614298080f, -0.540816778365796670f, 0.841036707833296650f,
	-0.540978055078882080f,
	0.840932961129779780f, -0.541139311901750800f, 0.840829183507561640f,
	-0.541300548828474120f,
	0.840725374970458070f, -0.541461765853123440f, 0.840621535522285690f,
	-0.541622962969771530f,
	0.840517665166862550f, -0.541784140172491550f, 0.840413763908007480f,
	-0.541945297455357360f,
	0.840309831749540770f, -0.542106434812443920f, 0.840205868695283580f,
	-0.542267552237826520f,
	0.840101874749058400f, -0.542428649725581250f, 0.839997849914688840f,
	-0.542589727269785270f,
	0.839893794195999520f, -0.542750784864515890f, 0.839789707596816370f,
	-0.542911822503851730f,
	0.839685590120966110f, -0.543072840181871740f, 0.839581441772277120f,
	-0.543233837892655890f,
	0.839477262554578550f, -0.543394815630284800f, 0.839373052471700690f,
	-0.543555773388839540f,
	0.839268811527475230f, -0.543716711162402280f, 0.839164539725734680f,
	-0.543877628945055980f,
	0.839060237070312740f, -0.544038526730883820f, 0.838955903565044460f,
	-0.544199404513970310f,
	0.838851539213765760f, -0.544360262288400400f, 0.838747144020313920f,
	-0.544521100048259600f,
	0.838642717988527300f, -0.544681917787634530f, 0.838538261122245280f,
	-0.544842715500612360f,
	0.838433773425308340f, -0.545003493181281160f, 0.838329254901558300f,
	-0.545164250823729320f,
	0.838224705554838080f, -0.545324988422046460f, 0.838120125388991500f,
	-0.545485705970322530f,
	0.838015514407863820f, -0.545646403462648590f, 0.837910872615301170f,
	-0.545807080893116140f,
	0.837806200015150940f, -0.545967738255817570f, 0.837701496611261700f,
	-0.546128375544845950f,
	0.837596762407483040f, -0.546288992754295210f, 0.837491997407665890f,
	-0.546449589878259650f,
	0.837387201615661940f, -0.546610166910834860f, 0.837282375035324320f,
	-0.546770723846116800f,
	0.837177517670507300f, -0.546931260678202190f, 0.837072629525066000f,
	-0.547091777401188530f,
	0.836967710602857020f, -0.547252274009174090f, 0.836862760907737920f,
	-0.547412750496257930f,
	0.836757780443567190f, -0.547573206856539760f, 0.836652769214204950f,
	-0.547733643084120090f,
	0.836547727223512010f, -0.547894059173100190f, 0.836442654475350380f,
	-0.548054455117581880f,
	0.836337550973583530f, -0.548214830911667780f, 0.836232416722075600f,
	-0.548375186549461600f,
	0.836127251724692270f, -0.548535522025067390f, 0.836022055985299880f,
	-0.548695837332590090f,
	0.835916829507766360f, -0.548856132466135290f, 0.835811572295960700f,
	-0.549016407419809390f,
	0.835706284353752600f, -0.549176662187719660f, 0.835600965685013410f,
	-0.549336896763974010f,
	0.835495616293615350f, -0.549497111142680960f, 0.835390236183431890f,
	-0.549657305317949870f,
	0.835284825358337370f, -0.549817479283890910f, 0.835179383822207690f,
	-0.549977633034614890f,
	0.835073911578919410f, -0.550137766564233630f, 0.834968408632350450f,
	-0.550297879866859190f,
	0.834862874986380010f, -0.550457972936604810f, 0.834757310644888230f,
	-0.550618045767584330f,
	0.834651715611756440f, -0.550778098353912120f, 0.834546089890866870f,
	-0.550938130689703880f,
	0.834440433486103190f, -0.551098142769075430f, 0.834334746401350080f,
	-0.551258134586143590f,
	0.834229028640493420f, -0.551418106135026060f, 0.834123280207420100f,
	-0.551578057409841000f,
	0.834017501106018130f, -0.551737988404707340f, 0.833911691340176840f,
	-0.551897899113745210f,
	0.833805850913786340f, -0.552057789531074980f, 0.833699979830738290f,
	-0.552217659650817930f,
	0.833594078094925140f, -0.552377509467096070f, 0.833488145710240770f,
	-0.552537338974032120f,
	0.833382182680579730f, -0.552697148165749770f, 0.833276189009838240f,
	-0.552856937036373290f,
	0.833170164701913190f, -0.553016705580027470f, 0.833064109760702890f,
	-0.553176453790838350f,
	0.832958024190106670f, -0.553336181662932300f, 0.832851907994025090f,
	-0.553495889190436570f,
	0.832745761176359460f, -0.553655576367479310f, 0.832639583741012770f,
	-0.553815243188189090f,
	0.832533375691888680f, -0.553974889646695500f, 0.832427137032892280f,
	-0.554134515737128910f,
	0.832320867767929680f, -0.554294121453620000f, 0.832214567900907980f,
	-0.554453706790300930f,
	0.832108237435735590f, -0.554613271741304040f, 0.832001876376321950f,
	-0.554772816300762470f,
	0.831895484726577590f, -0.554932340462810370f, 0.831789062490414400f,
	-0.555091844221582420f,
	0.831682609671745120f, -0.555251327571213980f, 0.831576126274483740f,
	-0.555410790505841630f,
	0.831469612302545240f, -0.555570233019602180f, 0.831363067759845920f,
	-0.555729655106633410f,
	0.831256492650303210f, -0.555889056761073810f, 0.831149886977835540f,
	-0.556048437977062600f,
	0.831043250746362320f, -0.556207798748739930f, 0.830936583959804410f,
	-0.556367139070246370f,
	0.830829886622083570f, -0.556526458935723610f, 0.830723158737122880f,
	-0.556685758339313890f,
	0.830616400308846310f, -0.556845037275160100f, 0.830509611341179070f,
	-0.557004295737405950f,
	0.830402791838047550f, -0.557163533720196220f, 0.830295941803379070f,
	-0.557322751217676160f,
	0.830189061241102370f, -0.557481948223991550f, 0.830082150155146970f,
	-0.557641124733289420f,
	0.829975208549443950f, -0.557800280739716990f, 0.829868236427924840f,
	-0.557959416237422960f,
	0.829761233794523050f, -0.558118531220556100f, 0.829654200653172640f,
	-0.558277625683266330f,
	0.829547137007808910f, -0.558436699619704100f, 0.829440042862368170f,
	-0.558595753024020760f,
	0.829332918220788250f, -0.558754785890368310f, 0.829225763087007570f,
	-0.558913798212899770f,
	0.829118577464965980f, -0.559072789985768480f, 0.829011361358604430f,
	-0.559231761203128900f,
	0.828904114771864870f, -0.559390711859136140f, 0.828796837708690610f,
	-0.559549641947945760f,
	0.828689530173025820f, -0.559708551463714680f, 0.828582192168815790f,
	-0.559867440400600210f,
	0.828474823700007130f, -0.560026308752760380f, 0.828367424770547480f,
	-0.560185156514354080f,
	0.828259995384385660f, -0.560343983679540860f, 0.828152535545471410f,
	-0.560502790242481060f,
	0.828045045257755800f, -0.560661576197336030f, 0.827937524525190870f,
	-0.560820341538267430f,
	0.827829973351729920f, -0.560979086259438150f, 0.827722391741327220f,
	-0.561137810355011420f,
	0.827614779697938400f, -0.561296513819151470f, 0.827507137225519830f,
	-0.561455196646023280f,
	0.827399464328029470f, -0.561613858829792420f, 0.827291761009425810f,
	-0.561772500364625340f,
	0.827184027273669130f, -0.561931121244689470f, 0.827076263124720270f,
	-0.562089721464152480f,
	0.826968468566541600f, -0.562248301017183150f, 0.826860643603096190f,
	-0.562406859897951140f,
	0.826752788238348520f, -0.562565398100626560f, 0.826644902476264320f,
	-0.562723915619380400f,
	0.826536986320809960f, -0.562882412448384440f, 0.826429039775953500f,
	-0.563040888581811230f,
	0.826321062845663530f, -0.563199344013834090f, 0.826213055533910220f,
	-0.563357778738627020f,
	0.826105017844664610f, -0.563516192750364800f, 0.825996949781899080f,
	-0.563674586043223070f,
	0.825888851349586780f, -0.563832958611378170f, 0.825780722551702430f,
	-0.563991310449006970f,
	0.825672563392221390f, -0.564149641550287680f, 0.825564373875120490f,
	-0.564307951909398640f,
	0.825456154004377550f, -0.564466241520519500f, 0.825347903783971380f,
	-0.564624510377830120f,
	0.825239623217882250f, -0.564782758475511400f, 0.825131312310091070f,
	-0.564940985807745210f,
	0.825022971064580220f, -0.565099192368713980f, 0.824914599485333190f,
	-0.565257378152600800f,
	0.824806197576334330f, -0.565415543153589660f, 0.824697765341569470f,
	-0.565573687365865330f,
	0.824589302785025290f, -0.565731810783613120f, 0.824480809910689500f,
	-0.565889913401019570f,
	0.824372286722551250f, -0.566047995212271450f, 0.824263733224600560f,
	-0.566206056211556730f,
	0.824155149420828570f, -0.566364096393063840f, 0.824046535315227760f,
	-0.566522115750982100f,
	0.823937890911791370f, -0.566680114279501600f, 0.823829216214513990f,
	-0.566838091972813320f,
	0.823720511227391430f, -0.566996048825108680f, 0.823611775954420260f,
	-0.567153984830580100f,
	0.823503010399598500f, -0.567311899983420800f, 0.823394214566925080f,
	-0.567469794277824510f,
	0.823285388460400110f, -0.567627667707986230f, 0.823176532084024860f,
	-0.567785520268101140f,
	0.823067645441801670f, -0.567943351952365560f, 0.822958728537734000f,
	-0.568101162754976460f,
	0.822849781375826430f, -0.568258952670131490f, 0.822740803960084420f,
	-0.568416721692029280f,
	0.822631796294514990f, -0.568574469814869140f, 0.822522758383125940f,
	-0.568732197032851050f,
	0.822413690229926390f, -0.568889903340175860f, 0.822304591838926350f,
	-0.569047588731045110f,
	0.822195463214137170f, -0.569205253199661200f, 0.822086304359571090f,
	-0.569362896740227220f,
	0.821977115279241550f, -0.569520519346947140f, 0.821867895977163250f,
	-0.569678121014025600f,
	0.821758646457351750f, -0.569835701735668000f, 0.821649366723823940f,
	-0.569993261506080540f,
	0.821540056780597610f, -0.570150800319470300f, 0.821430716631691870f,
	-0.570308318170044900f,
	0.821321346281126740f, -0.570465815052012990f, 0.821211945732923550f,
	-0.570623290959583750f,
	0.821102514991104650f, -0.570780745886967260f, 0.820993054059693580f,
	-0.570938179828374360f,
	0.820883562942714580f, -0.571095592778016690f, 0.820774041644193650f,
	-0.571252984730106660f,
	0.820664490168157460f, -0.571410355678857230f, 0.820554908518633890f,
	-0.571567705618482580f,
	0.820445296699652050f, -0.571725034543197120f, 0.820335654715241840f,
	-0.571882342447216590f,
	0.820225982569434690f, -0.572039629324757050f, 0.820116280266262820f,
	-0.572196895170035580f,
	0.820006547809759680f, -0.572354139977269920f, 0.819896785203959810f,
	-0.572511363740678790f,
	0.819786992452898990f, -0.572668566454481160f, 0.819677169560613870f,
	-0.572825748112897550f,
	0.819567316531142230f, -0.572982908710148560f, 0.819457433368523280f,
	-0.573140048240455950f,
	0.819347520076796900f, -0.573297166698042200f, 0.819237576660004520f,
	-0.573454264077130400f,
	0.819127603122188240f, -0.573611340371944610f, 0.819017599467391500f,
	-0.573768395576709560f,
	0.818907565699658950f, -0.573925429685650750f, 0.818797501823036010f,
	-0.574082442692994470f,
	0.818687407841569680f, -0.574239434592967890f, 0.818577283759307610f,
	-0.574396405379798750f,
	0.818467129580298660f, -0.574553355047715760f, 0.818356945308593150f,
	-0.574710283590948330f,
	0.818246730948242070f, -0.574867191003726740f, 0.818136486503297730f,
	-0.575024077280281710f,
	0.818026211977813440f, -0.575180942414845080f, 0.817915907375843850f,
	-0.575337786401649450f,
	0.817805572701444270f, -0.575494609234928120f, 0.817695207958671680f,
	-0.575651410908915140f,
	0.817584813151583710f, -0.575808191417845340f, 0.817474388284239240f,
	-0.575964950755954220f,
	0.817363933360698460f, -0.576121688917478280f, 0.817253448385022340f,
	-0.576278405896654910f,
	0.817142933361272970f, -0.576435101687721830f, 0.817032388293513880f,
	-0.576591776284917760f,
	0.816921813185809480f, -0.576748429682482410f, 0.816811208042225290f,
	-0.576905061874655960f,
	0.816700572866827850f, -0.577061672855679440f, 0.816589907663684890f,
	-0.577218262619794920f,
	0.816479212436865390f, -0.577374831161244880f, 0.816368487190439200f,
	-0.577531378474272720f,
	0.816257731928477390f, -0.577687904553122800f, 0.816146946655052270f,
	-0.577844409392039850f,
	0.816036131374236810f, -0.578000892985269910f, 0.815925286090105510f,
	-0.578157355327059360f,
	0.815814410806733780f, -0.578313796411655590f, 0.815703505528198260f,
	-0.578470216233306630f,
	0.815592570258576790f, -0.578626614786261430f, 0.815481605001947770f,
	-0.578782992064769690f,
	0.815370609762391290f, -0.578939348063081780f, 0.815259584543988280f,
	-0.579095682775449090f,
	0.815148529350820830f, -0.579251996196123550f, 0.815037444186972220f,
	-0.579408288319357870f,
	0.814926329056526620f, -0.579564559139405630f, 0.814815183963569440f,
	-0.579720808650521450f,
	0.814704008912187080f, -0.579877036846960350f, 0.814592803906467270f,
	-0.580033243722978150f,
	0.814481568950498610f, -0.580189429272831680f, 0.814370304048371070f,
	-0.580345593490778300f,
	0.814259009204175270f, -0.580501736371076490f, 0.814147684422003360f,
	-0.580657857907985300f,
	0.814036329705948410f, -0.580813958095764530f, 0.813924945060104600f,
	-0.580970036928674770f,
	0.813813530488567190f, -0.581126094400977620f, 0.813702085995432700f,
	-0.581282130506935000f,
	0.813590611584798510f, -0.581438145240810170f, 0.813479107260763220f,
	-0.581594138596866930f,
	0.813367573027426570f, -0.581750110569369650f, 0.813256008888889380f,
	-0.581906061152583810f,
	0.813144414849253590f, -0.582061990340775440f, 0.813032790912622040f,
	-0.582217898128211670f,
	0.812921137083098770f, -0.582373784509160110f, 0.812809453364789270f,
	-0.582529649477889320f,
	0.812697739761799490f, -0.582685493028668460f, 0.812585996278237130f,
	-0.582841315155767650f,
	0.812474222918210480f, -0.582997115853457700f, 0.812362419685829230f,
	-0.583152895116010430f,
	0.812250586585203880f, -0.583308652937698290f, 0.812138723620446480f,
	-0.583464389312794320f,
	0.812026830795669730f, -0.583620104235572760f, 0.811914908114987790f,
	-0.583775797700308070f,
	0.811802955582515470f, -0.583931469701276180f, 0.811690973202369050f,
	-0.584087120232753440f,
	0.811578960978665890f, -0.584242749289016980f, 0.811466918915524250f,
	-0.584398356864344600f,
	0.811354847017063730f, -0.584553942953015330f, 0.811242745287404810f,
	-0.584709507549308390f,
	0.811130613730669190f, -0.584865050647504490f, 0.811018452350979470f,
	-0.585020572241884530f,
	0.810906261152459670f, -0.585176072326730410f, 0.810794040139234730f,
	-0.585331550896324940f,
	0.810681789315430780f, -0.585487007944951340f, 0.810569508685174630f,
	-0.585642443466894420f,
	0.810457198252594770f, -0.585797857456438860f, 0.810344858021820550f,
	-0.585953249907870570f,
	0.810232487996982330f, -0.586108620815476430f, 0.810120088182211600f,
	-0.586263970173543590f,
	0.810007658581641140f, -0.586419297976360500f, 0.809895199199404450f,
	-0.586574604218216170f,
	0.809782710039636530f, -0.586729888893400390f, 0.809670191106473090f,
	-0.586885151996203950f,
	0.809557642404051260f, -0.587040393520917970f, 0.809445063936509170f,
	-0.587195613461834800f,
	0.809332455707985950f, -0.587350811813247660f, 0.809219817722621750f,
	-0.587505988569450020f,
	0.809107149984558240f, -0.587661143724736660f, 0.808994452497937670f,
	-0.587816277273402910f,
	0.808881725266903610f, -0.587971389209745010f, 0.808768968295600850f,
	-0.588126479528059850f,
	0.808656181588174980f, -0.588281548222645220f, 0.808543365148773010f,
	-0.588436595287799790f,
	0.808430518981542720f, -0.588591620717822890f, 0.808317643090633250f,
	-0.588746624507014540f,
	0.808204737480194720f, -0.588901606649675720f, 0.808091802154378370f,
	-0.589056567140108460f,
	0.807978837117336310f, -0.589211505972614960f, 0.807865842373222120f,
	-0.589366423141498790f,
	0.807752817926190360f, -0.589521318641063940f, 0.807639763780396480f,
	-0.589676192465615420f,
	0.807526679939997160f, -0.589831044609458790f, 0.807413566409150190f,
	-0.589985875066900920f,
	0.807300423192014450f, -0.590140683832248820f, 0.807187250292749960f,
	-0.590295470899810830f,
	0.807074047715517610f, -0.590450236263895810f, 0.806960815464479730f,
	-0.590604979918813330f,
	0.806847553543799330f, -0.590759701858874160f, 0.806734261957640860f,
	-0.590914402078389520f,
	0.806620940710169650f, -0.591069080571671400f, 0.806507589805552260f,
	-0.591223737333032910f,
	0.806394209247956240f, -0.591378372356787580f, 0.806280799041550480f,
	-0.591532985637249990f,
	0.806167359190504420f, -0.591687577168735430f, 0.806053889698989060f,
	-0.591842146945560140f,
	0.805940390571176280f, -0.591996694962040990f, 0.805826861811239300f,
	-0.592151221212495530f,
	0.805713303423352230f, -0.592305725691242290f, 0.805599715411690060f,
	-0.592460208392600830f,
	0.805486097780429230f, -0.592614669310891130f, 0.805372450533747060f,
	-0.592769108440434070f,
	0.805258773675822210f, -0.592923525775551300f, 0.805145067210834230f,
	-0.593077921310565470f,
	0.805031331142963660f, -0.593232295039799800f, 0.804917565476392260f,
	-0.593386646957578480f,
	0.804803770215302920f, -0.593540977058226390f, 0.804689945363879500f,
	-0.593695285336069190f,
	0.804576090926307110f, -0.593849571785433630f, 0.804462206906771840f,
	-0.594003836400646690f,
	0.804348293309460780f, -0.594158079176036800f, 0.804234350138562260f,
	-0.594312300105932830f,
	0.804120377398265810f, -0.594466499184664430f, 0.804006375092761520f,
	-0.594620676406562240f,
	0.803892343226241260f, -0.594774831765957580f, 0.803778281802897570f,
	-0.594928965257182420f,
	0.803664190826924090f, -0.595083076874569960f, 0.803550070302515680f,
	-0.595237166612453850f,
	0.803435920233868120f, -0.595391234465168730f, 0.803321740625178580f,
	-0.595545280427049790f,
	0.803207531480644940f, -0.595699304492433360f, 0.803093292804466400f,
	-0.595853306655656280f,
	0.802979024600843250f, -0.596007286911056530f, 0.802864726873976700f,
	-0.596161245252972540f,
	0.802750399628069160f, -0.596315181675743710f, 0.802636042867324150f,
	-0.596469096173710360f,
	0.802521656595946430f, -0.596622988741213220f, 0.802407240818141300f,
	-0.596776859372594390f,
	0.802292795538115720f, -0.596930708062196500f, 0.802178320760077450f,
	-0.597084534804362740f,
	0.802063816488235440f, -0.597238339593437420f, 0.801949282726799770f,
	-0.597392122423765710f,
	0.801834719479981310f, -0.597545883289693160f, 0.801720126751992330f,
	-0.597699622185566830f,
	0.801605504547046150f, -0.597853339105733910f, 0.801490852869356950f,
	-0.598007034044542700f,
	0.801376171723140240f, -0.598160706996342270f, 0.801261461112612540f,
	-0.598314357955482600f,
	0.801146721041991360f, -0.598467986916314310f, 0.801031951515495330f,
	-0.598621593873188920f,
	0.800917152537344300f, -0.598775178820458720f, 0.800802324111759110f,
	-0.598928741752476900f,
	0.800687466242961610f, -0.599082282663597310f, 0.800572578935174860f,
	-0.599235801548174570f,
	0.800457662192622820f, -0.599389298400564540f, 0.800342716019530660f,
	-0.599542773215123390f,
	0.800227740420124790f, -0.599696225986208310f, 0.800112735398632370f,
	-0.599849656708177250f,
	0.799997700959281910f, -0.600003065375388940f, 0.799882637106302810f,
	-0.600156451982203240f,
	0.799767543843925680f, -0.600309816522980430f, 0.799652421176382240f,
	-0.600463158992081580f,
	0.799537269107905010f, -0.600616479383868970f, 0.799422087642728040f,
	-0.600769777692705230f,
	0.799306876785086160f, -0.600923053912954090f, 0.799191636539215210f,
	-0.601076308038980160f,
	0.799076366909352350f, -0.601229540065148500f, 0.798961067899735760f,
	-0.601382749985825420f,
	0.798845739514604580f, -0.601535937795377730f, 0.798730381758199210f,
	-0.601689103488172950f,
	0.798614994634760820f, -0.601842247058580030f, 0.798499578148532120f,
	-0.601995368500968020f,
	0.798384132303756380f, -0.602148467809707210f, 0.798268657104678430f,
	-0.602301544979168550f,
	0.798153152555543750f, -0.602454600003723750f, 0.798037618660599410f,
	-0.602607632877745440f,
	0.797922055424093000f, -0.602760643595607220f, 0.797806462850273570f,
	-0.602913632151683030f,
	0.797690840943391160f, -0.603066598540348160f, 0.797575189707696700f,
	-0.603219542755978440f,
	0.797459509147442460f, -0.603372464792950260f, 0.797343799266881700f,
	-0.603525364645641550f,
	0.797228060070268700f, -0.603678242308430370f, 0.797112291561858920f,
	-0.603831097775695880f,
	0.796996493745908750f, -0.603983931041818020f, 0.796880666626675780f,
	-0.604136742101177520f,
	0.796764810208418830f, -0.604289530948155960f, 0.796648924495397260f,
	-0.604442297577135860f,
	0.796533009491872000f, -0.604595041982500360f, 0.796417065202104980f,
	-0.604747764158633410f,
	0.796301091630359110f, -0.604900464099919820f, 0.796185088780898440f,
	-0.605053141800745320f,
	0.796069056657987990f, -0.605205797255496500f, 0.795952995265893910f,
	-0.605358430458560530f,
	0.795836904608883570f, -0.605511041404325550f, 0.795720784691225090f,
	-0.605663630087180380f,
	0.795604635517188070f, -0.605816196501514970f, 0.795488457091042990f,
	-0.605968740641719680f,
	0.795372249417061310f, -0.606121262502186120f, 0.795256012499515610f,
	-0.606273762077306430f,
	0.795139746342679590f, -0.606426239361473550f, 0.795023450950828050f,
	-0.606578694349081290f,
	0.794907126328237010f, -0.606731127034524480f, 0.794790772479183170f,
	-0.606883537412198470f,
	0.794674389407944550f, -0.607035925476499650f, 0.794557977118800380f,
	-0.607188291221825160f,
	0.794441535616030590f, -0.607340634642572930f, 0.794325064903916520f,
	-0.607492955733141550f,
	0.794208564986740640f, -0.607645254487930830f, 0.794092035868785960f,
	-0.607797530901341140f,
	0.793975477554337170f, -0.607949784967773630f, 0.793858890047679730f,
	-0.608102016681630440f,
	0.793742273353100210f, -0.608254226037314490f, 0.793625627474886300f,
	-0.608406413029229150f,
	0.793508952417326660f, -0.608558577651779450f, 0.793392248184711100f,
	-0.608710719899370310f,
	0.793275514781330630f, -0.608862839766408200f, 0.793158752211477140f,
	-0.609014937247299830f,
	0.793041960479443640f, -0.609167012336453210f, 0.792925139589524260f,
	-0.609319065028276820f,
	0.792808289546014120f, -0.609471095317180240f, 0.792691410353209450f,
	-0.609623103197573730f,
	0.792574502015407690f, -0.609775088663868430f, 0.792457564536907080f,
	-0.609927051710476120f,
	0.792340597922007170f, -0.610078992331809620f, 0.792223602175008310f,
	-0.610230910522282620f,
	0.792106577300212390f, -0.610382806276309480f, 0.791989523301921850f,
	-0.610534679588305320f,
	0.791872440184440470f, -0.610686530452686280f, 0.791755327952073150f,
	-0.610838358863869170f,
	0.791638186609125880f, -0.610990164816271660f, 0.791521016159905220f,
	-0.611141948304312570f,
	0.791403816608719500f, -0.611293709322410890f, 0.791286587959877830f,
	-0.611445447864987000f,
	0.791169330217690200f, -0.611597163926461910f, 0.791052043386467950f,
	-0.611748857501257290f,
	0.790934727470523290f, -0.611900528583796070f, 0.790817382474169770f,
	-0.612052177168501470f,
	0.790700008401721610f, -0.612203803249797950f, 0.790582605257494460f,
	-0.612355406822110650f,
	0.790465173045804880f, -0.612506987879865570f, 0.790347711770970520f,
	-0.612658546417489290f,
	0.790230221437310030f, -0.612810082429409710f, 0.790112702049143300f,
	-0.612961595910055170f,
	0.789995153610791090f, -0.613113086853854910f, 0.789877576126575280f,
	-0.613264555255239040f,
	0.789759969600819070f, -0.613416001108638590f, 0.789642334037846340f,
	-0.613567424408485330f,
	0.789524669441982190f, -0.613718825149211720f, 0.789406975817552930f,
	-0.613870203325251330f,
	0.789289253168885650f, -0.614021558931038380f, 0.789171501500308900f,
	-0.614172891961007990f,
	0.789053720816151880f, -0.614324202409595950f, 0.788935911120745240f,
	-0.614475490271239040f,
	0.788818072418420280f, -0.614626755540375050f, 0.788700204713509660f,
	-0.614777998211442080f,
	0.788582308010347120f, -0.614929218278879590f, 0.788464382313267540f,
	-0.615080415737127460f,
	0.788346427626606340f, -0.615231590580626820f, 0.788228443954700490f,
	-0.615382742803819220f,
	0.788110431301888070f, -0.615533872401147320f, 0.787992389672507950f,
	-0.615684979367054570f,
	0.787874319070900220f, -0.615836063695985090f, 0.787756219501406060f,
	-0.615987125382383760f,
	0.787638090968367450f, -0.616138164420696910f, 0.787519933476127810f,
	-0.616289180805370980f,
	0.787401747029031430f, -0.616440174530853650f, 0.787283531631423620f,
	-0.616591145591593110f,
	0.787165287287651010f, -0.616742093982038720f, 0.787047014002060790f,
	-0.616893019696640680f,
	0.786928711779001810f, -0.617043922729849760f, 0.786810380622823490f,
	-0.617194803076117630f,
	0.786692020537876790f, -0.617345660729896830f, 0.786573631528513230f,
	-0.617496495685640910f,
	0.786455213599085770f, -0.617647307937803870f, 0.786336766753948260f,
	-0.617798097480841020f,
	0.786218290997455660f, -0.617948864309208150f, 0.786099786333963930f,
	-0.618099608417362000f,
	0.785981252767830150f, -0.618250329799760250f, 0.785862690303412600f,
	-0.618401028450860980f,
	0.785744098945070360f, -0.618551704365123740f, 0.785625478697163700f,
	-0.618702357537008530f,
	0.785506829564053930f, -0.618852987960976320f, 0.785388151550103550f,
	-0.619003595631488660f,
	0.785269444659675850f, -0.619154180543008410f, 0.785150708897135560f,
	-0.619304742689998690f,
	0.785031944266848080f, -0.619455282066924020f, 0.784913150773180020f,
	-0.619605798668249270f,
	0.784794328420499230f, -0.619756292488440660f, 0.784675477213174320f,
	-0.619906763521964720f,
	0.784556597155575240f, -0.620057211763289100f, 0.784437688252072830f,
	-0.620207637206882430f,
	0.784318750507038920f, -0.620358039847213720f, 0.784199783924846570f,
	-0.620508419678753360f,
	0.784080788509869950f, -0.620658776695972140f, 0.783961764266484120f,
	-0.620809110893341900f,
	0.783842711199065230f, -0.620959422265335180f, 0.783723629311990470f,
	-0.621109710806425630f,
	0.783604518609638200f, -0.621259976511087550f, 0.783485379096387820f,
	-0.621410219373796150f,
	0.783366210776619720f, -0.621560439389027160f, 0.783247013654715380f,
	-0.621710636551257690f,
	0.783127787735057310f, -0.621860810854965360f, 0.783008533022029110f,
	-0.622010962294628600f,
	0.782889249520015480f, -0.622161090864726820f, 0.782769937233402050f,
	-0.622311196559740320f,
	0.782650596166575730f, -0.622461279374149970f, 0.782531226323924240f,
	-0.622611339302437730f,
	0.782411827709836530f, -0.622761376339086350f, 0.782292400328702400f,
	-0.622911390478579460f,
	0.782172944184913010f, -0.623061381715401260f, 0.782053459282860300f,
	-0.623211350044037270f,
	0.781933945626937630f, -0.623361295458973230f, 0.781814403221538830f,
	-0.623511217954696440f,
	0.781694832071059390f, -0.623661117525694530f, 0.781575232179895550f,
	-0.623810994166456130f,
	0.781455603552444590f, -0.623960847871470660f, 0.781335946193104870f,
	-0.624110678635228510f,
	0.781216260106276090f, -0.624260486452220650f, 0.781096545296358520f,
	-0.624410271316939270f,
	0.780976801767753750f, -0.624560033223877210f, 0.780857029524864580f,
	-0.624709772167528100f,
	0.780737228572094490f, -0.624859488142386340f, 0.780617398913848400f,
	-0.625009181142947460f,
	0.780497540554531910f, -0.625158851163707620f, 0.780377653498552040f,
	-0.625308498199164010f,
	0.780257737750316590f, -0.625458122243814360f, 0.780137793314234610f,
	-0.625607723292157410f,
	0.780017820194715990f, -0.625757301338692900f, 0.779897818396172000f,
	-0.625906856377921090f,
	0.779777787923014550f, -0.626056388404343520f, 0.779657728779656890f,
	-0.626205897412462130f,
	0.779537640970513260f, -0.626355383396779990f, 0.779417524499998900f,
	-0.626504846351800810f,
	0.779297379372530300f, -0.626654286272029350f, 0.779177205592524680f,
	-0.626803703151971200f,
	0.779057003164400630f, -0.626953096986132660f, 0.778936772092577500f,
	-0.627102467769020900f,
	0.778816512381475980f, -0.627251815495144080f, 0.778696224035517530f,
	-0.627401140159011050f,
	0.778575907059125050f, -0.627550441755131530f, 0.778455561456721900f,
	-0.627699720278016240f,
	0.778335187232733210f, -0.627848975722176460f, 0.778214784391584540f,
	-0.627998208082124700f,
	0.778094352937702790f, -0.628147417352374000f, 0.777973892875516100f,
	-0.628296603527438320f,
	0.777853404209453150f, -0.628445766601832710f, 0.777732886943944050f,
	-0.628594906570072550f,
	0.777612341083420030f, -0.628744023426674680f, 0.777491766632313010f,
	-0.628893117166156480f,
	0.777371163595056310f, -0.629042187783036000f, 0.777250531976084070f,
	-0.629191235271832290f,
	0.777129871779831620f, -0.629340259627065630f, 0.777009183010735290f,
	-0.629489260843256630f,
	0.776888465673232440f, -0.629638238914926980f, 0.776767719771761510f,
	-0.629787193836599200f,
	0.776646945310762060f, -0.629936125602796440f, 0.776526142294674430f,
	-0.630085034208043180f,
	0.776405310727940390f, -0.630233919646864370f, 0.776284450615002510f,
	-0.630382781913785940f,
	0.776163561960304340f, -0.630531621003334600f, 0.776042644768290770f,
	-0.630680436910037940f,
	0.775921699043407690f, -0.630829229628424470f, 0.775800724790101650f,
	-0.630977999153023550f,
	0.775679722012820650f, -0.631126745478365340f, 0.775558690716013580f,
	-0.631275468598980760f,
	0.775437630904130540f, -0.631424168509401860f, 0.775316542581622530f,
	-0.631572845204161020f,
	0.775195425752941420f, -0.631721498677792260f, 0.775074280422540450f,
	-0.631870128924829850f,
	0.774953106594873930f, -0.632018735939809060f, 0.774831904274396850f,
	-0.632167319717265920f,
	0.774710673465565550f, -0.632315880251737570f, 0.774589414172837550f,
	-0.632464417537761840f,
	0.774468126400670860f, -0.632612931569877410f, 0.774346810153525130f,
	-0.632761422342624000f,
	0.774225465435860680f, -0.632909889850541750f, 0.774104092252139050f,
	-0.633058334088172140f,
	0.773982690606822900f, -0.633206755050057190f, 0.773861260504375540f,
	-0.633355152730739950f,
	0.773739801949261840f, -0.633503527124764320f, 0.773618314945947460f,
	-0.633651878226674900f,
	0.773496799498899050f, -0.633800206031017280f, 0.773375255612584470f,
	-0.633948510532337810f,
	0.773253683291472590f, -0.634096791725183740f, 0.773132082540033070f,
	-0.634245049604103330f,
	0.773010453362736990f, -0.634393284163645490f, 0.772888795764056220f,
	-0.634541495398360020f,
	0.772767109748463850f, -0.634689683302797740f, 0.772645395320433860f,
	-0.634837847871509990f,
	0.772523652484441330f, -0.634985989099049460f, 0.772401881244962450f,
	-0.635134106979969190f,
	0.772280081606474320f, -0.635282201508823420f, 0.772158253573455240f,
	-0.635430272680167160f,
	0.772036397150384520f, -0.635578320488556110f, 0.771914512341742350f,
	-0.635726344928547070f,
	0.771792599152010150f, -0.635874345994697720f, 0.771670657585670330f,
	-0.636022323681566300f,
	0.771548687647206300f, -0.636170277983712170f, 0.771426689341102590f,
	-0.636318208895695460f,
	0.771304662671844830f, -0.636466116412077180f, 0.771182607643919330f,
	-0.636614000527419120f,
	0.771060524261813820f, -0.636761861236284200f, 0.770938412530016940f,
	-0.636909698533235870f,
	0.770816272453018540f, -0.637057512412838590f, 0.770694104035309140f,
	-0.637205302869657600f,
	0.770571907281380810f, -0.637353069898259130f, 0.770449682195725960f,
	-0.637500813493210190f,
	0.770327428782838890f, -0.637648533649078810f, 0.770205147047214210f,
	-0.637796230360433540f,
	0.770082836993347900f, -0.637943903621844060f, 0.769960498625737230f,
	-0.638091553427880820f,
	0.769838131948879840f, -0.638239179773115280f, 0.769715736967275130f,
	-0.638386782652119570f,
	0.769593313685422940f, -0.638534362059466790f, 0.769470862107824670f,
	-0.638681917989730730f,
	0.769348382238982280f, -0.638829450437486290f, 0.769225874083399260f,
	-0.638976959397309140f,
	0.769103337645579700f, -0.639124444863775730f, 0.768980772930028870f,
	-0.639271906831463510f,
	0.768858179941253270f, -0.639419345294950700f, 0.768735558683760310f,
	-0.639566760248816310f,
	0.768612909162058380f, -0.639714151687640450f, 0.768490231380656860f,
	-0.639861519606003900f,
	0.768367525344066270f, -0.640008863998488440f, 0.768244791056798330f,
	-0.640156184859676510f,
	0.768122028523365420f, -0.640303482184151670f, 0.767999237748281270f,
	-0.640450755966498140f,
	0.767876418736060610f, -0.640598006201301030f, 0.767753571491219030f,
	-0.640745232883146440f,
	0.767630696018273380f, -0.640892436006621380f, 0.767507792321741270f,
	-0.641039615566313390f,
	0.767384860406141730f, -0.641186771556811250f, 0.767261900275994500f,
	-0.641333903972704290f,
	0.767138911935820400f, -0.641481012808583160f, 0.767015895390141480f,
	-0.641628098059038750f,
	0.766892850643480670f, -0.641775159718663500f, 0.766769777700361920f,
	-0.641922197782050170f,
	0.766646676565310380f, -0.642069212243792540f, 0.766523547242852210f,
	-0.642216203098485370f,
	0.766400389737514230f, -0.642363170340724320f, 0.766277204053824710f,
	-0.642510113965105710f,
	0.766153990196312920f, -0.642657033966226860f, 0.766030748169509000f,
	-0.642803930338685990f,
	0.765907477977944340f, -0.642950803077082080f, 0.765784179626150970f,
	-0.643097652176015110f,
	0.765660853118662500f, -0.643244477630085850f, 0.765537498460013070f,
	-0.643391279433895850f,
	0.765414115654738270f, -0.643538057582047740f, 0.765290704707374370f,
	-0.643684812069144850f,
	0.765167265622458960f, -0.643831542889791390f, 0.765043798404530520f,
	-0.643978250038592660f,
	0.764920303058128410f, -0.644124933510154540f, 0.764796779587793460f,
	-0.644271593299083790f,
	0.764673227998067140f, -0.644418229399988380f, 0.764549648293492150f,
	-0.644564841807476640f,
	0.764426040478612070f, -0.644711430516158310f, 0.764302404557971720f,
	-0.644857995520643710f,
	0.764178740536116670f, -0.645004536815543930f, 0.764055048417593970f,
	-0.645151054395471160f,
	0.763931328206951090f, -0.645297548255038380f, 0.763807579908737160f,
	-0.645444018388859230f,
	0.763683803527501870f, -0.645590464791548690f, 0.763559999067796150f,
	-0.645736887457722290f,
	0.763436166534172010f, -0.645883286381996320f, 0.763312305931182380f,
	-0.646029661558988330f,
	0.763188417263381270f, -0.646176012983316280f, 0.763064500535323710f,
	-0.646322340649599480f,
	0.762940555751565720f, -0.646468644552457780f, 0.762816582916664430f,
	-0.646614924686512050f,
	0.762692582035177980f, -0.646761181046383920f, 0.762568553111665380f,
	-0.646907413626696020f,
	0.762444496150687210f, -0.647053622422071540f, 0.762320411156804270f,
	-0.647199807427135230f,
	0.762196298134578900f, -0.647345968636512060f, 0.762072157088574560f,
	-0.647492106044828100f,
	0.761947988023355390f, -0.647638219646710310f, 0.761823790943486960f,
	-0.647784309436786440f,
	0.761699565853535380f, -0.647930375409685340f, 0.761575312758068000f,
	-0.648076417560036530f,
	0.761451031661653620f, -0.648222435882470420f, 0.761326722568861360f,
	-0.648368430371618290f,
	0.761202385484261780f, -0.648514401022112440f, 0.761078020412426560f,
	-0.648660347828585840f,
	0.760953627357928150f, -0.648806270785672550f, 0.760829206325340010f,
	-0.648952169888007300f,
	0.760704757319236920f, -0.649098045130225950f, 0.760580280344194450f,
	-0.649243896506964900f,
	0.760455775404789260f, -0.649389724012861660f, 0.760331242505599030f,
	-0.649535527642554730f,
	0.760206681651202420f, -0.649681307390683190f, 0.760082092846179340f,
	-0.649827063251887100f,
	0.759957476095110330f, -0.649972795220807530f, 0.759832831402577400f,
	-0.650118503292086200f,
	0.759708158773163440f, -0.650264187460365850f, 0.759583458211452010f,
	-0.650409847720290310f,
	0.759458729722028210f, -0.650555484066503880f, 0.759333973309477940f,
	-0.650701096493652040f,
	0.759209188978388070f, -0.650846684996380880f, 0.759084376733346610f,
	-0.650992249569337660f,
	0.758959536578942440f, -0.651137790207170330f, 0.758834668519765660f,
	-0.651283306904527740f,
	0.758709772560407390f, -0.651428799656059820f, 0.758584848705459610f,
	-0.651574268456416970f,
	0.758459896959515430f, -0.651719713300250910f, 0.758334917327168960f,
	-0.651865134182213920f,
	0.758209909813015280f, -0.652010531096959500f, 0.758084874421650730f,
	-0.652155904039141590f,
	0.757959811157672300f, -0.652301253003415460f, 0.757834720025678310f,
	-0.652446577984436730f,
	0.757709601030268080f, -0.652591878976862440f, 0.757584454176041810f,
	-0.652737155975350310f,
	0.757459279467600720f, -0.652882408974558850f, 0.757334076909547130f,
	-0.653027637969147530f,
	0.757208846506484570f, -0.653172842953776760f, 0.757083588263017140f,
	-0.653318023923107670f,
	0.756958302183750490f, -0.653463180871802330f, 0.756832988273290820f,
	-0.653608313794523890f,
	0.756707646536245670f, -0.653753422685936060f, 0.756582276977223470f,
	-0.653898507540703780f,
	0.756456879600833740f, -0.654043568353492640f, 0.756331454411686920f,
	-0.654188605118969040f,
	0.756206001414394540f, -0.654333617831800440f, 0.756080520613569120f,
	-0.654478606486655350f,
	0.755955012013824420f, -0.654623571078202680f, 0.755829475619774760f,
	-0.654768511601112600f,
	0.755703911436035880f, -0.654913428050056030f, 0.755578319467224540f,
	-0.655058320419704910f,
	0.755452699717958250f, -0.655203188704731820f, 0.755327052192855670f,
	-0.655348032899810470f,
	0.755201376896536550f, -0.655492852999615350f, 0.755075673833621620f,
	-0.655637648998821820f,
	0.754949943008732640f, -0.655782420892106030f, 0.754824184426492350f,
	-0.655927168674145360f,
	0.754698398091524500f, -0.656071892339617600f, 0.754572584008453840f,
	-0.656216591883201920f,
	0.754446742181906440f, -0.656361267299578000f, 0.754320872616508820f,
	-0.656505918583426550f,
	0.754194975316889170f, -0.656650545729428940f, 0.754069050287676120f,
	-0.656795148732268070f,
	0.753943097533499640f, -0.656939727586627110f, 0.753817117058990790f,
	-0.657084282287190180f,
	0.753691108868781210f, -0.657228812828642540f, 0.753565072967504300f,
	-0.657373319205670210f,
	0.753439009359793580f, -0.657517801412960120f, 0.753312918050284330f,
	-0.657662259445200070f,
	0.753186799043612520f, -0.657806693297078640f, 0.753060652344415100f,
	-0.657951102963285520f,
	0.752934477957330150f, -0.658095488438511180f, 0.752808275886996950f,
	-0.658239849717446870f,
	0.752682046138055340f, -0.658384186794785050f, 0.752555788715146390f,
	-0.658528499665218650f,
	0.752429503622912390f, -0.658672788323441890f, 0.752303190865996400f,
	-0.658817052764149480f,
	0.752176850449042810f, -0.658961292982037320f, 0.752050482376696360f,
	-0.659105508971802090f,
	0.751924086653603550f, -0.659249700728141490f, 0.751797663284411550f,
	-0.659393868245753860f,
	0.751671212273768430f, -0.659538011519338660f, 0.751544733626323680f,
	-0.659682130543596150f,
	0.751418227346727470f, -0.659826225313227320f, 0.751291693439630870f,
	-0.659970295822934540f,
	0.751165131909686480f, -0.660114342067420480f, 0.751038542761547360f,
	-0.660258364041389050f,
	0.750911925999867890f, -0.660402361739545030f, 0.750785281629303690f,
	-0.660546335156593890f,
	0.750658609654510700f, -0.660690284287242300f, 0.750531910080146410f,
	-0.660834209126197610f,
	0.750405182910869330f, -0.660978109668168060f, 0.750278428151338720f,
	-0.661121985907862860f,
	0.750151645806215070f, -0.661265837839992270f, 0.750024835880159780f,
	-0.661409665459266940f,
	0.749897998377835330f, -0.661553468760398890f, 0.749771133303905100f,
	-0.661697247738101010f,
	0.749644240663033480f, -0.661841002387086870f, 0.749517320459886170f,
	-0.661984732702070920f,
	0.749390372699129560f, -0.662128438677768720f, 0.749263397385431130f,
	-0.662272120308896590f,
	0.749136394523459370f, -0.662415777590171780f, 0.749009364117883880f,
	-0.662559410516312290f,
	0.748882306173375150f, -0.662703019082037440f, 0.748755220694604760f,
	-0.662846603282066900f,
	0.748628107686245440f, -0.662990163111121470f, 0.748500967152970430f,
	-0.663133698563923010f,
	0.748373799099454560f, -0.663277209635194100f, 0.748246603530373420f,
	-0.663420696319658280f,
	0.748119380450403600f, -0.663564158612039770f, 0.747992129864222700f,
	-0.663707596507064010f,
	0.747864851776509410f, -0.663851009999457340f, 0.747737546191943330f,
	-0.663994399083946640f,
	0.747610213115205150f, -0.664137763755260010f, 0.747482852550976570f,
	-0.664281104008126230f,
	0.747355464503940190f, -0.664424419837275180f, 0.747228048978779920f,
	-0.664567711237437520f,
	0.747100605980180130f, -0.664710978203344790f, 0.746973135512826850f,
	-0.664854220729729660f,
	0.746845637581406540f, -0.664997438811325340f, 0.746718112190607130f,
	-0.665140632442866140f,
	0.746590559345117310f, -0.665283801619087180f, 0.746462979049626770f,
	-0.665426946334724660f,
	0.746335371308826320f, -0.665570066584515450f, 0.746207736127407760f,
	-0.665713162363197550f,
	0.746080073510063780f, -0.665856233665509720f, 0.745952383461488290f,
	-0.665999280486191500f,
	0.745824665986376090f, -0.666142302819983540f, 0.745696921089422760f,
	-0.666285300661627280f,
	0.745569148775325430f, -0.666428274005865240f, 0.745441349048781680f,
	-0.666571222847440640f,
	0.745313521914490520f, -0.666714147181097670f, 0.745185667377151640f,
	-0.666857047001581220f,
	0.745057785441466060f, -0.666999922303637470f, 0.744929876112135350f,
	-0.667142773082013310f,
	0.744801939393862630f, -0.667285599331456370f, 0.744673975291351710f,
	-0.667428401046715520f,
	0.744545983809307370f, -0.667571178222540310f, 0.744417964952435620f,
	-0.667713930853681030f,
	0.744289918725443260f, -0.667856658934889320f, 0.744161845133038180f,
	-0.667999362460917400f,
	0.744033744179929290f, -0.668142041426518450f, 0.743905615870826490f,
	-0.668284695826446670f,
	0.743777460210440890f, -0.668427325655456820f, 0.743649277203484060f,
	-0.668569930908304970f,
	0.743521066854669120f, -0.668712511579747980f, 0.743392829168709970f,
	-0.668855067664543610f,
	0.743264564150321600f, -0.668997599157450270f, 0.743136271804219820f,
	-0.669140106053227600f,
	0.743007952135121720f, -0.669282588346636010f, 0.742879605147745200f,
	-0.669425046032436910f,
	0.742751230846809050f, -0.669567479105392490f, 0.742622829237033490f,
	-0.669709887560265840f,
	0.742494400323139180f, -0.669852271391821020f, 0.742365944109848460f,
	-0.669994630594823000f,
	0.742237460601884000f, -0.670136965164037650f, 0.742108949803969910f,
	-0.670279275094231800f,
	0.741980411720831070f, -0.670421560380173090f, 0.741851846357193480f,
	-0.670563821016630040f,
	0.741723253717784140f, -0.670706056998372160f, 0.741594633807331150f,
	-0.670848268320169640f,
	0.741465986630563290f, -0.670990454976794220f, 0.741337312192210660f,
	-0.671132616963017740f,
	0.741208610497004260f, -0.671274754273613490f, 0.741079881549676080f,
	-0.671416866903355450f,
	0.740951125354959110f, -0.671558954847018330f, 0.740822341917587330f,
	-0.671701018099378320f,
	0.740693531242295760f, -0.671843056655211930f, 0.740564693333820250f,
	-0.671985070509296900f,
	0.740435828196898020f, -0.672127059656411730f, 0.740306935836266940f,
	-0.672269024091335930f,
	0.740178016256666240f, -0.672410963808849790f, 0.740049069462835550f,
	-0.672552878803734710f,
	0.739920095459516200f, -0.672694769070772860f, 0.739791094251449950f,
	-0.672836634604747300f,
	0.739662065843380010f, -0.672978475400442090f, 0.739533010240050250f,
	-0.673120291452642070f,
	0.739403927446205760f, -0.673262082756132970f, 0.739274817466592520f,
	-0.673403849305701740f,
	0.739145680305957510f, -0.673545591096136100f, 0.739016515969048720f,
	-0.673687308122224330f,
	0.738887324460615110f, -0.673829000378756040f, 0.738758105785406900f,
	-0.673970667860521620f,
	0.738628859948174840f, -0.674112310562312360f, 0.738499586953671130f,
	-0.674253928478920410f,
	0.738370286806648620f, -0.674395521605139050f, 0.738240959511861310f,
	-0.674537089935762000f,
	0.738111605074064260f, -0.674678633465584540f, 0.737982223498013570f,
	-0.674820152189402170f,
	0.737852814788465980f, -0.674961646102011930f, 0.737723378950179700f,
	-0.675103115198211420f,
	0.737593915987913570f, -0.675244559472799270f, 0.737464425906427580f,
	-0.675385978920574840f,
	0.737334908710482910f, -0.675527373536338520f, 0.737205364404841190f,
	-0.675668743314891910f,
	0.737075792994265730f, -0.675810088251036940f, 0.736946194483520280f,
	-0.675951408339577010f,
	0.736816568877369900f, -0.676092703575315920f, 0.736686916180580460f,
	-0.676233973953058950f,
	0.736557236397919150f, -0.676375219467611590f, 0.736427529534153690f,
	-0.676516440113781090f,
	0.736297795594053170f, -0.676657635886374950f, 0.736168034582387330f,
	-0.676798806780201770f,
	0.736038246503927350f, -0.676939952790071130f, 0.735908431363445190f,
	-0.677081073910793530f,
	0.735778589165713590f, -0.677222170137180330f, 0.735648719915506510f,
	-0.677363241464043920f,
	0.735518823617598900f, -0.677504287886197430f, 0.735388900276766730f,
	-0.677645309398454910f,
	0.735258949897786840f, -0.677786305995631500f, 0.735128972485437180f,
	-0.677927277672543020f,
	0.734998968044496710f, -0.678068224424006600f, 0.734868936579745170f,
	-0.678209146244839860f,
	0.734738878095963500f, -0.678350043129861470f, 0.734608792597933550f,
	-0.678490915073891140f,
	0.734478680090438370f, -0.678631762071749360f, 0.734348540578261600f,
	-0.678772584118257690f,
	0.734218374066188280f, -0.678913381208238410f, 0.734088180559004040f,
	-0.679054153336514870f,
	0.733957960061495940f, -0.679194900497911200f, 0.733827712578451700f,
	-0.679335622687252560f,
	0.733697438114660370f, -0.679476319899364970f, 0.733567136674911360f,
	-0.679616992129075560f,
	0.733436808263995710f, -0.679757639371212030f, 0.733306452886705260f,
	-0.679898261620603290f,
	0.733176070547832740f, -0.680038858872078930f, 0.733045661252172080f,
	-0.680179431120469750f,
	0.732915225004517780f, -0.680319978360607200f, 0.732784761809665790f,
	-0.680460500587323880f,
	0.732654271672412820f, -0.680600997795453020f, 0.732523754597556700f,
	-0.680741469979829090f,
	0.732393210589896040f, -0.680881917135287230f, 0.732262639654230770f,
	-0.681022339256663670f,
	0.732132041795361290f, -0.681162736338795430f, 0.732001417018089630f,
	-0.681303108376520530f,
	0.731870765327218290f, -0.681443455364677870f, 0.731740086727550980f,
	-0.681583777298107480f,
	0.731609381223892630f, -0.681724074171649710f, 0.731478648821048520f,
	-0.681864345980146670f,
	0.731347889523825570f, -0.682004592718440830f, 0.731217103337031270f,
	-0.682144814381375640f,
	0.731086290265474340f, -0.682285010963795570f, 0.730955450313964360f,
	-0.682425182460546060f,
	0.730824583487312160f, -0.682565328866473250f, 0.730693689790329000f,
	-0.682705450176424590f,
	0.730562769227827590f, -0.682845546385248080f, 0.730431821804621520f,
	-0.682985617487792740f,
	0.730300847525525490f, -0.683125663478908680f, 0.730169846395354870f,
	-0.683265684353446700f,
	0.730038818418926260f, -0.683405680106258680f, 0.729907763601057140f,
	-0.683545650732197530f,
	0.729776681946566090f, -0.683685596226116580f, 0.729645573460272480f,
	-0.683825516582870720f,
	0.729514438146997010f, -0.683965411797315400f, 0.729383276011561050f,
	-0.684105281864307080f,
	0.729252087058786970f, -0.684245126778703080f, 0.729120871293498230f,
	-0.684384946535361750f,
	0.728989628720519420f, -0.684524741129142300f, 0.728858359344675800f,
	-0.684664510554904960f,
	0.728727063170793830f, -0.684804254807510620f, 0.728595740203700770f,
	-0.684943973881821490f,
	0.728464390448225200f, -0.685083667772700360f, 0.728333013909196360f,
	-0.685223336475011210f,
	0.728201610591444610f, -0.685362979983618730f, 0.728070180499801210f,
	-0.685502598293388550f,
	0.727938723639098620f, -0.685642191399187470f, 0.727807240014169960f,
	-0.685781759295883030f,
	0.727675729629849610f, -0.685921301978343560f, 0.727544192490972800f,
	-0.686060819441438710f,
	0.727412628602375770f, -0.686200311680038590f, 0.727281037968895870f,
	-0.686339778689014520f,
	0.727149420595371020f, -0.686479220463238950f, 0.727017776486640680f,
	-0.686618636997584630f,
	0.726886105647544970f, -0.686758028286925890f, 0.726754408082925020f,
	-0.686897394326137610f,
	0.726622683797622850f, -0.687036735110095660f, 0.726490932796481910f,
	-0.687176050633676820f,
	0.726359155084346010f, -0.687315340891759050f, 0.726227350666060370f,
	-0.687454605879221030f,
	0.726095519546471000f, -0.687593845590942170f, 0.725963661730424930f,
	-0.687733060021803230f,
	0.725831777222770370f, -0.687872249166685550f, 0.725699866028356120f,
	-0.688011413020471640f,
	0.725567928152032300f, -0.688150551578044830f, 0.725435963598649810f,
	-0.688289664834289330f,
	0.725303972373060770f, -0.688428752784090440f, 0.725171954480117950f,
	-0.688567815422334250f,
	0.725039909924675370f, -0.688706852743907750f, 0.724907838711587820f,
	-0.688845864743699020f,
	0.724775740845711280f, -0.688984851416597040f, 0.724643616331902550f,
	-0.689123812757491570f,
	0.724511465175019630f, -0.689262748761273470f, 0.724379287379921190f,
	-0.689401659422834270f,
	0.724247082951467000f, -0.689540544737066830f, 0.724114851894517850f,
	-0.689679404698864800f,
	0.723982594213935520f, -0.689818239303122470f, 0.723850309914582880f,
	-0.689957048544735390f,
	0.723717999001323500f, -0.690095832418599950f, 0.723585661479022150f,
	-0.690234590919613370f,
	0.723453297352544380f, -0.690373324042674040f, 0.723320906626756970f,
	-0.690512031782681060f,
	0.723188489306527460f, -0.690650714134534600f, 0.723056045396724410f,
	-0.690789371093135650f,
	0.722923574902217700f, -0.690928002653386160f, 0.722791077827877550f,
	-0.691066608810189220f,
	0.722658554178575610f, -0.691205189558448450f, 0.722526003959184540f,
	-0.691343744893068710f,
	0.722393427174577550f, -0.691482274808955850f, 0.722260823829629310f,
	-0.691620779301016290f,
	0.722128193929215350f, -0.691759258364157750f, 0.721995537478211880f,
	-0.691897711993288760f,
	0.721862854481496340f, -0.692036140183318720f, 0.721730144943947160f,
	-0.692174542929158140f,
	0.721597408870443770f, -0.692312920225718220f, 0.721464646265866370f,
	-0.692451272067911130f,
	0.721331857135096290f, -0.692589598450650380f, 0.721199041483015720f,
	-0.692727899368849820f,
	0.721066199314508110f, -0.692866174817424630f, 0.720933330634457530f,
	-0.693004424791290870f,
	0.720800435447749190f, -0.693142649285365400f, 0.720667513759269520f,
	-0.693280848294566040f,
	0.720534565573905270f, -0.693419021813811760f, 0.720401590896544760f,
	-0.693557169838022290f,
	0.720268589732077190f, -0.693695292362118240f, 0.720135562085392420f,
	-0.693833389381021350f,
	0.720002507961381650f, -0.693971460889654000f, 0.719869427364936860f,
	-0.694109506882939820f,
	0.719736320300951030f, -0.694247527355803310f, 0.719603186774318120f,
	-0.694385522303169740f,
	0.719470026789932990f, -0.694523491719965520f, 0.719336840352691740f,
	-0.694661435601117820f,
	0.719203627467491220f, -0.694799353941554900f, 0.719070388139229190f,
	-0.694937246736205830f,
	0.718937122372804490f, -0.695075113980000880f, 0.718803830173116890f,
	-0.695212955667870780f,
	0.718670511545067230f, -0.695350771794747690f, 0.718537166493557370f,
	-0.695488562355564440f,
	0.718403795023489830f, -0.695626327345254870f, 0.718270397139768260f,
	-0.695764066758753690f,
	0.718136972847297490f, -0.695901780590996830f, 0.718003522150983180f,
	-0.696039468836920690f,
	0.717870045055731710f, -0.696177131491462990f, 0.717736541566450950f,
	-0.696314768549562090f,
	0.717603011688049080f, -0.696452380006157830f, 0.717469455425435830f,
	-0.696589965856190370f,
	0.717335872783521730f, -0.696727526094601200f, 0.717202263767218070f,
	-0.696865060716332470f,
	0.717068628381437480f, -0.697002569716327460f, 0.716934966631093130f,
	-0.697140053089530420f,
	0.716801278521099540f, -0.697277510830886520f, 0.716667564056371890f,
	-0.697414942935341790f,
	0.716533823241826680f, -0.697552349397843160f, 0.716400056082381000f,
	-0.697689730213338800f,
	0.716266262582953120f, -0.697827085376777290f, 0.716132442748462330f,
	-0.697964414883108670f,
	0.715998596583828690f, -0.698101718727283770f, 0.715864724093973500f,
	-0.698238996904254280f,
	0.715730825283818590f, -0.698376249408972920f, 0.715596900158287470f,
	-0.698513476236393040f,
	0.715462948722303760f, -0.698650677381469460f, 0.715328970980792620f,
	-0.698787852839157670f,
	0.715194966938680120f, -0.698925002604414150f, 0.715060936600893090f,
	-0.699062126672196140f,
	0.714926879972359490f, -0.699199225037462120f, 0.714792797058008240f,
	-0.699336297695171140f,
	0.714658687862769090f, -0.699473344640283770f, 0.714524552391572860f,
	-0.699610365867761040f,
	0.714390390649351390f, -0.699747361372564990f, 0.714256202641037510f,
	-0.699884331149658760f,
	0.714121988371564820f, -0.700021275194006250f, 0.713987747845867830f,
	-0.700158193500572730f,
	0.713853481068882470f, -0.700295086064323780f, 0.713719188045545240f,
	-0.700431952880226420f,
	0.713584868780793640f, -0.700568793943248340f, 0.713450523279566260f,
	-0.700705609248358450f,
	0.713316151546802610f, -0.700842398790526120f, 0.713181753587443180f,
	-0.700979162564722370f,
	0.713047329406429340f, -0.701115900565918660f, 0.712912879008703480f,
	-0.701252612789087460f,
	0.712778402399208980f, -0.701389299229202230f, 0.712643899582890210f,
	-0.701525959881237340f,
	0.712509370564692320f, -0.701662594740168450f, 0.712374815349561710f,
	-0.701799203800971720f,
	0.712240233942445510f, -0.701935787058624360f, 0.712105626348291890f,
	-0.702072344508104630f,
	0.711970992572050100f, -0.702208876144391870f, 0.711836332618670080f,
	-0.702345381962465880f,
	0.711701646493102970f, -0.702481861957308000f, 0.711566934200300700f,
	-0.702618316123900130f,
	0.711432195745216430f, -0.702754744457225300f, 0.711297431132803970f,
	-0.702891146952267400f,
	0.711162640368018350f, -0.703027523604011220f, 0.711027823455815280f,
	-0.703163874407442770f,
	0.710892980401151680f, -0.703300199357548730f, 0.710758111208985350f,
	-0.703436498449316660f,
	0.710623215884275020f, -0.703572771677735580f, 0.710488294431980470f,
	-0.703709019037794810f,
	0.710353346857062420f, -0.703845240524484940f, 0.710218373164482220f,
	-0.703981436132797620f,
	0.710083373359202800f, -0.704117605857725310f, 0.709948347446187400f,
	-0.704253749694261470f,
	0.709813295430400840f, -0.704389867637400410f, 0.709678217316808580f,
	-0.704525959682137380f,
	0.709543113110376770f, -0.704662025823468820f, 0.709407982816072980f,
	-0.704798066056391950f,
	0.709272826438865690f, -0.704934080375904880f, 0.709137643983724030f,
	-0.705070068777006840f,
	0.709002435455618250f, -0.705206031254697830f, 0.708867200859519820f,
	-0.705341967803978840f,
	0.708731940200400650f, -0.705477878419852100f, 0.708596653483234080f,
	-0.705613763097320490f,
	0.708461340712994160f, -0.705749621831387790f, 0.708326001894655890f,
	-0.705885454617058980f,
	0.708190637033195400f, -0.706021261449339740f, 0.708055246133589500f,
	-0.706157042323237060f,
	0.707919829200816310f, -0.706292797233758480f, 0.707784386239854620f,
	-0.706428526175912790f,
	0.707648917255684350f, -0.706564229144709510f, 0.707513422253286280f,
	-0.706699906135159430f,
	0.707377901237642100f, -0.706835557142273750f, 0.707242354213734710f,
	-0.706971182161065360f,
	0.707106781186547570f, -0.707106781186547460f, 0.706971182161065360f,
	-0.707242354213734600f,
	0.706835557142273860f, -0.707377901237642100f, 0.706699906135159430f,
	-0.707513422253286170f,
	0.706564229144709620f, -0.707648917255684350f, 0.706428526175912790f,
	-0.707784386239854620f,
	0.706292797233758480f, -0.707919829200816310f, 0.706157042323237060f,
	-0.708055246133589500f,
	0.706021261449339740f, -0.708190637033195290f, 0.705885454617058980f,
	-0.708326001894655780f,
	0.705749621831387790f, -0.708461340712994050f, 0.705613763097320490f,
	-0.708596653483234080f,
	0.705477878419852210f, -0.708731940200400650f, 0.705341967803978950f,
	-0.708867200859519820f,
	0.705206031254697830f, -0.709002435455618250f, 0.705070068777006840f,
	-0.709137643983723920f,
	0.704934080375904990f, -0.709272826438865580f, 0.704798066056391950f,
	-0.709407982816072980f,
	0.704662025823468930f, -0.709543113110376770f, 0.704525959682137380f,
	-0.709678217316808470f,
	0.704389867637400410f, -0.709813295430400840f, 0.704253749694261580f,
	-0.709948347446187400f,
	0.704117605857725430f, -0.710083373359202690f, 0.703981436132797730f,
	-0.710218373164482220f,
	0.703845240524484940f, -0.710353346857062310f, 0.703709019037794810f,
	-0.710488294431980470f,
	0.703572771677735580f, -0.710623215884275020f, 0.703436498449316770f,
	-0.710758111208985350f,
	0.703300199357548730f, -0.710892980401151680f, 0.703163874407442770f,
	-0.711027823455815280f,
	0.703027523604011220f, -0.711162640368018350f, 0.702891146952267400f,
	-0.711297431132803970f,
	0.702754744457225300f, -0.711432195745216430f, 0.702618316123900130f,
	-0.711566934200300700f,
	0.702481861957308000f, -0.711701646493102970f, 0.702345381962465880f,
	-0.711836332618670080f,
	0.702208876144391870f, -0.711970992572049990f, 0.702072344508104740f,
	-0.712105626348291890f,
	0.701935787058624360f, -0.712240233942445510f, 0.701799203800971720f,
	-0.712374815349561710f,
	0.701662594740168570f, -0.712509370564692320f, 0.701525959881237450f,
	-0.712643899582890210f,
	0.701389299229202230f, -0.712778402399208870f, 0.701252612789087460f,
	-0.712912879008703370f,
	0.701115900565918660f, -0.713047329406429230f, 0.700979162564722480f,
	-0.713181753587443070f,
	0.700842398790526230f, -0.713316151546802610f, 0.700705609248358450f,
	-0.713450523279566150f,
	0.700568793943248450f, -0.713584868780793520f, 0.700431952880226420f,
	-0.713719188045545130f,
	0.700295086064323780f, -0.713853481068882470f, 0.700158193500572730f,
	-0.713987747845867830f,
	0.700021275194006360f, -0.714121988371564710f, 0.699884331149658760f,
	-0.714256202641037400f,
	0.699747361372564990f, -0.714390390649351390f, 0.699610365867761040f,
	-0.714524552391572860f,
	0.699473344640283770f, -0.714658687862768980f, 0.699336297695171250f,
	-0.714792797058008130f,
	0.699199225037462120f, -0.714926879972359370f, 0.699062126672196140f,
	-0.715060936600892980f,
	0.698925002604414150f, -0.715194966938680010f, 0.698787852839157790f,
	-0.715328970980792620f,
	0.698650677381469580f, -0.715462948722303650f, 0.698513476236393040f,
	-0.715596900158287360f,
	0.698376249408972920f, -0.715730825283818590f, 0.698238996904254390f,
	-0.715864724093973390f,
	0.698101718727283880f, -0.715998596583828690f, 0.697964414883108790f,
	-0.716132442748462330f,
	0.697827085376777290f, -0.716266262582953120f, 0.697689730213338800f,
	-0.716400056082380890f,
	0.697552349397843270f, -0.716533823241826570f, 0.697414942935341790f,
	-0.716667564056371890f,
	0.697277510830886630f, -0.716801278521099540f, 0.697140053089530530f,
	-0.716934966631093130f,
	0.697002569716327460f, -0.717068628381437480f, 0.696865060716332470f,
	-0.717202263767218070f,
	0.696727526094601200f, -0.717335872783521730f, 0.696589965856190370f,
	-0.717469455425435830f,
	0.696452380006157830f, -0.717603011688049080f, 0.696314768549562200f,
	-0.717736541566450840f,
	0.696177131491462990f, -0.717870045055731710f, 0.696039468836920690f,
	-0.718003522150983060f,
	0.695901780590996830f, -0.718136972847297490f, 0.695764066758753800f,
	-0.718270397139768260f,
	0.695626327345254870f, -0.718403795023489720f, 0.695488562355564440f,
	-0.718537166493557370f,
	0.695350771794747800f, -0.718670511545067230f, 0.695212955667870890f,
	-0.718803830173116890f,
	0.695075113980000880f, -0.718937122372804380f, 0.694937246736205940f,
	-0.719070388139229190f,
	0.694799353941554900f, -0.719203627467491220f, 0.694661435601117930f,
	-0.719336840352691740f,
	0.694523491719965520f, -0.719470026789932990f, 0.694385522303169860f,
	-0.719603186774318000f,
	0.694247527355803310f, -0.719736320300951030f, 0.694109506882939820f,
	-0.719869427364936860f,
	0.693971460889654000f, -0.720002507961381650f, 0.693833389381021350f,
	-0.720135562085392310f,
	0.693695292362118350f, -0.720268589732077080f, 0.693557169838022400f,
	-0.720401590896544760f,
	0.693419021813811880f, -0.720534565573905270f, 0.693280848294566150f,
	-0.720667513759269410f,
	0.693142649285365510f, -0.720800435447749190f, 0.693004424791290870f,
	-0.720933330634457530f,
	0.692866174817424740f, -0.721066199314508110f, 0.692727899368849820f,
	-0.721199041483015720f,
	0.692589598450650380f, -0.721331857135096180f, 0.692451272067911240f,
	-0.721464646265866370f,
	0.692312920225718220f, -0.721597408870443660f, 0.692174542929158140f,
	-0.721730144943947160f,
	0.692036140183318830f, -0.721862854481496340f, 0.691897711993288760f,
	-0.721995537478211880f,
	0.691759258364157750f, -0.722128193929215350f, 0.691620779301016400f,
	-0.722260823829629310f,
	0.691482274808955850f, -0.722393427174577550f, 0.691343744893068820f,
	-0.722526003959184430f,
	0.691205189558448450f, -0.722658554178575610f, 0.691066608810189220f,
	-0.722791077827877550f,
	0.690928002653386280f, -0.722923574902217700f, 0.690789371093135760f,
	-0.723056045396724410f,
	0.690650714134534720f, -0.723188489306527350f, 0.690512031782681170f,
	-0.723320906626756850f,
	0.690373324042674040f, -0.723453297352544380f, 0.690234590919613370f,
	-0.723585661479022040f,
	0.690095832418599950f, -0.723717999001323390f, 0.689957048544735390f,
	-0.723850309914582880f,
	0.689818239303122470f, -0.723982594213935520f, 0.689679404698864800f,
	-0.724114851894517850f,
	0.689540544737066940f, -0.724247082951466890f, 0.689401659422834380f,
	-0.724379287379921080f,
	0.689262748761273470f, -0.724511465175019520f, 0.689123812757491680f,
	-0.724643616331902550f,
	0.688984851416597150f, -0.724775740845711280f, 0.688845864743699130f,
	-0.724907838711587820f,
	0.688706852743907750f, -0.725039909924675370f, 0.688567815422334360f,
	-0.725171954480117840f,
	0.688428752784090550f, -0.725303972373060660f, 0.688289664834289440f,
	-0.725435963598649810f,
	0.688150551578044830f, -0.725567928152032300f, 0.688011413020471640f,
	-0.725699866028356120f,
	0.687872249166685550f, -0.725831777222770370f, 0.687733060021803230f,
	-0.725963661730424930f,
	0.687593845590942170f, -0.726095519546470890f, 0.687454605879221030f,
	-0.726227350666060260f,
	0.687315340891759160f, -0.726359155084346010f, 0.687176050633676930f,
	-0.726490932796481910f,
	0.687036735110095660f, -0.726622683797622850f, 0.686897394326137610f,
	-0.726754408082924910f,
	0.686758028286925890f, -0.726886105647544970f, 0.686618636997584740f,
	-0.727017776486640680f,
	0.686479220463238950f, -0.727149420595371020f, 0.686339778689014630f,
	-0.727281037968895760f,
	0.686200311680038700f, -0.727412628602375770f, 0.686060819441438710f,
	-0.727544192490972800f,
	0.685921301978343670f, -0.727675729629849610f, 0.685781759295883030f,
	-0.727807240014169960f,
	0.685642191399187470f, -0.727938723639098620f, 0.685502598293388670f,
	-0.728070180499801210f,
	0.685362979983618730f, -0.728201610591444500f, 0.685223336475011210f,
	-0.728333013909196360f,
	0.685083667772700360f, -0.728464390448225200f, 0.684943973881821490f,
	-0.728595740203700770f,
	0.684804254807510620f, -0.728727063170793720f, 0.684664510554904960f,
	-0.728858359344675690f,
	0.684524741129142300f, -0.728989628720519310f, 0.684384946535361750f,
	-0.729120871293498230f,
	0.684245126778703080f, -0.729252087058786970f, 0.684105281864307080f,
	-0.729383276011561050f,
	0.683965411797315510f, -0.729514438146996900f, 0.683825516582870830f,
	-0.729645573460272480f,
	0.683685596226116690f, -0.729776681946565970f, 0.683545650732197530f,
	-0.729907763601057140f,
	0.683405680106258790f, -0.730038818418926150f, 0.683265684353446700f,
	-0.730169846395354870f,
	0.683125663478908800f, -0.730300847525525380f, 0.682985617487792850f,
	-0.730431821804621520f,
	0.682845546385248080f, -0.730562769227827590f, 0.682705450176424590f,
	-0.730693689790328890f,
	0.682565328866473250f, -0.730824583487312050f, 0.682425182460546060f,
	-0.730955450313964360f,
	0.682285010963795570f, -0.731086290265474230f, 0.682144814381375640f,
	-0.731217103337031160f,
	0.682004592718440830f, -0.731347889523825460f, 0.681864345980146780f,
	-0.731478648821048520f,
	0.681724074171649820f, -0.731609381223892520f, 0.681583777298107480f,
	-0.731740086727550980f,
	0.681443455364677990f, -0.731870765327218290f, 0.681303108376520530f,
	-0.732001417018089520f,
	0.681162736338795430f, -0.732132041795361290f, 0.681022339256663670f,
	-0.732262639654230660f,
	0.680881917135287340f, -0.732393210589896040f, 0.680741469979829090f,
	-0.732523754597556590f,
	0.680600997795453130f, -0.732654271672412820f, 0.680460500587323880f,
	-0.732784761809665790f,
	0.680319978360607200f, -0.732915225004517780f, 0.680179431120469750f,
	-0.733045661252171970f,
	0.680038858872079040f, -0.733176070547832740f, 0.679898261620603290f,
	-0.733306452886705260f,
	0.679757639371212030f, -0.733436808263995710f, 0.679616992129075560f,
	-0.733567136674911360f,
	0.679476319899365080f, -0.733697438114660260f, 0.679335622687252670f,
	-0.733827712578451700f,
	0.679194900497911200f, -0.733957960061495940f, 0.679054153336514870f,
	-0.734088180559004040f,
	0.678913381208238410f, -0.734218374066188170f, 0.678772584118257690f,
	-0.734348540578261600f,
	0.678631762071749470f, -0.734478680090438370f, 0.678490915073891250f,
	-0.734608792597933550f,
	0.678350043129861580f, -0.734738878095963390f, 0.678209146244839860f,
	-0.734868936579745060f,
	0.678068224424006600f, -0.734998968044496600f, 0.677927277672543130f,
	-0.735128972485437180f,
	0.677786305995631500f, -0.735258949897786730f, 0.677645309398454910f,
	-0.735388900276766620f,
	0.677504287886197430f, -0.735518823617598900f, 0.677363241464044030f,
	-0.735648719915506400f,
	0.677222170137180450f, -0.735778589165713480f, 0.677081073910793530f,
	-0.735908431363445190f,
	0.676939952790071240f, -0.736038246503927350f, 0.676798806780201770f,
	-0.736168034582387330f,
	0.676657635886374950f, -0.736297795594053060f, 0.676516440113781090f,
	-0.736427529534153690f,
	0.676375219467611700f, -0.736557236397919150f, 0.676233973953058950f,
	-0.736686916180580460f,
	0.676092703575316030f, -0.736816568877369790f, 0.675951408339577010f,
	-0.736946194483520170f,
	0.675810088251037060f, -0.737075792994265620f, 0.675668743314891910f,
	-0.737205364404841190f,
	0.675527373536338630f, -0.737334908710482790f, 0.675385978920574950f,
	-0.737464425906427580f,
	0.675244559472799270f, -0.737593915987913460f, 0.675103115198211530f,
	-0.737723378950179590f,
	0.674961646102012040f, -0.737852814788465980f, 0.674820152189402280f,
	-0.737982223498013570f,
	0.674678633465584540f, -0.738111605074064260f, 0.674537089935762110f,
	-0.738240959511861310f,
	0.674395521605139050f, -0.738370286806648510f, 0.674253928478920520f,
	-0.738499586953671130f,
	0.674112310562312360f, -0.738628859948174840f, 0.673970667860521620f,
	-0.738758105785406900f,
	0.673829000378756150f, -0.738887324460615110f, 0.673687308122224330f,
	-0.739016515969048600f,
	0.673545591096136100f, -0.739145680305957400f, 0.673403849305701850f,
	-0.739274817466592520f,
	0.673262082756132970f, -0.739403927446205760f, 0.673120291452642070f,
	-0.739533010240050250f,
	0.672978475400442090f, -0.739662065843379900f, 0.672836634604747410f,
	-0.739791094251449950f,
	0.672694769070772970f, -0.739920095459516090f, 0.672552878803734820f,
	-0.740049069462835550f,
	0.672410963808849900f, -0.740178016256666240f, 0.672269024091336040f,
	-0.740306935836266940f,
	0.672127059656411840f, -0.740435828196898020f, 0.671985070509296900f,
	-0.740564693333820250f,
	0.671843056655211930f, -0.740693531242295640f, 0.671701018099378320f,
	-0.740822341917587330f,
	0.671558954847018330f, -0.740951125354959110f, 0.671416866903355450f,
	-0.741079881549676080f,
	0.671274754273613490f, -0.741208610497004260f, 0.671132616963017850f,
	-0.741337312192210660f,
	0.670990454976794220f, -0.741465986630563290f, 0.670848268320169750f,
	-0.741594633807331150f,
	0.670706056998372160f, -0.741723253717784140f, 0.670563821016630040f,
	-0.741851846357193480f,
	0.670421560380173090f, -0.741980411720830960f, 0.670279275094231910f,
	-0.742108949803969800f,
	0.670136965164037760f, -0.742237460601884000f, 0.669994630594823000f,
	-0.742365944109848460f,
	0.669852271391821130f, -0.742494400323139180f, 0.669709887560265840f,
	-0.742622829237033380f,
	0.669567479105392490f, -0.742751230846809050f, 0.669425046032436910f,
	-0.742879605147745090f,
	0.669282588346636010f, -0.743007952135121720f, 0.669140106053227710f,
	-0.743136271804219820f,
	0.668997599157450270f, -0.743264564150321490f, 0.668855067664543610f,
	-0.743392829168709970f,
	0.668712511579748090f, -0.743521066854669120f, 0.668569930908305080f,
	-0.743649277203484060f,
	0.668427325655456820f, -0.743777460210440780f, 0.668284695826446670f,
	-0.743905615870826490f,
	0.668142041426518560f, -0.744033744179929180f, 0.667999362460917510f,
	-0.744161845133038070f,
	0.667856658934889440f, -0.744289918725443140f, 0.667713930853681140f,
	-0.744417964952435620f,
	0.667571178222540310f, -0.744545983809307250f, 0.667428401046715640f,
	-0.744673975291351600f,
	0.667285599331456480f, -0.744801939393862630f, 0.667142773082013310f,
	-0.744929876112135350f,
	0.666999922303637470f, -0.745057785441465950f, 0.666857047001581220f,
	-0.745185667377151640f,
	0.666714147181097670f, -0.745313521914490410f, 0.666571222847440750f,
	-0.745441349048781680f,
	0.666428274005865350f, -0.745569148775325430f, 0.666285300661627390f,
	-0.745696921089422760f,
	0.666142302819983540f, -0.745824665986375980f, 0.665999280486191500f,
	-0.745952383461488180f,
	0.665856233665509720f, -0.746080073510063780f, 0.665713162363197660f,
	-0.746207736127407650f,
	0.665570066584515560f, -0.746335371308826320f, 0.665426946334724660f,
	-0.746462979049626770f,
	0.665283801619087180f, -0.746590559345117310f, 0.665140632442866140f,
	-0.746718112190607020f,
	0.664997438811325340f, -0.746845637581406540f, 0.664854220729729660f,
	-0.746973135512826740f,
	0.664710978203344900f, -0.747100605980180130f, 0.664567711237437520f,
	-0.747228048978779920f,
	0.664424419837275180f, -0.747355464503940190f, 0.664281104008126230f,
	-0.747482852550976570f,
	0.664137763755260010f, -0.747610213115205150f, 0.663994399083946640f,
	-0.747737546191943330f,
	0.663851009999457340f, -0.747864851776509410f, 0.663707596507064120f,
	-0.747992129864222700f,
	0.663564158612039880f, -0.748119380450403490f, 0.663420696319658280f,
	-0.748246603530373420f,
	0.663277209635194100f, -0.748373799099454560f, 0.663133698563923010f,
	-0.748500967152970430f,
	0.662990163111121470f, -0.748628107686245330f, 0.662846603282066900f,
	-0.748755220694604760f,
	0.662703019082037440f, -0.748882306173375030f, 0.662559410516312400f,
	-0.749009364117883770f,
	0.662415777590171780f, -0.749136394523459260f, 0.662272120308896590f,
	-0.749263397385431020f,
	0.662128438677768720f, -0.749390372699129560f, 0.661984732702071030f,
	-0.749517320459886170f,
	0.661841002387086870f, -0.749644240663033480f, 0.661697247738101120f,
	-0.749771133303904990f,
	0.661553468760399000f, -0.749897998377835220f, 0.661409665459266940f,
	-0.750024835880159780f,
	0.661265837839992270f, -0.750151645806214960f, 0.661121985907862970f,
	-0.750278428151338610f,
	0.660978109668168060f, -0.750405182910869220f, 0.660834209126197610f,
	-0.750531910080146410f,
	0.660690284287242300f, -0.750658609654510590f, 0.660546335156593890f,
	-0.750785281629303580f,
	0.660402361739545030f, -0.750911925999867890f, 0.660258364041389050f,
	-0.751038542761547250f,
	0.660114342067420480f, -0.751165131909686370f, 0.659970295822934540f,
	-0.751291693439630870f,
	0.659826225313227430f, -0.751418227346727360f, 0.659682130543596150f,
	-0.751544733626323570f,
	0.659538011519338770f, -0.751671212273768430f, 0.659393868245753970f,
	-0.751797663284411440f,
	0.659249700728141490f, -0.751924086653603550f, 0.659105508971802200f,
	-0.752050482376696360f,
	0.658961292982037320f, -0.752176850449042700f, 0.658817052764149480f,
	-0.752303190865996400f,
	0.658672788323441890f, -0.752429503622912390f, 0.658528499665218760f,
	-0.752555788715146390f,
	0.658384186794785050f, -0.752682046138055230f, 0.658239849717446980f,
	-0.752808275886996950f,
	0.658095488438511290f, -0.752934477957330150f, 0.657951102963285630f,
	-0.753060652344415100f,
	0.657806693297078640f, -0.753186799043612410f, 0.657662259445200070f,
	-0.753312918050284330f,
	0.657517801412960120f, -0.753439009359793580f, 0.657373319205670210f,
	-0.753565072967504190f,
	0.657228812828642650f, -0.753691108868781210f, 0.657084282287190180f,
	-0.753817117058990680f,
	0.656939727586627110f, -0.753943097533499640f, 0.656795148732268070f,
	-0.754069050287676120f,
	0.656650545729429050f, -0.754194975316889170f, 0.656505918583426550f,
	-0.754320872616508820f,
	0.656361267299578000f, -0.754446742181906330f, 0.656216591883202030f,
	-0.754572584008453840f,
	0.656071892339617710f, -0.754698398091524390f, 0.655927168674145360f,
	-0.754824184426492240f,
	0.655782420892106030f, -0.754949943008732640f, 0.655637648998821820f,
	-0.755075673833621510f,
	0.655492852999615460f, -0.755201376896536550f, 0.655348032899810580f,
	-0.755327052192855560f,
	0.655203188704731930f, -0.755452699717958140f, 0.655058320419704910f,
	-0.755578319467224540f,
	0.654913428050056150f, -0.755703911436035880f, 0.654768511601112600f,
	-0.755829475619774760f,
	0.654623571078202680f, -0.755955012013824310f, 0.654478606486655350f,
	-0.756080520613569120f,
	0.654333617831800550f, -0.756206001414394540f, 0.654188605118969040f,
	-0.756331454411686920f,
	0.654043568353492640f, -0.756456879600833630f, 0.653898507540703890f,
	-0.756582276977223470f,
	0.653753422685936170f, -0.756707646536245670f, 0.653608313794523890f,
	-0.756832988273290820f,
	0.653463180871802330f, -0.756958302183750490f, 0.653318023923107670f,
	-0.757083588263017140f,
	0.653172842953776760f, -0.757208846506484460f, 0.653027637969147650f,
	-0.757334076909547130f,
	0.652882408974558960f, -0.757459279467600720f, 0.652737155975350420f,
	-0.757584454176041810f,
	0.652591878976862550f, -0.757709601030268080f, 0.652446577984436840f,
	-0.757834720025678310f,
	0.652301253003415460f, -0.757959811157672300f, 0.652155904039141700f,
	-0.758084874421650620f,
	0.652010531096959500f, -0.758209909813015280f, 0.651865134182214030f,
	-0.758334917327168960f,
	0.651719713300251020f, -0.758459896959515320f, 0.651574268456417080f,
	-0.758584848705459500f,
	0.651428799656059820f, -0.758709772560407390f, 0.651283306904527850f,
	-0.758834668519765660f,
	0.651137790207170330f, -0.758959536578942440f, 0.650992249569337660f,
	-0.759084376733346500f,
	0.650846684996380990f, -0.759209188978387960f, 0.650701096493652040f,
	-0.759333973309477940f,
	0.650555484066503990f, -0.759458729722028210f, 0.650409847720290420f,
	-0.759583458211452010f,
	0.650264187460365960f, -0.759708158773163440f, 0.650118503292086200f,
	-0.759832831402577400f,
	0.649972795220807530f, -0.759957476095110330f, 0.649827063251887100f,
	-0.760082092846179220f,
	0.649681307390683190f, -0.760206681651202420f, 0.649535527642554730f,
	-0.760331242505599030f,
	0.649389724012861770f, -0.760455775404789260f, 0.649243896506965010f,
	-0.760580280344194340f,
	0.649098045130226060f, -0.760704757319236920f, 0.648952169888007410f,
	-0.760829206325340010f,
	0.648806270785672550f, -0.760953627357928040f, 0.648660347828585840f,
	-0.761078020412426560f,
	0.648514401022112550f, -0.761202385484261780f, 0.648368430371618400f,
	-0.761326722568861250f,
	0.648222435882470420f, -0.761451031661653510f, 0.648076417560036530f,
	-0.761575312758068000f,
	0.647930375409685460f, -0.761699565853535270f, 0.647784309436786550f,
	-0.761823790943486840f,
	0.647638219646710420f, -0.761947988023355390f, 0.647492106044828100f,
	-0.762072157088574560f,
	0.647345968636512060f, -0.762196298134578900f, 0.647199807427135230f,
	-0.762320411156804160f,
	0.647053622422071650f, -0.762444496150687100f, 0.646907413626696020f,
	-0.762568553111665380f,
	0.646761181046383920f, -0.762692582035177870f, 0.646614924686512050f,
	-0.762816582916664320f,
	0.646468644552457890f, -0.762940555751565720f, 0.646322340649599590f,
	-0.763064500535323710f,
	0.646176012983316390f, -0.763188417263381270f, 0.646029661558988330f,
	-0.763312305931182380f,
	0.645883286381996440f, -0.763436166534172010f, 0.645736887457722290f,
	-0.763559999067796150f,
	0.645590464791548800f, -0.763683803527501870f, 0.645444018388859230f,
	-0.763807579908737160f,
	0.645297548255038380f, -0.763931328206951090f, 0.645151054395471270f,
	-0.764055048417593860f,
	0.645004536815544040f, -0.764178740536116670f, 0.644857995520643710f,
	-0.764302404557971720f,
	0.644711430516158420f, -0.764426040478612070f, 0.644564841807476750f,
	-0.764549648293492150f,
	0.644418229399988380f, -0.764673227998067140f, 0.644271593299083900f,
	-0.764796779587793460f,
	0.644124933510154540f, -0.764920303058128410f, 0.643978250038592660f,
	-0.765043798404530410f,
	0.643831542889791500f, -0.765167265622458960f, 0.643684812069144960f,
	-0.765290704707374260f,
	0.643538057582047850f, -0.765414115654738160f, 0.643391279433895960f,
	-0.765537498460013070f,
	0.643244477630085850f, -0.765660853118662390f, 0.643097652176015110f,
	-0.765784179626150970f,
	0.642950803077082080f, -0.765907477977944230f, 0.642803930338686100f,
	-0.766030748169509000f,
	0.642657033966226860f, -0.766153990196312810f, 0.642510113965105710f,
	-0.766277204053824710f,
	0.642363170340724320f, -0.766400389737514120f, 0.642216203098485370f,
	-0.766523547242852100f,
	0.642069212243792540f, -0.766646676565310380f, 0.641922197782050170f,
	-0.766769777700361920f,
	0.641775159718663500f, -0.766892850643480670f, 0.641628098059038860f,
	-0.767015895390141480f,
	0.641481012808583160f, -0.767138911935820400f, 0.641333903972704290f,
	-0.767261900275994390f,
	0.641186771556811250f, -0.767384860406141620f, 0.641039615566313390f,
	-0.767507792321741270f,
	0.640892436006621380f, -0.767630696018273270f, 0.640745232883146440f,
	-0.767753571491219030f,
	0.640598006201301030f, -0.767876418736060610f, 0.640450755966498140f,
	-0.767999237748281270f,
	0.640303482184151670f, -0.768122028523365310f, 0.640156184859676620f,
	-0.768244791056798220f,
	0.640008863998488440f, -0.768367525344066270f, 0.639861519606004010f,
	-0.768490231380656750f,
	0.639714151687640450f, -0.768612909162058270f, 0.639566760248816420f,
	-0.768735558683760310f,
	0.639419345294950700f, -0.768858179941253270f, 0.639271906831463510f,
	-0.768980772930028870f,
	0.639124444863775730f, -0.769103337645579590f, 0.638976959397309140f,
	-0.769225874083399260f,
	0.638829450437486400f, -0.769348382238982280f, 0.638681917989730840f,
	-0.769470862107824560f,
	0.638534362059466790f, -0.769593313685422940f, 0.638386782652119680f,
	-0.769715736967275020f,
	0.638239179773115390f, -0.769838131948879840f, 0.638091553427880930f,
	-0.769960498625737230f,
	0.637943903621844170f, -0.770082836993347900f, 0.637796230360433540f,
	-0.770205147047214100f,
	0.637648533649078810f, -0.770327428782838770f, 0.637500813493210310f,
	-0.770449682195725960f,
	0.637353069898259130f, -0.770571907281380700f, 0.637205302869657600f,
	-0.770694104035309140f,
	0.637057512412838590f, -0.770816272453018430f, 0.636909698533235870f,
	-0.770938412530016940f,
	0.636761861236284200f, -0.771060524261813710f, 0.636614000527419230f,
	-0.771182607643919220f,
	0.636466116412077180f, -0.771304662671844720f, 0.636318208895695570f,
	-0.771426689341102590f,
	0.636170277983712170f, -0.771548687647206300f, 0.636022323681566300f,
	-0.771670657585670330f,
	0.635874345994697720f, -0.771792599152010150f, 0.635726344928547180f,
	-0.771914512341742350f,
	0.635578320488556230f, -0.772036397150384410f, 0.635430272680167160f,
	-0.772158253573455240f,
	0.635282201508823530f, -0.772280081606474320f, 0.635134106979969300f,
	-0.772401881244962340f,
	0.634985989099049460f, -0.772523652484441330f, 0.634837847871510100f,
	-0.772645395320433860f,
	0.634689683302797850f, -0.772767109748463740f, 0.634541495398360130f,
	-0.772888795764056220f,
	0.634393284163645490f, -0.773010453362736990f, 0.634245049604103330f,
	-0.773132082540033070f,
	0.634096791725183740f, -0.773253683291472590f, 0.633948510532337810f,
	-0.773375255612584470f,
	0.633800206031017280f, -0.773496799498899050f, 0.633651878226674900f,
	-0.773618314945947460f,
	0.633503527124764320f, -0.773739801949261840f, 0.633355152730740060f,
	-0.773861260504375540f,
	0.633206755050057190f, -0.773982690606822790f, 0.633058334088172250f,
	-0.774104092252138940f,
	0.632909889850541860f, -0.774225465435860570f, 0.632761422342624000f,
	-0.774346810153525020f,
	0.632612931569877520f, -0.774468126400670860f, 0.632464417537761840f,
	-0.774589414172837550f,
	0.632315880251737680f, -0.774710673465565550f, 0.632167319717266030f,
	-0.774831904274396850f,
	0.632018735939809060f, -0.774953106594873820f, 0.631870128924829850f,
	-0.775074280422540450f,
	0.631721498677792370f, -0.775195425752941310f, 0.631572845204161130f,
	-0.775316542581622410f,
	0.631424168509401860f, -0.775437630904130430f, 0.631275468598980870f,
	-0.775558690716013580f,
	0.631126745478365340f, -0.775679722012820540f, 0.630977999153023660f,
	-0.775800724790101540f,
	0.630829229628424470f, -0.775921699043407580f, 0.630680436910038060f,
	-0.776042644768290770f,
	0.630531621003334600f, -0.776163561960304340f, 0.630382781913785940f,
	-0.776284450615002400f,
	0.630233919646864480f, -0.776405310727940390f, 0.630085034208043290f,
	-0.776526142294674430f,
	0.629936125602796550f, -0.776646945310762060f, 0.629787193836599200f,
	-0.776767719771761510f,
	0.629638238914927100f, -0.776888465673232440f, 0.629489260843256740f,
	-0.777009183010735290f,
	0.629340259627065750f, -0.777129871779831620f, 0.629191235271832410f,
	-0.777250531976084070f,
	0.629042187783036000f, -0.777371163595056200f, 0.628893117166156480f,
	-0.777491766632312900f,
	0.628744023426674790f, -0.777612341083419920f, 0.628594906570072660f,
	-0.777732886943944050f,
	0.628445766601832710f, -0.777853404209453040f, 0.628296603527438440f,
	-0.777973892875515990f,
	0.628147417352374120f, -0.778094352937702790f, 0.627998208082124810f,
	-0.778214784391584420f,
	0.627848975722176570f, -0.778335187232733090f, 0.627699720278016240f,
	-0.778455561456721900f,
	0.627550441755131530f, -0.778575907059124940f, 0.627401140159011160f,
	-0.778696224035517530f,
	0.627251815495144190f, -0.778816512381475870f, 0.627102467769021010f,
	-0.778936772092577500f,
	0.626953096986132770f, -0.779057003164400630f, 0.626803703151971310f,
	-0.779177205592524680f,
	0.626654286272029460f, -0.779297379372530300f, 0.626504846351800930f,
	-0.779417524499998900f,
	0.626355383396779990f, -0.779537640970513150f, 0.626205897412462130f,
	-0.779657728779656780f,
	0.626056388404343520f, -0.779777787923014440f, 0.625906856377921210f,
	-0.779897818396171890f,
	0.625757301338692900f, -0.780017820194715990f, 0.625607723292157410f,
	-0.780137793314234500f,
	0.625458122243814360f, -0.780257737750316590f, 0.625308498199164010f,
	-0.780377653498552040f,
	0.625158851163707730f, -0.780497540554531910f, 0.625009181142947460f,
	-0.780617398913848290f,
	0.624859488142386450f, -0.780737228572094380f, 0.624709772167528100f,
	-0.780857029524864470f,
	0.624560033223877320f, -0.780976801767753750f, 0.624410271316939380f,
	-0.781096545296358410f,
	0.624260486452220650f, -0.781216260106276090f, 0.624110678635228510f,
	-0.781335946193104870f,
	0.623960847871470770f, -0.781455603552444480f, 0.623810994166456130f,
	-0.781575232179895550f,
	0.623661117525694640f, -0.781694832071059390f, 0.623511217954696550f,
	-0.781814403221538830f,
	0.623361295458973340f, -0.781933945626937630f, 0.623211350044037270f,
	-0.782053459282860300f,
	0.623061381715401370f, -0.782172944184912900f, 0.622911390478579460f,
	-0.782292400328702400f,
	0.622761376339086460f, -0.782411827709836420f, 0.622611339302437730f,
	-0.782531226323924240f,
	0.622461279374150080f, -0.782650596166575730f, 0.622311196559740320f,
	-0.782769937233402050f,
	0.622161090864726930f, -0.782889249520015480f, 0.622010962294628600f,
	-0.783008533022029110f,
	0.621860810854965360f, -0.783127787735057310f, 0.621710636551257690f,
	-0.783247013654715380f,
	0.621560439389027270f, -0.783366210776619720f, 0.621410219373796150f,
	-0.783485379096387820f,
	0.621259976511087660f, -0.783604518609638200f, 0.621109710806425740f,
	-0.783723629311990470f,
	0.620959422265335180f, -0.783842711199065230f, 0.620809110893341900f,
	-0.783961764266484010f,
	0.620658776695972140f, -0.784080788509869950f, 0.620508419678753360f,
	-0.784199783924846570f,
	0.620358039847213830f, -0.784318750507038920f, 0.620207637206882430f,
	-0.784437688252072720f,
	0.620057211763289210f, -0.784556597155575240f, 0.619906763521964830f,
	-0.784675477213174320f,
	0.619756292488440660f, -0.784794328420499230f, 0.619605798668249390f,
	-0.784913150773180020f,
	0.619455282066924020f, -0.785031944266848080f, 0.619304742689998690f,
	-0.785150708897135560f,
	0.619154180543008410f, -0.785269444659675850f, 0.619003595631488770f,
	-0.785388151550103550f,
	0.618852987960976320f, -0.785506829564053930f, 0.618702357537008640f,
	-0.785625478697163700f,
	0.618551704365123860f, -0.785744098945070360f, 0.618401028450860980f,
	-0.785862690303412600f,
	0.618250329799760250f, -0.785981252767830150f, 0.618099608417362110f,
	-0.786099786333963820f,
	0.617948864309208260f, -0.786218290997455550f, 0.617798097480841140f,
	-0.786336766753948260f,
	0.617647307937803980f, -0.786455213599085770f, 0.617496495685640910f,
	-0.786573631528513230f,
	0.617345660729896940f, -0.786692020537876680f, 0.617194803076117630f,
	-0.786810380622823490f,
	0.617043922729849760f, -0.786928711779001700f, 0.616893019696640790f,
	-0.787047014002060790f,
	0.616742093982038830f, -0.787165287287650890f, 0.616591145591593230f,
	-0.787283531631423620f,
	0.616440174530853650f, -0.787401747029031320f, 0.616289180805370980f,
	-0.787519933476127810f,
	0.616138164420696910f, -0.787638090968367450f, 0.615987125382383870f,
	-0.787756219501405950f,
	0.615836063695985090f, -0.787874319070900110f, 0.615684979367054570f,
	-0.787992389672507950f,
	0.615533872401147430f, -0.788110431301888070f, 0.615382742803819330f,
	-0.788228443954700490f,
	0.615231590580626820f, -0.788346427626606230f, 0.615080415737127460f,
	-0.788464382313267430f,
	0.614929218278879590f, -0.788582308010347120f, 0.614777998211442190f,
	-0.788700204713509660f,
	0.614626755540375050f, -0.788818072418420170f, 0.614475490271239160f,
	-0.788935911120745130f,
	0.614324202409595950f, -0.789053720816151880f, 0.614172891961007990f,
	-0.789171501500308790f,
	0.614021558931038490f, -0.789289253168885650f, 0.613870203325251440f,
	-0.789406975817552810f,
	0.613718825149211830f, -0.789524669441982190f, 0.613567424408485330f,
	-0.789642334037846340f,
	0.613416001108638590f, -0.789759969600819070f, 0.613264555255239150f,
	-0.789877576126575280f,
	0.613113086853854910f, -0.789995153610791090f, 0.612961595910055170f,
	-0.790112702049143300f,
	0.612810082429409710f, -0.790230221437310030f, 0.612658546417489290f,
	-0.790347711770970520f,
	0.612506987879865570f, -0.790465173045804880f, 0.612355406822110760f,
	-0.790582605257494460f,
	0.612203803249798060f, -0.790700008401721610f, 0.612052177168501580f,
	-0.790817382474169660f,
	0.611900528583796070f, -0.790934727470523290f, 0.611748857501257400f,
	-0.791052043386467950f,
	0.611597163926462020f, -0.791169330217690090f, 0.611445447864987110f,
	-0.791286587959877720f,
	0.611293709322411010f, -0.791403816608719500f, 0.611141948304312570f,
	-0.791521016159905220f,
	0.610990164816271770f, -0.791638186609125770f, 0.610838358863869280f,
	-0.791755327952073150f,
	0.610686530452686280f, -0.791872440184440470f, 0.610534679588305320f,
	-0.791989523301921850f,
	0.610382806276309480f, -0.792106577300212390f, 0.610230910522282620f,
	-0.792223602175008310f,
	0.610078992331809620f, -0.792340597922007060f, 0.609927051710476230f,
	-0.792457564536906970f,
	0.609775088663868430f, -0.792574502015407580f, 0.609623103197573730f,
	-0.792691410353209450f,
	0.609471095317180240f, -0.792808289546014120f, 0.609319065028276820f,
	-0.792925139589524260f,
	0.609167012336453210f, -0.793041960479443640f, 0.609014937247299940f,
	-0.793158752211477140f,
	0.608862839766408200f, -0.793275514781330630f, 0.608710719899370420f,
	-0.793392248184711100f,
	0.608558577651779450f, -0.793508952417326660f, 0.608406413029229260f,
	-0.793625627474886190f,
	0.608254226037314490f, -0.793742273353100100f, 0.608102016681630550f,
	-0.793858890047679620f,
	0.607949784967773740f, -0.793975477554337170f, 0.607797530901341140f,
	-0.794092035868785960f,
	0.607645254487930830f, -0.794208564986740640f, 0.607492955733141660f,
	-0.794325064903916520f,
	0.607340634642572930f, -0.794441535616030590f, 0.607188291221825160f,
	-0.794557977118800270f,
	0.607035925476499760f, -0.794674389407944550f, 0.606883537412198580f,
	-0.794790772479183170f,
	0.606731127034524480f, -0.794907126328237010f, 0.606578694349081400f,
	-0.795023450950828050f,
	0.606426239361473550f, -0.795139746342679590f, 0.606273762077306430f,
	-0.795256012499515500f,
	0.606121262502186230f, -0.795372249417061190f, 0.605968740641719790f,
	-0.795488457091042990f,
	0.605816196501515080f, -0.795604635517188070f, 0.605663630087180490f,
	-0.795720784691225090f,
	0.605511041404325550f, -0.795836904608883460f, 0.605358430458560530f,
	-0.795952995265893910f,
	0.605205797255496500f, -0.796069056657987990f, 0.605053141800745430f,
	-0.796185088780898440f,
	0.604900464099919930f, -0.796301091630359110f, 0.604747764158633410f,
	-0.796417065202104980f,
	0.604595041982500360f, -0.796533009491872000f, 0.604442297577135970f,
	-0.796648924495397150f,
	0.604289530948156070f, -0.796764810208418720f, 0.604136742101177630f,
	-0.796880666626675780f,
	0.603983931041818020f, -0.796996493745908750f, 0.603831097775695880f,
	-0.797112291561858920f,
	0.603678242308430370f, -0.797228060070268700f, 0.603525364645641550f,
	-0.797343799266881700f,
	0.603372464792950370f, -0.797459509147442460f, 0.603219542755978440f,
	-0.797575189707696590f,
	0.603066598540348280f, -0.797690840943391040f, 0.602913632151683140f,
	-0.797806462850273570f,
	0.602760643595607220f, -0.797922055424093000f, 0.602607632877745550f,
	-0.798037618660599410f,
	0.602454600003723860f, -0.798153152555543750f, 0.602301544979168550f,
	-0.798268657104678310f,
	0.602148467809707320f, -0.798384132303756380f, 0.601995368500968130f,
	-0.798499578148532010f,
	0.601842247058580030f, -0.798614994634760820f, 0.601689103488173060f,
	-0.798730381758199210f,
	0.601535937795377730f, -0.798845739514604580f, 0.601382749985825420f,
	-0.798961067899735760f,
	0.601229540065148620f, -0.799076366909352350f, 0.601076308038980160f,
	-0.799191636539215210f,
	0.600923053912954090f, -0.799306876785086160f, 0.600769777692705230f,
	-0.799422087642728040f,
	0.600616479383868970f, -0.799537269107905010f, 0.600463158992081690f,
	-0.799652421176382130f,
	0.600309816522980430f, -0.799767543843925680f, 0.600156451982203350f,
	-0.799882637106302810f,
	0.600003065375389060f, -0.799997700959281910f, 0.599849656708177360f,
	-0.800112735398632370f,
	0.599696225986208310f, -0.800227740420124790f, 0.599542773215123390f,
	-0.800342716019530660f,
	0.599389298400564540f, -0.800457662192622710f, 0.599235801548174570f,
	-0.800572578935174750f,
	0.599082282663597310f, -0.800687466242961500f, 0.598928741752476900f,
	-0.800802324111759110f,
	0.598775178820458720f, -0.800917152537344300f, 0.598621593873188920f,
	-0.801031951515495330f,
	0.598467986916314310f, -0.801146721041991250f, 0.598314357955482600f,
	-0.801261461112612540f,
	0.598160706996342380f, -0.801376171723140130f, 0.598007034044542700f,
	-0.801490852869356840f,
	0.597853339105733910f, -0.801605504547046040f, 0.597699622185566830f,
	-0.801720126751992330f,
	0.597545883289693270f, -0.801834719479981310f, 0.597392122423765710f,
	-0.801949282726799660f,
	0.597238339593437530f, -0.802063816488235440f, 0.597084534804362740f,
	-0.802178320760077450f,
	0.596930708062196500f, -0.802292795538115720f, 0.596776859372594500f,
	-0.802407240818141300f,
	0.596622988741213330f, -0.802521656595946320f, 0.596469096173710360f,
	-0.802636042867324150f,
	0.596315181675743820f, -0.802750399628069160f, 0.596161245252972540f,
	-0.802864726873976590f,
	0.596007286911056530f, -0.802979024600843140f, 0.595853306655656390f,
	-0.803093292804466400f,
	0.595699304492433470f, -0.803207531480644830f, 0.595545280427049790f,
	-0.803321740625178470f,
	0.595391234465168730f, -0.803435920233868120f, 0.595237166612453850f,
	-0.803550070302515570f,
	0.595083076874569960f, -0.803664190826924090f, 0.594928965257182420f,
	-0.803778281802897570f,
	0.594774831765957580f, -0.803892343226241260f, 0.594620676406562240f,
	-0.804006375092761520f,
	0.594466499184664540f, -0.804120377398265700f, 0.594312300105932830f,
	-0.804234350138562260f,
	0.594158079176036800f, -0.804348293309460780f, 0.594003836400646690f,
	-0.804462206906771840f,
	0.593849571785433630f, -0.804576090926307000f, 0.593695285336069300f,
	-0.804689945363879500f,
	0.593540977058226390f, -0.804803770215302810f, 0.593386646957578480f,
	-0.804917565476392150f,
	0.593232295039799800f, -0.805031331142963660f, 0.593077921310565580f,
	-0.805145067210834120f,
	0.592923525775551410f, -0.805258773675822210f, 0.592769108440434070f,
	-0.805372450533747060f,
	0.592614669310891130f, -0.805486097780429120f, 0.592460208392600940f,
	-0.805599715411689950f,
	0.592305725691242400f, -0.805713303423352120f, 0.592151221212495640f,
	-0.805826861811239300f,
	0.591996694962040990f, -0.805940390571176280f, 0.591842146945560250f,
	-0.806053889698988950f,
	0.591687577168735550f, -0.806167359190504310f, 0.591532985637249990f,
	-0.806280799041550370f,
	0.591378372356787580f, -0.806394209247956240f, 0.591223737333032910f,
	-0.806507589805552260f,
	0.591069080571671510f, -0.806620940710169650f, 0.590914402078389520f,
	-0.806734261957640750f,
	0.590759701858874280f, -0.806847553543799220f, 0.590604979918813440f,
	-0.806960815464479620f,
	0.590450236263895920f, -0.807074047715517610f, 0.590295470899810940f,
	-0.807187250292749850f,
	0.590140683832248940f, -0.807300423192014450f, 0.589985875066900920f,
	-0.807413566409150190f,
	0.589831044609458900f, -0.807526679939997160f, 0.589676192465615420f,
	-0.807639763780396370f,
	0.589521318641063940f, -0.807752817926190360f, 0.589366423141498790f,
	-0.807865842373222120f,
	0.589211505972615070f, -0.807978837117336310f, 0.589056567140108460f,
	-0.808091802154378260f,
	0.588901606649675840f, -0.808204737480194720f, 0.588746624507014650f,
	-0.808317643090633250f,
	0.588591620717822890f, -0.808430518981542720f, 0.588436595287799900f,
	-0.808543365148773010f,
	0.588281548222645330f, -0.808656181588174980f, 0.588126479528059850f,
	-0.808768968295600850f,
	0.587971389209745120f, -0.808881725266903610f, 0.587816277273403020f,
	-0.808994452497937560f,
	0.587661143724736770f, -0.809107149984558130f, 0.587505988569450020f,
	-0.809219817722621750f,
	0.587350811813247660f, -0.809332455707985840f, 0.587195613461834910f,
	-0.809445063936509170f,
	0.587040393520918080f, -0.809557642404051260f, 0.586885151996203950f,
	-0.809670191106473090f,
	0.586729888893400500f, -0.809782710039636420f, 0.586574604218216280f,
	-0.809895199199404450f,
	0.586419297976360500f, -0.810007658581641140f, 0.586263970173543700f,
	-0.810120088182211600f,
	0.586108620815476430f, -0.810232487996982330f, 0.585953249907870680f,
	-0.810344858021820550f,
	0.585797857456438860f, -0.810457198252594770f, 0.585642443466894420f,
	-0.810569508685174630f,
	0.585487007944951450f, -0.810681789315430670f, 0.585331550896324940f,
	-0.810794040139234730f,
	0.585176072326730410f, -0.810906261152459670f, 0.585020572241884530f,
	-0.811018452350979470f,
	0.584865050647504490f, -0.811130613730669190f, 0.584709507549308500f,
	-0.811242745287404810f,
	0.584553942953015330f, -0.811354847017063730f, 0.584398356864344710f,
	-0.811466918915524250f,
	0.584242749289016980f, -0.811578960978665890f, 0.584087120232753550f,
	-0.811690973202369050f,
	0.583931469701276300f, -0.811802955582515360f, 0.583775797700308070f,
	-0.811914908114987680f,
	0.583620104235572760f, -0.812026830795669730f, 0.583464389312794430f,
	-0.812138723620446480f,
	0.583308652937698290f, -0.812250586585203880f, 0.583152895116010540f,
	-0.812362419685829120f,
	0.582997115853457700f, -0.812474222918210480f, 0.582841315155767650f,
	-0.812585996278237020f,
	0.582685493028668460f, -0.812697739761799490f, 0.582529649477889320f,
	-0.812809453364789160f,
	0.582373784509160220f, -0.812921137083098770f, 0.582217898128211790f,
	-0.813032790912621930f,
	0.582061990340775550f, -0.813144414849253590f, 0.581906061152583920f,
	-0.813256008888889380f,
	0.581750110569369760f, -0.813367573027426570f, 0.581594138596866930f,
	-0.813479107260763220f,
	0.581438145240810280f, -0.813590611584798510f, 0.581282130506935110f,
	-0.813702085995432700f,
	0.581126094400977620f, -0.813813530488567190f, 0.580970036928674880f,
	-0.813924945060104490f,
	0.580813958095764530f, -0.814036329705948300f, 0.580657857907985410f,
	-0.814147684422003360f,
	0.580501736371076600f, -0.814259009204175270f, 0.580345593490778300f,
	-0.814370304048371070f,
	0.580189429272831680f, -0.814481568950498610f, 0.580033243722978150f,
	-0.814592803906467270f,
	0.579877036846960350f, -0.814704008912187080f, 0.579720808650521560f,
	-0.814815183963569330f,
	0.579564559139405740f, -0.814926329056526620f, 0.579408288319357980f,
	-0.815037444186972220f,
	0.579251996196123550f, -0.815148529350820830f, 0.579095682775449210f,
	-0.815259584543988280f,
	0.578939348063081890f, -0.815370609762391290f, 0.578782992064769690f,
	-0.815481605001947770f,
	0.578626614786261430f, -0.815592570258576680f, 0.578470216233306740f,
	-0.815703505528198260f,
	0.578313796411655590f, -0.815814410806733780f, 0.578157355327059360f,
	-0.815925286090105390f,
	0.578000892985269910f, -0.816036131374236700f, 0.577844409392039850f,
	-0.816146946655052160f,
	0.577687904553122800f, -0.816257731928477390f, 0.577531378474272830f,
	-0.816368487190439200f,
	0.577374831161244880f, -0.816479212436865390f, 0.577218262619794920f,
	-0.816589907663684890f,
	0.577061672855679550f, -0.816700572866827850f, 0.576905061874655960f,
	-0.816811208042225290f,
	0.576748429682482520f, -0.816921813185809480f, 0.576591776284917870f,
	-0.817032388293513880f,
	0.576435101687721830f, -0.817142933361272970f, 0.576278405896654910f,
	-0.817253448385022230f,
	0.576121688917478390f, -0.817363933360698460f, 0.575964950755954330f,
	-0.817474388284239240f,
	0.575808191417845340f, -0.817584813151583710f, 0.575651410908915250f,
	-0.817695207958671680f,
	0.575494609234928230f, -0.817805572701444270f, 0.575337786401649560f,
	-0.817915907375843740f,
	0.575180942414845190f, -0.818026211977813440f, 0.575024077280281820f,
	-0.818136486503297620f,
	0.574867191003726740f, -0.818246730948241960f, 0.574710283590948450f,
	-0.818356945308593150f,
	0.574553355047715760f, -0.818467129580298660f, 0.574396405379798750f,
	-0.818577283759307490f,
	0.574239434592967890f, -0.818687407841569570f, 0.574082442692994470f,
	-0.818797501823036010f,
	0.573925429685650750f, -0.818907565699658950f, 0.573768395576709560f,
	-0.819017599467391500f,
	0.573611340371944610f, -0.819127603122188240f, 0.573454264077130400f,
	-0.819237576660004520f,
	0.573297166698042320f, -0.819347520076796900f, 0.573140048240456060f,
	-0.819457433368523280f,
	0.572982908710148680f, -0.819567316531142230f, 0.572825748112897550f,
	-0.819677169560613760f,
	0.572668566454481160f, -0.819786992452898990f, 0.572511363740678790f,
	-0.819896785203959810f,
	0.572354139977270030f, -0.820006547809759680f, 0.572196895170035580f,
	-0.820116280266262710f,
	0.572039629324757050f, -0.820225982569434690f, 0.571882342447216590f,
	-0.820335654715241840f,
	0.571725034543197120f, -0.820445296699652050f, 0.571567705618482580f,
	-0.820554908518633890f,
	0.571410355678857340f, -0.820664490168157460f, 0.571252984730106660f,
	-0.820774041644193650f,
	0.571095592778016690f, -0.820883562942714580f, 0.570938179828374360f,
	-0.820993054059693470f,
	0.570780745886967370f, -0.821102514991104650f, 0.570623290959583860f,
	-0.821211945732923550f,
	0.570465815052012990f, -0.821321346281126740f, 0.570308318170045010f,
	-0.821430716631691760f,
	0.570150800319470300f, -0.821540056780597610f, 0.569993261506080650f,
	-0.821649366723823830f,
	0.569835701735668110f, -0.821758646457351640f, 0.569678121014025710f,
	-0.821867895977163140f,
	0.569520519346947250f, -0.821977115279241550f, 0.569362896740227330f,
	-0.822086304359571090f,
	0.569205253199661200f, -0.822195463214137170f, 0.569047588731045220f,
	-0.822304591838926350f,
	0.568889903340175970f, -0.822413690229926390f, 0.568732197032851160f,
	-0.822522758383125940f,
	0.568574469814869250f, -0.822631796294514990f, 0.568416721692029390f,
	-0.822740803960084420f,
	0.568258952670131490f, -0.822849781375826320f, 0.568101162754976570f,
	-0.822958728537734000f,
	0.567943351952365670f, -0.823067645441801670f, 0.567785520268101250f,
	-0.823176532084024860f,
	0.567627667707986230f, -0.823285388460400110f, 0.567469794277824620f,
	-0.823394214566925080f,
	0.567311899983420800f, -0.823503010399598390f, 0.567153984830580100f,
	-0.823611775954420260f,
	0.566996048825108680f, -0.823720511227391320f, 0.566838091972813320f,
	-0.823829216214513990f,
	0.566680114279501710f, -0.823937890911791370f, 0.566522115750982100f,
	-0.824046535315227760f,
	0.566364096393063950f, -0.824155149420828570f, 0.566206056211556840f,
	-0.824263733224600450f,
	0.566047995212271560f, -0.824372286722551250f, 0.565889913401019570f,
	-0.824480809910689500f,
	0.565731810783613230f, -0.824589302785025290f, 0.565573687365865440f,
	-0.824697765341569470f,
	0.565415543153589770f, -0.824806197576334330f, 0.565257378152600910f,
	-0.824914599485333080f,
	0.565099192368714090f, -0.825022971064580220f, 0.564940985807745320f,
	-0.825131312310090960f,
	0.564782758475511400f, -0.825239623217882130f, 0.564624510377830120f,
	-0.825347903783971380f,
	0.564466241520519500f, -0.825456154004377440f, 0.564307951909398750f,
	-0.825564373875120490f,
	0.564149641550287680f, -0.825672563392221390f, 0.563991310449007080f,
	-0.825780722551702430f,
	0.563832958611378170f, -0.825888851349586780f, 0.563674586043223180f,
	-0.825996949781898970f,
	0.563516192750364910f, -0.826105017844664610f, 0.563357778738627020f,
	-0.826213055533910110f,
	0.563199344013834090f, -0.826321062845663420f, 0.563040888581811230f,
	-0.826429039775953390f,
	0.562882412448384550f, -0.826536986320809960f, 0.562723915619380400f,
	-0.826644902476264210f,
	0.562565398100626560f, -0.826752788238348520f, 0.562406859897951140f,
	-0.826860643603096080f,
	0.562248301017183150f, -0.826968468566541490f, 0.562089721464152480f,
	-0.827076263124720270f,
	0.561931121244689470f, -0.827184027273669020f, 0.561772500364625450f,
	-0.827291761009425810f,
	0.561613858829792420f, -0.827399464328029350f, 0.561455196646023280f,
	-0.827507137225519830f,
	0.561296513819151470f, -0.827614779697938400f, 0.561137810355011530f,
	-0.827722391741327220f,
	0.560979086259438260f, -0.827829973351729810f, 0.560820341538267540f,
	-0.827937524525190870f,
	0.560661576197336030f, -0.828045045257755800f, 0.560502790242481060f,
	-0.828152535545471410f,
	0.560343983679540860f, -0.828259995384385550f, 0.560185156514354080f,
	-0.828367424770547480f,
	0.560026308752760380f, -0.828474823700007130f, 0.559867440400600320f,
	-0.828582192168815790f,
	0.559708551463714790f, -0.828689530173025710f, 0.559549641947945870f,
	-0.828796837708690610f,
	0.559390711859136140f, -0.828904114771864870f, 0.559231761203129010f,
	-0.829011361358604430f,
	0.559072789985768480f, -0.829118577464965980f, 0.558913798212899770f,
	-0.829225763087007570f,
	0.558754785890368310f, -0.829332918220788250f, 0.558595753024020760f,
	-0.829440042862368170f,
	0.558436699619704100f, -0.829547137007808800f, 0.558277625683266330f,
	-0.829654200653172640f,
	0.558118531220556100f, -0.829761233794523050f, 0.557959416237422960f,
	-0.829868236427924840f,
	0.557800280739717100f, -0.829975208549443840f, 0.557641124733289420f,
	-0.830082150155146970f,
	0.557481948223991660f, -0.830189061241102370f, 0.557322751217676160f,
	-0.830295941803379070f,
	0.557163533720196340f, -0.830402791838047550f, 0.557004295737406060f,
	-0.830509611341179070f,
	0.556845037275160100f, -0.830616400308846200f, 0.556685758339313890f,
	-0.830723158737122880f,
	0.556526458935723720f, -0.830829886622083570f, 0.556367139070246490f,
	-0.830936583959804410f,
	0.556207798748739930f, -0.831043250746362320f, 0.556048437977062720f,
	-0.831149886977835430f,
	0.555889056761073920f, -0.831256492650303210f, 0.555729655106633520f,
	-0.831363067759845920f,
	0.555570233019602290f, -0.831469612302545240f, 0.555410790505841740f,
	-0.831576126274483630f,
	0.555251327571214090f, -0.831682609671745120f, 0.555091844221582420f,
	-0.831789062490414400f,
	0.554932340462810370f, -0.831895484726577590f, 0.554772816300762580f,
	-0.832001876376321840f,
	0.554613271741304040f, -0.832108237435735480f, 0.554453706790301040f,
	-0.832214567900907980f,
	0.554294121453620110f, -0.832320867767929680f, 0.554134515737128910f,
	-0.832427137032892280f,
	0.553974889646695610f, -0.832533375691888680f, 0.553815243188189090f,
	-0.832639583741012770f,
	0.553655576367479310f, -0.832745761176359460f, 0.553495889190436570f,
	-0.832851907994024980f,
	0.553336181662932410f, -0.832958024190106670f, 0.553176453790838460f,
	-0.833064109760702890f,
	0.553016705580027580f, -0.833170164701913190f, 0.552856937036373290f,
	-0.833276189009838240f,
	0.552697148165749770f, -0.833382182680579730f, 0.552537338974032120f,
	-0.833488145710240770f,
	0.552377509467096070f, -0.833594078094925140f, 0.552217659650817930f,
	-0.833699979830738290f,
	0.552057789531074980f, -0.833805850913786340f, 0.551897899113745320f,
	-0.833911691340176730f,
	0.551737988404707450f, -0.834017501106018130f, 0.551578057409841000f,
	-0.834123280207419990f,
	0.551418106135026060f, -0.834229028640493420f, 0.551258134586143700f,
	-0.834334746401350080f,
	0.551098142769075430f, -0.834440433486103190f, 0.550938130689703880f,
	-0.834546089890866760f,
	0.550778098353912230f, -0.834651715611756330f, 0.550618045767584330f,
	-0.834757310644888230f,
	0.550457972936604810f, -0.834862874986380010f, 0.550297879866859190f,
	-0.834968408632350450f,
	0.550137766564233630f, -0.835073911578919300f, 0.549977633034615000f,
	-0.835179383822207580f,
	0.549817479283891020f, -0.835284825358337370f, 0.549657305317949980f,
	-0.835390236183431780f,
	0.549497111142680960f, -0.835495616293615350f, 0.549336896763974010f,
	-0.835600965685013410f,
	0.549176662187719770f, -0.835706284353752600f, 0.549016407419809390f,
	-0.835811572295960590f,
	0.548856132466135290f, -0.835916829507766360f, 0.548695837332590090f,
	-0.836022055985299880f,
	0.548535522025067390f, -0.836127251724692160f, 0.548375186549461600f,
	-0.836232416722075600f,
	0.548214830911667780f, -0.836337550973583530f, 0.548054455117581880f,
	-0.836442654475350380f,
	0.547894059173100190f, -0.836547727223511890f, 0.547733643084120200f,
	-0.836652769214204950f,
	0.547573206856539870f, -0.836757780443567190f, 0.547412750496257930f,
	-0.836862760907737810f,
	0.547252274009174090f, -0.836967710602857020f, 0.547091777401188530f,
	-0.837072629525066000f,
	0.546931260678202190f, -0.837177517670507190f, 0.546770723846116800f,
	-0.837282375035324320f,
	0.546610166910834860f, -0.837387201615661940f, 0.546449589878259760f,
	-0.837491997407665890f,
	0.546288992754295210f, -0.837596762407483040f, 0.546128375544846060f,
	-0.837701496611261700f,
	0.545967738255817680f, -0.837806200015150940f, 0.545807080893116140f,
	-0.837910872615301060f,
	0.545646403462648590f, -0.838015514407863700f, 0.545485705970322530f,
	-0.838120125388991500f,
	0.545324988422046460f, -0.838224705554837970f, 0.545164250823729320f,
	-0.838329254901558300f,
	0.545003493181281160f, -0.838433773425308340f, 0.544842715500612470f,
	-0.838538261122245170f,
	0.544681917787634530f, -0.838642717988527300f, 0.544521100048259710f,
	-0.838747144020313920f,
	0.544360262288400400f, -0.838851539213765760f, 0.544199404513970420f,
	-0.838955903565044350f,
	0.544038526730883930f, -0.839060237070312630f, 0.543877628945055980f,
	-0.839164539725734570f,
	0.543716711162402390f, -0.839268811527475230f, 0.543555773388839650f,
	-0.839373052471700690f,
	0.543394815630284800f, -0.839477262554578550f, 0.543233837892656000f,
	-0.839581441772277120f,
	0.543072840181871850f, -0.839685590120966110f, 0.542911822503851730f,
	-0.839789707596816260f,
	0.542750784864516000f, -0.839893794195999410f, 0.542589727269785270f,
	-0.839997849914688730f,
	0.542428649725581360f, -0.840101874749058400f, 0.542267552237826520f,
	-0.840205868695283580f,
	0.542106434812444030f, -0.840309831749540770f, 0.541945297455357470f,
	-0.840413763908007480f,
	0.541784140172491660f, -0.840517665166862440f, 0.541622962969771640f,
	-0.840621535522285690f,
	0.541461765853123560f, -0.840725374970458070f, 0.541300548828474120f,
	-0.840829183507561640f,
	0.541139311901750910f, -0.840932961129779670f, 0.540978055078882190f,
	-0.841036707833296650f,
	0.540816778365796670f, -0.841140423614298080f, 0.540655481768424260f,
	-0.841244108468970580f,
	0.540494165292695230f, -0.841347762393501950f, 0.540332828944540820f,
	-0.841451385384081260f,
	0.540171472729892970f, -0.841554977436898330f, 0.540010096654684020f,
	-0.841658538548144760f,
	0.539848700724847700f, -0.841762068714012490f, 0.539687284946317570f,
	-0.841865567930695340f,
	0.539525849325029010f, -0.841969036194387680f, 0.539364393866917150f,
	-0.842072473501285450f,
	0.539202918577918240f, -0.842175879847585570f, 0.539041423463969550f,
	-0.842279255229485880f,
	0.538879908531008420f, -0.842382599643185960f, 0.538718373784973670f,
	-0.842485913084885630f,
	0.538556819231804210f, -0.842589195550786600f, 0.538395244877439950f,
	-0.842692447037091560f,
	0.538233650727821700f, -0.842795667540004120f, 0.538072036788890600f,
	-0.842898857055729310f,
	0.537910403066588990f, -0.843002015580472830f, 0.537748749566859470f,
	-0.843105143110442050f,
	0.537587076295645510f, -0.843208239641845440f, 0.537425383258891660f,
	-0.843311305170892030f,
	0.537263670462542530f, -0.843414339693792760f, 0.537101937912544240f,
	-0.843517343206759080f,
	0.536940185614843020f, -0.843620315706004040f, 0.536778413575385920f,
	-0.843723257187741550f,
	0.536616621800121150f, -0.843826167648186740f, 0.536454810294997090f,
	-0.843929047083555870f,
	0.536292979065963180f, -0.844031895490066410f, 0.536131128118969350f,
	-0.844134712863936930f,
	0.535969257459966710f, -0.844237499201387020f, 0.535807367094906620f,
	-0.844340254498637590f,
	0.535645457029741090f, -0.844442978751910660f, 0.535483527270423370f,
	-0.844545671957429240f,
	0.535321577822907010f, -0.844648334111417820f, 0.535159608693146720f,
	-0.844750965210101510f,
	0.534997619887097260f, -0.844853565249707010f, 0.534835611410714670f,
	-0.844956134226462100f,
	0.534673583269955510f, -0.845058672136595470f, 0.534511535470777010f,
	-0.845161178976337140f,
	0.534349468019137520f, -0.845263654741918220f, 0.534187380920995600f,
	-0.845366099429570970f,
	0.534025274182310380f, -0.845468513035528830f, 0.533863147809042650f,
	-0.845570895556026270f,
	0.533701001807152960f, -0.845673246987299070f, 0.533538836182603120f,
	-0.845775567325583900f,
	0.533376650941355560f, -0.845877856567118890f, 0.533214446089372960f,
	-0.845980114708143270f,
	0.533052221632619670f, -0.846082341744896940f, 0.532889977577059690f,
	-0.846184537673621670f,
	0.532727713928658810f, -0.846286702490559710f, 0.532565430693382580f,
	-0.846388836191954930f,
	0.532403127877198010f, -0.846490938774052020f, 0.532240805486072330f,
	-0.846593010233097190f,
	0.532078463525973540f, -0.846695050565337450f, 0.531916102002870760f,
	-0.846797059767020910f,
	0.531753720922733320f, -0.846899037834397350f, 0.531591320291531780f,
	-0.847000984763716880f,
	0.531428900115236910f, -0.847102900551231500f, 0.531266460399820390f,
	-0.847204785193193980f,
	0.531104001151255000f, -0.847306638685858320f, 0.530941522375513510f,
	-0.847408461025479730f,
	0.530779024078570250f, -0.847510252208314330f, 0.530616506266399450f,
	-0.847612012230619660f,
	0.530453968944976320f, -0.847713741088654270f, 0.530291412120277420f,
	-0.847815438778677930f,
	0.530128835798278850f, -0.847917105296951410f, 0.529966239984958620f,
	-0.848018740639736810f,
	0.529803624686294830f, -0.848120344803297120f, 0.529640989908265910f,
	-0.848221917783896990f,
	0.529478335656852090f, -0.848323459577801530f, 0.529315661938033140f,
	-0.848424970181277600f,
	0.529152968757790720f, -0.848526449590592650f, 0.528990256122106040f,
	-0.848627897802015860f,
	0.528827524036961980f, -0.848729314811817010f, 0.528664772508341540f,
	-0.848830700616267530f,
	0.528502001542228480f, -0.848932055211639610f, 0.528339211144607690f,
	-0.849033378594206690f,
	0.528176401321464370f, -0.849134670760243630f, 0.528013572078784740f,
	-0.849235931706025960f,
	0.527850723422555460f, -0.849337161427830670f, 0.527687855358763720f,
	-0.849438359921935950f,
	0.527524967893398200f, -0.849539527184620890f, 0.527362061032447430f,
	-0.849640663212165910f,
	0.527199134781901390f, -0.849741768000852440f, 0.527036189147750190f,
	-0.849842841546963210f,
	0.526873224135984700f, -0.849943883846782210f, 0.526710239752597010f,
	-0.850044894896594070f,
	0.526547236003579330f, -0.850145874692685210f, 0.526384212894925210f,
	-0.850246823231342710f,
	0.526221170432628170f, -0.850347740508854980f, 0.526058108622682760f,
	-0.850448626521511650f,
	0.525895027471084740f, -0.850549481265603370f, 0.525731926983829640f,
	-0.850650304737422200f,
	0.525568807166914680f, -0.850751096933260790f, 0.525405668026336810f,
	-0.850851857849413640f,
	0.525242509568094710f, -0.850952587482175730f, 0.525079331798186890f,
	-0.851053285827843790f,
	0.524916134722612890f, -0.851153952882715340f, 0.524752918347373360f,
	-0.851254588643089120f,
	0.524589682678468840f, -0.851355193105265200f, 0.524426427721901510f,
	-0.851455766265544310f,
	0.524263153483673470f, -0.851556308120228870f, 0.524099859969787810f,
	-0.851656818665622370f,
	0.523936547186248600f, -0.851757297898029120f, 0.523773215139060170f,
	-0.851857745813754840f,
	0.523609863834228030f, -0.851958162409106380f, 0.523446493277757940f,
	-0.852058547680391580f,
	0.523283103475656430f, -0.852158901623919830f, 0.523119694433931250f,
	-0.852259224236001090f,
	0.522956266158590140f, -0.852359515512947090f, 0.522792818655642200f,
	-0.852459775451070100f,
	0.522629351931096720f, -0.852560004046683970f, 0.522465865990963900f,
	-0.852660201296103760f,
	0.522302360841254700f, -0.852760367195645300f, 0.522138836487980650f,
	-0.852860501741625860f,
	0.521975292937154390f, -0.852960604930363630f, 0.521811730194788550f,
	-0.853060676758178320f,
	0.521648148266897090f, -0.853160717221390420f, 0.521484547159494550f,
	-0.853260726316321770f,
	0.521320926878595550f, -0.853360704039295430f, 0.521157287430216610f,
	-0.853460650386635320f,
	0.520993628820373810f, -0.853560565354666840f, 0.520829951055084780f,
	-0.853660448939716270f,
	0.520666254140367270f, -0.853760301138111300f, 0.520502538082239790f,
	-0.853860121946180660f,
	0.520338802886721960f, -0.853959911360254060f, 0.520175048559833760f,
	-0.854059669376662780f,
	0.520011275107596040f, -0.854159395991738730f, 0.519847482536030300f,
	-0.854259091201815420f,
	0.519683670851158520f, -0.854358755003227440f, 0.519519840059003870f,
	-0.854458387392310060f,
	0.519355990165589530f, -0.854557988365400530f, 0.519192121176940360f,
	-0.854657557918836460f,
	0.519028233099080970f, -0.854757096048957110f, 0.518864325938037000f,
	-0.854856602752102850f,
	0.518700399699835170f, -0.854956078024614820f, 0.518536454390502110f,
	-0.855055521862835950f,
	0.518372490016066220f, -0.855154934263109620f, 0.518208506582555460f,
	-0.855254315221781080f,
	0.518044504095999340f, -0.855353664735196030f, 0.517880482562427800f,
	-0.855452982799701830f,
	0.517716441987871150f, -0.855552269411646970f, 0.517552382378360990f,
	-0.855651524567380690f,
	0.517388303739929060f, -0.855750748263253920f, 0.517224206078608310f,
	-0.855849940495618240f,
	0.517060089400432130f, -0.855949101260826790f, 0.516895953711434260f,
	-0.856048230555233820f,
	0.516731799017649980f, -0.856147328375194470f, 0.516567625325114350f,
	-0.856246394717065210f,
	0.516403432639863990f, -0.856345429577203610f, 0.516239220967935620f,
	-0.856444432951968480f,
	0.516074990315366630f, -0.856543404837719960f, 0.515910740688195650f,
	-0.856642345230818720f,
	0.515746472092461380f, -0.856741254127627470f, 0.515582184534203790f,
	-0.856840131524509220f,
	0.515417878019463150f, -0.856938977417828650f, 0.515253552554280290f,
	-0.857037791803951680f,
	0.515089208144697270f, -0.857136574679244870f, 0.514924844796756490f,
	-0.857235326040076460f,
	0.514760462516501200f, -0.857334045882815590f, 0.514596061309975040f,
	-0.857432734203832700f,
	0.514431641183222930f, -0.857531390999499040f, 0.514267202142289830f,
	-0.857630016266187620f,
	0.514102744193221660f, -0.857728610000272120f, 0.513938267342065490f,
	-0.857827172198127320f,
	0.513773771594868030f, -0.857925702856129790f, 0.513609256957677900f,
	-0.858024201970656540f,
	0.513444723436543570f, -0.858122669538086020f, 0.513280171037514330f,
	-0.858221105554798250f,
	0.513115599766640560f, -0.858319510017173440f, 0.512951009629972860f,
	-0.858417882921594040f,
	0.512786400633563070f, -0.858516224264442740f, 0.512621772783463100f,
	-0.858614534042104080f,
	0.512457126085725800f, -0.858712812250963520f, 0.512292460546404980f,
	-0.858811058887407500f,
	0.512127776171554690f, -0.858909273947823900f, 0.511963072967230200f,
	-0.859007457428601410f,
	0.511798350939487000f, -0.859105609326130340f, 0.511633610094381350f,
	-0.859203729636801920f,
	0.511468850437970520f, -0.859301818357008360f, 0.511304071976311890f,
	-0.859399875483143450f,
	0.511139274715464390f, -0.859497901011601620f, 0.510974458661486720f,
	-0.859595894938779080f,
	0.510809623820439040f, -0.859693857261072610f, 0.510644770198381730f,
	-0.859791787974880540f,
	0.510479897801375700f, -0.859889687076602290f, 0.510315006635483350f,
	-0.859987554562638200f,
	0.510150096706766700f, -0.860085390429390140f, 0.509985168021289570f,
	-0.860183194673260880f,
	0.509820220585115560f, -0.860280967290654510f, 0.509655254404309250f,
	-0.860378708277976130f,
	0.509490269484936360f, -0.860476417631632070f, 0.509325265833062480f,
	-0.860574095348029980f,
	0.509160243454754750f, -0.860671741423578380f, 0.508995202356080310f,
	-0.860769355854687060f,
	0.508830142543106990f, -0.860866938637767310f, 0.508665064021904260f,
	-0.860964489769230900f,
	0.508499966798540810f, -0.861062009245491480f, 0.508334850879087470f,
	-0.861159497062963350f,
	0.508169716269614710f, -0.861256953218062060f, 0.508004562976194010f,
	-0.861354377707204800f,
	0.507839391004897940f, -0.861451770526809210f, 0.507674200361798890f,
	-0.861549131673294720f,
	0.507508991052970870f, -0.861646461143081300f, 0.507343763084487920f,
	-0.861743758932590700f,
	0.507178516462425290f, -0.861841025038245330f, 0.507013251192858340f,
	-0.861938259456469180f,
	0.506847967281863320f, -0.862035462183687210f, 0.506682664735517600f,
	-0.862132633216325380f,
	0.506517343559898530f, -0.862229772550811240f, 0.506352003761084800f,
	-0.862326880183573060f,
	0.506186645345155450f, -0.862423956111040500f, 0.506021268318189830f,
	-0.862521000329644520f,
	0.505855872686268860f, -0.862618012835816740f, 0.505690458455473340f,
	-0.862714993625990690f,
	0.505525025631885510f, -0.862811942696600330f, 0.505359574221587390f,
	-0.862908860044081290f,
	0.505194104230662240f, -0.863005745664870210f, 0.505028615665194300f,
	-0.863102599555404800f,
	0.504863108531267480f, -0.863199421712124160f, 0.504697582834967680f,
	-0.863296212131468230f,
	0.504532038582380380f, -0.863392970809878310f, 0.504366475779592150f,
	-0.863489697743797140f,
	0.504200894432690560f, -0.863586392929667990f, 0.504035294547763080f,
	-0.863683056363935940f,
	0.503869676130898950f, -0.863779688043046610f, 0.503704039188186960f,
	-0.863876287963447510f,
	0.503538383725717580f, -0.863972856121586700f, 0.503372709749581150f,
	-0.864069392513913680f,
	0.503207017265869030f, -0.864165897136879300f, 0.503041306280673450f,
	-0.864262369986934950f,
	0.502875576800086880f, -0.864358811060534030f, 0.502709828830203100f,
	-0.864455220354130250f,
	0.502544062377115800f, -0.864551597864179230f, 0.502378277446919870f,
	-0.864647943587137480f,
	0.502212474045710900f, -0.864744257519462380f, 0.502046652179584660f,
	-0.864840539657612980f,
	0.501880811854638400f, -0.864936789998049020f, 0.501714953076969230f,
	-0.865033008537231750f,
	0.501549075852675390f, -0.865129195271623690f, 0.501383180187855880f,
	-0.865225350197688090f,
	0.501217266088609950f, -0.865321473311889800f, 0.501051333561038040f,
	-0.865417564610694410f,
	0.500885382611240940f, -0.865513624090568980f, 0.500719413245319880f,
	-0.865609651747981880f,
	0.500553425469377640f, -0.865705647579402270f, 0.500387419289516580f,
	-0.865801611581300760f,
	0.500221394711840680f, -0.865897543750148820f, 0.500055351742453860f,
	-0.865993444082419520f,
	0.499889290387461380f, -0.866089312574586770f, 0.499723210652968710f,
	-0.866185149223125730f,
	0.499557112545081890f, -0.866280954024512990f, 0.499390996069908220f,
	-0.866376726975225830f,
	0.499224861233555030f, -0.866472468071743050f, 0.499058708042130930f,
	-0.866568177310544360f,
	0.498892536501744750f, -0.866663854688111020f, 0.498726346618505960f,
	-0.866759500200925290f,
	0.498560138398525200f, -0.866855113845470320f, 0.498393911847913150f,
	-0.866950695618231020f,
	0.498227666972781870f, -0.867046245515692650f, 0.498061403779243520f,
	-0.867141763534342360f,
	0.497895122273410930f, -0.867237249670668400f, 0.497728822461398100f,
	-0.867332703921159690f,
	0.497562504349319090f, -0.867428126282306920f, 0.497396167943289340f,
	-0.867523516750601460f,
	0.497229813249424340f, -0.867618875322536230f, 0.497063440273840310f,
	-0.867714201994605140f,
	0.496897049022654640f, -0.867809496763303210f, 0.496730639501984710f,
	-0.867904759625126920f,
	0.496564211717949340f, -0.867999990576573400f, 0.496397765676667160f,
	-0.868095189614141670f,
	0.496231301384258310f, -0.868190356734331310f, 0.496064818846843060f,
	-0.868285491933643240f,
	0.495898318070542240f, -0.868380595208579800f, 0.495731799061478020f,
	-0.868475666555644120f,
	0.495565261825772490f, -0.868570705971340900f, 0.495398706369549080f,
	-0.868665713452175580f,
	0.495232132698931350f, -0.868760688994655190f, 0.495065540820043610f,
	-0.868855632595287750f,
	0.494898930739011310f, -0.868950544250582380f, 0.494732302461959820f,
	-0.869045423957049530f,
	0.494565655995016010f, -0.869140271711200560f, 0.494398991344306760f,
	-0.869235087509548250f,
	0.494232308515959730f, -0.869329871348606730f, 0.494065607516103730f,
	-0.869424623224890780f,
	0.493898888350867430f, -0.869519343134916970f, 0.493732151026381070f,
	-0.869614031075202300f,
	0.493565395548774880f, -0.869708687042265560f, 0.493398621924179830f,
	-0.869803311032626650f,
	0.493231830158728070f, -0.869897903042806340f, 0.493065020258551650f,
	-0.869992463069326870f,
	0.492898192229784090f, -0.870086991108711350f, 0.492731346078558840f,
	-0.870181487157484560f,
	0.492564481811010650f, -0.870275951212171830f, 0.492397599433274550f,
	-0.870370383269300160f,
	0.492230698951486080f, -0.870464783325397670f, 0.492063780371782060f,
	-0.870559151376993250f,
	0.491896843700299240f, -0.870653487420617540f, 0.491729888943175820f,
	-0.870747791452801790f,
	0.491562916106550060f, -0.870842063470078860f, 0.491395925196560830f,
	-0.870936303468982760f,
	0.491228916219348330f, -0.871030511446048260f, 0.491061889181052590f,
	-0.871124687397811900f,
	0.490894844087815140f, -0.871218831320810900f, 0.490727780945777570f,
	-0.871312943211583920f,
	0.490560699761082080f, -0.871407023066670950f, 0.490393600539872130f,
	-0.871501070882612530f,
	0.490226483288291100f, -0.871595086655951090f, 0.490059348012483910f,
	-0.871689070383229740f,
	0.489892194718595300f, -0.871783022060993010f, 0.489725023412770970f,
	-0.871876941685786890f,
	0.489557834101157550f, -0.871970829254157700f, 0.489390626789901920f,
	-0.872064684762653970f,
	0.489223401485152030f, -0.872158508207824480f, 0.489056158193055980f,
	-0.872252299586219860f,
	0.488888896919763230f, -0.872346058894391540f, 0.488721617671423250f,
	-0.872439786128892280f,
	0.488554320454186230f, -0.872533481286276060f, 0.488387005274203590f,
	-0.872627144363097960f,
	0.488219672137626740f, -0.872720775355914300f, 0.488052321050608310f,
	-0.872814374261282390f,
	0.487884952019301210f, -0.872907941075760970f, 0.487717565049858860f,
	-0.873001475795909920f,
	0.487550160148436050f, -0.873094978418290090f, 0.487382737321187310f,
	-0.873188448939463790f,
	0.487215296574268820f, -0.873281887355994210f, 0.487047837913836550f,
	-0.873375293664446000f,
	0.486880361346047400f, -0.873468667861384880f, 0.486712866877059340f,
	-0.873562009943377740f,
	0.486545354513030270f, -0.873655319906992630f, 0.486377824260119500f,
	-0.873748597748798870f,
	0.486210276124486530f, -0.873841843465366750f, 0.486042710112291390f,
	-0.873935057053268130f,
	0.485875126229695420f, -0.874028238509075630f, 0.485707524482859750f,
	-0.874121387829363330f,
	0.485539904877947020f, -0.874214505010706300f, 0.485372267421119770f,
	-0.874307590049680950f,
	0.485204612118541880f, -0.874400642942864790f, 0.485036938976377450f,
	-0.874493663686836450f,
	0.484869248000791120f, -0.874586652278176110f, 0.484701539197948730f,
	-0.874679608713464510f,
	0.484533812574016120f, -0.874772532989284150f, 0.484366068135160480f,
	-0.874865425102218210f,
	0.484198305887549140f, -0.874958285048851540f, 0.484030525837350010f,
	-0.875051112825769970f,
	0.483862727990732320f, -0.875143908429560250f, 0.483694912353865080f,
	-0.875236671856810870f,
	0.483527078932918740f, -0.875329403104110780f, 0.483359227734063980f,
	-0.875422102168050830f,
	0.483191358763471910f, -0.875514769045222740f, 0.483023472027315050f,
	-0.875607403732219240f,
	0.482855567531765670f, -0.875700006225634600f, 0.482687645282997510f,
	-0.875792576522063880f,
	0.482519705287184520f, -0.875885114618103700f, 0.482351747550501030f,
	-0.875977620510351660f,
	0.482183772079122830f, -0.876070094195406600f, 0.482015778879225530f,
	-0.876162535669868460f,
	0.481847767956986080f, -0.876254944930338400f, 0.481679739318581490f,
	-0.876347321973419020f,
	0.481511692970189920f, -0.876439666795713610f, 0.481343628917989870f,
	-0.876531979393827100f,
	0.481175547168160360f, -0.876624259764365310f, 0.481007447726881640f,
	-0.876716507903935400f,
	0.480839330600333900f, -0.876808723809145760f, 0.480671195794698690f,
	-0.876900907476605650f,
	0.480503043316157670f, -0.876993058902925780f, 0.480334873170893070f,
	-0.877085178084718310f,
	0.480166685365088440f, -0.877177265018595940f, 0.479998479904927220f,
	-0.877269319701173170f,
	0.479830256796594250f, -0.877361342129065140f, 0.479662016046274340f,
	-0.877453332298888560f,
	0.479493757660153060f, -0.877545290207261240f, 0.479325481644417130f,
	-0.877637215850802120f,
	0.479157188005253310f, -0.877729109226131570f, 0.478988876748849550f,
	-0.877820970329870500f,
	0.478820547881394050f, -0.877912799158641730f, 0.478652201409075550f,
	-0.878004595709069080f,
	0.478483837338084080f, -0.878096359977777130f, 0.478315455674609480f,
	-0.878188091961392250f,
	0.478147056424843120f, -0.878279791656541460f, 0.477978639594976110f,
	-0.878371459059853590f,
	0.477810205191201040f, -0.878463094167957870f, 0.477641753219710590f,
	-0.878554696977485340f,
	0.477473283686698060f, -0.878646267485068130f, 0.477304796598358010f,
	-0.878737805687339280f,
	0.477136291960884750f, -0.878829311580933360f, 0.476967769780474230f,
	-0.878920785162485840f,
	0.476799230063322250f, -0.879012226428633410f, 0.476630672815625380f,
	-0.879103635376014330f,
	0.476462098043581310f, -0.879195012001267370f, 0.476293505753387750f,
	-0.879286356301033250f,
	0.476124895951243630f, -0.879377668271953180f, 0.475956268643348220f,
	-0.879468947910670100f,
	0.475787623835901120f, -0.879560195213827890f, 0.475618961535103410f,
	-0.879651410178071470f,
	0.475450281747155870f, -0.879742592800047410f, 0.475281584478260800f,
	-0.879833743076402940f,
	0.475112869734620470f, -0.879924861003786860f, 0.474944137522437860f,
	-0.880015946578848960f,
	0.474775387847917230f, -0.880106999798240360f, 0.474606620717262560f,
	-0.880198020658613190f,
	0.474437836136679340f, -0.880289009156620890f, 0.474269034112372920f,
	-0.880379965288918260f,
	0.474100214650550020f, -0.880470889052160750f, 0.473931377757417560f,
	-0.880561780443005590f,
	0.473762523439182850f, -0.880652639458111010f, 0.473593651702054640f,
	-0.880743466094136230f,
	0.473424762552241530f, -0.880834260347742040f, 0.473255855995953380f,
	-0.880925022215589880f,
	0.473086932039400220f, -0.881015751694342760f, 0.472917990688792760f,
	-0.881106448780665130f,
	0.472749031950342900f, -0.881197113471221980f, 0.472580055830262250f,
	-0.881287745762680100f,
	0.472411062334764100f, -0.881378345651706810f, 0.472242051470061650f,
	-0.881468913134971330f,
	0.472073023242368660f, -0.881559448209143780f, 0.471903977657900320f,
	-0.881649950870895260f,
	0.471734914722871430f, -0.881740421116898320f, 0.471565834443498480f,
	-0.881830858943826620f,
	0.471396736825997810f, -0.881921264348354940f, 0.471227621876586400f,
	-0.882011637327159590f,
	0.471058489601482610f, -0.882101977876917580f, 0.470889340006904520f,
	-0.882192285994307430f,
	0.470720173099071710f, -0.882282561676008600f, 0.470550988884203490f,
	-0.882372804918702290f,
	0.470381787368520710f, -0.882463015719070040f, 0.470212568558244280f,
	-0.882553194073795400f,
	0.470043332459595620f, -0.882643339979562790f, 0.469874079078797470f,
	-0.882733453433057540f,
	0.469704808422072460f, -0.882823534430966730f, 0.469535520495644510f,
	-0.882913582969978020f,
	0.469366215305737630f, -0.883003599046780720f, 0.469196892858576630f,
	-0.883093582658065370f,
	0.469027553160387240f, -0.883183533800523280f, 0.468858196217395330f,
	-0.883273452470847430f,
	0.468688822035827960f, -0.883363338665731580f, 0.468519430621912420f,
	-0.883453192381870920f,
	0.468350021981876530f, -0.883543013615961880f, 0.468180596121949400f,
	-0.883632802364701760f,
	0.468011153048359830f, -0.883722558624789660f, 0.467841692767338220f,
	-0.883812282392925090f,
	0.467672215285114710f, -0.883901973665809470f, 0.467502720607920920f,
	-0.883991632440144890f,
	0.467333208741988530f, -0.884081258712634990f, 0.467163679693549770f,
	-0.884170852479984500f,
	0.466994133468838110f, -0.884260413738899080f, 0.466824570074086950f,
	-0.884349942486086120f,
	0.466654989515530970f, -0.884439438718253700f, 0.466485391799405010f,
	-0.884528902432111350f,
	0.466315776931944480f, -0.884618333624369920f, 0.466146144919386000f,
	-0.884707732291740930f,
	0.465976495767966130f, -0.884797098430937790f, 0.465806829483922770f,
	-0.884886432038674560f,
	0.465637146073493770f, -0.884975733111666660f, 0.465467445542917800f,
	-0.885065001646630930f,
	0.465297727898434650f, -0.885154237640285110f, 0.465127993146283950f,
	-0.885243441089348270f,
	0.464958241292706740f, -0.885332611990540590f, 0.464788472343944160f,
	-0.885421750340583570f,
	0.464618686306237820f, -0.885510856136199950f, 0.464448883185830770f,
	-0.885599929374113360f,
	0.464279062988965760f, -0.885688970051048960f, 0.464109225721887010f,
	-0.885777978163732940f,
	0.463939371390838460f, -0.885866953708892790f, 0.463769500002065680f,
	-0.885955896683257030f,
	0.463599611561814120f, -0.886044807083555490f, 0.463429706076329880f,
	-0.886133684906519340f,
	0.463259783551860260f, -0.886222530148880640f, 0.463089843994652470f,
	-0.886311342807372890f,
	0.462919887410955130f, -0.886400122878730490f, 0.462749913807016850f,
	-0.886488870359689600f,
	0.462579923189086810f, -0.886577585246987040f, 0.462409915563415540f,
	-0.886666267537360890f,
	0.462239890936253280f, -0.886754917227550950f, 0.462069849313851810f,
	-0.886843534314297300f,
	0.461899790702462840f, -0.886932118794342080f, 0.461729715108338770f,
	-0.887020670664428360f,
	0.461559622537733190f, -0.887109189921300060f, 0.461389512996899450f,
	-0.887197676561702900f,
	0.461219386492092430f, -0.887286130582383150f, 0.461049243029567010f,
	-0.887374551980088740f,
	0.460879082615578690f, -0.887462940751568840f, 0.460708905256384190f,
	-0.887551296893573370f,
	0.460538710958240010f, -0.887639620402853930f, 0.460368499727404070f,
	-0.887727911276163020f,
	0.460198271570134270f, -0.887816169510254550f, 0.460028026492689700f,
	-0.887904395101883240f,
	0.459857764501329650f, -0.887992588047805560f, 0.459687485602313870f,
	-0.888080748344778900f,
	0.459517189801903590f, -0.888168875989561620f, 0.459346877106359570f,
	-0.888256970978913870f,
	0.459176547521944150f, -0.888345033309596240f, 0.459006201054919680f,
	-0.888433062978371320f,
	0.458835837711549120f, -0.888521059982002260f, 0.458665457498096670f,
	-0.888609024317253750f,
	0.458495060420826220f, -0.888696955980891710f, 0.458324646486003300f,
	-0.888784854969682850f,
	0.458154215699893230f, -0.888872721280395520f, 0.457983768068762180f,
	-0.888960554909799310f,
	0.457813303598877290f, -0.889048355854664570f, 0.457642822296505770f,
	-0.889136124111763240f,
	0.457472324167916110f, -0.889223859677868210f, 0.457301809219376800f,
	-0.889311562549753850f,
	0.457131277457156980f, -0.889399232724195520f, 0.456960728887527030f,
	-0.889486870197969790f,
	0.456790163516757220f, -0.889574474967854580f, 0.456619581351118960f,
	-0.889662047030628790f,
	0.456448982396883860f, -0.889749586383072890f, 0.456278366660324670f,
	-0.889837093021967900f,
	0.456107734147714220f, -0.889924566944096720f, 0.455937084865326030f,
	-0.890012008146243260f,
	0.455766418819434750f, -0.890099416625192210f, 0.455595736016314920f,
	-0.890186792377730240f,
	0.455425036462242420f, -0.890274135400644480f, 0.455254320163493210f,
	-0.890361445690723730f,
	0.455083587126343840f, -0.890448723244757880f, 0.454912837357072050f,
	-0.890535968059537830f,
	0.454742070861955450f, -0.890623180131855930f, 0.454571287647273000f,
	-0.890710359458505520f,
	0.454400487719303750f, -0.890797506036281490f, 0.454229671084327320f,
	-0.890884619861979530f,
	0.454058837748624540f, -0.890971700932396750f, 0.453887987718476050f,
	-0.891058749244331590f,
	0.453717121000163930f, -0.891145764794583180f, 0.453546237599970260f,
	-0.891232747579952520f,
	0.453375337524177750f, -0.891319697597241390f, 0.453204420779070300f,
	-0.891406614843252900f,
	0.453033487370931580f, -0.891493499314791380f, 0.452862537306046810f,
	-0.891580351008662290f,
	0.452691570590700860f, -0.891667169921672390f, 0.452520587231180100f,
	-0.891753956050629460f,
	0.452349587233771000f, -0.891840709392342720f, 0.452178570604760410f,
	-0.891927429943622510f,
	0.452007537350436530f, -0.892014117701280360f, 0.451836487477087430f,
	-0.892100772662129170f,
	0.451665420991002540f, -0.892187394822982480f, 0.451494337898471210f,
	-0.892273984180655730f,
	0.451323238205783520f, -0.892360540731965360f, 0.451152121919230710f,
	-0.892447064473728680f,
	0.450980989045103810f, -0.892533555402764690f, 0.450809839589695340f,
	-0.892620013515893040f,
	0.450638673559297760f, -0.892706438809935280f, 0.450467490960204110f,
	-0.892792831281713610f,
	0.450296291798708730f, -0.892879190928051680f, 0.450125076081105750f,
	-0.892965517745774260f,
	0.449953843813690580f, -0.893051811731707450f, 0.449782595002758860f,
	-0.893138072882678210f,
	0.449611329654606600f, -0.893224301195515320f, 0.449440047775531260f,
	-0.893310496667048090f,
	0.449268749371829920f, -0.893396659294107610f, 0.449097434449801100f,
	-0.893482789073525850f,
	0.448926103015743260f, -0.893568886002136020f, 0.448754755075956020f,
	-0.893654950076772430f,
	0.448583390636739300f, -0.893740981294271040f, 0.448412009704393430f,
	-0.893826979651468620f,
	0.448240612285220000f, -0.893912945145203250f, 0.448069198385520340f,
	-0.893998877772314240f,
	0.447897768011597310f, -0.894084777529641990f, 0.447726321169753750f,
	-0.894170644414028270f,
	0.447554857866293010f, -0.894256478422316040f, 0.447383378107519710f,
	-0.894342279551349480f,
	0.447211881899738260f, -0.894428047797973800f, 0.447040369249254500f,
	-0.894513783159035620f,
	0.446868840162374330f, -0.894599485631382580f, 0.446697294645404090f,
	-0.894685155211863980f,
	0.446525732704651400f, -0.894770791897329550f, 0.446354154346423840f,
	-0.894856395684630930f,
	0.446182559577030120f, -0.894941966570620750f, 0.446010948402779110f,
	-0.895027504552152630f,
	0.445839320829980350f, -0.895113009626081760f, 0.445667676864944350f,
	-0.895198481789264200f,
	0.445496016513981740f, -0.895283921038557580f, 0.445324339783404240f,
	-0.895369327370820310f,
	0.445152646679523590f, -0.895454700782912450f, 0.444980937208652780f,
	-0.895540041271694840f,
	0.444809211377105000f, -0.895625348834030000f, 0.444637469191193790f,
	-0.895710623466781320f,
	0.444465710657234110f, -0.895795865166813420f, 0.444293935781540580f,
	-0.895881073930992370f,
	0.444122144570429260f, -0.895966249756185110f, 0.443950337030216250f,
	-0.896051392639260040f,
	0.443778513167218220f, -0.896136502577086770f, 0.443606672987753080f,
	-0.896221579566535920f,
	0.443434816498138430f, -0.896306623604479660f, 0.443262943704693380f,
	-0.896391634687790820f,
	0.443091054613736990f, -0.896476612813344010f, 0.442919149231588980f,
	-0.896561557978014960f,
	0.442747227564570130f, -0.896646470178680150f, 0.442575289619001170f,
	-0.896731349412217880f,
	0.442403335401204130f, -0.896816195675507190f, 0.442231364917501090f,
	-0.896901008965428680f,
	0.442059378174214760f, -0.896985789278863970f, 0.441887375177668960f,
	-0.897070536612695870f,
	0.441715355934187310f, -0.897155250963808550f, 0.441543320450094920f,
	-0.897239932329087050f,
	0.441371268731716620f, -0.897324580705418320f, 0.441199200785378660f,
	-0.897409196089689720f,
	0.441027116617407340f, -0.897493778478790190f, 0.440855016234129430f,
	-0.897578327869610230f,
	0.440682899641873020f, -0.897662844259040750f, 0.440510766846965880f,
	-0.897747327643974690f,
	0.440338617855737300f, -0.897831778021305650f, 0.440166452674516480f,
	-0.897916195387928550f,
	0.439994271309633260f, -0.898000579740739880f, 0.439822073767418610f,
	-0.898084931076636780f,
	0.439649860054203420f, -0.898169249392518080f, 0.439477630176319860f,
	-0.898253534685283570f,
	0.439305384140100060f, -0.898337786951834190f, 0.439133121951876930f,
	-0.898422006189072530f,
	0.438960843617984430f, -0.898506192393901840f, 0.438788549144756290f,
	-0.898590345563227030f,
	0.438616238538527710f, -0.898674465693953820f, 0.438443911805633860f,
	-0.898758552782989440f,
	0.438271568952410480f, -0.898842606827242260f, 0.438099209985194580f,
	-0.898926627823621870f,
	0.437926834910322860f, -0.899010615769039070f, 0.437754443734133470f,
	-0.899094570660405770f,
	0.437582036462964340f, -0.899178492494635330f, 0.437409613103154850f,
	-0.899262381268642000f,
	0.437237173661044200f, -0.899346236979341460f, 0.437064718142972370f,
	-0.899430059623650860f,
	0.436892246555280470f, -0.899513849198487870f, 0.436719758904309310f,
	-0.899597605700772180f,
	0.436547255196401250f, -0.899681329127423930f, 0.436374735437898510f,
	-0.899765019475365020f,
	0.436202199635143950f, -0.899848676741518580f, 0.436029647794481670f,
	-0.899932300922808400f,
	0.435857079922255470f, -0.900015892016160280f, 0.435684496024810520f,
	-0.900099450018500340f,
	0.435511896108492170f, -0.900182974926756700f, 0.435339280179646070f,
	-0.900266466737858480f,
	0.435166648244619370f, -0.900349925448735600f, 0.434994000309758710f,
	-0.900433351056319830f,
	0.434821336381412350f, -0.900516743557543520f, 0.434648656465928430f,
	-0.900600102949340790f,
	0.434475960569655710f, -0.900683429228646860f, 0.434303248698944100f,
	-0.900766722392397860f,
	0.434130520860143310f, -0.900849982437531450f, 0.433957777059604480f,
	-0.900933209360986200f,
	0.433785017303678520f, -0.901016403159702330f, 0.433612241598717640f,
	-0.901099563830620950f,
	0.433439449951074200f, -0.901182691370684410f, 0.433266642367100940f,
	-0.901265785776836580f,
	0.433093818853152010f, -0.901348847046022030f, 0.432920979415581220f,
	-0.901431875175186970f,
	0.432748124060743760f, -0.901514870161278630f, 0.432575252794994810f,
	-0.901597832001245660f,
	0.432402365624690140f, -0.901680760692037730f, 0.432229462556186770f,
	-0.901763656230605610f,
	0.432056543595841450f, -0.901846518613901860f, 0.431883608750012300f,
	-0.901929347838879350f,
	0.431710658025057370f, -0.902012143902493070f, 0.431537691427335500f,
	-0.902094906801698900f,
	0.431364708963206440f, -0.902177636533453510f, 0.431191710639030000f,
	-0.902260333094715540f,
	0.431018696461167080f, -0.902342996482444200f, 0.430845666435978820f,
	-0.902425626693600270f,
	0.430672620569826860f, -0.902508223725145830f, 0.430499558869073930f,
	-0.902590787574043870f,
	0.430326481340082610f, -0.902673318237258830f, 0.430153387989216930f,
	-0.902755815711756120f,
	0.429980278822840570f, -0.902838279994502830f, 0.429807153847318770f,
	-0.902920711082466630f,
	0.429634013069016500f, -0.903003108972617040f, 0.429460856494299490f,
	-0.903085473661924600f,
	0.429287684129534720f, -0.903167805147360610f, 0.429114495981088690f,
	-0.903250103425898400f,
	0.428941292055329550f, -0.903332368494511820f, 0.428768072358625240f,
	-0.903414600350176290f,
	0.428594836897344400f, -0.903496798989868450f, 0.428421585677856760f,
	-0.903578964410565950f,
	0.428248318706531910f, -0.903661096609247980f, 0.428075035989740780f,
	-0.903743195582894620f,
	0.427901737533854240f, -0.903825261328487390f, 0.427728423345243860f,
	-0.903907293843009050f,
	0.427555093430282200f, -0.903989293123443340f, 0.427381747795341770f,
	-0.904071259166775440f,
	0.427208386446796370f, -0.904153191969991670f, 0.427035009391019790f,
	-0.904235091530079750f,
	0.426861616634386490f, -0.904316957844028320f, 0.426688208183271970f,
	-0.904398790908827350f,
	0.426514784044051520f, -0.904480590721468250f, 0.426341344223101880f,
	-0.904562357278943190f,
	0.426167888726799620f, -0.904644090578246240f, 0.425994417561522450f,
	-0.904725790616371930f,
	0.425820930733648300f, -0.904807457390316540f, 0.425647428249555590f,
	-0.904889090897077470f,
	0.425473910115623910f, -0.904970691133653250f, 0.425300376338232590f,
	-0.905052258097043590f,
	0.425126826923762410f, -0.905133791784249580f, 0.424953261878594060f,
	-0.905215292192273480f,
	0.424779681209108810f, -0.905296759318118820f, 0.424606084921689220f,
	-0.905378193158789980f,
	0.424432473022717420f, -0.905459593711293250f, 0.424258845518577010f,
	-0.905540960972635480f,
	0.424085202415651670f, -0.905622294939825160f, 0.423911543720325580f,
	-0.905703595609872010f,
	0.423737869438983950f, -0.905784862979786440f, 0.423564179578011960f,
	-0.905866097046580940f,
	0.423390474143796100f, -0.905947297807268460f, 0.423216753142722780f,
	-0.906028465258863490f,
	0.423043016581179100f, -0.906109599398381980f, 0.422869264465553170f,
	-0.906190700222840540f,
	0.422695496802232950f, -0.906271767729257660f, 0.422521713597607870f,
	-0.906352801914652280f,
	0.422347914858067000f, -0.906433802776045460f, 0.422174100590000820f,
	-0.906514770310458800f,
	0.422000270799799790f, -0.906595704514915330f, 0.421826425493854910f,
	-0.906676605386439460f,
	0.421652564678558380f, -0.906757472922056550f, 0.421478688360302220f,
	-0.906838307118793540f,
	0.421304796545479700f, -0.906919107973678030f, 0.421130889240484140f,
	-0.906999875483739610f,
	0.420956966451709440f, -0.907080609646008450f, 0.420783028185550630f,
	-0.907161310457516250f,
	0.420609074448402510f, -0.907241977915295930f, 0.420435105246661220f,
	-0.907322612016381310f,
	0.420261120586723050f, -0.907403212757808000f, 0.420087120474984590f,
	-0.907483780136612570f,
	0.419913104917843730f, -0.907564314149832520f, 0.419739073921698180f,
	-0.907644814794507090f,
	0.419565027492946940f, -0.907725282067676330f, 0.419390965637989050f,
	-0.907805715966381820f,
	0.419216888363223960f, -0.907886116487666150f, 0.419042795675052480f,
	-0.907966483628573240f,
	0.418868687579875110f, -0.908046817386148340f, 0.418694564084093610f,
	-0.908127117757437600f,
	0.418520425194109700f, -0.908207384739488700f, 0.418346270916326310f,
	-0.908287618329350450f,
	0.418172101257146430f, -0.908367818524072780f, 0.417997916222973550f,
	-0.908447985320707250f,
	0.417823715820212380f, -0.908528118716306120f, 0.417649500055267410f,
	-0.908608218707923190f,
	0.417475268934544340f, -0.908688285292613360f, 0.417301022464449060f,
	-0.908768318467432780f,
	0.417126760651387870f, -0.908848318229439120f, 0.416952483501768280f,
	-0.908928284575690640f,
	0.416778191021997590f, -0.909008217503247450f, 0.416603883218484410f,
	-0.909088117009170580f,
	0.416429560097637320f, -0.909167983090522270f, 0.416255221665865480f,
	-0.909247815744366310f,
	0.416080867929579320f, -0.909327614967767260f, 0.415906498895188770f,
	-0.909407380757791260f,
	0.415732114569105420f, -0.909487113111505430f, 0.415557714957740580f,
	-0.909566812025978220f,
	0.415383300067506290f, -0.909646477498279540f, 0.415208869904815650f,
	-0.909726109525480160f,
	0.415034424476081630f, -0.909805708104652220f, 0.414859963787718390f,
	-0.909885273232869160f,
	0.414685487846140010f, -0.909964804907205660f, 0.414510996657761810f,
	-0.910044303124737390f,
	0.414336490228999210f, -0.910123767882541570f, 0.414161968566268080f,
	-0.910203199177696540f,
	0.413987431675985510f, -0.910282597007281760f, 0.413812879564568300f,
	-0.910361961368377990f,
	0.413638312238434560f, -0.910441292258067140f, 0.413463729704002580f,
	-0.910520589673432630f,
	0.413289131967690960f, -0.910599853611558930f, 0.413114519035919560f,
	-0.910679084069531570f,
	0.412939890915108020f, -0.910758281044437570f, 0.412765247611677320f,
	-0.910837444533365010f,
	0.412590589132048380f, -0.910916574533403240f, 0.412415915482642730f,
	-0.910995671041643140f,
	0.412241226669883000f, -0.911074734055176250f, 0.412066522700191560f,
	-0.911153763571095900f,
	0.411891803579992220f, -0.911232759586496190f, 0.411717069315708670f,
	-0.911311722098472670f,
	0.411542319913765280f, -0.911390651104122320f, 0.411367555380587340f,
	-0.911469546600543020f,
	0.411192775722600160f, -0.911548408584833990f, 0.411017980946230270f,
	-0.911627237054095650f,
	0.410843171057903910f, -0.911706032005429880f, 0.410668346064048780f,
	-0.911784793435939430f,
	0.410493505971092520f, -0.911863521342728520f, 0.410318650785463260f,
	-0.911942215722902570f,
	0.410143780513590350f, -0.912020876573568230f, 0.409968895161902820f,
	-0.912099503891833470f,
	0.409793994736831200f, -0.912178097674807060f, 0.409619079244805840f,
	-0.912256657919599650f,
	0.409444148692257590f, -0.912335184623322750f, 0.409269203085618700f,
	-0.912413677783089020f,
	0.409094242431320920f, -0.912492137396012650f, 0.408919266735797480f,
	-0.912570563459208730f,
	0.408744276005481520f, -0.912648955969793900f, 0.408569270246806780f,
	-0.912727314924885900f,
	0.408394249466208110f, -0.912805640321603500f, 0.408219213670120100f,
	-0.912883932157067200f,
	0.408044162864978740f, -0.912962190428398100f, 0.407869097057219960f,
	-0.913040415132719160f,
	0.407694016253280170f, -0.913118606267154130f, 0.407518920459597030f,
	-0.913196763828828200f,
	0.407343809682607970f, -0.913274887814867760f, 0.407168683928751610f,
	-0.913352978222400250f,
	0.406993543204466460f, -0.913431035048554720f, 0.406818387516192370f,
	-0.913509058290461140f,
	0.406643216870369140f, -0.913587047945250810f, 0.406468031273437000f,
	-0.913665004010056350f,
	0.406292830731837470f, -0.913742926482011390f, 0.406117615252011790f,
	-0.913820815358251100f,
	0.405942384840402570f, -0.913898670635911680f, 0.405767139503452220f,
	-0.913976492312130520f,
	0.405591879247603870f, -0.914054280384046460f, 0.405416604079301750f,
	-0.914132034848799460f,
	0.405241314004989860f, -0.914209755703530690f, 0.405066009031113390f,
	-0.914287442945382440f,
	0.404890689164117750f, -0.914365096571498450f, 0.404715354410448650f,
	-0.914442716579023870f,
	0.404540004776553110f, -0.914520302965104450f, 0.404364640268877810f,
	-0.914597855726887790f,
	0.404189260893870750f, -0.914675374861522390f, 0.404013866657980060f,
	-0.914752860366158100f,
	0.403838457567654130f, -0.914830312237946090f, 0.403663033629342750f,
	-0.914907730474038620f,
	0.403487594849495310f, -0.914985115071589310f, 0.403312141234562660f,
	-0.915062466027752760f,
	0.403136672790995240f, -0.915139783339685260f, 0.402961189525244960f,
	-0.915217067004543750f,
	0.402785691443763640f, -0.915294317019487050f, 0.402610178553003680f,
	-0.915371533381674760f,
	0.402434650859418540f, -0.915448716088267830f, 0.402259108369461440f,
	-0.915525865136428530f,
	0.402083551089587040f, -0.915602980523320230f, 0.401907979026249860f,
	-0.915680062246107650f,
	0.401732392185905010f, -0.915757110301956720f, 0.401556790575008650f,
	-0.915834124688034710f,
	0.401381174200016790f, -0.915911105401509880f, 0.401205543067386760f,
	-0.915988052439551840f,
	0.401029897183575790f, -0.916064965799331610f, 0.400854236555041650f,
	-0.916141845478021350f,
	0.400678561188243350f, -0.916218691472794110f, 0.400502871089639500f,
	-0.916295503780824800f,
	0.400327166265690150f, -0.916372282399289140f, 0.400151446722855300f,
	-0.916449027325364040f,
	0.399975712467595390f, -0.916525738556228100f, 0.399799963506372090f,
	-0.916602416089060680f,
	0.399624199845646790f, -0.916679059921042700f, 0.399448421491882260f,
	-0.916755670049355990f,
	0.399272628451540930f, -0.916832246471183890f, 0.399096820731086600f,
	-0.916908789183710990f,
	0.398920998336983020f, -0.916985298184122890f, 0.398745161275694480f,
	-0.917061773469606820f,
	0.398569309553686360f, -0.917138215037350710f, 0.398393443177423920f,
	-0.917214622884544250f,
	0.398217562153373620f, -0.917290997008377910f, 0.398041666488001930f,
	-0.917367337406043810f,
	0.397865756187775750f, -0.917443644074735220f, 0.397689831259163240f,
	-0.917519917011646260f,
	0.397513891708632330f, -0.917596156213972950f, 0.397337937542652120f,
	-0.917672361678911750f,
	0.397161968767691720f, -0.917748533403661250f, 0.396985985390220900f,
	-0.917824671385420570f,
	0.396809987416710420f, -0.917900775621390390f, 0.396633974853630830f,
	-0.917976846108772730f,
	0.396457947707453960f, -0.918052882844770380f, 0.396281905984651680f,
	-0.918128885826587910f,
	0.396105849691696320f, -0.918204855051430900f, 0.395929778835061360f,
	-0.918280790516506130f,
	0.395753693421220080f, -0.918356692219021720f, 0.395577593456646950f,
	-0.918432560156186790f,
	0.395401478947816300f, -0.918508394325212250f, 0.395225349901203730f,
	-0.918584194723309540f,
	0.395049206323284880f, -0.918659961347691900f, 0.394873048220535760f,
	-0.918735694195573550f,
	0.394696875599433670f, -0.918811393264169940f, 0.394520688466455550f,
	-0.918887058550697970f,
	0.394344486828079650f, -0.918962690052375630f, 0.394168270690784250f,
	-0.919038287766421940f,
	0.393992040061048100f, -0.919113851690057770f, 0.393815794945351130f,
	-0.919189381820504470f,
	0.393639535350172880f, -0.919264878154985250f, 0.393463261281994380f,
	-0.919340340690724230f,
	0.393286972747296570f, -0.919415769424946960f, 0.393110669752560760f,
	-0.919491164354880100f,
	0.392934352304269600f, -0.919566525477751530f, 0.392758020408905280f,
	-0.919641852790790470f,
	0.392581674072951530f, -0.919717146291227360f, 0.392405313302891860f,
	-0.919792405976293750f,
	0.392228938105210370f, -0.919867631843222950f, 0.392052548486392200f,
	-0.919942823889248640f,
	0.391876144452922350f, -0.920017982111606570f, 0.391699726011287050f,
	-0.920093106507533070f,
	0.391523293167972350f, -0.920168197074266450f, 0.391346845929465610f,
	-0.920243253809045370f,
	0.391170384302253980f, -0.920318276709110480f, 0.390993908292825380f,
	-0.920393265771703550f,
	0.390817417907668610f, -0.920468220994067110f, 0.390640913153272370f,
	-0.920543142373445480f,
	0.390464394036126650f, -0.920618029907083860f, 0.390287860562721360f,
	-0.920692883592229010f,
	0.390111312739546910f, -0.920767703426128790f, 0.389934750573094790f,
	-0.920842489406032080f,
	0.389758174069856410f, -0.920917241529189520f, 0.389581583236324360f,
	-0.920991959792852310f,
	0.389404978078991100f, -0.921066644194273530f, 0.389228358604349730f,
	-0.921141294730707270f,
	0.389051724818894500f, -0.921215911399408730f, 0.388875076729119250f,
	-0.921290494197634540f,
	0.388698414341519250f, -0.921365043122642340f, 0.388521737662589740f,
	-0.921439558171691320f,
	0.388345046698826300f, -0.921514039342041900f, 0.388168341456725850f,
	-0.921588486630955380f,
	0.387991621942784910f, -0.921662900035694730f, 0.387814888163501290f,
	-0.921737279553523800f,
	0.387638140125372680f, -0.921811625181708120f, 0.387461377834897920f,
	-0.921885936917513970f,
	0.387284601298575890f, -0.921960214758209110f, 0.387107810522905990f,
	-0.922034458701062820f,
	0.386931005514388690f, -0.922108668743345070f, 0.386754186279524130f,
	-0.922182844882327600f,
	0.386577352824813980f, -0.922256987115283030f, 0.386400505156759610f,
	-0.922331095439485330f,
	0.386223643281862980f, -0.922405169852209880f, 0.386046767206627280f,
	-0.922479210350733100f,
	0.385869876937555310f, -0.922553216932332830f, 0.385692972481151200f,
	-0.922627189594287800f,
	0.385516053843919020f, -0.922701128333878520f, 0.385339121032363340f,
	-0.922775033148386380f,
	0.385162174052989970f, -0.922848904035094120f, 0.384985212912304200f,
	-0.922922740991285680f,
	0.384808237616812930f, -0.922996544014246250f, 0.384631248173022740f,
	-0.923070313101262420f,
	0.384454244587440870f, -0.923144048249621820f, 0.384277226866575620f,
	-0.923217749456613500f,
	0.384100195016935040f, -0.923291416719527640f, 0.383923149045028500f,
	-0.923365050035655610f,
	0.383746088957365010f, -0.923438649402290370f, 0.383569014760454960f,
	-0.923512214816725520f,
	0.383391926460808770f, -0.923585746276256560f, 0.383214824064937180f,
	-0.923659243778179980f,
	0.383037707579352130f, -0.923732707319793180f, 0.382860577010565360f,
	-0.923806136898395410f,
	0.382683432365089840f, -0.923879532511286740f, 0.382506273649438400f,
	-0.923952894155768640f,
	0.382329100870124510f, -0.924026221829143850f, 0.382151914033662720f,
	-0.924099515528716280f,
	0.381974713146567220f, -0.924172775251791200f, 0.381797498215353690f,
	-0.924246000995674890f,
	0.381620269246537520f, -0.924319192757675160f, 0.381443026246634730f,
	-0.924392350535101050f,
	0.381265769222162490f, -0.924465474325262600f, 0.381088498179637520f,
	-0.924538564125471420f,
	0.380911213125578130f, -0.924611619933039970f, 0.380733914066502090f,
	-0.924684641745282530f,
	0.380556601008928570f, -0.924757629559513910f, 0.380379273959376710f,
	-0.924830583373050800f,
	0.380201932924366050f, -0.924903503183210910f, 0.380024577910417380f,
	-0.924976388987313050f,
	0.379847208924051110f, -0.925049240782677580f, 0.379669825971789000f,
	-0.925122058566625770f,
	0.379492429060152740f, -0.925194842336480420f, 0.379315018195664430f,
	-0.925267592089565550f,
	0.379137593384847430f, -0.925340307823206200f, 0.378960154634224720f,
	-0.925412989534729060f,
	0.378782701950320600f, -0.925485637221461490f, 0.378605235339659290f,
	-0.925558250880732620f,
	0.378427754808765620f, -0.925630830509872720f, 0.378250260364165310f,
	-0.925703376106213120f,
	0.378072752012383990f, -0.925775887667086740f, 0.377895229759948550f,
	-0.925848365189827270f,
	0.377717693613385810f, -0.925920808671769960f, 0.377540143579222940f,
	-0.925993218110251480f,
	0.377362579663988450f, -0.926065593502609310f, 0.377185001874210450f,
	-0.926137934846182560f,
	0.377007410216418310f, -0.926210242138311270f, 0.376829804697141220f,
	-0.926282515376337210f,
	0.376652185322909620f, -0.926354754557602860f, 0.376474552100253880f,
	-0.926426959679452100f,
	0.376296905035704790f, -0.926499130739230510f, 0.376119244135794390f,
	-0.926571267734284220f,
	0.375941569407054420f, -0.926643370661961230f, 0.375763880856017750f,
	-0.926715439519610330f,
	0.375586178489217330f, -0.926787474304581750f, 0.375408462313186590f,
	-0.926859475014227160f,
	0.375230732334460030f, -0.926931441645899130f, 0.375052988559571860f,
	-0.927003374196951670f,
	0.374875230995057600f, -0.927075272664740100f, 0.374697459647452770f,
	-0.927147137046620880f,
	0.374519674523293210f, -0.927218967339951790f, 0.374341875629116030f,
	-0.927290763542091720f,
	0.374164062971457990f, -0.927362525650401110f, 0.373986236556857090f,
	-0.927434253662241300f,
	0.373808396391851370f, -0.927505947574975180f, 0.373630542482979280f,
	-0.927577607385966730f,
	0.373452674836780410f, -0.927649233092581180f, 0.373274793459794030f,
	-0.927720824692185200f,
	0.373096898358560690f, -0.927792382182146320f, 0.372918989539620770f,
	-0.927863905559833780f,
	0.372741067009515810f, -0.927935394822617890f, 0.372563130774787370f,
	-0.928006849967869970f,
	0.372385180841977360f, -0.928078270992963140f, 0.372207217217628950f,
	-0.928149657895271150f,
	0.372029239908284960f, -0.928221010672169440f, 0.371851248920489540f,
	-0.928292329321034560f,
	0.371673244260786630f, -0.928363613839244370f, 0.371495225935720760f,
	-0.928434864224177980f,
	0.371317193951837600f, -0.928506080473215480f, 0.371139148315682510f,
	-0.928577262583738850f,
	0.370961089033802040f, -0.928648410553130520f, 0.370783016112742720f,
	-0.928719524378774700f,
	0.370604929559051670f, -0.928790604058057020f, 0.370426829379276900f,
	-0.928861649588363700f,
	0.370248715579966360f, -0.928932660967082820f, 0.370070588167669130f,
	-0.929003638191603360f,
	0.369892447148934270f, -0.929074581259315750f, 0.369714292530311240f,
	-0.929145490167611720f,
	0.369536124318350760f, -0.929216364913883930f, 0.369357942519603190f,
	-0.929287205495526790f,
	0.369179747140620070f, -0.929358011909935500f, 0.369001538187952780f,
	-0.929428784154506800f,
	0.368823315668153960f, -0.929499522226638560f, 0.368645079587776150f,
	-0.929570226123729860f,
	0.368466829953372320f, -0.929640895843181330f, 0.368288566771496680f,
	-0.929711531382394370f,
	0.368110290048703050f, -0.929782132738772190f, 0.367931999791546500f,
	-0.929852699909718750f,
	0.367753696006582090f, -0.929923232892639560f, 0.367575378700365330f,
	-0.929993731684941480f,
	0.367397047879452820f, -0.930064196284032360f, 0.367218703550400930f,
	-0.930134626687321390f,
	0.367040345719767240f, -0.930205022892219070f, 0.366861974394109220f,
	-0.930275384896137040f,
	0.366683589579984930f, -0.930345712696488470f, 0.366505191283953480f,
	-0.930416006290687550f,
	0.366326779512573590f, -0.930486265676149780f, 0.366148354272405390f,
	-0.930556490850291800f,
	0.365969915570008910f, -0.930626681810531650f, 0.365791463411944570f,
	-0.930696838554288860f,
	0.365612997804773960f, -0.930766961078983710f, 0.365434518755058390f,
	-0.930837049382038150f,
	0.365256026269360380f, -0.930907103460875020f, 0.365077520354242180f,
	-0.930977123312918930f,
	0.364899001016267380f, -0.931047108935595170f, 0.364720468261999390f,
	-0.931117060326330790f,
	0.364541922098002180f, -0.931186977482553750f, 0.364363362530840730f,
	-0.931256860401693420f,
	0.364184789567079840f, -0.931326709081180430f, 0.364006203213285530f,
	-0.931396523518446600f,
	0.363827603476023610f, -0.931466303710925090f, 0.363648990361860550f,
	-0.931536049656050300f,
	0.363470363877363870f, -0.931605761351257830f, 0.363291724029100700f,
	-0.931675438793984620f,
	0.363113070823639530f, -0.931745081981668720f, 0.362934404267548750f,
	-0.931814690911749620f,
	0.362755724367397230f, -0.931884265581668150f, 0.362577031129754870f,
	-0.931953805988865900f,
	0.362398324561191310f, -0.932023312130786490f, 0.362219604668277570f,
	-0.932092784004874050f,
	0.362040871457584350f, -0.932162221608574320f, 0.361862124935682980f,
	-0.932231624939334540f,
	0.361683365109145950f, -0.932300993994602640f, 0.361504591984545260f,
	-0.932370328771828460f,
	0.361325805568454340f, -0.932439629268462360f, 0.361147005867446190f,
	-0.932508895481956700f,
	0.360968192888095290f, -0.932578127409764420f, 0.360789366636975690f,
	-0.932647325049340340f,
	0.360610527120662270f, -0.932716488398140250f, 0.360431674345730810f,
	-0.932785617453620990f,
	0.360252808318756830f, -0.932854712213241230f, 0.360073929046317080f,
	-0.932923772674460140f,
	0.359895036534988280f, -0.932992798834738850f, 0.359716130791347570f,
	-0.933061790691539380f,
	0.359537211821973180f, -0.933130748242325110f, 0.359358279633443080f,
	-0.933199671484560730f,
	0.359179334232336560f, -0.933268560415712050f, 0.359000375625232630f,
	-0.933337415033246080f,
	0.358821403818710920f, -0.933406235334631520f, 0.358642418819352100f,
	-0.933475021317337950f,
	0.358463420633736540f, -0.933543772978836170f, 0.358284409268445900f,
	-0.933612490316598540f,
	0.358105384730061760f, -0.933681173328098300f, 0.357926347025166070f,
	-0.933749822010810580f,
	0.357747296160342010f, -0.933818436362210960f, 0.357568232142172260f,
	-0.933887016379776890f,
	0.357389154977241000f, -0.933955562060986730f, 0.357210064672131900f,
	-0.934024073403320500f,
	0.357030961233430030f, -0.934092550404258870f, 0.356851844667720410f,
	-0.934160993061284420f,
	0.356672714981588260f, -0.934229401371880820f, 0.356493572181620200f,
	-0.934297775333532530f,
	0.356314416274402360f, -0.934366114943725900f, 0.356135247266522180f,
	-0.934434420199948050f,
	0.355956065164567010f, -0.934502691099687870f, 0.355776869975124640f,
	-0.934570927640435030f,
	0.355597661704783960f, -0.934639129819680780f, 0.355418440360133590f,
	-0.934707297634917440f,
	0.355239205947763370f, -0.934775431083638700f, 0.355059958474263030f,
	-0.934843530163339430f,
	0.354880697946222790f, -0.934911594871516090f, 0.354701424370233940f,
	-0.934979625205665800f,
	0.354522137752887430f, -0.935047621163287430f, 0.354342838100775600f,
	-0.935115582741880890f,
	0.354163525420490510f, -0.935183509938947500f, 0.353984199718624830f,
	-0.935251402751989810f,
	0.353804861001772160f, -0.935319261178511500f, 0.353625509276525970f,
	-0.935387085216017770f,
	0.353446144549480870f, -0.935454874862014620f, 0.353266766827231180f,
	-0.935522630114009930f,
	0.353087376116372530f, -0.935590350969512370f, 0.352907972423500360f,
	-0.935658037426032040f,
	0.352728555755210730f, -0.935725689481080370f, 0.352549126118100580f,
	-0.935793307132169900f,
	0.352369683518766630f, -0.935860890376814640f, 0.352190227963806890f,
	-0.935928439212529660f,
	0.352010759459819240f, -0.935995953636831300f, 0.351831278013402030f,
	-0.936063433647237540f,
	0.351651783631154680f, -0.936130879241266920f, 0.351472276319676260f,
	-0.936198290416440090f,
	0.351292756085567150f, -0.936265667170278260f, 0.351113222935427630f,
	-0.936333009500304180f,
	0.350933676875858360f, -0.936400317404042060f, 0.350754117913461170f,
	-0.936467590879016880f,
	0.350574546054837570f, -0.936534829922755500f, 0.350394961306590200f,
	-0.936602034532785570f,
	0.350215363675321740f, -0.936669204706636060f, 0.350035753167635300f,
	-0.936736340441837620f,
	0.349856129790135030f, -0.936803441735921560f, 0.349676493549424760f,
	-0.936870508586420960f,
	0.349496844452109600f, -0.936937540990869900f, 0.349317182504794320f,
	-0.937004538946803690f,
	0.349137507714085030f, -0.937071502451759190f, 0.348957820086587600f,
	-0.937138431503274140f,
	0.348778119628908420f, -0.937205326098887960f, 0.348598406347655040f,
	-0.937272186236140950f,
	0.348418680249434510f, -0.937339011912574960f, 0.348238941340855310f,
	-0.937405803125732850f,
	0.348059189628525780f, -0.937472559873159140f, 0.347879425119054510f,
	-0.937539282152399230f,
	0.347699647819051490f, -0.937605969960999990f, 0.347519857735126110f,
	-0.937672623296509470f,
	0.347340054873889190f, -0.937739242156476970f, 0.347160239241951330f,
	-0.937805826538453010f,
	0.346980410845923680f, -0.937872376439989890f, 0.346800569692418400f,
	-0.937938891858640210f,
	0.346620715788047320f, -0.938005372791958840f, 0.346440849139423580f,
	-0.938071819237501160f,
	0.346260969753160170f, -0.938138231192824360f, 0.346081077635870480f,
	-0.938204608655486490f,
	0.345901172794169100f, -0.938270951623047080f, 0.345721255234670120f,
	-0.938337260093066950f,
	0.345541324963989150f, -0.938403534063108060f, 0.345361381988741170f,
	-0.938469773530733800f,
	0.345181426315542610f, -0.938535978493508560f, 0.345001457951009780f,
	-0.938602148948998290f,
	0.344821476901759290f, -0.938668284894770170f, 0.344641483174409070f,
	-0.938734386328392460f,
	0.344461476775576480f, -0.938800453247434770f, 0.344281457711880230f,
	-0.938866485649468060f,
	0.344101425989938980f, -0.938932483532064490f, 0.343921381616371700f,
	-0.938998446892797540f,
	0.343741324597798600f, -0.939064375729241950f, 0.343561254940839330f,
	-0.939130270038973650f,
	0.343381172652115100f, -0.939196129819569900f, 0.343201077738246710f,
	-0.939261955068609100f,
	0.343020970205855540f, -0.939327745783671400f, 0.342840850061564060f,
	-0.939393501962337510f,
	0.342660717311994380f, -0.939459223602189920f, 0.342480571963769850f,
	-0.939524910700812120f,
	0.342300414023513690f, -0.939590563255789160f, 0.342120243497849590f,
	-0.939656181264707070f,
	0.341940060393402300f, -0.939721764725153340f, 0.341759864716796310f,
	-0.939787313634716570f,
	0.341579656474657210f, -0.939852827990986680f, 0.341399435673610360f,
	-0.939918307791555050f,
	0.341219202320282410f, -0.939983753034013940f, 0.341038956421299830f,
	-0.940049163715957370f,
	0.340858697983289440f, -0.940114539834980280f, 0.340678427012879310f,
	-0.940179881388678810f,
	0.340498143516697100f, -0.940245188374650880f, 0.340317847501371730f,
	-0.940310460790495070f,
	0.340137538973531880f, -0.940375698633811540f, 0.339957217939806880f,
	-0.940440901902201750f,
	0.339776884406826960f, -0.940506070593268300f, 0.339596538381222060f,
	-0.940571204704615190f,
	0.339416179869623410f, -0.940636304233847590f, 0.339235808878662120f,
	-0.940701369178571940f,
	0.339055425414969640f, -0.940766399536396070f, 0.338875029485178560f,
	-0.940831395304928870f,
	0.338694621095921190f, -0.940896356481780830f, 0.338514200253831000f,
	-0.940961283064563280f,
	0.338333766965541290f, -0.941026175050889260f, 0.338153321237685990f,
	-0.941091032438372780f,
	0.337972863076899830f, -0.941155855224629190f, 0.337792392489817460f,
	-0.941220643407275180f,
	0.337611909483074680f, -0.941285396983928660f, 0.337431414063306790f,
	-0.941350115952208970f,
	0.337250906237150650f, -0.941414800309736230f, 0.337070386011242730f,
	-0.941479450054132580f,
	0.336889853392220050f, -0.941544065183020810f, 0.336709308386720700f,
	-0.941608645694025140f,
	0.336528751001382350f, -0.941673191584771360f, 0.336348181242844100f,
	-0.941737702852886160f,
	0.336167599117744690f, -0.941802179495997650f, 0.335987004632723350f,
	-0.941866621511735280f,
	0.335806397794420560f, -0.941931028897729510f, 0.335625778609476230f,
	-0.941995401651612550f,
	0.335445147084531660f, -0.942059739771017310f, 0.335264503226227970f,
	-0.942124043253578460f,
	0.335083847041206580f, -0.942188312096931770f, 0.334903178536110290f,
	-0.942252546298714020f,
	0.334722497717581220f, -0.942316745856563780f, 0.334541804592262960f,
	-0.942380910768120470f,
	0.334361099166798900f, -0.942445041031024890f, 0.334180381447832740f,
	-0.942509136642919240f,
	0.333999651442009490f, -0.942573197601446870f, 0.333818909155973620f,
	-0.942637223904252530f,
	0.333638154596370920f, -0.942701215548981900f, 0.333457387769846790f,
	-0.942765172533282510f,
	0.333276608683047980f, -0.942829094854802710f, 0.333095817342620890f,
	-0.942892982511192130f,
	0.332915013755212650f, -0.942956835500102120f, 0.332734197927471160f,
	-0.943020653819184650f,
	0.332553369866044220f, -0.943084437466093490f, 0.332372529577580680f,
	-0.943148186438483420f,
	0.332191677068729320f, -0.943211900734010620f, 0.332010812346139380f,
	-0.943275580350332540f,
	0.331829935416461220f, -0.943339225285107720f, 0.331649046286344620f,
	-0.943402835535996240f,
	0.331468144962440920f, -0.943466411100659320f, 0.331287231451400990f,
	-0.943529951976759370f,
	0.331106305759876430f, -0.943593458161960390f, 0.330925367894519650f,
	-0.943656929653927110f,
	0.330744417861982890f, -0.943720366450326200f, 0.330563455668919590f,
	-0.943783768548825060f,
	0.330382481321982950f, -0.943847135947092690f, 0.330201494827826620f,
	-0.943910468642799150f,
	0.330020496193105530f, -0.943973766633615980f, 0.329839485424473940f,
	-0.944037029917215830f,
	0.329658462528587550f, -0.944100258491272660f, 0.329477427512101680f,
	-0.944163452353461770f,
	0.329296380381672800f, -0.944226611501459810f, 0.329115321143957360f,
	-0.944289735932944410f,
	0.328934249805612200f, -0.944352825645594750f, 0.328753166373295100f,
	-0.944415880637091250f,
	0.328572070853663690f, -0.944478900905115550f, 0.328390963253376630f,
	-0.944541886447350380f,
	0.328209843579092660f, -0.944604837261480260f, 0.328028711837470730f,
	-0.944667753345190490f,
	0.327847568035170960f, -0.944730634696167800f, 0.327666412178853060f,
	-0.944793481312100280f,
	0.327485244275178060f, -0.944856293190677210f, 0.327304064330806830f,
	-0.944919070329589220f,
	0.327122872352400510f, -0.944981812726528150f, 0.326941668346621530f,
	-0.945044520379187070f,
	0.326760452320131790f, -0.945107193285260610f, 0.326579224279594460f,
	-0.945169831442444150f,
	0.326397984231672660f, -0.945232434848434890f, 0.326216732183029770f,
	-0.945295003500931100f,
	0.326035468140330350f, -0.945357537397632290f, 0.325854192110238580f,
	-0.945420036536239070f,
	0.325672904099419900f, -0.945482500914453740f, 0.325491604114539260f,
	-0.945544930529979680f,
	0.325310292162262980f, -0.945607325380521280f, 0.325128968249257190f,
	-0.945669685463784710f,
	0.324947632382188430f, -0.945732010777477150f, 0.324766284567724330f,
	-0.945794301319306860f,
	0.324584924812532150f, -0.945856557086983910f, 0.324403553123280290f,
	-0.945918778078219110f,
	0.324222169506637130f, -0.945980964290724760f, 0.324040773969271450f,
	-0.946043115722214560f,
	0.323859366517852960f, -0.946105232370403340f, 0.323677947159051180f,
	-0.946167314233007370f,
	0.323496515899536760f, -0.946229361307743820f, 0.323315072745980150f,
	-0.946291373592331510f,
	0.323133617705052330f, -0.946353351084490590f, 0.322952150783425370f,
	-0.946415293781942110f,
	0.322770671987770710f, -0.946477201682408680f, 0.322589181324761390f,
	-0.946539074783614100f,
	0.322407678801070020f, -0.946600913083283530f, 0.322226164423369650f,
	-0.946662716579143360f,
	0.322044638198334620f, -0.946724485268921170f, 0.321863100132638580f,
	-0.946786219150346000f,
	0.321681550232956640f, -0.946847918221148000f, 0.321499988505963450f,
	-0.946909582479058760f,
	0.321318414958334910f, -0.946971211921810880f, 0.321136829596746780f,
	-0.947032806547138620f,
	0.320955232427875210f, -0.947094366352777220f, 0.320773623458397440f,
	-0.947155891336463270f,
	0.320592002694990330f, -0.947217381495934820f, 0.320410370144331880f,
	-0.947278836828930880f,
	0.320228725813100020f, -0.947340257333191940f, 0.320047069707973140f,
	-0.947401643006459900f,
	0.319865401835630610f, -0.947462993846477700f, 0.319683722202751370f,
	-0.947524309850989570f,
	0.319502030816015750f, -0.947585591017741090f, 0.319320327682103720f,
	-0.947646837344479190f,
	0.319138612807695900f, -0.947708048828952100f, 0.318956886199473770f,
	-0.947769225468909180f,
	0.318775147864118480f, -0.947830367262101010f, 0.318593397808312470f,
	-0.947891474206279730f,
	0.318411636038737960f, -0.947952546299198560f, 0.318229862562077580f,
	-0.948013583538612200f,
	0.318048077385015060f, -0.948074585922276230f, 0.317866280514233660f,
	-0.948135553447947980f,
	0.317684471956418020f, -0.948196486113385580f, 0.317502651718252260f,
	-0.948257383916349060f,
	0.317320819806421790f, -0.948318246854599090f, 0.317138976227611890f,
	-0.948379074925898120f,
	0.316957120988508150f, -0.948439868128009620f, 0.316775254095797380f,
	-0.948500626458698260f,
	0.316593375556165850f, -0.948561349915730270f, 0.316411485376301090f,
	-0.948622038496872990f,
	0.316229583562890490f, -0.948682692199895090f, 0.316047670122621860f,
	-0.948743311022566480f,
	0.315865745062184070f, -0.948803894962658380f, 0.315683808388265600f,
	-0.948864444017943340f,
	0.315501860107556040f, -0.948924958186195160f, 0.315319900226745050f,
	-0.948985437465188710f,
	0.315137928752522440f, -0.949045881852700560f, 0.314955945691579250f,
	-0.949106291346508260f,
	0.314773951050606070f, -0.949166665944390700f, 0.314591944836294710f,
	-0.949227005644128210f,
	0.314409927055336820f, -0.949287310443502010f, 0.314227897714424500f,
	-0.949347580340295210f,
	0.314045856820250820f, -0.949407815332291460f, 0.313863804379508500f,
	-0.949468015417276550f,
	0.313681740398891570f, -0.949528180593036670f, 0.313499664885093450f,
	-0.949588310857359950f,
	0.313317577844809070f, -0.949648406208035480f, 0.313135479284732950f,
	-0.949708466642853800f,
	0.312953369211560200f, -0.949768492159606680f, 0.312771247631986880f,
	-0.949828482756087000f,
	0.312589114552708660f, -0.949888438430089300f, 0.312406969980422500f,
	-0.949948359179409010f,
	0.312224813921825050f, -0.950008245001843000f, 0.312042646383613510f,
	-0.950068095895189590f,
	0.311860467372486130f, -0.950127911857248100f, 0.311678276895140550f,
	-0.950187692885819280f,
	0.311496074958275970f, -0.950247438978705230f, 0.311313861568591090f,
	-0.950307150133709140f,
	0.311131636732785270f, -0.950366826348635780f, 0.310949400457558760f,
	-0.950426467621290900f,
	0.310767152749611470f, -0.950486073949481700f, 0.310584893615644560f,
	-0.950545645331016600f,
	0.310402623062358880f, -0.950605181763705230f, 0.310220341096455910f,
	-0.950664683245358910f,
	0.310038047724638000f, -0.950724149773789610f, 0.309855742953607130f,
	-0.950783581346811070f,
	0.309673426790066490f, -0.950842977962238160f, 0.309491099240719050f,
	-0.950902339617887060f,
	0.309308760312268780f, -0.950961666311575080f, 0.309126410011419550f,
	-0.951020958041121080f,
	0.308944048344875710f, -0.951080214804345010f, 0.308761675319342570f,
	-0.951139436599068190f,
	0.308579290941525030f, -0.951198623423113230f, 0.308396895218129240f,
	-0.951257775274304000f,
	0.308214488155861220f, -0.951316892150465550f, 0.308032069761427330f,
	-0.951375974049424420f,
	0.307849640041534980f, -0.951435020969008340f, 0.307667199002891190f,
	-0.951494032907046370f,
	0.307484746652204160f, -0.951553009861368590f, 0.307302282996181950f,
	-0.951611951829806730f,
	0.307119808041533100f, -0.951670858810193860f, 0.306937321794967020f,
	-0.951729730800363720f,
	0.306754824263192780f, -0.951788567798152130f, 0.306572315452920800f,
	-0.951847369801395620f,
	0.306389795370861080f, -0.951906136807932230f, 0.306207264023724280f,
	-0.951964868815601380f,
	0.306024721418221900f, -0.952023565822243570f, 0.305842167561065080f,
	-0.952082227825700620f,
	0.305659602458966230f, -0.952140854823815830f, 0.305477026118637360f,
	-0.952199446814433580f,
	0.305294438546791720f, -0.952258003795399600f, 0.305111839750142220f,
	-0.952316525764560830f,
	0.304929229735402430f, -0.952375012719765880f, 0.304746608509286640f,
	-0.952433464658864030f,
	0.304563976078509050f, -0.952491881579706320f, 0.304381332449784940f,
	-0.952550263480144930f,
	0.304198677629829270f, -0.952608610358033240f, 0.304016011625357570f,
	-0.952666922211226170f,
	0.303833334443086470f, -0.952725199037579570f, 0.303650646089731910f,
	-0.952783440834950920f,
	0.303467946572011370f, -0.952841647601198720f, 0.303285235896641910f,
	-0.952899819334182880f,
	0.303102514070341060f, -0.952957956031764700f, 0.302919781099827420f,
	-0.953016057691806530f,
	0.302737036991819140f, -0.953074124312172200f, 0.302554281753035670f,
	-0.953132155890726750f,
	0.302371515390196130f, -0.953190152425336560f, 0.302188737910020040f,
	-0.953248113913869320f,
	0.302005949319228200f, -0.953306040354193750f, 0.301823149624540650f,
	-0.953363931744180330f,
	0.301640338832678880f, -0.953421788081700310f, 0.301457516950363940f,
	-0.953479609364626610f,
	0.301274683984318000f, -0.953537395590833280f, 0.301091839941263210f,
	-0.953595146758195680f,
	0.300908984827921890f, -0.953652862864590500f, 0.300726118651017620f,
	-0.953710543907895560f,
	0.300543241417273400f, -0.953768189885990330f, 0.300360353133413580f,
	-0.953825800796755050f,
	0.300177453806162120f, -0.953883376638071770f, 0.299994543442243580f,
	-0.953940917407823500f,
	0.299811622048383460f, -0.953998423103894490f, 0.299628689631306790f,
	-0.954055893724170660f,
	0.299445746197739950f, -0.954113329266538800f, 0.299262791754409010f,
	-0.954170729728887280f,
	0.299079826308040480f, -0.954228095109105670f, 0.298896849865361910f,
	-0.954285425405084650f,
	0.298713862433100390f, -0.954342720614716480f, 0.298530864017984230f,
	-0.954399980735894490f,
	0.298347854626741570f, -0.954457205766513490f, 0.298164834266100910f,
	-0.954514395704469500f,
	0.297981802942791920f, -0.954571550547659630f, 0.297798760663543550f,
	-0.954628670293982680f,
	0.297615707435086310f, -0.954685754941338340f, 0.297432643264150030f,
	-0.954742804487627940f,
	0.297249568157465890f, -0.954799818930753720f, 0.297066482121764840f,
	-0.954856798268619580f,
	0.296883385163778270f, -0.954913742499130520f, 0.296700277290238460f,
	-0.954970651620192790f,
	0.296517158507877410f, -0.955027525629714160f, 0.296334028823428240f,
	-0.955084364525603410f,
	0.296150888243623960f, -0.955141168305770670f, 0.295967736775197890f,
	-0.955197936968127710f,
	0.295784574424884370f, -0.955254670510586990f, 0.295601401199417360f,
	-0.955311368931062720f,
	0.295418217105532070f, -0.955368032227470240f, 0.295235022149963390f,
	-0.955424660397726330f,
	0.295051816339446720f, -0.955481253439748770f, 0.294868599680718380f,
	-0.955537811351456770f,
	0.294685372180514330f, -0.955594334130771110f, 0.294502133845571720f,
	-0.955650821775613220f,
	0.294318884682627570f, -0.955707274283906560f, 0.294135624698419080f,
	-0.955763691653575440f,
	0.293952353899684770f, -0.955820073882545420f, 0.293769072293162400f,
	-0.955876420968743590f,
	0.293585779885591310f, -0.955932732910098170f, 0.293402476683710060f,
	-0.955989009704538930f,
	0.293219162694258680f, -0.956045251349996410f, 0.293035837923976920f,
	-0.956101457844403040f,
	0.292852502379604810f, -0.956157629185692140f, 0.292669156067883570f,
	-0.956213765371798470f,
	0.292485798995553830f, -0.956269866400658140f, 0.292302431169357610f,
	-0.956325932270208230f,
	0.292119052596036540f, -0.956381962978387620f, 0.291935663282332780f,
	-0.956437958523136180f,
	0.291752263234989370f, -0.956493918902394990f, 0.291568852460749040f,
	-0.956549844114106820f,
	0.291385430966355720f, -0.956605734156215080f, 0.291201998758553020f,
	-0.956661589026664980f,
	0.291018555844085090f, -0.956717408723403050f, 0.290835102229696940f,
	-0.956773193244376930f,
	0.290651637922133220f, -0.956828942587535370f, 0.290468162928139870f,
	-0.956884656750828900f,
	0.290284677254462330f, -0.956940335732208940f, 0.290101180907847140f,
	-0.956995979529628230f,
	0.289917673895040860f, -0.957051588141040970f, 0.289734156222790250f,
	-0.957107161564402790f,
	0.289550627897843140f, -0.957162699797670100f, 0.289367088926946960f,
	-0.957218202838801210f,
	0.289183539316850310f, -0.957273670685755200f, 0.288999979074301530f,
	-0.957329103336492790f,
	0.288816408206049480f, -0.957384500788975860f, 0.288632826718843940f,
	-0.957439863041167570f,
	0.288449234619434170f, -0.957495190091032570f, 0.288265631914570830f,
	-0.957550481936536470f,
	0.288082018611004300f, -0.957605738575646240f, 0.287898394715485170f,
	-0.957660960006330610f,
	0.287714760234765280f, -0.957716146226558870f, 0.287531115175595930f,
	-0.957771297234302320f,
	0.287347459544729570f, -0.957826413027532910f, 0.287163793348918560f,
	-0.957881493604224250f,
	0.286980116594915570f, -0.957936538962351420f, 0.286796429289474190f,
	-0.957991549099890370f,
	0.286612731439347790f, -0.958046524014818600f, 0.286429023051290750f,
	-0.958101463705114620f,
	0.286245304132057120f, -0.958156368168758820f, 0.286061574688402100f,
	-0.958211237403732260f,
	0.285877834727080730f, -0.958266071408017670f, 0.285694084254848320f,
	-0.958320870179598880f,
	0.285510323278461380f, -0.958375633716461170f, 0.285326551804675810f,
	-0.958430362016591040f,
	0.285142769840248720f, -0.958485055077976100f, 0.284958977391937150f,
	-0.958539712898605730f,
	0.284775174466498300f, -0.958594335476470220f, 0.284591361070690550f,
	-0.958648922809561040f,
	0.284407537211271820f, -0.958703474895871600f, 0.284223702895001100f,
	-0.958757991733395710f,
	0.284039858128637360f, -0.958812473320129200f, 0.283856002918939750f,
	-0.958866919654069010f,
	0.283672137272668550f, -0.958921330733213060f, 0.283488261196583550f,
	-0.958975706555561080f,
	0.283304374697445790f, -0.959030047119113550f, 0.283120477782015990f,
	-0.959084352421872730f,
	0.282936570457055390f, -0.959138622461841890f, 0.282752652729326040f,
	-0.959192857237025740f,
	0.282568724605589740f, -0.959247056745430090f, 0.282384786092609420f,
	-0.959301220985062210f,
	0.282200837197147500f, -0.959355349953930790f, 0.282016877925967690f,
	-0.959409443650045550f,
	0.281832908285833460f, -0.959463502071417510f, 0.281648928283508680f,
	-0.959517525216059260f,
	0.281464937925758050f, -0.959571513081984520f, 0.281280937219346110f,
	-0.959625465667208300f,
	0.281096926171038320f, -0.959679382969746750f, 0.280912904787600120f,
	-0.959733264987617680f,
	0.280728873075797190f, -0.959787111718839900f, 0.280544831042396360f,
	-0.959840923161433660f,
	0.280360778694163810f, -0.959894699313420530f, 0.280176716037867040f,
	-0.959948440172823210f,
	0.279992643080273380f, -0.960002145737665850f, 0.279808559828150390f,
	-0.960055816005973890f,
	0.279624466288266700f, -0.960109450975773940f, 0.279440362467390510f,
	-0.960163050645094000f,
	0.279256248372291240f, -0.960216615011963430f, 0.279072124009737970f,
	-0.960270144074412800f,
	0.278887989386500280f, -0.960323637830473920f, 0.278703844509348600f,
	-0.960377096278180130f,
	0.278519689385053060f, -0.960430519415565790f, 0.278335524020384970f,
	-0.960483907240666790f,
	0.278151348422115090f, -0.960537259751520050f, 0.277967162597015430f,
	-0.960590576946164120f,
	0.277782966551857800f, -0.960643858822638470f, 0.277598760293414290f,
	-0.960697105378984450f,
	0.277414543828458200f, -0.960750316613243950f, 0.277230317163762120f,
	-0.960803492523460760f,
	0.277046080306099950f, -0.960856633107679660f, 0.276861833262245390f,
	-0.960909738363946770f,
	0.276677576038972420f, -0.960962808290309780f, 0.276493308643056100f,
	-0.961015842884817230f,
	0.276309031081271030f, -0.961068842145519350f, 0.276124743360392890f,
	-0.961121806070467380f,
	0.275940445487197320f, -0.961174734657714080f, 0.275756137468460120f,
	-0.961227627905313460f,
	0.275571819310958250f, -0.961280485811320640f, 0.275387491021468140f,
	-0.961333308373792270f,
	0.275203152606767370f, -0.961386095590786250f, 0.275018804073633380f,
	-0.961438847460361570f,
	0.274834445428843940f, -0.961491563980579000f, 0.274650076679177790f,
	-0.961544245149499990f,
	0.274465697831413220f, -0.961596890965187860f, 0.274281308892329710f,
	-0.961649501425706820f,
	0.274096909868706330f, -0.961702076529122540f, 0.273912500767323320f,
	-0.961754616273502010f,
	0.273728081594960650f, -0.961807120656913540f, 0.273543652358398730f,
	-0.961859589677426570f,
	0.273359213064418790f, -0.961912023333112100f, 0.273174763719801870f,
	-0.961964421622042320f,
	0.272990304331329980f, -0.962016784542290560f, 0.272805834905784920f,
	-0.962069112091931580f,
	0.272621355449948980f, -0.962121404269041580f, 0.272436865970605350f,
	-0.962173661071697770f,
	0.272252366474536660f, -0.962225882497979020f, 0.272067856968526980f,
	-0.962278068545965090f,
	0.271883337459359890f, -0.962330219213737400f, 0.271698807953819510f,
	-0.962382334499378380f,
	0.271514268458690810f, -0.962434414400971990f, 0.271329718980758420f,
	-0.962486458916603450f,
	0.271145159526808070f, -0.962538468044359160f, 0.270960590103625330f,
	-0.962590441782326780f,
	0.270776010717996010f, -0.962642380128595710f, 0.270591421376707050f,
	-0.962694283081255930f,
	0.270406822086544820f, -0.962746150638399410f, 0.270222212854296930f,
	-0.962797982798119010f,
	0.270037593686750510f, -0.962849779558509030f, 0.269852964590693910f,
	-0.962901540917665000f,
	0.269668325572915200f, -0.962953266873683880f, 0.269483676640202840f,
	-0.963004957424663850f,
	0.269299017799346230f, -0.963056612568704340f, 0.269114349057134330f,
	-0.963108232303906190f,
	0.268929670420357310f, -0.963159816628371360f, 0.268744981895805090f,
	-0.963211365540203480f,
	0.268560283490267890f, -0.963262879037507070f, 0.268375575210537010f,
	-0.963314357118388090f,
	0.268190857063403180f, -0.963365799780954050f, 0.268006129055658350f,
	-0.963417207023313350f,
	0.267821391194094320f, -0.963468578843575950f, 0.267636643485503090f,
	-0.963519915239853140f,
	0.267451885936677740f, -0.963571216210257210f, 0.267267118554410930f,
	-0.963622481752902220f,
	0.267082341345496350f, -0.963673711865903230f, 0.266897554316727510f,
	-0.963724906547376410f,
	0.266712757474898420f, -0.963776065795439840f, 0.266527950826803810f,
	-0.963827189608212340f,
	0.266343134379238180f, -0.963878277983814200f, 0.266158308138997050f,
	-0.963929330920367140f,
	0.265973472112875530f, -0.963980348415994110f, 0.265788626307669970f,
	-0.964031330468819280f,
	0.265603770730176440f, -0.964082277076968140f, 0.265418905387191260f,
	-0.964133188238567640f,
	0.265234030285511900f, -0.964184063951745720f, 0.265049145431935200f,
	-0.964234904214632200f,
	0.264864250833259320f, -0.964285709025357370f, 0.264679346496282050f,
	-0.964336478382053720f,
	0.264494432427801630f, -0.964387212282854290f, 0.264309508634617220f,
	-0.964437910725893910f,
	0.264124575123527490f, -0.964488573709308410f, 0.263939631901332410f,
	-0.964539201231235150f,
	0.263754678974831510f, -0.964589793289812650f, 0.263569716350824880f,
	-0.964640349883180930f,
	0.263384744036113390f, -0.964690871009480920f, 0.263199762037497560f,
	-0.964741356666855340f,
	0.263014770361779060f, -0.964791806853447900f, 0.262829769015759330f,
	-0.964842221567403510f,
	0.262644758006240100f, -0.964892600806868890f, 0.262459737340024090f,
	-0.964942944569991410f,
	0.262274707023913590f, -0.964993252854920320f, 0.262089667064712100f,
	-0.965043525659805890f,
	0.261904617469222560f, -0.965093762982799590f, 0.261719558244249080f,
	-0.965143964822054450f,
	0.261534489396595630f, -0.965194131175724720f, 0.261349410933066350f,
	-0.965244262041965780f,
	0.261164322860466590f, -0.965294357418934660f, 0.260979225185601020f,
	-0.965344417304789370f,
	0.260794117915275570f, -0.965394441697689400f, 0.260609001056295920f,
	-0.965444430595795430f,
	0.260423874615468010f, -0.965494383997269500f, 0.260238738599598950f,
	-0.965544301900275070f,
	0.260053593015495130f, -0.965594184302976830f, 0.259868437869964330f,
	-0.965644031203540590f,
	0.259683273169813930f, -0.965693842600133690f, 0.259498098921851660f,
	-0.965743618490924830f,
	0.259312915132886350f, -0.965793358874083570f, 0.259127721809726150f,
	-0.965843063747781510f,
	0.258942518959180580f, -0.965892733110190860f, 0.258757306588058840f,
	-0.965942366959485540f,
	0.258572084703170390f, -0.965991965293840570f, 0.258386853311325710f,
	-0.966041528111432400f,
	0.258201612419334870f, -0.966091055410438830f, 0.258016362034009070f,
	-0.966140547189038750f,
	0.257831102162158930f, -0.966190003445412620f, 0.257645832810596440f,
	-0.966239424177741890f,
	0.257460553986133210f, -0.966288809384209580f, 0.257275265695581120f,
	-0.966338159063000130f,
	0.257089967945753230f, -0.966387473212298790f, 0.256904660743461850f,
	-0.966436751830292650f,
	0.256719344095520720f, -0.966485994915169840f, 0.256534018008743200f,
	-0.966535202465119700f,
	0.256348682489942910f, -0.966584374478333120f, 0.256163337545934570f,
	-0.966633510953002100f,
	0.255977983183532380f, -0.966682611887320190f, 0.255792619409551670f,
	-0.966731677279481840f,
	0.255607246230807550f, -0.966780707127683270f, 0.255421863654115460f,
	-0.966829701430121810f,
	0.255236471686291820f, -0.966878660184995910f, 0.255051070334152530f,
	-0.966927583390505660f,
	0.254865659604514630f, -0.966976471044852070f, 0.254680239504194990f,
	-0.967025323146237900f,
	0.254494810040010790f, -0.967074139692867040f, 0.254309371218780110f,
	-0.967122920682944360f,
	0.254123923047320620f, -0.967171666114676640f, 0.253938465532451140f,
	-0.967220375986271310f,
	0.253752998680989940f, -0.967269050295937790f, 0.253567522499756610f,
	-0.967317689041886310f,
	0.253382036995570270f, -0.967366292222328510f, 0.253196542175250560f,
	-0.967414859835477480f,
	0.253011038045617980f, -0.967463391879547440f, 0.252825524613492610f,
	-0.967511888352754150f,
	0.252640001885695580f, -0.967560349253314360f, 0.252454469869047900f,
	-0.967608774579446380f,
	0.252268928570370810f, -0.967657164329369880f, 0.252083377996486560f,
	-0.967705518501305480f,
	0.251897818154216910f, -0.967753837093475510f, 0.251712249050384750f,
	-0.967802120104103270f,
	0.251526670691812780f, -0.967850367531413620f, 0.251341083085323880f,
	-0.967898579373632660f,
	0.251155486237742030f, -0.967946755628987800f, 0.250969880155890720f,
	-0.967994896295707670f,
	0.250784264846594550f, -0.968043001372022260f, 0.250598640316677830f,
	-0.968091070856162970f,
	0.250413006572965280f, -0.968139104746362330f, 0.250227363622282540f,
	-0.968187103040854420f,
	0.250041711471454650f, -0.968235065737874320f, 0.249856050127308050f,
	-0.968282992835658660f,
	0.249670379596668520f, -0.968330884332445300f, 0.249484699886363010f,
	-0.968378740226473300f,
	0.249299011003218300f, -0.968426560515983190f, 0.249113312954061360f,
	-0.968474345199216820f,
	0.248927605745720260f, -0.968522094274417270f, 0.248741889385022420f,
	-0.968569807739828930f,
	0.248556163878796620f, -0.968617485593697540f, 0.248370429233871150f,
	-0.968665127834269950f,
	0.248184685457074780f, -0.968712734459794780f, 0.247998932555237220f,
	-0.968760305468521430f,
	0.247813170535187620f, -0.968807840858700970f, 0.247627399403756330f,
	-0.968855340628585580f,
	0.247441619167773440f, -0.968902804776428870f, 0.247255829834069320f,
	-0.968950233300485800f,
	0.247070031409475370f, -0.968997626199012310f, 0.246884223900822430f,
	-0.969044983470266240f,
	0.246698407314942500f, -0.969092305112506100f, 0.246512581658667380f,
	-0.969139591123992280f,
	0.246326746938829060f, -0.969186841502985950f, 0.246140903162260640f,
	-0.969234056247750050f,
	0.245955050335794590f, -0.969281235356548530f, 0.245769188466264670f,
	-0.969328378827646660f,
	0.245583317560504000f, -0.969375486659311280f, 0.245397437625346990f,
	-0.969422558849810320f,
	0.245211548667627680f, -0.969469595397412950f, 0.245025650694180470f,
	-0.969516596300390000f,
	0.244839743711840750f, -0.969563561557013180f, 0.244653827727443320f,
	-0.969610491165555870f,
	0.244467902747824210f, -0.969657385124292450f, 0.244281968779819170f,
	-0.969704243431498750f,
	0.244096025830264210f, -0.969751066085452140f, 0.243910073905996370f,
	-0.969797853084430890f,
	0.243724113013852130f, -0.969844604426714830f, 0.243538143160669180f,
	-0.969891320110585100f,
	0.243352164353284880f, -0.969938000134323960f, 0.243166176598536930f,
	-0.969984644496215240f,
	0.242980179903263980f, -0.970031253194543970f, 0.242794174274304190f,
	-0.970077826227596420f,
	0.242608159718496890f, -0.970124363593660280f, 0.242422136242681050f,
	-0.970170865291024360f,
	0.242236103853696040f, -0.970217331317979160f, 0.242050062558382180f,
	-0.970263761672816140f,
	0.241864012363579210f, -0.970310156353828110f, 0.241677953276128090f,
	-0.970356515359309450f,
	0.241491885302869300f, -0.970402838687555500f, 0.241305808450644390f,
	-0.970449126336863090f,
	0.241119722726294730f, -0.970495378305530450f, 0.240933628136661910f,
	-0.970541594591857070f,
	0.240747524688588540f, -0.970587775194143630f, 0.240561412388916620f,
	-0.970633920110692160f,
	0.240375291244489500f, -0.970680029339806130f, 0.240189161262150040f,
	-0.970726102879790110f,
	0.240003022448741500f, -0.970772140728950350f, 0.239816874811108110f,
	-0.970818142885593870f,
	0.239630718356093560f, -0.970864109348029470f, 0.239444553090542720f,
	-0.970910040114567050f,
	0.239258379021300120f, -0.970955935183517970f, 0.239072196155210660f,
	-0.971001794553194690f,
	0.238886004499120170f, -0.971047618221911100f, 0.238699804059873950f,
	-0.971093406187982460f,
	0.238513594844318500f, -0.971139158449725090f, 0.238327376859299970f,
	-0.971184875005457030f,
	0.238141150111664870f, -0.971230555853497380f, 0.237954914608260650f,
	-0.971276200992166490f,
	0.237768670355934210f, -0.971321810419786160f, 0.237582417361533650f,
	-0.971367384134679490f,
	0.237396155631906550f, -0.971412922135170940f, 0.237209885173901620f,
	-0.971458424419585960f,
	0.237023605994367340f, -0.971503890986251780f, 0.236837318100152380f,
	-0.971549321833496630f,
	0.236651021498106460f, -0.971594716959650160f, 0.236464716195078750f,
	-0.971640076363043390f,
	0.236278402197919620f, -0.971685400042008540f, 0.236092079513479050f,
	-0.971730687994879160f,
	0.235905748148607370f, -0.971775940219990140f, 0.235719408110155930f,
	-0.971821156715677700f,
	0.235533059404975460f, -0.971866337480279400f, 0.235346702039917920f,
	-0.971911482512134000f,
	0.235160336021834860f, -0.971956591809581600f, 0.234973961357578310f,
	-0.972001665370963890f,
	0.234787578054001080f, -0.972046703194623380f, 0.234601186117955550f,
	-0.972091705278904430f,
	0.234414785556295250f, -0.972136671622152120f, 0.234228376375873380f,
	-0.972181602222713440f,
	0.234041958583543460f, -0.972226497078936270f, 0.233855532186159950f,
	-0.972271356189170040f,
	0.233669097190576820f, -0.972316179551765300f, 0.233482653603649170f,
	-0.972360967165074140f,
	0.233296201432231560f, -0.972405719027449770f, 0.233109740683179740f,
	-0.972450435137246830f,
	0.232923271363349120f, -0.972495115492821190f, 0.232736793479595420f,
	-0.972539760092530180f,
	0.232550307038775330f, -0.972584368934732210f, 0.232363812047745010f,
	-0.972628942017787270f,
	0.232177308513361770f, -0.972673479340056430f, 0.231990796442482580f,
	-0.972717980899902250f,
	0.231804275841964780f, -0.972762446695688570f, 0.231617746718666580f,
	-0.972806876725780370f,
	0.231431209079445730f, -0.972851270988544180f, 0.231244662931161110f,
	-0.972895629482347760f,
	0.231058108280671280f, -0.972939952205560070f, 0.230871545134835070f,
	-0.972984239156551740f,
	0.230684973500512310f, -0.973028490333694100f, 0.230498393384562320f,
	-0.973072705735360530f,
	0.230311804793845530f, -0.973116885359925130f, 0.230125207735222020f,
	-0.973161029205763530f,
	0.229938602215552260f, -0.973205137271252800f, 0.229751988241697600f,
	-0.973249209554771120f,
	0.229565365820518870f, -0.973293246054698250f, 0.229378734958878120f,
	-0.973337246769414800f,
	0.229192095663636740f, -0.973381211697303290f, 0.229005447941657390f,
	-0.973425140836747030f,
	0.228818791799802360f, -0.973469034186130950f, 0.228632127244934230f,
	-0.973512891743841370f,
	0.228445454283916550f, -0.973556713508265560f, 0.228258772923612350f,
	-0.973600499477792370f,
	0.228072083170885790f, -0.973644249650811870f, 0.227885385032600700f,
	-0.973687964025715670f,
	0.227698678515621170f, -0.973731642600896400f, 0.227511963626812390f,
	-0.973775285374748000f,
	0.227325240373038830f, -0.973818892345666100f, 0.227138508761166260f,
	-0.973862463512047300f,
	0.226951768798059980f, -0.973905998872289460f, 0.226765020490585720f,
	-0.973949498424792170f,
	0.226578263845610110f, -0.973992962167955830f, 0.226391498869999210f,
	-0.974036390100182610f,
	0.226204725570620270f, -0.974079782219875680f, 0.226017943954340190f,
	-0.974123138525439520f,
	0.225831154028026200f, -0.974166459015280320f, 0.225644355798546440f,
	-0.974209743687805110f,
	0.225457549272768540f, -0.974252992541422500f, 0.225270734457561240f,
	-0.974296205574542330f,
	0.225083911359792780f, -0.974339382785575860f, 0.224897079986332540f,
	-0.974382524172935470f,
	0.224710240344049570f, -0.974425629735034990f, 0.224523392439813170f,
	-0.974468699470289580f,
	0.224336536280493690f, -0.974511733377115720f, 0.224149671872960840f,
	-0.974554731453931230f,
	0.223962799224085520f, -0.974597693699155050f, 0.223775918340738290f,
	-0.974640620111207560f,
	0.223589029229790020f, -0.974683510688510670f, 0.223402131898112480f,
	-0.974726365429487320f,
	0.223215226352576960f, -0.974769184332561770f, 0.223028312600055870f,
	-0.974811967396159830f,
	0.222841390647421280f, -0.974854714618708430f, 0.222654460501545550f,
	-0.974897425998635820f,
	0.222467522169301990f, -0.974940101534371720f, 0.222280575657563370f,
	-0.974982741224347140f,
	0.222093620973203590f, -0.975025345066994120f, 0.221906658123096260f,
	-0.975067913060746360f,
	0.221719687114115240f, -0.975110445204038890f, 0.221532707953135340f,
	-0.975152941495307620f,
	0.221345720647030810f, -0.975195401932990370f, 0.221158725202677100f,
	-0.975237826515525820f,
	0.220971721626949060f, -0.975280215241354220f, 0.220784709926722670f,
	-0.975322568108916930f,
	0.220597690108873650f, -0.975364885116656870f, 0.220410662180277940f,
	-0.975407166263018270f,
	0.220223626147812460f, -0.975449411546446380f, 0.220036582018353550f,
	-0.975491620965388110f,
	0.219849529798778750f, -0.975533794518291360f, 0.219662469495965180f,
	-0.975575932203605610f,
	0.219475401116790340f, -0.975618034019781750f, 0.219288324668132580f,
	-0.975660099965271590f,
	0.219101240156869770f, -0.975702130038528570f, 0.218914147589880900f,
	-0.975744124238007270f,
	0.218727046974044600f, -0.975786082562163930f, 0.218539938316239830f,
	-0.975828005009455550f,
	0.218352821623346430f, -0.975869891578341030f, 0.218165696902243770f,
	-0.975911742267280170f,
	0.217978564159812290f, -0.975953557074734300f, 0.217791423402932120f,
	-0.975995335999165880f,
	0.217604274638483670f, -0.976037079039039020f, 0.217417117873348300f,
	-0.976078786192818850f,
	0.217229953114406790f, -0.976120457458971910f, 0.217042780368541080f,
	-0.976162092835966110f,
	0.216855599642632570f, -0.976203692322270560f, 0.216668410943563790f,
	-0.976245255916355800f,
	0.216481214278216900f, -0.976286783616693630f, 0.216294009653474370f,
	-0.976328275421757260f,
	0.216106797076219600f, -0.976369731330021140f, 0.215919576553335460f,
	-0.976411151339961040f,
	0.215732348091705940f, -0.976452535450054060f, 0.215545111698214660f,
	-0.976493883658778540f,
	0.215357867379745550f, -0.976535195964614470f, 0.215170615143183500f,
	-0.976576472366042610f,
	0.214983354995412820f, -0.976617712861545640f, 0.214796086943318920f,
	-0.976658917449606980f,
	0.214608810993786920f, -0.976700086128711840f, 0.214421527153702190f,
	-0.976741218897346550f,
	0.214234235429951100f, -0.976782315753998650f, 0.214046935829419330f,
	-0.976823376697157240f,
	0.213859628358993830f, -0.976864401725312640f, 0.213672313025561140f,
	-0.976905390836956490f,
	0.213484989836008080f, -0.976946344030581560f, 0.213297658797222430f,
	-0.976987261304682390f,
	0.213110319916091360f, -0.977028142657754390f, 0.212922973199503260f,
	-0.977068988088294450f,
	0.212735618654345870f, -0.977109797594800880f, 0.212548256287508120f,
	-0.977150571175773200f,
	0.212360886105878580f, -0.977191308829712280f, 0.212173508116346080f,
	-0.977232010555120320f,
	0.211986122325800410f, -0.977272676350500860f, 0.211798728741130820f,
	-0.977313306214358750f,
	0.211611327369227610f, -0.977353900145199960f, 0.211423918216980810f,
	-0.977394458141532250f,
	0.211236501291280710f, -0.977434980201864260f, 0.211049076599018500f,
	-0.977475466324706050f,
	0.210861644147084830f, -0.977515916508569280f, 0.210674203942371490f,
	-0.977556330751966460f,
	0.210486755991769890f, -0.977596709053411780f, 0.210299300302171750f,
	-0.977637051411420770f,
	0.210111836880469720f, -0.977677357824509930f, 0.209924365733555860f,
	-0.977717628291197570f,
	0.209736886868323370f, -0.977757862810002760f, 0.209549400291665110f,
	-0.977798061379446360f,
	0.209361906010474190f, -0.977838223998050430f, 0.209174404031644700f,
	-0.977878350664338150f,
	0.208986894362070070f, -0.977918441376834370f, 0.208799377008644980f,
	-0.977958496134064830f,
	0.208611851978263460f, -0.977998514934557140f, 0.208424319277820650f,
	-0.978038497776839600f,
	0.208236778914211470f, -0.978078444659442380f, 0.208049230894330940f,
	-0.978118355580896660f,
	0.207861675225075150f, -0.978158230539735050f, 0.207674111913339540f,
	-0.978198069534491400f,
	0.207486540966020700f, -0.978237872563701090f, 0.207298962390014880f,
	-0.978277639625900420f,
	0.207111376192218560f, -0.978317370719627650f, 0.206923782379529210f,
	-0.978357065843421640f,
	0.206736180958843660f, -0.978396724995823090f, 0.206548571937059940f,
	-0.978436348175373730f,
	0.206360955321075680f, -0.978475935380616830f, 0.206173331117788770f,
	-0.978515486610096910f,
	0.205985699334098050f, -0.978555001862359550f, 0.205798059976901760f,
	-0.978594481135952270f,
	0.205610413053099320f, -0.978633924429423100f, 0.205422758569589780f,
	-0.978673331741322210f,
	0.205235096533272380f, -0.978712703070200420f, 0.205047426951047380f,
	-0.978752038414610340f,
	0.204859749829814420f, -0.978791337773105670f, 0.204672065176474290f,
	-0.978830601144241470f,
	0.204484372997927180f, -0.978869828526574120f, 0.204296673301074430f,
	-0.978909019918661310f,
	0.204108966092817010f, -0.978948175319062200f, 0.203921251380056150f,
	-0.978987294726337050f,
	0.203733529169694010f, -0.979026378139047580f, 0.203545799468632190f,
	-0.979065425555756930f,
	0.203358062283773370f, -0.979104436975029250f, 0.203170317622019920f,
	-0.979143412395430230f,
	0.202982565490274460f, -0.979182351815526930f, 0.202794805895440550f,
	-0.979221255233887700f,
	0.202607038844421110f, -0.979260122649082020f, 0.202419264344120220f,
	-0.979298954059681040f,
	0.202231482401441620f, -0.979337749464256780f, 0.202043693023289280f,
	-0.979376508861383170f,
	0.201855896216568160f, -0.979415232249634780f, 0.201668091988182500f,
	-0.979453919627588210f,
	0.201480280345037820f, -0.979492570993820700f, 0.201292461294039190f,
	-0.979531186346911390f,
	0.201104634842091960f, -0.979569765685440520f, 0.200916800996102370f,
	-0.979608309007989450f,
	0.200728959762976140f, -0.979646816313141210f, 0.200541111149620090f,
	-0.979685287599479930f,
	0.200353255162940420f, -0.979723722865591170f, 0.200165391809844500f,
	-0.979762122110061640f,
	0.199977521097239290f, -0.979800485331479680f, 0.199789643032032120f,
	-0.979838812528434740f,
	0.199601757621131050f, -0.979877103699517640f, 0.199413864871443750f,
	-0.979915358843320480f,
	0.199225964789878890f, -0.979953577958436740f, 0.199038057383344820f,
	-0.979991761043461200f,
	0.198850142658750120f, -0.980029908096989980f, 0.198662220623004320f,
	-0.980068019117620650f,
	0.198474291283016360f, -0.980106094103951770f, 0.198286354645696270f,
	-0.980144133054583590f,
	0.198098410717953730f, -0.980182135968117320f, 0.197910459506698720f,
	-0.980220102843155970f,
	0.197722501018842030f, -0.980258033678303550f, 0.197534535261294000f,
	-0.980295928472165290f,
	0.197346562240966000f, -0.980333787223347960f, 0.197158581964769040f,
	-0.980371609930459690f,
	0.196970594439614370f, -0.980409396592109910f, 0.196782599672414240f,
	-0.980447147206909060f,
	0.196594597670080220f, -0.980484861773469380f, 0.196406588439525050f,
	-0.980522540290404090f,
	0.196218571987660850f, -0.980560182756327950f, 0.196030548321400880f,
	-0.980597789169856850f,
	0.195842517447657990f, -0.980635359529608120f, 0.195654479373345370f,
	-0.980672893834200530f,
	0.195466434105377090f, -0.980710392082253970f, 0.195278381650666520f,
	-0.980747854272389750f,
	0.195090322016128330f, -0.980785280403230430f, 0.194902255208676660f,
	-0.980822670473399990f,
	0.194714181235225990f, -0.980860024481523870f, 0.194526100102691720f,
	-0.980897342426228390f,
	0.194338011817988600f, -0.980934624306141640f, 0.194149916388032530f,
	-0.980971870119892840f,
	0.193961813819739010f, -0.981009079866112630f, 0.193773704120023840f,
	-0.981046253543432780f,
	0.193585587295803750f, -0.981083391150486590f, 0.193397463353994740f,
	-0.981120492685908730f,
	0.193209332301514080f, -0.981157558148334830f, 0.193021194145278320f,
	-0.981194587536402320f,
	0.192833048892205290f, -0.981231580848749730f, 0.192644896549212240f,
	-0.981268538084016710f,
	0.192456737123216840f, -0.981305459240844670f, 0.192268570621137590f,
	-0.981342344317875930f,
	0.192080397049892380f, -0.981379193313754560f, 0.191892216416400310f,
	-0.981416006227125550f,
	0.191704028727579940f, -0.981452783056635520f, 0.191515833990350240f,
	-0.981489523800932130f,
	0.191327632211630990f, -0.981526228458664660f, 0.191139423398341420f,
	-0.981562897028483650f,
	0.190951207557401860f, -0.981599529509040720f, 0.190762984695732250f,
	-0.981636125898989080f,
	0.190574754820252800f, -0.981672686196983110f, 0.190386517937884580f,
	-0.981709210401678800f,
	0.190198274055548120f, -0.981745698511732990f, 0.190010023180165050f,
	-0.981782150525804310f,
	0.189821765318656580f, -0.981818566442552500f, 0.189633500477944220f,
	-0.981854946260638630f,
	0.189445228664950340f, -0.981891289978724990f, 0.189256949886596720f,
	-0.981927597595475540f,
	0.189068664149806280f, -0.981963869109555240f, 0.188880371461501330f,
	-0.982000104519630490f,
	0.188692071828605260f, -0.982036303824369020f, 0.188503765258041080f,
	-0.982072467022439890f,
	0.188315451756732120f, -0.982108594112513610f, 0.188127131331602530f,
	-0.982144685093261580f,
	0.187938803989575850f, -0.982180739963357200f, 0.187750469737576840f,
	-0.982216758721474510f,
	0.187562128582529740f, -0.982252741366289370f, 0.187373780531359110f,
	-0.982288687896478830f,
	0.187185425590990440f, -0.982324598310721160f, 0.186997063768348510f,
	-0.982360472607696210f,
	0.186808695070359330f, -0.982396310786084690f, 0.186620319503948420f,
	-0.982432112844569110f,
	0.186431937076041640f, -0.982467878781833170f, 0.186243547793565670f,
	-0.982503608596561720f,
	0.186055151663446630f, -0.982539302287441240f, 0.185866748692611720f,
	-0.982574959853159240f,
	0.185678338887987790f, -0.982610581292404750f, 0.185489922256501900f,
	-0.982646166603868050f,
	0.185301498805082040f, -0.982681715786240860f, 0.185113068540655510f,
	-0.982717228838215990f,
	0.184924631470150870f, -0.982752705758487830f, 0.184736187600495930f,
	-0.982788146545751970f,
	0.184547736938619640f, -0.982823551198705240f, 0.184359279491450640f,
	-0.982858919716046110f,
	0.184170815265917720f, -0.982894252096474070f, 0.183982344268950600f,
	-0.982929548338690060f,
	0.183793866507478390f, -0.982964808441396440f, 0.183605381988431350f,
	-0.983000032403296590f,
	0.183416890718739230f, -0.983035220223095640f, 0.183228392705332140f,
	-0.983070371899499640f,
	0.183039887955141060f, -0.983105487431216290f, 0.182851376475096310f,
	-0.983140566816954500f,
	0.182662858272129360f, -0.983175610055424420f, 0.182474333353171260f,
	-0.983210617145337640f,
	0.182285801725153320f, -0.983245588085407070f, 0.182097263395007760f,
	-0.983280522874346970f,
	0.181908718369666160f, -0.983315421510872810f, 0.181720166656061170f,
	-0.983350283993701500f,
	0.181531608261125130f, -0.983385110321551180f, 0.181343043191790590f,
	-0.983419900493141540f,
	0.181154471454990920f, -0.983454654507193270f, 0.180965893057658980f,
	-0.983489372362428730f,
	0.180777308006728670f, -0.983524054057571260f, 0.180588716309133280f,
	-0.983558699591345900f,
	0.180400117971807270f, -0.983593308962478650f, 0.180211513001684590f,
	-0.983627882169697210f,
	0.180022901405699510f, -0.983662419211730250f, 0.179834283190787180f,
	-0.983696920087308020f,
	0.179645658363882100f, -0.983731384795162090f, 0.179457026931919950f,
	-0.983765813334025240f,
	0.179268388901835880f, -0.983800205702631490f, 0.179079744280565390f,
	-0.983834561899716630f,
	0.178891093075044830f, -0.983868881924017220f, 0.178702435292209940f,
	-0.983903165774271500f,
	0.178513770938997590f, -0.983937413449218920f, 0.178325100022344140f,
	-0.983971624947600270f,
	0.178136422549186320f, -0.984005800268157870f, 0.177947738526461670f,
	-0.984039939409634970f,
	0.177759047961107140f, -0.984074042370776450f, 0.177570350860060790f,
	-0.984108109150328540f,
	0.177381647230260200f, -0.984142139747038570f, 0.177192937078643310f,
	-0.984176134159655320f,
	0.177004220412148860f, -0.984210092386929030f, 0.176815497237715000f,
	-0.984244014427611110f,
	0.176626767562280960f, -0.984277900280454370f, 0.176438031392785350f,
	-0.984311749944212780f,
	0.176249288736167940f, -0.984345563417641900f, 0.176060539599367960f,
	-0.984379340699498510f,
	0.175871783989325040f, -0.984413081788540700f, 0.175683021912979580f,
	-0.984446786683527920f,
	0.175494253377271400f, -0.984480455383220930f, 0.175305478389141370f,
	-0.984514087886381840f,
	0.175116696955530060f, -0.984547684191773960f, 0.174927909083378160f,
	-0.984581244298162180f,
	0.174739114779627310f, -0.984614768204312600f, 0.174550314051218490f,
	-0.984648255908992630f,
	0.174361506905093830f, -0.984681707410970940f, 0.174172693348194960f,
	-0.984715122709017620f,
	0.173983873387463850f, -0.984748501801904210f, 0.173795047029843270f,
	-0.984781844688403350f,
	0.173606214282275410f, -0.984815151367289140f, 0.173417375151703520f,
	-0.984848421837337010f,
	0.173228529645070490f, -0.984881656097323700f, 0.173039677769319390f,
	-0.984914854146027200f,
	0.172850819531394200f, -0.984948015982227030f, 0.172661954938238270f,
	-0.984981141604703960f,
	0.172473083996796030f, -0.985014231012239840f, 0.172284206714011350f,
	-0.985047284203618200f,
	0.172095323096829040f, -0.985080301177623800f, 0.171906433152193700f,
	-0.985113281933042590f,
	0.171717536887049970f, -0.985146226468662230f, 0.171528634308343500f,
	-0.985179134783271020f,
	0.171339725423019260f, -0.985212006875659460f, 0.171150810238023340f,
	-0.985244842744618540f,
	0.170961888760301360f, -0.985277642388941220f, 0.170772960996799230f,
	-0.985310405807421570f,
	0.170584026954463700f, -0.985343132998854790f, 0.170395086640240920f,
	-0.985375823962037710f,
	0.170206140061078120f, -0.985408478695768420f, 0.170017187223922090f,
	-0.985441097198846210f,
	0.169828228135719880f, -0.985473679470071810f, 0.169639262803419400f,
	-0.985506225508247290f,
	0.169450291233967930f, -0.985538735312176060f, 0.169261313434313890f,
	-0.985571208880662740f,
	0.169072329411405180f, -0.985603646212513400f, 0.168883339172190010f,
	-0.985636047306535420f,
	0.168694342723617440f, -0.985668412161537550f, 0.168505340072635900f,
	-0.985700740776329850f,
	0.168316331226194910f, -0.985733033149723490f, 0.168127316191243350f,
	-0.985765289280531310f,
	0.167938294974731230f, -0.985797509167567370f, 0.167749267583608030f,
	-0.985829692809647050f,
	0.167560234024823590f, -0.985861840205586980f, 0.167371194305328540f,
	-0.985893951354205210f,
	0.167182148432072880f, -0.985926026254321130f, 0.166993096412007770f,
	-0.985958064904755460f,
	0.166804038252083870f, -0.985990067304330030f, 0.166614973959252090f,
	-0.986022033451868560f,
	0.166425903540464220f, -0.986053963346195440f, 0.166236827002671390f,
	-0.986085856986136820f,
	0.166047744352825850f, -0.986117714370520090f, 0.165858655597879430f,
	-0.986149535498173860f,
	0.165669560744784140f, -0.986181320367928270f, 0.165480459800492890f,
	-0.986213068978614490f,
	0.165291352771957970f, -0.986244781329065460f, 0.165102239666132720f,
	-0.986276457418114980f,
	0.164913120489970090f, -0.986308097244598670f, 0.164723995250423190f,
	-0.986339700807353000f,
	0.164534863954446110f, -0.986371268105216030f, 0.164345726608992190f,
	-0.986402799137027220f,
	0.164156583221015890f, -0.986434293901627070f, 0.163967433797471110f,
	-0.986465752397857940f,
	0.163778278345312690f, -0.986497174624562880f, 0.163589116871495160f,
	-0.986528560580586690f,
	0.163399949382973230f, -0.986559910264775410f, 0.163210775886702460f,
	-0.986591223675976400f,
	0.163021596389637810f, -0.986622500813038480f, 0.162832410898735260f,
	-0.986653741674811350f,
	0.162643219420950450f, -0.986684946260146690f, 0.162454021963239190f,
	-0.986716114567897100f,
	0.162264818532558110f, -0.986747246596916480f, 0.162075609135863330f,
	-0.986778342346060430f,
	0.161886393780111910f, -0.986809401814185420f, 0.161697172472260540f,
	-0.986840425000149680f,
	0.161507945219266150f, -0.986871411902812470f, 0.161318712028086540f,
	-0.986902362521034470f,
	0.161129472905678780f, -0.986933276853677710f, 0.160940227859001140f,
	-0.986964154899605650f,
	0.160750976895011390f, -0.986994996657682870f, 0.160561720020667510f,
	-0.987025802126775600f,
	0.160372457242928400f, -0.987056571305750970f, 0.160183188568752240f,
	-0.987087304193477900f,
	0.159993914005098350f, -0.987118000788826280f, 0.159804633558925380f,
	-0.987148661090667570f,
	0.159615347237193090f, -0.987179285097874340f, 0.159426055046860750f,
	-0.987209872809320820f,
	0.159236756994887850f, -0.987240424223882250f, 0.159047453088234840f,
	-0.987270939340435420f,
	0.158858143333861390f, -0.987301418157858430f, 0.158668827738728370f,
	-0.987331860675030430f,
	0.158479506309796100f, -0.987362266890832400f, 0.158290179054025180f,
	-0.987392636804146240f,
	0.158100845978377090f, -0.987422970413855410f, 0.157911507089812640f,
	-0.987453267718844560f,
	0.157722162395293690f, -0.987483528717999710f, 0.157532811901781670f,
	-0.987513753410208420f,
	0.157343455616238280f, -0.987543941794359230f, 0.157154093545626010f,
	-0.987574093869342360f,
	0.156964725696906750f, -0.987604209634049160f, 0.156775352077043430f,
	-0.987634289087372160f,
	0.156585972692998590f, -0.987664332228205710f, 0.156396587551734940f,
	-0.987694339055445130f,
	0.156207196660216040f, -0.987724309567986960f, 0.156017800025404830f,
	-0.987754243764729530f,
	0.155828397654265320f, -0.987784141644572180f, 0.155638989553760850f,
	-0.987814003206415550f,
	0.155449575730855880f, -0.987843828449161740f, 0.155260156192514380f,
	-0.987873617371714200f,
	0.155070730945700510f, -0.987903369972977790f, 0.154881299997379400f,
	-0.987933086251858380f,
	0.154691863354515400f, -0.987962766207263420f, 0.154502421024073990f,
	-0.987992409838101880f,
	0.154312973013020240f, -0.988022017143283530f, 0.154123519328319360f,
	-0.988051588121720110f,
	0.153934059976937460f, -0.988081122772324070f, 0.153744594965840000f,
	-0.988110621094009820f,
	0.153555124301993500f, -0.988140083085692570f, 0.153365647992364020f,
	-0.988169508746289060f,
	0.153176166043917870f, -0.988198898074717610f, 0.152986678463622160f,
	-0.988228251069897420f,
	0.152797185258443410f, -0.988257567730749460f, 0.152607686435349140f,
	-0.988286848056195710f,
	0.152418182001306500f, -0.988316092045159690f, 0.152228671963282770f,
	-0.988345299696566150f,
	0.152039156328246160f, -0.988374471009341280f, 0.151849635103164180f,
	-0.988403605982412390f,
	0.151660108295005400f, -0.988432704614708340f, 0.151470575910737760f,
	-0.988461766905159300f,
	0.151281037957330250f, -0.988490792852696590f, 0.151091494441751430f,
	-0.988519782456253270f,
	0.150901945370970040f, -0.988548735714763200f, 0.150712390751955720f,
	-0.988577652627162020f,
	0.150522830591677370f, -0.988606533192386450f, 0.150333264897105050f,
	-0.988635377409374790f,
	0.150143693675208330f, -0.988664185277066230f, 0.149954116932956990f,
	-0.988692956794401940f,
	0.149764534677321620f, -0.988721691960323780f, 0.149574946915272210f,
	-0.988750390773775360f,
	0.149385353653779810f, -0.988779053233701520f, 0.149195754899814960f,
	-0.988807679339048340f,
	0.149006150660348470f, -0.988836269088763540f, 0.148816540942352030f,
	-0.988864822481795640f,
	0.148626925752796540f, -0.988893339517095130f, 0.148437305098654050f,
	-0.988921820193613190f,
	0.148247678986896200f, -0.988950264510302990f, 0.148058047424494740f,
	-0.988978672466118480f,
	0.147868410418422360f, -0.989007044060015270f, 0.147678767975650970f,
	-0.989035379290950310f,
	0.147489120103153680f, -0.989063678157881540f, 0.147299466807902820f,
	-0.989091940659768800f,
	0.147109808096871850f, -0.989120166795572690f, 0.146920143977033760f,
	-0.989148356564255590f,
	0.146730474455361750f, -0.989176509964781010f, 0.146540799538829870f,
	-0.989204626996113780f,
	0.146351119234411440f, -0.989232707657220050f, 0.146161433549080950f,
	-0.989260751947067640f,
	0.145971742489812370f, -0.989288759864625170f, 0.145782046063579860f,
	-0.989316731408863000f,
	0.145592344277358450f, -0.989344666578752640f, 0.145402637138122540f,
	-0.989372565373267010f,
	0.145212924652847520f, -0.989400427791380380f, 0.145023206828508360f,
	-0.989428253832068230f,
	0.144833483672080240f, -0.989456043494307710f, 0.144643755190539150f,
	-0.989483796777076760f,
	0.144454021390860440f, -0.989511513679355190f, 0.144264282280020530f,
	-0.989539194200123930f,
	0.144074537864995330f, -0.989566838338365120f, 0.143884788152761010f,
	-0.989594446093062460f,
	0.143695033150294580f, -0.989622017463200780f, 0.143505272864572290f,
	-0.989649552447766530f,
	0.143315507302571590f, -0.989677051045747210f, 0.143125736471269140f,
	-0.989704513256131850f,
	0.142935960377642700f, -0.989731939077910570f, 0.142746179028669620f,
	-0.989759328510075200f,
	0.142556392431327340f, -0.989786681551618640f, 0.142366600592594260f,
	-0.989813998201535260f,
	0.142176803519448000f, -0.989841278458820530f, 0.141987001218867340f,
	-0.989868522322471580f,
	0.141797193697830530f, -0.989895729791486660f, 0.141607380963316020f,
	-0.989922900864865450f,
	0.141417563022303130f, -0.989950035541608990f, 0.141227739881770480f,
	-0.989977133820719610f,
	0.141037911548697770f, -0.990004195701200910f, 0.140848078030064220f,
	-0.990031221182058000f,
	0.140658239332849240f, -0.990058210262297120f, 0.140468395464033110f,
	-0.990085162940925970f,
	0.140278546430595420f, -0.990112079216953770f, 0.140088692239516780f,
	-0.990138959089390650f,
	0.139898832897777380f, -0.990165802557248400f, 0.139708968412357580f,
	-0.990192609619540030f,
	0.139519098790238600f, -0.990219380275280000f, 0.139329224038400980f,
	-0.990246114523483990f,
	0.139139344163826280f, -0.990272812363169110f, 0.138949459173495440f,
	-0.990299473793353590f,
	0.138759569074390380f, -0.990326098813057330f, 0.138569673873492640f,
	-0.990352687421301340f,
	0.138379773577783890f, -0.990379239617108160f, 0.138189868194246640f,
	-0.990405755399501260f,
	0.137999957729862760f, -0.990432234767505970f, 0.137810042191615130f,
	-0.990458677720148620f,
	0.137620121586486180f, -0.990485084256456980f, 0.137430195921458550f,
	-0.990511454375460290f,
	0.137240265203515700f, -0.990537788076188750f, 0.137050329439640380f,
	-0.990564085357674370f,
	0.136860388636816430f, -0.990590346218950150f, 0.136670442802027230f,
	-0.990616570659050620f,
	0.136480491942256310f, -0.990642758677011570f, 0.136290536064488070f,
	-0.990668910271869980f,
	0.136100575175706200f, -0.990695025442664630f, 0.135910609282895440f,
	-0.990721104188435180f,
	0.135720638393040080f, -0.990747146508222710f, 0.135530662513124620f,
	-0.990773152401069780f,
	0.135340681650134330f, -0.990799121866020370f, 0.135150695811053850f,
	-0.990825054902119470f,
	0.134960705002868830f, -0.990850951508413620f, 0.134770709232564290f,
	-0.990876811683950810f,
	0.134580708507126220f, -0.990902635427780010f, 0.134390702833540240f,
	-0.990928422738951990f,
	0.134200692218792020f, -0.990954173616518500f, 0.134010676669868210f,
	-0.990979888059532740f,
	0.133820656193754690f, -0.991005566067049370f, 0.133630630797438390f,
	-0.991031207638124130f,
	0.133440600487905820f, -0.991056812771814340f, 0.133250565272143570f,
	-0.991082381467178640f,
	0.133060525157139180f, -0.991107913723276780f, 0.132870480149879400f,
	-0.991133409539170170f,
	0.132680430257352130f, -0.991158868913921350f, 0.132490375486544710f,
	-0.991184291846594180f,
	0.132300315844444680f, -0.991209678336254060f, 0.132110251338040470f,
	-0.991235028381967420f,
	0.131920181974319760f, -0.991260341982802440f, 0.131730107760271280f,
	-0.991285619137828200f,
	0.131540028702883280f, -0.991310859846115440f, 0.131349944809144220f,
	-0.991336064106736140f,
	0.131159856086043410f, -0.991361231918763460f, 0.130969762540569380f,
	-0.991386363281272280f,
	0.130779664179711790f, -0.991411458193338540f, 0.130589561010459600f,
	-0.991436516654039420f,
	0.130399453039802740f, -0.991461538662453790f, 0.130209340274730770f,
	-0.991486524217661480f,
	0.130019222722233350f, -0.991511473318743900f, 0.129829100389301010f,
	-0.991536385964783880f,
	0.129638973282923540f, -0.991561262154865290f, 0.129448841410091830f,
	-0.991586101888073500f,
	0.129258704777796270f, -0.991610905163495370f, 0.129068563393027410f,
	-0.991635671980218740f,
	0.128878417262776660f, -0.991660402337333210f, 0.128688266394034690f,
	-0.991685096233929530f,
	0.128498110793793220f, -0.991709753669099530f, 0.128307950469043590f,
	-0.991734374641936810f,
	0.128117785426777150f, -0.991758959151536110f, 0.127927615673986190f,
	-0.991783507196993490f,
	0.127737441217662280f, -0.991808018777406430f, 0.127547262064798050f,
	-0.991832493891873780f,
	0.127357078222385570f, -0.991856932539495360f, 0.127166889697417180f,
	-0.991881334719373010f,
	0.126976696496885980f, -0.991905700430609330f, 0.126786498627784430f,
	-0.991930029672308480f,
	0.126596296097105960f, -0.991954322443575950f, 0.126406088911843320f,
	-0.991978578743518580f,
	0.126215877078990400f, -0.992002798571244520f, 0.126025660605540460f,
	-0.992026981925863360f,
	0.125835439498487020f, -0.992051128806485720f, 0.125645213764824380f,
	-0.992075239212224070f,
	0.125454983411546210f, -0.992099313142191800f, 0.125264748445647110f,
	-0.992123350595503720f,
	0.125074508874121300f, -0.992147351571276090f, 0.124884264703963150f,
	-0.992171316068626520f,
	0.124694015942167770f, -0.992195244086673920f, 0.124503762595729650f,
	-0.992219135624538450f,
	0.124313504671644300f, -0.992242990681341700f, 0.124123242176906760f,
	-0.992266809256206580f,
	0.123932975118512200f, -0.992290591348257370f, 0.123742703503456630f,
	-0.992314336956619640f,
	0.123552427338735370f, -0.992338046080420420f, 0.123362146631344750f,
	-0.992361718718787870f,
	0.123171861388280650f, -0.992385354870851670f, 0.122981571616539080f,
	-0.992408954535742850f,
	0.122791277323116900f, -0.992432517712593550f, 0.122600978515010240f,
	-0.992456044400537700f,
	0.122410675199216280f, -0.992479534598709970f, 0.122220367382731500f,
	-0.992502988306246950f,
	0.122030055072553410f, -0.992526405522286100f, 0.121839738275679020f,
	-0.992549786245966570f,
	0.121649416999105540f, -0.992573130476428810f, 0.121459091249830950f,
	-0.992596438212814290f,
	0.121268761034852550f, -0.992619709454266140f, 0.121078426361168710f,
	-0.992642944199928820f,
	0.120888087235777220f, -0.992666142448948020f, 0.120697743665676120f,
	-0.992689304200470750f,
	0.120507395657864240f, -0.992712429453645460f, 0.120317043219339670f,
	-0.992735518207621850f,
	0.120126686357101580f, -0.992758570461551140f, 0.119936325078148620f,
	-0.992781586214585570f,
	0.119745959389479630f, -0.992804565465879140f, 0.119555589298094230f,
	-0.992827508214586760f,
	0.119365214810991350f, -0.992850414459865100f, 0.119174835935170960f,
	-0.992873284200871730f,
	0.118984452677632520f, -0.992896117436765980f, 0.118794065045375670f,
	-0.992918914166708300f,
	0.118603673045400840f, -0.992941674389860470f, 0.118413276684707770f,
	-0.992964398105385610f,
	0.118222875970297250f, -0.992987085312448390f, 0.118032470909169300f,
	-0.993009736010214580f,
	0.117842061508325020f, -0.993032350197851410f, 0.117651647774765000f,
	-0.993054927874527320f,
	0.117461229715489990f, -0.993077469039412300f, 0.117270807337501560f,
	-0.993099973691677570f,
	0.117080380647800550f, -0.993122441830495580f, 0.116889949653388850f,
	-0.993144873455040430f,
	0.116699514361267840f, -0.993167268564487230f, 0.116509074778439050f,
	-0.993189627158012620f,
	0.116318630911904880f, -0.993211949234794500f, 0.116128182768666920f,
	-0.993234234794012290f,
	0.115937730355727850f, -0.993256483834846440f, 0.115747273680089870f,
	-0.993278696356479030f,
	0.115556812748755290f, -0.993300872358093280f, 0.115366347568727250f,
	-0.993323011838873950f,
	0.115175878147008180f, -0.993345114798006910f, 0.114985404490601530f,
	-0.993367181234679600f,
	0.114794926606510250f, -0.993389211148080650f, 0.114604444501737460f,
	-0.993411204537400060f,
	0.114413958183287050f, -0.993433161401829360f, 0.114223467658162260f,
	-0.993455081740560960f,
	0.114032972933367300f, -0.993476965552789190f, 0.113842474015905660f,
	-0.993498812837709360f,
	0.113651970912781920f, -0.993520623594518090f, 0.113461463631000080f,
	-0.993542397822413600f,
	0.113270952177564360f, -0.993564135520595300f, 0.113080436559479720f,
	-0.993585836688263950f,
	0.112889916783750470f, -0.993607501324621610f, 0.112699392857381910f,
	-0.993629129428871720f,
	0.112508864787378830f, -0.993650721000219120f, 0.112318332580746190f,
	-0.993672276037870010f,
	0.112127796244489750f, -0.993693794541031680f, 0.111937255785614560f,
	-0.993715276508913230f,
	0.111746711211126660f, -0.993736721940724600f, 0.111556162528031630f,
	-0.993758130835677430f,
	0.111365609743335190f, -0.993779503192984580f, 0.111175052864043830f,
	-0.993800839011860120f,
	0.110984491897163380f, -0.993822138291519660f, 0.110793926849700630f,
	-0.993843401031180180f,
	0.110603357728661910f, -0.993864627230059750f, 0.110412784541053660f,
	-0.993885816887378090f,
	0.110222207293883180f, -0.993906970002356060f, 0.110031625994157000f,
	-0.993928086574215830f,
	0.109841040648882680f, -0.993949166602181130f, 0.109650451265067080f,
	-0.993970210085476920f,
	0.109459857849718030f, -0.993991217023329380f, 0.109269260409842920f,
	-0.994012187414966220f,
	0.109078658952449240f, -0.994033121259616400f, 0.108888053484545310f,
	-0.994054018556510210f,
	0.108697444013138670f, -0.994074879304879370f, 0.108506830545237980f,
	-0.994095703503956930f,
	0.108316213087851300f, -0.994116491152977070f, 0.108125591647986880f,
	-0.994137242251175720f,
	0.107934966232653760f, -0.994157956797789730f, 0.107744336848860260f,
	-0.994178634792057590f,
	0.107553703503615710f, -0.994199276233218910f, 0.107363066203928920f,
	-0.994219881120514850f,
	0.107172424956808870f, -0.994240449453187900f, 0.106981779769265340f,
	-0.994260981230481790f,
	0.106791130648307380f, -0.994281476451641550f, 0.106600477600945030f,
	-0.994301935115913580f,
	0.106409820634187840f, -0.994322357222545810f, 0.106219159755045520f,
	-0.994342742770787270f,
	0.106028494970528530f, -0.994363091759888570f, 0.105837826287646670f,
	-0.994383404189101430f,
	0.105647153713410700f, -0.994403680057679100f, 0.105456477254830660f,
	-0.994423919364875950f,
	0.105265796918917650f, -0.994444122109948040f, 0.105075112712682180f,
	-0.994464288292152390f,
	0.104884424643134970f, -0.994484417910747600f, 0.104693732717287500f,
	-0.994504510964993590f,
	0.104503036942150550f, -0.994524567454151740f, 0.104312337324735870f,
	-0.994544587377484300f,
	0.104121633872054730f, -0.994564570734255420f, 0.103930926591118540f,
	-0.994584517523730340f,
	0.103740215488939480f, -0.994604427745175660f, 0.103549500572529040f,
	-0.994624301397859400f,
	0.103358781848899700f, -0.994644138481050710f, 0.103168059325063390f,
	-0.994663938994020280f,
	0.102977333008032250f, -0.994683702936040250f, 0.102786602904819150f,
	-0.994703430306383860f,
	0.102595869022436280f, -0.994723121104325700f, 0.102405131367896790f,
	-0.994742775329142010f,
	0.102214389948213370f, -0.994762392980109930f, 0.102023644770398800f,
	-0.994781974056508260f,
	0.101832895841466670f, -0.994801518557617110f, 0.101642143168429830f,
	-0.994821026482717860f,
	0.101451386758302160f, -0.994840497831093180f, 0.101260626618096800f,
	-0.994859932602027320f,
	0.101069862754827880f, -0.994879330794805620f, 0.100879095175509010f,
	-0.994898692408714870f,
	0.100688323887153970f, -0.994918017443043200f, 0.100497548896777310f,
	-0.994937305897080070f,
	0.100306770211392820f, -0.994956557770116380f, 0.100115987838015370f,
	-0.994975773061444140f,
	0.099925201783659226f, -0.994994951770357020f, 0.099734412055338839f,
	-0.995014093896149700f,
	0.099543618660069444f, -0.995033199438118630f, 0.099352821604865513f,
	-0.995052268395561160f,
	0.099162020896742573f, -0.995071300767776170f, 0.098971216542715582f,
	-0.995090296554063890f,
	0.098780408549799664f, -0.995109255753726110f, 0.098589596925010708f,
	-0.995128178366065490f,
	0.098398781675363881f, -0.995147064390386470f, 0.098207962807875346f,
	-0.995165913825994620f,
	0.098017140329560770f, -0.995184726672196820f, 0.097826314247435903f,
	-0.995203502928301510f,
	0.097635484568517339f, -0.995222242593618240f, 0.097444651299820870f,
	-0.995240945667458130f,
	0.097253814448363354f, -0.995259612149133390f, 0.097062974021160875f,
	-0.995278242037957670f,
	0.096872130025230527f, -0.995296835333246090f, 0.096681282467588864f,
	-0.995315392034315070f,
	0.096490431355252607f, -0.995333912140482280f, 0.096299576695239225f,
	-0.995352395651066810f,
	0.096108718494565468f, -0.995370842565388990f, 0.095917856760249096f,
	-0.995389252882770690f,
	0.095726991499307315f, -0.995407626602534900f, 0.095536122718757485f,
	-0.995425963724006160f,
	0.095345250425617742f, -0.995444264246510340f, 0.095154374626905472f,
	-0.995462528169374420f,
	0.094963495329639061f, -0.995480755491926940f, 0.094772612540836410f,
	-0.995498946213497770f,
	0.094581726267515473f, -0.995517100333418110f, 0.094390836516695067f,
	-0.995535217851020390f,
	0.094199943295393190f, -0.995553298765638470f, 0.094009046610628907f,
	-0.995571343076607770f,
	0.093818146469420494f, -0.995589350783264600f, 0.093627242878787237f,
	-0.995607321884947050f,
	0.093436335845747912f, -0.995625256380994310f, 0.093245425377321389f,
	-0.995643154270746900f,
	0.093054511480527333f, -0.995661015553546910f, 0.092863594162384697f,
	-0.995678840228737540f,
	0.092672673429913366f, -0.995696628295663520f, 0.092481749290132753f,
	-0.995714379753670610f,
	0.092290821750062355f, -0.995732094602106430f, 0.092099890816722485f,
	-0.995749772840319400f,
	0.091908956497132696f, -0.995767414467659820f, 0.091718018798313525f,
	-0.995785019483478750f,
	0.091527077727284981f, -0.995802587887129160f, 0.091336133291067212f,
	-0.995820119677964910f,
	0.091145185496681130f, -0.995837614855341610f, 0.090954234351146898f,
	-0.995855073418615790f,
	0.090763279861485704f, -0.995872495367145730f, 0.090572322034718156f,
	-0.995889880700290720f,
	0.090381360877865011f, -0.995907229417411720f, 0.090190396397947820f,
	-0.995924541517870690f,
	0.089999428601987341f, -0.995941817001031350f, 0.089808457497005362f,
	-0.995959055866258320f,
	0.089617483090022917f, -0.995976258112917790f, 0.089426505388062016f,
	-0.995993423740377360f,
	0.089235524398144139f, -0.996010552748005870f, 0.089044540127290905f,
	-0.996027645135173610f,
	0.088853552582524684f, -0.996044700901251970f, 0.088662561770867121f,
	-0.996061720045614000f,
	0.088471567699340822f, -0.996078702567633980f, 0.088280570374967879f,
	-0.996095648466687300f,
	0.088089569804770507f, -0.996112557742151130f, 0.087898565995771685f,
	-0.996129430393403740f,
	0.087707558954993645f, -0.996146266419824620f, 0.087516548689459586f,
	-0.996163065820794950f,
	0.087325535206192226f, -0.996179828595696870f, 0.087134518512214321f,
	-0.996196554743914220f,
	0.086943498614549489f, -0.996213244264832040f, 0.086752475520220515f,
	-0.996229897157836500f,
	0.086561449236251239f, -0.996246513422315520f, 0.086370419769664919f,
	-0.996263093057658030f,
	0.086179387127484922f, -0.996279636063254650f, 0.085988351316735448f,
	-0.996296142438496850f,
	0.085797312344439880f, -0.996312612182778000f, 0.085606270217622613f,
	-0.996329045295492380f,
	0.085415224943307277f, -0.996345441776035900f, 0.085224176528518519f,
	-0.996361801623805720f,
	0.085033124980280414f, -0.996378124838200210f, 0.084842070305617148f,
	-0.996394411418619290f,
	0.084651012511553700f, -0.996410661364464100f, 0.084459951605114297f,
	-0.996426874675137240f,
	0.084268887593324127f, -0.996443051350042630f, 0.084077820483207846f,
	-0.996459191388585410f,
	0.083886750281790226f, -0.996475294790172160f, 0.083695676996096827f,
	-0.996491361554210920f,
	0.083504600633152404f, -0.996507391680110820f, 0.083313521199982740f,
	-0.996523385167282450f,
	0.083122438703613077f, -0.996539342015137940f, 0.082931353151068726f,
	-0.996555262223090540f,
	0.082740264549375803f, -0.996571145790554840f, 0.082549172905559659f,
	-0.996586992716946950f,
	0.082358078226646619f, -0.996602803001684130f, 0.082166980519662466f,
	-0.996618576644185070f,
	0.081975879791633108f, -0.996634313643869900f, 0.081784776049585201f,
	-0.996650014000160070f,
	0.081593669300544638f, -0.996665677712478160f, 0.081402559551538328f,
	-0.996681304780248300f,
	0.081211446809592386f, -0.996696895202896060f, 0.081020331081733912f,
	-0.996712448979848010f,
	0.080829212374989468f, -0.996727966110532490f, 0.080638090696385709f,
	-0.996743446594378860f,
	0.080446966052950097f, -0.996758890430818000f, 0.080255838451709291f,
	-0.996774297619282050f,
	0.080064707899690932f, -0.996789668159204560f, 0.079873574403922148f,
	-0.996805002050020320f,
	0.079682437971430126f, -0.996820299291165670f, 0.079491298609242866f,
	-0.996835559882078170f,
	0.079300156324387569f, -0.996850783822196610f, 0.079109011123892431f,
	-0.996865971110961310f,
	0.078917863014785095f, -0.996881121747813850f, 0.078726712004093313f,
	-0.996896235732197210f,
	0.078535558098845590f, -0.996911313063555740f, 0.078344401306069678f,
	-0.996926353741335090f,
	0.078153241632794315f, -0.996941357764982160f, 0.077962079086047645f,
	-0.996956325133945280f,
	0.077770913672857989f, -0.996971255847674320f, 0.077579745400254363f,
	-0.996986149905620180f,
	0.077388574275265049f, -0.997001007307235290f, 0.077197400304919297f,
	-0.997015828051973310f,
	0.077006223496245585f, -0.997030612139289450f, 0.076815043856273399f,
	-0.997045359568640040f,
	0.076623861392031617f, -0.997060070339482960f, 0.076432676110549283f,
	-0.997074744451277310f,
	0.076241488018856149f, -0.997089381903483400f, 0.076050297123981231f,
	-0.997103982695563330f,
	0.075859103432954503f, -0.997118546826979980f, 0.075667906952805383f,
	-0.997133074297198110f,
	0.075476707690563416f, -0.997147565105683480f, 0.075285505653258880f,
	-0.997162019251903290f,
	0.075094300847921291f, -0.997176436735326190f, 0.074903093281581137f,
	-0.997190817555421940f,
	0.074711882961268378f, -0.997205161711661850f, 0.074520669894013014f,
	-0.997219469203518670f,
	0.074329454086845867f, -0.997233740030466160f, 0.074138235546796952f,
	-0.997247974191979860f,
	0.073947014280897269f, -0.997262171687536170f, 0.073755790296177265f,
	-0.997276332516613180f,
	0.073564563599667454f, -0.997290456678690210f, 0.073373334198399157f,
	-0.997304544173247990f,
	0.073182102099402888f, -0.997318594999768600f, 0.072990867309710133f,
	-0.997332609157735470f,
	0.072799629836351618f, -0.997346586646633230f, 0.072608389686359048f,
	-0.997360527465947940f,
	0.072417146866763538f, -0.997374431615167030f, 0.072225901384596336f,
	-0.997388299093779460f,
	0.072034653246889416f, -0.997402129901275300f, 0.071843402460674000f,
	-0.997415924037145960f,
	0.071652149032982254f, -0.997429681500884180f, 0.071460892970845832f,
	-0.997443402291984360f,
	0.071269634281296415f, -0.997457086409941910f, 0.071078372971366502f,
	-0.997470733854253670f,
	0.070887109048087787f, -0.997484344624417930f, 0.070695842518492924f,
	-0.997497918719934210f,
	0.070504573389614009f, -0.997511456140303450f, 0.070313301668483263f,
	-0.997524956885027960f,
	0.070122027362133646f, -0.997538420953611230f, 0.069930750477597295f,
	-0.997551848345558430f,
	0.069739471021907376f, -0.997565239060375750f, 0.069548189002096472f,
	-0.997578593097570800f,
	0.069356904425197236f, -0.997591910456652630f, 0.069165617298243109f,
	-0.997605191137131640f,
	0.068974327628266732f, -0.997618435138519550f, 0.068783035422301728f,
	-0.997631642460329320f,
	0.068591740687380900f, -0.997644813102075420f, 0.068400443430538069f,
	-0.997657947063273710f,
	0.068209143658806454f, -0.997671044343441000f, 0.068017841379219388f,
	-0.997684104942096030f,
	0.067826536598810966f, -0.997697128858758500f, 0.067635229324614451f,
	-0.997710116092949570f,
	0.067443919563664106f, -0.997723066644191640f, 0.067252607322993652f,
	-0.997735980512008620f,
	0.067061292609636836f, -0.997748857695925690f, 0.066869975430628226f,
	-0.997761698195469560f,
	0.066678655793001543f, -0.997774502010167820f, 0.066487333703791507f,
	-0.997787269139549960f,
	0.066296009170032283f, -0.997799999583146470f, 0.066104682198758091f,
	-0.997812693340489280f,
	0.065913352797003930f, -0.997825350411111640f, 0.065722020971803977f,
	-0.997837970794548280f,
	0.065530686730193397f, -0.997850554490335110f, 0.065339350079206798f,
	-0.997863101498009500f,
	0.065148011025878860f, -0.997875611817110150f, 0.064956669577245010f,
	-0.997888085447177110f,
	0.064765325740339871f, -0.997900522387751620f, 0.064573979522199065f,
	-0.997912922638376610f,
	0.064382630929857410f, -0.997925286198596000f, 0.064191279970350679f,
	-0.997937613067955250f,
	0.063999926650714078f, -0.997949903246001190f, 0.063808570977982898f,
	-0.997962156732281950f,
	0.063617212959193190f, -0.997974373526346990f, 0.063425852601380200f,
	-0.997986553627747020f,
	0.063234489911580136f, -0.997998697036034390f, 0.063043124896828631f,
	-0.998010803750762450f,
	0.062851757564161420f, -0.998022873771486240f, 0.062660387920614985f,
	-0.998034907097761770f,
	0.062469015973224969f, -0.998046903729146840f, 0.062277641729028041f,
	-0.998058863665200250f,
	0.062086265195060247f, -0.998070786905482340f, 0.061894886378357744f,
	-0.998082673449554590f,
	0.061703505285957416f, -0.998094523296980010f, 0.061512121924895365f,
	-0.998106336447323050f,
	0.061320736302208648f, -0.998118112900149180f, 0.061129348424933755f,
	-0.998129852655025520f,
	0.060937958300107238f, -0.998141555711520520f, 0.060746565934766412f,
	-0.998153222069203650f,
	0.060555171335947781f, -0.998164851727646240f, 0.060363774510688827f,
	-0.998176444686420530f,
	0.060172375466026218f, -0.998188000945100300f, 0.059980974208997596f,
	-0.998199520503260660f,
	0.059789570746640007f, -0.998211003360478190f, 0.059598165085990598f,
	-0.998222449516330550f,
	0.059406757234087247f, -0.998233858970396850f, 0.059215347197967026f,
	-0.998245231722257880f,
	0.059023934984667986f, -0.998256567771495180f, 0.058832520601227581f,
	-0.998267867117692110f,
	0.058641104054683348f, -0.998279129760433200f, 0.058449685352073573f,
	-0.998290355699304350f,
	0.058258264500435732f, -0.998301544933892890f, 0.058066841506808263f,
	-0.998312697463787260f,
	0.057875416378229017f, -0.998323813288577560f, 0.057683989121735932f,
	-0.998334892407855000f,
	0.057492559744367684f, -0.998345934821212370f, 0.057301128253162144f,
	-0.998356940528243420f,
	0.057109694655158132f, -0.998367909528543820f, 0.056918258957393907f,
	-0.998378841821709990f,
	0.056726821166907783f, -0.998389737407340160f, 0.056535381290738825f,
	-0.998400596285033640f,
	0.056343939335925283f, -0.998411418454391300f, 0.056152495309506383f,
	-0.998422203915015020f,
	0.055961049218520520f, -0.998432952666508440f, 0.055769601070007072f,
	-0.998443664708476340f,
	0.055578150871004817f, -0.998454340040524800f, 0.055386698628552604f,
	-0.998464978662261250f,
	0.055195244349690031f, -0.998475580573294770f, 0.055003788041455885f,
	-0.998486145773235360f,
	0.054812329710889909f, -0.998496674261694640f, 0.054620869365031251f,
	-0.998507166038285490f,
	0.054429407010919147f, -0.998517621102622210f, 0.054237942655593556f,
	-0.998528039454320230f,
	0.054046476306093640f, -0.998538421092996730f, 0.053855007969459509f,
	-0.998548766018269920f,
	0.053663537652730679f, -0.998559074229759310f, 0.053472065362946755f,
	-0.998569345727086110f,
	0.053280591107148056f, -0.998579580509872500f, 0.053089114892374119f,
	-0.998589778577742230f,
	0.052897636725665401f, -0.998599939930320370f, 0.052706156614061798f,
	-0.998610064567233340f,
	0.052514674564603257f, -0.998620152488108870f, 0.052323190584330471f,
	-0.998630203692576050f,
	0.052131704680283317f, -0.998640218180265270f, 0.051940216859502626f,
	-0.998650195950808280f,
	0.051748727129028414f, -0.998660137003838490f, 0.051557235495901653f,
	-0.998670041338990070f,
	0.051365741967162731f, -0.998679908955899090f, 0.051174246549852087f,
	-0.998689739854202620f,
	0.050982749251010900f, -0.998699534033539280f, 0.050791250077679546f,
	-0.998709291493549030f,
	0.050599749036899337f, -0.998719012233872940f, 0.050408246135710995f,
	-0.998728696254153720f,
	0.050216741381155325f, -0.998738343554035230f, 0.050025234780273840f,
	-0.998747954133162860f,
	0.049833726340107257f, -0.998757527991183340f, 0.049642216067697226f,
	-0.998767065127744380f,
	0.049450703970084824f, -0.998776565542495610f, 0.049259190054311168f,
	-0.998786029235087640f,
	0.049067674327418126f, -0.998795456205172410f, 0.048876156796446746f,
	-0.998804846452403420f,
	0.048684637468439020f, -0.998814199976435390f, 0.048493116350436342f,
	-0.998823516776924380f,
	0.048301593449480172f, -0.998832796853527990f, 0.048110068772612716f,
	-0.998842040205904840f,
	0.047918542326875327f, -0.998851246833715180f, 0.047727014119310344f,
	-0.998860416736620520f,
	0.047535484156959261f, -0.998869549914283560f, 0.047343952446864526f,
	-0.998878646366368690f,
	0.047152418996068000f, -0.998887706092541290f, 0.046960883811611599f,
	-0.998896729092468410f,
	0.046769346900537960f, -0.998905715365818290f, 0.046577808269888908f,
	-0.998914664912260440f,
	0.046386267926707213f, -0.998923577731465780f, 0.046194725878035046f,
	-0.998932453823106690f,
	0.046003182130914644f, -0.998941293186856870f, 0.045811636692388955f,
	-0.998950095822391250f,
	0.045620089569500123f, -0.998958861729386080f, 0.045428540769291224f,
	-0.998967590907519300f,
	0.045236990298804750f, -0.998976283356469820f, 0.045045438165083225f,
	-0.998984939075918010f,
	0.044853884375169933f, -0.998993558065545680f, 0.044662328936107311f,
	-0.999002140325035980f,
	0.044470771854938744f, -0.999010685854073380f, 0.044279213138707016f,
	-0.999019194652343460f,
	0.044087652794454979f, -0.999027666719533690f, 0.043896090829226200f,
	-0.999036102055332330f,
	0.043704527250063421f, -0.999044500659429290f, 0.043512962064010327f,
	-0.999052862531515930f,
	0.043321395278109784f, -0.999061187671284600f, 0.043129826899405595f,
	-0.999069476078429330f,
	0.042938256934940959f, -0.999077727752645360f, 0.042746685391759139f,
	-0.999085942693629270f,
	0.042555112276904117f, -0.999094120901079070f, 0.042363537597419038f,
	-0.999102262374694130f,
	0.042171961360348002f, -0.999110367114174890f, 0.041980383572734502f,
	-0.999118435119223490f,
	0.041788804241622082f, -0.999126466389543390f, 0.041597223374055005f,
	-0.999134460924839150f,
	0.041405640977076712f, -0.999142418724816910f, 0.041214057057731589f,
	-0.999150339789184110f,
	0.041022471623063397f, -0.999158224117649430f, 0.040830884680115968f,
	-0.999166071709923000f,
	0.040639296235933854f, -0.999173882565716380f, 0.040447706297560768f,
	-0.999181656684742350f,
	0.040256114872041358f, -0.999189394066714920f, 0.040064521966419686f,
	-0.999197094711349880f,
	0.039872927587739845f, -0.999204758618363890f, 0.039681331743046659f,
	-0.999212385787475290f,
	0.039489734439384118f, -0.999219976218403530f, 0.039298135683797149f,
	-0.999227529910869610f,
	0.039106535483329839f, -0.999235046864595850f, 0.038914933845027241f,
	-0.999242527079305830f,
	0.038723330775933762f, -0.999249970554724420f, 0.038531726283093877f,
	-0.999257377290578060f,
	0.038340120373552791f, -0.999264747286594420f, 0.038148513054354856f,
	-0.999272080542502610f,
	0.037956904332545366f, -0.999279377058032710f, 0.037765294215169005f,
	-0.999286636832916740f,
	0.037573682709270514f, -0.999293859866887790f, 0.037382069821895340f,
	-0.999301046159680070f,
	0.037190455560088091f, -0.999308195711029470f, 0.036998839930894332f,
	-0.999315308520673070f,
	0.036807222941358991f, -0.999322384588349540f, 0.036615604598527057f,
	-0.999329423913798420f,
	0.036423984909444228f, -0.999336426496761240f, 0.036232363881155374f,
	-0.999343392336980220f,
	0.036040741520706299f, -0.999350321434199440f, 0.035849117835142184f,
	-0.999357213788164000f,
	0.035657492831508264f, -0.999364069398620550f, 0.035465866516850478f,
	-0.999370888265317060f,
	0.035274238898213947f, -0.999377670388002850f, 0.035082609982644702f,
	-0.999384415766428560f,
	0.034890979777187955f, -0.999391124400346050f, 0.034699348288889847f,
	-0.999397796289508640f,
	0.034507715524795889f, -0.999404431433671300f, 0.034316081491951658f,
	-0.999411029832589780f,
	0.034124446197403423f, -0.999417591486021720f, 0.033932809648196623f,
	-0.999424116393725640f,
	0.033741171851377642f, -0.999430604555461730f, 0.033549532813992221f,
	-0.999437055970991530f,
	0.033357892543086159f, -0.999443470640077770f, 0.033166251045705968f,
	-0.999449848562484530f,
	0.032974608328897315f, -0.999456189737977340f, 0.032782964399706793f,
	-0.999462494166323160f,
	0.032591319265180385f, -0.999468761847290050f, 0.032399672932364114f,
	-0.999474992780647780f,
	0.032208025408304704f, -0.999481186966166950f, 0.032016376700048046f,
	-0.999487344403620080f,
	0.031824726814640963f, -0.999493465092780590f, 0.031633075759129645f,
	-0.999499549033423640f,
	0.031441423540560343f, -0.999505596225325310f, 0.031249770165979990f,
	-0.999511606668263440f,
	0.031058115642434700f, -0.999517580362016990f, 0.030866459976971503f,
	-0.999523517306366350f,
	0.030674803176636581f, -0.999529417501093140f, 0.030483145248477058f,
	-0.999535280945980540f,
	0.030291486199539423f, -0.999541107640812940f, 0.030099826036870208f,
	-0.999546897585375960f,
	0.029908164767516655f, -0.999552650779456990f, 0.029716502398525156f,
	-0.999558367222844300f,
	0.029524838936943035f, -0.999564046915327740f, 0.029333174389816984f,
	-0.999569689856698580f,
	0.029141508764193740f, -0.999575296046749220f, 0.028949842067120746f,
	-0.999580865485273700f,
	0.028758174305644590f, -0.999586398172067070f, 0.028566505486812797f,
	-0.999591894106925950f,
	0.028374835617672258f, -0.999597353289648380f, 0.028183164705269902f,
	-0.999602775720033530f,
	0.027991492756653365f, -0.999608161397882110f, 0.027799819778869434f,
	-0.999613510322995950f,
	0.027608145778965820f, -0.999618822495178640f, 0.027416470763989606f,
	-0.999624097914234570f,
	0.027224794740987910f, -0.999629336579970110f, 0.027033117717008563f,
	-0.999634538492192300f,
	0.026841439699098527f, -0.999639703650710200f, 0.026649760694305708f,
	-0.999644832055333610f,
	0.026458080709677145f, -0.999649923705874240f, 0.026266399752260809f,
	-0.999654978602144690f,
	0.026074717829104040f, -0.999659996743959220f, 0.025883034947254208f,
	-0.999664978131133310f,
	0.025691351113759395f, -0.999669922763483760f, 0.025499666335666818f,
	-0.999674830640828740f,
	0.025307980620024630f, -0.999679701762987930f, 0.025116293973880335f,
	-0.999684536129782140f,
	0.024924606404281485f, -0.999689333741033640f, 0.024732917918276334f,
	-0.999694094596566000f,
	0.024541228522912264f, -0.999698818696204250f, 0.024349538225237600f,
	-0.999703506039774650f,
	0.024157847032300020f, -0.999708156627104880f, 0.023966154951147241f,
	-0.999712770458023870f,
	0.023774461988827676f, -0.999717347532362190f, 0.023582768152388880f,
	-0.999721887849951310f,
	0.023391073448879338f, -0.999726391410624470f, 0.023199377885346890f,
	-0.999730858214216030f,
	0.023007681468839410f, -0.999735288260561680f, 0.022815984206405477f,
	-0.999739681549498660f,
	0.022624286105092803f, -0.999744038080865430f, 0.022432587171950024f,
	-0.999748357854501780f,
	0.022240887414024919f, -0.999752640870248840f, 0.022049186838366180f,
	-0.999756887127949080f,
	0.021857485452021874f, -0.999761096627446610f, 0.021665783262040089f,
	-0.999765269368586450f,
	0.021474080275469605f, -0.999769405351215280f, 0.021282376499358355f,
	-0.999773504575180990f,
	0.021090671940755180f, -0.999777567040332940f, 0.020898966606708289f,
	-0.999781592746521670f,
	0.020707260504265912f, -0.999785581693599210f, 0.020515553640476986f,
	-0.999789533881418780f,
	0.020323846022389572f, -0.999793449309835270f, 0.020132137657052664f,
	-0.999797327978704690f,
	0.019940428551514598f, -0.999801169887884260f, 0.019748718712823757f,
	-0.999804975037232870f,
	0.019557008148029204f, -0.999808743426610520f, 0.019365296864179146f,
	-0.999812475055878780f,
	0.019173584868322699f, -0.999816169924900410f, 0.018981872167508348f,
	-0.999819828033539420f,
	0.018790158768784596f, -0.999823449381661570f, 0.018598444679200642f,
	-0.999827033969133420f,
	0.018406729905804820f, -0.999830581795823400f, 0.018215014455646376f,
	-0.999834092861600960f,
	0.018023298335773701f, -0.999837567166337090f, 0.017831581553236088f,
	-0.999841004709904000f,
	0.017639864115082195f, -0.999844405492175240f, 0.017448146028360704f,
	-0.999847769513025900f,
	0.017256427300120978f, -0.999851096772332190f, 0.017064707937411529f,
	-0.999854387269971890f,
	0.016872987947281773f, -0.999857641005823860f, 0.016681267336780482f,
	-0.999860857979768540f,
	0.016489546112956454f, -0.999864038191687680f, 0.016297824282859176f,
	-0.999867181641464380f,
	0.016106101853537263f, -0.999870288328982950f, 0.015914378832040249f,
	-0.999873358254129260f,
	0.015722655225417017f, -0.999876391416790410f, 0.015530931040716478f,
	-0.999879387816854930f,
	0.015339206284988220f, -0.999882347454212560f, 0.015147480965280975f,
	-0.999885270328754520f,
	0.014955755088644378f, -0.999888156440373320f, 0.014764028662127416f,
	-0.999891005788962950f,
	0.014572301692779104f, -0.999893818374418490f, 0.014380574187649138f,
	-0.999896594196636680f,
	0.014188846153786343f, -0.999899333255515390f, 0.013997117598240459f,
	-0.999902035550953920f,
	0.013805388528060349f, -0.999904701082852900f, 0.013613658950295789f,
	-0.999907329851114300f,
	0.013421928871995907f, -0.999909921855641540f, 0.013230198300209845f,
	-0.999912477096339240f,
	0.013038467241987433f, -0.999914995573113470f, 0.012846735704377631f,
	-0.999917477285871770f,
	0.012655003694430301f, -0.999919922234522750f, 0.012463271219194662f,
	-0.999922330418976490f,
	0.012271538285719944f, -0.999924701839144500f, 0.012079804901056066f,
	-0.999927036494939640f,
	0.011888071072252072f, -0.999929334386276070f, 0.011696336806357907f,
	-0.999931595513069200f,
	0.011504602110422875f, -0.999933819875236000f, 0.011312866991496287f,
	-0.999936007472694620f,
	0.011121131456628141f, -0.999938158305364590f, 0.010929395512867561f,
	-0.999940272373166960f,
	0.010737659167264572f, -0.999942349676023910f, 0.010545922426868548f,
	-0.999944390213859060f,
	0.010354185298728884f, -0.999946393986597460f, 0.010162447789895645f,
	-0.999948360994165400f,
	0.009970709907418029f, -0.999950291236490480f, 0.009778971658346134f,
	-0.999952184713501780f,
	0.009587233049729183f, -0.999954041425129780f, 0.009395494088617302f,
	-0.999955861371306100f,
	0.009203754782059960f, -0.999957644551963900f, 0.009012015137106642f,
	-0.999959390967037450f,
	0.008820275160807512f, -0.999961100616462820f, 0.008628534860211857f,
	-0.999962773500176930f,
	0.008436794242369860f, -0.999964409618118280f, 0.008245053314331058f,
	-0.999966008970226920f,
	0.008053312083144991f, -0.999967571556443780f, 0.007861570555861883f,
	-0.999969097376711580f,
	0.007669828739531077f, -0.999970586430974140f, 0.007478086641202815f,
	-0.999972038719176730f,
	0.007286344267926684f, -0.999973454241265940f, 0.007094601626752279f,
	-0.999974832997189810f,
	0.006902858724729877f, -0.999976174986897610f, 0.006711115568908869f,
	-0.999977480210339940f,
	0.006519372166339549f, -0.999978748667468830f, 0.006327628524071549f,
	-0.999979980358237650f,
	0.006135884649154515f, -0.999981175282601110f, 0.005944140548638765f,
	-0.999982333440515350f,
	0.005752396229573737f, -0.999983454831937730f, 0.005560651699009764f,
	-0.999984539456826970f,
	0.005368906963996303f, -0.999985587315143200f, 0.005177162031583702f,
	-0.999986598406848000f,
	0.004985416908821652f, -0.999987572731904080f, 0.004793671602759852f,
	-0.999988510290275690f,
	0.004601926120448672f, -0.999989411081928400f, 0.004410180468937601f,
	-0.999990275106828920f,
	0.004218434655277024f, -0.999991102364945590f, 0.004026688686516664f,
	-0.999991892856248010f,
	0.003834942569706248f, -0.999992646580707190f, 0.003643196311896179f,
	-0.999993363538295150f,
	0.003451449920135975f, -0.999994043728985820f, 0.003259703401476044f,
	-0.999994687152754080f,
	0.003067956762966138f, -0.999995293809576190f, 0.002876210011656010f,
	-0.999995863699429940f,
	0.002684463154596083f, -0.999996396822294350f, 0.002492716198835898f,
	-0.999996893178149880f,
	0.002300969151425887f, -0.999997352766978210f, 0.002109222019415816f,
	-0.999997775588762350f,
	0.001917474809855460f, -0.999998161643486980f, 0.001725727529795258f,
	-0.999998510931137790f,
	0.001533980186284766f, -0.999998823451701880f, 0.001342232786374430f,
	-0.999999099205167830f,
	0.001150485337113809f, -0.999999338191525530f, 0.000958737845553352f,
	-0.999999540410766110f,
	0.000766990318742846f, -0.999999705862882230f, 0.000575242763732077f,
	-0.999999834547867670f,
	0.000383495187571497f, -0.999999926465717890f, 0.000191747597310674f,
	-0.999999981616429330f
};

/**
* \par
* cosFactor tables are generated using the formula : <pre>cos_factors[n] = 2 * cos((2n+1)*pi/(4*N))</pre>
* \par
* C command to generate the table
* \par
* <pre> for(i = 0; i< N; i++)
* {
*    cos_factors[i]= 2 * cos((2*i+1)*c/2);
* } </pre>
* \par
* where <code>N</code> is the number of factors to generate and <code>c</code> is <code>pi/(2*N)</code>
*/
static const float32_t cos_factors_128[128] = {
	0.999981175282601110f, 0.999830581795823400f, 0.999529417501093140f,
	0.999077727752645360f,
	0.998475580573294770f, 0.997723066644191640f, 0.996820299291165670f,
	0.995767414467659820f,
	0.994564570734255420f, 0.993211949234794500f, 0.991709753669099530f,
	0.990058210262297120f,
	0.988257567730749460f, 0.986308097244598670f, 0.984210092386929030f,
	0.981963869109555240f,
	0.979569765685440520f, 0.977028142657754390f, 0.974339382785575860f,
	0.971503890986251780f,
	0.968522094274417380f, 0.965394441697689400f, 0.962121404269041580f,
	0.958703474895871600f,
	0.955141168305770780f, 0.951435020969008340f, 0.947585591017741090f,
	0.943593458161960390f,
	0.939459223602189920f, 0.935183509938947610f, 0.930766961078983710f,
	0.926210242138311380f,
	0.921514039342042010f, 0.916679059921042700f, 0.911706032005429880f,
	0.906595704514915330f,
	0.901348847046022030f, 0.895966249756185220f, 0.890448723244757880f,
	0.884797098430937790f,
	0.879012226428633530f, 0.873094978418290090f, 0.867046245515692650f,
	0.860866938637767310f,
	0.854557988365400530f, 0.848120344803297230f, 0.841554977436898440f,
	0.834862874986380010f,
	0.828045045257755800f, 0.821102514991104650f, 0.814036329705948410f,
	0.806847553543799330f,
	0.799537269107905010f, 0.792106577300212390f, 0.784556597155575240f,
	0.776888465673232440f,
	0.769103337645579700f, 0.761202385484261780f, 0.753186799043612520f,
	0.745057785441466060f,
	0.736816568877369900f, 0.728464390448225200f, 0.720002507961381650f,
	0.711432195745216430f,
	0.702754744457225300f, 0.693971460889654000f, 0.685083667772700360f,
	0.676092703575316030f,
	0.666999922303637470f, 0.657806693297078640f, 0.648514401022112550f,
	0.639124444863775730f,
	0.629638238914927100f, 0.620057211763289210f, 0.610382806276309480f,
	0.600616479383868970f,
	0.590759701858874280f, 0.580813958095764530f, 0.570780745886967370f,
	0.560661576197336030f,
	0.550457972936604810f, 0.540171472729892970f, 0.529803624686294830f,
	0.519355990165589530f,
	0.508830142543106990f, 0.498227666972781870f, 0.487550160148436050f,
	0.476799230063322250f,
	0.465976495767966130f, 0.455083587126343840f, 0.444122144570429260f,
	0.433093818853152010f,
	0.422000270799799790f, 0.410843171057903910f, 0.399624199845646790f,
	0.388345046698826300f,
	0.377007410216418310f, 0.365612997804773960f, 0.354163525420490510f,
	0.342660717311994380f,
	0.331106305759876430f, 0.319502030816015750f, 0.307849640041534980f,
	0.296150888243623960f,
	0.284407537211271820f, 0.272621355449948980f, 0.260794117915275570f,
	0.248927605745720260f,
	0.237023605994367340f, 0.225083911359792780f, 0.213110319916091360f,
	0.201104634842091960f,
	0.189068664149806280f, 0.177004220412148860f, 0.164913120489970090f,
	0.152797185258443410f,
	0.140658239332849240f, 0.128498110793793220f, 0.116318630911904880f,
	0.104121633872054730f,
	0.091908956497132696f, 0.079682437971430126f, 0.067443919563664106f,
	0.055195244349690031f,
	0.042938256934940959f, 0.030674803176636581f, 0.018406729905804820f,
	0.006135884649154515f
};

static const float32_t cos_factors_512[512] = {
	0.999998823451701880f, 0.999989411081928400f, 0.999970586430974140f,
	0.999942349676023910f,
	0.999904701082852900f, 0.999857641005823860f, 0.999801169887884260f,
	0.999735288260561680f,
	0.999659996743959220f, 0.999575296046749220f, 0.999481186966166950f,
	0.999377670388002850f,
	0.999264747286594420f, 0.999142418724816910f, 0.999010685854073380f,
	0.998869549914283560f,
	0.998719012233872940f, 0.998559074229759310f, 0.998389737407340160f,
	0.998211003360478190f,
	0.998022873771486240f, 0.997825350411111640f, 0.997618435138519550f,
	0.997402129901275300f,
	0.997176436735326190f, 0.996941357764982160f, 0.996696895202896060f,
	0.996443051350042630f,
	0.996179828595696980f, 0.995907229417411720f, 0.995625256380994310f,
	0.995333912140482280f,
	0.995033199438118630f, 0.994723121104325700f, 0.994403680057679100f,
	0.994074879304879370f,
	0.993736721940724600f, 0.993389211148080650f, 0.993032350197851410f,
	0.992666142448948020f,
	0.992290591348257370f, 0.991905700430609330f, 0.991511473318743900f,
	0.991107913723276890f,
	0.990695025442664630f, 0.990272812363169110f, 0.989841278458820530f,
	0.989400427791380380f,
	0.988950264510302990f, 0.988490792852696590f, 0.988022017143283530f,
	0.987543941794359230f,
	0.987056571305750970f, 0.986559910264775410f, 0.986053963346195440f,
	0.985538735312176060f,
	0.985014231012239840f, 0.984480455383220930f, 0.983937413449218920f,
	0.983385110321551180f,
	0.982823551198705240f, 0.982252741366289370f, 0.981672686196983110f,
	0.981083391150486710f,
	0.980484861773469380f, 0.979877103699517640f, 0.979260122649082020f,
	0.978633924429423210f,
	0.977998514934557140f, 0.977353900145199960f, 0.976700086128711840f,
	0.976037079039039020f,
	0.975364885116656980f, 0.974683510688510670f, 0.973992962167955830f,
	0.973293246054698250f,
	0.972584368934732210f, 0.971866337480279400f, 0.971139158449725090f,
	0.970402838687555500f,
	0.969657385124292450f, 0.968902804776428870f, 0.968139104746362440f,
	0.967366292222328510f,
	0.966584374478333120f, 0.965793358874083680f, 0.964993252854920320f,
	0.964184063951745830f,
	0.963365799780954050f, 0.962538468044359160f, 0.961702076529122540f,
	0.960856633107679660f,
	0.960002145737665960f, 0.959138622461841890f, 0.958266071408017670f,
	0.957384500788975860f,
	0.956493918902395100f, 0.955594334130771110f, 0.954685754941338340f,
	0.953768189885990330f,
	0.952841647601198720f, 0.951906136807932350f, 0.950961666311575080f,
	0.950008245001843000f,
	0.949045881852700560f, 0.948074585922276230f, 0.947094366352777220f,
	0.946105232370403450f,
	0.945107193285260610f, 0.944100258491272660f, 0.943084437466093490f,
	0.942059739771017310f,
	0.941026175050889260f, 0.939983753034014050f, 0.938932483532064600f,
	0.937872376439989890f,
	0.936803441735921560f, 0.935725689481080370f, 0.934639129819680780f,
	0.933543772978836170f,
	0.932439629268462360f, 0.931326709081180430f, 0.930205022892219070f,
	0.929074581259315860f,
	0.927935394822617890f, 0.926787474304581750f, 0.925630830509872720f,
	0.924465474325262600f,
	0.923291416719527640f, 0.922108668743345180f, 0.920917241529189520f,
	0.919717146291227360f,
	0.918508394325212250f, 0.917290997008377910f, 0.916064965799331720f,
	0.914830312237946200f,
	0.913587047945250810f, 0.912335184623322750f, 0.911074734055176360f,
	0.909805708104652220f,
	0.908528118716306120f, 0.907241977915295820f, 0.905947297807268460f,
	0.904644090578246240f,
	0.903332368494511820f, 0.902012143902493180f, 0.900683429228646970f,
	0.899346236979341570f,
	0.898000579740739880f, 0.896646470178680150f, 0.895283921038557580f,
	0.893912945145203250f,
	0.892533555402764580f, 0.891145764794583180f, 0.889749586383072780f,
	0.888345033309596350f,
	0.886932118794342190f, 0.885510856136199950f, 0.884081258712634990f,
	0.882643339979562790f,
	0.881197113471222090f, 0.879742592800047410f, 0.878279791656541580f,
	0.876808723809145650f,
	0.875329403104110890f, 0.873841843465366860f, 0.872346058894391540f,
	0.870842063470078980f,
	0.869329871348606840f, 0.867809496763303320f, 0.866280954024512990f,
	0.864744257519462380f,
	0.863199421712124160f, 0.861646461143081300f, 0.860085390429390140f,
	0.858516224264442740f,
	0.856938977417828760f, 0.855353664735196030f, 0.853760301138111410f,
	0.852158901623919830f,
	0.850549481265603480f, 0.848932055211639610f, 0.847306638685858320f,
	0.845673246987299070f,
	0.844031895490066410f, 0.842382599643185850f, 0.840725374970458070f,
	0.839060237070312740f,
	0.837387201615661940f, 0.835706284353752600f, 0.834017501106018130f,
	0.832320867767929680f,
	0.830616400308846310f, 0.828904114771864870f, 0.827184027273669130f,
	0.825456154004377550f,
	0.823720511227391430f, 0.821977115279241550f, 0.820225982569434690f,
	0.818467129580298660f,
	0.816700572866827850f, 0.814926329056526620f, 0.813144414849253590f,
	0.811354847017063730f,
	0.809557642404051260f, 0.807752817926190360f, 0.805940390571176280f,
	0.804120377398265810f,
	0.802292795538115720f, 0.800457662192622820f, 0.798614994634760820f,
	0.796764810208418830f,
	0.794907126328237010f, 0.793041960479443640f, 0.791169330217690200f,
	0.789289253168885650f,
	0.787401747029031430f, 0.785506829564053930f, 0.783604518609638200f,
	0.781694832071059390f,
	0.779777787923014550f, 0.777853404209453150f, 0.775921699043407690f,
	0.773982690606822900f,
	0.772036397150384520f, 0.770082836993347900f, 0.768122028523365420f,
	0.766153990196312920f,
	0.764178740536116670f, 0.762196298134578900f, 0.760206681651202420f,
	0.758209909813015280f,
	0.756206001414394540f, 0.754194975316889170f, 0.752176850449042810f,
	0.750151645806215070f,
	0.748119380450403600f, 0.746080073510063780f, 0.744033744179929290f,
	0.741980411720831070f,
	0.739920095459516200f, 0.737852814788465980f, 0.735778589165713590f,
	0.733697438114660370f,
	0.731609381223892630f, 0.729514438146997010f, 0.727412628602375770f,
	0.725303972373060770f,
	0.723188489306527460f, 0.721066199314508110f, 0.718937122372804490f,
	0.716801278521099540f,
	0.714658687862769090f, 0.712509370564692320f, 0.710353346857062420f,
	0.708190637033195400f,
	0.706021261449339740f, 0.703845240524484940f, 0.701662594740168570f,
	0.699473344640283770f,
	0.697277510830886630f, 0.695075113980000880f, 0.692866174817424740f,
	0.690650714134534720f,
	0.688428752784090550f, 0.686200311680038700f, 0.683965411797315510f,
	0.681724074171649820f,
	0.679476319899365080f, 0.677222170137180450f, 0.674961646102012040f,
	0.672694769070772970f,
	0.670421560380173090f, 0.668142041426518560f, 0.665856233665509720f,
	0.663564158612039880f,
	0.661265837839992270f, 0.658961292982037320f, 0.656650545729429050f,
	0.654333617831800550f,
	0.652010531096959500f, 0.649681307390683190f, 0.647345968636512060f,
	0.645004536815544040f,
	0.642657033966226860f, 0.640303482184151670f, 0.637943903621844170f,
	0.635578320488556230f,
	0.633206755050057190f, 0.630829229628424470f, 0.628445766601832710f,
	0.626056388404343520f,
	0.623661117525694640f, 0.621259976511087660f, 0.618852987960976320f,
	0.616440174530853650f,
	0.614021558931038490f, 0.611597163926462020f, 0.609167012336453210f,
	0.606731127034524480f,
	0.604289530948156070f, 0.601842247058580030f, 0.599389298400564540f,
	0.596930708062196500f,
	0.594466499184664540f, 0.591996694962040990f, 0.589521318641063940f,
	0.587040393520918080f,
	0.584553942953015330f, 0.582061990340775550f, 0.579564559139405740f,
	0.577061672855679550f,
	0.574553355047715760f, 0.572039629324757050f, 0.569520519346947250f,
	0.566996048825108680f,
	0.564466241520519500f, 0.561931121244689470f, 0.559390711859136140f,
	0.556845037275160100f,
	0.554294121453620110f, 0.551737988404707450f, 0.549176662187719770f,
	0.546610166910834860f,
	0.544038526730883930f, 0.541461765853123560f, 0.538879908531008420f,
	0.536292979065963180f,
	0.533701001807152960f, 0.531104001151255000f, 0.528502001542228480f,
	0.525895027471084740f,
	0.523283103475656430f, 0.520666254140367270f, 0.518044504095999340f,
	0.515417878019463150f,
	0.512786400633563070f, 0.510150096706766700f, 0.507508991052970870f,
	0.504863108531267480f,
	0.502212474045710900f, 0.499557112545081890f, 0.496897049022654640f,
	0.494232308515959730f,
	0.491562916106550060f, 0.488888896919763230f, 0.486210276124486530f,
	0.483527078932918740f,
	0.480839330600333900f, 0.478147056424843120f, 0.475450281747155870f,
	0.472749031950342900f,
	0.470043332459595620f, 0.467333208741988530f, 0.464618686306237820f,
	0.461899790702462840f,
	0.459176547521944150f, 0.456448982396883860f, 0.453717121000163930f,
	0.450980989045103810f,
	0.448240612285220000f, 0.445496016513981740f, 0.442747227564570130f,
	0.439994271309633260f,
	0.437237173661044200f, 0.434475960569655710f, 0.431710658025057370f,
	0.428941292055329550f,
	0.426167888726799620f, 0.423390474143796100f, 0.420609074448402510f,
	0.417823715820212380f,
	0.415034424476081630f, 0.412241226669883000f, 0.409444148692257590f,
	0.406643216870369140f,
	0.403838457567654130f, 0.401029897183575790f, 0.398217562153373620f,
	0.395401478947816300f,
	0.392581674072951530f, 0.389758174069856410f, 0.386931005514388690f,
	0.384100195016935040f,
	0.381265769222162490f, 0.378427754808765620f, 0.375586178489217330f,
	0.372741067009515810f,
	0.369892447148934270f, 0.367040345719767240f, 0.364184789567079840f,
	0.361325805568454340f,
	0.358463420633736540f, 0.355597661704783960f, 0.352728555755210730f,
	0.349856129790135030f,
	0.346980410845923680f, 0.344101425989938980f, 0.341219202320282410f,
	0.338333766965541290f,
	0.335445147084531660f, 0.332553369866044220f, 0.329658462528587550f,
	0.326760452320131790f,
	0.323859366517852960f, 0.320955232427875210f, 0.318048077385015060f,
	0.315137928752522440f,
	0.312224813921825050f, 0.309308760312268780f, 0.306389795370861080f,
	0.303467946572011370f,
	0.300543241417273400f, 0.297615707435086310f, 0.294685372180514330f,
	0.291752263234989370f,
	0.288816408206049480f, 0.285877834727080730f, 0.282936570457055390f,
	0.279992643080273380f,
	0.277046080306099950f, 0.274096909868706330f, 0.271145159526808070f,
	0.268190857063403180f,
	0.265234030285511900f, 0.262274707023913590f, 0.259312915132886350f,
	0.256348682489942910f,
	0.253382036995570270f, 0.250413006572965280f, 0.247441619167773440f,
	0.244467902747824210f,
	0.241491885302869300f, 0.238513594844318500f, 0.235533059404975460f,
	0.232550307038775330f,
	0.229565365820518870f, 0.226578263845610110f, 0.223589029229790020f,
	0.220597690108873650f,
	0.217604274638483670f, 0.214608810993786920f, 0.211611327369227610f,
	0.208611851978263460f,
	0.205610413053099320f, 0.202607038844421110f, 0.199601757621131050f,
	0.196594597670080220f,
	0.193585587295803750f, 0.190574754820252800f, 0.187562128582529740f,
	0.184547736938619640f,
	0.181531608261125130f, 0.178513770938997590f, 0.175494253377271400f,
	0.172473083996796030f,
	0.169450291233967930f, 0.166425903540464220f, 0.163399949382973230f,
	0.160372457242928400f,
	0.157343455616238280f, 0.154312973013020240f, 0.151281037957330250f,
	0.148247678986896200f,
	0.145212924652847520f, 0.142176803519448000f, 0.139139344163826280f,
	0.136100575175706200f,
	0.133060525157139180f, 0.130019222722233350f, 0.126976696496885980f,
	0.123932975118512200f,
	0.120888087235777220f, 0.117842061508325020f, 0.114794926606510250f,
	0.111746711211126660f,
	0.108697444013138670f, 0.105647153713410700f, 0.102595869022436280f,
	0.099543618660069444f,
	0.096490431355252607f, 0.093436335845747912f, 0.090381360877865011f,
	0.087325535206192226f,
	0.084268887593324127f, 0.081211446809592386f, 0.078153241632794315f,
	0.075094300847921291f,
	0.072034653246889416f, 0.068974327628266732f, 0.065913352797003930f,
	0.062851757564161420f,
	0.059789570746640007f, 0.056726821166907783f, 0.053663537652730679f,
	0.050599749036899337f,
	0.047535484156959261f, 0.044470771854938744f, 0.041405640977076712f,
	0.038340120373552791f,
	0.035274238898213947f, 0.032208025408304704f, 0.029141508764193740f,
	0.026074717829104040f,
	0.023007681468839410f, 0.019940428551514598f, 0.016872987947281773f,
	0.013805388528060349f,
	0.010737659167264572f, 0.007669828739531077f, 0.004601926120448672f,
	0.001533980186284766f
};

static const float32_t cos_factors_2048[2048] = {
	0.999999926465717890f, 0.999999338191525530f, 0.999998161643486980f,
	0.999996396822294350f,
	0.999994043728985820f, 0.999991102364945590f, 0.999987572731904080f,
	0.999983454831937730f,
	0.999978748667468830f, 0.999973454241265940f, 0.999967571556443780f,
	0.999961100616462820f,
	0.999954041425129780f, 0.999946393986597460f, 0.999938158305364590f,
	0.999929334386276070f,
	0.999919922234522750f, 0.999909921855641540f, 0.999899333255515390f,
	0.999888156440373320f,
	0.999876391416790410f, 0.999864038191687680f, 0.999851096772332190f,
	0.999837567166337090f,
	0.999823449381661570f, 0.999808743426610520f, 0.999793449309835270f,
	0.999777567040332940f,
	0.999761096627446610f, 0.999744038080865430f, 0.999726391410624470f,
	0.999708156627104880f,
	0.999689333741033640f, 0.999669922763483760f, 0.999649923705874240f,
	0.999629336579970110f,
	0.999608161397882110f, 0.999586398172067070f, 0.999564046915327740f,
	0.999541107640812940f,
	0.999517580362016990f, 0.999493465092780590f, 0.999468761847290050f,
	0.999443470640077770f,
	0.999417591486021720f, 0.999391124400346050f, 0.999364069398620550f,
	0.999336426496761240f,
	0.999308195711029470f, 0.999279377058032710f, 0.999249970554724420f,
	0.999219976218403530f,
	0.999189394066714920f, 0.999158224117649430f, 0.999126466389543390f,
	0.999094120901079070f,
	0.999061187671284600f, 0.999027666719533690f, 0.998993558065545680f,
	0.998958861729386080f,
	0.998923577731465780f, 0.998887706092541290f, 0.998851246833715180f,
	0.998814199976435390f,
	0.998776565542495610f, 0.998738343554035230f, 0.998699534033539280f,
	0.998660137003838490f,
	0.998620152488108870f, 0.998579580509872500f, 0.998538421092996730f,
	0.998496674261694640f,
	0.998454340040524800f, 0.998411418454391300f, 0.998367909528543820f,
	0.998323813288577560f,
	0.998279129760433200f, 0.998233858970396850f, 0.998188000945100300f,
	0.998141555711520520f,
	0.998094523296980010f, 0.998046903729146840f, 0.997998697036034390f,
	0.997949903246001190f,
	0.997900522387751620f, 0.997850554490335110f, 0.997799999583146470f,
	0.997748857695925690f,
	0.997697128858758500f, 0.997644813102075420f, 0.997591910456652630f,
	0.997538420953611340f,
	0.997484344624417930f, 0.997429681500884180f, 0.997374431615167150f,
	0.997318594999768600f,
	0.997262171687536170f, 0.997205161711661850f, 0.997147565105683480f,
	0.997089381903483400f,
	0.997030612139289450f, 0.996971255847674320f, 0.996911313063555740f,
	0.996850783822196610f,
	0.996789668159204560f, 0.996727966110532490f, 0.996665677712478160f,
	0.996602803001684130f,
	0.996539342015137940f, 0.996475294790172160f, 0.996410661364464100f,
	0.996345441776035900f,
	0.996279636063254650f, 0.996213244264832040f, 0.996146266419824620f,
	0.996078702567633980f,
	0.996010552748005870f, 0.995941817001031350f, 0.995872495367145730f,
	0.995802587887129160f,
	0.995732094602106430f, 0.995661015553546910f, 0.995589350783264600f,
	0.995517100333418110f,
	0.995444264246510340f, 0.995370842565388990f, 0.995296835333246090f,
	0.995222242593618360f,
	0.995147064390386470f, 0.995071300767776170f, 0.994994951770357020f,
	0.994918017443043200f,
	0.994840497831093180f, 0.994762392980109930f, 0.994683702936040250f,
	0.994604427745175660f,
	0.994524567454151740f, 0.994444122109948040f, 0.994363091759888570f,
	0.994281476451641550f,
	0.994199276233218910f, 0.994116491152977070f, 0.994033121259616400f,
	0.993949166602181130f,
	0.993864627230059750f, 0.993779503192984580f, 0.993693794541031790f,
	0.993607501324621610f,
	0.993520623594518090f, 0.993433161401829360f, 0.993345114798006910f,
	0.993256483834846440f,
	0.993167268564487230f, 0.993077469039412300f, 0.992987085312448390f,
	0.992896117436765980f,
	0.992804565465879140f, 0.992712429453645460f, 0.992619709454266140f,
	0.992526405522286100f,
	0.992432517712593660f, 0.992338046080420420f, 0.992242990681341700f,
	0.992147351571276090f,
	0.992051128806485720f, 0.991954322443575950f, 0.991856932539495470f,
	0.991758959151536110f,
	0.991660402337333210f, 0.991561262154865290f, 0.991461538662453790f,
	0.991361231918763460f,
	0.991260341982802440f, 0.991158868913921350f, 0.991056812771814340f,
	0.990954173616518500f,
	0.990850951508413620f, 0.990747146508222710f, 0.990642758677011570f,
	0.990537788076188750f,
	0.990432234767505970f, 0.990326098813057330f, 0.990219380275280000f,
	0.990112079216953770f,
	0.990004195701200910f, 0.989895729791486660f, 0.989786681551618640f,
	0.989677051045747210f,
	0.989566838338365120f, 0.989456043494307710f, 0.989344666578752640f,
	0.989232707657220050f,
	0.989120166795572690f, 0.989007044060015270f, 0.988893339517095130f,
	0.988779053233701520f,
	0.988664185277066230f, 0.988548735714763200f, 0.988432704614708340f,
	0.988316092045159690f,
	0.988198898074717610f, 0.988081122772324070f, 0.987962766207263420f,
	0.987843828449161740f,
	0.987724309567986960f, 0.987604209634049160f, 0.987483528717999710f,
	0.987362266890832400f,
	0.987240424223882250f, 0.987118000788826280f, 0.986994996657682980f,
	0.986871411902812470f,
	0.986747246596916590f, 0.986622500813038480f, 0.986497174624562880f,
	0.986371268105216030f,
	0.986244781329065460f, 0.986117714370520090f, 0.985990067304330140f,
	0.985861840205586980f,
	0.985733033149723490f, 0.985603646212513400f, 0.985473679470071810f,
	0.985343132998854790f,
	0.985212006875659350f, 0.985080301177623800f, 0.984948015982227030f,
	0.984815151367289140f,
	0.984681707410970940f, 0.984547684191773960f, 0.984413081788540700f,
	0.984277900280454370f,
	0.984142139747038570f, 0.984005800268157870f, 0.983868881924017220f,
	0.983731384795162090f,
	0.983593308962478650f, 0.983454654507193270f, 0.983315421510872810f,
	0.983175610055424420f,
	0.983035220223095640f, 0.982894252096474070f, 0.982752705758487830f,
	0.982610581292404750f,
	0.982467878781833170f, 0.982324598310721280f, 0.982180739963357090f,
	0.982036303824369020f,
	0.981891289978725100f, 0.981745698511732990f, 0.981599529509040720f,
	0.981452783056635520f,
	0.981305459240844670f, 0.981157558148334830f, 0.981009079866112630f,
	0.980860024481523870f,
	0.980710392082253970f, 0.980560182756327840f, 0.980409396592109910f,
	0.980258033678303550f,
	0.980106094103951770f, 0.979953577958436740f, 0.979800485331479790f,
	0.979646816313141210f,
	0.979492570993820810f, 0.979337749464256780f, 0.979182351815526930f,
	0.979026378139047580f,
	0.978869828526574120f, 0.978712703070200420f, 0.978555001862359550f,
	0.978396724995823090f,
	0.978237872563701090f, 0.978078444659442380f, 0.977918441376834370f,
	0.977757862810002760f,
	0.977596709053411890f, 0.977434980201864260f, 0.977272676350500860f,
	0.977109797594800880f,
	0.976946344030581670f, 0.976782315753998650f, 0.976617712861545640f,
	0.976452535450054060f,
	0.976286783616693630f, 0.976120457458971910f, 0.975953557074734300f,
	0.975786082562163930f,
	0.975618034019781750f, 0.975449411546446380f, 0.975280215241354220f,
	0.975110445204038890f,
	0.974940101534371830f, 0.974769184332561770f, 0.974597693699155050f,
	0.974425629735034990f,
	0.974252992541422500f, 0.974079782219875680f, 0.973905998872289570f,
	0.973731642600896400f,
	0.973556713508265560f, 0.973381211697303290f, 0.973205137271252800f,
	0.973028490333694210f,
	0.972851270988544180f, 0.972673479340056430f, 0.972495115492821190f,
	0.972316179551765300f,
	0.972136671622152230f, 0.971956591809581720f, 0.971775940219990140f,
	0.971594716959650160f,
	0.971412922135170940f, 0.971230555853497380f, 0.971047618221911100f,
	0.970864109348029470f,
	0.970680029339806130f, 0.970495378305530560f, 0.970310156353828110f,
	0.970124363593660280f,
	0.969938000134323960f, 0.969751066085452140f, 0.969563561557013180f,
	0.969375486659311280f,
	0.969186841502985950f, 0.968997626199012420f, 0.968807840858700970f,
	0.968617485593697540f,
	0.968426560515983190f, 0.968235065737874320f, 0.968043001372022260f,
	0.967850367531413620f,
	0.967657164329369880f, 0.967463391879547550f, 0.967269050295937790f,
	0.967074139692867040f,
	0.966878660184995910f, 0.966682611887320080f, 0.966485994915169840f,
	0.966288809384209690f,
	0.966091055410438830f, 0.965892733110190860f, 0.965693842600133690f,
	0.965494383997269500f,
	0.965294357418934660f, 0.965093762982799590f, 0.964892600806868890f,
	0.964690871009481030f,
	0.964488573709308410f, 0.964285709025357480f, 0.964082277076968140f,
	0.963878277983814200f,
	0.963673711865903230f, 0.963468578843575950f, 0.963262879037507070f,
	0.963056612568704340f,
	0.962849779558509030f, 0.962642380128595710f, 0.962434414400972100f,
	0.962225882497979020f,
	0.962016784542290560f, 0.961807120656913540f, 0.961596890965187860f,
	0.961386095590786250f,
	0.961174734657714080f, 0.960962808290309780f, 0.960750316613243950f,
	0.960537259751520050f,
	0.960323637830473920f, 0.960109450975773940f, 0.959894699313420530f,
	0.959679382969746750f,
	0.959463502071417510f, 0.959247056745430090f, 0.959030047119113660f,
	0.958812473320129310f,
	0.958594335476470220f, 0.958375633716461170f, 0.958156368168758820f,
	0.957936538962351420f,
	0.957716146226558870f, 0.957495190091032570f, 0.957273670685755200f,
	0.957051588141040970f,
	0.956828942587535370f, 0.956605734156215080f, 0.956381962978387730f,
	0.956157629185692140f,
	0.955932732910098280f, 0.955707274283906560f, 0.955481253439748770f,
	0.955254670510586990f,
	0.955027525629714160f, 0.954799818930753720f, 0.954571550547659630f,
	0.954342720614716480f,
	0.954113329266538800f, 0.953883376638071770f, 0.953652862864590500f,
	0.953421788081700310f,
	0.953190152425336670f, 0.952957956031764700f, 0.952725199037579570f,
	0.952491881579706320f,
	0.952258003795399600f, 0.952023565822243570f, 0.951788567798152130f,
	0.951553009861368590f,
	0.951316892150465550f, 0.951080214804345010f, 0.950842977962238160f,
	0.950605181763705340f,
	0.950366826348635780f, 0.950127911857248100f, 0.949888438430089300f,
	0.949648406208035480f,
	0.949407815332291570f, 0.949166665944390700f, 0.948924958186195160f,
	0.948682692199895090f,
	0.948439868128009620f, 0.948196486113385580f, 0.947952546299198670f,
	0.947708048828952100f,
	0.947462993846477700f, 0.947217381495934820f, 0.946971211921810880f,
	0.946724485268921170f,
	0.946477201682408680f, 0.946229361307743820f, 0.945980964290724760f,
	0.945732010777477150f,
	0.945482500914453740f, 0.945232434848435000f, 0.944981812726528150f,
	0.944730634696167800f,
	0.944478900905115550f, 0.944226611501459810f, 0.943973766633615980f,
	0.943720366450326200f,
	0.943466411100659320f, 0.943211900734010620f, 0.942956835500102120f,
	0.942701215548981900f,
	0.942445041031024890f, 0.942188312096931770f, 0.941931028897729620f,
	0.941673191584771360f,
	0.941414800309736340f, 0.941155855224629190f, 0.940896356481780830f,
	0.940636304233847590f,
	0.940375698633811540f, 0.940114539834980280f, 0.939852827990986680f,
	0.939590563255789270f,
	0.939327745783671400f, 0.939064375729241950f, 0.938800453247434770f,
	0.938535978493508560f,
	0.938270951623047190f, 0.938005372791958840f, 0.937739242156476970f,
	0.937472559873159250f,
	0.937205326098887960f, 0.936937540990869900f, 0.936669204706636170f,
	0.936400317404042060f,
	0.936130879241267030f, 0.935860890376814640f, 0.935590350969512370f,
	0.935319261178511610f,
	0.935047621163287430f, 0.934775431083638700f, 0.934502691099687870f,
	0.934229401371880820f,
	0.933955562060986730f, 0.933681173328098410f, 0.933406235334631520f,
	0.933130748242325230f,
	0.932854712213241120f, 0.932578127409764420f, 0.932300993994602760f,
	0.932023312130786490f,
	0.931745081981668720f, 0.931466303710925090f, 0.931186977482553750f,
	0.930907103460875130f,
	0.930626681810531760f, 0.930345712696488470f, 0.930064196284032360f,
	0.929782132738772190f,
	0.929499522226638560f, 0.929216364913884040f, 0.928932660967082820f,
	0.928648410553130520f,
	0.928363613839244370f, 0.928078270992963140f, 0.927792382182146320f,
	0.927505947574975180f,
	0.927218967339951790f, 0.926931441645899130f, 0.926643370661961230f,
	0.926354754557602860f,
	0.926065593502609310f, 0.925775887667086740f, 0.925485637221461490f,
	0.925194842336480530f,
	0.924903503183210910f, 0.924611619933039970f, 0.924319192757675160f,
	0.924026221829143850f,
	0.923732707319793290f, 0.923438649402290370f, 0.923144048249621930f,
	0.922848904035094120f,
	0.922553216932332830f, 0.922256987115283030f, 0.921960214758209220f,
	0.921662900035694730f,
	0.921365043122642340f, 0.921066644194273640f, 0.920767703426128790f,
	0.920468220994067110f,
	0.920168197074266340f, 0.919867631843222950f, 0.919566525477751530f,
	0.919264878154985370f,
	0.918962690052375630f, 0.918659961347691900f, 0.918356692219021720f,
	0.918052882844770380f,
	0.917748533403661250f, 0.917443644074735220f, 0.917138215037350710f,
	0.916832246471183890f,
	0.916525738556228210f, 0.916218691472794220f, 0.915911105401509880f,
	0.915602980523320230f,
	0.915294317019487050f, 0.914985115071589310f, 0.914675374861522390f,
	0.914365096571498560f,
	0.914054280384046570f, 0.913742926482011390f, 0.913431035048554720f,
	0.913118606267154240f,
	0.912805640321603500f, 0.912492137396012650f, 0.912178097674807180f,
	0.911863521342728520f,
	0.911548408584833990f, 0.911232759586496190f, 0.910916574533403360f,
	0.910599853611558930f,
	0.910282597007281760f, 0.909964804907205660f, 0.909646477498279540f,
	0.909327614967767260f,
	0.909008217503247450f, 0.908688285292613360f, 0.908367818524072890f,
	0.908046817386148340f,
	0.907725282067676440f, 0.907403212757808110f, 0.907080609646008450f,
	0.906757472922056550f,
	0.906433802776045460f, 0.906109599398381980f, 0.905784862979786550f,
	0.905459593711293250f,
	0.905133791784249690f, 0.904807457390316540f, 0.904480590721468250f,
	0.904153191969991780f,
	0.903825261328487510f, 0.903496798989868450f, 0.903167805147360720f,
	0.902838279994502830f,
	0.902508223725145940f, 0.902177636533453620f, 0.901846518613901750f,
	0.901514870161278740f,
	0.901182691370684520f, 0.900849982437531450f, 0.900516743557543520f,
	0.900182974926756810f,
	0.899848676741518580f, 0.899513849198487980f, 0.899178492494635330f,
	0.898842606827242370f,
	0.898506192393901950f, 0.898169249392518080f, 0.897831778021305650f,
	0.897493778478790310f,
	0.897155250963808550f, 0.896816195675507300f, 0.896476612813344120f,
	0.896136502577086770f,
	0.895795865166813530f, 0.895454700782912450f, 0.895113009626081760f,
	0.894770791897329550f,
	0.894428047797973800f, 0.894084777529641990f, 0.893740981294271040f,
	0.893396659294107720f,
	0.893051811731707450f, 0.892706438809935390f, 0.892360540731965360f,
	0.892014117701280470f,
	0.891667169921672280f, 0.891319697597241390f, 0.890971700932396860f,
	0.890623180131855930f,
	0.890274135400644600f, 0.889924566944096720f, 0.889574474967854580f,
	0.889223859677868210f,
	0.888872721280395630f, 0.888521059982002260f, 0.888168875989561730f,
	0.887816169510254440f,
	0.887462940751568840f, 0.887109189921300170f, 0.886754917227550840f,
	0.886400122878730600f,
	0.886044807083555600f, 0.885688970051048960f, 0.885332611990540590f,
	0.884975733111666660f,
	0.884618333624369920f, 0.884260413738899190f, 0.883901973665809470f,
	0.883543013615961880f,
	0.883183533800523390f, 0.882823534430966620f, 0.882463015719070150f,
	0.882101977876917580f,
	0.881740421116898320f, 0.881378345651706920f, 0.881015751694342870f,
	0.880652639458111010f,
	0.880289009156621010f, 0.879924861003786860f, 0.879560195213827890f,
	0.879195012001267480f,
	0.878829311580933360f, 0.878463094167957870f, 0.878096359977777130f,
	0.877729109226131570f,
	0.877361342129065140f, 0.876993058902925890f, 0.876624259764365310f,
	0.876254944930338510f,
	0.875885114618103810f, 0.875514769045222850f, 0.875143908429560360f,
	0.874772532989284150f,
	0.874400642942864790f, 0.874028238509075740f, 0.873655319906992630f,
	0.873281887355994210f,
	0.872907941075761080f, 0.872533481286276170f, 0.872158508207824480f,
	0.871783022060993120f,
	0.871407023066670950f, 0.871030511446048260f, 0.870653487420617430f,
	0.870275951212171940f,
	0.869897903042806340f, 0.869519343134916860f, 0.869140271711200560f,
	0.868760688994655310f,
	0.868380595208579800f, 0.867999990576573510f, 0.867618875322536230f,
	0.867237249670668400f,
	0.866855113845470430f, 0.866472468071743050f, 0.866089312574586770f,
	0.865705647579402380f,
	0.865321473311889800f, 0.864936789998049020f, 0.864551597864179340f,
	0.864165897136879300f,
	0.863779688043046720f, 0.863392970809878420f, 0.863005745664870320f,
	0.862618012835816740f,
	0.862229772550811240f, 0.861841025038245330f, 0.861451770526809320f,
	0.861062009245491480f,
	0.860671741423578380f, 0.860280967290654510f, 0.859889687076602290f,
	0.859497901011601730f,
	0.859105609326130450f, 0.858712812250963520f, 0.858319510017173440f,
	0.857925702856129790f,
	0.857531390999499150f, 0.857136574679244980f, 0.856741254127627470f,
	0.856345429577203610f,
	0.855949101260826910f, 0.855552269411646860f, 0.855154934263109620f,
	0.854757096048957220f,
	0.854358755003227440f, 0.853959911360254180f, 0.853560565354666840f,
	0.853160717221390420f,
	0.852760367195645300f, 0.852359515512947090f, 0.851958162409106380f,
	0.851556308120228980f,
	0.851153952882715340f, 0.850751096933260790f, 0.850347740508854980f,
	0.849943883846782210f,
	0.849539527184620890f, 0.849134670760243630f, 0.848729314811817130f,
	0.848323459577801640f,
	0.847917105296951410f, 0.847510252208314330f, 0.847102900551231500f,
	0.846695050565337450f,
	0.846286702490559710f, 0.845877856567119000f, 0.845468513035528830f,
	0.845058672136595470f,
	0.844648334111417820f, 0.844237499201387020f, 0.843826167648186740f,
	0.843414339693792760f,
	0.843002015580472940f, 0.842589195550786710f, 0.842175879847585570f,
	0.841762068714012490f,
	0.841347762393501950f, 0.840932961129779780f, 0.840517665166862550f,
	0.840101874749058400f,
	0.839685590120966110f, 0.839268811527475230f, 0.838851539213765760f,
	0.838433773425308340f,
	0.838015514407863820f, 0.837596762407483040f, 0.837177517670507300f,
	0.836757780443567190f,
	0.836337550973583530f, 0.835916829507766360f, 0.835495616293615350f,
	0.835073911578919410f,
	0.834651715611756440f, 0.834229028640493420f, 0.833805850913786340f,
	0.833382182680579730f,
	0.832958024190106670f, 0.832533375691888680f, 0.832108237435735590f,
	0.831682609671745120f,
	0.831256492650303210f, 0.830829886622083570f, 0.830402791838047550f,
	0.829975208549443950f,
	0.829547137007808910f, 0.829118577464965980f, 0.828689530173025820f,
	0.828259995384385660f,
	0.827829973351729920f, 0.827399464328029470f, 0.826968468566541600f,
	0.826536986320809960f,
	0.826105017844664610f, 0.825672563392221390f, 0.825239623217882250f,
	0.824806197576334330f,
	0.824372286722551250f, 0.823937890911791370f, 0.823503010399598500f,
	0.823067645441801670f,
	0.822631796294514990f, 0.822195463214137170f, 0.821758646457351750f,
	0.821321346281126740f,
	0.820883562942714580f, 0.820445296699652050f, 0.820006547809759680f,
	0.819567316531142230f,
	0.819127603122188240f, 0.818687407841569680f, 0.818246730948242070f,
	0.817805572701444270f,
	0.817363933360698460f, 0.816921813185809480f, 0.816479212436865390f,
	0.816036131374236810f,
	0.815592570258576790f, 0.815148529350820830f, 0.814704008912187080f,
	0.814259009204175270f,
	0.813813530488567190f, 0.813367573027426570f, 0.812921137083098770f,
	0.812474222918210480f,
	0.812026830795669730f, 0.811578960978665890f, 0.811130613730669190f,
	0.810681789315430780f,
	0.810232487996982330f, 0.809782710039636530f, 0.809332455707985950f,
	0.808881725266903610f,
	0.808430518981542720f, 0.807978837117336310f, 0.807526679939997160f,
	0.807074047715517610f,
	0.806620940710169650f, 0.806167359190504420f, 0.805713303423352230f,
	0.805258773675822210f,
	0.804803770215302920f, 0.804348293309460780f, 0.803892343226241260f,
	0.803435920233868120f,
	0.802979024600843250f, 0.802521656595946430f, 0.802063816488235440f,
	0.801605504547046150f,
	0.801146721041991360f, 0.800687466242961610f, 0.800227740420124790f,
	0.799767543843925680f,
	0.799306876785086160f, 0.798845739514604580f, 0.798384132303756380f,
	0.797922055424093000f,
	0.797459509147442460f, 0.796996493745908750f, 0.796533009491872000f,
	0.796069056657987990f,
	0.795604635517188070f, 0.795139746342679590f, 0.794674389407944550f,
	0.794208564986740640f,
	0.793742273353100210f, 0.793275514781330630f, 0.792808289546014120f,
	0.792340597922007170f,
	0.791872440184440470f, 0.791403816608719500f, 0.790934727470523290f,
	0.790465173045804880f,
	0.789995153610791090f, 0.789524669441982190f, 0.789053720816151880f,
	0.788582308010347120f,
	0.788110431301888070f, 0.787638090968367450f, 0.787165287287651010f,
	0.786692020537876790f,
	0.786218290997455660f, 0.785744098945070360f, 0.785269444659675850f,
	0.784794328420499230f,
	0.784318750507038920f, 0.783842711199065230f, 0.783366210776619720f,
	0.782889249520015480f,
	0.782411827709836530f, 0.781933945626937630f, 0.781455603552444590f,
	0.780976801767753750f,
	0.780497540554531910f, 0.780017820194715990f, 0.779537640970513260f,
	0.779057003164400630f,
	0.778575907059125050f, 0.778094352937702790f, 0.777612341083420030f,
	0.777129871779831620f,
	0.776646945310762060f, 0.776163561960304340f, 0.775679722012820650f,
	0.775195425752941420f,
	0.774710673465565550f, 0.774225465435860680f, 0.773739801949261840f,
	0.773253683291472590f,
	0.772767109748463850f, 0.772280081606474320f, 0.771792599152010150f,
	0.771304662671844830f,
	0.770816272453018540f, 0.770327428782838890f, 0.769838131948879840f,
	0.769348382238982280f,
	0.768858179941253270f, 0.768367525344066270f, 0.767876418736060610f,
	0.767384860406141730f,
	0.766892850643480670f, 0.766400389737514230f, 0.765907477977944340f,
	0.765414115654738270f,
	0.764920303058128410f, 0.764426040478612070f, 0.763931328206951090f,
	0.763436166534172010f,
	0.762940555751565720f, 0.762444496150687210f, 0.761947988023355390f,
	0.761451031661653620f,
	0.760953627357928150f, 0.760455775404789260f, 0.759957476095110330f,
	0.759458729722028210f,
	0.758959536578942440f, 0.758459896959515430f, 0.757959811157672300f,
	0.757459279467600720f,
	0.756958302183750490f, 0.756456879600833740f, 0.755955012013824420f,
	0.755452699717958250f,
	0.754949943008732640f, 0.754446742181906440f, 0.753943097533499640f,
	0.753439009359793580f,
	0.752934477957330150f, 0.752429503622912390f, 0.751924086653603550f,
	0.751418227346727470f,
	0.750911925999867890f, 0.750405182910869330f, 0.749897998377835330f,
	0.749390372699129560f,
	0.748882306173375150f, 0.748373799099454560f, 0.747864851776509410f,
	0.747355464503940190f,
	0.746845637581406540f, 0.746335371308826320f, 0.745824665986376090f,
	0.745313521914490520f,
	0.744801939393862630f, 0.744289918725443260f, 0.743777460210440890f,
	0.743264564150321600f,
	0.742751230846809050f, 0.742237460601884000f, 0.741723253717784140f,
	0.741208610497004260f,
	0.740693531242295760f, 0.740178016256666240f, 0.739662065843380010f,
	0.739145680305957510f,
	0.738628859948174840f, 0.738111605074064260f, 0.737593915987913570f,
	0.737075792994265730f,
	0.736557236397919150f, 0.736038246503927350f, 0.735518823617598900f,
	0.734998968044496710f,
	0.734478680090438370f, 0.733957960061495940f, 0.733436808263995710f,
	0.732915225004517780f,
	0.732393210589896040f, 0.731870765327218290f, 0.731347889523825570f,
	0.730824583487312160f,
	0.730300847525525490f, 0.729776681946566090f, 0.729252087058786970f,
	0.728727063170793830f,
	0.728201610591444610f, 0.727675729629849610f, 0.727149420595371020f,
	0.726622683797622850f,
	0.726095519546471000f, 0.725567928152032300f, 0.725039909924675370f,
	0.724511465175019630f,
	0.723982594213935520f, 0.723453297352544380f, 0.722923574902217700f,
	0.722393427174577550f,
	0.721862854481496340f, 0.721331857135096290f, 0.720800435447749190f,
	0.720268589732077190f,
	0.719736320300951030f, 0.719203627467491220f, 0.718670511545067230f,
	0.718136972847297490f,
	0.717603011688049080f, 0.717068628381437480f, 0.716533823241826680f,
	0.715998596583828690f,
	0.715462948722303760f, 0.714926879972359490f, 0.714390390649351390f,
	0.713853481068882470f,
	0.713316151546802610f, 0.712778402399208980f, 0.712240233942445510f,
	0.711701646493102970f,
	0.711162640368018350f, 0.710623215884275020f, 0.710083373359202800f,
	0.709543113110376770f,
	0.709002435455618250f, 0.708461340712994160f, 0.707919829200816310f,
	0.707377901237642100f,
	0.706835557142273860f, 0.706292797233758480f, 0.705749621831387790f,
	0.705206031254697830f,
	0.704662025823468930f, 0.704117605857725430f, 0.703572771677735580f,
	0.703027523604011220f,
	0.702481861957308000f, 0.701935787058624360f, 0.701389299229202230f,
	0.700842398790526230f,
	0.700295086064323780f, 0.699747361372564990f, 0.699199225037462120f,
	0.698650677381469580f,
	0.698101718727283880f, 0.697552349397843270f, 0.697002569716327460f,
	0.696452380006157830f,
	0.695901780590996830f, 0.695350771794747800f, 0.694799353941554900f,
	0.694247527355803310f,
	0.693695292362118350f, 0.693142649285365510f, 0.692589598450650380f,
	0.692036140183318830f,
	0.691482274808955850f, 0.690928002653386280f, 0.690373324042674040f,
	0.689818239303122470f,
	0.689262748761273470f, 0.688706852743907750f, 0.688150551578044830f,
	0.687593845590942170f,
	0.687036735110095660f, 0.686479220463238950f, 0.685921301978343670f,
	0.685362979983618730f,
	0.684804254807510620f, 0.684245126778703080f, 0.683685596226116690f,
	0.683125663478908800f,
	0.682565328866473250f, 0.682004592718440830f, 0.681443455364677990f,
	0.680881917135287340f,
	0.680319978360607200f, 0.679757639371212030f, 0.679194900497911200f,
	0.678631762071749470f,
	0.678068224424006600f, 0.677504287886197430f, 0.676939952790071240f,
	0.676375219467611700f,
	0.675810088251037060f, 0.675244559472799270f, 0.674678633465584540f,
	0.674112310562312360f,
	0.673545591096136100f, 0.672978475400442090f, 0.672410963808849900f,
	0.671843056655211930f,
	0.671274754273613490f, 0.670706056998372160f, 0.670136965164037760f,
	0.669567479105392490f,
	0.668997599157450270f, 0.668427325655456820f, 0.667856658934889440f,
	0.667285599331456480f,
	0.666714147181097670f, 0.666142302819983540f, 0.665570066584515560f,
	0.664997438811325340f,
	0.664424419837275180f, 0.663851009999457340f, 0.663277209635194100f,
	0.662703019082037440f,
	0.662128438677768720f, 0.661553468760399000f, 0.660978109668168060f,
	0.660402361739545030f,
	0.659826225313227430f, 0.659249700728141490f, 0.658672788323441890f,
	0.658095488438511290f,
	0.657517801412960120f, 0.656939727586627110f, 0.656361267299578000f,
	0.655782420892106030f,
	0.655203188704731930f, 0.654623571078202680f, 0.654043568353492640f,
	0.653463180871802330f,
	0.652882408974558960f, 0.652301253003415460f, 0.651719713300251020f,
	0.651137790207170330f,
	0.650555484066503990f, 0.649972795220807530f, 0.649389724012861770f,
	0.648806270785672550f,
	0.648222435882470420f, 0.647638219646710420f, 0.647053622422071650f,
	0.646468644552457890f,
	0.645883286381996440f, 0.645297548255038380f, 0.644711430516158420f,
	0.644124933510154540f,
	0.643538057582047850f, 0.642950803077082080f, 0.642363170340724320f,
	0.641775159718663500f,
	0.641186771556811250f, 0.640598006201301030f, 0.640008863998488440f,
	0.639419345294950700f,
	0.638829450437486400f, 0.638239179773115390f, 0.637648533649078810f,
	0.637057512412838590f,
	0.636466116412077180f, 0.635874345994697720f, 0.635282201508823530f,
	0.634689683302797850f,
	0.634096791725183740f, 0.633503527124764320f, 0.632909889850541860f,
	0.632315880251737680f,
	0.631721498677792370f, 0.631126745478365340f, 0.630531621003334600f,
	0.629936125602796550f,
	0.629340259627065750f, 0.628744023426674790f, 0.628147417352374120f,
	0.627550441755131530f,
	0.626953096986132770f, 0.626355383396779990f, 0.625757301338692900f,
	0.625158851163707730f,
	0.624560033223877320f, 0.623960847871470770f, 0.623361295458973340f,
	0.622761376339086460f,
	0.622161090864726930f, 0.621560439389027270f, 0.620959422265335180f,
	0.620358039847213830f,
	0.619756292488440660f, 0.619154180543008410f, 0.618551704365123860f,
	0.617948864309208260f,
	0.617345660729896940f, 0.616742093982038830f, 0.616138164420696910f,
	0.615533872401147430f,
	0.614929218278879590f, 0.614324202409595950f, 0.613718825149211830f,
	0.613113086853854910f,
	0.612506987879865570f, 0.611900528583796070f, 0.611293709322411010f,
	0.610686530452686280f,
	0.610078992331809620f, 0.609471095317180240f, 0.608862839766408200f,
	0.608254226037314490f,
	0.607645254487930830f, 0.607035925476499760f, 0.606426239361473550f,
	0.605816196501515080f,
	0.605205797255496500f, 0.604595041982500360f, 0.603983931041818020f,
	0.603372464792950370f,
	0.602760643595607220f, 0.602148467809707320f, 0.601535937795377730f,
	0.600923053912954090f,
	0.600309816522980430f, 0.599696225986208310f, 0.599082282663597310f,
	0.598467986916314310f,
	0.597853339105733910f, 0.597238339593437530f, 0.596622988741213330f,
	0.596007286911056530f,
	0.595391234465168730f, 0.594774831765957580f, 0.594158079176036800f,
	0.593540977058226390f,
	0.592923525775551410f, 0.592305725691242400f, 0.591687577168735550f,
	0.591069080571671510f,
	0.590450236263895920f, 0.589831044609458900f, 0.589211505972615070f,
	0.588591620717822890f,
	0.587971389209745120f, 0.587350811813247660f, 0.586729888893400500f,
	0.586108620815476430f,
	0.585487007944951450f, 0.584865050647504490f, 0.584242749289016980f,
	0.583620104235572760f,
	0.582997115853457700f, 0.582373784509160220f, 0.581750110569369760f,
	0.581126094400977620f,
	0.580501736371076600f, 0.579877036846960350f, 0.579251996196123550f,
	0.578626614786261430f,
	0.578000892985269910f, 0.577374831161244880f, 0.576748429682482520f,
	0.576121688917478390f,
	0.575494609234928230f, 0.574867191003726740f, 0.574239434592967890f,
	0.573611340371944610f,
	0.572982908710148680f, 0.572354139977270030f, 0.571725034543197120f,
	0.571095592778016690f,
	0.570465815052012990f, 0.569835701735668110f, 0.569205253199661200f,
	0.568574469814869250f,
	0.567943351952365670f, 0.567311899983420800f, 0.566680114279501710f,
	0.566047995212271560f,
	0.565415543153589770f, 0.564782758475511400f, 0.564149641550287680f,
	0.563516192750364910f,
	0.562882412448384550f, 0.562248301017183150f, 0.561613858829792420f,
	0.560979086259438260f,
	0.560343983679540860f, 0.559708551463714790f, 0.559072789985768480f,
	0.558436699619704100f,
	0.557800280739717100f, 0.557163533720196340f, 0.556526458935723720f,
	0.555889056761073920f,
	0.555251327571214090f, 0.554613271741304040f, 0.553974889646695610f,
	0.553336181662932410f,
	0.552697148165749770f, 0.552057789531074980f, 0.551418106135026060f,
	0.550778098353912230f,
	0.550137766564233630f, 0.549497111142680960f, 0.548856132466135290f,
	0.548214830911667780f,
	0.547573206856539870f, 0.546931260678202190f, 0.546288992754295210f,
	0.545646403462648590f,
	0.545003493181281160f, 0.544360262288400400f, 0.543716711162402390f,
	0.543072840181871850f,
	0.542428649725581360f, 0.541784140172491660f, 0.541139311901750910f,
	0.540494165292695230f,
	0.539848700724847700f, 0.539202918577918240f, 0.538556819231804210f,
	0.537910403066588990f,
	0.537263670462542530f, 0.536616621800121150f, 0.535969257459966710f,
	0.535321577822907010f,
	0.534673583269955510f, 0.534025274182310380f, 0.533376650941355560f,
	0.532727713928658810f,
	0.532078463525973540f, 0.531428900115236910f, 0.530779024078570250f,
	0.530128835798278850f,
	0.529478335656852090f, 0.528827524036961980f, 0.528176401321464370f,
	0.527524967893398200f,
	0.526873224135984700f, 0.526221170432628170f, 0.525568807166914680f,
	0.524916134722612890f,
	0.524263153483673470f, 0.523609863834228030f, 0.522956266158590140f,
	0.522302360841254700f,
	0.521648148266897090f, 0.520993628820373810f, 0.520338802886721960f,
	0.519683670851158520f,
	0.519028233099080970f, 0.518372490016066220f, 0.517716441987871150f,
	0.517060089400432130f,
	0.516403432639863990f, 0.515746472092461380f, 0.515089208144697270f,
	0.514431641183222930f,
	0.513773771594868030f, 0.513115599766640560f, 0.512457126085725800f,
	0.511798350939487000f,
	0.511139274715464390f, 0.510479897801375700f, 0.509820220585115560f,
	0.509160243454754750f,
	0.508499966798540810f, 0.507839391004897940f, 0.507178516462425290f,
	0.506517343559898530f,
	0.505855872686268860f, 0.505194104230662240f, 0.504532038582380380f,
	0.503869676130898950f,
	0.503207017265869030f, 0.502544062377115800f, 0.501880811854638400f,
	0.501217266088609950f,
	0.500553425469377640f, 0.499889290387461380f, 0.499224861233555030f,
	0.498560138398525200f,
	0.497895122273410930f, 0.497229813249424340f, 0.496564211717949340f,
	0.495898318070542240f,
	0.495232132698931350f, 0.494565655995016010f, 0.493898888350867430f,
	0.493231830158728070f,
	0.492564481811010650f, 0.491896843700299240f, 0.491228916219348330f,
	0.490560699761082080f,
	0.489892194718595300f, 0.489223401485152030f, 0.488554320454186230f,
	0.487884952019301210f,
	0.487215296574268820f, 0.486545354513030270f, 0.485875126229695420f,
	0.485204612118541880f,
	0.484533812574016120f, 0.483862727990732320f, 0.483191358763471910f,
	0.482519705287184520f,
	0.481847767956986080f, 0.481175547168160360f, 0.480503043316157670f,
	0.479830256796594250f,
	0.479157188005253310f, 0.478483837338084080f, 0.477810205191201040f,
	0.477136291960884750f,
	0.476462098043581310f, 0.475787623835901120f, 0.475112869734620470f,
	0.474437836136679340f,
	0.473762523439182850f, 0.473086932039400220f, 0.472411062334764100f,
	0.471734914722871430f,
	0.471058489601482610f, 0.470381787368520710f, 0.469704808422072460f,
	0.469027553160387240f,
	0.468350021981876530f, 0.467672215285114710f, 0.466994133468838110f,
	0.466315776931944480f,
	0.465637146073493770f, 0.464958241292706740f, 0.464279062988965760f,
	0.463599611561814120f,
	0.462919887410955130f, 0.462239890936253280f, 0.461559622537733190f,
	0.460879082615578690f,
	0.460198271570134270f, 0.459517189801903590f, 0.458835837711549120f,
	0.458154215699893230f,
	0.457472324167916110f, 0.456790163516757220f, 0.456107734147714220f,
	0.455425036462242420f,
	0.454742070861955450f, 0.454058837748624540f, 0.453375337524177750f,
	0.452691570590700860f,
	0.452007537350436530f, 0.451323238205783520f, 0.450638673559297760f,
	0.449953843813690580f,
	0.449268749371829920f, 0.448583390636739300f, 0.447897768011597360f,
	0.447211881899738260f,
	0.446525732704651400f, 0.445839320829980350f, 0.445152646679523590f,
	0.444465710657234110f,
	0.443778513167218280f, 0.443091054613736990f, 0.442403335401204130f,
	0.441715355934187310f,
	0.441027116617407340f, 0.440338617855737300f, 0.439649860054203420f,
	0.438960843617984430f,
	0.438271568952410480f, 0.437582036462964340f, 0.436892246555280470f,
	0.436202199635143950f,
	0.435511896108492170f, 0.434821336381412350f, 0.434130520860143310f,
	0.433439449951074200f,
	0.432748124060743760f, 0.432056543595841450f, 0.431364708963206440f,
	0.430672620569826860f,
	0.429980278822840570f, 0.429287684129534720f, 0.428594836897344400f,
	0.427901737533854240f,
	0.427208386446796370f, 0.426514784044051520f, 0.425820930733648350f,
	0.425126826923762410f,
	0.424432473022717420f, 0.423737869438983950f, 0.423043016581179100f,
	0.422347914858067000f,
	0.421652564678558380f, 0.420956966451709440f, 0.420261120586723050f,
	0.419565027492946940f,
	0.418868687579875110f, 0.418172101257146430f, 0.417475268934544340f,
	0.416778191021997590f,
	0.416080867929579320f, 0.415383300067506290f, 0.414685487846140010f,
	0.413987431675985510f,
	0.413289131967690960f, 0.412590589132048380f, 0.411891803579992220f,
	0.411192775722600160f,
	0.410493505971092520f, 0.409793994736831200f, 0.409094242431320920f,
	0.408394249466208110f,
	0.407694016253280170f, 0.406993543204466460f, 0.406292830731837470f,
	0.405591879247603870f,
	0.404890689164117750f, 0.404189260893870750f, 0.403487594849495310f,
	0.402785691443763640f,
	0.402083551089587040f, 0.401381174200016790f, 0.400678561188243350f,
	0.399975712467595390f,
	0.399272628451540930f, 0.398569309553686360f, 0.397865756187775750f,
	0.397161968767691720f,
	0.396457947707453960f, 0.395753693421220080f, 0.395049206323284880f,
	0.394344486828079650f,
	0.393639535350172880f, 0.392934352304269600f, 0.392228938105210370f,
	0.391523293167972350f,
	0.390817417907668610f, 0.390111312739546910f, 0.389404978078991100f,
	0.388698414341519250f,
	0.387991621942784910f, 0.387284601298575890f, 0.386577352824813980f,
	0.385869876937555310f,
	0.385162174052989970f, 0.384454244587440870f, 0.383746088957365010f,
	0.383037707579352130f,
	0.382329100870124510f, 0.381620269246537520f, 0.380911213125578130f,
	0.380201932924366050f,
	0.379492429060152740f, 0.378782701950320600f, 0.378072752012383990f,
	0.377362579663988450f,
	0.376652185322909620f, 0.375941569407054420f, 0.375230732334460030f,
	0.374519674523293210f,
	0.373808396391851370f, 0.373096898358560690f, 0.372385180841977360f,
	0.371673244260786630f,
	0.370961089033802040f, 0.370248715579966360f, 0.369536124318350760f,
	0.368823315668153960f,
	0.368110290048703050f, 0.367397047879452820f, 0.366683589579984930f,
	0.365969915570008910f,
	0.365256026269360380f, 0.364541922098002180f, 0.363827603476023610f,
	0.363113070823639530f,
	0.362398324561191310f, 0.361683365109145950f, 0.360968192888095290f,
	0.360252808318756830f,
	0.359537211821973180f, 0.358821403818710860f, 0.358105384730061760f,
	0.357389154977241000f,
	0.356672714981588260f, 0.355956065164567010f, 0.355239205947763370f,
	0.354522137752887430f,
	0.353804861001772160f, 0.353087376116372530f, 0.352369683518766630f,
	0.351651783631154680f,
	0.350933676875858360f, 0.350215363675321740f, 0.349496844452109600f,
	0.348778119628908420f,
	0.348059189628525780f, 0.347340054873889190f, 0.346620715788047320f,
	0.345901172794169100f,
	0.345181426315542610f, 0.344461476775576480f, 0.343741324597798600f,
	0.343020970205855540f,
	0.342300414023513690f, 0.341579656474657210f, 0.340858697983289440f,
	0.340137538973531880f,
	0.339416179869623410f, 0.338694621095921190f, 0.337972863076899830f,
	0.337250906237150650f,
	0.336528751001382350f, 0.335806397794420560f, 0.335083847041206580f,
	0.334361099166798900f,
	0.333638154596370920f, 0.332915013755212650f, 0.332191677068729320f,
	0.331468144962440920f,
	0.330744417861982890f, 0.330020496193105530f, 0.329296380381672800f,
	0.328572070853663690f,
	0.327847568035170960f, 0.327122872352400510f, 0.326397984231672660f,
	0.325672904099419900f,
	0.324947632382188430f, 0.324222169506637130f, 0.323496515899536760f,
	0.322770671987770710f,
	0.322044638198334620f, 0.321318414958334910f, 0.320592002694990330f,
	0.319865401835630610f,
	0.319138612807695900f, 0.318411636038737960f, 0.317684471956418020f,
	0.316957120988508150f,
	0.316229583562890490f, 0.315501860107556040f, 0.314773951050606070f,
	0.314045856820250820f,
	0.313317577844809070f, 0.312589114552708660f, 0.311860467372486130f,
	0.311131636732785270f,
	0.310402623062358880f, 0.309673426790066490f, 0.308944048344875710f,
	0.308214488155861220f,
	0.307484746652204160f, 0.306754824263192780f, 0.306024721418221900f,
	0.305294438546791720f,
	0.304563976078509050f, 0.303833334443086470f, 0.303102514070341060f,
	0.302371515390196130f,
	0.301640338832678880f, 0.300908984827921890f, 0.300177453806162120f,
	0.299445746197739950f,
	0.298713862433100390f, 0.297981802942791920f, 0.297249568157465890f,
	0.296517158507877410f,
	0.295784574424884370f, 0.295051816339446720f, 0.294318884682627570f,
	0.293585779885591310f,
	0.292852502379604810f, 0.292119052596036540f, 0.291385430966355720f,
	0.290651637922133220f,
	0.289917673895040860f, 0.289183539316850310f, 0.288449234619434170f,
	0.287714760234765280f,
	0.286980116594915570f, 0.286245304132057120f, 0.285510323278461380f,
	0.284775174466498300f,
	0.284039858128637360f, 0.283304374697445790f, 0.282568724605589740f,
	0.281832908285833460f,
	0.281096926171038320f, 0.280360778694163810f, 0.279624466288266700f,
	0.278887989386500280f,
	0.278151348422115090f, 0.277414543828458200f, 0.276677576038972420f,
	0.275940445487197320f,
	0.275203152606767370f, 0.274465697831413220f, 0.273728081594960650f,
	0.272990304331329980f,
	0.272252366474536660f, 0.271514268458690810f, 0.270776010717996010f,
	0.270037593686750510f,
	0.269299017799346230f, 0.268560283490267890f, 0.267821391194094320f,
	0.267082341345496350f,
	0.266343134379238180f, 0.265603770730176440f, 0.264864250833259320f,
	0.264124575123527490f,
	0.263384744036113390f, 0.262644758006240100f, 0.261904617469222560f,
	0.261164322860466590f,
	0.260423874615468010f, 0.259683273169813930f, 0.258942518959180580f,
	0.258201612419334870f,
	0.257460553986133210f, 0.256719344095520720f, 0.255977983183532380f,
	0.255236471686291820f,
	0.254494810040010790f, 0.253752998680989940f, 0.253011038045617980f,
	0.252268928570370810f,
	0.251526670691812780f, 0.250784264846594550f, 0.250041711471454650f,
	0.249299011003218300f,
	0.248556163878796620f, 0.247813170535187620f, 0.247070031409475370f,
	0.246326746938829060f,
	0.245583317560504000f, 0.244839743711840750f, 0.244096025830264210f,
	0.243352164353284880f,
	0.242608159718496890f, 0.241864012363579210f, 0.241119722726294730f,
	0.240375291244489500f,
	0.239630718356093560f, 0.238886004499120170f, 0.238141150111664870f,
	0.237396155631906550f,
	0.236651021498106460f, 0.235905748148607370f, 0.235160336021834860f,
	0.234414785556295250f,
	0.233669097190576820f, 0.232923271363349120f, 0.232177308513361770f,
	0.231431209079445730f,
	0.230684973500512310f, 0.229938602215552260f, 0.229192095663636740f,
	0.228445454283916550f,
	0.227698678515621170f, 0.226951768798059980f, 0.226204725570620270f,
	0.225457549272768540f,
	0.224710240344049570f, 0.223962799224085520f, 0.223215226352576960f,
	0.222467522169301990f,
	0.221719687114115240f, 0.220971721626949060f, 0.220223626147812460f,
	0.219475401116790340f,
	0.218727046974044600f, 0.217978564159812290f, 0.217229953114406790f,
	0.216481214278216900f,
	0.215732348091705940f, 0.214983354995412820f, 0.214234235429951100f,
	0.213484989836008080f,
	0.212735618654345870f, 0.211986122325800410f, 0.211236501291280710f,
	0.210486755991769890f,
	0.209736886868323370f, 0.208986894362070070f, 0.208236778914211470f,
	0.207486540966020700f,
	0.206736180958843660f, 0.205985699334098050f, 0.205235096533272380f,
	0.204484372997927180f,
	0.203733529169694010f, 0.202982565490274460f, 0.202231482401441620f,
	0.201480280345037820f,
	0.200728959762976140f, 0.199977521097239290f, 0.199225964789878890f,
	0.198474291283016360f,
	0.197722501018842030f, 0.196970594439614370f, 0.196218571987660850f,
	0.195466434105377090f,
	0.194714181235225990f, 0.193961813819739010f, 0.193209332301514080f,
	0.192456737123216840f,
	0.191704028727579940f, 0.190951207557401860f, 0.190198274055548120f,
	0.189445228664950340f,
	0.188692071828605260f, 0.187938803989575850f, 0.187185425590990440f,
	0.186431937076041640f,
	0.185678338887987790f, 0.184924631470150870f, 0.184170815265917720f,
	0.183416890718739230f,
	0.182662858272129360f, 0.181908718369666160f, 0.181154471454990920f,
	0.180400117971807270f,
	0.179645658363882100f, 0.178891093075044830f, 0.178136422549186320f,
	0.177381647230260200f,
	0.176626767562280960f, 0.175871783989325040f, 0.175116696955530060f,
	0.174361506905093830f,
	0.173606214282275410f, 0.172850819531394200f, 0.172095323096829040f,
	0.171339725423019260f,
	0.170584026954463700f, 0.169828228135719880f, 0.169072329411405180f,
	0.168316331226194910f,
	0.167560234024823590f, 0.166804038252083870f, 0.166047744352825850f,
	0.165291352771957970f,
	0.164534863954446110f, 0.163778278345312690f, 0.163021596389637810f,
	0.162264818532558110f,
	0.161507945219266150f, 0.160750976895011390f, 0.159993914005098350f,
	0.159236756994887850f,
	0.158479506309796100f, 0.157722162395293690f, 0.156964725696906750f,
	0.156207196660216040f,
	0.155449575730855880f, 0.154691863354515400f, 0.153934059976937460f,
	0.153176166043917870f,
	0.152418182001306500f, 0.151660108295005400f, 0.150901945370970040f,
	0.150143693675208330f,
	0.149385353653779810f, 0.148626925752796540f, 0.147868410418422360f,
	0.147109808096871850f,
	0.146351119234411440f, 0.145592344277358450f, 0.144833483672080240f,
	0.144074537864995330f,
	0.143315507302571590f, 0.142556392431327340f, 0.141797193697830530f,
	0.141037911548697770f,
	0.140278546430595420f, 0.139519098790238600f, 0.138759569074390380f,
	0.137999957729862760f,
	0.137240265203515700f, 0.136480491942256310f, 0.135720638393040080f,
	0.134960705002868830f,
	0.134200692218792020f, 0.133440600487905820f, 0.132680430257352130f,
	0.131920181974319760f,
	0.131159856086043410f, 0.130399453039802740f, 0.129638973282923540f,
	0.128878417262776660f,
	0.128117785426777150f, 0.127357078222385570f, 0.126596296097105960f,
	0.125835439498487020f,
	0.125074508874121300f, 0.124313504671644300f, 0.123552427338735370f,
	0.122791277323116900f,
	0.122030055072553410f, 0.121268761034852550f, 0.120507395657864240f,
	0.119745959389479630f,
	0.118984452677632520f, 0.118222875970297250f, 0.117461229715489990f,
	0.116699514361267840f,
	0.115937730355727850f, 0.115175878147008180f, 0.114413958183287050f,
	0.113651970912781920f,
	0.112889916783750470f, 0.112127796244489750f, 0.111365609743335190f,
	0.110603357728661910f,
	0.109841040648882680f, 0.109078658952449240f, 0.108316213087851300f,
	0.107553703503615710f,
	0.106791130648307380f, 0.106028494970528530f, 0.105265796918917650f,
	0.104503036942150550f,
	0.103740215488939480f, 0.102977333008032250f, 0.102214389948213370f,
	0.101451386758302160f,
	0.100688323887153970f, 0.099925201783659226f, 0.099162020896742573f,
	0.098398781675363881f,
	0.097635484568517339f, 0.096872130025230527f, 0.096108718494565468f,
	0.095345250425617742f,
	0.094581726267515473f, 0.093818146469420494f, 0.093054511480527333f,
	0.092290821750062355f,
	0.091527077727284981f, 0.090763279861485704f, 0.089999428601987341f,
	0.089235524398144139f,
	0.088471567699340822f, 0.087707558954993645f, 0.086943498614549489f,
	0.086179387127484922f,
	0.085415224943307277f, 0.084651012511553700f, 0.083886750281790226f,
	0.083122438703613077f,
	0.082358078226646619f, 0.081593669300544638f, 0.080829212374989468f,
	0.080064707899690932f,
	0.079300156324387569f, 0.078535558098845590f, 0.077770913672857989f,
	0.077006223496245585f,
	0.076241488018856149f, 0.075476707690563416f, 0.074711882961268378f,
	0.073947014280897269f,
	0.073182102099402888f, 0.072417146866763538f, 0.071652149032982254f,
	0.070887109048087787f,
	0.070122027362133646f, 0.069356904425197236f, 0.068591740687380900f,
	0.067826536598810966f,
	0.067061292609636836f, 0.066296009170032283f, 0.065530686730193397f,
	0.064765325740339871f,
	0.063999926650714078f, 0.063234489911580136f, 0.062469015973224969f,
	0.061703505285957416f,
	0.060937958300107238f, 0.060172375466026218f, 0.059406757234087247f,
	0.058641104054683348f,
	0.057875416378229017f, 0.057109694655158132f, 0.056343939335925283f,
	0.055578150871004817f,
	0.054812329710889909f, 0.054046476306093640f, 0.053280591107148056f,
	0.052514674564603257f,
	0.051748727129028414f, 0.050982749251010900f, 0.050216741381155325f,
	0.049450703970084824f,
	0.048684637468439020f, 0.047918542326875327f, 0.047152418996068000f,
	0.046386267926707213f,
	0.045620089569500123f, 0.044853884375169933f, 0.044087652794454979f,
	0.043321395278109784f,
	0.042555112276904117f, 0.041788804241622082f, 0.041022471623063397f,
	0.040256114872041358f,
	0.039489734439384118f, 0.038723330775933762f, 0.037956904332545366f,
	0.037190455560088091f,
	0.036423984909444228f, 0.035657492831508264f, 0.034890979777187955f,
	0.034124446197403423f,
	0.033357892543086159f, 0.032591319265180385f, 0.031824726814640963f,
	0.031058115642434700f,
	0.030291486199539423f, 0.029524838936943035f, 0.028758174305644590f,
	0.027991492756653365f,
	0.027224794740987910f, 0.026458080709677145f, 0.025691351113759395f,
	0.024924606404281485f,
	0.024157847032300020f, 0.023391073448879338f, 0.022624286105092803f,
	0.021857485452021874f,
	0.021090671940755180f, 0.020323846022389572f, 0.019557008148029204f,
	0.018790158768784596f,
	0.018023298335773701f, 0.017256427300120978f, 0.016489546112956454f,
	0.015722655225417017f,
	0.014955755088644378f, 0.014188846153786343f, 0.013421928871995907f,
	0.012655003694430301f,
	0.011888071072252072f, 0.011121131456628141f, 0.010354185298728884f,
	0.009587233049729183f,
	0.008820275160807512f, 0.008053312083144991f, 0.007286344267926684f,
	0.006519372166339549f,
	0.005752396229573737f, 0.004985416908821652f, 0.004218434655277024f,
	0.003451449920135975f,
	0.002684463154596083f, 0.001917474809855460f, 0.001150485337113809f,
	0.000383495187571497f
};

static const float32_t cos_factors_8192[8192] = {
	1.999999990808214700f, 1.999999917273932200f, 1.999999770205369800f,
	1.999999549602533100f,
	1.999999255465430200f, 1.999998887794072000f, 1.999998446588471700f,
	1.999997931848645600f,
	1.999997343574612800f, 1.999996681766395000f, 1.999995946424016200f,
	1.999995137547503600f,
	1.999994255136887000f, 1.999993299192198700f, 1.999992269713474200f,
	1.999991166700750800f,
	1.999989990154069600f, 1.999988740073473500f, 1.999987416459008600f,
	1.999986019310723500f,
	1.999984548628669600f, 1.999983004412901000f, 1.999981386663474400f,
	1.999979695380449400f,
	1.999977930563888100f, 1.999976092213855400f, 1.999974180330418700f,
	1.999972194913648900f,
	1.999970135963618400f, 1.999968003480403000f, 1.999965797464081200f,
	1.999963517914734100f,
	1.999961164832445800f, 1.999958738217302300f, 1.999956238069392900f,
	1.999953664388809800f,
	1.999951017175647600f, 1.999948296430003500f, 1.999945502151977600f,
	1.999942634341672600f,
	1.999939692999193900f, 1.999936678124649700f, 1.999933589718150700f,
	1.999930427779810900f,
	1.999927192309745900f, 1.999923883308075200f, 1.999920500774920300f,
	1.999917044710405500f,
	1.999913515114657900f, 1.999909911987807200f, 1.999906235329986100f,
	1.999902485141329400f,
	1.999898661421975400f, 1.999894764172064600f, 1.999890793391740000f,
	1.999886749081147800f,
	1.999882631240436700f, 1.999878439869758200f, 1.999874174969266300f,
	1.999869836539117700f,
	1.999865424579472000f, 1.999860939090491600f, 1.999856380072341000f,
	1.999851747525188200f,
	1.999847041449203300f, 1.999842261844559700f, 1.999837408711432600f,
	1.999832482050000900f,
	1.999827481860445300f, 1.999822408142949900f, 1.999817260897701400f,
	1.999812040124888700f,
	1.999806745824704000f, 1.999801377997341800f, 1.999795936642999600f,
	1.999790421761877400f,
	1.999784833354177900f, 1.999779171420106700f, 1.999773435959872000f,
	1.999767626973684400f,
	1.999761744461757700f, 1.999755788424308200f, 1.999749758861554900f,
	1.999743655773719400f,
	1.999737479161026100f, 1.999731229023702200f, 1.999724905361977200f,
	1.999718508176084000f,
	1.999712037466257600f, 1.999705493232735800f, 1.999698875475759600f,
	1.999692184195571900f,
	1.999685419392419000f, 1.999678581066549400f, 1.999671669218214600f,
	1.999664683847668800f,
	1.999657624955168700f, 1.999650492540973900f, 1.999643286605346800f,
	1.999636007148552400f,
	1.999628654170857900f, 1.999621227672533800f, 1.999613727653853500f,
	1.999606154115092500f,
	1.999598507056529000f, 1.999590786478444600f, 1.999582992381123000f,
	1.999575124764850800f,
	1.999567183629917100f, 1.999559168976613900f, 1.999551080805236100f,
	1.999542919116081000f,
	1.999534683909448600f, 1.999526375185641800f, 1.999517992944965800f,
	1.999509537187729200f,
	1.999501007914242600f, 1.999492405124819700f, 1.999483728819776900f,
	1.999474978999432800f,
	1.999466155664109600f, 1.999457258814131500f, 1.999448288449825500f,
	1.999439244571521700f,
	1.999430127179552500f, 1.999420936274252800f, 1.999411671855960900f,
	1.999402333925017300f,
	1.999392922481765500f, 1.999383437526551300f, 1.999373879059723500f,
	1.999364247081633500f,
	1.999354541592635500f, 1.999344762593086500f, 1.999334910083345700f,
	1.999324984063775700f,
	1.999314984534741100f, 1.999304911496609700f, 1.999294764949752100f,
	1.999284544894541100f,
	1.999274251331352400f, 1.999263884260564600f, 1.999253443682558900f,
	1.999242929597719200f,
	1.999232342006432000f, 1.999221680909086400f, 1.999210946306074500f,
	1.999200138197791100f,
	1.999189256584633600f, 1.999178301467001900f, 1.999167272845298900f,
	1.999156170719930100f,
	1.999144995091303600f, 1.999133745959830600f, 1.999122423325924200f,
	1.999111027190001000f,
	1.999099557552479900f, 1.999088014413782800f, 1.999076397774334000f,
	1.999064707634560700f,
	1.999052943994892300f, 1.999041106855761900f, 1.999029196217604100f,
	1.999017212080857400f,
	1.999005154445962200f, 1.998993023313361700f, 1.998980818683502100f,
	1.998968540556831800f,
	1.998956188933802800f, 1.998943763814868800f, 1.998931265200486900f,
	1.998918693091116200f,
	1.998906047487219600f, 1.998893328389261400f, 1.998880535797709700f,
	1.998867669713034500f,
	1.998854730135709400f, 1.998841717066209400f, 1.998828630505013400f,
	1.998815470452602400f,
	1.998802236909460500f, 1.998788929876074100f, 1.998775549352932400f,
	1.998762095340527400f,
	1.998748567839354000f, 1.998734966849909000f, 1.998721292372693100f,
	1.998707544408208700f,
	1.998693722956961500f, 1.998679828019459300f, 1.998665859596213500f,
	1.998651817687737300f,
	1.998637702294547000f, 1.998623513417161700f, 1.998609251056103100f,
	1.998594915211895600f,
	1.998580505885066100f, 1.998566023076144600f, 1.998551466785663400f,
	1.998536837014157900f,
	1.998522133762165900f, 1.998507357030227900f, 1.998492506818887200f,
	1.998477583128690100f,
	1.998462585960185000f, 1.998447515313923400f, 1.998432371190459500f,
	1.998417153590349900f,
	1.998401862514154200f, 1.998386497962434800f, 1.998371059935756300f,
	1.998355548434686400f,
	1.998339963459795400f, 1.998324305011656600f, 1.998308573090845200f,
	1.998292767697940100f,
	1.998276888833522300f, 1.998260936498175400f, 1.998244910692486000f,
	1.998228811417043700f,
	1.998212638672439900f, 1.998196392459269400f, 1.998180072778129600f,
	1.998163679629620500f,
	1.998147213014344900f, 1.998130672932908000f, 1.998114059385918400f,
	1.998097372373986300f,
	1.998080611897725700f, 1.998063777957752600f, 1.998046870554686100f,
	1.998029889689147700f,
	1.998012835361761900f, 1.997995707573155600f, 1.997978506323958600f,
	1.997961231614803200f,
	1.997943883446324800f, 1.997926461819161000f, 1.997908966733952500f,
	1.997891398191342400f,
	1.997873756191977000f, 1.997856040736504500f, 1.997838251825576400f,
	1.997820389459846700f,
	1.997802453639972300f, 1.997784444366612600f, 1.997766361640429800f,
	1.997748205462088500f,
	1.997729975832256600f, 1.997711672751604200f, 1.997693296220804000f,
	1.997674846240532000f,
	1.997656322811466500f, 1.997637725934288300f, 1.997619055609681600f,
	1.997600311838332500f,
	1.997581494620930300f, 1.997562603958166600f, 1.997543639850736200f,
	1.997524602299336500f,
	1.997505491304667000f, 1.997486306867430900f, 1.997467048988333000f,
	1.997447717668082000f,
	1.997428312907388200f, 1.997408834706965000f, 1.997389283067528800f,
	1.997369657989798400f,
	1.997349959474495200f, 1.997330187522343700f, 1.997310342134070800f,
	1.997290423310406100f,
	1.997270431052081900f, 1.997250365359833200f, 1.997230226234397900f,
	1.997210013676516700f,
	1.997189727686932400f, 1.997169368266390900f, 1.997148935415640600f,
	1.997128429135433400f,
	1.997107849426522600f, 1.997087196289665000f, 1.997066469725620200f,
	1.997045669735150000f,
	1.997024796319019300f, 1.997003849477995600f, 1.996982829212848900f,
	1.996961735524351900f,
	1.996940568413280600f, 1.996919327880412900f, 1.996898013926530000f,
	1.996876626552415400f,
	1.996855165758855600f, 1.996833631546639300f, 1.996812023916558800f,
	1.996790342869408000f,
	1.996768588405984300f, 1.996746760527087700f, 1.996724859233520500f,
	1.996702884526087900f,
	1.996680836405598100f, 1.996658714872861800f, 1.996636519928692000f,
	1.996614251573904900f,
	1.996591909809319400f, 1.996569494635756600f, 1.996547006054041100f,
	1.996524444064999400f,
	1.996501808669461000f, 1.996479099868258400f, 1.996456317662226300f,
	1.996433462052202600f,
	1.996410533039027400f, 1.996387530623543900f, 1.996364454806597500f,
	1.996341305589037100f,
	1.996318082971713500f, 1.996294786955480800f, 1.996271417541195300f,
	1.996247974729716200f,
	1.996224458521905600f, 1.996200868918628100f, 1.996177205920750800f,
	1.996153469529144100f,
	1.996129659744680300f, 1.996105776568235100f, 1.996081820000686500f,
	1.996057790042915500f,
	1.996033686695805300f, 1.996009509960242400f, 1.995985259837115500f,
	1.995960936327316300f,
	1.995936539431739000f, 1.995912069151280800f, 1.995887525486841300f,
	1.995862908439323100f,
	1.995838218009630800f, 1.995813454198672700f, 1.995788617007359100f,
	1.995763706436603200f,
	1.995738722487320600f, 1.995713665160430600f, 1.995688534456853800f,
	1.995663330377514400f,
	1.995638052923339300f, 1.995612702095257400f, 1.995587277894201400f,
	1.995561780321105600f,
	1.995536209376907600f, 1.995510565062547800f, 1.995484847378968600f,
	1.995459056327116000f,
	1.995433191907938000f, 1.995407254122385700f, 1.995381242971412600f,
	1.995355158455975200f,
	1.995329000577032800f, 1.995302769335546500f, 1.995276464732481200f,
	1.995250086768804100f,
	1.995223635445484900f, 1.995197110763496000f, 1.995170512723813100f,
	1.995143841327413400f,
	1.995117096575278200f, 1.995090278468390600f, 1.995063387007736600f,
	1.995036422194304700f,
	1.995009384029086800f, 1.994982272513076600f, 1.994955087647271000f,
	1.994927829432669800f,
	1.994900497870274900f, 1.994873092961091200f, 1.994845614706126400f,
	1.994818063106391000f,
	1.994790438162897600f, 1.994762739876662100f, 1.994734968248702800f,
	1.994707123280041100f,
	1.994679204971700100f, 1.994651213324707000f, 1.994623148340090700f,
	1.994595010018883000f,
	1.994566798362118300f, 1.994538513370834200f, 1.994510155046070700f,
	1.994481723388870100f,
	1.994453218400277900f, 1.994424640081342100f, 1.994395988433113700f,
	1.994367263456646100f,
	1.994338465152995000f, 1.994309593523219600f, 1.994280648568381500f,
	1.994251630289544600f,
	1.994222538687776100f, 1.994193373764145500f, 1.994164135519725000f,
	1.994134823955589800f,
	1.994105439072817700f, 1.994075980872488800f, 1.994046449355686200f,
	1.994016844523496000f,
	1.993987166377006600f, 1.993957414917308700f, 1.993927590145496900f,
	1.993897692062667200f,
	1.993867720669919400f, 1.993837675968354700f, 1.993807557959078600f,
	1.993777366643197900f,
	1.993747102021822900f, 1.993716764096066200f, 1.993686352867043200f,
	1.993655868335872300f,
	1.993625310503674100f, 1.993594679371572200f, 1.993563974940692800f,
	1.993533197212164800f,
	1.993502346187119700f, 1.993471421866692200f, 1.993440424252018900f,
	1.993409353344239600f,
	1.993378209144496700f, 1.993346991653935300f, 1.993315700873703200f,
	1.993284336804950900f,
	1.993252899448831400f, 1.993221388806500900f, 1.993189804879117500f,
	1.993158147667842800f,
	1.993126417173840500f, 1.993094613398277400f, 1.993062736342323000f,
	1.993030786007148800f,
	1.992998762393930000f, 1.992966665503844000f, 1.992934495338070800f,
	1.992902251897793000f,
	1.992869935184196300f, 1.992837545198469000f, 1.992805081941801700f,
	1.992772545415388200f,
	1.992739935620424700f, 1.992707252558110200f, 1.992674496229646500f,
	1.992641666636237700f,
	1.992608763779091000f, 1.992575787659416100f, 1.992542738278425300f,
	1.992509615637334100f,
	1.992476419737359900f, 1.992443150579723500f, 1.992409808165648100f,
	1.992376392496359300f,
	1.992342903573086000f, 1.992309341397059600f, 1.992275705969513800f,
	1.992241997291685400f,
	1.992208215364813700f, 1.992174360190140900f, 1.992140431768911500f,
	1.992106430102373400f,
	1.992072355191776300f, 1.992038207038373300f, 1.992003985643419700f,
	1.991969691008174100f,
	1.991935323133897000f, 1.991900882021852200f, 1.991866367673306200f,
	1.991831780089527500f,
	1.991797119271788300f, 1.991762385221362600f, 1.991727577939527600f,
	1.991692697427563300f,
	1.991657743686751700f, 1.991622716718378400f, 1.991587616523731000f,
	1.991552443104099800f,
	1.991517196460778500f, 1.991481876595062800f, 1.991446483508251500f,
	1.991411017201645500f,
	1.991375477676549100f, 1.991339864934268800f, 1.991304178976114100f,
	1.991268419803397200f,
	1.991232587417432600f, 1.991196681819537900f, 1.991160703011033200f,
	1.991124650993241400f,
	1.991088525767488200f, 1.991052327335101300f, 1.991016055697411900f,
	1.990979710855753900f,
	1.990943292811463000f, 1.990906801565878600f, 1.990870237120342400f,
	1.990833599476198800f,
	1.990796888634794400f, 1.990760104597479400f, 1.990723247365606200f,
	1.990686316940529800f,
	1.990649313323608100f, 1.990612236516201300f, 1.990575086519673200f,
	1.990537863335389400f,
	1.990500566964718400f, 1.990463197409031700f, 1.990425754669703100f,
	1.990388238748109100f,
	1.990350649645629600f, 1.990312987363646000f, 1.990275251903543600f,
	1.990237443266709400f,
	1.990199561454533600f, 1.990161606468409300f, 1.990123578309731700f,
	1.990085476979899000f,
	1.990047302480312300f, 1.990009054812374800f, 1.989970733977493000f,
	1.989932339977075900f,
	1.989893872812535000f, 1.989855332485284800f, 1.989816718996742200f,
	1.989778032348326700f,
	1.989739272541461100f, 1.989700439577570400f, 1.989661533458082100f,
	1.989622554184426800f,
	1.989583501758037700f, 1.989544376180350600f, 1.989505177452804100f,
	1.989465905576839600f,
	1.989426560553900500f, 1.989387142385433900f, 1.989347651072888900f,
	1.989308086617717500f,
	1.989268449021374300f, 1.989228738285316900f, 1.989188954411005100f,
	1.989149097399901500f,
	1.989109167253472000f, 1.989069163973184300f, 1.989029087560509700f,
	1.988988938016921000f,
	1.988948715343894900f, 1.988908419542910100f, 1.988868050615448100f,
	1.988827608562993200f,
	1.988787093387032600f, 1.988746505089055600f, 1.988705843670554500f,
	1.988665109133024500f,
	1.988624301477963200f, 1.988583420706871100f, 1.988542466821251000f,
	1.988501439822608900f,
	1.988460339712453200f, 1.988419166492295000f, 1.988377920163648000f,
	1.988336600728029000f,
	1.988295208186956700f, 1.988253742541953800f, 1.988212203794544000f,
	1.988170591946255100f,
	1.988128906998616800f, 1.988087148953161700f, 1.988045317811425700f,
	1.988003413574946000f,
	1.987961436245263800f, 1.987919385823922400f, 1.987877262312467600f,
	1.987835065712448600f,
	1.987792796025416500f, 1.987750453252925500f, 1.987708037396532800f,
	1.987665548457797400f,
	1.987622986438281700f, 1.987580351339550700f, 1.987537643163171700f,
	1.987494861910715100f,
	1.987452007583754100f, 1.987409080183863800f, 1.987366079712622900f,
	1.987323006171612500f,
	1.987279859562415900f, 1.987236639886619700f, 1.987193347145813000f,
	1.987149981341587400f,
	1.987106542475537400f, 1.987063030549260300f, 1.987019445564355700f,
	1.986975787522426100f,
	1.986932056425076800f, 1.986888252273915500f, 1.986844375070552900f,
	1.986800424816602200f,
	1.986756401513679400f, 1.986712305163403000f, 1.986668135767394300f,
	1.986623893327277500f,
	1.986579577844678900f, 1.986535189321228000f, 1.986490727758556800f,
	1.986446193158300400f,
	1.986401585522095600f, 1.986356904851583000f, 1.986312151148405200f,
	1.986267324414207500f,
	1.986222424650638400f, 1.986177451859348200f, 1.986132406041990900f,
	1.986087287200222700f,
	1.986042095335702300f, 1.985996830450091200f, 1.985951492545054100f,
	1.985906081622257300f,
	1.985860597683371000f, 1.985815040730067200f, 1.985769410764020900f,
	1.985723707786909900f,
	1.985677931800414500f, 1.985632082806217900f, 1.985586160806005700f,
	1.985540165801466200f,
	1.985494097794290800f, 1.985447956786173100f, 1.985401742778809500f,
	1.985355455773899500f,
	1.985309095773144500f, 1.985262662778249300f, 1.985216156790921000f,
	1.985169577812869500f,
	1.985122925845807400f, 1.985076200891450000f, 1.985029402951515200f,
	1.984982532027723700f,
	1.984935588121798700f, 1.984888571235466200f, 1.984841481370454900f,
	1.984794318528496200f,
	1.984747082711324100f, 1.984699773920675300f, 1.984652392158289500f,
	1.984604937425908300f,
	1.984557409725276700f, 1.984509809058142300f, 1.984462135426255000f,
	1.984414388831367900f,
	1.984366569275236400f, 1.984318676759618400f, 1.984270711286275200f,
	1.984222672856969800f,
	1.984174561473469200f, 1.984126377137541700f, 1.984078119850959200f,
	1.984029789615495900f,
	1.983981386432928800f, 1.983932910305037400f, 1.983884361233604100f,
	1.983835739220414000f,
	1.983787044267254700f, 1.983738276375916800f, 1.983689435548192900f,
	1.983640521785879200f,
	1.983591535090773800f, 1.983542475464678000f, 1.983493342909395500f,
	1.983444137426732600f,
	1.983394859018498900f, 1.983345507686505900f, 1.983296083432567900f,
	1.983246586258502700f,
	1.983197016166129400f, 1.983147373157271300f, 1.983097657233753100f,
	1.983047868397403100f,
	1.982998006650051400f, 1.982948071993531700f, 1.982898064429679900f,
	1.982847983960334600f,
	1.982797830587336800f, 1.982747604312531200f, 1.982697305137763700f,
	1.982646933064884200f,
	1.982596488095744300f, 1.982545970232199000f, 1.982495379476105800f,
	1.982444715829324600f,
	1.982393979293718200f, 1.982343169871152000f, 1.982292287563494300f,
	1.982241332372615600f,
	1.982190304300389400f, 1.982139203348692200f, 1.982088029519402300f,
	1.982036782814401900f,
	1.981985463235574700f, 1.981934070784807400f, 1.981882605463990200f,
	1.981831067275015000f,
	1.981779456219776600f, 1.981727772300172500f, 1.981676015518103500f,
	1.981624185875472000f,
	1.981572283374183800f, 1.981520308016147200f, 1.981468259803273300f,
	1.981416138737475800f,
	1.981363944820670800f, 1.981311678054777500f, 1.981259338441717400f,
	1.981206925983415300f,
	1.981154440681797800f, 1.981101882538794900f, 1.981049251556338900f,
	1.980996547736364900f,
	1.980943771080810700f, 1.980890921591616600f, 1.980837999270726100f,
	1.980785004120084700f,
	1.980731936141640900f, 1.980678795337345900f, 1.980625581709153600f,
	1.980572295259020600f,
	1.980518935988905700f, 1.980465503900771000f, 1.980411998996581200f,
	1.980358421278303200f,
	1.980304770747907300f, 1.980251047407365600f, 1.980197251258653900f,
	1.980143382303749500f,
	1.980089440544633600f, 1.980035425983289300f, 1.979981338621702200f,
	1.979927178461861500f,
	1.979872945505758000f, 1.979818639755386100f, 1.979764261212742400f,
	1.979709809879825800f,
	1.979655285758638900f, 1.979600688851186100f, 1.979546019159474900f,
	1.979491276685515300f,
	1.979436461431320000f, 1.979381573398904400f, 1.979326612590286400f,
	1.979271579007487100f,
	1.979216472652529900f, 1.979161293527440500f, 1.979106041634248100f,
	1.979050716974983800f,
	1.978995319551682100f, 1.978939849366379700f, 1.978884306421115900f,
	1.978828690717932900f,
	1.978773002258875600f, 1.978717241045991700f, 1.978661407081331100f,
	1.978605500366946700f,
	1.978549520904894000f, 1.978493468697231300f, 1.978437343746019600f,
	1.978381146053322000f,
	1.978324875621205300f, 1.978268532451738200f, 1.978212116546992100f,
	1.978155627909041300f,
	1.978099066539962900f, 1.978042432441836400f, 1.977985725616743900f,
	1.977928946066770600f,
	1.977872093794004200f, 1.977815168800534500f, 1.977758171088455100f,
	1.977701100659861300f,
	1.977643957516851400f, 1.977586741661526500f, 1.977529453095990200f,
	1.977472091822348700f,
	1.977414657842711200f, 1.977357151159189400f, 1.977299571773897700f,
	1.977241919688953000f,
	1.977184194906475000f, 1.977126397428586000f, 1.977068527257411300f,
	1.977010584395078300f,
	1.976952568843717700f, 1.976894480605462500f, 1.976836319682448300f,
	1.976778086076813600f,
	1.976719779790699500f, 1.976661400826249500f, 1.976602949185610500f,
	1.976544424870931400f,
	1.976485827884363800f, 1.976427158228062100f, 1.976368415904183900f,
	1.976309600914888400f,
	1.976250713262338600f, 1.976191752948699200f, 1.976132719976138000f,
	1.976073614346825800f,
	1.976014436062935700f, 1.975955185126643300f, 1.975895861540127200f,
	1.975836465305568400f,
	1.975776996425151000f, 1.975717454901061400f, 1.975657840735488800f,
	1.975598153930624900f,
	1.975538394488664200f, 1.975478562411804100f, 1.975418657702244300f,
	1.975358680362187400f,
	1.975298630393838500f, 1.975238507799405500f, 1.975178312581099100f,
	1.975118044741132300f,
	1.975057704281721000f, 1.974997291205083700f, 1.974936805513442000f,
	1.974876247209019100f,
	1.974815616294042200f, 1.974754912770740200f, 1.974694136641345300f,
	1.974633287908091500f,
	1.974572366573216400f, 1.974511372638960000f, 1.974450306107564900f,
	1.974389166981275900f,
	1.974327955262341400f, 1.974266670953011400f, 1.974205314055540000f,
	1.974143884572182400f,
	1.974082382505197400f, 1.974020807856846400f, 1.973959160629393100f,
	1.973897440825104200f,
	1.973835648446248900f, 1.973773783495099500f, 1.973711845973930000f,
	1.973649835885018100f,
	1.973587753230643400f, 1.973525598013088800f, 1.973463370234639600f,
	1.973401069897583200f,
	1.973338697004211100f, 1.973276251556815600f, 1.973213733557693400f,
	1.973151143009142800f,
	1.973088479913465100f, 1.973025744272964200f, 1.972962936089946800f,
	1.972900055366722000f,
	1.972837102105601900f, 1.972774076308901200f, 1.972710977978936900f,
	1.972647807118029300f,
	1.972584563728500700f, 1.972521247812676600f, 1.972457859372884500f,
	1.972394398411455800f,
	1.972330864930723200f, 1.972267258933022600f, 1.972203580420693000f,
	1.972139829396075200f,
	1.972076005861513700f, 1.972012109819354600f, 1.971948141271947500f,
	1.971884100221644300f,
	1.971819986670799500f, 1.971755800621770400f, 1.971691542076916800f,
	1.971627211038601500f,
	1.971562807509189800f, 1.971498331491049700f, 1.971433782986551400f,
	1.971369161998068400f,
	1.971304468527976800f, 1.971239702578655000f, 1.971174864152484400f,
	1.971109953251848600f,
	1.971044969879134600f, 1.970979914036731500f, 1.970914785727030800f,
	1.970849584952427900f,
	1.970784311715319400f, 1.970718966018105500f, 1.970653547863188600f,
	1.970588057252973900f,
	1.970522494189869800f, 1.970456858676286300f, 1.970391150714636800f,
	1.970325370307337100f,
	1.970259517456806100f, 1.970193592165464700f, 1.970127594435737000f,
	1.970061524270049400f,
	1.969995381670831100f, 1.969929166640514100f, 1.969862879181532700f,
	1.969796519296324300f,
	1.969730086987328900f, 1.969663582256988600f, 1.969597005107748900f,
	1.969530355542057800f,
	1.969463633562365400f, 1.969396839171125200f, 1.969329972370792700f,
	1.969263033163826800f,
	1.969196021552688500f, 1.969128937539841500f, 1.969061781127752400f,
	1.968994552318890300f,
	1.968927251115727200f, 1.968859877520737300f, 1.968792431536398000f,
	1.968724913165188900f,
	1.968657322409592500f, 1.968589659272094000f, 1.968521923755181000f,
	1.968454115861344000f,
	1.968386235593076300f, 1.968318282952873600f, 1.968250257943234200f,
	1.968182160566659000f,
	1.968113990825652200f, 1.968045748722719900f, 1.967977434260371300f,
	1.967909047441118100f,
	1.967840588267474500f, 1.967772056741957900f, 1.967703452867087800f,
	1.967634776645386600f,
	1.967566028079379200f, 1.967497207171593500f, 1.967428313924559600f,
	1.967359348340810700f,
	1.967290310422882700f, 1.967221200173313400f, 1.967152017594644200f,
	1.967082762689418500f,
	1.967013435460182700f, 1.966944035909485600f, 1.966874564039879300f,
	1.966805019853917500f,
	1.966735403354157500f, 1.966665714543159000f, 1.966595953423483800f,
	1.966526119997697100f,
	1.966456214268366600f, 1.966386236238062200f, 1.966316185909357200f,
	1.966246063284826700f,
	1.966175868367049400f, 1.966105601158605600f, 1.966035261662079300f,
	1.965964849880056600f,
	1.965894365815126000f, 1.965823809469879400f, 1.965753180846910900f,
	1.965682479948817100f,
	1.965611706778197700f, 1.965540861337654600f, 1.965469943629792700f,
	1.965398953657219600f,
	1.965327891422544900f, 1.965256756928382100f, 1.965185550177345900f,
	1.965114271172054800f,
	1.965042919915129400f, 1.964971496409193100f, 1.964900000656872000f,
	1.964828432660794500f,
	1.964756792423592200f, 1.964685079947899200f, 1.964613295236352000f,
	1.964541438291590000f,
	1.964469509116255000f, 1.964397507712991800f, 1.964325434084447600f,
	1.964253288233272400f,
	1.964181070162119000f, 1.964108779873642100f, 1.964036417370500300f,
	1.963963982655353400f,
	1.963891475730865400f, 1.963818896599701400f, 1.963746245264530700f,
	1.963673521728023900f,
	1.963600725992855200f, 1.963527858061700600f, 1.963454917937239800f,
	1.963381905622154400f,
	1.963308821119128700f, 1.963235664430850200f, 1.963162435560008100f,
	1.963089134509295300f,
	1.963015761281406800f, 1.962942315879040000f, 1.962868798304895400f,
	1.962795208561676200f,
	1.962721546652088200f, 1.962647812578839400f, 1.962574006344640900f,
	1.962500127952206300f,
	1.962426177404252200f, 1.962352154703497200f, 1.962278059852663000f,
	1.962203892854473800f,
	1.962129653711656800f, 1.962055342426941400f, 1.961980959003059500f,
	1.961906503442746300f,
	1.961831975748739200f, 1.961757375923778700f, 1.961682703970607100f,
	1.961607959891970200f,
	1.961533143690616000f, 1.961458255369295400f, 1.961383294930761700f,
	1.961308262377770900f,
	1.961233157713082200f, 1.961157980939456400f, 1.961082732059657800f,
	1.961007411076453000f,
	1.960932017992611500f, 1.960856552810905200f, 1.960781015534108800f,
	1.960705406164999300f,
	1.960629724706357100f, 1.960553971160964500f, 1.960478145531606700f,
	1.960402247821071900f,
	1.960326278032150200f, 1.960250236167635100f, 1.960174122230322400f,
	1.960097936223010400f,
	1.960021678148500500f, 1.959945348009596500f, 1.959868945809104500f,
	1.959792471549834000f,
	1.959715925234596600f, 1.959639306866206600f, 1.959562616447480900f,
	1.959485853981239600f,
	1.959409019470304700f, 1.959332112917501400f, 1.959255134325657000f,
	1.959178083697602300f,
	1.959100961036169800f, 1.959023766344195200f, 1.958946499624516700f,
	1.958869160879975500f,
	1.958791750113414700f, 1.958714267327680500f, 1.958636712525621900f,
	1.958559085710090500f,
	1.958481386883940100f, 1.958403616050027600f, 1.958325773211212300f,
	1.958247858370356400f,
	1.958169871530324600f, 1.958091812693984400f, 1.958013681864205500f,
	1.957935479043860600f,
	1.957857204235825100f, 1.957778857442976900f, 1.957700438668196700f,
	1.957621947914367500f,
	1.957543385184375300f, 1.957464750481108700f, 1.957386043807458800f,
	1.957307265166319500f,
	1.957228414560587200f, 1.957149491993160900f, 1.957070497466942400f,
	1.956991430984836400f,
	1.956912292549749500f, 1.956833082164591600f, 1.956753799832275300f,
	1.956674445555715000f,
	1.956595019337829000f, 1.956515521181537000f, 1.956435951089762200f,
	1.956356309065430100f,
	1.956276595111468900f, 1.956196809230809500f, 1.956116951426385600f,
	1.956037021701132900f,
	1.955957020057990500f, 1.955876946499899700f, 1.955796801029804800f,
	1.955716583650652000f,
	1.955636294365391300f, 1.955555933176974300f, 1.955475500088355900f,
	1.955394995102493100f,
	1.955314418222346100f, 1.955233769450877200f, 1.955153048791052000f,
	1.955072256245838000f,
	1.954991391818206000f, 1.954910455511129000f, 1.954829447327582900f,
	1.954748367270545900f,
	1.954667215342999600f, 1.954585991547927100f, 1.954504695888315000f,
	1.954423328367152600f,
	1.954341888987431100f, 1.954260377752145000f, 1.954178794664291200f,
	1.954097139726869600f,
	1.954015412942881900f, 1.953933614315333200f, 1.953851743847231100f,
	1.953769801541585400f,
	1.953687787401409400f, 1.953605701429718100f, 1.953523543629529700f,
	1.953441314003864900f,
	1.953359012555747200f, 1.953276639288202400f, 1.953194194204259200f,
	1.953111677306948800f,
	1.953029088599305100f, 1.952946428084364900f, 1.952863695765167100f,
	1.952780891644753500f,
	1.952698015726169100f, 1.952615068012460300f, 1.952532048506677300f,
	1.952448957211872200f,
	1.952365794131100300f, 1.952282559267419100f, 1.952199252623889200f,
	1.952115874203572900f,
	1.952032424009536600f, 1.951948902044847900f, 1.951865308312577900f,
	1.951781642815800100f,
	1.951697905557590700f, 1.951614096541028500f, 1.951530215769194700f,
	1.951446263245173500f,
	1.951362238972051500f, 1.951278142952918200f, 1.951193975190865600f,
	1.951109735688987900f,
	1.951025424450382900f, 1.950941041478150100f, 1.950856586775392200f,
	1.950772060345214300f,
	1.950687462190724200f, 1.950602792315032200f, 1.950518050721251600f,
	1.950433237412498000f,
	1.950348352391889600f, 1.950263395662547700f, 1.950178367227595900f,
	1.950093267090159800f,
	1.950008095253369200f, 1.949922851720355100f, 1.949837536494251700f,
	1.949752149578196000f,
	1.949666690975327100f, 1.949581160688787400f, 1.949495558721721500f,
	1.949409885077276500f,
	1.949324139758602700f, 1.949238322768852800f, 1.949152434111181700f,
	1.949066473788747300f,
	1.948980441804710300f, 1.948894338162233900f, 1.948808162864483600f,
	1.948721915914628100f,
	1.948635597315838200f, 1.948549207071288000f, 1.948462745184153400f,
	1.948376211657613500f,
	1.948289606494849800f, 1.948202929699046800f, 1.948116181273391100f,
	1.948029361221072400f,
	1.947942469545282500f, 1.947855506249216700f, 1.947768471336071700f,
	1.947681364809048100f,
	1.947594186671348000f, 1.947506936926177300f, 1.947419615576743600f,
	1.947332222626257500f,
	1.947244758077932200f, 1.947157221934983500f, 1.947069614200629900f,
	1.946981934878092300f,
	1.946894183970594900f, 1.946806361481363500f, 1.946718467413627300f,
	1.946630501770618000f,
	1.946542464555569800f, 1.946454355771719300f, 1.946366175422306500f,
	1.946277923510573200f,
	1.946189600039764300f, 1.946101205013127000f, 1.946012738433911600f,
	1.945924200305370700f,
	1.945835590630759400f, 1.945746909413335900f, 1.945658156656360700f,
	1.945569332363096700f,
	1.945480436536810100f, 1.945391469180769200f, 1.945302430298244900f,
	1.945213319892511200f,
	1.945124137966844200f, 1.945034884524523100f, 1.944945559568829200f,
	1.944856163103046800f,
	1.944766695130463000f, 1.944677155654366900f, 1.944587544678050900f,
	1.944497862204809900f,
	1.944408108237940700f, 1.944318282780743900f, 1.944228385836521700f,
	1.944138417408579400f,
	1.944048377500225100f, 1.943958266114769200f, 1.943868083255524800f,
	1.943777828925807600f,
	1.943687503128936200f, 1.943597105868231500f, 1.943506637147017300f,
	1.943416096968619400f,
	1.943325485336367300f, 1.943234802253592400f, 1.943144047723628400f,
	1.943053221749812400f,
	1.942962324335484100f, 1.942871355483985200f, 1.942780315198660200f,
	1.942689203482856900f,
	1.942598020339924700f, 1.942506765773216500f, 1.942415439786087300f,
	1.942324042381895000f,
	1.942232573564000000f, 1.942141033335765400f, 1.942049421700556600f,
	1.941957738661741900f,
	1.941865984222692900f, 1.941774158386782200f, 1.941682261157386700f,
	1.941590292537884700f,
	1.941498252531658200f, 1.941406141142090600f, 1.941313958372568900f,
	1.941221704226482500f,
	1.941129378707223000f, 1.941036981818185400f, 1.940944513562766300f,
	1.940851973944365900f,
	1.940759362966386600f, 1.940666680632233200f, 1.940573926945313700f,
	1.940481101909038200f,
	1.940388205526819600f, 1.940295237802073500f, 1.940202198738217900f,
	1.940109088338673600f,
	1.940015906606864300f, 1.939922653546215500f, 1.939829329160156500f,
	1.939735933452118000f,
	1.939642466425534300f, 1.939548928083841800f, 1.939455318430479500f,
	1.939361637468889100f,
	1.939267885202515400f, 1.939174061634805000f, 1.939080166769207700f,
	1.938986200609175600f,
	1.938892163158163700f, 1.938798054419629500f, 1.938703874397032800f,
	1.938609623093837000f,
	1.938515300513506700f, 1.938420906659510600f, 1.938326441535318500f,
	1.938231905144404400f,
	1.938137297490243500f, 1.938042618576314400f, 1.937947868406098500f,
	1.937853046983079300f,
	1.937758154310742900f, 1.937663190392578500f, 1.937568155232077600f,
	1.937473048832734500f,
	1.937377871198045600f, 1.937282622331510500f, 1.937187302236631500f,
	1.937091910916912900f,
	1.936996448375861900f, 1.936900914616988900f, 1.936805309643805800f,
	1.936709633459828200f,
	1.936613886068573500f, 1.936518067473562300f, 1.936422177678317300f,
	1.936326216686364400f,
	1.936230184501231500f, 1.936134081126449800f, 1.936037906565552400f,
	1.935941660822075600f,
	1.935845343899558000f, 1.935748955801540800f, 1.935652496531568000f,
	1.935555966093186300f,
	1.935459364489944500f, 1.935362691725394500f, 1.935265947803090900f,
	1.935169132726590500f,
	1.935072246499453000f, 1.934975289125240500f, 1.934878260607517900f,
	1.934781160949852600f,
	1.934683990155814800f, 1.934586748228977100f, 1.934489435172914900f,
	1.934392050991206300f,
	1.934294595687431300f, 1.934197069265173500f, 1.934099471728018700f,
	1.934001803079554700f,
	1.933904063323373300f, 1.933806252463067500f, 1.933708370502233800f,
	1.933610417444471000f,
	1.933512393293380600f, 1.933414298052566600f, 1.933316131725635800f,
	1.933217894316197300f,
	1.933119585827862900f, 1.933021206264247600f, 1.932922755628968100f,
	1.932824233925644300f,
	1.932725641157898600f, 1.932626977329356100f, 1.932528242443643900f,
	1.932429436504392800f,
	1.932330559515235100f, 1.932231611479806800f, 1.932132592401745400f,
	1.932033502284691700f,
	1.931934341132289100f, 1.931835108948183300f, 1.931735805736022800f,
	1.931636431499459000f,
	1.931536986242145200f, 1.931437469967737900f, 1.931337882679895900f,
	1.931238224382281000f,
	1.931138495078557300f, 1.931038694772391200f, 1.930938823467452500f,
	1.930838881167413100f,
	1.930738867875947400f, 1.930638783596732700f, 1.930538628333448900f,
	1.930438402089778200f,
	1.930338104869405900f, 1.930237736676019500f, 1.930137297513309300f,
	1.930036787384968200f,
	1.929936206294691400f, 1.929835554246177400f, 1.929734831243126600f,
	1.929634037289242400f,
	1.929533172388230700f, 1.929432236543799900f, 1.929331229759661200f,
	1.929230152039528500f,
	1.929129003387117800f, 1.929027783806148300f, 1.928926493300341400f,
	1.928825131873421500f,
	1.928723699529115000f, 1.928622196271151800f, 1.928520622103263400f,
	1.928418977029184600f,
	1.928317261052652700f, 1.928215474177407100f, 1.928113616407190600f,
	1.928011687745748300f,
	1.927909688196827400f, 1.927807617764178300f, 1.927705476451554000f,
	1.927603264262709900f,
	1.927500981201404100f, 1.927398627271397000f, 1.927296202476451900f,
	1.927193706820335100f,
	1.927091140306814500f, 1.926988502939661400f, 1.926885794722649600f,
	1.926783015659555300f,
	1.926680165754157500f, 1.926577245010237400f, 1.926474253431579500f,
	1.926371191021970100f,
	1.926268057785198700f, 1.926164853725057300f, 1.926061578845340600f,
	1.925958233149845000f,
	1.925854816642371000f, 1.925751329326720600f, 1.925647771206698600f,
	1.925544142286112800f,
	1.925440442568773000f, 1.925336672058492300f, 1.925232830759086000f,
	1.925128918674371900f,
	1.925024935808170600f, 1.924920882164305300f, 1.924816757746601800f,
	1.924712562558888100f,
	1.924608296604995800f, 1.924503959888757900f, 1.924399552414010700f,
	1.924295074184593000f,
	1.924190525204346300f, 1.924085905477114400f, 1.923981215006744100f,
	1.923876453797084300f,
	1.923771621851986700f, 1.923666719175306100f, 1.923561745770898900f,
	1.923456701642625200f,
	1.923351586794346900f, 1.923246401229928600f, 1.923141144953238300f,
	1.923035817968145300f,
	1.922930420278522500f, 1.922824951888245000f, 1.922719412801190600f,
	1.922613803021239600f,
	1.922508122552275100f, 1.922402371398182600f, 1.922296549562850100f,
	1.922190657050168800f,
	1.922084693864031700f, 1.921978660008334600f, 1.921872555486976700f,
	1.921766380303858500f,
	1.921660134462884100f, 1.921553817967959900f, 1.921447430822994500f,
	1.921340973031900000f,
	1.921234444598590100f, 1.921127845526981600f, 1.921021175820994100f,
	1.920914435484549100f,
	1.920807624521571700f, 1.920700742935988600f, 1.920593790731729600f,
	1.920486767912727300f,
	1.920379674482916500f, 1.920272510446234400f, 1.920165275806621400f,
	1.920057970568020100f,
	1.919950594734376000f, 1.919843148309637000f, 1.919735631297753400f,
	1.919628043702678300f,
	1.919520385528367300f, 1.919412656778779000f, 1.919304857457874200f,
	1.919196987569616200f,
	1.919089047117971100f, 1.918981036106907700f, 1.918872954540397300f,
	1.918764802422413500f,
	1.918656579756932800f, 1.918548286547934400f, 1.918439922799399800f,
	1.918331488515313300f,
	1.918222983699661600f, 1.918114408356434300f, 1.918005762489623400f,
	1.917897046103223200f,
	1.917788259201231200f, 1.917679401787647100f, 1.917570473866473200f,
	1.917461475441714500f,
	1.917352406517378600f, 1.917243267097475700f, 1.917134057186018300f,
	1.917024776787022100f,
	1.916915425904504700f, 1.916806004542486800f, 1.916696512704991500f,
	1.916586950396044400f,
	1.916477317619674100f, 1.916367614379911100f, 1.916257840680788900f,
	1.916147996526343700f,
	1.916038081920614400f, 1.915928096867641800f, 1.915818041371470000f,
	1.915707915436145200f,
	1.915597719065716700f, 1.915487452264236000f, 1.915377115035757200f,
	1.915266707384337200f,
	1.915156229314035200f, 1.915045680828913400f, 1.914935061933036300f,
	1.914824372630470800f,
	1.914713612925287100f, 1.914602782821557000f, 1.914491882323355700f,
	1.914380911434760500f,
	1.914269870159851700f, 1.914158758502712000f, 1.914047576467426500f,
	1.913936324058083100f,
	1.913825001278772100f, 1.913713608133586600f, 1.913602144626622500f,
	1.913490610761977600f,
	1.913379006543752800f, 1.913267331976051400f, 1.913155587062979500f,
	1.913043771808645700f,
	1.912931886217160900f, 1.912819930292639000f, 1.912707904039196300f,
	1.912595807460951500f,
	1.912483640562026200f, 1.912371403346544400f, 1.912259095818632700f,
	1.912146717982420500f,
	1.912034269842039600f, 1.911921751401624200f, 1.911809162665311500f,
	1.911696503637241100f,
	1.911583774321554700f, 1.911470974722397500f, 1.911358104843916500f,
	1.911245164690262000f,
	1.911132154265586100f, 1.911019073574044200f, 1.910905922619793800f,
	1.910792701406995000f,
	1.910679409939810600f, 1.910566048222406300f, 1.910452616258949900f,
	1.910339114053611900f,
	1.910225541610565800f, 1.910111898933986900f, 1.909998186028053700f,
	1.909884402896947100f,
	1.909770549544850500f, 1.909656625975950200f, 1.909542632194434700f,
	1.909428568204495100f,
	1.909314434010325400f, 1.909200229616121700f, 1.909085955026083200f,
	1.908971610244411600f,
	1.908857195275310800f, 1.908742710122987700f, 1.908628154791651300f,
	1.908513529285513500f,
	1.908398833608789100f, 1.908284067765694900f, 1.908169231760450400f,
	1.908054325597278200f,
	1.907939349280402400f, 1.907824302814050900f, 1.907709186202453600f,
	1.907593999449842800f,
	1.907478742560453600f, 1.907363415538523700f, 1.907248018388293400f,
	1.907132551114005600f,
	1.907017013719905600f, 1.906901406210241200f, 1.906785728589263300f,
	1.906669980861224900f,
	1.906554163030381500f, 1.906438275100991600f, 1.906322317077316300f,
	1.906206288963618700f,
	1.906090190764164700f, 1.905974022483223300f, 1.905857784125065500f,
	1.905741475693964800f,
	1.905625097194197900f, 1.905508648630043700f, 1.905392130005783400f,
	1.905275541325701400f,
	1.905158882594083900f, 1.905042153815220700f, 1.904925354993402900f,
	1.904808486132925300f,
	1.904691547238084800f, 1.904574538313180700f, 1.904457459362515200f,
	1.904340310390393100f,
	1.904223091401121600f, 1.904105802399010300f, 1.903988443388371600f,
	1.903871014373520700f,
	1.903753515358774800f, 1.903635946348454500f, 1.903518307346881800f,
	1.903400598358382600f,
	1.903282819387284200f, 1.903164970437917400f, 1.903047051514615000f,
	1.902929062621712600f,
	1.902811003763547900f, 1.902692874944462300f, 1.902574676168798700f,
	1.902456407440902700f,
	1.902338068765123200f, 1.902219660145810800f, 1.902101181587319000f,
	1.901982633094004200f,
	1.901864014670225000f, 1.901745326320342500f, 1.901626568048721000f,
	1.901507739859726200f,
	1.901388841757727600f, 1.901269873747096600f, 1.901150835832207100f,
	1.901031728017436300f,
	1.900912550307162700f, 1.900793302705768900f, 1.900673985217638900f,
	1.900554597847159400f,
	1.900435140598720500f, 1.900315613476714100f, 1.900196016485534700f,
	1.900076349629579600f,
	1.899956612913248800f, 1.899836806340944300f, 1.899716929917071500f,
	1.899596983646037600f,
	1.899476967532252900f, 1.899356881580129800f, 1.899236725794083600f,
	1.899116500178532200f,
	1.898996204737895900f, 1.898875839476597700f, 1.898755404399062900f,
	1.898634899509719500f,
	1.898514324812998300f, 1.898393680313332600f, 1.898272966015157800f,
	1.898152181922912600f,
	1.898031328041037700f, 1.897910404373976500f, 1.897789410926175000f,
	1.897668347702081900f,
	1.897547214706148300f, 1.897426011942827900f, 1.897304739416577200f,
	1.897183397131854600f,
	1.897061985093121800f, 1.896940503304842800f, 1.896818951771484000f,
	1.896697330497514800f,
	1.896575639487406300f, 1.896453878745633100f, 1.896332048276672100f,
	1.896210148085002400f,
	1.896088178175106200f, 1.895966138551467700f, 1.895844029218574100f,
	1.895721850180915000f,
	1.895599601442982600f, 1.895477283009271400f, 1.895354894884279100f,
	1.895232437072505300f,
	1.895109909578452500f, 1.894987312406625700f, 1.894864645561532100f,
	1.894741909047682500f,
	1.894619102869589100f, 1.894496227031767100f, 1.894373281538734400f,
	1.894250266395011600f,
	1.894127181605121100f, 1.894004027173588700f, 1.893880803104942600f,
	1.893757509403713100f,
	1.893634146074433500f, 1.893510713121639300f, 1.893387210549869000f,
	1.893263638363663400f,
	1.893139996567565900f, 1.893016285166122500f, 1.892892504163881600f,
	1.892768653565394300f,
	1.892644733375214300f, 1.892520743597897700f, 1.892396684238003300f,
	1.892272555300092300f,
	1.892148356788728700f, 1.892024088708479200f, 1.891899751063912200f,
	1.891775343859599400f,
	1.891650867100115300f, 1.891526320790036100f, 1.891401704933941100f,
	1.891277019536412400f,
	1.891152264602033800f, 1.891027440135392600f, 1.890902546141078000f,
	1.890777582623682300f,
	1.890652549587799700f, 1.890527447038027300f, 1.890402274978965100f,
	1.890277033415215200f,
	1.890151722351382200f, 1.890026341792073500f, 1.889900891741899100f,
	1.889775372205471300f,
	1.889649783187405100f, 1.889524124692318200f, 1.889398396724830500f,
	1.889272599289564900f,
	1.889146732391146400f, 1.889020796034202700f, 1.888894790223364600f,
	1.888768714963264400f,
	1.888642570258537700f, 1.888516356113822700f, 1.888390072533759700f,
	1.888263719522991900f,
	1.888137297086165000f, 1.888010805227927000f, 1.887884243952928600f,
	1.887757613265823400f,
	1.887630913171267000f, 1.887504143673917700f, 1.887377304778437000f,
	1.887250396489487800f,
	1.887123418811736500f, 1.886996371749851700f, 1.886869255308504200f,
	1.886742069492368000f,
	1.886614814306119400f, 1.886487489754437300f, 1.886360095842002600f,
	1.886232632573499700f,
	1.886105099953614900f, 1.885977497987037000f, 1.885849826678457800f,
	1.885722086032571200f,
	1.885594276054074300f, 1.885466396747665700f, 1.885338448118047700f,
	1.885210430169924200f,
	1.885082342908002400f, 1.884954186336991400f, 1.884825960461603100f,
	1.884697665286552400f,
	1.884569300816556000f, 1.884440867056333700f, 1.884312364010607600f,
	1.884183791684102400f,
	1.884055150081545200f, 1.883926439207665800f, 1.883797659067196800f,
	1.883668809664872600f,
	1.883539891005431100f, 1.883410903093611900f, 1.883281845934157800f,
	1.883152719531813800f,
	1.883023523891327300f, 1.882894259017448900f, 1.882764924914930700f,
	1.882635521588528400f,
	1.882506049042999700f, 1.882376507283104900f, 1.882246896313606800f,
	1.882117216139270700f,
	1.881987466764865100f, 1.881857648195159900f, 1.881727760434928500f,
	1.881597803488946500f,
	1.881467777361992100f, 1.881337682058845700f, 1.881207517584290600f,
	1.881077283943112900f,
	1.880946981140100500f, 1.880816609180044700f, 1.880686168067738500f,
	1.880555657807977800f,
	1.880425078405561600f, 1.880294429865290600f, 1.880163712191968300f,
	1.880032925390400900f,
	1.879902069465397200f, 1.879771144421768200f, 1.879640150264327600f,
	1.879509086997891900f,
	1.879377954627279700f, 1.879246753157312700f, 1.879115482592814500f,
	1.878984142938611600f,
	1.878852734199532900f, 1.878721256380410100f, 1.878589709486077300f,
	1.878458093521370800f,
	1.878326408491130200f, 1.878194654400196600f, 1.878062831253414900f,
	1.877930939055631100f,
	1.877798977811695200f, 1.877666947526458700f, 1.877534848204775800f,
	1.877402679851504000f,
	1.877270442471502100f, 1.877138136069632400f, 1.877005760650759500f,
	1.876873316219750200f,
	1.876740802781474500f, 1.876608220340804100f, 1.876475568902614000f,
	1.876342848471781200f,
	1.876210059053185600f, 1.876077200651709500f, 1.875944273272237800f,
	1.875811276919657500f,
	1.875678211598858800f, 1.875545077314734000f, 1.875411874072178100f,
	1.875278601876088700f,
	1.875145260731365700f, 1.875011850642911600f, 1.874878371615631900f,
	1.874744823654434000f,
	1.874611206764227800f, 1.874477520949926500f, 1.874343766216444800f,
	1.874209942568701100f,
	1.874076050011615400f, 1.873942088550110400f, 1.873808058189111700f,
	1.873673958933546900f,
	1.873539790788347100f, 1.873405553758444600f, 1.873271247848775400f,
	1.873136873064277000f,
	1.873002429409890600f, 1.872867916890558900f, 1.872733335511227700f,
	1.872598685276845000f,
	1.872463966192361900f, 1.872329178262731200f, 1.872194321492908700f,
	1.872059395887852900f,
	1.871924401452524700f, 1.871789338191887100f, 1.871654206110906500f,
	1.871519005214550700f,
	1.871383735507791100f, 1.871248396995601300f, 1.871112989682956800f,
	1.870977513574836500f,
	1.870841968676221400f, 1.870706354992095000f, 1.870570672527443600f,
	1.870434921287255700f,
	1.870299101276522400f, 1.870163212500237900f, 1.870027254963397800f,
	1.869891228671001200f,
	1.869755133628049600f, 1.869618969839546500f, 1.869482737310498100f,
	1.869346436045913800f,
	1.869210066050804600f, 1.869073627330184700f, 1.868937119889070300f,
	1.868800543732480600f,
	1.868663898865437200f, 1.868527185292963700f, 1.868390403020087100f,
	1.868253552051836200f,
	1.868116632393243000f, 1.867979644049341200f, 1.867842587025167800f,
	1.867705461325761800f,
	1.867568266956164800f, 1.867431003921421500f, 1.867293672226578300f,
	1.867156271876684500f,
	1.867018802876792200f, 1.866881265231955500f, 1.866743658947231300f,
	1.866605984027679000f,
	1.866468240478360600f, 1.866330428304340300f, 1.866192547510685300f,
	1.866054598102465000f,
	1.865916580084751500f, 1.865778493462619100f, 1.865640338241145100f,
	1.865502114425408900f,
	1.865363822020492700f, 1.865225461031480900f, 1.865087031463460900f,
	1.864948533321522300f,
	1.864809966610757400f, 1.864671331336260600f, 1.864532627503129100f,
	1.864393855116463200f,
	1.864255014181364500f, 1.864116104702938000f, 1.863977126686291200f,
	1.863838080136534000f,
	1.863698965058778300f, 1.863559781458139300f, 1.863420529339734100f,
	1.863281208708683000f,
	1.863141819570107900f, 1.863002361929134500f, 1.862862835790889400f,
	1.862723241160503300f,
	1.862583578043108100f, 1.862443846443839300f, 1.862304046367834200f,
	1.862164177820232700f,
	1.862024240806177800f, 1.861884235330814300f, 1.861744161399289600f,
	1.861604019016754200f,
	1.861463808188360500f, 1.861323528919263800f, 1.861183181214621600f,
	1.861042765079594200f,
	1.860902280519344500f, 1.860761727539037300f, 1.860621106143840500f,
	1.860480416338924600f,
	1.860339658129461800f, 1.860198831520627900f, 1.860057936517600700f,
	1.859916973125560000f,
	1.859775941349689000f, 1.859634841195173100f, 1.859493672667199800f,
	1.859352435770959900f,
	1.859211130511645900f, 1.859069756894453400f, 1.858928314924580300f,
	1.858786804607227100f,
	1.858645225947596300f, 1.858503578950893900f, 1.858361863622327400f,
	1.858220079967107600f,
	1.858078227990447300f, 1.857936307697561900f, 1.857794319093669900f,
	1.857652262183991000f,
	1.857510136973749000f, 1.857367943468169100f, 1.857225681672479300f,
	1.857083351591910300f,
	1.856940953231694900f, 1.856798486597069000f, 1.856655951693270600f,
	1.856513348525540300f,
	1.856370677099121100f, 1.856227937419258700f, 1.856085129491201100f,
	1.855942253320199200f,
	1.855799308911506100f, 1.855656296270377300f, 1.855513215402071000f,
	1.855370066311848000f,
	1.855226849004971500f, 1.855083563486706900f, 1.854940209762322700f,
	1.854796787837089500f,
	1.854653297716280400f, 1.854509739405171300f, 1.854366112909040300f,
	1.854222418233168400f,
	1.854078655382838300f, 1.853934824363336200f, 1.853790925179950500f,
	1.853646957837971500f,
	1.853502922342692600f, 1.853358818699409900f, 1.853214646913421200f,
	1.853070406990027500f,
	1.852926098934532200f, 1.852781722752241000f, 1.852637278448462200f,
	1.852492766028506400f,
	1.852348185497687300f, 1.852203536861320600f, 1.852058820124724300f,
	1.851914035293219700f,
	1.851769182372129600f, 1.851624261366780400f, 1.851479272282500000f,
	1.851334215124619300f,
	1.851189089898471800f, 1.851043896609393400f, 1.850898635262721900f,
	1.850753305863798800f,
	1.850607908417967200f, 1.850462442930572900f, 1.850316909406964200f,
	1.850171307852492200f,
	1.850025638272510000f, 1.849879900672373600f, 1.849734095057441200f,
	1.849588221433073700f,
	1.849442279804634600f, 1.849296270177489800f, 1.849150192557007300f,
	1.849004046948558200f,
	1.848857833357515900f, 1.848711551789256300f, 1.848565202249157400f,
	1.848418784742600400f,
	1.848272299274968500f, 1.848125745851647800f, 1.847979124478026100f,
	1.847832435159495000f,
	1.847685677901447200f, 1.847538852709279100f, 1.847391959588388300f,
	1.847244998544176300f,
	1.847097969582046200f, 1.846950872707404000f, 1.846803707925657600f,
	1.846656475242218300f,
	1.846509174662499300f, 1.846361806191916000f, 1.846214369835887500f,
	1.846066865599834000f,
	1.845919293489179000f, 1.845771653509348200f, 1.845623945665770100f,
	1.845476169963875500f,
	1.845328326409097400f, 1.845180415006871800f, 1.845032435762637100f,
	1.844884388681833800f,
	1.844736273769905300f, 1.844588091032297400f, 1.844439840474458200f,
	1.844291522101838800f,
	1.844143135919891900f, 1.843994681934073600f, 1.843846160149842200f,
	1.843697570572658200f,
	1.843548913207985000f, 1.843400188061288000f, 1.843251395138035800f,
	1.843102534443698900f,
	1.842953605983750400f, 1.842804609763666100f, 1.842655545788924000f,
	1.842506414065004900f,
	1.842357214597392100f, 1.842207947391570900f, 1.842058612453029600f,
	1.841909209787258900f,
	1.841759739399751800f, 1.841610201296003800f, 1.841460595481513100f,
	1.841310921961780500f,
	1.841161180742308500f, 1.841011371828603200f, 1.840861495226172600f,
	1.840711550940526700f,
	1.840561538977179200f, 1.840411459341645400f, 1.840261312039443100f,
	1.840111097076092800f,
	1.839960814457117600f, 1.839810464188043100f, 1.839660046274397100f,
	1.839509560721709800f,
	1.839359007535514400f, 1.839208386721346500f, 1.839057698284743500f,
	1.838906942231246100f,
	1.838756118566397200f, 1.838605227295741800f, 1.838454268424828400f,
	1.838303241959206700f,
	1.838152147904429800f, 1.838000986266052900f, 1.837849757049633900f,
	1.837698460260732900f,
	1.837547095904912700f, 1.837395663987738700f, 1.837244164514778600f,
	1.837092597491602100f,
	1.836940962923782700f, 1.836789260816895000f, 1.836637491176516600f,
	1.836485654008228200f,
	1.836333749317611700f, 1.836181777110252900f, 1.836029737391738700f,
	1.835877630167659800f,
	1.835725455443608200f, 1.835573213225179400f, 1.835420903517970500f,
	1.835268526327581900f,
	1.835116081659615700f, 1.834963569519677100f, 1.834810989913373500f,
	1.834658342846314800f,
	1.834505628324113200f, 1.834352846352383700f, 1.834199996936744000f,
	1.834047080082813300f,
	1.833894095796214400f, 1.833741044082571900f, 1.833587924947513100f,
	1.833434738396668000f,
	1.833281484435668400f, 1.833128163070149300f, 1.832974774305747600f,
	1.832821318148103500f,
	1.832667794602858400f, 1.832514203675657600f, 1.832360545372147900f,
	1.832206819697979000f,
	1.832053026658802700f, 1.831899166260273700f, 1.831745238508049300f,
	1.831591243407788300f,
	1.831437180965153100f, 1.831283051185808300f, 1.831128854075420500f,
	1.830974589639659000f,
	1.830820257884196100f, 1.830665858814705600f, 1.830511392436864800f,
	1.830356858756352800f,
	1.830202257778851300f, 1.830047589510044500f, 1.829892853955619200f,
	1.829738051121264600f,
	1.829583181012672400f, 1.829428243635536500f, 1.829273238995553700f,
	1.829118167098423100f,
	1.828963027949846100f, 1.828807821555527000f, 1.828652547921171900f,
	1.828497207052490100f,
	1.828341798955192900f, 1.828186323634994200f, 1.828030781097610400f,
	1.827875171348760400f,
	1.827719494394165500f, 1.827563750239549400f, 1.827407938890638600f,
	1.827252060353161500f,
	1.827096114632849700f, 1.826940101735436500f, 1.826784021666658400f,
	1.826627874432253700f,
	1.826471660037963800f, 1.826315378489531800f, 1.826159029792704400f,
	1.826002613953229500f,
	1.825846130976858100f, 1.825689580869344100f, 1.825532963636443000f,
	1.825376279283913200f,
	1.825219527817515800f, 1.825062709243013800f, 1.824905823566173000f,
	1.824748870792761900f,
	1.824591850928550800f, 1.824434763979313300f, 1.824277609950824700f,
	1.824120388848863300f,
	1.823963100679209600f, 1.823805745447646600f, 1.823648323159960100f,
	1.823490833821937600f,
	1.823333277439369600f, 1.823175654018049300f, 1.823017963563772000f,
	1.822860206082335300f,
	1.822702381579539800f, 1.822544490061187800f, 1.822386531533084900f,
	1.822228506001038800f,
	1.822070413470859600f, 1.821912253948359700f, 1.821754027439354400f,
	1.821595733949661100f,
	1.821437373485099900f, 1.821278946051493100f, 1.821120451654665700f,
	1.820961890300445400f,
	1.820803261994661500f, 1.820644566743146800f, 1.820485804551735800f,
	1.820326975426265600f,
	1.820168079372576300f, 1.820009116396509800f, 1.819850086503910700f,
	1.819690989700625900f,
	1.819531825992505500f, 1.819372595385401000f, 1.819213297885166900f,
	1.819053933497660300f,
	1.818894502228740600f, 1.818735004084269600f, 1.818575439070111200f,
	1.818415807192132600f,
	1.818256108456203000f, 1.818096342868193800f, 1.817936510433979300f,
	1.817776611159436000f,
	1.817616645050443000f, 1.817456612112881900f, 1.817296512352636300f,
	1.817136345775592900f,
	1.816976112387640700f, 1.816815812194670700f, 1.816655445202576700f,
	1.816495011417255300f,
	1.816334510844604700f, 1.816173943490526400f, 1.816013309360923900f,
	1.815852608461703300f,
	1.815691840798773000f, 1.815531006378043900f, 1.815370105205429600f,
	1.815209137286846200f,
	1.815048102628211500f, 1.814887001235446600f, 1.814725833114474700f,
	1.814564598271221300f,
	1.814403296711615000f, 1.814241928441585800f, 1.814080493467067300f,
	1.813918991793994900f,
	1.813757423428306000f, 1.813595788375941700f, 1.813434086642844400f,
	1.813272318234959700f,
	1.813110483158235400f, 1.812948581418621500f, 1.812786613022070700f,
	1.812624577974538000f,
	1.812462476281981200f, 1.812300307950360300f, 1.812138072985637800f,
	1.811975771393778300f,
	1.811813403180749300f, 1.811650968352521000f, 1.811488466915065000f,
	1.811325898874356800f,
	1.811163264236372900f, 1.811000563007093100f, 1.810837795192499400f,
	1.810674960798576600f,
	1.810512059831311400f, 1.810349092296693400f, 1.810186058200714100f,
	1.810022957549368000f,
	1.809859790348652200f, 1.809696556604565300f, 1.809533256323109200f,
	1.809369889510288100f,
	1.809206456172108200f, 1.809042956314578900f, 1.808879389943711200f,
	1.808715757065519200f,
	1.808552057686019200f, 1.808388291811230000f, 1.808224459447172800f,
	1.808060560599871200f,
	1.807896595275351200f, 1.807732563479641300f, 1.807568465218772900f,
	1.807404300498778800f,
	1.807240069325695400f, 1.807075771705560800f, 1.806911407644415700f,
	1.806746977148303300f,
	1.806582480223269500f, 1.806417916875362000f, 1.806253287110631600f,
	1.806088590935131000f,
	1.805923828354915900f, 1.805758999376044100f, 1.805594104004575800f,
	1.805429142246573600f,
	1.805264114108102900f, 1.805099019595231200f, 1.804933858714028700f,
	1.804768631470567500f,
	1.804603337870923000f, 1.804437977921172300f, 1.804272551627395400f,
	1.804107058995674500f,
	1.803941500032094200f, 1.803775874742741500f, 1.803610183133706400f,
	1.803444425211080400f,
	1.803278600980958300f, 1.803112710449436900f, 1.802946753622615400f,
	1.802780730506595700f,
	1.802614641107481900f, 1.802448485431380900f, 1.802282263484401300f,
	1.802115975272655000f,
	1.801949620802255600f, 1.801783200079319900f, 1.801616713109966300f,
	1.801450159900316300f,
	1.801283540456493700f, 1.801116854784624400f, 1.800950102890836800f,
	1.800783284781262200f,
	1.800616400462033800f, 1.800449449939287800f, 1.800282433219162000f,
	1.800115350307797600f,
	1.799948201211337500f, 1.799780985935927300f, 1.799613704487715200f,
	1.799446356872851400f,
	1.799278943097489100f, 1.799111463167783400f, 1.798943917089892000f,
	1.798776304869975200f,
	1.798608626514195800f, 1.798440882028718500f, 1.798273071419711000f,
	1.798105194693343500f,
	1.797937251855787700f, 1.797769242913218800f, 1.797601167871813800f,
	1.797433026737752700f,
	1.797264819517217200f, 1.797096546216391900f, 1.796928206841463800f,
	1.796759801398622100f,
	1.796591329894058800f, 1.796422792333968000f, 1.796254188724546500f,
	1.796085519071992900f,
	1.795916783382509200f, 1.795747981662299200f, 1.795579113917569200f,
	1.795410180154527900f,
	1.795241180379386800f, 1.795072114598359200f, 1.794902982817661500f,
	1.794733785043511900f,
	1.794564521282131300f, 1.794395191539743400f, 1.794225795822573600f,
	1.794056334136850300f,
	1.793886806488804100f, 1.793717212884667900f, 1.793547553330677300f,
	1.793377827833070100f,
	1.793208036398086900f, 1.793038179031970000f, 1.792868255740965000f,
	1.792698266531319400f,
	1.792528211409282900f, 1.792358090381108300f, 1.792187903453050100f,
	1.792017650631366100f,
	1.791847331922315600f, 1.791676947332161000f, 1.791506496867166600f,
	1.791335980533599300f,
	1.791165398337728900f, 1.790994750285827000f, 1.790824036384167900f,
	1.790653256639028100f,
	1.790482411056686800f, 1.790311499643425500f, 1.790140522405528200f,
	1.789969479349281100f,
	1.789798370480973000f, 1.789627195806895200f, 1.789455955333341100f,
	1.789284649066606800f,
	1.789113277012990900f, 1.788941839178794100f, 1.788770335570319700f,
	1.788598766193873600f,
	1.788427131055763600f, 1.788255430162300400f, 1.788083663519796800f,
	1.787911831134568300f,
	1.787739933012932900f, 1.787567969161210300f, 1.787395939585723500f,
	1.787223844292797500f,
	1.787051683288759500f, 1.786879456579939700f, 1.786707164172670200f,
	1.786534806073285700f,
	1.786362382288123400f, 1.786189892823522700f, 1.786017337685825700f,
	1.785844716881376700f,
	1.785672030416522300f, 1.785499278297612000f, 1.785326460530997300f,
	1.785153577123032000f,
	1.784980628080072900f, 1.784807613408478300f, 1.784634533114609800f,
	1.784461387204831400f,
	1.784288175685508700f, 1.784114898563010200f, 1.783941555843707100f,
	1.783768147533972200f,
	1.783594673640181800f, 1.783421134168713800f, 1.783247529125948900f,
	1.783073858518269700f,
	1.782900122352062000f, 1.782726320633713200f, 1.782552453369613800f,
	1.782378520566156200f,
	1.782204522229735600f, 1.782030458366749200f, 1.781856328983596900f,
	1.781682134086680900f,
	1.781507873682406200f, 1.781333547777179200f, 1.781159156377410100f,
	1.780984699489510200f,
	1.780810177119894100f, 1.780635589274978600f, 1.780460935961182300f,
	1.780286217184927000f,
	1.780111432952636600f, 1.779936583270737400f, 1.779761668145658300f,
	1.779586687583830200f,
	1.779411641591686500f, 1.779236530175663600f, 1.779061353342199500f,
	1.778886111097735000f,
	1.778710803448713400f, 1.778535430401580100f, 1.778359991962783000f,
	1.778184488138772900f,
	1.778008918936002000f, 1.777833284360925900f, 1.777657584420002000f,
	1.777481819119690200f,
	1.777305988466453000f, 1.777130092466755200f, 1.776954131127064200f,
	1.776778104453849100f,
	1.776602012453582400f, 1.776425855132738100f, 1.776249632497793200f,
	1.776073344555227000f,
	1.775896991311520800f, 1.775720572773158900f, 1.775544088946627600f,
	1.775367539838415700f,
	1.775190925455014400f, 1.775014245802917200f, 1.774837500888620400f,
	1.774660690718622000f,
	1.774483815299423100f, 1.774306874637527000f, 1.774129868739439100f,
	1.773952797611667100f,
	1.773775661260722100f, 1.773598459693116500f, 1.773421192915365400f,
	1.773243860933986400f,
	1.773066463755499800f, 1.772889001386427800f, 1.772711473833295200f,
	1.772533881102629000f,
	1.772356223200959100f, 1.772178500134817100f, 1.772000711910737700f,
	1.771822858535257600f,
	1.771644940014915700f, 1.771466956356254000f, 1.771288907565816000f,
	1.771110793650148500f,
	1.770932614615799800f, 1.770754370469321400f, 1.770576061217266500f,
	1.770397686866191300f,
	1.770219247422653700f, 1.770040742893215000f, 1.769862173284438000f,
	1.769683538602888000f,
	1.769504838855133100f, 1.769326074047743700f, 1.769147244187292200f,
	1.768968349280353800f,
	1.768789389333506000f, 1.768610364353328600f, 1.768431274346403900f,
	1.768252119319316400f,
	1.768072899278653200f, 1.767893614231003800f, 1.767714264182959500f,
	1.767534849141115100f,
	1.767355369112067100f, 1.767175824102414000f, 1.766996214118757800f,
	1.766816539167701800f,
	1.766636799255852300f, 1.766456994389817600f, 1.766277124576209000f,
	1.766097189821639300f,
	1.765917190132724600f, 1.765737125516083000f, 1.765556995978334800f,
	1.765376801526102700f,
	1.765196542166012100f, 1.765016217904690900f, 1.764835828748768400f,
	1.764655374704877700f,
	1.764474855779653200f, 1.764294271979732100f, 1.764113623311754000f,
	1.763932909782361100f,
	1.763752131398197200f, 1.763571288165909400f, 1.763390380092146400f,
	1.763209407183560200f,
	1.763028369446804500f, 1.762847266888535100f, 1.762666099515411100f,
	1.762484867334093400f,
	1.762303570351245300f, 1.762122208573532600f, 1.761940782007623600f,
	1.761759290660188400f,
	1.761577734537900500f, 1.761396113647435000f, 1.761214427995469100f,
	1.761032677588683800f,
	1.760850862433760700f, 1.760668982537384900f, 1.760487037906243600f,
	1.760305028547026500f,
	1.760122954466425600f, 1.759940815671135100f, 1.759758612167851700f,
	1.759576343963274600f,
	1.759394011064105100f, 1.759211613477047200f, 1.759029151208807400f,
	1.758846624266093800f,
	1.758664032655617500f, 1.758481376384092500f, 1.758298655458233600f,
	1.758115869884759700f,
	1.757933019670390800f, 1.757750104821850000f, 1.757567125345862700f,
	1.757384081249156100f,
	1.757200972538460700f, 1.757017799220508500f, 1.756834561302034400f,
	1.756651258789775800f,
	1.756467891690471700f, 1.756284460010864200f, 1.756100963757697900f,
	1.755917402937718900f,
	1.755733777557676500f, 1.755550087624322000f, 1.755366333144409200f,
	1.755182514124693900f,
	1.754998630571935200f, 1.754814682492893600f, 1.754630669894332600f,
	1.754446592783017500f,
	1.754262451165716300f, 1.754078245049199600f, 1.753893974440240000f,
	1.753709639345612600f,
	1.753525239772095100f, 1.753340775726466700f, 1.753156247215510400f,
	1.752971654246010300f,
	1.752786996824753600f, 1.752602274958529500f, 1.752417488654129700f,
	1.752232637918348200f,
	1.752047722757981600f, 1.751862743179828600f, 1.751677699190690400f,
	1.751492590797370600f,
	1.751307418006674800f, 1.751122180825411800f, 1.750936879260391700f,
	1.750751513318427700f,
	1.750566083006335600f, 1.750380588330932500f, 1.750195029299038900f,
	1.750009405917477100f,
	1.749823718193071800f, 1.749637966132650900f, 1.749452149743043100f,
	1.749266269031080700f,
	1.749080324003598100f, 1.748894314667431800f, 1.748708241029421000f,
	1.748522103096407300f,
	1.748335900875233900f, 1.748149634372747200f, 1.747963303595795500f,
	1.747776908551230000f,
	1.747590449245904000f, 1.747403925686672500f, 1.747217337880393900f,
	1.747030685833928200f,
	1.746843969554138200f, 1.746657189047889200f, 1.746470344322048200f,
	1.746283435383485100f,
	1.746096462239072000f, 1.745909424895683200f, 1.745722323360195900f,
	1.745535157639489100f,
	1.745347927740444200f, 1.745160633669945200f, 1.744973275434878300f,
	1.744785853042132300f,
	1.744598366498598200f, 1.744410815811169300f, 1.744223200986741100f,
	1.744035522032211900f,
	1.743847778954482000f, 1.743659971760454200f, 1.743472100457033700f,
	1.743284165051127700f,
	1.743096165549646400f, 1.742908101959502100f, 1.742719974287608900f,
	1.742531782540884100f,
	1.742343526726246800f, 1.742155206850618800f, 1.741966822920923800f,
	1.741778374944088000f,
	1.741589862927040800f, 1.741401286876712800f, 1.741212646800037300f,
	1.741023942703950200f,
	1.740835174595389600f, 1.740646342481295900f, 1.740457446368612000f,
	1.740268486264283200f,
	1.740079462175256900f, 1.739890374108482600f, 1.739701222070913200f,
	1.739512006069502800f,
	1.739322726111208500f, 1.739133382202989500f, 1.738943974351807600f,
	1.738754502564626700f,
	1.738564966848413100f, 1.738375367210135400f, 1.738185703656765200f,
	1.737995976195275000f,
	1.737806184832640900f, 1.737616329575841300f, 1.737426410431856200f,
	1.737236427407668800f,
	1.737046380510263800f, 1.736856269746629000f, 1.736666095123754000f,
	1.736475856648631400f,
	1.736285554328254900f, 1.736095188169622500f, 1.735904758179732400f,
	1.735714264365586700f,
	1.735523706734189100f, 1.735333085292545900f, 1.735142400047666100f,
	1.734951651006560100f,
	1.734760838176241400f, 1.734569961563725600f, 1.734379021176030600f,
	1.734188017020177100f,
	1.733996949103187500f, 1.733805817432086900f, 1.733614622013902600f,
	1.733423362855664100f,
	1.733232039964403900f, 1.733040653347156300f, 1.732849203010957900f,
	1.732657688962847600f,
	1.732466111209867200f, 1.732274469759060200f, 1.732082764617472800f,
	1.731890995792153600f,
	1.731699163290153100f, 1.731507267118524500f, 1.731315307284323700f,
	1.731123283794607800f,
	1.730931196656437600f, 1.730739045876875200f, 1.730546831462985500f,
	1.730354553421835600f,
	1.730162211760495300f, 1.729969806486036500f, 1.729777337605533000f,
	1.729584805126061400f,
	1.729392209054700900f, 1.729199549398532400f, 1.729006826164639400f,
	1.728814039360108100f,
	1.728621188992026400f, 1.728428275067485100f, 1.728235297593577100f,
	1.728042256577397200f,
	1.727849152026043500f, 1.727655983946615700f, 1.727462752346216000f,
	1.727269457231948900f,
	1.727076098610921500f, 1.726882676490243000f, 1.726689190877025000f,
	1.726495641778381200f,
	1.726302029201427900f, 1.726108353153283900f, 1.725914613641069900f,
	1.725720810671909300f,
	1.725526944252927700f, 1.725333014391252900f, 1.725139021094015200f,
	1.724944964368347000f,
	1.724750844221383500f, 1.724556660660261800f, 1.724362413692121400f,
	1.724168103324104300f,
	1.723973729563354600f, 1.723779292417019200f, 1.723584791892246700f,
	1.723390227996188600f,
	1.723195600735998100f, 1.723000910118831300f, 1.722806156151846400f,
	1.722611338842204000f,
	1.722416458197066900f, 1.722221514223600100f, 1.722026506928971500f,
	1.721831436320350800f,
	1.721636302404910200f, 1.721441105189824000f, 1.721245844682269600f,
	1.721050520889425600f,
	1.720855133818473900f, 1.720659683476597900f, 1.720464169870984200f,
	1.720268593008821100f,
	1.720072952897299100f, 1.719877249543611900f, 1.719681482954954500f,
	1.719485653138524800f,
	1.719289760101522900f, 1.719093803851151400f, 1.718897784394614900f,
	1.718701701739120400f,
	1.718505555891877400f, 1.718309346860097600f, 1.718113074650995200f,
	1.717916739271786500f,
	1.717720340729689700f, 1.717523879031926500f, 1.717327354185719900f,
	1.717130766198295700f,
	1.716934115076881800f, 1.716737400828708400f, 1.716540623461008100f,
	1.716343782981016200f,
	1.716146879395969500f, 1.715949912713108100f, 1.715752882939673300f,
	1.715555790082909900f,
	1.715358634150064000f, 1.715161415148384500f, 1.714964133085122900f,
	1.714766787967532600f,
	1.714569379802868900f, 1.714371908598390800f, 1.714174374361358000f,
	1.713976777099033700f,
	1.713779116818682900f, 1.713581393527573000f, 1.713383607232973600f,
	1.713185757942156800f,
	1.712987845662396800f, 1.712789870400970700f, 1.712591832165157200f,
	1.712393730962237500f,
	1.712195566799495500f, 1.711997339684216700f, 1.711799049623689900f,
	1.711600696625205300f,
	1.711402280696055800f, 1.711203801843536700f, 1.711005260074945200f,
	1.710806655397581600f,
	1.710607987818747700f, 1.710409257345748100f, 1.710210463985889500f,
	1.710011607746480600f,
	1.709812688634833300f, 1.709613706658261100f, 1.709414661824080000f,
	1.709215554139608400f,
	1.709016383612166600f, 1.708817150249077900f, 1.708617854057667300f,
	1.708418495045262300f,
	1.708219073219193300f, 1.708019588586791700f, 1.707820041155392500f,
	1.707620430932332400f,
	1.707420757924950300f, 1.707221022140587900f, 1.707021223586588700f,
	1.706821362270298600f,
	1.706621438199066300f, 1.706421451380242000f, 1.706221401821179200f,
	1.706021289529232800f,
	1.705821114511760300f, 1.705620876776121600f, 1.705420576329679000f,
	1.705220213179796900f,
	1.705019787333842200f, 1.704819298799183700f, 1.704618747583193100f,
	1.704418133693243800f,
	1.704217457136711900f, 1.704016717920976000f, 1.703815916053416300f,
	1.703615051541415900f,
	1.703414124392360000f, 1.703213134613636100f, 1.703012082212634000f,
	1.702810967196746000f,
	1.702609789573366300f, 1.702408549349891500f, 1.702207246533721000f,
	1.702005881132255800f,
	1.701804453152900000f, 1.701602962603059100f, 1.701401409490141300f,
	1.701199793821557300f,
	1.700998115604720000f, 1.700796374847044300f, 1.700594571555948100f,
	1.700392705738850400f,
	1.700190777403173700f, 1.699988786556342300f, 1.699786733205783000f,
	1.699584617358924400f,
	1.699382439023197700f, 1.699180198206036600f, 1.698977894914877100f,
	1.698775529157156700f,
	1.698573100940316400f, 1.698370610271798800f, 1.698168057159048700f,
	1.697965441609513300f,
	1.697762763630642700f, 1.697560023229888200f, 1.697357220414704500f,
	1.697154355192547900f,
	1.696951427570877000f, 1.696748437557152900f, 1.696545385158839200f,
	1.696342270383401200f,
	1.696139093238307400f, 1.695935853731027600f, 1.695732551869034300f,
	1.695529187659802400f,
	1.695325761110809200f, 1.695122272229534000f, 1.694918721023458600f,
	1.694715107500066800f,
	1.694511431666845000f, 1.694307693531282000f, 1.694103893100868100f,
	1.693900030383096900f,
	1.693696105385463800f, 1.693492118115466500f, 1.693288068580604900f,
	1.693083956788381500f,
	1.692879782746300700f, 1.692675546461869900f, 1.692471247942597600f,
	1.692266887195995600f,
	1.692062464229577600f, 1.691857979050859900f, 1.691653431667360600f,
	1.691448822086600400f,
	1.691244150316102000f, 1.691039416363390800f, 1.690834620235994300f,
	1.690629761941442100f,
	1.690424841487266700f, 1.690219858881001800f, 1.690014814130184300f,
	1.689809707242353200f,
	1.689604538225049700f, 1.689399307085817300f, 1.689194013832201500f,
	1.688988658471750600f,
	1.688783241012014700f, 1.688577761460546800f, 1.688372219824901400f,
	1.688166616112636100f,
	1.687960950331309800f, 1.687755222488484600f, 1.687549432591724400f,
	1.687343580648595700f,
	1.687137666666667100f, 1.686931690653509000f, 1.686725652616694900f,
	1.686519552563800400f,
	1.686313390502403000f, 1.686107166440082600f, 1.685900880384421800f,
	1.685694532343004600f,
	1.685488122323418400f, 1.685281650333251900f, 1.685075116380096800f,
	1.684868520471546600f,
	1.684661862615197000f, 1.684455142818646700f, 1.684248361089495800f,
	1.684041517435347400f,
	1.683834611863806100f, 1.683627644382479800f, 1.683420614998977900f,
	1.683213523720911800f,
	1.683006370555896400f, 1.682799155511547600f, 1.682591878595484300f,
	1.682384539815327400f,
	1.682177139178700400f, 1.681969676693228600f, 1.681762152366539600f,
	1.681554566206263900f,
	1.681346918220033800f, 1.681139208415483700f, 1.680931436800250600f,
	1.680723603381973500f,
	1.680515708168294200f, 1.680307751166856300f, 1.680099732385305300f,
	1.679891651831290100f,
	1.679683509512460900f, 1.679475305436470600f, 1.679267039610974300f,
	1.679058712043629300f,
	1.678850322742095200f, 1.678641871714033900f, 1.678433358967109400f,
	1.678224784508988400f,
	1.678016148347339300f, 1.677807450489833300f, 1.677598690944143400f,
	1.677389869717945000f,
	1.677180986818916300f, 1.676972042254736900f, 1.676763036033089600f,
	1.676553968161658600f,
	1.676344838648130600f, 1.676135647500194700f, 1.675926394725542700f,
	1.675717080331867900f,
	1.675507704326866200f, 1.675298266718235900f, 1.675088767513677200f,
	1.674879206720892900f,
	1.674669584347587800f, 1.674459900401469700f, 1.674250154890247300f,
	1.674040347821632800f,
	1.673830479203340000f, 1.673620549043085500f, 1.673410557348587600f,
	1.673200504127567000f,
	1.672990389387746700f, 1.672780213136852300f, 1.672569975382611300f,
	1.672359676132753500f,
	1.672149315395010900f, 1.671938893177118000f, 1.671728409486811500f,
	1.671517864331830000f,
	1.671307257719914800f, 1.671096589658809500f, 1.670885860156259300f,
	1.670675069220012500f,
	1.670464216857819200f, 1.670253303077431800f, 1.670042327886605200f,
	1.669831291293095900f,
	1.669620193304663500f, 1.669409033929069500f, 1.669197813174077200f,
	1.668986531047453000f,
	1.668775187556965000f, 1.668563782710383600f, 1.668352316515481700f,
	1.668140788980034400f,
	1.667929200111818400f, 1.667717549918614100f, 1.667505838408202700f,
	1.667294065588368100f,
	1.667082231466896900f, 1.666870336051577800f, 1.666658379350201000f,
	1.666446361370560000f,
	1.666234282120450100f, 1.666022141607668600f, 1.665809939840015500f,
	1.665597676825292700f,
	1.665385352571304500f, 1.665172967085857700f, 1.664960520376761000f,
	1.664748012451825200f,
	1.664535443318863900f, 1.664322812985692600f, 1.664110121460129000f,
	1.663897368749993400f,
	1.663684554863107800f, 1.663471679807296800f, 1.663258743590387400f,
	1.663045746220208600f,
	1.662832687704591800f, 1.662619568051370500f, 1.662406387268380100f,
	1.662193145363459100f,
	1.661979842344447600f, 1.661766478219188300f, 1.661553052995526000f,
	1.661339566681307600f,
	1.661126019284382200f, 1.660912410812601900f, 1.660698741273819700f,
	1.660485010675892400f,
	1.660271219026677700f, 1.660057366334036300f, 1.659843452605831200f,
	1.659629477849926800f,
	1.659415442074190900f, 1.659201345286492900f, 1.658987187494704200f,
	1.658772968706699000f,
	1.658558688930353400f, 1.658344348173546300f, 1.658129946444157700f,
	1.657915483750071100f,
	1.657700960099171200f, 1.657486375499345900f, 1.657271729958484500f,
	1.657057023484479000f,
	1.656842256085223800f, 1.656627427768615000f, 1.656412538542551200f,
	1.656197588414933600f,
	1.655982577393664700f, 1.655767505486650500f, 1.655552372701798200f,
	1.655337179047017700f,
	1.655121924530220900f, 1.654906609159322500f, 1.654691232942238500f,
	1.654475795886888300f,
	1.654260298001192200f, 1.654044739293073900f, 1.653829119770458900f,
	1.653613439441274500f,
	1.653397698313451300f, 1.653181896394921000f, 1.652966033693617800f,
	1.652750110217479100f,
	1.652534125974443000f, 1.652318080972451400f, 1.652101975219447200f,
	1.651885808723375900f,
	1.651669581492185300f, 1.651453293533826000f, 1.651236944856249600f,
	1.651020535467411200f,
	1.650804065375267400f, 1.650587534587776700f, 1.650370943112901000f,
	1.650154290958603300f,
	1.649937578132849400f, 1.649720804643607400f, 1.649503970498847200f,
	1.649287075706541200f,
	1.649070120274664000f, 1.648853104211192700f, 1.648636027524106100f,
	1.648418890221385400f,
	1.648201692311014300f, 1.647984433800978600f, 1.647767114699266100f,
	1.647549735013867000f,
	1.647332294752774200f, 1.647114793923981600f, 1.646897232535486500f,
	1.646679610595287900f,
	1.646461928111387300f, 1.646244185091788400f, 1.646026381544496400f,
	1.645808517477519700f,
	1.645590592898868600f, 1.645372607816555400f, 1.645154562238594800f,
	1.644936456173004000f,
	1.644718289627801600f, 1.644500062611009300f, 1.644281775130650900f,
	1.644063427194751600f,
	1.643845018811340300f, 1.643626549988446200f, 1.643408020734102600f,
	1.643189431056343700f,
	1.642970780963206800f, 1.642752070462730800f, 1.642533299562957100f,
	1.642314468271929300f,
	1.642095576597693200f, 1.641876624548297000f, 1.641657612131790500f,
	1.641438539356226500f,
	1.641219406229659700f, 1.641000212760146800f, 1.640780958955747200f,
	1.640561644824521700f,
	1.640342270374534500f, 1.640122835613851100f, 1.639903340550539200f,
	1.639683785192669600f,
	1.639464169548314100f, 1.639244493625547900f, 1.639024757432447500f,
	1.638804960977092100f,
	1.638585104267562800f, 1.638365187311943400f, 1.638145210118319400f,
	1.637925172694778800f,
	1.637705075049411800f, 1.637484917190310800f, 1.637264699125570200f,
	1.637044420863286600f,
	1.636824082411559600f, 1.636603683778490100f, 1.636383224972181500f,
	1.636162706000739300f,
	1.635942126872271800f, 1.635721487594888400f, 1.635500788176702100f,
	1.635280028625826900f,
	1.635059208950379700f, 1.634838329158479200f, 1.634617389258246700f,
	1.634396389257805700f,
	1.634175329165281400f, 1.633954208988801700f, 1.633733028736496400f,
	1.633511788416498000f,
	1.633290488036940500f, 1.633069127605960800f, 1.632847707131697600f,
	1.632626226622291700f,
	1.632404686085886300f, 1.632183085530627200f, 1.631961424964661700f,
	1.631739704396139900f,
	1.631517923833213400f, 1.631296083284036900f, 1.631074182756766300f,
	1.630852222259560700f,
	1.630630201800580900f, 1.630408121387990000f, 1.630185981029953000f,
	1.629963780734637400f,
	1.629741520510213000f, 1.629519200364851800f, 1.629296820306727700f,
	1.629074380344017100f,
	1.628851880484898200f, 1.628629320737551700f, 1.628406701110161100f,
	1.628184021610910700f,
	1.627961282247988300f, 1.627738483029583100f, 1.627515623963887000f,
	1.627292705059093700f,
	1.627069726323399500f, 1.626846687765002700f, 1.626623589392103500f,
	1.626400431212904800f,
	1.626177213235611400f, 1.625953935468430500f, 1.625730597919571300f,
	1.625507200597245500f,
	1.625283743509666300f, 1.625060226665050000f, 1.624836650071614500f,
	1.624613013737580000f,
	1.624389317671169500f, 1.624165561880607000f, 1.623941746374119500f,
	1.623717871159936300f,
	1.623493936246288300f, 1.623269941641409400f, 1.623045887353534900f,
	1.622821773390902700f,
	1.622597599761753000f, 1.622373366474327800f, 1.622149073536871800f,
	1.621924720957631300f,
	1.621700308744855200f, 1.621475836906794500f, 1.621251305451702400f,
	1.621026714387834300f,
	1.620802063723447700f, 1.620577353466802700f, 1.620352583626160500f,
	1.620127754209786100f,
	1.619902865225945300f, 1.619677916682906700f, 1.619452908588941300f,
	1.619227840952321800f,
	1.619002713781323200f, 1.618777527084222800f, 1.618552280869300300f,
	1.618326975144837000f,
	1.618101609919117200f, 1.617876185200426600f, 1.617650700997053500f,
	1.617425157317288200f,
	1.617199554169423500f, 1.616973891561754200f, 1.616748169502577200f,
	1.616522388000191500f,
	1.616296547062898500f, 1.616070646699001800f, 1.615844686916807300f,
	1.615618667724622700f,
	1.615392589130757900f, 1.615166451143525300f, 1.614940253771239400f,
	1.614713997022216900f,
	1.614487680904776600f, 1.614261305427239200f, 1.614034870597928400f,
	1.613808376425168900f,
	1.613581822917288900f, 1.613355210082617800f, 1.613128537929487500f,
	1.612901806466232200f,
	1.612675015701188000f, 1.612448165642693400f, 1.612221256299089200f,
	1.611994287678718100f,
	1.611767259789925100f, 1.611540172641057200f, 1.611313026240463800f,
	1.611085820596496600f,
	1.610858555717509200f, 1.610631231611857800f, 1.610403848287899700f,
	1.610176405753995800f,
	1.609948904018508200f, 1.609721343089801600f, 1.609493722976242900f,
	1.609266043686200700f,
	1.609038305228046400f, 1.608810507610153100f, 1.608582650840896200f,
	1.608354734928653800f,
	1.608126759881805400f, 1.607898725708732900f, 1.607670632417820500f,
	1.607442480017454700f,
	1.607214268516024000f, 1.606985997921919000f, 1.606757668243532500f,
	1.606529279489259600f,
	1.606300831667497600f, 1.606072324786645500f, 1.605843758855105300f,
	1.605615133881280700f,
	1.605386449873577300f, 1.605157706840403300f, 1.604928904790168700f,
	1.604700043731286200f,
	1.604471123672170500f, 1.604242144621237800f, 1.604013106586907400f,
	1.603784009577600100f,
	1.603554853601739700f, 1.603325638667751000f, 1.603096364784061900f,
	1.602867031959102100f,
	1.602637640201303400f, 1.602408189519099800f, 1.602178679920927900f,
	1.601949111415226000f,
	1.601719484010434300f, 1.601489797714996000f, 1.601260052537355700f,
	1.601030248485960900f,
	1.600800385569260300f, 1.600570463795705700f, 1.600340483173750400f,
	1.600110443711850300f,
	1.599880345418463100f, 1.599650188302049100f, 1.599419972371070500f,
	1.599189697633991400f,
	1.598959364099278700f, 1.598728971775401000f, 1.598498520670828900f,
	1.598268010794035900f,
	1.598037442153496900f, 1.597806814757689200f, 1.597576128615092200f,
	1.597345383734188000f,
	1.597114580123460100f, 1.596883717791394800f, 1.596652796746479600f,
	1.596421816997205500f,
	1.596190778552064800f, 1.595959681419551800f, 1.595728525608163700f,
	1.595497311126399300f,
	1.595266037982759500f, 1.595034706185747500f, 1.594803315743869000f,
	1.594571866665631700f,
	1.594340358959544800f, 1.594108792634120600f, 1.593877167697873100f,
	1.593645484159318200f,
	1.593413742026974500f, 1.593181941309362400f, 1.592950082015004700f,
	1.592718164152426000f,
	1.592486187730153300f, 1.592254152756715600f, 1.592022059240644400f,
	1.591789907190473100f,
	1.591557696614737100f, 1.591325427521974100f, 1.591093099920724200f,
	1.590860713819529400f,
	1.590628269226933600f, 1.590395766151483400f, 1.590163204601727100f,
	1.589930584586215500f,
	1.589697906113501000f, 1.589465169192139100f, 1.589232373830686400f,
	1.588999520037702300f,
	1.588766607821748200f, 1.588533637191387400f, 1.588300608155185600f,
	1.588067520721711000f,
	1.587834374899533400f, 1.587601170697224600f, 1.587367908123358900f,
	1.587134587186513000f,
	1.586901207895265300f, 1.586667770258196600f, 1.586434274283889500f,
	1.586200719980929200f,
	1.585967107357902700f, 1.585733436423399000f, 1.585499707186010200f,
	1.585265919654329300f,
	1.585032073836952100f, 1.584798169742476400f, 1.584564207379502500f,
	1.584330186756632200f,
	1.584096107882470000f, 1.583861970765622100f, 1.583627775414697000f,
	1.583393521838305700f,
	1.583159210045060900f, 1.582924840043577400f, 1.582690411842472700f,
	1.582455925450365600f,
	1.582221380875877800f, 1.581986778127632700f, 1.581752117214255900f,
	1.581517398144375800f,
	1.581282620926621300f, 1.581047785569625400f, 1.580812892082021900f,
	1.580577940472447200f,
	1.580342930749539800f, 1.580107862921940700f, 1.579872736998292100f,
	1.579637552987239100f,
	1.579402310897428900f, 1.579167010737510600f, 1.578931652516135700f,
	1.578696236241957200f,
	1.578460761923630800f, 1.578225229569814700f, 1.577989639189168100f,
	1.577753990790353500f,
	1.577518284382034800f, 1.577282519972878200f, 1.577046697571552000f,
	1.576810817186727000f,
	1.576574878827075700f, 1.576338882501273000f, 1.576102828217995600f,
	1.575866715985922500f,
	1.575630545813735200f, 1.575394317710116600f, 1.575158031683752300f,
	1.574921687743330300f,
	1.574685285897539800f, 1.574448826155072400f, 1.574212308524622500f,
	1.573975733014886000f,
	1.573739099634561500f, 1.573502408392348600f, 1.573265659296950300f,
	1.573028852357070800f,
	1.572791987581417100f, 1.572555064978698100f, 1.572318084557624800f,
	1.572081046326909900f,
	1.571843950295269000f, 1.571606796471419100f, 1.571369584864080100f,
	1.571132315481973200f,
	1.570894988333822400f, 1.570657603428353300f, 1.570420160774294000f,
	1.570182660380374600f,
	1.569945102255327200f, 1.569707486407886600f, 1.569469812846788500f,
	1.569232081580771900f,
	1.568994292618577400f, 1.568756445968948000f, 1.568518541640628400f,
	1.568280579642366000f,
	1.568042559982909500f, 1.567804482671010500f, 1.567566347715422500f,
	1.567328155124900800f,
	1.567089904908203200f, 1.566851597074089500f, 1.566613231631321500f,
	1.566374808588663300f,
	1.566136327954881000f, 1.565897789738742900f, 1.565659193949019400f,
	1.565420540594482800f,
	1.565181829683907700f, 1.564943061226071100f, 1.564704235229751500f,
	1.564465351703730400f,
	1.564226410656790000f, 1.563987412097716200f, 1.563748356035296000f,
	1.563509242478319000f,
	1.563270071435576500f, 1.563030842915862100f, 1.562791556927971800f,
	1.562552213480703300f,
	1.562312812582856500f, 1.562073354243233700f, 1.561833838470639200f,
	1.561594265273878800f,
	1.561354634661761300f, 1.561114946643096900f, 1.560875201226698900f,
	1.560635398421381400f,
	1.560395538235961800f, 1.560155620679258400f, 1.559915645760092900f,
	1.559675613487288200f,
	1.559435523869669500f, 1.559195376916064700f, 1.558955172635302800f,
	1.558714911036215700f,
	1.558474592127637100f, 1.558234215918402600f, 1.557993782417350400f,
	1.557753291633320500f,
	1.557512743575155000f, 1.557272138251698300f, 1.557031475671796400f,
	1.556790755844298400f,
	1.556549978778054300f, 1.556309144481917300f, 1.556068252964741600f,
	1.555827304235384500f,
	1.555586298302704900f, 1.555345235175563900f, 1.555104114862824600f,
	1.554862937373352500f,
	1.554621702716015000f, 1.554380410899681300f, 1.554139061933223200f,
	1.553897655825514600f,
	1.553656192585431100f, 1.553414672221850700f, 1.553173094743653300f,
	1.552931460159721100f,
	1.552689768478938500f, 1.552448019710191300f, 1.552206213862368500f,
	1.551964350944360100f,
	1.551722430965059000f, 1.551480453933359800f, 1.551238419858159700f,
	1.550996328748356800f,
	1.550754180612852900f, 1.550511975460550500f, 1.550269713300355100f,
	1.550027394141174000f,
	1.549785017991916400f, 1.549542584861493900f, 1.549300094758820000f,
	1.549057547692810600f,
	1.548814943672383300f, 1.548572282706457900f, 1.548329564803956300f,
	1.548086789973802700f,
	1.547843958224923000f, 1.547601069566245900f, 1.547358124006701400f,
	1.547115121555221700f,
	1.546872062220741700f, 1.546628946012197800f, 1.546385772938528600f,
	1.546142543008675300f,
	1.545899256231580300f, 1.545655912616188800f, 1.545412512171447700f,
	1.545169054906306200f,
	1.544925540829715600f, 1.544681969950629300f, 1.544438342278002600f,
	1.544194657820792800f,
	1.543950916587959700f, 1.543707118588464800f, 1.543463263831272000f,
	1.543219352325347200f,
	1.542975384079658300f, 1.542731359103175300f, 1.542487277404870100f,
	1.542243138993717000f,
	1.541998943878692300f, 1.541754692068774600f, 1.541510383572944000f,
	1.541266018400183200f,
	1.541021596559476700f, 1.540777118059811100f, 1.540532582910175500f,
	1.540287991119560600f,
	1.540043342696959100f, 1.539798637651366400f, 1.539553875991779300f,
	1.539309057727197300f,
	1.539064182866621400f, 1.538819251419055100f, 1.538574263393503800f,
	1.538329218798974800f,
	1.538084117644477900f, 1.537838959939025200f, 1.537593745691629500f,
	1.537348474911307300f,
	1.537103147607076200f, 1.536857763787956400f, 1.536612323462969800f,
	1.536366826641140800f,
	1.536121273331495300f, 1.535875663543061700f, 1.535629997284870400f,
	1.535384274565953600f,
	1.535138495395346400f, 1.534892659782085100f, 1.534646767735208000f,
	1.534400819263756400f,
	1.534154814376772700f, 1.533908753083302200f, 1.533662635392391700f,
	1.533416461313090100f,
	1.533170230854448400f, 1.532923944025520200f, 1.532677600835360600f,
	1.532431201293027000f,
	1.532184745407578500f, 1.531938233188077100f, 1.531691664643585900f,
	1.531445039783170500f,
	1.531198358615898800f, 1.530951621150840700f, 1.530704827397067800f,
	1.530457977363654000f,
	1.530211071059675200f, 1.529964108494209700f, 1.529717089676337500f,
	1.529470014615140800f,
	1.529222883319703700f, 1.528975695799112500f, 1.528728452062455600f,
	1.528481152118823700f,
	1.528233795977309400f, 1.527986383647006500f, 1.527738915137012400f,
	1.527491390456425600f,
	1.527243809614346600f, 1.526996172619878900f, 1.526748479482126700f,
	1.526500730210197200f,
	1.526252924813199500f, 1.526005063300244900f, 1.525757145680446200f,
	1.525509171962918800f,
	1.525261142156779900f, 1.525013056271149000f, 1.524764914315147200f,
	1.524516716297898300f,
	1.524268462228527900f, 1.524020152116163200f, 1.523771785969934000f,
	1.523523363798972000f,
	1.523274885612411200f, 1.523026351419387100f, 1.522777761229038100f,
	1.522529115050503600f,
	1.522280412892925900f, 1.522031654765448900f, 1.521782840677218700f,
	1.521533970637383800f,
	1.521285044655094300f, 1.521036062739502300f, 1.520787024899762100f,
	1.520537931145030400f,
	1.520288781484465700f, 1.520039575927228500f, 1.519790314482481100f,
	1.519540997159388300f,
	1.519291623967116600f, 1.519042194914835200f, 1.518792710011714500f,
	1.518543169266927600f,
	1.518293572689648900f, 1.518043920289055900f, 1.517794212074327500f,
	1.517544448054644500f,
	1.517294628239190400f, 1.517044752637150000f, 1.516794821257710500f,
	1.516544834110061600f,
	1.516294791203394200f, 1.516044692546901800f, 1.515794538149779700f,
	1.515544328021225500f,
	1.515294062170438700f, 1.515043740606620800f, 1.514793363338975600f,
	1.514542930376708600f,
	1.514292441729027300f, 1.514041897405141700f, 1.513791297414263800f,
	1.513540641765606800f,
	1.513289930468387300f, 1.513039163531823000f, 1.512788340965133500f,
	1.512537462777541200f,
	1.512286528978270300f, 1.512035539576546600f, 1.511784494581598600f,
	1.511533394002656100f,
	1.511282237848951400f, 1.511031026129719100f, 1.510779758854195400f,
	1.510528436031618900f,
	1.510277057671229400f, 1.510025623782270000f, 1.509774134373984800f,
	1.509522589455620600f,
	1.509270989036425800f, 1.509019333125651200f, 1.508767621732549400f,
	1.508515854866375100f,
	1.508264032536385000f, 1.508012154751837700f, 1.507760221521994700f,
	1.507508232856118200f,
	1.507256188763473200f, 1.507004089253327000f, 1.506751934334948000f,
	1.506499724017607900f,
	1.506247458310579400f, 1.505995137223137500f, 1.505742760764559300f,
	1.505490328944124200f,
	1.505237841771113200f, 1.504985299254809800f, 1.504732701404498900f,
	1.504480048229468000f,
	1.504227339739006500f, 1.503974575942405700f, 1.503721756848958700f,
	1.503468882467961600f,
	1.503215952808711500f, 1.502962967880507600f, 1.502709927692651900f,
	1.502456832254447600f,
	1.502203681575200700f, 1.501950475664218600f, 1.501697214530810700f,
	1.501443898184289200f,
	1.501190526633967600f, 1.500937099889161600f, 1.500683617959188900f,
	1.500430080853369500f,
	1.500176488581024900f, 1.499922841151479600f, 1.499669138574058800f,
	1.499415380858090800f,
	1.499161568012905300f, 1.498907700047834600f, 1.498653776972212600f,
	1.498399798795375000f,
	1.498145765526660300f, 1.497891677175408500f, 1.497637533750961300f,
	1.497383335262663300f,
	1.497129081719860400f, 1.496874773131900800f, 1.496620409508134800f,
	1.496365990857914600f,
	1.496111517190594300f, 1.495856988515530400f, 1.495602404842080800f,
	1.495347766179606400f,
	1.495093072537469100f, 1.494838323925033400f, 1.494583520351665500f,
	1.494328661826734200f,
	1.494073748359609600f, 1.493818779959664300f, 1.493563756636272500f,
	1.493308678398810800f,
	1.493053545256657800f, 1.492798357219194100f, 1.492543114295801900f,
	1.492287816495866200f,
	1.492032463828773200f, 1.491777056303911700f, 1.491521593930672100f,
	1.491266076718446900f,
	1.491010504676631500f, 1.490754877814621800f, 1.490499196141816600f,
	1.490243459667616600f,
	1.489987668401424800f, 1.489731822352645500f, 1.489475921530685900f,
	1.489219965944954300f,
	1.488963955604861500f, 1.488707890519820600f, 1.488451770699245900f,
	1.488195596152554800f,
	1.487939366889165600f, 1.487683082918499300f, 1.487426744249978400f,
	1.487170350893028500f,
	1.486913902857075700f, 1.486657400151549600f, 1.486400842785880100f,
	1.486144230769501000f,
	1.485887564111846500f, 1.485630842822354100f, 1.485374066910462500f,
	1.485117236385612200f,
	1.484860351257246500f, 1.484603411534810300f, 1.484346417227750700f,
	1.484089368345516300f,
	1.483832264897558400f, 1.483575106893329600f, 1.483317894342285100f,
	1.483060627253882000f,
	1.482803305637578900f, 1.482545929502837100f, 1.482288498859119400f,
	1.482031013715890700f,
	1.481773474082618300f, 1.481515879968770900f, 1.481258231383819800f,
	1.481000528337237800f,
	1.480742770838499900f, 1.480484958897083200f, 1.480227092522466500f,
	1.479969171724131200f,
	1.479711196511560100f, 1.479453166894238100f, 1.479195082881652200f,
	1.478936944483291600f,
	1.478678751708647000f, 1.478420504567211900f, 1.478162203068481100f,
	1.477903847221951400f,
	1.477645437037121900f, 1.477386972523493800f, 1.477128453690569800f,
	1.476869880547855300f,
	1.476611253104856700f, 1.476352571371083700f, 1.476093835356046700f,
	1.475835045069259000f,
	1.475576200520235500f, 1.475317301718493300f, 1.475058348673551100f,
	1.474799341394929900f,
	1.474540279892153000f, 1.474281164174744900f, 1.474021994252233000f,
	1.473762770134145800f,
	1.473503491830014300f, 1.473244159349371700f, 1.472984772701752900f,
	1.472725331896694400f,
	1.472465836943735600f, 1.472206287852416900f, 1.471946684632281500f,
	1.471687027292874400f,
	1.471427315843742100f, 1.471167550294433700f, 1.470907730654499800f,
	1.470647856933493300f,
	1.470387929140969200f, 1.470127947286484100f, 1.469867911379596900f,
	1.469607821429868500f,
	1.469347677446861500f, 1.469087479440140300f, 1.468827227419272200f,
	1.468566921393825700f,
	1.468306561373371900f, 1.468046147367482600f, 1.467785679385733300f,
	1.467525157437700200f,
	1.467264581532962100f, 1.467003951681099800f, 1.466743267891695800f,
	1.466482530174334500f,
	1.466221738538602500f, 1.465960892994088800f, 1.465699993550383400f,
	1.465439040217079400f,
	1.465178033003770700f, 1.464916971920054100f, 1.464655856975527900f,
	1.464394688179792900f,
	1.464133465542451200f, 1.463872189073107500f, 1.463610858781367900f,
	1.463349474676840700f,
	1.463088036769136600f, 1.462826545067867700f, 1.462564999582648600f,
	1.462303400323095000f,
	1.462041747298825900f, 1.461780040519460800f, 1.461518279994622200f,
	1.461256465733934400f,
	1.460994597747023600f, 1.460732676043517800f, 1.460470700633046800f,
	1.460208671525243400f,
	1.459946588729741100f, 1.459684452256176300f, 1.459422262114186800f,
	1.459160018313412400f,
	1.458897720863495500f, 1.458635369774079500f, 1.458372965054810700f,
	1.458110506715337000f,
	1.457847994765308200f, 1.457585429214375700f, 1.457322810072193800f,
	1.457060137348418000f,
	1.456797411052706200f, 1.456534631194717800f, 1.456271797784114900f,
	1.456008910830560500f,
	1.455745970343720800f, 1.455482976333263100f, 1.455219928808857200f,
	1.454956827780174100f,
	1.454693673256887600f, 1.454430465248673300f, 1.454167203765208000f,
	1.453903888816171900f,
	1.453640520411245900f, 1.453377098560113100f, 1.453113623272459100f,
	1.452850094557971000f,
	1.452586512426338000f, 1.452322876887251400f, 1.452059187950404100f,
	1.451795445625491300f,
	1.451531649922210200f, 1.451267800850259500f, 1.451003898419340500f,
	1.450739942639155800f,
	1.450475933519410400f, 1.450211871069811300f, 1.449947755300067500f,
	1.449683586219889400f,
	1.449419363838989800f, 1.449155088167083600f, 1.448890759213887100f,
	1.448626376989119400f,
	1.448361941502500900f, 1.448097452763754000f, 1.447832910782603100f,
	1.447568315568775100f,
	1.447303667131997900f, 1.447038965482002200f, 1.446774210628520200f,
	1.446509402581286400f,
	1.446244541350036700f, 1.445979626944509300f, 1.445714659374444500f,
	1.445449638649584500f,
	1.445184564779673500f, 1.444919437774456700f, 1.444654257643682900f,
	1.444389024397101600f,
	1.444123738044464900f, 1.443858398595526400f, 1.443593006060042100f,
	1.443327560447769600f,
	1.443062061768468400f, 1.442796510031900500f, 1.442530905247829200f,
	1.442265247426020200f,
	1.441999536576240800f, 1.441733772708260600f, 1.441467955831850800f,
	1.441202085956784900f,
	1.440936163092837900f, 1.440670187249787600f, 1.440404158437412500f,
	1.440138076665494100f,
	1.439871941943815300f, 1.439605754282161400f, 1.439339513690319100f,
	1.439073220178077400f,
	1.438806873755226900f, 1.438540474431560600f, 1.438274022216873500f,
	1.438007517120961900f,
	1.437740959153624500f, 1.437474348324662100f, 1.437207684643876800f,
	1.436940968121073600f,
	1.436674198766058500f, 1.436407376588640000f, 1.436140501598628400f,
	1.435873573805835900f,
	1.435606593220076600f, 1.435339559851166500f, 1.435072473708924000f,
	1.434805334803169100f,
	1.434538143143723200f, 1.434270898740410700f, 1.434003601603057300f,
	1.433736251741490700f,
	1.433468849165540500f, 1.433201393885038500f, 1.432933885909818000f,
	1.432666325249714700f,
	1.432398711914566200f, 1.432131045914211600f, 1.431863327258492400f,
	1.431595555957251700f,
	1.431327732020334800f, 1.431059855457588600f, 1.430791926278862400f,
	1.430523944494007400f,
	1.430255910112876000f, 1.429987823145323100f, 1.429719683601205800f,
	1.429451491490382900f,
	1.429183246822714800f, 1.428914949608064200f, 1.428646599856295400f,
	1.428378197577275100f,
	1.428109742780871800f, 1.427841235476955400f, 1.427572675675398600f,
	1.427304063386075200f,
	1.427035398618861500f, 1.426766681383635500f, 1.426497911690277000f,
	1.426229089548668200f,
	1.425960214968693000f, 1.425691287960236600f, 1.425422308533187200f,
	1.425153276697434000f,
	1.424884192462868800f, 1.424615055839385300f, 1.424345866836878200f,
	1.424076625465245500f,
	1.423807331734385800f, 1.423537985654200800f, 1.423268587234593400f,
	1.422999136485468600f,
	1.422729633416733200f, 1.422460078038296300f, 1.422190470360068300f,
	1.421920810391962500f,
	1.421651098143893000f, 1.421381333625776600f, 1.421111516847531700f,
	1.420841647819078600f,
	1.420571726550339700f, 1.420301753051239400f, 1.420031727331703800f,
	1.419761649401660500f,
	1.419491519271040000f, 1.419221336949774100f, 1.418951102447796800f,
	1.418680815775043500f,
	1.418410476941452100f, 1.418140085956961900f, 1.417869642831514700f,
	1.417599147575054000f,
	1.417328600197524900f, 1.417058000708874700f, 1.416787349119052600f,
	1.416516645438009600f,
	1.416245889675698900f, 1.415975081842075300f, 1.415704221947095700f,
	1.415433310000718600f,
	1.415162346012905000f, 1.414891329993617200f, 1.414620261952819600f,
	1.414349141900479000f,
	1.414077969846563500f, 1.413806745801043500f, 1.413535469773890700f,
	1.413264141775079300f,
	1.412992761814585400f, 1.412721329902386900f, 1.412449846048463600f,
	1.412178310262796900f,
	1.411906722555370500f, 1.411635082936170100f, 1.411363391415182900f,
	1.411091648002398500f,
	1.410819852707807700f, 1.410548005541404100f, 1.410276106513182400f,
	1.410004155633139500f,
	1.409732152911274500f, 1.409460098357588200f, 1.409187991982083100f,
	1.408915833794763800f,
	1.408643623805636800f, 1.408371362024710500f, 1.408099048461995300f,
	1.407826683127503000f,
	1.407554266031248100f, 1.407281797183246500f, 1.407009276593515800f,
	1.406736704272076400f,
	1.406464080228949600f, 1.406191404474159000f, 1.405918677017730100f,
	1.405645897869690400f,
	1.405373067040069300f, 1.405100184538898000f, 1.404827250376209400f,
	1.404554264562038400f,
	1.404281227106422400f, 1.404008138019399800f, 1.403734997311011600f,
	1.403461804991300100f,
	1.403188561070310100f, 1.402915265558087700f, 1.402641918464681400f,
	1.402368519800141200f,
	1.402095069574519800f, 1.401821567797870300f, 1.401548014480249000f,
	1.401274409631713600f,
	1.401000753262323900f, 1.400727045382141400f, 1.400453286001229800f,
	1.400179475129653700f,
	1.399905612777481200f, 1.399631698954780800f, 1.399357733671623900f,
	1.399083716938083600f,
	1.398809648764234100f, 1.398535529160152400f, 1.398261358135917300f,
	1.397987135701609200f,
	1.397712861867310300f, 1.397438536643105000f, 1.397164160039079200f,
	1.396889732065321300f,
	1.396615252731921100f, 1.396340722048970300f, 1.396066140026562800f,
	1.395791506674794100f,
	1.395516822003761700f, 1.395242086023564800f, 1.394967298744304900f,
	1.394692460176085300f,
	1.394417570329010700f, 1.394142629213188000f, 1.393867636838725900f,
	1.393592593215735600f,
	1.393317498354329300f, 1.393042352264621600f, 1.392767154956728400f,
	1.392491906440768600f,
	1.392216606726861800f, 1.391941255825130100f, 1.391665853745697400f,
	1.391390400498689700f,
	1.391114896094234100f, 1.390839340542460600f, 1.390563733853500200f,
	1.390288076037486500f,
	1.390012367104554600f, 1.389736607064841100f, 1.389460795928485500f,
	1.389184933705628300f,
	1.388909020406412100f, 1.388633056040981600f, 1.388357040619483200f,
	1.388080974152065200f,
	1.387804856648877600f, 1.387528688120072600f, 1.387252468575804100f,
	1.386976198026228100f,
	1.386699876481501900f, 1.386423503951785200f, 1.386147080447239600f,
	1.385870605978028100f,
	1.385594080554316100f, 1.385317504186270900f, 1.385040876884061000f,
	1.384764198657857200f,
	1.384487469517832200f, 1.384210689474160600f, 1.383933858537019100f,
	1.383656976716585600f,
	1.383380044023040400f, 1.383103060466565300f, 1.382826026057344600f,
	1.382548940805563800f,
	1.382271804721410600f, 1.381994617815074400f, 1.381717380096746800f,
	1.381440091576620700f,
	1.381162752264891500f, 1.380885362171756300f, 1.380607921307413400f,
	1.380330429682064000f,
	1.380052887305910400f, 1.379775294189157000f, 1.379497650342010400f,
	1.379219955774678700f,
	1.378942210497371600f, 1.378664414520301500f, 1.378386567853681700f,
	1.378108670507728300f,
	1.377830722492658500f, 1.377552723818691500f, 1.377274674496048700f,
	1.376996574534953300f,
	1.376718423945630000f, 1.376440222738305700f, 1.376161970923209400f,
	1.375883668510570900f,
	1.375605315510623200f, 1.375326911933600200f, 1.375048457789738400f,
	1.374769953089275400f,
	1.374491397842451100f, 1.374212792059507100f, 1.373934135750687100f,
	1.373655428926236400f,
	1.373376671596402400f, 1.373097863771434200f, 1.372819005461582500f,
	1.372540096677100200f,
	1.372261137428242300f, 1.371982127725264800f, 1.371703067578426700f,
	1.371423956997988000f,
	1.371144795994210500f, 1.370865584577358300f, 1.370586322757697500f,
	1.370307010545495500f,
	1.370027647951022100f, 1.369748234984548000f, 1.369468771656347200f,
	1.369189257976694200f,
	1.368909693955866000f, 1.368630079604142000f, 1.368350414931802000f,
	1.368070699949128800f,
	1.367790934666406600f, 1.367511119093921800f, 1.367231253241962200f,
	1.366951337120818000f,
	1.366671370740780500f, 1.366391354112143500f, 1.366111287245202400f,
	1.365831170150254300f,
	1.365551002837598600f, 1.365270785317536100f, 1.364990517600369400f,
	1.364710199696403300f,
	1.364429831615944200f, 1.364149413369300600f, 1.363868944966782900f,
	1.363588426418702600f,
	1.363307857735373900f, 1.363027238927112300f, 1.362746570004235400f,
	1.362465850977062900f,
	1.362185081855915600f, 1.361904262651116900f, 1.361623393372991300f,
	1.361342474031866000f,
	1.361061504638069400f, 1.360780485201932300f, 1.360499415733786400f,
	1.360218296243966200f,
	1.359937126742807300f, 1.359655907240648000f, 1.359374637747827700f,
	1.359093318274687800f,
	1.358811948831571500f, 1.358530529428824400f, 1.358249060076792900f,
	1.357967540785826300f,
	1.357685971566275200f, 1.357404352428492000f, 1.357122683382830900f,
	1.356840964439648200f,
	1.356559195609301700f, 1.356277376902151900f, 1.355995508328559500f,
	1.355713589898888800f,
	1.355431621623504700f, 1.355149603512774400f, 1.354867535577067200f,
	1.354585417826753800f,
	1.354303250272206500f, 1.354021032923800300f, 1.353738765791911100f,
	1.353456448886917200f,
	1.353174082219199100f, 1.352891665799137900f, 1.352609199637117500f,
	1.352326683743523300f,
	1.352044118128742600f, 1.351761502803164900f, 1.351478837777180700f,
	1.351196123061183100f,
	1.350913358665566400f, 1.350630544600727200f, 1.350347680877063800f,
	1.350064767504976400f,
	1.349781804494866600f, 1.349498791857138400f, 1.349215729602197400f,
	1.348932617740450600f,
	1.348649456282307700f, 1.348366245238179500f, 1.348082984618478800f,
	1.347799674433620500f,
	1.347516314694020800f, 1.347232905410098200f, 1.346949446592273100f,
	1.346665938250967100f,
	1.346382380396604000f, 1.346098773039609700f, 1.345815116190411300f,
	1.345531409859438200f,
	1.345247654057121700f, 1.344963848793894200f, 1.344679994080190800f,
	1.344396089926448000f,
	1.344112136343103900f, 1.343828133340598800f, 1.343544080929374800f,
	1.343259979119875600f,
	1.342975827922546600f, 1.342691627347835500f, 1.342407377406191500f,
	1.342123078108065700f,
	1.341838729463910900f, 1.341554331484181600f, 1.341269884179334700f,
	1.340985387559828100f,
	1.340700841636122400f, 1.340416246418678800f, 1.340131601917961900f,
	1.339846908144436600f,
	1.339562165108570700f, 1.339277372820833400f, 1.338992531291695500f,
	1.338707640531629800f,
	1.338422700551110900f, 1.338137711360615200f, 1.337852672970621300f,
	1.337567585391608900f,
	1.337282448634059800f, 1.336997262708457900f, 1.336712027625288600f,
	1.336426743395039000f,
	1.336141410028198500f, 1.335856027535258000f, 1.335570595926709700f,
	1.335285115213048500f,
	1.334999585404770700f, 1.334714006512374400f, 1.334428378546359500f,
	1.334142701517227600f,
	1.333856975435482300f, 1.333571200311629100f, 1.333285376156174700f,
	1.332999502979628700f,
	1.332713580792501500f, 1.332427609605305400f, 1.332141589428554900f,
	1.331855520272766200f,
	1.331569402148457400f, 1.331283235066148100f, 1.330997019036359800f,
	1.330710754069615700f,
	1.330424440176441300f, 1.330138077367363200f, 1.329851665652910500f,
	1.329565205043613800f,
	1.329278695550004700f, 1.328992137182618100f, 1.328705529951989400f,
	1.328418873868656900f,
	1.328132168943159800f, 1.327845415186039000f, 1.327558612607838500f,
	1.327271761219102500f,
	1.326984861030378000f, 1.326697912052213500f, 1.326410914295159400f,
	1.326123867769767500f,
	1.325836772486591800f, 1.325549628456188100f, 1.325262435689113600f,
	1.324975194195928000f,
	1.324687903987191900f, 1.324400565073468300f, 1.324113177465321900f,
	1.323825741173318700f,
	1.323538256208027800f, 1.323250722580018500f, 1.322963140299862500f,
	1.322675509378133900f,
	1.322387829825407700f, 1.322100101652261100f, 1.321812324869273500f,
	1.321524499487024800f,
	1.321236625516098100f, 1.320948702967077400f, 1.320660731850549000f,
	1.320372712177100700f,
	1.320084643957322400f, 1.319796527201805300f, 1.319508361921142500f,
	1.319220148125929100f,
	1.318931885826762000f, 1.318643575034239800f, 1.318355215758962900f,
	1.318066808011533200f,
	1.317778351802554800f, 1.317489847142633300f, 1.317201294042376300f,
	1.316912692512393300f,
	1.316624042563294900f, 1.316335344205694200f, 1.316046597450205800f,
	1.315757802307445900f,
	1.315468958788033000f, 1.315180066902586800f, 1.314891126661728900f,
	1.314602138076083300f,
	1.314313101156274800f, 1.314024015912930600f, 1.313734882356679900f,
	1.313445700498152800f,
	1.313156470347981900f, 1.312867191916801100f, 1.312577865215246900f,
	1.312288490253956900f,
	1.311999067043570200f, 1.311709595594728000f, 1.311420075918073900f,
	1.311130508024252400f,
	1.310840891923910100f, 1.310551227627695400f, 1.310261515146258200f,
	1.309971754490250700f,
	1.309681945670326400f, 1.309392088697140900f, 1.309102183581351200f,
	1.308812230333616500f,
	1.308522228964597500f, 1.308232179484956500f, 1.307942081905358000f,
	1.307651936236467800f,
	1.307361742488954300f, 1.307071500673486800f, 1.306781210800736200f,
	1.306490872881376200f,
	1.306200486926081700f, 1.305910052945529200f, 1.305619570950396800f,
	1.305329040951365100f,
	1.305038462959116100f, 1.304747836984333300f, 1.304457163037702200f,
	1.304166441129910300f,
	1.303875671271646400f, 1.303584853473601200f, 1.303293987746467300f,
	1.303003074100939100f,
	1.302712112547712800f, 1.302421103097485900f, 1.302130045760958100f,
	1.301838940548830600f,
	1.301547787471806900f, 1.301256586540591600f, 1.300965337765891600f,
	1.300674041158414800f,
	1.300382696728871400f, 1.300091304487973800f, 1.299799864446435200f,
	1.299508376614971500f,
	1.299216841004299200f, 1.298925257625137800f, 1.298633626488207500f,
	1.298341947604231300f,
	1.298050220983932900f, 1.297758446638038700f, 1.297466624577275900f,
	1.297174754812374400f,
	1.296882837354065100f, 1.296590872213081200f, 1.296298859400157700f,
	1.296006798926030200f,
	1.295714690801437600f, 1.295422535037119800f, 1.295130331643818500f,
	1.294838080632277000f,
	1.294545782013240900f, 1.294253435797456900f, 1.293961041995673700f,
	1.293668600618642000f,
	1.293376111677113900f, 1.293083575181843500f, 1.292790991143586200f,
	1.292498359573099700f,
	1.292205680481143500f, 1.291912953878477900f, 1.291620179775866400f,
	1.291327358184073200f,
	1.291034489113864100f, 1.290741572576007400f, 1.290448608581273000f,
	1.290155597140431700f,
	1.289862538264257700f, 1.289569431963524900f, 1.289276278249010600f,
	1.288983077131493000f,
	1.288689828621752300f, 1.288396532730570400f, 1.288103189468731400f,
	1.287809798847019800f,
	1.287516360876223500f, 1.287222875567130900f, 1.286929342930532800f,
	1.286635762977221800f,
	1.286342135717991600f, 1.286048461163638000f, 1.285754739324958900f,
	1.285460970212753500f,
	1.285167153837822900f, 1.284873290210969900f, 1.284579379342998700f,
	1.284285421244715900f,
	1.283991415926929400f, 1.283697363400448900f, 1.283403263676086100f,
	1.283109116764654000f,
	1.282814922676967400f, 1.282520681423843000f, 1.282226393016099500f,
	1.281932057464557000f,
	1.281637674780037100f, 1.281343244973363700f, 1.281048768055361900f,
	1.280754244036858900f,
	1.280459672928683500f, 1.280165054741666300f, 1.279870389486639400f,
	1.279575677174437100f,
	1.279280917815894600f, 1.278986111421849900f, 1.278691258003142000f,
	1.278396357570611900f,
	1.278101410135101800f, 1.277806415707456700f, 1.277511374298522200f,
	1.277216285919146500f,
	1.276921150580179200f, 1.276625968292471000f, 1.276330739066875400f,
	1.276035462914247000f,
	1.275740139845442400f, 1.275444769871319600f, 1.275149353002738700f,
	1.274853889250561200f,
	1.274558378625650200f, 1.274262821138871300f, 1.273967216801090900f,
	1.273671565623178100f,
	1.273375867616002300f, 1.273080122790436000f, 1.272784331157352800f,
	1.272488492727628100f,
	1.272192607512139300f, 1.271896675521764900f, 1.271600696767385400f,
	1.271304671259883200f,
	1.271008599010142500f, 1.270712480029048800f, 1.270416314327489800f,
	1.270120101916354600f,
	1.269823842806533800f, 1.269527537008920300f, 1.269231184534408200f,
	1.268934785393893700f,
	1.268638339598274500f, 1.268341847158450200f, 1.268045308085321800f,
	1.267748722389792100f,
	1.267452090082765900f, 1.267155411175149500f, 1.266858685677851000f,
	1.266561913601780100f,
	1.266265094957848000f, 1.265968229756968100f, 1.265671318010055400f,
	1.265374359728026500f,
	1.265077354921799300f, 1.264780303602294200f, 1.264483205780432700f,
	1.264186061467138500f,
	1.263888870673336400f, 1.263591633409954000f, 1.263294349687918800f,
	1.262997019518161700f,
	1.262699642911614600f, 1.262402219879211300f, 1.262104750431887000f,
	1.261807234580578900f,
	1.261509672336225600f, 1.261212063709767900f, 1.260914408712147800f,
	1.260616707354309500f,
	1.260318959647198400f, 1.260021165601761900f, 1.259723325228949000f,
	1.259425438539710300f,
	1.259127505544998600f, 1.258829526255768000f, 1.258531500682973800f,
	1.258233428837574300f,
	1.257935310730528000f, 1.257637146372796400f, 1.257338935775342200f,
	1.257040678949129500f,
	1.256742375905124400f, 1.256444026654294400f, 1.256145631207609400f,
	1.255847189576040100f,
	1.255548701770560000f, 1.255250167802143000f, 1.254951587681765600f,
	1.254652961420405600f,
	1.254354289029042900f, 1.254055570518658500f, 1.253756805900235700f,
	1.253457995184759300f,
	1.253159138383215200f, 1.252860235506592100f, 1.252561286565879300f,
	1.252262291572068900f,
	1.251963250536153500f, 1.251664163469128300f, 1.251365030381989700f,
	1.251065851285736200f,
	1.250766626191367500f, 1.250467355109885500f, 1.250168038052293500f,
	1.249868675029596200f,
	1.249569266052800800f, 1.249269811132915200f, 1.248970310280950200f,
	1.248670763507917100f,
	1.248371170824829300f, 1.248071532242702100f, 1.247771847772552300f,
	1.247472117425398700f,
	1.247172341212261500f, 1.246872519144162300f, 1.246572651232124700f,
	1.246272737487174300f,
	1.245972777920338000f, 1.245672772542644400f, 1.245372721365123600f,
	1.245072624398807900f,
	1.244772481654731000f, 1.244472293143928300f, 1.244172058877436800f,
	1.243871778866295400f,
	1.243571453121544000f, 1.243271081654225400f, 1.242970664475383100f,
	1.242670201596062700f,
	1.242369693027311200f, 1.242069138780177400f, 1.241768538865712000f,
	1.241467893294967200f,
	1.241167202078996800f, 1.240866465228856100f, 1.240565682755603100f,
	1.240264854670295900f,
	1.239963980983995300f, 1.239663061707763700f, 1.239362096852665300f,
	1.239061086429765300f,
	1.238760030450130900f, 1.238458928924831600f, 1.238157781864937400f,
	1.237856589281521000f,
	1.237555351185656500f, 1.237254067588419400f, 1.236952738500886900f,
	1.236651363934138300f,
	1.236349943899254000f, 1.236048478407316500f, 1.235746967469409900f,
	1.235445411096619500f,
	1.235143809300033300f, 1.234842162090739700f, 1.234540469479829900f,
	1.234238731478396000f,
	1.233936948097532400f, 1.233635119348334400f, 1.233333245241899200f,
	1.233031325789326400f,
	1.232729361001716500f, 1.232427350890172000f, 1.232125295465796600f,
	1.231823194739696300f,
	1.231521048722978200f, 1.231218857426751700f, 1.230916620862127400f,
	1.230614339040217800f,
	1.230312011972136500f, 1.230009639668999500f, 1.229707222141924100f,
	1.229404759402029400f,
	1.229102251460436400f, 1.228799698328266700f, 1.228497100016644900f,
	1.228194456536696500f,
	1.227891767899548700f, 1.227589034116330700f, 1.227286255198173100f,
	1.226983431156208200f,
	1.226680562001569900f, 1.226377647745394000f, 1.226074688398817600f,
	1.225771683972980200f,
	1.225468634479021500f, 1.225165539928084300f, 1.224862400331312400f,
	1.224559215699851500f,
	1.224255986044848500f, 1.223952711377453100f, 1.223649391708814700f,
	1.223346027050086400f,
	1.223042617412421600f, 1.222739162806975900f, 1.222435663244906700f,
	1.222132118737372400f,
	1.221828529295533800f, 1.221524894930552800f, 1.221221215653593100f,
	1.220917491475820500f,
	1.220613722408401900f, 1.220309908462505800f, 1.220006049649302800f,
	1.219702145979964600f,
	1.219398197465665400f, 1.219094204117580300f, 1.218790165946886100f,
	1.218486082964761500f,
	1.218181955182386500f, 1.217877782610943700f, 1.217573565261616000f,
	1.217269303145589000f,
	1.216964996274049400f, 1.216660644658185600f, 1.216356248309187600f,
	1.216051807238247800f,
	1.215747321456559300f, 1.215442790975316700f, 1.215138215805717300f,
	1.214833595958959300f,
	1.214528931446242600f, 1.214224222278769100f, 1.213919468467741900f,
	1.213614670024366000f,
	1.213309826959847700f, 1.213004939285395400f, 1.212700007012219100f,
	1.212395030151530300f,
	1.212090008714541600f, 1.211784942712468300f, 1.211479832156526800f,
	1.211174677057934800f,
	1.210869477427912300f, 1.210564233277680500f, 1.210258944618462200f,
	1.209953611461482200f,
	1.209648233817966600f, 1.209342811699143600f, 1.209037345116242400f,
	1.208731834080493800f,
	1.208426278603131200f, 1.208120678695388600f, 1.207815034368502100f,
	1.207509345633709600f,
	1.207203612502250300f, 1.206897834985365000f, 1.206592013094296200f,
	1.206286146840288300f,
	1.205980236234587100f, 1.205674281288440000f, 1.205368282013096200f,
	1.205062238419806200f,
	1.204756150519822300f, 1.204450018324398900f, 1.204143841844791200f,
	1.203837621092256800f,
	1.203531356078054100f, 1.203225046813444000f, 1.202918693309688300f,
	1.202612295578050900f,
	1.202305853629797500f, 1.201999367476194400f, 1.201692837128510700f,
	1.201386262598016500f,
	1.201079643895983700f, 1.200772981033685800f, 1.200466274022397900f,
	1.200159522873396800f,
	1.199852727597960700f, 1.199545888207369700f, 1.199239004712905300f,
	1.198932077125851100f,
	1.198625105457491700f, 1.198318089719113200f, 1.198011029922004400f,
	1.197703926077454200f,
	1.197396778196754700f, 1.197089586291198500f, 1.196782350372080300f,
	1.196475070450696100f,
	1.196167746538343600f, 1.195860378646322700f, 1.195552966785933900f,
	1.195245510968480300f,
	1.194938011205265900f, 1.194630467507596500f, 1.194322879886780000f,
	1.194015248354125100f,
	1.193707572920943000f, 1.193399853598545500f, 1.193092090398246900f,
	1.192784283331362700f,
	1.192476432409210100f, 1.192168537643107900f, 1.191860599044376500f,
	1.191552616624337800f,
	1.191244590394315400f, 1.190936520365635000f, 1.190628406549622900f,
	1.190320248957608100f,
	1.190012047600920200f, 1.189703802490891000f, 1.189395513638853900f,
	1.189087181056143900f,
	1.188778804754097300f, 1.188470384744052100f, 1.188161921037348400f,
	1.187853413645327100f,
	1.187544862579331500f, 1.187236267850706000f, 1.186927629470796900f,
	1.186618947450951600f,
	1.186310221802519900f, 1.186001452536852300f, 1.185692639665301600f,
	1.185383783199222000f,
	1.185074883149969100f, 1.184765939528900500f, 1.184456952347374900f,
	1.184147921616753200f,
	1.183838847348397400f, 1.183529729553671500f, 1.183220568243940300f,
	1.182911363430571200f,
	1.182602115124932900f, 1.182292823338395100f, 1.181983488082330300f,
	1.181674109368111300f,
	1.181364687207113100f, 1.181055221610712400f, 1.180745712590287400f,
	1.180436160157217800f,
	1.180126564322885100f, 1.179816925098671900f, 1.179507242495962900f,
	1.179197516526144600f,
	1.178887747200604300f, 1.178577934530731700f, 1.178268078527917200f,
	1.177958179203553800f,
	1.177648236569035300f, 1.177338250635757700f, 1.177028221415118200f,
	1.176718148918515700f,
	1.176408033157350300f, 1.176097874143024600f, 1.175787671886942000f,
	1.175477426400507700f,
	1.175167137695128900f, 1.174856805782213500f, 1.174546430673171900f,
	1.174236012379415600f,
	1.173925550912357800f, 1.173615046283413200f, 1.173304498503998400f,
	1.172993907585530900f,
	1.172683273539430800f, 1.172372596377118800f, 1.172061876110017700f,
	1.171751112749551900f,
	1.171440306307147200f, 1.171129456794231200f, 1.170818564222232800f,
	1.170507628602582800f,
	1.170196649946713100f, 1.169885628266057900f, 1.169574563572052300f,
	1.169263455876133200f,
	1.168952305189739200f, 1.168641111524310700f, 1.168329874891289400f,
	1.168018595302118000f,
	1.167707272768241800f, 1.167395907301107100f, 1.167084498912162300f,
	1.166773047612856400f,
	1.166461553414641000f, 1.166150016328968600f, 1.165838436367293800f,
	1.165526813541072100f,
	1.165215147861761400f, 1.164903439340820900f, 1.164591687989710500f,
	1.164279893819892800f,
	1.163968056842831700f, 1.163656177069992500f, 1.163344254512841800f,
	1.163032289182848800f,
	1.162720281091483000f, 1.162408230250216100f, 1.162096136670521600f,
	1.161784000363874000f,
	1.161471821341749900f, 1.161159599615627000f, 1.160847335196984800f,
	1.160535028097304600f,
	1.160222678328068700f, 1.159910285900761700f, 1.159597850826869200f,
	1.159285373117878500f,
	1.158972852785278500f, 1.158660289840559800f, 1.158347684295214300f,
	1.158035036160735900f,
	1.157722345448619400f, 1.157409612170361600f, 1.157096836337461000f,
	1.156784017961417500f,
	1.156471157053732300f, 1.156158253625908700f, 1.155845307689450800f,
	1.155532319255865300f,
	1.155219288336659400f, 1.154906214943342700f, 1.154593099087426000f,
	1.154279940780421400f,
	1.153966740033842900f, 1.153653496859206000f, 1.153340211268028000f,
	1.153026883271827300f,
	1.152713512882124400f, 1.152400100110440700f, 1.152086644968299400f,
	1.151773147467225300f,
	1.151459607618745300f, 1.151146025434387000f, 1.150832400925680100f,
	1.150518734104155400f,
	1.150205024981345800f, 1.149891273568785400f, 1.149577479878009800f,
	1.149263643920556800f,
	1.148949765707964600f, 1.148635845251773800f, 1.148321882563526400f,
	1.148007877654766200f,
	1.147693830537038100f, 1.147379741221888500f, 1.147065609720865600f,
	1.146751436045519300f,
	1.146437220207400700f, 1.146122962218062600f, 1.145808662089060000f,
	1.145494319831947800f,
	1.145179935458284100f, 1.144865508979627800f, 1.144551040407539400f,
	1.144236529753581000f,
	1.143921977029316500f, 1.143607382246310600f, 1.143292745416130600f,
	1.142978066550344400f,
	1.142663345660522000f, 1.142348582758234900f, 1.142033777855056000f,
	1.141718930962559500f,
	1.141404042092321500f, 1.141089111255919800f, 1.140774138464933700f,
	1.140459123730943200f,
	1.140144067065530700f, 1.139828968480280300f, 1.139513827986776900f,
	1.139198645596607400f,
	1.138883421321360600f, 1.138568155172625700f, 1.138252847161994400f,
	1.137937497301059600f,
	1.137622105601416000f, 1.137306672074659900f, 1.136991196732388200f,
	1.136675679586200500f,
	1.136360120647697200f, 1.136044519928480800f, 1.135728877440154800f,
	1.135413193194324800f,
	1.135097467202597100f, 1.134781699476580300f, 1.134465890027884300f,
	1.134150038868120500f,
	1.133834146008902100f, 1.133518211461843200f, 1.133202235238559800f,
	1.132886217350669500f,
	1.132570157809791500f, 1.132254056627546300f, 1.131937913815556300f,
	1.131621729385444900f,
	1.131305503348837300f, 1.130989235717360100f, 1.130672926502642100f,
	1.130356575716312500f,
	1.130040183370002900f, 1.129723749475346000f, 1.129407274043976200f,
	1.129090757087529500f,
	1.128774198617643200f, 1.128457598645956600f, 1.128140957184109700f,
	1.127824274243744500f,
	1.127507549836505000f, 1.127190783974035800f, 1.126873976667983800f,
	1.126557127929996800f,
	1.126240237771724700f, 1.125923306204818400f, 1.125606333240930700f,
	1.125289318891715900f,
	1.124972263168829500f, 1.124655166083928800f, 1.124338027648672500f,
	1.124020847874721100f,
	1.123703626773736100f, 1.123386364357381200f, 1.123069060637320600f,
	1.122751715625221400f,
	1.122434329332750800f, 1.122116901771578400f, 1.121799432953375600f,
	1.121481922889814300f,
	1.121164371592568300f, 1.120846779073313400f, 1.120529145343726500f,
	1.120211470415486200f,
	1.119893754300272300f, 1.119575997009766300f, 1.119258198555651300f,
	1.118940358949611900f,
	1.118622478203333800f, 1.118304556328505200f, 1.117986593336814700f,
	1.117668589239953200f,
	1.117350544049612300f, 1.117032457777486200f, 1.116714330435269600f,
	1.116396162034659600f,
	1.116077952587353600f, 1.115759702105052000f, 1.115441410599455500f,
	1.115123078082267000f,
	1.114804704565190500f, 1.114486290059931900f, 1.114167834578198200f,
	1.113849338131698300f,
	1.113530800732142100f, 1.113212222391241500f, 1.112893603120710000f,
	1.112574942932261600f,
	1.112256241837613000f, 1.111937499848481900f, 1.111618716976587700f,
	1.111299893233650600f,
	1.110981028631393700f, 1.110662123181539900f, 1.110343176895814500f,
	1.110024189785944900f,
	1.109705161863658600f, 1.109386093140686000f, 1.109066983628758100f,
	1.108747833339607200f,
	1.108428642284968100f, 1.108109410476576300f, 1.107790137926169200f,
	1.107470824645485600f,
	1.107151470646265300f, 1.106832075940250600f, 1.106512640539184100f,
	1.106193164454811100f,
	1.105873647698877300f, 1.105554090283131100f, 1.105234492219321100f,
	1.104914853519198400f,
	1.104595174194514800f, 1.104275454257024300f, 1.103955693718482200f,
	1.103635892590644900f,
	1.103316050885270600f, 1.102996168614119000f, 1.102676245788951400f,
	1.102356282421530300f,
	1.102036278523620000f, 1.101716234106985700f, 1.101396149183395000f,
	1.101076023764616400f,
	1.100755857862419700f, 1.100435651488577100f, 1.100115404654861100f,
	1.099795117373046200f,
	1.099474789654909100f, 1.099154421512226600f, 1.098834012956778200f,
	1.098513564000344300f,
	1.098193074654706800f, 1.097872544931649100f, 1.097551974842956500f,
	1.097231364400415000f,
	1.096910713615813200f, 1.096590022500939700f, 1.096269291067585700f,
	1.095948519327543800f,
	1.095627707292607700f, 1.095306854974572800f, 1.094985962385235800f,
	1.094665029536395100f,
	1.094344056439850600f, 1.094023043107403200f, 1.093701989550856000f,
	1.093380895782013000f,
	1.093059761812680100f, 1.092738587654664300f, 1.092417373319774200f,
	1.092096118819820200f,
	1.091774824166613600f, 1.091453489371968100f, 1.091132114447697300f,
	1.090810699405617900f,
	1.090489244257547300f, 1.090167749015304300f, 1.089846213690709900f,
	1.089524638295585400f,
	1.089203022841754400f, 1.088881367341041800f, 1.088559671805274100f,
	1.088237936246279100f,
	1.087916160675885800f, 1.087594345105925300f, 1.087272489548229700f,
	1.086950594014632700f,
	1.086628658516969500f, 1.086306683067076900f, 1.085984667676792600f,
	1.085662612357956500f,
	1.085340517122409800f, 1.085018381981994500f, 1.084696206948555300f,
	1.084373992033937000f,
	1.084051737249986900f, 1.083729442608553300f, 1.083407108121486000f,
	1.083084733800636200f,
	1.082762319657857100f, 1.082439865705002500f, 1.082117371953928300f,
	1.081794838416491700f,
	1.081472265104551200f, 1.081149652029967000f, 1.080826999204601100f,
	1.080504306640315500f,
	1.080181574348975500f, 1.079858802342446900f, 1.079535990632596800f,
	1.079213139231294500f,
	1.078890248150409700f, 1.078567317401815100f, 1.078244346997383300f,
	1.077921336948988600f,
	1.077598287268508400f, 1.077275197967819000f, 1.076952069058800400f,
	1.076628900553332700f,
	1.076305692463297900f, 1.075982444800579700f, 1.075659157577062200f,
	1.075335830804633000f,
	1.075012464495178800f, 1.074689058660589700f, 1.074365613312755900f,
	1.074042128463569500f,
	1.073718604124924500f, 1.073395040308715400f, 1.073071437026839500f,
	1.072747794291194300f,
	1.072424112113678600f, 1.072100390506194500f, 1.071776629480643500f,
	1.071452829048929800f,
	1.071128989222958500f, 1.070805110014635900f, 1.070481191435870500f,
	1.070157233498571600f,
	1.069833236214650800f, 1.069509199596019800f, 1.069185123654592600f,
	1.068861008402285200f,
	1.068536853851013600f, 1.068212660012696700f, 1.067888426899253500f,
	1.067564154522606000f,
	1.067239842894676100f, 1.066915492027387600f, 1.066591101932666800f,
	1.066266672622439700f,
	1.065942204108635300f, 1.065617696403183400f, 1.065293149518014500f,
	1.064968563465062100f,
	1.064643938256259400f, 1.064319273903543000f, 1.063994570418849400f,
	1.063669827814116300f,
	1.063345046101285000f, 1.063020225292295300f, 1.062695365399091200f,
	1.062370466433616400f,
	1.062045528407815900f, 1.061720551333637600f, 1.061395535223029500f,
	1.061070480087941800f,
	1.060745385940325500f, 1.060420252792134000f, 1.060095080655320900f,
	1.059769869541841800f,
	1.059444619463654400f, 1.059119330432716700f, 1.058794002460989000f,
	1.058468635560432500f,
	1.058143229743009600f, 1.057817785020685100f, 1.057492301405424500f,
	1.057166778909195000f,
	1.056841217543965200f, 1.056515617321704500f, 1.056189978254385100f,
	1.055864300353978900f,
	1.055538583632461100f, 1.055212828101807200f, 1.054887033773993300f,
	1.054561200660999200f,
	1.054235328774803900f, 1.053909418127389400f, 1.053583468730738200f,
	1.053257480596834700f,
	1.052931453737664600f, 1.052605388165214700f, 1.052279283891473600f,
	1.051953140928431100f,
	1.051626959288079100f, 1.051300738982409800f, 1.050974480023417500f,
	1.050648182423098000f,
	1.050321846193448000f, 1.049995471346466300f, 1.049669057894152800f,
	1.049342605848508200f,
	1.049016115221536000f, 1.048689586025239700f, 1.048363018271625300f,
	1.048036411972699500f,
	1.047709767140470500f, 1.047383083786948700f, 1.047056361924144400f,
	1.046729601564071200f,
	1.046402802718742400f, 1.046075965400174300f, 1.045749089620383200f,
	1.045422175391386800f,
	1.045095222725206200f, 1.044768231633861100f, 1.044441202129375200f,
	1.044114134223771900f,
	1.043787027929076000f, 1.043459883257315400f, 1.043132700220517300f,
	1.042805478830712200f,
	1.042478219099930400f, 1.042150921040204200f, 1.041823584663568200f,
	1.041496209982056600f,
	1.041168797007707000f, 1.040841345752557200f, 1.040513856228645800f,
	1.040186328448014800f,
	1.039858762422705600f, 1.039531158164762400f, 1.039203515686230000f,
	1.038875834999155100f,
	1.038548116115585800f, 1.038220359047570500f, 1.037892563807160800f,
	1.037564730406408200f,
	1.037236858857366600f, 1.036908949172090900f, 1.036581001362636600f,
	1.036253015441062700f,
	1.035924991419427100f, 1.035596929309791300f, 1.035268829124216700f,
	1.034940690874766300f,
	1.034612514573505700f, 1.034284300232500000f, 1.033956047863817500f,
	1.033627757479526700f,
	1.033299429091697700f, 1.032971062712402700f, 1.032642658353714300f,
	1.032314216027707700f,
	1.031985735746457900f, 1.031657217522042900f, 1.031328661366541300f,
	1.031000067292032300f,
	1.030671435310598600f, 1.030342765434322200f, 1.030014057675287900f,
	1.029685312045581100f,
	1.029356528557288300f, 1.029027707222499100f, 1.028698848053302100f,
	1.028369951061789600f,
	1.028041016260053500f, 1.027712043660187600f, 1.027383033274288400f,
	1.027053985114451100f,
	1.026724899192775300f, 1.026395775521359500f, 1.026066614112305600f,
	1.025737414977715200f,
	1.025408178129692000f, 1.025078903580341600f, 1.024749591341769700f,
	1.024420241426085200f,
	1.024090853845396800f, 1.023761428611814600f, 1.023431965737451800f,
	1.023102465234420700f,
	1.022772927114837100f, 1.022443351390816400f, 1.022113738074476300f,
	1.021784087177936000f,
	1.021454398713315600f, 1.021124672692737000f, 1.020794909128323000f,
	1.020465108032198300f,
	1.020135269416488700f, 1.019805393293321100f, 1.019475479674824900f,
	1.019145528573129000f,
	1.018815540000365800f, 1.018485513968667500f, 1.018155450490168000f,
	1.017825349577003300f,
	1.017495211241309800f, 1.017165035495226400f, 1.016834822350892300f,
	1.016504571820448000f,
	1.016174283916036800f, 1.015843958649801600f, 1.015513596033888400f,
	1.015183196080442900f,
	1.014852758801613200f, 1.014522284209548900f, 1.014191772316400000f,
	1.013861223134318900f,
	1.013530636675459100f, 1.013200012951974700f, 1.012869351976022300f,
	1.012538653759758900f,
	1.012207918315344300f, 1.011877145654937400f, 1.011546335790700600f,
	1.011215488734796800f,
	1.010884604499389800f, 1.010553683096645900f, 1.010222724538731600f,
	1.009891728837815700f,
	1.009560696006067900f, 1.009229626055658800f, 1.008898518998761800f,
	1.008567374847549900f,
	1.008236193614199000f, 1.007904975310885300f, 1.007573719949786700f,
	1.007242427543082900f,
	1.006911098102953900f, 1.006579731641582500f, 1.006248328171152100f,
	1.005916887703846500f,
	1.005585410251852700f, 1.005253895827357800f, 1.004922344442551000f,
	1.004590756109621900f,
	1.004259130840762700f, 1.003927468648166100f, 1.003595769544025900f,
	1.003264033540538500f,
	1.002932260649900000f, 1.002600450884309800f, 1.002268604255967200f,
	1.001936720777072400f,
	1.001604800459829000f, 1.001272843316440000f, 1.000940849359111000f,
	1.000608818600048100f,
	1.000276751051459200f, 0.999944646725553720f, 0.999612505634541740f,
	0.999280327790635690f,
	0.998948113206048590f, 0.998615861892994560f, 0.998283573863690270f,
	0.997951249130352380f,
	0.997618887705200020f, 0.997286489600452630f, 0.996954054828332210f,
	0.996621583401061110f,
	0.996289075330862860f, 0.995956530629963810f, 0.995623949310589620f,
	0.995291331384969390f,
	0.994958676865332010f, 0.994625985763907820f, 0.994293258092929790f,
	0.993960493864630480f,
	0.993627693091245660f, 0.993294855785010760f, 0.992961981958163210f,
	0.992629071622942340f,
	0.992296124791587690f, 0.991963141476341460f, 0.991630121689446090f,
	0.991297065443145440f,
	0.990963972749685840f, 0.990630843621313260f, 0.990297678070276800f,
	0.989964476108825210f,
	0.989631237749210020f, 0.989297963003683330f, 0.988964651884498000f,
	0.988631304403909890f,
	0.988297920574174430f, 0.987964500407549910f, 0.987631043916294970f,
	0.987297551112669370f,
	0.986964022008935520f, 0.986630456617355380f, 0.986296854950194260f,
	0.985963217019717120f,
	0.985629542838190490f, 0.985295832417883540f, 0.984962085771065030f,
	0.984628302910006580f,
	0.984294483846980150f, 0.983960628594258810f, 0.983626737164118190f,
	0.983292809568833910f,
	0.982958845820684270f, 0.982624845931947320f, 0.982290809914904140f,
	0.981956737781835790f,
	0.981622629545024770f, 0.981288485216756160f, 0.980954304809314670f,
	0.980620088334987930f,
	0.980285835806063770f, 0.979951547234831130f, 0.979617222633581860f,
	0.979282862014607240f,
	0.978948465390201530f, 0.978614032772659240f, 0.978279564174275860f,
	0.977945059607349900f,
	0.977610519084179290f, 0.977275942617064740f, 0.976941330218307540f,
	0.976606681900209830f,
	0.976271997675076550f, 0.975937277555212310f, 0.975602521552924600f,
	0.975267729680520560f,
	0.974932901950310350f, 0.974598038374604350f, 0.974263138965714040f,
	0.973928203735953460f,
	0.973593232697636530f, 0.973258225863079970f, 0.972923183244600480f,
	0.972588104854516410f,
	0.972252990705148370f, 0.971917840808816710f, 0.971582655177844700f,
	0.971247433824555920f,
	0.970912176761274950f, 0.970576884000329040f, 0.970241555554045230f,
	0.969906191434753320f,
	0.969570791654783330f, 0.969235356226466500f, 0.968899885162136650f,
	0.968564378474127350f,
	0.968228836174775060f, 0.967893258276415700f, 0.967557644791388500f,
	0.967221995732032490f,
	0.966886311110688230f, 0.966550590939698640f, 0.966214835231406500f,
	0.965879043998157160f,
	0.965543217252296420f, 0.965207355006171270f, 0.964871457272131190f,
	0.964535524062525410f,
	0.964199555389706030f, 0.963863551266025300f, 0.963527511703836660f,
	0.963191436715496120f,
	0.962855326313359350f, 0.962519180509785130f, 0.962182999317132030f,
	0.961846782747760140f,
	0.961510530814032040f, 0.961174243528309820f, 0.960837920902958720f,
	0.960501562950343390f,
	0.960165169682831830f, 0.959828741112791590f, 0.959492277252591900f,
	0.959155778114604400f,
	0.958819243711200310f, 0.958482674054753960f, 0.958146069157639560f,
	0.957809429032232760f,
	0.957472753690911670f, 0.957136043146054050f, 0.956799297410040440f,
	0.956462516495251940f,
	0.956125700414070300f, 0.955788849178880300f, 0.955451962802066120f,
	0.955115041296014880f,
	0.954778084673113870f, 0.954441092945751630f, 0.954104066126319150f,
	0.953767004227207060f,
	0.953429907260809120f, 0.953092775239518630f, 0.952755608175731570f,
	0.952418406081844360f,
	0.952081168970254520f, 0.951743896853362140f, 0.951406589743566950f,
	0.951069247653271500f,
	0.950731870594878510f, 0.950394458580791970f, 0.950057011623418380f,
	0.949719529735163940f,
	0.949382012928437600f, 0.949044461215648560f, 0.948706874609207220f,
	0.948369253121526420f,
	0.948031596765018910f, 0.947693905552099870f, 0.947356179495185020f,
	0.947018418606691230f,
	0.946680622899037650f, 0.946342792384643360f, 0.946004927075930090f,
	0.945667026985319680f,
	0.945329092125236190f, 0.944991122508104350f, 0.944653118146349890f,
	0.944315079052401090f,
	0.943977005238685770f, 0.943638896717634900f, 0.943300753501679190f,
	0.942962575603250920f,
	0.942624363034784580f, 0.942286115808714690f, 0.941947833937478270f,
	0.941609517433512730f,
	0.941271166309256450f, 0.940932780577150460f, 0.940594360249635500f,
	0.940255905339155150f,
	0.939917415858152920f, 0.939578891819073720f, 0.939240333234364950f,
	0.938901740116473540f,
	0.938563112477849630f, 0.938224450330942590f, 0.937885753688204820f,
	0.937547022562088990f,
	0.937208256965048840f, 0.936869456909540490f, 0.936530622408019990f,
	0.936191753472946030f,
	0.935852850116777430f, 0.935513912351974450f, 0.935174940190999560f,
	0.934835933646314900f,
	0.934496892730385720f, 0.934157817455677160f, 0.933818707834655590f,
	0.933479563879790030f,
	0.933140385603548840f, 0.932801173018403480f, 0.932461926136825660f,
	0.932122644971287830f,
	0.931783329534265240f, 0.931443979838232900f, 0.931104595895668410f,
	0.930765177719049210f,
	0.930425725320855430f, 0.930086238713567440f, 0.929746717909666790f,
	0.929407162921637610f,
	0.929067573761963250f, 0.928727950443130500f, 0.928388292977625930f,
	0.928048601377937210f,
	0.927708875656554800f, 0.927369115825968480f, 0.927029321898671270f,
	0.926689493887155820f,
	0.926349631803916270f, 0.926009735661449170f, 0.925669805472250860f,
	0.925329841248820340f,
	0.924989843003656610f, 0.924649810749260110f, 0.924309744498133750f,
	0.923969644262779830f,
	0.923629510055703820f, 0.923289341889410480f, 0.922949139776407800f,
	0.922608903729203570f,
	0.922268633760306990f, 0.921928329882229390f, 0.921587992107482210f,
	0.921247620448579440f,
	0.920907214918035070f, 0.920566775528364410f, 0.920226302292085460f,
	0.919885795221715540f,
	0.919545254329774850f, 0.919204679628783720f, 0.918864071131263780f,
	0.918523428849739030f,
	0.918182752796733110f, 0.917842042984772340f, 0.917501299426383480f,
	0.917160522134094160f,
	0.916819711120434700f, 0.916478866397934850f, 0.916137987979127270f,
	0.915797075876544350f,
	0.915456130102721200f, 0.915115150670193110f, 0.914774137591496510f,
	0.914433090879170130f,
	0.914092010545752620f, 0.913750896603785280f, 0.913409749065809520f,
	0.913068567944367970f,
	0.912727353252005710f, 0.912386105001267270f, 0.912044823204700370f,
	0.911703507874852440f,
	0.911362159024272310f, 0.911020776665511290f, 0.910679360811120000f,
	0.910337911473652390f,
	0.909996428665661990f, 0.909654912399703860f, 0.909313362688335290f,
	0.908971779544113350f,
	0.908630162979597760f, 0.908288513007348140f, 0.907946829639926790f,
	0.907605112889895870f,
	0.907263362769819000f, 0.906921579292262250f, 0.906579762469791110f,
	0.906237912314974080f,
	0.905896028840379560f, 0.905554112058577170f, 0.905212161982139160f,
	0.904870178623637170f,
	0.904528161995645670f, 0.904186112110739510f, 0.903844028981494190f,
	0.903501912620488070f,
	0.903159763040298880f, 0.902817580253507450f, 0.902475364272694370f,
	0.902133115110441470f,
	0.901790832779333250f, 0.901448517291953520f, 0.901106168660889110f,
	0.900763786898726380f,
	0.900421372018054500f, 0.900078924031462610f, 0.899736442951541320f,
	0.899393928790883420f,
	0.899051381562081310f, 0.898708801277730340f, 0.898366187950425780f,
	0.898023541592764210f,
	0.897680862217344440f, 0.897338149836764960f, 0.896995404463627350f,
	0.896652626110532870f,
	0.896309814790084090f, 0.895966970514885940f, 0.895624093297543110f,
	0.895281183150662960f,
	0.894938240086852970f, 0.894595264118721810f, 0.894252255258880410f,
	0.893909213519939460f,
	0.893566138914512420f, 0.893223031455212530f, 0.892879891154655380f,
	0.892536718025457090f,
	0.892193512080234670f, 0.891850273331607600f, 0.891507001792195000f,
	0.891163697474618880f,
	0.890820360391500920f, 0.890476990555464480f, 0.890133587979135000f,
	0.889790152675137610f,
	0.889446684656100330f, 0.889103183934650930f, 0.888759650523418650f,
	0.888416084435035060f,
	0.888072485682131150f, 0.887728854277341050f, 0.887385190233298650f,
	0.887041493562639060f,
	0.886697764277999840f, 0.886354002392018110f, 0.886010207917333760f,
	0.885666380866586560f,
	0.885322521252418610f, 0.884978629087472270f, 0.884634704384391180f,
	0.884290747155821230f,
	0.883946757414407980f, 0.883602735172799640f, 0.883258680443644530f,
	0.882914593239592320f,
	0.882570473573294660f, 0.882226321457403320f, 0.881882136904572400f,
	0.881537919927456340f,
	0.881193670538710450f, 0.880849388750992610f, 0.880505074576960370f,
	0.880160728029273920f,
	0.879816349120593590f, 0.879471937863580690f, 0.879127494270899090f,
	0.878783018355212220f,
	0.878438510129186170f, 0.878093969605486800f, 0.877749396796782770f,
	0.877404791715742370f,
	0.877060154375035710f, 0.876715484787334630f, 0.876370782965310900f,
	0.876026048921639160f,
	0.875681282668993700f, 0.875336484220050390f, 0.874991653587487090f,
	0.874646790783981660f,
	0.874301895822214290f, 0.873956968714865500f, 0.873612009474616810f,
	0.873267018114152300f,
	0.872921994646155390f, 0.872576939083312460f, 0.872231851438309840f,
	0.871886731723835020f,
	0.871541579952577750f, 0.871196396137227660f, 0.870851180290476810f,
	0.870505932425017060f,
	0.870160652553543020f, 0.869815340688749220f, 0.869469996843331370f,
	0.869124621029987670f,
	0.868779213261415610f, 0.868433773550315810f, 0.868088301909388680f,
	0.867742798351335720f,
	0.867397262888861100f, 0.867051695534668210f, 0.866706096301463340f,
	0.866360465201952980f,
	0.866014802248844420f, 0.865669107454847490f, 0.865323380832671800f,
	0.864977622395029290f,
	0.864631832154632240f, 0.864286010124194040f, 0.863940156316430170f,
	0.863594270744056040f,
	0.863248353419789670f, 0.862902404356348570f, 0.862556423566453230f,
	0.862210411062823810f,
	0.861864366858181910f, 0.861518290965251340f, 0.861172183396755500f,
	0.860826044165420630f,
	0.860479873283972910f, 0.860133670765139580f, 0.859787436621650360f,
	0.859441170866234390f,
	0.859094873511623840f, 0.858748544570550610f, 0.858402184055747750f,
	0.858055791979950740f,
	0.857709368355894840f, 0.857362913196317630f, 0.857016426513956930f,
	0.856669908321551650f,
	0.856323358631843170f, 0.855976777457572280f, 0.855630164811482460f,
	0.855283520706317080f,
	0.854936845154821930f, 0.854590138169742830f, 0.854243399763827020f,
	0.853896629949823630f,
	0.853549828740481690f, 0.853202996148552880f, 0.852856132186788910f,
	0.852509236867942440f,
	0.852162310204768740f, 0.851815352210022470f, 0.851468362896461110f,
	0.851121342276842110f,
	0.850774290363923820f, 0.850427207170467380f, 0.850080092709233130f,
	0.849732946992984290f,
	0.849385770034483680f, 0.849038561846496730f, 0.848691322441788910f,
	0.848344051833126780f,
	0.847996750033279350f, 0.847649417055015060f, 0.847302052911105160f,
	0.846954657614320980f,
	0.846607231177434640f, 0.846259773613221020f, 0.845912284934454140f,
	0.845564765153910990f,
	0.845217214284368690f, 0.844869632338605130f, 0.844522019329400630f,
	0.844174375269535320f,
	0.843826700171791620f, 0.843478994048952440f, 0.843131256913801420f,
	0.842783488779124570f,
	0.842435689657707650f, 0.842087859562339000f, 0.841739998505806610f,
	0.841392106500900900f,
	0.841044183560412770f, 0.840696229697133760f, 0.840348244923857960f,
	0.840000229253379030f,
	0.839652182698493290f, 0.839304105271996950f, 0.838955996986687550f,
	0.838607857855364740f,
	0.838259687890827830f, 0.837911487105878820f, 0.837563255513319780f,
	0.837214993125953600f,
	0.836866699956585690f, 0.836518376018021260f, 0.836170021323067610f,
	0.835821635884532730f,
	0.835473219715225040f, 0.835124772827955830f, 0.834776295235535540f,
	0.834427786950777460f,
	0.834079247986494690f, 0.833730678355502630f, 0.833382078070616820f,
	0.833033447144653880f,
	0.832684785590432690f, 0.832336093420771970f, 0.831987370648492710f,
	0.831638617286416190f,
	0.831289833347364620f, 0.830941018844162600f, 0.830592173789634240f,
	0.830243298196606360f,
	0.829894392077905720f, 0.829545455446360270f, 0.829196488314800080f,
	0.828847490696055010f,
	0.828498462602957340f, 0.828149404048339590f, 0.827800315045035150f,
	0.827451195605879990f,
	0.827102045743709160f, 0.826752865471360950f, 0.826403654801672770f,
	0.826054413747485010f,
	0.825705142321637720f, 0.825355840536972420f, 0.825006508406332490f,
	0.824657145942561230f,
	0.824307753158504460f, 0.823958330067008030f, 0.823608876680918760f,
	0.823259393013085820f,
	0.822909879076357930f, 0.822560334883586490f, 0.822210760447622980f,
	0.821861155781319800f,
	0.821511520897531660f, 0.821161855809112830f, 0.820812160528920360f,
	0.820462435069811090f,
	0.820112679444643060f, 0.819762893666276530f, 0.819413077747571440f,
	0.819063231701390170f,
	0.818713355540594880f, 0.818363449278050270f, 0.818013512926620940f,
	0.817663546499172720f,
	0.817313550008573640f, 0.816963523467691410f, 0.816613466889396070f,
	0.816263380286557980f,
	0.815913263672048310f, 0.815563117058740630f, 0.815212940459508210f,
	0.814862733887226740f,
	0.814512497354771830f, 0.814162230875020380f, 0.813811934460851430f,
	0.813461608125143560f,
	0.813111251880778150f, 0.812760865740636440f, 0.812410449717600570f,
	0.812060003824555230f,
	0.811709528074384460f, 0.811359022479975040f, 0.811008487054213360f,
	0.810657921809988410f,
	0.810307326760189020f, 0.809956701917705080f, 0.809606047295428950f,
	0.809255362906252440f,
	0.808904648763069890f, 0.808553904878775760f, 0.808203131266265420f,
	0.807852327938436750f,
	0.807501494908186900f, 0.807150632188415760f, 0.806799739792023240f,
	0.806448817731910130f,
	0.806097866020979660f, 0.805746884672134620f, 0.805395873698280360f,
	0.805044833112322000f,
	0.804693762927166100f, 0.804342663155721230f, 0.803991533810895500f,
	0.803640374905599810f,
	0.803289186452744390f, 0.802937968465242240f, 0.802586720956006250f,
	0.802235443937950320f,
	0.801884137423990890f, 0.801532801427043530f, 0.801181435960026780f,
	0.800830041035858750f,
	0.800478616667459010f, 0.800127162867749210f, 0.799775679649650460f,
	0.799424167026086540f,
	0.799072625009981330f, 0.798721053614259490f, 0.798369452851848020f,
	0.798017822735673680f,
	0.797666163278665570f, 0.797314474493752810f, 0.796962756393865600f,
	0.796611008991936490f,
	0.796259232300897350f, 0.795907426333682830f, 0.795555591103226930f,
	0.795203726622466520f,
	0.794851832904338360f, 0.794499909961779990f, 0.794147957807731400f,
	0.793795976455132220f,
	0.793443965916924570f, 0.793091926206050400f, 0.792739857335452710f,
	0.792387759318077150f,
	0.792035632166868230f, 0.791683475894773720f, 0.791331290514740830f,
	0.790979076039718180f,
	0.790626832482656310f, 0.790274559856505520f, 0.789922258174218570f,
	0.789569927448748320f,
	0.789217567693048520f, 0.788865178920075130f, 0.788512761142783790f,
	0.788160314374132590f,
	0.787807838627079260f, 0.787455333914584220f, 0.787102800249607550f,
	0.786750237645110430f,
	0.786397646114056490f, 0.786045025669408700f, 0.785692376324132690f,
	0.785339698091194080f,
	0.784986990983559170f, 0.784634255014197040f, 0.784281490196075850f,
	0.783928696542166680f,
	0.783575874065440270f, 0.783223022778868350f, 0.782870142695425320f,
	0.782517233828084580f,
	0.782164296189822530f, 0.781811329793615120f, 0.781458334652439630f,
	0.781105310779275470f,
	0.780752258187101480f, 0.780399176888899150f, 0.780046066897649550f,
	0.779692928226336290f,
	0.779339760887942880f, 0.778986564895453810f, 0.778633340261856040f,
	0.778280087000135730f,
	0.777926805123281830f, 0.777573494644283050f, 0.777220155576129220f,
	0.776866787931812410f,
	0.776513391724324210f, 0.776159966966658680f, 0.775806513671809860f,
	0.775453031852772920f,
	0.775099521522545020f, 0.774745982694123090f, 0.774392415380506400f,
	0.774038819594694230f,
	0.773685195349686940f, 0.773331542658487140f, 0.772977861534096640f,
	0.772624151989520280f,
	0.772270414037761980f, 0.771916647691828660f, 0.771562852964726710f,
	0.771209029869463940f,
	0.770855178419050050f, 0.770501298626494410f, 0.770147390504808960f,
	0.769793454067005500f,
	0.769439489326096850f, 0.769085496295098040f, 0.768731474987023660f,
	0.768377425414890850f,
	0.768023347591716640f, 0.767669241530518850f, 0.767315107244318060f,
	0.766960944746133740f,
	0.766606754048988260f, 0.766252535165903970f, 0.765898288109903900f,
	0.765544012894013530f,
	0.765189709531257760f, 0.764835378034664170f, 0.764481018417259680f,
	0.764126630692073870f,
	0.763772214872136200f, 0.763417770970477140f, 0.763063299000129260f,
	0.762708798974124800f,
	0.762354270905498450f, 0.761999714807284790f, 0.761645130692519490f,
	0.761290518574240350f,
	0.760935878465484720f, 0.760581210379292380f, 0.760226514328703140f,
	0.759871790326757670f,
	0.759517038386499090f, 0.759162258520969860f, 0.758807450743214760f,
	0.758452615066278920f,
	0.758097751503208020f, 0.757742860067050380f, 0.757387940770853360f,
	0.757032993627667290f,
	0.756678018650541630f, 0.756323015852528700f, 0.755967985246680520f,
	0.755612926846050080f,
	0.755257840663692730f, 0.754902726712663120f, 0.754547585006018600f,
	0.754192415556816380f,
	0.753837218378114460f, 0.753481993482973400f, 0.753126740884452970f,
	0.752771460595615500f,
	0.752416152629523330f, 0.752060816999239660f, 0.751705453717829930f,
	0.751350062798359140f,
	0.750994644253894730f, 0.750639198097504010f, 0.750283724342255320f,
	0.749928223001219310f,
	0.749572694087465850f, 0.749217137614067500f, 0.748861553594096340f,
	0.748505942040627040f,
	0.748150302966733790f, 0.747794636385492150f, 0.747438942309979870f,
	0.747083220753273820f,
	0.746727471728453770f, 0.746371695248599140f, 0.746015891326790470f,
	0.745660059976110400f,
	0.745304201209641030f, 0.744948315040467210f, 0.744592401481673270f,
	0.744236460546344850f,
	0.743880492247569580f, 0.743524496598434670f, 0.743168473612029980f,
	0.742812423301444810f,
	0.742456345679769810f, 0.742100240760097840f, 0.741744108555520860f,
	0.741387949079133860f,
	0.741031762344030790f, 0.740675548363308620f, 0.740319307150063780f,
	0.739963038717393880f,
	0.739606743078398690f, 0.739250420246177380f, 0.738894070233831800f,
	0.738537693054463370f,
	0.738181288721174830f, 0.737824857247070810f, 0.737468398645255490f,
	0.737111912928835710f,
	0.736755400110918000f, 0.736398860204609870f, 0.736042293223021060f,
	0.735685699179260850f,
	0.735329078086440880f, 0.734972429957672760f, 0.734615754806068890f,
	0.734259052644744230f,
	0.733902323486812610f, 0.733545567345390890f, 0.733188784233595240f,
	0.732831974164544150f,
	0.732475137151356370f, 0.732118273207151170f, 0.731761382345050280f,
	0.731404464578174760f,
	0.731047519919648340f, 0.730690548382594280f, 0.730333549980137110f,
	0.729976524725403530f,
	0.729619472631519270f, 0.729262393711613280f, 0.728905287978813600f,
	0.728548155446249730f,
	0.728190996127053180f, 0.727833810034354990f, 0.727476597181288540f,
	0.727119357580987220f,
	0.726762091246585200f, 0.726404798191218950f, 0.726047478428024420f,
	0.725690131970139980f,
	0.725332758830703360f, 0.724975359022855150f, 0.724617932559735390f,
	0.724260479454485130f,
	0.723902999720247850f, 0.723545493370166160f, 0.723187960417385530f,
	0.722830400875050790f,
	0.722472814756308090f, 0.722115202074305680f, 0.721757562842191060f,
	0.721399897073114470f,
	0.721042204780225960f, 0.720684485976676230f, 0.720326740675618530f,
	0.719968968890205230f,
	0.719611170633591480f, 0.719253345918932090f, 0.718895494759382860f,
	0.718537617168101610f,
	0.718179713158245800f, 0.717821782742975370f, 0.717463825935449550f,
	0.717105842748830160f,
	0.716747833196278770f, 0.716389797290958090f, 0.716031735046032900f,
	0.715673646474667140f,
	0.715315531590027700f, 0.714957390405280950f, 0.714599222933594240f,
	0.714241029188137260f,
	0.713882809182079030f, 0.713524562928591010f, 0.713166290440844450f,
	0.712807991732011590f,
	0.712449666815266890f, 0.712091315703784260f, 0.711732938410739810f,
	0.711374534949309800f,
	0.711016105332671340f, 0.710657649574003460f, 0.710299167686484930f,
	0.709940659683296890f,
	0.709582125577619790f, 0.709223565382636760f, 0.708864979111530680f,
	0.708506366777485130f,
	0.708147728393686340f, 0.707789063973319310f, 0.707430373529572170f,
	0.707071657075632460f,
	0.706712914624688770f, 0.706354146189931750f, 0.705995351784551530f,
	0.705636531421740880f,
	0.705277685114692020f, 0.704918812876598410f, 0.704559914720655490f,
	0.704200990660058150f,
	0.703842040708003820f, 0.703483064877689630f, 0.703124063182313690f,
	0.702765035635076310f,
	0.702405982249177160f, 0.702046903037818250f, 0.701687798014201110f,
	0.701328667191529980f,
	0.700969510583008600f, 0.700610328201841660f, 0.700251120061236020f,
	0.699891886174398130f,
	0.699532626554536630f, 0.699173341214860190f, 0.698814030168578240f,
	0.698454693428902320f,
	0.698095331009043640f, 0.697735942922215520f, 0.697376529181631400f,
	0.697017089800505250f,
	0.696657624792053730f, 0.696298134169492380f, 0.695938617946039510f,
	0.695579076134912990f,
	0.695219508749331800f, 0.694859915802517050f, 0.694500297307689140f,
	0.694140653278070950f,
	0.693780983726884790f, 0.693421288667355530f, 0.693061568112707690f,
	0.692701822076166820f,
	0.692342050570960430f, 0.691982253610315510f, 0.691622431207461700f,
	0.691262583375628180f,
	0.690902710128045050f, 0.690542811477944610f, 0.690182887438558710f,
	0.689822938023121220f,
	0.689462963244866330f, 0.689102963117028790f, 0.688742937652845550f,
	0.688382886865552930f,
	0.688022810768389670f, 0.687662709374594510f, 0.687302582697406850f,
	0.686942430750068330f,
	0.686582253545819920f, 0.686222051097905130f, 0.685861823419566700f,
	0.685501570524050140f,
	0.685141292424600310f, 0.684780989134463280f, 0.684420660666887120f,
	0.684060307035119440f,
	0.683699928252410110f, 0.683339524332008840f, 0.682979095287166160f,
	0.682618641131135020f,
	0.682258161877167370f, 0.681897657538517720f, 0.681537128128440470f,
	0.681176573660190910f,
	0.680815994147026320f, 0.680455389602203310f, 0.680094760038981280f,
	0.679734105470619080f,
	0.679373425910376310f, 0.679012721371515250f, 0.678651991867297080f,
	0.678291237410985510f,
	0.677930458015843620f, 0.677569653695137220f, 0.677208824462131490f,
	0.676847970330092700f,
	0.676487091312289350f, 0.676126187421989040f, 0.675765258672461950f,
	0.675404305076978020f,
	0.675043326648808170f, 0.674682323401225250f, 0.674321295347501510f,
	0.673960242500911690f,
	0.673599164874730370f, 0.673238062482232950f, 0.672876935336696900f,
	0.672515783451398950f,
	0.672154606839618470f, 0.671793405514634180f, 0.671432179489727110f,
	0.671070928778178090f,
	0.670709653393269050f, 0.670348353348283690f, 0.669987028656505170f,
	0.669625679331219300f,
	0.669264305385711360f, 0.668902906833267590f, 0.668541483687176590f,
	0.668180035960725840f,
	0.667818563667205600f, 0.667457066819905800f, 0.667095545432117240f,
	0.666733999517132860f,
	0.666372429088244790f, 0.666010834158747840f, 0.665649214741936390f,
	0.665287570851105680f,
	0.664925902499553190f, 0.664564209700575500f, 0.664202492467472090f,
	0.663840750813541210f,
	0.663478984752084110f, 0.663117194296401260f, 0.662755379459794350f,
	0.662393540255567070f,
	0.662031676697022450f, 0.661669788797465960f, 0.661307876570202740f,
	0.660945940028538900f,
	0.660583979185782600f, 0.660221994055241400f, 0.659859984650225110f,
	0.659497950984043510f,
	0.659135893070007080f, 0.658773810921428500f, 0.658411704551619570f,
	0.658049573973894850f,
	0.657687419201568260f, 0.657325240247955020f, 0.656963037126372160f,
	0.656600809850135910f,
	0.656238558432565400f, 0.655876282886978410f, 0.655513983226695960f,
	0.655151659465038060f,
	0.654789311615326050f, 0.654426939690883280f, 0.654064543705032310f,
	0.653702123671098150f,
	0.653339679602405470f, 0.652977211512280050f, 0.652614719414049580f,
	0.652252203321041060f,
	0.651889663246583930f, 0.651527099204007310f, 0.651164511206641320f,
	0.650801899267818060f,
	0.650439263400868990f, 0.650076603619127890f, 0.649713919935928420f,
	0.649351212364604910f,
	0.648988480918494040f, 0.648625725610931460f, 0.648262946455255510f,
	0.647900143464803730f,
	0.647537316652916140f, 0.647174466032932490f, 0.646811591618193350f,
	0.646448693422041360f,
	0.646085771457818310f, 0.645722825738868860f, 0.645359856278536980f,
	0.644996863090167570f,
	0.644633846187107620f, 0.644270805582703550f, 0.643907741290304040f,
	0.643544653323257610f,
	0.643181541694913480f, 0.642818406418622980f, 0.642455247507736860f,
	0.642092064975608220f,
	0.641728858835589830f, 0.641365629101035340f, 0.641002375785300500f,
	0.640639098901740200f,
	0.640275798463712080f, 0.639912474484572560f, 0.639549126977681070f,
	0.639185755956396480f,
	0.638822361434078330f, 0.638458943424088490f, 0.638095501939787920f,
	0.637732036994540290f,
	0.637368548601708660f, 0.637005036774657030f, 0.636641501526751590f,
	0.636277942871357530f,
	0.635914360821842830f, 0.635550755391574910f, 0.635187126593922070f,
	0.634823474442254840f,
	0.634459798949942640f, 0.634096100130357660f, 0.633732377996871770f,
	0.633368632562857470f,
	0.633004863841689520f, 0.632641071846741790f, 0.632277256591390780f,
	0.631913418089012020f,
	0.631549556352983710f, 0.631185671396683470f, 0.630821763233490040f,
	0.630457831876783950f,
	0.630093877339945260f, 0.629729899636356280f, 0.629365898779399080f,
	0.629001874782456500f,
	0.628637827658913300f, 0.628273757422153860f, 0.627909664085564810f,
	0.627545547662532230f,
	0.627181408166443410f, 0.626817245610687520f, 0.626453060008652860f,
	0.626088851373730380f,
	0.625724619719310480f, 0.625360365058784670f, 0.624996087405546350f,
	0.624631786772988030f,
	0.624267463174504880f, 0.623903116623491180f, 0.623538747133343780f,
	0.623174354717459190f,
	0.622809939389234460f, 0.622445501162069090f, 0.622081040049361490f,
	0.621716556064512820f,
	0.621352049220923570f, 0.620987519531995270f, 0.620622967011131400f,
	0.620258391671734690f,
	0.619893793527210410f, 0.619529172590963410f, 0.619164528876399280f,
	0.618799862396925750f,
	0.618435173165949760f, 0.618070461196880800f, 0.617705726503127720f,
	0.617340969098100430f,
	0.616976188995210780f, 0.616611386207870040f, 0.616246560749491690f,
	0.615881712633488340f,
	0.615516841873275490f, 0.615151948482267840f, 0.614787032473881110f,
	0.614422093861533010f,
	0.614057132658640590f, 0.613692148878623000f, 0.613327142534899510f,
	0.612962113640889710f,
	0.612597062210015750f, 0.612231988255698470f, 0.611866891791361560f,
	0.611501772830428060f,
	0.611136631386322020f, 0.610771467472469460f, 0.610406281102295440f,
	0.610041072289227990f,
	0.609675841046694030f, 0.609310587388121830f, 0.608945311326941520f,
	0.608580012876582370f,
	0.608214692050476290f, 0.607849348862054220f, 0.607483983324749510f,
	0.607118595451995420f,
	0.606753185257225550f, 0.606387752753876020f, 0.606022297955381760f,
	0.605656820875180360f,
	0.605291321526709060f, 0.604925799923405670f, 0.604560256078710220f,
	0.604194690006061960f,
	0.603829101718902580f, 0.603463491230673220f, 0.603097858554815790f,
	0.602732203704774650f,
	0.602366526693992930f, 0.602000827535916330f, 0.601635106243990190f,
	0.601269362831660550f,
	0.600903597312375640f, 0.600537809699582810f, 0.600172000006731770f,
	0.599806168247271620f,
	0.599440314434653620f, 0.599074438582328780f, 0.598708540703749010f,
	0.598342620812368000f,
	0.597976678921638860f, 0.597610715045016950f, 0.597244729195957500f,
	0.596878721387916090f,
	0.596512691634350830f, 0.596146639948718640f, 0.595780566344478960f,
	0.595414470835091030f,
	0.595048353434014630f, 0.594682214154711790f, 0.594316053010643270f,
	0.593949870015273000f,
	0.593583665182063740f, 0.593217438524479500f, 0.592851190055986300f,
	0.592484919790049140f,
	0.592118627740135460f, 0.591752313919712170f, 0.591385978342248260f,
	0.591019621021212420f,
	0.590653241970074180f, 0.590286841202305120f, 0.589920418731375800f,
	0.589553974570759530f,
	0.589187508733928890f, 0.588821021234357310f, 0.588454512085520460f,
	0.588087981300892900f,
	0.587721428893951850f, 0.587354854878173850f, 0.586988259267036350f,
	0.586621642074019120f,
	0.586255003312600500f, 0.585888342996261690f, 0.585521661138483250f,
	0.585154957752746730f,
	0.584788232852535560f, 0.584421486451332410f, 0.584054718562622140f,
	0.583687929199888990f,
	0.583321118376619710f, 0.582954286106300290f, 0.582587432402417840f,
	0.582220557278461340f,
	0.581853660747918780f, 0.581486742824280810f, 0.581119803521037650f,
	0.580752842851679940f,
	0.580385860829700780f, 0.580018857468592270f, 0.579651832781848730f,
	0.579284786782964360f,
	0.578917719485433800f, 0.578550630902754050f, 0.578183521048421080f,
	0.577816389935933090f,
	0.577449237578788300f, 0.577082063990485340f, 0.576714869184524860f,
	0.576347653174406840f,
	0.575980415973633590f, 0.575613157595706530f, 0.575245878054129520f,
	0.574878577362406000f,
	0.574511255534040030f, 0.574143912582537940f, 0.573776548521405030f,
	0.573409163364148930f,
	0.573041757124277180f, 0.572674329815297640f, 0.572306881450720390f,
	0.571939412044054740f,
	0.571571921608812320f, 0.571204410158504090f, 0.570836877706642270f,
	0.570469324266740570f,
	0.570101749852312100f, 0.569734154476872480f, 0.569366538153936560f,
	0.568998900897020210f,
	0.568631242719641270f, 0.568263563635316600f, 0.567895863657565500f,
	0.567528142799906490f,
	0.567160401075860410f, 0.566792638498947680f, 0.566424855082689470f,
	0.566057050840608870f,
	0.565689225786228160f, 0.565321379933072190f, 0.564953513294665140f,
	0.564585625884531870f,
	0.564217717716199550f, 0.563849788803194140f, 0.563481839159044150f,
	0.563113868797277870f,
	0.562745877731423820f, 0.562377865975012940f, 0.562009833541575080f,
	0.561641780444642640f,
	0.561273706697747450f, 0.560905612314422150f, 0.560537497308201240f,
	0.560169361692618440f,
	0.559801205481210040f, 0.559433028687510990f, 0.559064831325059240f,
	0.558696613407391630f,
	0.558328374948046320f, 0.557960115960563050f, 0.557591836458480870f,
	0.557223536455341280f,
	0.556855215964685120f, 0.556486875000054000f, 0.556118513574991650f,
	0.555750131703040880f,
	0.555381729397746880f, 0.555013306672654360f, 0.554644863541308600f,
	0.554276400017257090f,
	0.553907916114046440f, 0.553539411845225590f, 0.553170887224342820f,
	0.552802342264947400f,
	0.552433776980590490f, 0.552065191384822350f, 0.551696585491195710f,
	0.551327959313262280f,
	0.550959312864576220f, 0.550590646158691240f, 0.550221959209161620f,
	0.549853252029543830f,
	0.549484524633393480f, 0.549115777034268170f, 0.548747009245725500f,
	0.548378221281323520f,
	0.548009413154622370f, 0.547640584879181100f, 0.547271736468561530f,
	0.546902867936324590f,
	0.546533979296032200f, 0.546165070561248080f, 0.545796141745535150f,
	0.545427192862458780f,
	0.545058223925583670f, 0.544689234948475210f, 0.544320225944701200f,
	0.543951196927828010f,
	0.543582147911424560f, 0.543213078909059120f, 0.542843989934301940f,
	0.542474881000723050f,
	0.542105752121893050f, 0.541736603311384620f, 0.541367434582769480f,
	0.540998245949621760f,
	0.540629037425515050f, 0.540259809024023600f, 0.539890560758723770f,
	0.539521292643190930f,
	0.539152004691002770f, 0.538782696915736770f, 0.538413369330970610f,
	0.538044021950284450f,
	0.537674654787257180f, 0.537305267855470390f, 0.536935861168504670f,
	0.536566434739941920f,
	0.536196988583365510f, 0.535827522712358230f, 0.535458037140505110f,
	0.535088531881390050f,
	0.534719006948599860f, 0.534349462355720230f, 0.533979898116337950f,
	0.533610314244041710f,
	0.533240710752419080f, 0.532871087655060300f, 0.532501444965554960f,
	0.532131782697493170f,
	0.531762100864467290f, 0.531392399480068670f, 0.531022678557890980f,
	0.530652938111527360f,
	0.530283178154571710f, 0.529913398700619820f, 0.529543599763266700f,
	0.529173781356109600f,
	0.528803943492745180f, 0.528434086186771010f, 0.528064209451786560f,
	0.527694313301390160f,
	0.527324397749182720f, 0.526954462808764120f, 0.526584508493736840f,
	0.526214534817702310f,
	0.525844541794263210f, 0.525474529437023890f, 0.525104497759587900f,
	0.524734446775560910f,
	0.524364376498548390f, 0.523994286942156220f, 0.523624178119992400f,
	0.523254050045663940f,
	0.522883902732780290f, 0.522513736194950230f, 0.522143550445783310f,
	0.521773345498891090f,
	0.521403121367884030f, 0.521032878066375100f, 0.520662615607976660f,
	0.520292334006301820f,
	0.519922033274965560f, 0.519551713427582000f, 0.519181374477767470f,
	0.518811016439137520f,
	0.518440639325310040f, 0.518070243149902240f, 0.517699827926532130f,
	0.517329393668819580f,
	0.516958940390383700f, 0.516588468104845820f, 0.516217976825826600f,
	0.515847466566947580f,
	0.515476937341832310f, 0.515106389164103120f, 0.514735822047384990f,
	0.514365236005302040f,
	0.513994631051479240f, 0.513624007199543600f, 0.513253364463121090f,
	0.512882702855839920f,
	0.512512022391327980f, 0.512141323083213470f, 0.511770604945127050f,
	0.511399867990697920f,
	0.511029112233557960f, 0.510658337687338040f, 0.510287544365671140f,
	0.509916732282189920f,
	0.509545901450527690f, 0.509175051884319660f, 0.508804183597200140f,
	0.508433296602805670f,
	0.508062390914772230f, 0.507691466546736580f, 0.507320523512337470f,
	0.506949561825212450f,
	0.506578581499001590f, 0.506207582547344550f, 0.505836564983881190f,
	0.505465528822253710f,
	0.505094474076103310f, 0.504723400759073290f, 0.504352308884806750f,
	0.503981198466947000f,
	0.503610069519139780f, 0.503238922055029400f, 0.502867756088262840f,
	0.502496571632486070f,
	0.502125368701347050f, 0.501754147308493770f, 0.501382907467574190f,
	0.501011649192238950f,
	0.500640372496137020f, 0.500269077392920150f, 0.499897763896239410f,
	0.499526432019746450f,
	0.499155081777094940f, 0.498783713181937540f, 0.498412326247929250f,
	0.498040920988724490f,
	0.497669497417978280f, 0.497298055549347750f, 0.496926595396488870f,
	0.496555116973059980f,
	0.496183620292718900f, 0.495812105369124070f, 0.495440572215935850f,
	0.495069020846813650f,
	0.494697451275419140f, 0.494325863515413130f, 0.493954257580458580f,
	0.493582633484217940f,
	0.493210991240354450f, 0.492839330862533120f, 0.492467652364417970f,
	0.492095955759675460f,
	0.491724241061971320f, 0.491352508284972070f, 0.490980757442346090f,
	0.490608988547760690f,
	0.490237201614885710f, 0.489865396657390210f, 0.489493573688943970f,
	0.489121732723218740f,
	0.488749873773885120f, 0.488377996854616250f, 0.488006101979084450f,
	0.487634189160962910f,
	0.487262258413926560f, 0.486890309751649490f, 0.486518343187807900f,
	0.486146358736077200f,
	0.485774356410135000f, 0.485402336223658360f, 0.485030298190324950f,
	0.484658242323814380f,
	0.484286168637805270f, 0.483914077145978560f, 0.483541967862014480f,
	0.483169840799594130f,
	0.482797695972400300f, 0.482425533394114920f, 0.482053353078422120f,
	0.481681155039005550f,
	0.481308939289549380f, 0.480936705843739820f, 0.480564454715261990f,
	0.480192185917803270f,
	0.479819899465050160f, 0.479447595370691370f, 0.479075273648415010f,
	0.478702934311909910f,
	0.478330577374866780f, 0.477958202850975230f, 0.477585810753927250f,
	0.477213401097414220f,
	0.476840973895128200f, 0.476468529160763100f, 0.476096066908011760f,
	0.475723587150569390f,
	0.475351089902130650f, 0.474978575176390750f, 0.474606042987046840f,
	0.474233493347795020f,
	0.473860926272333670f, 0.473488341774360670f, 0.473115739867574380f,
	0.472743120565675250f,
	0.472370483882362520f, 0.471997829831337810f, 0.471625158426301700f,
	0.471252469680957190f,
	0.470879763609006460f, 0.470507040224152460f, 0.470134299540099940f,
	0.469761541570552780f,
	0.469388766329217000f, 0.469015973829798090f, 0.468643164086002100f,
	0.468270337111537040f,
	0.467897492920109850f, 0.467524631525429830f, 0.467151752941205530f,
	0.466778857181146260f,
	0.466405944258963200f, 0.466033014188366350f, 0.465660066983068220f,
	0.465287102656780530f,
	0.464914121223215740f, 0.464541122696088100f, 0.464168107089110940f,
	0.463795074415999760f,
	0.463422024690469060f, 0.463048957926235630f, 0.462675874137015720f,
	0.462302773336526080f,
	0.461929655538485470f, 0.461556520756611410f, 0.461183369004623920f,
	0.460810200296242310f,
	0.460437014645186440f, 0.460063812065178160f, 0.459690592569938270f,
	0.459317356173189750f,
	0.458944102888655060f, 0.458570832730057170f, 0.458197545711121090f,
	0.457824241845570630f,
	0.457450921147131930f, 0.457077583629530550f, 0.456704229306492570f,
	0.456330858191746010f,
	0.455957470299017840f, 0.455584065642037350f, 0.455210644234532610f,
	0.454837206090234200f,
	0.454463751222871910f, 0.454090279646176210f, 0.453716791373879380f,
	0.453343286419712720f,
	0.452969764797409750f, 0.452596226520703360f, 0.452222671603327130f,
	0.451849100059016350f,
	0.451475511901505420f, 0.451101907144530910f, 0.450728285801828830f,
	0.450354647887135640f,
	0.449980993414189900f, 0.449607322396728900f, 0.449233634848492320f,
	0.448859930783219170f,
	0.448486210214649020f, 0.448112473156523420f, 0.447738719622582710f,
	0.447364949626569590f,
	0.446991163182225700f, 0.446617360303294910f, 0.446243541003520480f,
	0.445869705296646270f,
	0.445495853196417930f, 0.445121984716580210f, 0.444748099870879880f,
	0.444374198673063330f,
	0.444000281136877280f, 0.443626347276070590f, 0.443252397104390790f,
	0.442878430635587910f,
	0.442504447883411090f, 0.442130448861610240f, 0.441756433583937120f,
	0.441382402064142250f,
	0.441008354315978680f, 0.440634290353198510f, 0.440260210189554690f,
	0.439886113838801880f,
	0.439512001314693700f, 0.439137872630986080f, 0.438763727801433690f,
	0.438389566839793740f,
	0.438015389759822630f, 0.437641196575277220f, 0.437266987299916590f,
	0.436892761947498260f,
	0.436518520531782470f, 0.436144263066528480f, 0.435769989565496290f,
	0.435395700042447710f,
	0.435021394511143410f, 0.434647072985346380f, 0.434272735478819010f,
	0.433898382005324050f,
	0.433524012578626440f, 0.433149627212489670f, 0.432775225920679740f,
	0.432400808716961900f,
	0.432026375615101930f, 0.431651926628867530f, 0.431277461772025310f,
	0.430902981058344070f,
	0.430528484501591540f, 0.430153972115537800f, 0.429779443913952170f,
	0.429404899910604490f,
	0.429030340119266550f, 0.428655764553708960f, 0.428281173227704760f,
	0.427906566155026040f,
	0.427531943349445720f, 0.427157304824738350f, 0.426782650594677570f,
	0.426407980673039090f,
	0.426033295073598160f, 0.425658593810130330f, 0.425283876896413280f,
	0.424909144346223290f,
	0.424534396173339160f, 0.424159632391538870f, 0.423784853014600950f,
	0.423410058056305830f,
	0.423035247530432810f, 0.422660421450763490f, 0.422285579831078230f,
	0.421910722685159720f,
	0.421535850026790060f, 0.421160961869751720f, 0.420786058227829220f,
	0.420411139114805770f,
	0.420036204544466940f, 0.419661254530597550f, 0.419286289086983070f,
	0.418911308227410740f,
	0.418536311965666650f, 0.418161300315539220f, 0.417786273290816130f,
	0.417411230905285650f,
	0.417036173172737830f, 0.416661100106961610f, 0.416286011721748230f,
	0.415910908030888200f,
	0.415535789048172620f, 0.415160654787394280f, 0.414785505262345030f,
	0.414410340486818910f,
	0.414035160474608700f, 0.413659965239509710f, 0.413284754795316230f,
	0.412909529155823300f,
	0.412534288334827750f, 0.412159032346125280f, 0.411783761203513790f,
	0.411408474920790520f,
	0.411033173511753220f, 0.410657856990201580f, 0.410282525369933980f,
	0.409907178664751180f,
	0.409531816888453190f, 0.409156440054840590f, 0.408781048177715660f,
	0.408405641270879690f,
	0.408030219348136270f, 0.407654782423288010f, 0.407279330510138260f,
	0.406903863622492260f,
	0.406528381774153900f, 0.406152884978929480f, 0.405777373250624070f,
	0.405401846603045010f,
	0.405026305049998980f, 0.404650748605293040f, 0.404275177282736260f,
	0.403899591096136380f,
	0.403523990059303620f, 0.403148374186047210f, 0.402772743490177110f,
	0.402397097985504990f,
	0.402021437685841480f, 0.401645762604999350f, 0.401270072756790610f,
	0.400894368155027990f,
	0.400518648813525830f, 0.400142914746097480f, 0.399767165966558420f,
	0.399391402488723400f,
	0.399015624326407800f, 0.398639831493428740f, 0.398264024003602220f,
	0.397888201870746420f,
	0.397512365108678430f, 0.397136513731217500f, 0.396760647752182230f,
	0.396384767185391620f,
	0.396008872044666730f, 0.395632962343827170f, 0.395257038096694990f,
	0.394881099317091370f,
	0.394505146018838130f, 0.394129178215758820f, 0.393753195921675850f,
	0.393377199150413860f,
	0.393001187915796750f, 0.392625162231649010f, 0.392249122111796800f,
	0.391873067570065240f,
	0.391496998620281590f, 0.391120915276272410f, 0.390744817551864850f,
	0.390368705460887750f,
	0.389992579017168830f, 0.389616438234538010f, 0.389240283126824070f,
	0.388864113707858060f,
	0.388487929991470140f, 0.388111731991491180f, 0.387735519721753690f,
	0.387359293196089140f,
	0.386983052428331030f, 0.386606797432312350f, 0.386230528221866430f,
	0.385854244810828530f,
	0.385477947213032580f, 0.385101635442314900f, 0.384725309512510880f,
	0.384348969437456610f,
	0.383972615230989860f, 0.383596246906947210f, 0.383219864479167560f,
	0.382843467961488940f,
	0.382467057367749940f, 0.382090632711791060f, 0.381714194007451380f,
	0.381337741268572390f,
	0.380961274508994250f, 0.380584793742559550f, 0.380208298983109930f,
	0.379831790244487540f,
	0.379455267540536490f, 0.379078730885099520f, 0.378702180292021630f,
	0.378325615775147170f,
	0.377949037348320800f, 0.377572445025389230f, 0.377195838820197690f,
	0.376819218746593910f,
	0.376442584818424570f, 0.376065937049537060f, 0.375689275453780500f,
	0.375312600045002780f,
	0.374935910837054080f, 0.374559207843783660f, 0.374182491079041500f,
	0.373805760556679190f,
	0.373429016290547200f, 0.373052258294498230f, 0.372675486582383640f,
	0.372298701168057190f,
	0.371921902065371730f, 0.371545089288180640f, 0.371168262850339210f,
	0.370791422765701320f,
	0.370414569048123140f, 0.370037701711460170f, 0.369660820769568240f,
	0.369283926236305070f,
	0.368907018125527120f, 0.368530096451093140f, 0.368153161226860980f,
	0.367776212466689010f,
	0.367399250184437480f, 0.367022274393965340f, 0.366645285109133750f,
	0.366268282343803150f,
	0.365891266111834370f, 0.365514236427090080f, 0.365137193303431750f,
	0.364760136754723020f,
	0.364383066794826350f, 0.364005983437606320f, 0.363628886696926890f,
	0.363251776586652310f,
	0.362874653120648700f, 0.362497516312780990f, 0.362120366176916230f,
	0.361743202726920790f,
	0.361366025976661450f, 0.360988835940006750f, 0.360611632630824020f,
	0.360234416062982840f,
	0.359857186250351960f, 0.359479943206800550f, 0.359102686946199680f,
	0.358725417482419150f,
	0.358348134829330870f, 0.357970839000806010f, 0.357593530010716310f,
	0.357216207872935120f,
	0.356838872601334680f, 0.356461524209789380f, 0.356084162712172360f,
	0.355706788122359060f,
	0.355329400454223950f, 0.354951999721642100f, 0.354574585938490280f,
	0.354197159118644080f,
	0.353819719275981330f, 0.353442266424378930f, 0.353064800577714280f,
	0.352687321749866610f,
	0.352309829954713830f, 0.351932325206136210f, 0.351554807518012990f,
	0.351177276904224070f,
	0.350799733378650890f, 0.350422176955173910f, 0.350044607647675640f,
	0.349667025470037810f,
	0.349289430436142520f, 0.348911822559873850f, 0.348534201855114360f,
	0.348156568335749040f,
	0.347778922015661520f, 0.347401262908737570f, 0.347023591028862320f,
	0.346645906389921150f,
	0.346268209005801410f, 0.345890498890388980f, 0.345512776057572080f,
	0.345135040521238170f,
	0.344757292295274910f, 0.344379531393571970f, 0.344001757830017680f,
	0.343623971618502560f,
	0.343246172772916250f, 0.342868361307148980f, 0.342490537235092600f,
	0.342112700570637750f,
	0.341734851327677280f, 0.341356989520103240f, 0.340979115161808070f,
	0.340601228266685980f,
	0.340223328848629880f, 0.339845416921535030f, 0.339467492499295200f,
	0.339089555595806560f,
	0.338711606224964210f, 0.338333644400663940f, 0.337955670136803170f,
	0.337577683447278010f,
	0.337199684345986910f, 0.336821672846827290f, 0.336443648963697160f,
	0.336065612710496290f,
	0.335687564101123050f, 0.335309503149478110f, 0.334931429869461230f,
	0.334553344274972690f,
	0.334175246379914470f, 0.333797136198187240f, 0.333419013743693980f,
	0.333040879030336690f,
	0.332662732072017800f, 0.332284572882641680f, 0.331906401476111280f,
	0.331528217866331690f,
	0.331150022067206780f, 0.330771814092642610f, 0.330393593956544440f,
	0.330015361672817750f,
	0.329637117255370090f, 0.329258860718107450f, 0.328880592074938190f,
	0.328502311339769700f,
	0.328124018526509800f, 0.327745713649068180f, 0.327367396721353070f,
	0.326989067757275040f,
	0.326610726770743760f, 0.326232373775669270f, 0.325854008785963320f,
	0.325475631815536570f,
	0.325097242878301660f, 0.324718841988170470f, 0.324340429159055250f,
	0.323962004404870050f,
	0.323583567739527570f, 0.323205119176942720f, 0.322826658731029110f,
	0.322448186415702550f,
	0.322069702244877910f, 0.321691206232470550f, 0.321312698392397570f,
	0.320934178738574720f,
	0.320555647284919980f, 0.320177104045350440f, 0.319798549033783570f,
	0.319419982264138650f,
	0.319041403750333630f, 0.318662813506288670f, 0.318284211545923010f,
	0.317905597883156250f,
	0.317526972531909870f, 0.317148335506103940f, 0.316769686819660780f,
	0.316391026486501690f,
	0.316012354520548600f, 0.315633670935725030f, 0.315254975745953180f,
	0.314876268965157470f,
	0.314497550607261090f, 0.314118820686189180f, 0.313740079215866160f,
	0.313361326210216840f,
	0.312982561683167790f, 0.312603785648644220f, 0.312224998120573420f,
	0.311846199112882030f,
	0.311467388639496860f, 0.311088566714346650f, 0.310709733351358600f,
	0.310330888564462340f,
	0.309952032367586390f, 0.309573164774659850f, 0.309194285799613390f,
	0.308815395456376430f,
	0.308436493758880660f, 0.308057580721056660f, 0.307678656356835560f,
	0.307299720680150270f,
	0.306920773704932260f, 0.306541815445115160f, 0.306162845914631390f,
	0.305783865127415400f,
	0.305404873097400780f, 0.305025869838521590f, 0.304646855364713530f,
	0.304267829689911010f,
	0.303888792828050650f, 0.303509744793068030f, 0.303130685598899270f,
	0.302751615259482190f,
	0.302372533788753170f, 0.301993441200650910f, 0.301614337509113100f,
	0.301235222728077840f,
	0.300856096871485010f, 0.300476959953273060f, 0.300097811987382670f,
	0.299718652987753580f,
	0.299339482968325970f, 0.298960301943041680f, 0.298581109925841300f,
	0.298201906930667390f,
	0.297822692971461410f, 0.297443468062166820f, 0.297064232216726120f,
	0.296684985449082390f,
	0.296305727773180260f, 0.295926459202963120f, 0.295547179752376430f,
	0.295167889435364820f,
	0.294788588265873170f, 0.294409276257848300f, 0.294029953425235520f,
	0.293650619781982260f,
	0.293271275342035120f, 0.292891920119341120f, 0.292512554127848930f,
	0.292133177381505850f,
	0.291753789894261320f, 0.291374391680063520f, 0.290994982752862730f,
	0.290615563126608250f,
	0.290236132815249790f, 0.289856691832738880f, 0.289477240193025510f,
	0.289097777910061970f,
	0.288718304997799550f, 0.288338821470189910f, 0.287959327341186510f,
	0.287579822624741350f,
	0.287200307334808670f, 0.286820781485341620f, 0.286441245090293950f,
	0.286061698163620930f,
	0.285682140719276560f, 0.285302572771216960f, 0.284922994333397350f,
	0.284543405419773240f,
	0.284163806044301910f, 0.283784196220939370f, 0.283404575963643550f,
	0.283024945286371230f,
	0.282645304203081090f, 0.282265652727731130f, 0.281885990874279570f,
	0.281506318656686290f,
	0.281126636088910030f, 0.280746943184911340f, 0.280367239958650150f,
	0.279987526424086530f,
	0.279607802595182420f, 0.279228068485898210f, 0.278848324110196550f,
	0.278468569482039130f,
	0.278088804615388040f, 0.277709029524206950f, 0.277329244222458250f,
	0.276949448724106480f,
	0.276569643043115150f, 0.276189827193448200f, 0.275810001189071290f,
	0.275430165043948570f,
	0.275050318772046500f, 0.274670462387330010f, 0.274290595903766200f,
	0.273910719335321300f,
	0.273530832695961790f, 0.273150935999655950f, 0.272771029260370560f,
	0.272391112492074590f,
	0.272011185708736060f, 0.271631248924323390f, 0.271251302152806570f,
	0.270871345408154380f,
	0.270491378704337540f, 0.270111402055325910f, 0.269731415475089780f,
	0.269351418977600950f,
	0.268971412576829990f, 0.268591396286749500f, 0.268211370121331170f,
	0.267831334094547010f,
	0.267451288220370730f, 0.267071232512774700f, 0.266691166985733360f,
	0.266311091653219700f,
	0.265931006529208920f, 0.265550911627675250f, 0.265170806962593210f,
	0.264790692547939020f,
	0.264410568397687560f, 0.264030434525815760f, 0.263650290946299660f,
	0.263270137673115630f,
	0.262889974720241610f, 0.262509802101654310f, 0.262129619831332370f,
	0.261749427923253670f,
	0.261369226391396310f, 0.260989015249740050f, 0.260608794512263380f,
	0.260228564192946710f,
	0.259848324305769600f, 0.259468074864711960f, 0.259087815883755400f,
	0.258707547376880010f,
	0.258327269358068100f, 0.257946981841300490f, 0.257566684840560170f,
	0.257186378369829110f,
	0.256806062443089680f, 0.256425737074325920f, 0.256045402277520320f,
	0.255665058066657680f,
	0.255284704455721660f, 0.254904341458696390f, 0.254523969089567590f,
	0.254143587362319620f,
	0.253763196290938850f, 0.253382795889410710f, 0.253002386171721110f,
	0.252621967151857420f,
	0.252241538843805680f, 0.251861101261554090f, 0.251480654419089730f,
	0.251100198330400150f,
	0.250719733009474530f, 0.250339258470300590f, 0.249958774726868170f,
	0.249578281793165680f,
	0.249197779683183660f, 0.248817268410911650f, 0.248436747990339490f,
	0.248056218435458720f,
	0.247675679760259450f, 0.247295131978733870f, 0.246914575104873220f,
	0.246534009152669040f,
	0.246153434136114490f, 0.245772850069201410f, 0.245392256965923620f,
	0.245011654840274010f,
	0.244631043706245800f, 0.244250423577833860f, 0.243869794469031620f,
	0.243489156393834590f,
	0.243108509366237320f, 0.242727853400234670f, 0.242347188509823150f,
	0.241966514708997830f,
	0.241585832011755900f, 0.241205140432093070f, 0.240824439984007180f,
	0.240443730681495050f,
	0.240063012538553830f, 0.239682285569182310f, 0.239301549787377890f,
	0.238920805207139960f,
	0.238540051842467020f, 0.238159289707357810f, 0.237778518815812740f,
	0.237397739181830820f,
	0.237016950819413100f, 0.236636153742559610f, 0.236255347965270780f,
	0.235874533501548580f,
	0.235493710365393630f, 0.235112878570808560f, 0.234732038131795020f,
	0.234351189062355030f,
	0.233970331376492150f, 0.233589465088208580f, 0.233208590211508550f,
	0.232827706760394850f,
	0.232446814748872410f, 0.232065914190945020f, 0.231685005100616930f,
	0.231304087491893930f,
	0.230923161378780380f, 0.230542226775282770f, 0.230161283695406500f,
	0.229780332153157300f,
	0.229399372162542610f, 0.229018403737568290f, 0.228637426892242400f,
	0.228256441640571880f,
	0.227875447996564060f, 0.227494445974227850f, 0.227113435587570770f,
	0.226732416850602300f,
	0.226351389777330990f, 0.225970354381765690f, 0.225589310677916880f,
	0.225208258679793520f,
	0.224827198401406690f, 0.224446129856766040f, 0.224065053059883250f,
	0.223683968024768950f,
	0.223302874765434120f, 0.222921773295891380f, 0.222540663630151820f,
	0.222159545782228660f,
	0.221778419766134050f, 0.221397285595880480f, 0.221016143285482050f,
	0.220634992848951380f,
	0.220253834300303180f, 0.219872667653551100f, 0.219491492922709110f,
	0.219110310121792800f,
	0.218729119264816280f, 0.218347920365795780f, 0.217966713438746380f,
	0.217585498497683580f,
	0.217204275556624420f, 0.216823044629584520f, 0.216441805730581500f,
	0.216060558873631570f,
	0.215679304072752960f, 0.215298041341962870f, 0.214916770695278810f,
	0.214535492146719880f,
	0.214154205710303750f, 0.213772911400050090f, 0.213391609229977570f,
	0.213010299214105140f,
	0.212628981366453330f, 0.212247655701041290f, 0.211866322231890090f,
	0.211484980973019880f,
	0.211103631938451000f, 0.210722275142205480f, 0.210340910598303870f,
	0.209959538320768660f,
	0.209578158323621420f, 0.209196770620883960f, 0.208815375226579670f,
	0.208433972154730530f,
	0.208052561419360520f, 0.207671143034492080f, 0.207289717014149830f,
	0.206908283372357230f,
	0.206526842123138070f, 0.206145393280517730f, 0.205763936858520150f,
	0.205382472871171230f,
	0.205001001332495910f, 0.204619522256519300f, 0.204238035657268250f,
	0.203856541548768030f,
	0.203475039945045950f, 0.203093530860128300f, 0.202712014308041620f,
	0.202330490302814110f,
	0.201948958858472420f, 0.201567419989045200f, 0.201185873708560170f,
	0.200804320031045230f,
	0.200422758970529910f, 0.200041190541042220f, 0.199659614756612230f,
	0.199278031631268500f,
	0.198896441179041650f, 0.198514843413961220f, 0.198133238350057030f,
	0.197751626001360480f,
	0.197370006381901520f, 0.196988379505712050f, 0.196606745386822960f,
	0.196225104039265410f,
	0.195843455477072190f, 0.195461799714274460f, 0.195080136764905570f,
	0.194698466642997730f,
	0.194316789362583340f, 0.193935104937696560f, 0.193553413382369890f,
	0.193171714710637930f,
	0.192790008936534220f, 0.192408296074092570f, 0.192026576137348330f,
	0.191644849140335360f,
	0.191263115097089540f, 0.190881374021645320f, 0.190499625928039040f,
	0.190117870830306100f,
	0.189736108742482030f, 0.189354339678604100f, 0.188972563652707950f,
	0.188590780678831250f,
	0.188208990771010640f, 0.187827193943283040f, 0.187445390209686870f,
	0.187063579584259070f,
	0.186681762081038650f, 0.186299937714063470f, 0.185918106497371700f,
	0.185536268445003070f,
	0.185154423570995760f, 0.184772571889390000f, 0.184390713414225000f,
	0.184008848159540110f,
	0.183626976139376310f, 0.183245097367773090f, 0.182863211858771880f,
	0.182481319626412670f,
	0.182099420684737420f, 0.181717515047787020f, 0.181335602729602590f,
	0.180953683744226880f,
	0.180571758105701030f, 0.180189825828068250f, 0.179807886925370670f,
	0.179425941411650660f,
	0.179043989300952110f, 0.178662030607317450f, 0.178280065344791100f,
	0.177898093527416370f,
	0.177516115169236820f, 0.177134130284297610f, 0.176752138886642350f,
	0.176370140990316640f,
	0.175988136609365020f, 0.175606125757832240f, 0.175224108449764660f,
	0.174842084699207030f,
	0.174460054520206240f, 0.174078017926807490f, 0.173695974933058080f,
	0.173313925553004180f,
	0.172931869800692250f, 0.172549807690170230f, 0.172167739235484620f,
	0.171785664450683800f,
	0.171403583349815180f, 0.171021495946926340f, 0.170639402256066410f,
	0.170257302291283000f,
	0.169875196066625710f, 0.169493083596143100f, 0.169110964893883830f,
	0.168728839973898290f,
	0.168346708850235140f, 0.167964571536945220f, 0.167582428048078130f,
	0.167200278397683750f,
	0.166818122599813570f, 0.166435960668517400f, 0.166053792617847200f,
	0.165671618461853270f,
	0.165289438214587970f, 0.164907251890102520f, 0.164525059502448390f,
	0.164142861065678550f,
	0.163760656593844480f, 0.163378446100999640f, 0.162996229601196390f,
	0.162614007108487250f,
	0.162231778636926370f, 0.161849544200566300f, 0.161467303813461580f,
	0.161085057489665670f,
	0.160702805243232240f, 0.160320547088216470f, 0.159938283038672050f,
	0.159556013108654580f,
	0.159173737312218650f, 0.158791455663418930f, 0.158409168176311760f,
	0.158026874864951870f,
	0.157644575743395960f, 0.157262270825699210f, 0.156879960125918730f,
	0.156497643658110590f,
	0.156115321436331000f, 0.155732993474637760f, 0.155350659787087090f,
	0.154968320387737170f,
	0.154585975290645110f, 0.154203624509868190f, 0.153821268059465250f,
	0.153438905953493550f,
	0.153056538206012340f, 0.152674164831079730f, 0.152291785842754070f,
	0.151909401255095250f,
	0.151527011082161540f, 0.151144615338013210f, 0.150762214036709470f,
	0.150379807192309620f,
	0.149997394818874590f, 0.149614976930463660f, 0.149232553541138180f,
	0.148850124664957870f,
	0.148467690315984390f, 0.148085250508278370f, 0.147702805255900570f,
	0.147320354572913260f,
	0.146937898473377210f, 0.146555436971355090f, 0.146172970080908520f,
	0.145790497816099230f,
	0.145408020190990560f, 0.145025537219644170f, 0.144643048916123810f,
	0.144260555294492000f,
	0.143878056368811510f, 0.143495552153146630f, 0.143113042661560050f,
	0.142730527908116440f,
	0.142348007906879320f, 0.141965482671912420f, 0.141582952217280980f,
	0.141200416557048680f,
	0.140817875705281120f, 0.140435329676042390f, 0.140052778483398480f,
	0.139670222141414250f,
	0.139287660664154770f, 0.138905094065686600f, 0.138522522360074780f,
	0.138139945561386200f,
	0.137757363683686740f, 0.137374776741042340f, 0.136992184747520560f,
	0.136609587717187310f,
	0.136226985664110460f, 0.135844378602356760f, 0.135461766545993150f,
	0.135079149509088060f,
	0.134696527505708320f, 0.134313900549922760f, 0.133931268655799020f,
	0.133548631837404950f,
	0.133165990108809860f, 0.132783343484081580f, 0.132400691977289760f,
	0.132018035602502530f,
	0.131635374373789940f, 0.131252708305220960f, 0.130870037410864640f,
	0.130487361704791580f,
	0.130104681201070800f, 0.129721995913773260f, 0.129339305856968730f,
	0.128956611044727220f,
	0.128573911491120210f, 0.128191207210217570f, 0.127808498216091110f,
	0.127425784522811530f,
	0.127043066144449680f, 0.126660343095077900f, 0.126277615388766920f,
	0.125894883039589430f,
	0.125512146061616980f, 0.125129404468921260f, 0.124746658275575490f,
	0.124363907495651240f,
	0.123981152143222060f, 0.123598392232359880f, 0.123215627777138580f,
	0.122832858791630880f,
	0.122450085289909640f, 0.122067307286049230f, 0.121684524794122440f,
	0.121301737828203960f,
	0.120918946402367330f, 0.120536150530686250f, 0.120153350227235940f,
	0.119770545506089950f,
	0.119387736381323830f, 0.119004922867011920f, 0.118622104977228730f,
	0.118239282726050290f,
	0.117856456127550970f, 0.117473625195807100f, 0.117090789944893860f,
	0.116707950388886520f,
	0.116325106541861910f, 0.115942258417895240f, 0.115559406031063570f,
	0.115176549395442460f,
	0.114793688525109290f, 0.114410823434140360f, 0.114027954136612060f,
	0.113645080646602280f,
	0.113262202978187320f, 0.112879321145445350f, 0.112496435162453430f,
	0.112113545043288730f,
	0.111730650802029900f, 0.111347752452754000f, 0.110964850009539970f,
	0.110581943486465610f,
	0.110199032897608850f, 0.109816118257049110f, 0.109433199578864170f,
	0.109050276877133770f,
	0.108667350165936400f, 0.108284419459350770f, 0.107901484771457020f,
	0.107518546116333660f,
	0.107135603508061170f, 0.106752656960718350f, 0.106369706488385940f,
	0.105986752105143480f,
	0.105603793825070680f, 0.105220831662248700f, 0.104837865630757090f,
	0.104454895744677270f,
	0.104071922018089540f, 0.103688944465074300f, 0.103305963099713400f,
	0.102922977936087120f,
	0.102539988988277600f, 0.102156996270365800f, 0.101773999796432830f,
	0.101390999580561250f,
	0.101007995636832020f, 0.100624987979327970f, 0.100241976622130760f,
	0.099858961579322170f,
	0.099475942864985456f, 0.099092920493202258f, 0.098709894478056073f,
	0.098326864833628791f,
	0.097943831574004214f, 0.097560794713264939f, 0.097177754265493674f,
	0.096794710244774623f,
	0.096411662665190329f, 0.096028611540825232f, 0.095645556885762609f,
	0.095262498714085819f,
	0.094879437039879722f, 0.094496371877227495f, 0.094113303240214247f,
	0.093730231142923864f,
	0.093347155599440373f, 0.092964076623849271f, 0.092580994230234359f,
	0.092197908432681386f,
	0.091814819245274432f, 0.091431726682099479f, 0.091048630757241303f,
	0.090665531484784803f,
	0.090282428878816323f, 0.089899322953420582f, 0.089516213722684160f,
	0.089133101200692441f,
	0.088749985401530951f, 0.088366866339286629f, 0.087983744028044805f,
	0.087600618481892656f,
	0.087217489714916191f, 0.086834357741201490f, 0.086451222574836131f,
	0.086068084229906014f,
	0.085684942720498897f, 0.085301798060701386f, 0.084918650264600160f,
	0.084535499346283349f,
	0.084152345319837438f, 0.083769188199350780f, 0.083386027998910095f,
	0.083002864732603973f,
	0.082619698414519799f, 0.082236529058745025f, 0.081853356679368619f,
	0.081470181290477811f,
	0.081087002906161790f, 0.080703821540508452f, 0.080320637207605849f,
	0.079937449921543474f,
	0.079554259696409127f, 0.079171066546292510f, 0.078787870485282088f,
	0.078404671527466441f,
	0.078021469686935602f, 0.077638264977777913f, 0.077255057414083589f,
	0.076871847009941652f,
	0.076488633779441206f, 0.076105417736672773f, 0.075722198895725248f,
	0.075338977270689375f,
	0.074955752875654230f, 0.074572525724710764f, 0.074189295831948693f,
	0.073806063211457842f,
	0.073422827877329483f, 0.073039589843653177f, 0.072656349124520389f,
	0.072273105734021334f,
	0.071889859686246352f, 0.071506610995287156f, 0.071123359675233852f,
	0.070740105740178361f,
	0.070356849204211397f, 0.069973590081423773f, 0.069590328385907715f,
	0.069207064131753759f,
	0.068823797333054326f, 0.068440528003900616f, 0.068057256158383886f,
	0.067673981810596848f,
	0.067290704974630494f, 0.066907425664577733f, 0.066524143894529736f,
	0.066140859678579578f,
	0.065757573030819083f, 0.065374283965340146f, 0.064990992496236119f,
	0.064607698637598646f,
	0.064224402403521202f, 0.063841103808096086f, 0.063457802865415636f,
	0.063074499589573618f,
	0.062691193994662109f, 0.062307886094775049f, 0.061924575904005130f,
	0.061541263436445129f,
	0.061157948706189229f, 0.060774631727329942f, 0.060391312513961619f,
	0.060007991080177375f,
	0.059624667440070382f, 0.059241341607735261f, 0.058858013597264912f,
	0.058474683422754095f,
	0.058091351098295878f, 0.057708016637985186f, 0.057324680055915692f,
	0.056941341366181127f,
	0.056558000582876661f, 0.056174657720095743f, 0.055791312791933681f,
	0.055407965812484541f,
	0.055024616795842439f, 0.054641265756102911f, 0.054257912707359794f,
	0.053874557663708772f,
	0.053491200639244271f, 0.053107841648060788f, 0.052724480704254229f,
	0.052341117821918783f,
	0.051957753015150501f, 0.051574386298044173f, 0.051191017684694640f,
	0.050807647189198162f,
	0.050424274825649297f, 0.050040900608144430f, 0.049657524550778251f,
	0.049274146667647289f,
	0.048890766972846805f, 0.048507385480472134f, 0.048124002204620014f,
	0.047740617159385448f,
	0.047357230358865306f, 0.046973841817155179f, 0.046590451548350717f,
	0.046207059566548990f,
	0.045823665885845313f, 0.045440270520336883f, 0.045056873484119603f,
	0.044673474791289434f,
	0.044290074455943754f, 0.043906672492178188f, 0.043523268914090238f,
	0.043139863735776100f,
	0.042756456971332048f, 0.042373048634855741f, 0.041989638740443119f,
	0.041606227302191955f,
	0.041222814334198304f, 0.040839399850560058f, 0.040455983865373815f,
	0.040072566392736257f,
	0.039689147446745419f, 0.039305727041497644f, 0.038922305191091085f,
	0.038538881909622631f,
	0.038155457211189216f, 0.037772031109889144f, 0.037388603619819022f,
	0.037005174755077273f,
	0.036621744529761024f, 0.036238312957967478f, 0.035854880053795196f,
	0.035471445831341021f,
	0.035088010304703626f, 0.034704573487980395f, 0.034321135395268765f,
	0.033937696040667535f,
	0.033554255438273790f, 0.033170813602186440f, 0.032787370546502645f,
	0.032403926285321405f,
	0.032020480832740429f, 0.031637034202857461f, 0.031253586409771626f,
	0.030870137467580314f,
	0.030486687390382738f, 0.030103236192276818f, 0.029719783887360508f,
	0.029336330489733147f,
	0.028952876013492331f, 0.028569420472737472f, 0.028185963881566689f,
	0.027802506254078142f,
	0.027419047604371360f, 0.027035587946544135f, 0.026652127294696067f,
	0.026268665662925468f,
	0.025885203065330677f, 0.025501739516011413f, 0.025118275029065638f,
	0.024734809618593138f,
	0.024351343298691951f, 0.023967876083461924f, 0.023584407987001611f,
	0.023200939023409587f,
	0.022817469206785804f, 0.022433998551228459f, 0.022050527070837558f,
	0.021667054779711814f,
	0.021283581691949955f, 0.020900107821652084f, 0.020516633182916549f,
	0.020133157789843505f,
	0.019749681656531803f, 0.019366204797080316f, 0.018982727225589285f,
	0.018599248956157190f,
	0.018215770002884327f, 0.017832290379869671f, 0.017448810101212228f,
	0.017065329181012358f,
	0.016681847633368677f, 0.016298365472381587f, 0.015914882712149747f,
	0.015531399366773606f,
	0.015147915450352307f, 0.014764430976985016f, 0.014380945960772247f,
	0.013997460415812761f,
	0.013613974356207112f, 0.013230487796054543f, 0.012847000749454314f,
	0.012463513230507034f,
	0.012080025253311559f, 0.011696536831968529f, 0.011313047980577277f,
	0.010929558713237145f,
	0.010546069044048827f, 0.010162578987111254f, 0.009779088556525145f,
	0.009395597766389905f,
	0.009012106630804949f, 0.008628615163871038f, 0.008245123379687167f,
	0.007861631292354124f,
	0.007478138915970929f, 0.007094646264638386f, 0.006711153352455981f,
	0.006327660193523208f,
	0.005944166801940901f, 0.005560673191808128f, 0.005177179377225743f,
	0.004793685372293270f,
	0.004410191191110246f, 0.004026696847777542f, 0.003643202356394263f,
	0.003259707731061291f,
	0.002876212985878184f, 0.002492718134944503f, 0.002109223192361147f,
	0.001725728172227238f,
	0.001342233088643682f, 0.000958737955710053f, 0.000575242787525925f,
	0.000191747598192208f
};

/**
 * @} end of DCT4_IDCT4_Table group
 */

/**
 * @addtogroup DCT4_IDCT4
 * @{
 */

/**
 * @brief  Initialization function for the floating-point DCT4/IDCT4.
 * @param[in,out] *S         points to an instance of floating-point DCT4/IDCT4 structure.
 * @param[in]     *S_RFFT    points to an instance of floating-point RFFT/RIFFT structure.
 * @param[in]     *S_CFFT    points to an instance of floating-point CFFT/CIFFT structure.
 * @param[in]     N			 length of the DCT4.
 * @param[in]     Nby2       half of the length of the DCT4.
 * @param[in]     normalize  normalizing factor.
 * @return        arm_status function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>fftLenReal</code> is not a supported transform length.
 * \par Normalizing factor:
 * The normalizing factor is <code>sqrt(2/N)</code>, which depends on the size of transform <code>N</code>.
 * Floating-point normalizing factors are mentioned in the table below for different DCT sizes:
 * \image html dct4NormalizingF32Table.gif
 */

arm_status arm_dct4_init_f32(
	arm_dct4_instance_f32 *S,
	arm_rfft_instance_f32 *S_RFFT,
	arm_cfft_radix4_instance_f32 *S_CFFT,
	uint16_t N,
	uint16_t Nby2,
	float32_t normalize)
{
	/*  Initialize the default arm status */
	arm_status status = ARM_MATH_SUCCESS;

	/* Initializing the pointer array with the weight table base addresses of different lengths */
	float32_t *twiddlePtr[4] = {
		(float32_t *) Weights_128, (float32_t *) Weights_512,
		(float32_t *) Weights_2048, (float32_t *) Weights_8192
	};

	/* Initializing the pointer array with the cos factor table base addresses of different lengths */
	float32_t *pCosFactor[4] = {
		(float32_t *) cos_factors_128, (float32_t *) cos_factors_512,
		(float32_t *) cos_factors_2048, (float32_t *) cos_factors_8192
	};

	/* Initialize the DCT4 length */
	S->N = N;

	/* Initialize the half of DCT4 length */
	S->Nby2 = Nby2;

	/* Initialize the DCT4 Normalizing factor */
	S->normalize = normalize;

	/* Initialize Real FFT Instance */
	S->pRfft = S_RFFT;

	/* Initialize Complex FFT Instance */
	S->pCfft = S_CFFT;

	switch (N) {
	/* Initialize the table modifier values */
	case 8192U:
		S->pTwiddle = twiddlePtr[3];
		S->pCosFactor = pCosFactor[3];
		break;
	case 2048U:
		S->pTwiddle = twiddlePtr[2];
		S->pCosFactor = pCosFactor[2];
		break;
	case 512U:
		S->pTwiddle = twiddlePtr[1];
		S->pCosFactor = pCosFactor[1];
		break;
	case 128U:
		S->pTwiddle = twiddlePtr[0];
		S->pCosFactor = pCosFactor[0];
		break;
	default:
		status = ARM_MATH_ARGUMENT_ERROR;
	}

	/* Initialize the RFFT/RIFFT */
	arm_rfft_init_f32(S->pRfft, S->pCfft, S->N, 0U, 1U);

	/* return the status of DCT4 Init function */
	return (status);
}

/**
 * @} end of DCT4_IDCT4 group
 */
