

#include "serial_api.h"
#include "crypto_api.h"
#include "memory.h"
#include "platform_stdlib.h"

#include "FreeRTOS.h"
#include "task.h"
#include "device_lock.h"

#define STACKSIZE 2048

#undef __crypto_mem_dump
#define __crypto_mem_dump(start,size,prefix) do{ \
dbg_printf(prefix "\r\n"); \
  dump_bytes(start,size); \
}while(0)

#define AUTH_SHA2_256            0
#define AUTH_HMAC_SHA2_256       1

#define CIPHER_AES_256_ECB       0
#define CIPHER_AES_256_CBC       1

const u8 plaintext[] = "12345678901234567890123456789012345678901234567890123456789012" \
					   "345678901234567890";
const u8 md5_digest[] = {0x57, 0xED, 0xF4, 0xA2, 0x2B, 0xE3, 0xC9, 0x55,
						 0xAC, 0x49, 0xDA, 0x2E, 0x21, 0x07, 0xB6, 0x7A
						};

const u8 auth_plaintext_test_buf[14][81] = {
	{ "a" },
	{ "abc" },
	{ "message digest" },
	{ "abcdefghijklmnopqrstuvwxyz" },
	{ "The quick brown fox jumps over the lazy dog" },
	{ "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789" },
	{
		"12345678901234567890123456789012345678901234567890123456789012" \
		"345678901234567890"
	},
	{ "a" },
	{ "abc" },
	{ "message digest" },
	{ "abcdefghijklmnopqrstuvwxyz" },
	{ "The quick brown fox jumps over the lazy dog" },
	{ "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789" },
	{
		"12345678901234567890123456789012345678901234567890123456789012" \
		"345678901234567890"
	}
};

const int auth_plaintext_test_buflen[14] = {
	1, 3, 14, 26, 43, 62, 80, 1, 3, 14, 26, 43, 62, 80
};

const u8 md5_test_sum[14][16] = {
	{
		0x0C, 0xC1, 0x75, 0xB9, 0xC0, 0xF1, 0xB6, 0xA8,
		0x31, 0xC3, 0x99, 0xE2, 0x69, 0x77, 0x26, 0x61
	},
	{
		0x90, 0x01, 0x50, 0x98, 0x3C, 0xD2, 0x4F, 0xB0,
		0xD6, 0x96, 0x3F, 0x7D, 0x28, 0xE1, 0x7F, 0x72
	},
	{
		0xF9, 0x6B, 0x69, 0x7D, 0x7C, 0xB7, 0x93, 0x8D,
		0x52, 0x5A, 0x2F, 0x31, 0xAA, 0xF1, 0x61, 0xD0
	},
	{
		0xC3, 0xFC, 0xD3, 0xD7, 0x61, 0x92, 0xE4, 0x00,
		0x7D, 0xFB, 0x49, 0x6C, 0xCA, 0x67, 0xE1, 0x3B
	},
	{
		"\x9e\x10\x7d\x9d\x37\x2b\xb6\x82"
		"\x6b\xd8\x1d\x35\x42\xa4\x19\xd6"
	},
	{
		0xD1, 0x74, 0xAB, 0x98, 0xD2, 0x77, 0xD9, 0xF5,
		0xA5, 0x61, 0x1C, 0x2C, 0x9F, 0x41, 0x9D, 0x9F
	},
	{
		0x57, 0xED, 0xF4, 0xA2, 0x2B, 0xE3, 0xC9, 0x55,
		0xAC, 0x49, 0xDA, 0x2E, 0x21, 0x07, 0xB6, 0x7A
	},
	{
		0x0C, 0xC1, 0x75, 0xB9, 0xC0, 0xF1, 0xB6, 0xA8,
		0x31, 0xC3, 0x99, 0xE2, 0x69, 0x77, 0x26, 0x61
	},
	{
		0x90, 0x01, 0x50, 0x98, 0x3C, 0xD2, 0x4F, 0xB0,
		0xD6, 0x96, 0x3F, 0x7D, 0x28, 0xE1, 0x7F, 0x72
	},
	{
		0xF9, 0x6B, 0x69, 0x7D, 0x7C, 0xB7, 0x93, 0x8D,
		0x52, 0x5A, 0x2F, 0x31, 0xAA, 0xF1, 0x61, 0xD0
	},
	{
		0xC3, 0xFC, 0xD3, 0xD7, 0x61, 0x92, 0xE4, 0x00,
		0x7D, 0xFB, 0x49, 0x6C, 0xCA, 0x67, 0xE1, 0x3B
	},
	{
		"\x9e\x10\x7d\x9d\x37\x2b\xb6\x82"
		"\x6b\xd8\x1d\x35\x42\xa4\x19\xd6"
	},
	{
		0xD1, 0x74, 0xAB, 0x98, 0xD2, 0x77, 0xD9, 0xF5,
		0xA5, 0x61, 0x1C, 0x2C, 0x9F, 0x41, 0x9D, 0x9F
	},
	{
		0x57, 0xED, 0xF4, 0xA2, 0x2B, 0xE3, 0xC9, 0x55,
		0xAC, 0x49, 0xDA, 0x2E, 0x21, 0x07, 0xB6, 0x7A
	}
};

const u8 sha2_256_digest[2][32] = {
	{
		0xf3, 0x71, 0xbc, 0x4a, 0x31, 0x1f, 0x2b, 0x00,
		0x9e, 0xef, 0x95, 0x2d, 0xd8, 0x3c, 0xa8, 0x0e,
		0x2b, 0x60, 0x02, 0x6c, 0x8e, 0x93, 0x55, 0x92,
		0xd0, 0xf9, 0xc3, 0x08, 0x45, 0x3c, 0x81, 0x3e
	},
	{
		0x74, 0xfd, 0x17, 0x5c, 0xd9, 0x44, 0x8e, 0x6a,
		0x4f, 0x0f, 0xde, 0x8e, 0x95, 0xc5, 0x05, 0x50,
		0xaa, 0x0a, 0x73, 0xeb, 0xc0, 0x58, 0x9c, 0xf3,
		0xc1, 0x9b, 0x4e, 0x3e, 0x14, 0xd1, 0x80, 0xa8
	}
};

// key start address needs to be 32bytes aligned
const u8 sha2_256_hmac_key[] __attribute__((aligned(32))) = "key";

const u8 sha2_256_test_sum[2][14][32] = {
	{
		{
			0xca, 0x97, 0x81, 0x12, 0xca, 0x1b, 0xbd, 0xca,
			0xfa, 0xc2, 0x31, 0xb3, 0x9a, 0x23, 0xdc, 0x4d,
			0xa7, 0x86, 0xef, 0xf8, 0x14, 0x7c, 0x4e, 0x72,
			0xb9, 0x80, 0x77, 0x85, 0xaf, 0xee, 0x48, 0xbb
		},
		{
			0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
			0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
			0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
			0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad
		},
		{
			0xf7, 0x84, 0x6f, 0x55, 0xcf, 0x23, 0xe1, 0x4e,
			0xeb, 0xea, 0xb5, 0xb4, 0xe1, 0x55, 0x0c, 0xad,
			0x5b, 0x50, 0x9e, 0x33, 0x48, 0xfb, 0xc4, 0xef,
			0xa3, 0xa1, 0x41, 0x3d, 0x39, 0x3c, 0xb6, 0x50
		},
		{
			0x71, 0xc4, 0x80, 0xdf, 0x93, 0xd6, 0xae, 0x2f,
			0x1e, 0xfa, 0xd1, 0x44, 0x7c, 0x66, 0xc9, 0x52,
			0x5e, 0x31, 0x62, 0x18, 0xcf, 0x51, 0xfc, 0x8d,
			0x9e, 0xd8, 0x32, 0xf2, 0xda, 0xf1, 0x8b, 0x73
		},
		{
			0xd7, 0xa8, 0xfb, 0xb3, 0x07, 0xd7, 0x80, 0x94,
			0x69, 0xca, 0x9a, 0xbc, 0xb0, 0x08, 0x2e, 0x4f,
			0x8d, 0x56, 0x51, 0xe4, 0x6d, 0x3c, 0xdb, 0x76,
			0x2d, 0x02, 0xd0, 0xbf, 0x37, 0xc9, 0xe5, 0x92
		},
		{
			0xdb, 0x4b, 0xfc, 0xbd, 0x4d, 0xa0, 0xcd, 0x85,
			0xa6, 0x0c, 0x3c, 0x37, 0xd3, 0xfb, 0xd8, 0x80,
			0x5c, 0x77, 0xf1, 0x5f, 0xc6, 0xb1, 0xfd, 0xfe,
			0x61, 0x4e, 0xe0, 0xa7, 0xc8, 0xfd, 0xb4, 0xc0
		},
		{
			0xf3, 0x71, 0xbc, 0x4a, 0x31, 0x1f, 0x2b, 0x00,
			0x9e, 0xef, 0x95, 0x2d, 0xd8, 0x3c, 0xa8, 0x0e,
			0x2b, 0x60, 0x02, 0x6c, 0x8e, 0x93, 0x55, 0x92,
			0xd0, 0xf9, 0xc3, 0x08, 0x45, 0x3c, 0x81, 0x3e
		},
		{
			0xca, 0x97, 0x81, 0x12, 0xca, 0x1b, 0xbd, 0xca,
			0xfa, 0xc2, 0x31, 0xb3, 0x9a, 0x23, 0xdc, 0x4d,
			0xa7, 0x86, 0xef, 0xf8, 0x14, 0x7c, 0x4e, 0x72,
			0xb9, 0x80, 0x77, 0x85, 0xaf, 0xee, 0x48, 0xbb
		},
		{
			0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
			0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
			0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
			0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad
		},
		{
			0xf7, 0x84, 0x6f, 0x55, 0xcf, 0x23, 0xe1, 0x4e,
			0xeb, 0xea, 0xb5, 0xb4, 0xe1, 0x55, 0x0c, 0xad,
			0x5b, 0x50, 0x9e, 0x33, 0x48, 0xfb, 0xc4, 0xef,
			0xa3, 0xa1, 0x41, 0x3d, 0x39, 0x3c, 0xb6, 0x50
		},
		{
			0x71, 0xc4, 0x80, 0xdf, 0x93, 0xd6, 0xae, 0x2f,
			0x1e, 0xfa, 0xd1, 0x44, 0x7c, 0x66, 0xc9, 0x52,
			0x5e, 0x31, 0x62, 0x18, 0xcf, 0x51, 0xfc, 0x8d,
			0x9e, 0xd8, 0x32, 0xf2, 0xda, 0xf1, 0x8b, 0x73
		},
		{
			0xd7, 0xa8, 0xfb, 0xb3, 0x07, 0xd7, 0x80, 0x94,
			0x69, 0xca, 0x9a, 0xbc, 0xb0, 0x08, 0x2e, 0x4f,
			0x8d, 0x56, 0x51, 0xe4, 0x6d, 0x3c, 0xdb, 0x76,
			0x2d, 0x02, 0xd0, 0xbf, 0x37, 0xc9, 0xe5, 0x92
		},
		{
			0xdb, 0x4b, 0xfc, 0xbd, 0x4d, 0xa0, 0xcd, 0x85,
			0xa6, 0x0c, 0x3c, 0x37, 0xd3, 0xfb, 0xd8, 0x80,
			0x5c, 0x77, 0xf1, 0x5f, 0xc6, 0xb1, 0xfd, 0xfe,
			0x61, 0x4e, 0xe0, 0xa7, 0xc8, 0xfd, 0xb4, 0xc0
		},
		{
			0xf3, 0x71, 0xbc, 0x4a, 0x31, 0x1f, 0x2b, 0x00,
			0x9e, 0xef, 0x95, 0x2d, 0xd8, 0x3c, 0xa8, 0x0e,
			0x2b, 0x60, 0x02, 0x6c, 0x8e, 0x93, 0x55, 0x92,
			0xd0, 0xf9, 0xc3, 0x08, 0x45, 0x3c, 0x81, 0x3e
		}
	},
	{
		{
			0x78, 0x0c, 0x3d, 0xb4, 0xce, 0x3d, 0xe5, 0xb9,
			0xe5, 0x58, 0x16, 0xfb, 0xa9, 0x8f, 0x59, 0x06,
			0x31, 0xd9, 0x6c, 0x07, 0x52, 0x71, 0xb2, 0x69,
			0x76, 0x23, 0x8d, 0x5f, 0x44, 0x44, 0x21, 0x9b
		},
		{
			0x9c, 0x19, 0x6e, 0x32, 0xdc, 0x01, 0x75, 0xf8,
			0x6f, 0x4b, 0x1c, 0xb8, 0x92, 0x89, 0xd6, 0x61,
			0x9d, 0xe6, 0xbe, 0xe6, 0x99, 0xe4, 0xc3, 0x78,
			0xe6, 0x83, 0x09, 0xed, 0x97, 0xa1, 0xa6, 0xab
		},
		{
			0x32, 0xb3, 0x13, 0x60, 0x74, 0x0c, 0xca, 0x47,
			0x1a, 0x8c, 0x52, 0x8d, 0xcc, 0x26, 0x21, 0x8e,
			0x69, 0x79, 0xe4, 0xf7, 0x96, 0x27, 0xdf, 0xff,
			0x08, 0x2d, 0xa9, 0x89, 0xa6, 0xbd, 0x3d, 0xa6
		},
		{
			0xd9, 0xe4, 0xab, 0xfa, 0xc1, 0xbc, 0xff, 0xec,
			0x48, 0x83, 0x44, 0x77, 0xf1, 0xbc, 0xf9, 0x04,
			0x03, 0x06, 0xec, 0x09, 0x12, 0xcf, 0xb7, 0xaf,
			0xe2, 0x8f, 0x1c, 0xa7, 0xea, 0x5c, 0x26, 0x0f
		},
		{
			0xf7, 0xbc, 0x83, 0xf4, 0x30, 0x53, 0x84, 0x24,
			0xb1, 0x32, 0x98, 0xe6, 0xaa, 0x6f, 0xb1, 0x43,
			0xef, 0x4d, 0x59, 0xa1, 0x49, 0x46, 0x17, 0x59,
			0x97, 0x47, 0x9d, 0xbc, 0x2d, 0x1a, 0x3c, 0xd8
		},
		{
			0xa6, 0x70, 0xf3, 0xd1, 0x51, 0x48, 0x5f, 0x0c,
			0x48, 0x61, 0x85, 0x47, 0x3c, 0x94, 0xe2, 0x02,
			0xe0, 0x63, 0xdb, 0xc4, 0x3d, 0xc8, 0x3e, 0x1f,
			0x0c, 0xbf, 0x89, 0xfa, 0x87, 0xdf, 0xd9, 0xd7
		},
		{
			0x74, 0xfd, 0x17, 0x5c, 0xd9, 0x44, 0x8e, 0x6a,
			0x4f, 0x0f, 0xde, 0x8e, 0x95, 0xc5, 0x05, 0x50,
			0xaa, 0x0a, 0x73, 0xeb, 0xc0, 0x58, 0x9c, 0xf3,
			0xc1, 0x9b, 0x4e, 0x3e, 0x14, 0xd1, 0x80, 0xa8
		},
		{
			0x78, 0x0c, 0x3d, 0xb4, 0xce, 0x3d, 0xe5, 0xb9,
			0xe5, 0x58, 0x16, 0xfb, 0xa9, 0x8f, 0x59, 0x06,
			0x31, 0xd9, 0x6c, 0x07, 0x52, 0x71, 0xb2, 0x69,
			0x76, 0x23, 0x8d, 0x5f, 0x44, 0x44, 0x21, 0x9b
		},
		{
			0x9c, 0x19, 0x6e, 0x32, 0xdc, 0x01, 0x75, 0xf8,
			0x6f, 0x4b, 0x1c, 0xb8, 0x92, 0x89, 0xd6, 0x61,
			0x9d, 0xe6, 0xbe, 0xe6, 0x99, 0xe4, 0xc3, 0x78,
			0xe6, 0x83, 0x09, 0xed, 0x97, 0xa1, 0xa6, 0xab
		},
		{
			0x32, 0xb3, 0x13, 0x60, 0x74, 0x0c, 0xca, 0x47,
			0x1a, 0x8c, 0x52, 0x8d, 0xcc, 0x26, 0x21, 0x8e,
			0x69, 0x79, 0xe4, 0xf7, 0x96, 0x27, 0xdf, 0xff,
			0x08, 0x2d, 0xa9, 0x89, 0xa6, 0xbd, 0x3d, 0xa6
		},
		{
			0xd9, 0xe4, 0xab, 0xfa, 0xc1, 0xbc, 0xff, 0xec,
			0x48, 0x83, 0x44, 0x77, 0xf1, 0xbc, 0xf9, 0x04,
			0x03, 0x06, 0xec, 0x09, 0x12, 0xcf, 0xb7, 0xaf,
			0xe2, 0x8f, 0x1c, 0xa7, 0xea, 0x5c, 0x26, 0x0f
		},
		{
			0xf7, 0xbc, 0x83, 0xf4, 0x30, 0x53, 0x84, 0x24,
			0xb1, 0x32, 0x98, 0xe6, 0xaa, 0x6f, 0xb1, 0x43,
			0xef, 0x4d, 0x59, 0xa1, 0x49, 0x46, 0x17, 0x59,
			0x97, 0x47, 0x9d, 0xbc, 0x2d, 0x1a, 0x3c, 0xd8
		},
		{
			0xa6, 0x70, 0xf3, 0xd1, 0x51, 0x48, 0x5f, 0x0c,
			0x48, 0x61, 0x85, 0x47, 0x3c, 0x94, 0xe2, 0x02,
			0xe0, 0x63, 0xdb, 0xc4, 0x3d, 0xc8, 0x3e, 0x1f,
			0x0c, 0xbf, 0x89, 0xfa, 0x87, 0xdf, 0xd9, 0xd7
		},
		{
			0x74, 0xfd, 0x17, 0x5c, 0xd9, 0x44, 0x8e, 0x6a,
			0x4f, 0x0f, 0xde, 0x8e, 0x95, 0xc5, 0x05, 0x50,
			0xaa, 0x0a, 0x73, 0xeb, 0xc0, 0x58, 0x9c, 0xf3,
			0xc1, 0x9b, 0x4e, 0x3e, 0x14, 0xd1, 0x80, 0xa8
		}
	}
};

u8 digest[64];
u8 cipher_result[1024];
u8 tag[16];

serial_t    sobj;

int test_md5(void)
{
	int i;
	int ret;
	u8 md5sum[16];

	dbg_printf("MD5 test \r\n");
	memset(digest, 0, sizeof(digest));
	device_mutex_lock(RT_DEV_LOCK_CRYPTO);
	ret = crypto_md5(plaintext, strlen((const char *)plaintext), digest); // the length of MD5's digest is 16 bytes.
	device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
	if (SUCCESS != ret) {
		dbg_printf("crypto_md5 failed,ret = %d \r\n", ret);
		goto test_md5_end;
	}
	if (memcmp(digest, md5_digest, 16) == 0) {
		dbg_printf("MD5 test result is correct \r\n");
		ret = SUCCESS;
	} else {
		dbg_printf("MD5 test result is WRONG!! \r\n");
		__crypto_mem_dump((u8 *)md5_digest, 16, "sw digest");
		__crypto_mem_dump(digest, 16, "hw digest");
		ret = FAIL;
		goto test_md5_end;
	}

	for (i = 0; i < 14; i++) {
		dbg_printf(" MD5 test #%d: \r\n", (i + 1));
		device_mutex_lock(RT_DEV_LOCK_CRYPTO);
		ret = crypto_md5(auth_plaintext_test_buf[i], auth_plaintext_test_buflen[i], md5sum); // the length of MD5's digest is 16 bytes.
		device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
		if (SUCCESS != ret) {
			dbg_printf("crypto_md5 failed, ret = %d \r\n", ret);
			break;
		}

		if (memcmp(md5sum, md5_test_sum[i], 16) != 0) {
			dbg_printf("MD5 test result is WRONG!!\r\n");
			__crypto_mem_dump((u8 *)md5_test_sum[i], 16, "sw digest");
			__crypto_mem_dump(md5sum, 16, "hw digest");
			memset(md5sum, 0, 16);
			ret = FAIL;
			break;
		} else {
			dbg_printf("pass! \r\n");
			memset(md5sum, 0, 16);
			ret = SUCCESS;
		}
	}

test_md5_end:
	return ret;
}

int test_sha2_256(void)
{
	int i, j;
	int ret;
	u8 sha2_256sum[32];

	for (i = 0 ; i < 2; i++) {
		memset(digest, 0, sizeof(digest));
		if (AUTH_SHA2_256 == i) {
			dbg_printf("SHA2 test \r\n");
			device_mutex_lock(RT_DEV_LOCK_CRYPTO);
			ret = crypto_sha2_256(plaintext, strlen((const char *)plaintext), digest); // the length of SHA2_256's digest is 32 bytes.
			device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
			if (SUCCESS != ret) {
				dbg_printf("crypto_sha2_256 failed,ret = %d \r\n", ret);
				goto test_sha2_end;
			}
		} else if (AUTH_HMAC_SHA2_256 == i) {
			dbg_printf("SHA2 HMAC test \r\n");
			device_mutex_lock(RT_DEV_LOCK_CRYPTO);
			ret = crypto_hmac_sha2_256(plaintext, strlen((const char *)plaintext), sha2_256_hmac_key, strlen((const char *)sha2_256_hmac_key),
									   digest); // the length of SHA2_256's digest is 32 bytes.
			device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
			if (SUCCESS != ret) {
				dbg_printf("crypto_hmac_sha2_256 failed,ret = %d \r\n", ret);
				goto test_sha2_end;
			}
		}

		if (memcmp(digest, &sha2_256_digest[i][0], 32) == 0) {
			dbg_printf("SHA2_256 test result is correct \r\n");
			ret = SUCCESS;
		} else {
			dbg_printf("SHA2_256 test result is WRONG!! \r\n");
			__crypto_mem_dump((u8 *)&sha2_256_digest[i][0], 32, "sw digest");
			__crypto_mem_dump(digest, 32, "hw digest");
			ret = FAIL;
			goto test_sha2_end;
		}

		for (j = 0; j < 14; j++) {
			if (0 == i) {
				dbg_printf(" SHA2 test #%d: \r\n", (j + 1));
				device_mutex_lock(RT_DEV_LOCK_CRYPTO);
				ret = crypto_sha2_256(auth_plaintext_test_buf[j], auth_plaintext_test_buflen[j], sha2_256sum); // the length of SHA2_256's digest is 32 bytes.
				device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
				if (SUCCESS != ret) {
					dbg_printf("crypto_sha2_256 failed,ret = %d \r\n", ret);
					break;
				}
			} else if (1 == i) {
				dbg_printf(" HMAC SHA2 test #%d: ", (j + 1));
				device_mutex_lock(RT_DEV_LOCK_CRYPTO);
				ret = crypto_hmac_sha2_256(auth_plaintext_test_buf[j], auth_plaintext_test_buflen[j], sha2_256_hmac_key, strlen((const char *)sha2_256_hmac_key),
										   sha2_256sum); // the length of SHA2_256's digest is 32 bytes.
				device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
				if (SUCCESS != ret) {
					dbg_printf("crypto_hmac_sha2_256 failed,ret = %d \r\n", ret);
					break;
				}
			}

			if (memcmp(sha2_256sum, &sha2_256_test_sum[i][j][0], 32) != 0) {
				dbg_printf("SHA2_256 test result is WRONG!!\r\n");
				__crypto_mem_dump((u8 *)&sha2_256_test_sum[i][j][0], 32, "sw digest");
				__crypto_mem_dump(sha2_256sum, 32, "hw digest");
				memset(sha2_256sum, 0, 32);
				ret = FAIL;
				break;
			} else {
				dbg_printf("pass! \r\n");
				memset(sha2_256sum, 0, 32);
				ret = SUCCESS;
			}
		}
	}
test_sha2_end:
	return ret;
}

// vector from NIST: AES ECB/CBC 256 bits :
// key,IV,AAD,tag start address needs to be 32bytes aligned
const unsigned char aes_256_test_key[32] __attribute__((aligned(32))) = {
	0x60, 0x3D, 0xEB, 0x10, 0x15, 0xCA, 0x71, 0xBE,
	0x2B, 0x73, 0xAE, 0xF0, 0x85, 0x7D, 0x77, 0x81,
	0x1F, 0x35, 0x2C, 0x07, 0x3B, 0x61, 0x08, 0xD7,
	0x2D, 0x98, 0x10, 0xA3, 0x09, 0x14, 0xDF, 0xF4
};

const unsigned char aes_test_iv[16] __attribute__((aligned(32))) = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
};

const unsigned char aes_test_buf[16] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a
};

const unsigned char aes_test_res[2][16] = {
	{
		0xF3, 0xEE, 0xD1, 0xBD, 0xB5, 0xD2, 0xA0, 0x3C,
		0x06, 0x4B, 0x5A, 0x7E, 0x3D, 0xB1, 0x81, 0xF8
	},
	{
		0xF5, 0x8C, 0x4C, 0x04, 0xD6, 0xE5, 0xF1, 0xBA,
		0x77, 0x9E, 0xAB, 0xFB, 0x5F, 0x7B, 0xFB, 0xD6
	}
};

// vector from NIST: AES-GCM 256 bits :
const unsigned char aes_256_gcm_test_key[32] __attribute__((aligned(32))) = {
	0x58, 0x53, 0xc0, 0x20, 0x94, 0x6b, 0x35, 0xf2,
	0xc5, 0x8e, 0xc4, 0x27, 0x15, 0x2b, 0x84, 0x04,
	0x20, 0xc4, 0x00, 0x29, 0x63, 0x6a, 0xdc, 0xbb,
	0x02, 0x74, 0x71, 0x37, 0x8c, 0xfd, 0xde, 0x0f
};

const unsigned char aes_gcm_test_iv[12] __attribute__((aligned(32))) = {
	0xee, 0xc3, 0x13, 0xdd, 0x07, 0xcc, 0x1b, 0x3e, 0x6b, 0x06, 0x8a, 0x47
};

const unsigned char aes_gcm_test_aad[20] __attribute__((aligned(32))) = {
	0x13, 0x89, 0xb5, 0x22,
	0xc2, 0x4a, 0x77, 0x41,
	0x81, 0x70, 0x05, 0x53,
	0xf0, 0x24, 0x6b, 0xba,
	0xbd, 0xd3, 0x8d, 0x6f
};

const unsigned char aes_gcm_test_buf[32] = {
	0xce, 0x74, 0x58, 0xe5, 0x6a, 0xef, 0x90, 0x61,
	0xcb, 0x0c, 0x42, 0xec, 0x23, 0x15, 0x56, 0x5e,
	0x61, 0x68, 0xf5, 0xa6, 0x24, 0x9f, 0xfd, 0x31,
	0x61, 0x0b, 0x6d, 0x17, 0xab, 0x64, 0x93, 0x5e
};

const unsigned char aes_gcm_test_res[60] = {
	0xea, 0xdc, 0x3b, 0x87, 0x66, 0xa7, 0x7d, 0xed,
	0x1a, 0x58, 0xcb, 0x72, 0x7e, 0xca, 0x2a, 0x97,
	0x90, 0x49, 0x6c, 0x29, 0x86, 0x54, 0xcd, 0xa7,
	0x8f, 0xeb, 0xf0, 0xda, 0x16, 0xb6, 0x90, 0x3b
};

const unsigned char aes_gcm_test_tag[16] __attribute__((aligned(32))) = {
	0x3d, 0x49, 0xa5, 0xb3, 0x2f, 0xde, 0x7e, 0xaf,
	0xcc, 0xe9, 0x00, 0x79, 0x21, 0x7f, 0xfb, 0x57
};

int test_aes_256(void)
{
	const u8 *key, *pIv;
	u32 keylen = 0;
	u32 ivlen = 0;
	u8 *message;
	u32 msglen;
	u8 *pResult;
	int i;
	int ret;

	for (i = 0; i < 2; i++) {
		key = aes_256_test_key;
		keylen = 32;
		pIv = aes_test_iv;
		ivlen = 16;
		memset(cipher_result, 0, sizeof(cipher_result));
		pResult = cipher_result;
		message = (unsigned char *)aes_test_buf;
		msglen = sizeof(aes_test_buf);

		if (CIPHER_AES_256_ECB == i) {
			dbg_printf("AES 256 ECB test Encrypt \r\n");
			pIv = NULL;
			ivlen = 0;
			device_mutex_lock(RT_DEV_LOCK_CRYPTO);
			ret = crypto_aes_ecb_init(key, keylen);
			if (SUCCESS != ret) {
				device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
				dbg_printf("AES ECB init failed, ret = %d \r\n", ret);
				goto test_aes_end;
			}
			ret = crypto_aes_ecb_encrypt(message, msglen, pIv, ivlen, pResult);
			device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
			if (SUCCESS != ret) {
				dbg_printf("AES ECB encrypt failed, ret = %d \r\n", ret);
				goto test_aes_end;
			}
		} else if (CIPHER_AES_256_CBC == i) {
			dbg_printf("AES 256 CBC test Encrypt \r\n");
			device_mutex_lock(RT_DEV_LOCK_CRYPTO);
			ret = crypto_aes_cbc_init(key, keylen);
			if (SUCCESS != ret) {
				device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
				dbg_printf("AES CBC init failed, ret = %d \r\n", ret);
				goto test_aes_end;
			}
			ret = crypto_aes_cbc_encrypt(message, msglen, pIv, ivlen, pResult);
			device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
			if (SUCCESS != ret) {
				dbg_printf("AES CBC encrypt failed, ret = %d \r\n", ret);
				goto test_aes_end;
			}
		}
		if (memcmp(&aes_test_res[i][0], pResult, msglen) == 0) {
			dbg_printf("AES 256 encrypt result success \r\n");
			ret = SUCCESS;
		} else {
			dbg_printf("AES 256 encrypt result failed \r\n");
			__crypto_mem_dump((u8 *)&aes_test_res[i][0], msglen, "sw encrypt msg");
			__crypto_mem_dump(pResult, msglen, "hw encrypt msg");
			ret = FAIL;
			goto test_aes_end;
		}

		message = pResult;

		if (CIPHER_AES_256_ECB == i) {
			dbg_printf("AES 256 ECB test Decrypt \r\n");
			device_mutex_lock(RT_DEV_LOCK_CRYPTO);
			ret = crypto_aes_ecb_init(key, keylen);
			if (SUCCESS != ret) {
				device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
				dbg_printf("AES ECB init failed, ret = %d \r\n", ret);
				goto test_aes_end;
			}
			ret = crypto_aes_ecb_decrypt(message, msglen, pIv, ivlen, pResult);
			device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
			if (SUCCESS != ret) {
				dbg_printf("AES ECB decrypt failed, ret = %d \r\n", ret);
				goto test_aes_end;
			}
		} else if (CIPHER_AES_256_CBC == i) {
			dbg_printf("AES 256 CBC test Decrypt \r\n");
			device_mutex_lock(RT_DEV_LOCK_CRYPTO);
			ret = crypto_aes_cbc_init(key, keylen);
			if (SUCCESS != ret) {
				device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
				dbg_printf("AES CBC init failed, ret = %d \r\n", ret);
				goto test_aes_end;
			}
			ret = crypto_aes_cbc_decrypt(message, msglen, pIv, ivlen, pResult);
			device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
			if (SUCCESS != ret) {
				dbg_printf("AES CBC decrypt failed, ret = %d \r\n", ret);
				goto test_aes_end;
			}
		}

		if (memcmp(aes_test_buf, pResult, msglen) == 0) {
			dbg_printf("AES CBC decrypt result success \r\n");
			ret = SUCCESS;
		} else {
			dbg_printf("AES CBC decrypt result failed \r\n");
			__crypto_mem_dump((u8 *)aes_test_buf, msglen, "sw decrypt msg");
			__crypto_mem_dump(pResult, msglen, "hw decrypt msg");
			ret = FAIL;
		}
	}

test_aes_end:
	return ret;
}

int test_aes_256_gcm(void)
{
	const u8 *key, *nonce, *pAAD;
	u32 keylen = 0;
	u8 *message;
	u32 msglen, aadlen;
	u8 *pResult, *pTag;
	int i;
	int ret;

	for (i = 0; i < 1; i++) {
		key = aes_256_gcm_test_key;
		keylen = 32;
		nonce = aes_gcm_test_iv;
		memset(cipher_result, 0, sizeof(cipher_result));
		memset(tag, 0, sizeof(tag));
		pResult = cipher_result;
		pTag = tag;
		pAAD = (unsigned char *)aes_gcm_test_aad;
		aadlen = sizeof(aes_gcm_test_aad);
		message = (unsigned char *)aes_gcm_test_buf;
		msglen = sizeof(aes_gcm_test_buf);

		dbg_printf("AES 256 GCM test Encrypt \r\n");
		device_mutex_lock(RT_DEV_LOCK_CRYPTO);
		ret = crypto_aes_gcm_init(key, keylen);
		if (SUCCESS != ret) {
			device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
			dbg_printf("AES GCM init failed, ret = %d \r\n", ret);
			goto test_aes_gcm_end;
		}
		ret = crypto_aes_gcm_encrypt(message, msglen, nonce, pAAD, aadlen, pResult, pTag);
		device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
		if (SUCCESS != ret) {
			dbg_printf("AES GCM encrypt failed, ret = %d \r\n", ret);
			goto test_aes_gcm_end;
		}

		if (memcmp(aes_gcm_test_res, pResult, msglen) == 0) {
			dbg_printf("AES GCM 256 encrypt result success \r\n");
			if (memcmp(aes_gcm_test_tag, pTag, 16) == 0) {
				dbg_printf("AES GCM 256 encrypt tag success \r\n");
				ret = SUCCESS;
			} else {
				dbg_printf("AES GCM 256 encrypt tag failed \r\n");
				__crypto_mem_dump((u8 *)aes_gcm_test_tag, 16, "sw encrypt tag");
				__crypto_mem_dump(pTag, 16, "hw encrypt tag");
				ret = FAIL;
				goto test_aes_gcm_end;
			}
		} else {
			dbg_printf("AES GCM 256 encrypt result failed \r\n");
			__crypto_mem_dump((u8 *)aes_gcm_test_res, msglen, "sw encrypt msg");
			__crypto_mem_dump(pResult, msglen, "hw encrypt msg");
			ret = FAIL;
			goto test_aes_gcm_end;
		}

		message = pResult;
		memset(tag, 0, sizeof(tag));

		dbg_printf("AES 256 GCM test Decrypt \r\n");
		device_mutex_lock(RT_DEV_LOCK_CRYPTO);
		ret = crypto_aes_gcm_init(key, keylen);
		if (SUCCESS != ret) {
			device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
			dbg_printf("AES GCM init failed, ret = %d \r\n", ret);
			goto test_aes_gcm_end;
		}
		ret = crypto_aes_gcm_decrypt(message, msglen, nonce, pAAD, aadlen, pResult, pTag);
		device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
		if (memcmp(aes_gcm_test_buf, pResult, msglen) == 0) {
			dbg_printf("AES GCM 256 decrypt result success \r\n");
			if (memcmp(aes_gcm_test_tag, pTag, 16) == 0) {
				dbg_printf("AES GCM 256 decrypt tag success \r\n");
				ret = SUCCESS;
			} else {
				dbg_printf("AES GCM 256 decrypt tag failed \r\n");
				__crypto_mem_dump((u8 *)aes_gcm_test_tag, 16, "sw decrypt tag");
				__crypto_mem_dump(pTag, 16, "hw decrypt tag");
				ret = FAIL;
				goto test_aes_gcm_end;
			}
		} else {
			dbg_printf("AES GCM 256 decrypt result failed \r\n");
			__crypto_mem_dump((u8 *)aes_gcm_test_buf, msglen, "sw decrypt msg");
			__crypto_mem_dump(pResult, msglen, "hw decrypt msg");
			ret = FAIL;
			goto test_aes_gcm_end;
		}
	}

test_aes_gcm_end:
	return ret;
}

static void test_thread(void *param)
{
	// sample text
	int ret;

	dbg_printf("\r\n   CRYPTO Demo   \r\n");

	device_mutex_lock(RT_DEV_LOCK_CRYPTO);
	ret = crypto_init();
	device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
	if (SUCCESS != ret) {
		dbg_printf("crypto engine init failed \r\n");
	}
	ret = test_md5();
	if (SUCCESS != ret) {
		dbg_printf("crypto test_md5 failed \r\n");
		goto vrf_end;
	}

	ret = test_sha2_256();
	if (SUCCESS != ret) {
		dbg_printf("crypto test_sha2_256 failed \r\n");
		goto vrf_end;
	}

	ret = test_aes_256();
	if (SUCCESS != ret) {
		dbg_printf("crypto test_aes_256 failed \r\n");
		goto vrf_end;
	}
	ret = test_aes_256_gcm();
	if (SUCCESS != ret) {
		dbg_printf("crypto test_aes_256_gcm failed \r\n");
		goto vrf_end;
	}
vrf_end:
	device_mutex_lock(RT_DEV_LOCK_CRYPTO);
	ret = crypto_deinit();
	device_mutex_unlock(RT_DEV_LOCK_CRYPTO);
	if (SUCCESS != ret) {
		dbg_printf("crypto engine deinit failed \r\n");
	}

	vTaskDelete(NULL);
}

int main(void)
{

	/* Note that it should be protected by device_mutex_lock() if using HW crypto */
	if (xTaskCreate(test_thread, "test_thread", STACKSIZE, NULL, tskIDLE_PRIORITY + 1, NULL) != pdPASS) {
		printf("\n\r%s xTaskCreate failed", __FUNCTION__);
	}

	vTaskStartScheduler();
	while (1);
}
